﻿using RimWorld;
using RimWorld.Planet;
using System.Collections.Generic;
using UnityEngine;
using Verse;
using Verse.Sound;

namespace MedTimes
{
    class Building_TrapAnimal : Building_TrapRearmable
    {
        protected bool HuntOnSpring = false;

        protected virtual void SpringDebilitateBottom(Pawn p)
        {
            SoundDefOf.TrapSpring.PlayOneShot(new TargetInfo(base.Position, base.Map));
            if (p == null)
            {
                return;
            }
            float damageAmountTotal = this.GetStatValue(StatDefOf.TrapMeleeDamage) * DamageRandomFactorRange.RandomInRange;
            float damageAmountAtomic = damageAmountTotal / DamageCount;
            float armorPenetration = damageAmountAtomic * 0.015f;
                                                          
            DamageInfo dinfo = new DamageInfo(DamageDefOf.Stun, damageAmountTotal, armorPenetration, -1f, this);
            DamageWorker.DamageResult damageResult = p.TakeDamage(dinfo);
            
            BattleLogEntry_DamageTaken battleLogEntry_DamageTaken = new BattleLogEntry_DamageTaken(p, RulePackDefOf.DamageEvent_TrapSpike);
            Find.BattleLog.Add(battleLogEntry_DamageTaken);
            damageResult.AssociateWithLog(battleLogEntry_DamageTaken);
            

            for (int i = 0; (float)i < DamageCount; i++)
            {
                dinfo = new DamageInfo(DamageDefOf.Bite, damageAmountAtomic, armorPenetration, -1f, this);
                dinfo.SetBodyRegion(BodyPartHeight.Bottom);  
                damageResult = p.TakeDamage(dinfo);
                if (i == 0)
                {
                    battleLogEntry_DamageTaken = new BattleLogEntry_DamageTaken(p, RulePackDefOf.DamageEvent_TrapSpike);
                    Find.BattleLog.Add(battleLogEntry_DamageTaken);
                    damageResult.AssociateWithLog(battleLogEntry_DamageTaken);
                }
            }
        }

        public override void ExposeData()
        {
            base.ExposeData();

            Scribe_Values.Look(ref HuntOnSpring, "HuntOnSpring", false);
        }

        public override bool KnowsOfTrap(Pawn p)
        {
            if (p.Faction != null && !p.Faction.HostileTo(base.Faction))
            {
                return true;
            }
            if (p.Faction == null && p.RaceProps.Animal)
            {
                return false;
            }
            if (p.guest != null && p.guest.Released)
            {
                return true;
            }
            if (!p.IsPrisoner && base.Faction != null && p.HostFaction == base.Faction)
            {
                return true;
            }
            if (p.RaceProps.Humanlike && p.IsFormingCaravan())
            {
                return true;
            }
            if (p.IsPrisoner && p.guest.ShouldWaitInsteadOfEscaping && base.Faction == p.HostFaction)
            {
                return true;
            }
            if (p.Faction == null && p.RaceProps.Humanlike)
            {
                return true;
            }
            return false;
        }         

        protected override void SpringSub(Pawn pawn)
        {
            SpringDebilitateBottom(pawn);
            var desHunt = Find.ReverseDesignatorDatabase.Get<Designator_Hunt>();
            if (HuntOnSpring && desHunt.CanDesignateThing(pawn).Accepted)
            {
                desHunt.DesignateThing(pawn);
            }  
        }

        protected override float SpringChance(Pawn p)
        {
            if (p.RaceProps.Animal && !p.RaceProps.ToolUser)
            {
                return 1.0f * this.GetStatValue(StatDefOf.TrapSpringChance);
            }      
            return base.SpringChance(p); ;
        }

        public override IEnumerable<Gizmo> GetGizmos()
        {
            foreach (Gizmo gizmo in base.GetGizmos())
            {
                yield return gizmo;
            }
            yield return new Command_Toggle
            {                                                              
                defaultLabel = ResourceBank.Strings.CommandHuntOnSpring.Translate(),
                defaultDesc = ResourceBank.Strings.CommandHuntOnSpringDesc.Translate(),     
                icon = ContentFinder<Texture2D>.Get("UI/Designators/Hunt"),
                isActive = (() => HuntOnSpring),
                toggleAction = delegate
                {
                    HuntOnSpring = !HuntOnSpring;
                }
            };
        }
    }
}
