﻿using RimWorld;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using UnityEngine;
using Verse;

namespace MedTimes
{
    class CompQuarryWorker : ThingComp
    {
        private ThingDef resDef;

        private float portionProgress;

        private float portionYieldPct;               

        private const float WorkPerPortionBase = 8000f;

        private const float YieldPerWorkDone = 0.5f;

        public static float WorkPerPortionCurrentDifficulty => WorkPerPortionBase / Find.Storyteller.difficulty.mineYieldFactor;

        public float ProgressToNextPortionPercent => portionProgress / WorkPerPortionCurrentDifficulty;

        public static List<ThingDef> ResourceDefs = DefDatabase<ThingDef>.AllDefs.Where(def => def.deepCommonality > 0).ToList();

        public static float ResourceChancePer10kCells = (DefDatabase<ThingDef>.AllDefs.Where(def => def.deepCommonality > 0).Sum(def => def.deepCommonality * def.deepLumpSizeRange.Average) / 10000f);

        public override void PostExposeData()
        {
            Scribe_Values.Look(ref portionProgress, "portionProgress", 0f);
            Scribe_Values.Look(ref portionYieldPct, "portionYieldPct", 0f);
            Scribe_Defs.Look(ref resDef, "resDef");
        }

        public override void PostSpawnSetup(bool respawningAfterLoad)
        {
            base.PostSpawnSetup(respawningAfterLoad);
            if(!respawningAfterLoad && resDef== null)
            {          
                GetNextResource(out resDef);
            }
        }

        public void MineWorkDone(Pawn driller)
        {
            float statValue = driller.GetStatValue(StatDefOf.MiningSpeed);
            portionProgress += statValue;
            portionYieldPct += YieldPerWorkDone * statValue * driller.GetStatValue(StatDefOf.MiningYield) / WorkPerPortionCurrentDifficulty; 
            if (portionProgress > WorkPerPortionCurrentDifficulty)
            {
                TryProducePortion(portionYieldPct, driller);
                portionProgress = 0f;
                portionYieldPct = 0f;
            }
        }

        public override void PostDeSpawn(Map map)
        {
            portionProgress = 0f;
            portionYieldPct = 0f;       
        }

        private void TryProducePortion(float yieldPct, Pawn driller)
        {  
            if(resDef == null)
                GetNextResource(out resDef);
            if (resDef == null)
            {
                return;
            }
            int num = Mathf.Max(1, resDef.deepCountPerPortion);

            int stackCount = Mathf.Max(1, GenMath.RoundRandom((float)num * yieldPct));
            Thing thing = ThingMaker.MakeThing(resDef);
            thing.stackCount = stackCount;
            GenPlace.TryPlaceThing(thing, parent.InteractionCell, parent.Map, ThingPlaceMode.Near);
            if(parent is IBillGiver billGiver)
            {
                var bills = billGiver.BillStack.Bills;
                for (int i = 0; i < bills.Count; ++i)
                {
                    if (bills[i].ShouldDoNow() && bills[i].IsFixedOrAllowedIngredient(thing)) {
                        bills[i].Notify_IterationCompleted(driller, new List<Thing>());
                        break;
                    }    
                }             
            }

            GetNextResource(out resDef);
        }

        private void GetNextResource(out ThingDef resDef)
        {
            if(Rand.Value < ResourceChancePer10kCells)
            {
                resDef = ResourceDefs.RandomElementByWeight(def => def.deepCommonality * def.deepLumpSizeRange.Average);
                
            }
            else
            {
                resDef = (from rock in Find.World.NaturalRockTypesIn(parent.Map.Tile)
                        select rock.building.mineableThing).RandomElement();
            }     
        }

        public bool CanMineNow()
        {
            if (resDef != null || DeepDrillUtility.GetBaseResource(parent.Map) != null)
            {                
                return true;
            }
            return false;
        }

        public override string CompInspectStringExtra()
        {
            if (parent.Spawned)
            {
                if (resDef == null)
                {
                    return ResourceBank.Strings.DeepDrillNoResources.Translate();
                }
                return ResourceBank.Strings.ResourceBelow.Translate() + ": " + resDef.LabelCap + "\n" + ResourceBank.Strings.ProgressToNextPortion.Translate() + ": " + ProgressToNextPortionPercent.ToStringPercent("F0");
            }
            return null;
        }
    }
}
