﻿using MedTimes;
using RimWorld;
using System;
using System.Collections.Generic;
using UnityEngine;
using Verse;
using Verse.AI;
using Verse.Sound;

namespace MedTimes
{
    class JobDriver_ReadBook : JobDriver
    {
        //redo this
        protected bool readingFromInventory;

        protected const TargetIndex BookIndex = TargetIndex.A;

        protected const TargetIndex PawnReadToIndex = TargetIndex.B;

        protected const TargetIndex BookSurfaceIndex = TargetIndex.C;

        protected Thing BookSource => job.GetTarget(TargetIndex.A).Thing;

        protected float ReadDurationMultiplier
        {
            get
            {
                return 1f / pawn.GetStatValue(StatDefOf.UnskilledLaborSpeed);
            }
        }

        protected virtual SkillDef GetBookSkill()
        {
            return null;
        }

        public override void ExposeData()
        {
            base.ExposeData();
            Scribe_Values.Look(ref readingFromInventory, "readingFromInventory", defaultValue: false);
        }

        public override void Notify_Starting()
        {
            base.Notify_Starting();
            readingFromInventory = (pawn.inventory != null && pawn.inventory.Contains(BookSource));
            this.TargetThingB = pawn;
        }

        public override bool TryMakePreToilReservations(bool errorOnFailed)
        {
            Pawn pawn = base.pawn;
            LocalTargetInfo target = BookSource;
            Job job = base.job;
            if (!pawn.Reserve(target, job, 1, -1, null, errorOnFailed))
            {
                return false;
            }

            return true;
        }

        protected override IEnumerable<Toil> MakeNewToils()
        {
            this.FailOn(() => BookSource.Destroyed);
            Toil readToil = this.GetReadToil(Mathf.RoundToInt(300 * ReadDurationMultiplier));

            if (readingFromInventory)
            {
                yield return Toils_Misc.TakeItemFromInventoryToCarrier(pawn, TargetIndex.A);
            }
            else
            {
                yield return ReserveBook();
                Toil gotoToPickup = Toils_Goto.GotoThing(TargetIndex.A, PathEndMode.ClosestTouch).FailOnDespawnedNullOrForbidden(TargetIndex.A);
                yield return Toils_Jump.JumpIf(gotoToPickup, () => pawn.health.capacities.CapableOf(PawnCapacityDefOf.Manipulation));
                yield return Toils_Goto.GotoThing(TargetIndex.A, PathEndMode.Touch).FailOnDespawnedNullOrForbidden(TargetIndex.A);
                yield return Toils_Jump.Jump(readToil);
                yield return gotoToPickup;
                yield return PickupBook();
            }
            yield return CarryBookToReadSpot(pawn, BookIndex).FailOnDestroyedOrNull(BookIndex);
            yield return Toils_Ingest.FindAdjacentEatSurface(BookSurfaceIndex, BookIndex);

            yield return readToil;
            yield return Toils_Jump.JumpIf(readToil, () => Find.TickManager.TicksGame < (startTick + job.def.joyDuration) && pawn.needs.joy.CurLevel < 0.99999f);

            if (readingFromInventory)
            {
                yield return Toils_Haul.TakeToInventory(TargetIndex.A, BookSource.stackCount);
            }
            else
            {
                Toil haulJobToil = new Toil();
                haulJobToil.initAction = delegate
                {
                    StoragePriority currentPriority = StoreUtility.CurrentStoragePriorityOf(BookSource);
                    if (!StoreUtility.TryFindBestBetterStorageFor(BookSource, pawn, pawn.Map, currentPriority, pawn.Faction, out IntVec3 foundCell, out IHaulDestination haulDestination))
                    {
                        JobFailReason.Is("NoEmptyPlaceLower".Translate());
                        return;
                    }

                    Job haul = HaulAIUtility.HaulToCellStorageJob(pawn, BookSource, foundCell, fitInStoreCell: false);

                    pawn.jobs.StartJob(haul, JobCondition.Succeeded);
                };
                haulJobToil.defaultCompleteMode = ToilCompleteMode.Instant;
                yield return haulJobToil;
            }
        }

        private Toil PickupBook()
        {
            Toil toil = new Toil();
            toil.initAction = delegate
            {
                Pawn actor = toil.actor;
                Job curJob = actor.jobs.curJob;
                Thing haulThing = curJob.GetTarget(BookIndex).Thing;
                if (!haulThing.Spawned || haulThing.stackCount <= 0)
                {
                    pawn.jobs.EndCurrentJob(JobCondition.Incompletable);
                }
                else
                {
                    int stackCount = haulThing.stackCount;
                    actor.carryTracker.TryStartCarry(haulThing, stackCount);
                    if (haulThing != actor.carryTracker.CarriedThing && actor.Map.reservationManager.ReservedBy(haulThing, actor, curJob))
                    {
                        actor.Map.reservationManager.Release(haulThing, actor, curJob);
                    }
                    actor.jobs.curJob.targetA = actor.carryTracker.CarriedThing;
                }
            };
            return toil;
        }

        private Toil ReserveBook()
        {
            Toil toil = new Toil();
            toil.initAction = delegate
            {
                if (pawn.Faction != null)
                {
                    Thing thing = job.GetTarget(TargetIndex.A).Thing;
                    if (pawn.carryTracker.CarriedThing != thing)
                    {
                        pawn.Reserve(thing, job, 1, 1);
                    }
                }
            };
            toil.defaultCompleteMode = ToilCompleteMode.Instant;
            toil.atomicWithPrevious = true;
            return toil;
        }

        protected virtual Toil GetReadToil(int duration)
        {
            Toil readToil = new Toil();
            SkillDef skill = GetBookSkill();
            readToil.initAction = delegate
            {
                if (BookSource.Spawned)
                {
                    BookSource.Map.physicalInteractionReservationManager.Reserve(pawn, pawn.CurJob, BookSource);

                    ResourceBank.SoundDefOf.BookPageFlips.PlayOneShot(new TargetInfo(this.pawn.Position, this.pawn.Map));
                }
                if (pawn.CurJob.GetTarget(BookSurfaceIndex).IsValid)
                    this.pawn.rotationTracker.FaceCell(pawn.CurJob.GetTarget(BookSurfaceIndex).Cell);
            };
            readToil.defaultDuration = duration;
            readToil.tickAction = delegate
            {
                this.pawn.GainComfortFromCellIfPossible();

                ReadTick(pawn, BookSource, skill, 1f, JoyTickFullJoyAction.GoToNextToil);
            };
            readToil.defaultCompleteMode = ToilCompleteMode.Delay;
            readToil.handlingFacing = true;
            readToil.AddFinishAction(delegate
            {
                JoyUtility.TryGainRecRoomThought(pawn);
            });

            return readToil;
        }

        protected virtual void ReadTick(Pawn pawn, Thing bookSource, SkillDef skill, float extraJoyGainFactor = 1f, JoyTickFullJoyAction fullJoyAction = JoyTickFullJoyAction.EndJob)
        {
            Job curJob = pawn.CurJob;
            if (curJob.def.joyKind == null)
            {
                Log.Warning("This method can only be called for jobs with joyKind.");
                return;
            }
            if (bookSource != null)
            {
                extraJoyGainFactor *= bookSource.GetStatValue(ResourceBank.StatDefOf.BookQualityFactor);
            }

            pawn.rotationTracker.FaceCell(bookSource.Position);

            pawn.needs.joy.GainJoy(extraJoyGainFactor * curJob.def.joyGainRate * 0.36f / 2500f, curJob.def.joyKind);
            if (curJob.def.joySkill != null)
            {
                pawn.skills.GetSkill(curJob.def.joySkill).Learn(curJob.def.joyXpPerTick);
            }
            if (!curJob.ignoreJoyTimeAssignment && !pawn.GetTimeAssignment().allowJoy)
            {
                pawn.jobs.curDriver.EndJobWith(JobCondition.InterruptForced);
            }
            if (pawn.needs.joy.CurLevel > 0.9999f)
            {
                pawn.jobs.curDriver.ReadyForNextToil();
            }
        }

        /*
     public override bool ModifyCarriedThingDrawPos(ref Vector3 drawPos, ref bool behind, ref bool flip)
     {
         IntVec3 cell = job.GetTarget(TargetIndex.B).Cell;
         return ModifyCarriedThingDrawPosWorker(ref drawPos, ref behind, ref flip, cell, pawn);
     }

     public static bool ModifyCarriedThingDrawPosWorker(ref Vector3 drawPos, ref bool behind, ref bool flip, IntVec3 placeCell, Pawn pawn)
     {
         if (pawn.pather.Moving)
         {
             return false;
         }
         Thing carriedThing = pawn.carryTracker.CarriedThing;
         if (carriedThing == null)
         {
             return false;
         }
         if (placeCell.IsValid && placeCell.AdjacentToCardinal(pawn.Position) && placeCell.HasEatSurface(pawn.Map) && carriedThing.def.ingestible.ingestHoldUsesTable)
         {
             drawPos = new Vector3((float)placeCell.x + 0.5f, drawPos.y, (float)placeCell.z + 0.5f);
             return true;
         }
         if (carriedThing.def.ingestible.ingestHoldOffsetStanding != null)
         {
             HoldOffset holdOffset = carriedThing.def.ingestible.ingestHoldOffsetStanding.Pick(pawn.Rotation);
             if (holdOffset != null)
             {
                 drawPos += holdOffset.offset;
                 behind = holdOffset.behind;
                 flip = holdOffset.flip;
                 return true;
             }
         }
         return false;
     }     */

        public static Toil CarryBookToReadSpot(Pawn pawn, TargetIndex ingestibleInd)
        {
            Toil toil = new Toil();
            toil.initAction = delegate
            {
                Pawn actor = toil.actor;
                IntVec3 intVec = IntVec3.Invalid;
                Thing thing = null;
                Thing thing2 = actor.CurJob.GetTarget(ingestibleInd).Thing;
                Predicate<Thing> baseChairValidator = delegate (Thing t)
                {
                    if (t.def.building == null || !t.def.building.isSittable)
                    {
                        return false;
                    }
                    if (t.IsForbidden(pawn))
                    {
                        return false;
                    }
                    if (!actor.CanReserve(t))
                    {
                        return false;
                    }
                    if (!t.IsSociallyProper(actor))
                    {
                        return false;
                    }
                    if (t.IsBurning())
                    {
                        return false;
                    }
                    if (t.HostileTo(pawn))
                    {
                        return false;
                    }
                    bool flag = false;

                    IntVec3 c = t.Position + t.Rotation.FacingCell;
                    Thing readLocThing = c.GetEdifice(t.Map);
                    if (readLocThing != null && readLocThing.def.passability != Traversability.Impassable)
                    {
                        flag = true;
                    }
                    else
                    {
                        flag = true;
                    }

                    return flag;
                };
                if (thing2.GetStatValue(ResourceBank.StatDefOf.BookChairSearchRadius) > 0f)
                {
                    thing = GenClosest.ClosestThingReachable(actor.Position, actor.Map, ThingRequest.ForGroup(ThingRequestGroup.BuildingArtificial), PathEndMode.OnCell, TraverseParms.For(actor), thing2.GetStatValue(ResourceBank.StatDefOf.BookChairSearchRadius), (Thing t) => baseChairValidator(t) && t.Position.GetDangerFor(pawn, t.Map) == Danger.None);
                }
                if (thing == null)
                {
                    intVec = RCellFinder.SpotToChewStandingNear(actor, actor.CurJob.GetTarget(ingestibleInd).Thing);
                    Danger chewSpotDanger = intVec.GetDangerFor(pawn, actor.Map);
                    if (chewSpotDanger != Danger.None)
                    {
                        thing = GenClosest.ClosestThingReachable(actor.Position, actor.Map, ThingRequest.ForGroup(ThingRequestGroup.BuildingArtificial), PathEndMode.OnCell, TraverseParms.For(actor), thing2.GetStatValue(ResourceBank.StatDefOf.BookChairSearchRadius), (Thing t) => baseChairValidator(t) && (int)t.Position.GetDangerFor(pawn, t.Map) <= (int)chewSpotDanger);
                    }
                }
                if (thing != null)
                {
                    intVec = thing.Position;
                    actor.Reserve(thing, actor.CurJob);
                }
                actor.Map.pawnDestinationReservationManager.Reserve(actor, actor.CurJob, intVec);
                actor.pather.StartPath(intVec, PathEndMode.OnCell);
            };
            toil.defaultCompleteMode = ToilCompleteMode.PatherArrival;
            return toil;
        }
    }
}
