﻿using RimWorld;
using System;
using System.Collections.Generic;
using System.Linq;
using UnityEngine;
using Verse;
using Verse.AI;

namespace MedTimes
{
    public class JobDriver_WriteBook : JobDriver_DoBill
    {

        public static Toil PlaceHauledThingInCell(TargetIndex cellInd, Toil nextToilOnPlaceFailOrIncomplete, bool storageMode)
        {
            Toil toil = new Toil();
            toil.initAction = delegate
            {
                Pawn actor = toil.actor;
                Job curJob = actor.jobs.curJob;
                IntVec3 cell = curJob.GetTarget(cellInd).Cell;
                if (actor.carryTracker.CarriedThing == null)
                {
                    Log.Error(actor + " tried to place hauled thing in cell but is not hauling anything.");
                }
                else
                {
                    SlotGroup slotGroup = actor.Map.haulDestinationManager.SlotGroupAt(cell);
                    if (slotGroup != null && slotGroup.Settings.AllowedToAccept(actor.carryTracker.CarriedThing))
                    {
                        actor.Map.designationManager.TryRemoveDesignationOn(actor.carryTracker.CarriedThing, DesignationDefOf.Haul);
                    }
                    Action<Thing, int> placedAction = null;
                    if (curJob.def == JobDefOf.DoBill || curJob.def == ResourceBank.JobDefOf.WriteBook || curJob.def == JobDefOf.RefuelAtomic || curJob.def == JobDefOf.RearmTurretAtomic)
                    {
                        placedAction = delegate (Thing th, int added)
                        {
                            if (curJob.placedThings == null)
                            {
                                curJob.placedThings = new List<ThingCountClass>();
                            }
                            ThingCountClass thingCountClass = curJob.placedThings.Find((ThingCountClass x) => x.thing == th);
                            if (thingCountClass != null)
                            {
                                thingCountClass.Count += added;
                            }
                            else
                            {
                                curJob.placedThings.Add(new ThingCountClass(th, added));
                            }
                        };
                    }
                    if (!actor.carryTracker.TryDropCarriedThing(cell, ThingPlaceMode.Direct, out Thing _, placedAction))
                    {
                        if (storageMode)
                        {
                            if (nextToilOnPlaceFailOrIncomplete != null && StoreUtility.TryFindBestBetterStoreCellFor(actor.carryTracker.CarriedThing, actor, actor.Map, StoragePriority.Unstored, actor.Faction, out IntVec3 foundCell))
                            {
                                if (actor.CanReserve(foundCell))
                                {
                                    actor.Reserve(foundCell, actor.CurJob);
                                }
                                actor.CurJob.SetTarget(cellInd, foundCell);
                                actor.jobs.curDriver.JumpToToil(nextToilOnPlaceFailOrIncomplete);
                            }
                            else
                            {
                                Job job = HaulAIUtility.HaulAsideJobFor(actor, actor.carryTracker.CarriedThing);
                                if (job != null)
                                {
                                    curJob.targetA = job.targetA;
                                    curJob.targetB = job.targetB;
                                    curJob.targetC = job.targetC;
                                    curJob.count = job.count;
                                    curJob.haulOpportunisticDuplicates = job.haulOpportunisticDuplicates;
                                    curJob.haulMode = job.haulMode;
                                    actor.jobs.curDriver.JumpToToil(nextToilOnPlaceFailOrIncomplete);
                                }
                                else
                                {
                                    Log.Error("Incomplete haul for " + actor + ": Could not find anywhere to put " + actor.carryTracker.CarriedThing + " near " + actor.Position + ". Destroying. This should never happen!");
                                    actor.carryTracker.CarriedThing.Destroy();
                                }
                            }
                        }
                        else if (nextToilOnPlaceFailOrIncomplete != null)
                        {
                            actor.jobs.curDriver.JumpToToil(nextToilOnPlaceFailOrIncomplete);
                        }
                    }
                }
            };
            return toil;
        }

        private static IEnumerable<Thing> MakeRecipeProducts(RecipeDef recipeDef, Pawn worker, List<Thing> ingredients, Thing dominantIngredient, IBillGiver billGiver)
        {
            float efficiency = (recipeDef.efficiencyStat != null) ? worker.GetStatValue(recipeDef.efficiencyStat) : 1f;
            if (recipeDef.workTableEfficiencyStat != null)
            {
                Building_WorkTable building_WorkTable = billGiver as Building_WorkTable;
                if (building_WorkTable != null)
                {
                    efficiency *= building_WorkTable.GetStatValue(recipeDef.workTableEfficiencyStat);
                }
            }
            if (recipeDef.products != null)
            {
                for (int k = 0; k < recipeDef.products.Count; k++)
                {
                    ThingDefCountClass prod = recipeDef.products[k];
                    Thing product3 = ThingMaker.MakeThing(stuff: (!prod.thingDef.MadeFromStuff) ? null : dominantIngredient.def, def: prod.thingDef);
                    product3.stackCount = Mathf.CeilToInt((float)prod.count * efficiency);
                    if (dominantIngredient != null)
                    {
                        product3.SetColor(dominantIngredient.DrawColor, reportFailure: false);
                    }
                    CompIngredients ingredientsComp = product3.TryGetComp<CompIngredients>();
                    if (ingredientsComp != null)
                    {
                        for (int l = 0; l < ingredients.Count; l++)
                        {
                            ingredientsComp.RegisterIngredient(ingredients[l].def);
                        }
                    }                                                                               
                   
                    yield return TryPostProcessBook(product3, recipeDef, worker);
                }
            }
            if (recipeDef.specialProducts == null)
            {
                yield break;
            }
            for (int j = 0; j < recipeDef.specialProducts.Count; j++)
            {
                for (int i = 0; i < ingredients.Count; i++)
                {
                    Thing ing = ingredients[i];
                    switch (recipeDef.specialProducts[j])
                    {
                        case SpecialProductType.Butchery:
                            foreach (Thing product in ing.ButcherProducts(worker, efficiency))
                            {
                                yield return TryPostProcessBook(product, recipeDef, worker);
                            }
                            break;
                        case SpecialProductType.Smelted:
                            foreach (Thing product2 in ing.SmeltProducts(efficiency))
                            {
                                yield return TryPostProcessBook(product2, recipeDef, worker);
                            }
                            break;
                    }
                }
            }
        }

        protected static Thing TryPostProcessBook(Thing product, RecipeDef recipeDef, Pawn worker)
        {
            Log.Message(product.def.defName);
            CompQuality compQuality = product.TryGetComp<CompQuality>();
            if (compQuality != null)
            {
                CompBook bookComp = product.TryGetComp<CompBook>();
                QualityCategory q;
                if(bookComp != null)
                {
                    bool inspired  = worker.InspirationDef == InspirationDefOf.Inspired_Creativity;
                    if (bookComp.Props.Skill != null)
                    {
                        q = GenerateQualityCreatedByPawn((int)(worker.skills.GetSkill(bookComp.Props.Skill).Level * worker.GetStatValue(ResourceBank.StatDefOf.ComposingQuality)), inspired);
                    }
                    else
                    {
                        q = GenerateQualityCreatedByPawn((int)(worker.skills.GetSkill(SkillDefOf.Artistic).Level * worker.GetStatValue(ResourceBank.StatDefOf.ComposingQuality)), inspired);
                    }
                }
                else
                {            
                    if (recipeDef.workSkill == null)
                    {
                        Log.Error(recipeDef + " needs workSkill because it creates a product with a quality.");
                    }  

                    q = QualityUtility.GenerateQualityCreatedByPawn(worker, recipeDef.workSkill);
                }
                compQuality.SetQuality(q, ArtGenerationContext.Colony);
                QualityUtility.SendCraftNotification(product, worker);
            }
            CompArt compArt = product.TryGetComp<CompArt>();
            if (compArt != null)
            {
                compArt.JustCreatedBy(worker);
                if ((int)compQuality.Quality >= 4)
                {
                    TaleRecorder.RecordTale(TaleDefOf.CraftedArt, worker, product);
                }
            }
            if (product.def.Minifiable)
            {
                product = product.MakeMinified();
            }
            return product;
        }

        private static QualityCategory GenerateQualityCreatedByPawn(int relevantSkillLevel, bool inspired)
        {
            float num = 0f;
            switch (relevantSkillLevel)
            {
                case 0:
                    num += 0.7f;
                    break;
                case 1:
                    num += 1.1f;
                    break;
                case 2:
                    num += 1.5f;
                    break;
                case 3:
                    num += 1.8f;
                    break;
                case 4:
                    num += 2f;
                    break;
                case 5:
                    num += 2.2f;
                    break;
                case 6:
                    num += 2.4f;
                    break;
                case 7:
                    num += 2.6f;
                    break;
                case 8:
                    num += 2.8f;
                    break;
                case 9:
                    num += 2.95f;
                    break;
                case 10:
                    num += 3.1f;
                    break;
                case 11:
                    num += 3.25f;
                    break;
                case 12:
                    num += 3.4f;
                    break;
                case 13:
                    num += 3.5f;
                    break;
                case 14:
                    num += 3.6f;
                    break;
                case 15:
                    num += 3.7f;
                    break;
                case 16:
                    num += 3.8f;
                    break;
                case 17:
                    num += 3.9f;
                    break;
                case 18:
                    num += 4f;
                    break;
                case 19:
                    num += 4.1f;
                    break;
                case 20:
                    num += 4.2f;
                    break;
            }
            int value = (int)Rand.GaussianAsymmetric(num, 0.6f, 0.8f);
            value = Mathf.Clamp(value, 0, 5);
            if (value == 5 && Rand.Value < 0.5f)
            {
                value = (int)Rand.GaussianAsymmetric(num, 0.6f, 0.95f);
                value = Mathf.Clamp(value, 0, 5);
            }
            QualityCategory qualityCategory = (QualityCategory)value;
            if (inspired)
            {
                qualityCategory = AddLevels(qualityCategory, 2);
            }
            return qualityCategory;
        }

        private static QualityCategory AddLevels(QualityCategory quality, int levels)
        {
            return (QualityCategory)Mathf.Min((int)quality + levels, 6);
        }

        private static Toil FinishBookAndHaulIt()
        {
            Toil toil = new Toil();
            toil.initAction = delegate
            {
                Pawn actor = toil.actor;
                Job curJob = actor.jobs.curJob;
                JobDriver_DoBill jobDriver_DoBill = (JobDriver_DoBill)actor.jobs.curDriver;
                if (curJob.RecipeDef.workSkill != null && !curJob.RecipeDef.UsesUnfinishedThing)
                {
                    float xp = (float)jobDriver_DoBill.ticksSpentDoingRecipeWork * 0.1f * curJob.RecipeDef.workSkillLearnFactor;
                    actor.skills.GetSkill(curJob.RecipeDef.workSkill).Learn(xp);
                }
                List<Thing> ingredients = CalculateIngredients(curJob, actor);
                Thing dominantIngredient = CalculateDominantIngredient(curJob, ingredients);
                List<Thing> list = MakeRecipeProducts(curJob.RecipeDef, actor, ingredients, dominantIngredient, jobDriver_DoBill.BillGiver).ToList();
                ConsumeIngredients(ingredients, curJob.RecipeDef, actor.Map);
                curJob.bill.Notify_IterationCompleted(actor, ingredients);
                RecordsUtility.Notify_BillDone(actor, list);
                UnfinishedThing unfinishedThing = curJob.GetTarget(TargetIndex.B).Thing as UnfinishedThing;
                if (curJob.bill.recipe.WorkAmountTotal(unfinishedThing?.Stuff) >= 10000f && list.Count > 0)
                {
                    TaleRecorder.RecordTale(TaleDefOf.CompletedLongCraftingProject, actor, list[0].GetInnerIfMinified().def);
                }
                if (list.Count == 0)
                {
                    actor.jobs.EndCurrentJob(JobCondition.Succeeded);
                }
                else if (curJob.bill.GetStoreMode() == BillStoreModeDefOf.DropOnFloor)
                {
                    for (int i = 0; i < list.Count; i++)
                    {
                        if (!GenPlace.TryPlaceThing(list[i], actor.Position, actor.Map, ThingPlaceMode.Near))
                        {
                            Log.Error(actor + " could not drop recipe product " + list[i] + " near " + actor.Position);
                        }
                    }
                    actor.jobs.EndCurrentJob(JobCondition.Succeeded);
                }
                else
                {
                    if (list.Count > 1)
                    {
                        for (int j = 1; j < list.Count; j++)
                        {
                            if (!GenPlace.TryPlaceThing(list[j], actor.Position, actor.Map, ThingPlaceMode.Near))
                            {
                                Log.Error(actor + " could not drop recipe product " + list[j] + " near " + actor.Position);
                            }
                        }
                    }
                    IntVec3 foundCell = IntVec3.Invalid;
                    if (curJob.bill.GetStoreMode() == BillStoreModeDefOf.BestStockpile)
                    {
                        StoreUtility.TryFindBestBetterStoreCellFor(list[0], actor, actor.Map, StoragePriority.Unstored, actor.Faction, out foundCell);
                    }
                    else if (curJob.bill.GetStoreMode() == BillStoreModeDefOf.SpecificStockpile)
                    {
                        StoreUtility.TryFindBestBetterStoreCellForIn(list[0], actor, actor.Map, StoragePriority.Unstored, actor.Faction, curJob.bill.GetStoreZone().slotGroup, out foundCell);
                    }
                    else
                    {
                        Log.ErrorOnce("Unknown store mode", 9158246);
                    }
                    if (foundCell.IsValid)
                    {
                        actor.carryTracker.TryStartCarry(list[0]);
                        curJob.targetB = foundCell;
                        curJob.targetA = list[0];
                        curJob.count = 99999;
                    }
                    else
                    {
                        if (!GenPlace.TryPlaceThing(list[0], actor.Position, actor.Map, ThingPlaceMode.Near))
                        {
                            Log.Error("Bill doer could not drop product " + list[0] + " near " + actor.Position);
                        }
                        actor.jobs.EndCurrentJob(JobCondition.Succeeded);
                    }
                }
            };
            return toil;  
        }

        protected override IEnumerable<Toil> MakeNewToils()
        {
            AddEndCondition(delegate
            {
                Thing thing = GetActor().jobs.curJob.GetTarget(TargetIndex.A).Thing;
                if (thing is Building && !thing.Spawned)
                {
                    return JobCondition.Incompletable;
                }
                return JobCondition.Ongoing;
            });
            this.FailOnBurningImmobile(TargetIndex.A);
            this.FailOn(delegate
            {
                IBillGiver billGiver = job.GetTarget(TargetIndex.A).Thing as IBillGiver;
                if (billGiver != null)
                {
                    if (job.bill.DeletedOrDereferenced)
                    {
                        return true;
                    }
                    if (!billGiver.CurrentlyUsableForBills())
                    {
                        return true;
                    }
                }
                return false;
            });
            Toil gotoBillGiver = Toils_Goto.GotoThing(TargetIndex.A, PathEndMode.InteractionCell);
            yield return new Toil
            {
                initAction = delegate
                {
                    if (job.targetQueueB != null && job.targetQueueB.Count == 1)
                    {
                        UnfinishedThing unfinishedThing = job.targetQueueB[0].Thing as UnfinishedThing;
                        if (unfinishedThing != null)
                        {
                            unfinishedThing.BoundBill = (Bill_ProductionWithUft)job.bill;
                        }
                    }
                }
            };
            yield return Toils_Jump.JumpIf(gotoBillGiver, () => job.GetTargetQueue(TargetIndex.B).NullOrEmpty());
            Toil extract = Toils_JobTransforms.ExtractNextTargetFromQueue(TargetIndex.B);
            yield return extract;
            Toil getToHaulTarget = Toils_Goto.GotoThing(TargetIndex.B, PathEndMode.ClosestTouch).FailOnDespawnedNullOrForbidden(TargetIndex.B).FailOnSomeonePhysicallyInteracting(TargetIndex.B);
            yield return getToHaulTarget;
            yield return Toils_Haul.StartCarryThing(TargetIndex.B, putRemainderInQueue: true, subtractNumTakenFromJobCount: false, failIfStackCountLessThanJobCount: true);
            yield return JumpToCollectNextIntoHandsForBill(getToHaulTarget, TargetIndex.B);
            yield return Toils_Goto.GotoThing(TargetIndex.A, PathEndMode.InteractionCell).FailOnDestroyedOrNull(TargetIndex.B);
            Toil findPlaceTarget = Toils_JobTransforms.SetTargetToIngredientPlaceCell(TargetIndex.A, TargetIndex.B, TargetIndex.C);
            yield return findPlaceTarget;
            yield return PlaceHauledThingInCell(TargetIndex.C, findPlaceTarget, storageMode: false);
            yield return Toils_Jump.JumpIfHaveTargetInQueue(TargetIndex.B, extract);
            yield return gotoBillGiver;
            yield return Toils_Recipe.MakeUnfinishedThingIfNeeded();
            yield return Toils_Recipe.DoRecipeWork()
                .FailOnDespawnedNullOrForbiddenPlacedThings()
                .FailOnCannotTouch(TargetIndex.A, PathEndMode.InteractionCell);
            yield return FinishBookAndHaulIt();
            if (!job.RecipeDef.products.NullOrEmpty() || !job.RecipeDef.specialProducts.NullOrEmpty())
            {
                yield return Toils_Reserve.Reserve(TargetIndex.B);
                Toil carryToCell = Toils_Haul.CarryHauledThingToCell(TargetIndex.B);
                yield return carryToCell;
                yield return PlaceHauledThingInCell(TargetIndex.B, carryToCell, storageMode: true);
                Toil recount = new Toil();
                recount.initAction = delegate
                {
                    Bill_Production bill_Production = recount.actor.jobs.curJob.bill as Bill_Production;
                    if (bill_Production != null && bill_Production.repeatMode == BillRepeatModeDefOf.TargetCount)
                    {
                        Map.resourceCounter.UpdateResourceCounts();
                    }
                };
                yield return recount;
            }
        }

        private static Toil JumpToCollectNextIntoHandsForBill(Toil gotoGetTargetToil, TargetIndex ind)
        {
            Toil toil = new Toil();
            toil.initAction = delegate
            {
                Pawn actor = toil.actor;
                if (actor.carryTracker.CarriedThing == null)
                {
                    Log.Error("JumpToAlsoCollectTargetInQueue run on " + actor + " who is not carrying something.");
                }
                else if (!actor.carryTracker.Full)
                {
                    Job curJob = actor.jobs.curJob;
                    List<LocalTargetInfo> targetQueue = curJob.GetTargetQueue(ind);
                    if (!targetQueue.NullOrEmpty())
                    {
                        int num = 0;
                        int a;
                        while (true)
                        {
                            if (num >= targetQueue.Count)
                            {
                                return;
                            }
                            if (GenAI.CanUseItemForWork(actor, targetQueue[num].Thing) && targetQueue[num].Thing.CanStackWith(actor.carryTracker.CarriedThing) && !((float)(actor.Position - targetQueue[num].Thing.Position).LengthHorizontalSquared > 64f))
                            {
                                int num2 = (actor.carryTracker.CarriedThing != null) ? actor.carryTracker.CarriedThing.stackCount : 0;
                                a = curJob.countQueue[num];
                                a = Mathf.Min(a, targetQueue[num].Thing.def.stackLimit - num2);
                                a = Mathf.Min(a, actor.carryTracker.AvailableStackSpace(targetQueue[num].Thing.def));
                                if (a > 0)
                                {
                                    break;
                                }
                            }
                            num++;
                        }
                        curJob.count = a;
                        curJob.SetTarget(ind, targetQueue[num].Thing);
                        List<int> countQueue;
                        int index;
                        (countQueue = curJob.countQueue)[index = num] = countQueue[index] - a;
                        if (curJob.countQueue[num] <= 0)
                        {
                            curJob.countQueue.RemoveAt(num);
                            targetQueue.RemoveAt(num);
                        }
                        actor.jobs.curDriver.JumpToToil(gotoGetTargetToil);
                    }
                }
            };
            return toil;
        }

        private static List<Thing> CalculateIngredients(Job job, Pawn actor)
        {
            UnfinishedThing unfinishedThing = job.GetTarget(TargetIndex.B).Thing as UnfinishedThing;
            if (unfinishedThing != null)
            {
                List<Thing> ingredients = unfinishedThing.ingredients;
                job.RecipeDef.Worker.ConsumeIngredient(unfinishedThing, job.RecipeDef, actor.Map);
                job.placedThings = null;
                return ingredients;
            }
            List<Thing> list = new List<Thing>();
            if (job.placedThings != null)
            {
                for (int i = 0; i < job.placedThings.Count; i++)
                {
                    if (job.placedThings[i].Count <= 0)
                    {
                        Log.Error("PlacedThing " + job.placedThings[i] + " with count " + job.placedThings[i].Count + " for job " + job);
                        continue;
                    }
                    Thing thing = (job.placedThings[i].Count >= job.placedThings[i].thing.stackCount) ? job.placedThings[i].thing : job.placedThings[i].thing.SplitOff(job.placedThings[i].Count);
                    job.placedThings[i].Count = 0;
                    if (list.Contains(thing))
                    {
                        Log.Error("Tried to add ingredient from job placed targets twice: " + thing);
                        continue;
                    }
                    list.Add(thing);
                    if (job.RecipeDef.autoStripCorpses)
                    {
                        (thing as IStrippable)?.Strip();
                    }
                }
            }
            job.placedThings = null;
            return list;
        }

        private static Thing CalculateDominantIngredient(Job job, List<Thing> ingredients)
        {
            UnfinishedThing uft = job.GetTarget(TargetIndex.B).Thing as UnfinishedThing;
            if (uft != null && uft.def.MadeFromStuff)
            {
                return uft.ingredients.First((Thing ing) => ing.def == uft.Stuff);
            }
            if (!ingredients.NullOrEmpty())
            {
                if (job.RecipeDef.productHasIngredientStuff)
                {
                    return ingredients[0];
                }
                if (job.RecipeDef.products.Any((ThingDefCountClass x) => x.thingDef.MadeFromStuff))
                {
                    return (from x in ingredients
                            where x.def.IsStuff
                            select x).RandomElementByWeight((Thing x) => x.stackCount);
                }
                return ingredients.RandomElementByWeight((Thing x) => x.stackCount);
            }
            return null;
        }

        private static void ConsumeIngredients(List<Thing> ingredients, RecipeDef recipe, Map map)
        {
            for (int i = 0; i < ingredients.Count; i++)
            {
                recipe.Worker.ConsumeIngredient(ingredients[i], recipe, map);
                Log.Message(ingredients[i].def.defName);
            }
        }
    }
}
/*
      Toil toil = new Toil();
            toil.initAction = delegate
            {
                Pawn actor = toil.actor;
                Job curJob = actor.jobs.curJob;
                JobDriver_DoBill jobDriver_DoBill = (JobDriver_DoBill)actor.jobs.curDriver;
                if (curJob.RecipeDef.workSkill != null && !curJob.RecipeDef.UsesUnfinishedThing)
                {
                    float xp = (float)jobDriver_DoBill.ticksSpentDoingRecipeWork * 0.1f * curJob.RecipeDef.workSkillLearnFactor;
                    actor.skills.GetSkill(curJob.RecipeDef.workSkill).Learn(xp);
                }
                List<Thing> ingredients = CalculateIngredients(curJob, actor);
                Thing dominantIngredient = CalculateDominantIngredient(curJob, ingredients);
                List<Thing> list = MakeRecipeProducts(curJob.RecipeDef, actor, ingredients, dominantIngredient, jobDriver_DoBill.BillGiver).ToList();                                        

                ConsumeIngredients(ingredients, curJob.RecipeDef, actor.Map);
                curJob.bill.Notify_IterationCompleted(actor, ingredients);
                RecordsUtility.Notify_BillDone(actor, list);
                UnfinishedThing unfinishedThing = curJob.GetTarget(TargetIndex.B).Thing as UnfinishedThing;
                if (curJob.bill.recipe.WorkAmountTotal(unfinishedThing?.Stuff) >= 10000f && list.Count > 0)
                {
                    TaleRecorder.RecordTale(TaleDefOf.CompletedLongCraftingProject, actor, list[0].GetInnerIfMinified().def);
                }
                if (list.Count == 0)
                {
                    actor.jobs.EndCurrentJob(JobCondition.Succeeded);
                }
                else if (curJob.bill.GetStoreMode() == BillStoreModeDefOf.DropOnFloor)
                {
                    for (int i = 0; i < list.Count; i++)
                    {
                        if (!GenPlace.TryPlaceThing(list[i], actor.Position, actor.Map, ThingPlaceMode.Near))
                        {
                            Log.Error(actor + " could not drop recipe product " + list[i] + " near " + actor.Position);
                        }
                    }
                    actor.jobs.EndCurrentJob(JobCondition.Succeeded);
                }
                else
                {
                    if (list.Count > 1)
                    {
                        for (int j = 1; j < list.Count; j++)
                        {
                            if (!GenPlace.TryPlaceThing(list[j], actor.Position, actor.Map, ThingPlaceMode.Near))
                            {
                                Log.Error(actor + " could not drop recipe product " + list[j] + " near " + actor.Position);
                            }
                        }
                    }
                    IntVec3 foundCell = IntVec3.Invalid;
                    if (curJob.bill.GetStoreMode() == BillStoreModeDefOf.BestStockpile)
                    {
                        StoreUtility.TryFindBestBetterStoreCellFor(list[0], actor, actor.Map, StoragePriority.Unstored, actor.Faction, out foundCell);
                    }
                    else if (curJob.bill.GetStoreMode() == BillStoreModeDefOf.SpecificStockpile)
                    {
                        StoreUtility.TryFindBestBetterStoreCellForIn(list[0], actor, actor.Map, StoragePriority.Unstored, actor.Faction, curJob.bill.GetStoreZone().slotGroup, out foundCell);
                    }
                    else
                    {
                        Log.ErrorOnce("Unknown store mode", 9158246);
                    }
                    if (foundCell.IsValid)
                    {
                        actor.carryTracker.TryStartCarry(list[0]);
                        curJob.targetB = foundCell;
                        curJob.targetA = list[0];
                        curJob.count = 99999;
                    }
                    else
                    {
                        if (!GenPlace.TryPlaceThing(list[0], actor.Position, actor.Map, ThingPlaceMode.Near))
                        {
                            Log.Error("Bill doer could not drop product " + list[0] + " near " + actor.Position);
                        }
                        actor.jobs.EndCurrentJob(JobCondition.Succeeded);
                    }
                }
            };
            return toil;
    */
