﻿using RimWorld;
using System;
using System.Collections.Generic;
using UnityEngine;
using Verse;
using Verse.AI;
using Verse.Sound;

namespace MedTimes
{
    class JobDriver_ReadNote : JobDriver
    {                              
        protected bool readingFromInventory = false;
        protected int LastExaustTick = -1;
        protected int ReservedScrolls = -1;

        protected const TargetIndex ScrollIndex = TargetIndex.A;

        protected const TargetIndex PawnReadToIndex = TargetIndex.B;

        protected const TargetIndex ScrollSurfaceIndex = TargetIndex.C;

        protected Thing ScrollSource => job.GetTarget(TargetIndex.A).Thing;

        protected float ReadDurationMultiplier
        {
            get
            {
                return 1f / pawn.GetStatValue(StatDefOf.GeneralLaborSpeed);
            }
        }

        protected virtual SkillDef GetScrollSkill()
        {
            return null;
        }

        public override void ExposeData()
        {
            base.ExposeData();
            Scribe_Values.Look(ref readingFromInventory, "readingFromInventory", defaultValue: false);
            Scribe_Values.Look(ref LastExaustTick, "ExaustTicks", defaultValue: -1);
            Scribe_Values.Look(ref ReservedScrolls, "ReservedScrolls", defaultValue: -1);
        }

        public override void Notify_Starting()
        {
            base.Notify_Starting();
            readingFromInventory = (pawn.inventory != null && pawn.inventory.Contains(ScrollSource));
            this.TargetThingB = pawn;
        }

        public override bool TryMakePreToilReservations(bool errorOnFailed)
        {
            Pawn pawn = base.pawn;
            LocalTargetInfo target = ScrollSource;
            Job job = base.job;
            if (!pawn.Reserve(target, job, 1, -1, null, errorOnFailed))
            {
                return false;
            }

            return true;
        }

        protected override IEnumerable<Toil> MakeNewToils()
        {
            this.FailOn(() => ScrollSource.Destroyed);
            Toil readToil = this.GetReadToil(Mathf.RoundToInt(300 * ReadDurationMultiplier));

            if (readingFromInventory)
            {
                yield return Toils_Misc.TakeItemFromInventoryToCarrier(pawn, TargetIndex.A);
            }
            else
            {
                yield return ReserveScroll();
                Toil gotoToPickup = Toils_Goto.GotoThing(TargetIndex.A, PathEndMode.ClosestTouch).FailOnDespawnedNullOrForbidden(TargetIndex.A);
                yield return Toils_Jump.JumpIf(gotoToPickup, () => pawn.health.capacities.CapableOf(PawnCapacityDefOf.Manipulation));
                yield return Toils_Goto.GotoThing(TargetIndex.A, PathEndMode.Touch).FailOnDespawnedNullOrForbidden(TargetIndex.A);
                yield return Toils_Jump.Jump(readToil);
                yield return gotoToPickup;
                yield return PickupScroll();
            }
            yield return CarryScrollToReadSpot(pawn, ScrollIndex).FailOnDestroyedOrNull(ScrollIndex);
            yield return Toils_Ingest.FindAdjacentEatSurface(ScrollSurfaceIndex, ScrollIndex);
            yield return InitReadToil();
            yield return readToil;
            yield return ExaustScroll();
            yield return Toils_Jump.JumpIf(readToil, () => Find.TickManager.TicksGame < (startTick + job.def.joyDuration) && pawn.needs.joy.CurLevel < 0.99999f);

            if (readingFromInventory)
            {
                yield return Toils_Haul.TakeToInventory(TargetIndex.A, ScrollSource.stackCount);
            }
            else
            {         
                Toil haulJobToil = new Toil();
                haulJobToil.initAction = delegate
                {
                    StoragePriority currentPriority = StoreUtility.CurrentStoragePriorityOf(ScrollSource);
                    if (!StoreUtility.TryFindBestBetterStorageFor(ScrollSource, pawn, pawn.Map, currentPriority, pawn.Faction, out IntVec3 foundCell, out IHaulDestination haulDestination))
                    {
                        JobFailReason.Is("NoEmptyPlaceLower".Translate());
                        return;
                    }

                    Job haul = HaulAIUtility.HaulToCellStorageJob(pawn, ScrollSource, foundCell, fitInStoreCell: false);

                    pawn.jobs.StartJob(haul, JobCondition.Succeeded);
                };
                haulJobToil.defaultCompleteMode = ToilCompleteMode.Instant;
                yield return haulJobToil;
            }
        }

        private Toil PickupScroll()
        {
            Toil toil = new Toil();
            toil.initAction = delegate
            {
                Pawn actor = toil.actor;
                Job curJob = actor.jobs.curJob;
                Thing haulThing = curJob.GetTarget(ScrollIndex).Thing;
                if (!haulThing.Spawned || haulThing.stackCount <= 0)
                {
                    pawn.jobs.EndCurrentJob(JobCondition.Incompletable);
                }
                else
                {                                          
                    actor.carryTracker.TryStartCarry(haulThing, ReservedScrolls);
                    if (haulThing != actor.carryTracker.CarriedThing && actor.Map.reservationManager.ReservedBy(haulThing, actor, curJob))
                    {
                        actor.Map.reservationManager.Release(haulThing, actor, curJob);
                    }
                    actor.jobs.curJob.targetA = actor.carryTracker.CarriedThing;
                }
            };
            return toil;
        }

        private Toil ReserveScroll()
        {
            Toil toil = new Toil();
            toil.initAction = delegate
            {
                if (pawn.Faction != null)
                {
                    Thing thing = job.GetTarget(TargetIndex.A).Thing;
                    ReservedScrolls = Math.Min((job.def.joyDuration / ResourceBank.Numbers.TicksToDestroyScroll) * 2 + 1, thing.stackCount);

                    if (pawn.carryTracker.CarriedThing != thing)
                    {
                        pawn.Reserve(thing, job, 1, ReservedScrolls);
                    }
                }
            };
            toil.defaultCompleteMode = ToilCompleteMode.Instant;
            toil.atomicWithPrevious = true;
            return toil;
        }                    

        private Toil InitReadToil()
        {
            Toil toil = new Toil()
            {
                defaultCompleteMode = ToilCompleteMode.Instant,
                initAction = delegate
                {
                    LastExaustTick = Find.TickManager.TicksGame;
                }  
            };                           
            return toil;
        }

        protected virtual Toil GetReadToil(int duration)
        {
            Toil readToil = new Toil();
            SkillDef skill = GetScrollSkill();
            readToil.initAction = delegate
            {
                if (ScrollSource.Spawned)
                {
                    ScrollSource.Map.physicalInteractionReservationManager.Reserve(pawn, pawn.CurJob, ScrollSource);

                    ResourceBank.SoundDefOf.BookPageFlips.PlayOneShot(new TargetInfo(this.pawn.Position, this.pawn.Map));
                }
                if (pawn.CurJob.GetTarget(ScrollSurfaceIndex).IsValid)
                    this.pawn.rotationTracker.FaceCell(pawn.CurJob.GetTarget(ScrollSurfaceIndex).Cell);
            };
            readToil.defaultDuration = duration;
            readToil.tickAction = delegate
            {                  
                this.pawn.GainComfortFromCellIfPossible();  

                ReadTick(pawn, ScrollSource, skill, 1f, JoyTickFullJoyAction.GoToNextToil);
            };
            readToil.defaultCompleteMode = ToilCompleteMode.Delay;
            readToil.handlingFacing = true;
            readToil.AddFinishAction(delegate
            {
                JoyUtility.TryGainRecRoomThought(pawn);
            });

            return readToil;
        }

        private Toil ExaustScroll()
        {                         
            Toil toil = new Toil();
            toil.initAction = delegate
            {    
                if(LastExaustTick + ResourceBank.Numbers.TicksToDestroyScroll < Find.TickManager.TicksGame )    

                LastExaustTick = Find.TickManager.TicksGame;

                if (Rand.Value < ResourceBank.Numbers.ChanceToDestroyScroll)
                {                                       
                    Thing thing = job.GetTarget(TargetIndex.A).Thing;
                    if (thing.stackCount == 1)
                    {
                        thing.Destroy();
                        pawn.jobs.EndCurrentJob(JobCondition.InterruptForced);
                    }
                    else
                    {
                        --thing.stackCount;
                    }
                }  
            };
            toil.defaultCompleteMode = ToilCompleteMode.Instant;
            toil.atomicWithPrevious = true;
            return toil;
        }

        protected virtual void ReadTick(Pawn pawn, Thing scrollSource, SkillDef skill, float extraJoyGainFactor = 1f, JoyTickFullJoyAction fullJoyAction = JoyTickFullJoyAction.EndJob)
        {
            Job curJob = pawn.CurJob;
            if (curJob.def.joyKind == null)
            {
                Log.Warning("This method can only be called for jobs with joyKind.");
                return;
            }
            if (scrollSource != null)
            {
                extraJoyGainFactor *= scrollSource.GetStatValue(ResourceBank.StatDefOf.BookQualityFactor);
            }

            pawn.rotationTracker.FaceCell(scrollSource.Position);

            pawn.needs.joy.GainJoy(extraJoyGainFactor * curJob.def.joyGainRate * 0.36f / 2500f, curJob.def.joyKind);
            if (curJob.def.joySkill != null)
            {
                pawn.skills.GetSkill(curJob.def.joySkill).Learn(curJob.def.joyXpPerTick);
            }
            if (!curJob.ignoreJoyTimeAssignment && !pawn.GetTimeAssignment().allowJoy)
            {
                pawn.jobs.curDriver.EndJobWith(JobCondition.InterruptForced);
            }
            if (pawn.needs.joy.CurLevel > 0.9999f) {                                        
                pawn.jobs.curDriver.ReadyForNextToil();
            }
        }       

           /*
        public override bool ModifyCarriedThingDrawPos(ref Vector3 drawPos, ref bool behind, ref bool flip)
        {
            IntVec3 cell = job.GetTarget(TargetIndex.B).Cell;
            return ModifyCarriedThingDrawPosWorker(ref drawPos, ref behind, ref flip, cell, pawn);
        }

        public static bool ModifyCarriedThingDrawPosWorker(ref Vector3 drawPos, ref bool behind, ref bool flip, IntVec3 placeCell, Pawn pawn)
        {
            if (pawn.pather.Moving)
            {
                return false;
            }
            Thing carriedThing = pawn.carryTracker.CarriedThing;
            if (carriedThing == null)
            {
                return false;
            }
            if (placeCell.IsValid && placeCell.AdjacentToCardinal(pawn.Position) && placeCell.HasEatSurface(pawn.Map) && carriedThing.def.ingestible.ingestHoldUsesTable)
            {
                drawPos = new Vector3((float)placeCell.x + 0.5f, drawPos.y, (float)placeCell.z + 0.5f);
                return true;
            }
            if (carriedThing.def.ingestible.ingestHoldOffsetStanding != null)
            {
                HoldOffset holdOffset = carriedThing.def.ingestible.ingestHoldOffsetStanding.Pick(pawn.Rotation);
                if (holdOffset != null)
                {
                    drawPos += holdOffset.offset;
                    behind = holdOffset.behind;
                    flip = holdOffset.flip;
                    return true;
                }
            }
            return false;
        }     */

        public static Toil CarryScrollToReadSpot(Pawn pawn, TargetIndex ingestibleInd)
        {
            Toil toil = new Toil();
            toil.initAction = delegate
            {
                Pawn actor = toil.actor;
                IntVec3 intVec = IntVec3.Invalid;
                Thing thing = null;
                Thing thing2 = actor.CurJob.GetTarget(ingestibleInd).Thing;
                Predicate<Thing> baseChairValidator = delegate (Thing t)
                {
                    if (t.def.building == null || !t.def.building.isSittable)
                    {
                        return false;
                    }
                    if (t.IsForbidden(pawn))
                    {
                        return false;
                    }
                    if (!actor.CanReserve(t))
                    {
                        return false;
                    }
                    if (!t.IsSociallyProper(actor))
                    {
                        return false;
                    }
                    if (t.IsBurning())
                    {
                        return false;
                    }
                    if (t.HostileTo(pawn))
                    {
                        return false;
                    }
                    bool flag = false;
                    
                    IntVec3 c = t.Position + t.Rotation.FacingCell;
                    Thing readLocThing = c.GetEdifice(t.Map);
                    if (readLocThing != null && readLocThing.def.passability != Traversability.Impassable)
                    {                                                       
                        flag = true;
                    }
                    else
                    {
                        flag = true;
                    }        

                    return flag;
                };
                if (thing2.GetStatValue(ResourceBank.StatDefOf.BookChairSearchRadius) > 0f)
                {
                    thing = GenClosest.ClosestThingReachable(actor.Position, actor.Map, ThingRequest.ForGroup(ThingRequestGroup.BuildingArtificial), PathEndMode.OnCell, TraverseParms.For(actor), thing2.GetStatValue(ResourceBank.StatDefOf.BookChairSearchRadius), (Thing t) => baseChairValidator(t) && t.Position.GetDangerFor(pawn, t.Map) == Danger.None);
                }
                if (thing == null)
                {
                    intVec = RCellFinder.SpotToChewStandingNear(actor, actor.CurJob.GetTarget(ingestibleInd).Thing);
                    Danger chewSpotDanger = intVec.GetDangerFor(pawn, actor.Map);
                    if (chewSpotDanger != Danger.None)
                    {
                        thing = GenClosest.ClosestThingReachable(actor.Position, actor.Map, ThingRequest.ForGroup(ThingRequestGroup.BuildingArtificial), PathEndMode.OnCell, TraverseParms.For(actor), thing2.GetStatValue(ResourceBank.StatDefOf.BookChairSearchRadius), (Thing t) => baseChairValidator(t) && (int)t.Position.GetDangerFor(pawn, t.Map) <= (int)chewSpotDanger);
                    }
                }
                if (thing != null)
                {
                    intVec = thing.Position;
                    actor.Reserve(thing, actor.CurJob);
                }      

                actor.Map.pawnDestinationReservationManager.Reserve(actor, actor.CurJob, intVec);
                actor.pather.StartPath(intVec, PathEndMode.OnCell);
            };
            toil.defaultCompleteMode = ToilCompleteMode.PatherArrival;
            return toil;
        }     
    }
}
