﻿using RimWorld;
using System;
using System.Collections.Generic;
using UnityEngine;
using Verse;
using Verse.AI;

namespace MedTimes
{
    abstract class IJobDriver_Read : JobDriver
    {                                          /*
        public override bool ModifyCarriedThingDrawPos(ref Vector3 drawPos, ref bool behind, ref bool flip)
        {
            IntVec3 cell = job.GetTarget(TargetIndex.B).Cell;
            return ModifyCarriedThingDrawPosWorker(ref drawPos, ref behind, ref flip, cell, pawn);
        }

        public static bool ModifyCarriedThingDrawPosWorker(ref Vector3 drawPos, ref bool behind, ref bool flip, IntVec3 placeCell, Pawn pawn)
        {
            if (pawn.pather.Moving)
            {
                return false;
            }
            Thing carriedThing = pawn.carryTracker.CarriedThing;
            if (carriedThing == null)
            {
                return false;
            }
            if (placeCell.IsValid && placeCell.AdjacentToCardinal(pawn.Position) && placeCell.HasEatSurface(pawn.Map) && carriedThing.def.ingestible.ingestHoldUsesTable)
            {
                drawPos = new Vector3((float)placeCell.x + 0.5f, drawPos.y, (float)placeCell.z + 0.5f);
                return true;
            }
            if (carriedThing.def.ingestible.ingestHoldOffsetStanding != null)
            {
                HoldOffset holdOffset = carriedThing.def.ingestible.ingestHoldOffsetStanding.Pick(pawn.Rotation);
                if (holdOffset != null)
                {
                    drawPos += holdOffset.offset;
                    behind = holdOffset.behind;
                    flip = holdOffset.flip;
                    return true;
                }
            }
            return false;
        }     */
        protected bool readingFromInventory = false;

        public const TargetIndex ReadSourceIndex = TargetIndex.A;

        public const TargetIndex ReadItemIndex = TargetIndex.B;

        public const TargetIndex ReadSurfaceIndex = TargetIndex.C;

        protected Building_ContainerShelf ReadSource => job.GetTarget(ReadSourceIndex).Thing as Building_ContainerShelf;

        protected Thing ReadItem => job.GetTarget(ReadItemIndex).Thing;

        protected float ReadDurationMultiplier
        {
            get
            {
                return 1f / pawn.GetStatValue(StatDefOf.GeneralLaborSpeed);
            }
        }         

        public override void ExposeData()
        {
            base.ExposeData();
            Scribe_Values.Look(ref readingFromInventory, "readingFromInventory", defaultValue: false);
        }

        public override void Notify_Starting()
        {
            base.Notify_Starting();
            readingFromInventory = (pawn.inventory != null && pawn.inventory.Contains(ReadItem));
        }

        public override bool TryMakePreToilReservations(bool errorOnFailed)
        {
            Pawn pawn = base.pawn;

            Job job = base.job;
            if (ReadSource != null)
            {
                if (!pawn.Reserve(ReadSource, job, 1, -1, null, errorOnFailed))
                {
                    return false;
                }
            }
            else
            {
                if (!pawn.Reserve(ReadItem, job, 1, job.count, null, errorOnFailed))
                {
                    return false;
                }
            }

            return true;
        }

        protected Toil PickupReadable()
        {
            Toil toil = new Toil();
            toil.initAction = delegate
            {
                Pawn actor = toil.actor;
                Job curJob = actor.jobs.curJob;

                if (ReadSource != null && ReadSource.Destroyed)
                {
                    pawn.jobs.EndCurrentJob(JobCondition.Incompletable);
                }
                else
                if (ReadSource is Building_ContainerShelf bookShelf)
                {
                    Thing readThing = bookShelf.GetDirectlyHeldThings().Take(ReadItem, job.count);

                    job.SetTarget(ReadItemIndex, readThing);

                    if (!actor.carryTracker.TryStartCarry(readThing))
                    {
                        pawn.jobs.EndCurrentJob(JobCondition.Incompletable);
                        GenSpawn.Spawn(readThing, pawn.Position, pawn.Map, WipeMode.VanishOrMoveAside);
                    }

                    if (ReadSource.Spawned && actor.Map.reservationManager.ReservedBy(ReadSource, actor, curJob))
                    {
                        actor.Map.reservationManager.Release(ReadSource, actor, curJob);
                    }
                    actor.jobs.curJob.SetTarget(ReadItemIndex, actor.carryTracker.CarriedThing);

                    /*
                    float value = BookUtilites.GetReadSourceWeight(scroll, pawn);

                    if (value == maxValue) scrollsToPickup.Add(scroll);
                    else
                    if (value > maxValue)
                    {
                        value = maxValue;
                        scrollsToPickup.Clear();
                        scrollsToPickup.Add(scroll);
                    }   */
                }
                else
                if (!ReadItem.Spawned)
                {
                    pawn.jobs.EndCurrentJob(JobCondition.Incompletable);
                }
                else
                {
                    Thing splitOffThing = ReadItem;

                    Thing readThing = splitOffThing.SplitOff(curJob.count);

                    job.SetTarget(ReadItemIndex, readThing);
                    actor.carryTracker.TryStartCarry(readThing);

                    if (splitOffThing.Spawned && splitOffThing != actor.carryTracker.CarriedThing && actor.Map.reservationManager.ReservedBy(splitOffThing, actor, curJob))
                    {
                        actor.Map.reservationManager.Release(splitOffThing, actor, curJob);
                    }
                    actor.jobs.curJob.targetA = actor.carryTracker.CarriedThing;
                }
            };
            return toil;
        }

        protected abstract Toil GetReadToil(int duration);

        protected virtual Toil InitRead()
        {
            Toil toil = new Toil()
            {
                defaultCompleteMode = ToilCompleteMode.Instant
            };
            toil.atomicWithPrevious = true;
            toil.initAction = delegate
            {
                if (ReadItem.Spawned)
                {
                    ReadItem.Map.physicalInteractionReservationManager.Reserve(pawn, pawn.CurJob, ReadItem);
                }
            };
            return toil;
        }

        protected virtual Toil EndRead()
        {
            Toil toil = new Toil()
            {
                defaultCompleteMode = ToilCompleteMode.Instant
            };
            toil.atomicWithPrevious = true;
            return toil;
        }

        private Toil ReserveToil()
        {
            Toil toil = new Toil();
            toil.initAction = delegate
            {
                if (pawn.Faction != null)
                {
                    if (ReadSource is Building_ContainerShelf && ReadSource.Spawned)
                    {
                        pawn.Reserve(ReadSource, job);
                    }
                    else
                    if (ReadItem != null && ReadItem.Spawned)
                    {
                        pawn.Reserve(ReadItem, job, -1, job.count);
                    }
                }
            };
            toil.defaultCompleteMode = ToilCompleteMode.Instant;
            toil.atomicWithPrevious = true;
            return toil;
        }

        protected override IEnumerable<Toil> MakeNewToils()
        {
            if (ReadItem != null)
            {
                this.FailOn(() => ReadItem.Destroyed);
            }

            Toil initReadToil = InitRead();
            Toil readToil = this.GetReadToil(Mathf.RoundToInt(300 * ReadDurationMultiplier)).FailOnDestroyedOrNull(ReadItemIndex);
            Toil endReadToil = EndRead();

            if (readingFromInventory)
            {
                yield return Toils_Misc.TakeItemFromInventoryToCarrier(pawn, ReadItemIndex);
            }
            else
            {
                yield return ReserveToil();
                if (ReadSource != null)
                {
                    yield return Toils_Goto.GotoThing(ReadSourceIndex, PathEndMode.Touch).FailOnDespawnedNullOrForbidden(ReadSourceIndex);
                }
                else
                {
                    Toil gotoToPickup = Toils_Goto.GotoThing(ReadItemIndex, PathEndMode.ClosestTouch).FailOnDespawnedNullOrForbidden(ReadItemIndex);
                    yield return Toils_Jump.JumpIf(gotoToPickup, () => pawn.health.capacities.CapableOf(PawnCapacityDefOf.Manipulation));
                    yield return Toils_Goto.GotoThing(ReadItemIndex, PathEndMode.Touch).FailOnDespawnedNullOrForbidden(ReadItemIndex);
                    yield return Toils_Jump.Jump(initReadToil);
                    yield return gotoToPickup;
                }

                yield return PickupReadable();
            }

            yield return CarryReadableToReadSpot(pawn, ReadItemIndex).FailOnDestroyedOrNull(ReadItemIndex);

            yield return Toils_Ingest.FindAdjacentEatSurface(ReadSurfaceIndex, ReadItemIndex);
            yield return initReadToil;
            yield return readToil;         
            yield return Toils_Jump.JumpIf(readToil, () => Find.TickManager.TicksGame < (startTick + job.def.joyDuration) && pawn.needs.joy.CurLevel < 0.99999f);
            yield return endReadToil;

            if (readingFromInventory)
            {
                yield return Toils_Haul.TakeToInventory(ReadItemIndex, ReadItem.stackCount);
            }
            else
            {
                Toil haulJobToil = new Toil();
                haulJobToil.initAction = delegate
                {
                    Job haul = HaulAIUtility.HaulToStorageJob(pawn, ReadItem);

                    pawn.jobs.StartJob(haul, JobCondition.Succeeded);
                };
                haulJobToil.defaultCompleteMode = ToilCompleteMode.Instant;
                yield return haulJobToil;
            }
        }        

        protected virtual void ReadTick(Pawn pawn, Thing scrollSource, SkillDef skill, float extraJoyGainFactor = 1f, JoyTickFullJoyAction fullJoyAction = JoyTickFullJoyAction.EndJob)
        {
            Job curJob = pawn.CurJob;
            if (curJob.def.joyKind == null)
            {
                Log.Warning("This method can only be called for jobs with joyKind.");
                return;
            }
            if (scrollSource != null)
            {
                extraJoyGainFactor *= scrollSource.GetStatValue(ResourceBank.StatDefOf.BookQualityFactor);
            }

            if (skill != null)
            {
                var skillRecord = pawn.skills.GetSkill(skill);

                skillRecord.Learn(2 * extraJoyGainFactor * curJob.def.joyXpPerTick / (skillRecord.levelInt + 1));
            }

            if(pawn.CurJob.GetTarget(ReadSurfaceIndex).IsValid)
                pawn.rotationTracker.FaceCell(pawn.CurJob.GetTarget(ReadSurfaceIndex).Cell);

            pawn.needs.joy.GainJoy(extraJoyGainFactor * curJob.def.joyGainRate * 0.36f / 2500f, curJob.def.joyKind);
            if (curJob.def.joySkill != null)
            {
                pawn.skills.GetSkill(curJob.def.joySkill).Learn(curJob.def.joyXpPerTick);
            }
            if (!curJob.ignoreJoyTimeAssignment && !pawn.GetTimeAssignment().allowJoy)
            {                                                                    
                Job haul = HaulAIUtility.HaulToStorageJob(pawn, ReadItem);

                pawn.jobs.StartJob(haul, JobCondition.InterruptForced);
            }
            if (pawn.needs.joy.CurLevel > 0.9999f)
            {
                switch (fullJoyAction)
                {
                    case JoyTickFullJoyAction.EndJob:
                        pawn.jobs.curDriver.EndJobWith(JobCondition.Succeeded);
                        break;
                    case JoyTickFullJoyAction.GoToNextToil:
                        pawn.jobs.curDriver.ReadyForNextToil();
                        break;
                }
            }
        }

        public static Toil CarryReadableToReadSpot(Pawn pawn, TargetIndex readableIndex)
        {
            Toil toil = new Toil();
            toil.initAction = delegate
            {
                Pawn actor = toil.actor;
                IntVec3 intVec = IntVec3.Invalid;
                Thing thing = null;
                Thing thing2 = actor.CurJob.GetTarget(readableIndex).Thing;
                Predicate<Thing> baseChairValidator = delegate (Thing t)
                {
                    if (t.def.building == null || !t.def.building.isSittable)
                    {
                        return false;
                    }
                    if (t.IsForbidden(pawn))
                    {
                        return false;
                    }
                    if (!actor.CanReserve(t))
                    {
                        return false;
                    }
                    if (!t.IsSociallyProper(actor))
                    {
                        return false;
                    }
                    if (t.IsBurning())
                    {
                        return false;
                    }
                    if (t.HostileTo(pawn))
                    {
                        return false;
                    }
                    bool flag = false;

                    IntVec3 c = t.Position + t.Rotation.FacingCell;
                    Thing readLocThing = c.GetEdifice(t.Map);

                    if (readLocThing != null && readLocThing.def.passability != Traversability.Impassable)
                    {
                        flag = true;
                    }
                    else if (t.Map.thingGrid.ThingAt(c, ThingCategory.Building) != null)
                    {
                        flag = true;
                    }

                    return flag;
                };
                if (thing2.GetStatValue(ResourceBank.StatDefOf.BookChairSearchRadius) > 0f)
                {
                    thing = GenClosest.ClosestThingReachable(actor.Position, actor.Map, ThingRequest.ForGroup(ThingRequestGroup.BuildingArtificial), PathEndMode.OnCell, TraverseParms.For(actor), thing2.GetStatValue(ResourceBank.StatDefOf.BookChairSearchRadius), (Thing t) => baseChairValidator(t) && t.Position.GetDangerFor(pawn, t.Map) == Danger.None);
                }
                if (thing == null)
                {
                    intVec = RCellFinder.SpotToChewStandingNear(actor, actor.CurJob.GetTarget(readableIndex).Thing);
                    Danger chewSpotDanger = intVec.GetDangerFor(pawn, actor.Map);
                    if (chewSpotDanger != Danger.None)
                    {
                        thing = GenClosest.ClosestThingReachable(actor.Position, actor.Map, ThingRequest.ForGroup(ThingRequestGroup.BuildingArtificial), PathEndMode.OnCell, TraverseParms.For(actor), thing2.GetStatValue(ResourceBank.StatDefOf.BookChairSearchRadius), (Thing t) => baseChairValidator(t) && (int)t.Position.GetDangerFor(pawn, t.Map) <= (int)chewSpotDanger);
                    }
                }
                if (thing != null)
                {
                    intVec = thing.Position;
                    actor.Reserve(thing, actor.CurJob);
                }

                actor.Map.pawnDestinationReservationManager.Reserve(actor, actor.CurJob, intVec);
                actor.pather.StartPath(intVec, PathEndMode.OnCell);
            };
            toil.defaultCompleteMode = ToilCompleteMode.PatherArrival;
            return toil;
        }
    }
}
