﻿using RimWorld;
using System.Collections.Generic;
using System.Text;
using UnityEngine;
using Verse;

namespace MedTimes
{
    class Building_CheesePress : Building
    {
        
        public const int MaxCapacity = 40;
        private int milkCount;

        private const int BaseFermentationDuration = 420000;
        
        private float progressInt;
        private const float ProgressPerTick = 1.0f / BaseFermentationDuration;
        private float ProgressPerTickAtCurrentTemp => ProgressPerTick * CurrentTempProgressSpeedFactor;
        private int EstimatedTicksLeft => Mathf.Max(Mathf.RoundToInt((1f - Progress) / ProgressPerTickAtCurrentTemp), 0);
        
        public const float MinIdealTemperature = 15f;
        private bool Empty => milkCount <= 0;
        public bool Coagulated => !Empty && Progress >= 1f;

        public float Progress
        {
            get
            {
                return progressInt;
            }
            set
            {
                if (value != progressInt)
                {
                    progressInt = value;
                }
            }
        }

        public int SpaceLeftForMilk
        {
            get
            {
                if (Coagulated)
                {
                    return 0;
                }
                return MaxCapacity - milkCount;
            }
        }

        private float CurrentTempProgressSpeedFactor
        {
            get
            {
                CompProperties_TemperatureRuinable compProperties = def.GetCompProperties<CompProperties_TemperatureRuinable>();
                float ambientTemperature = base.AmbientTemperature;
                if (ambientTemperature < compProperties.minSafeTemperature)
                {
                    return 0.1f;
                }
                if (ambientTemperature < MinIdealTemperature)
                {
                    return GenMath.LerpDouble(compProperties.minSafeTemperature, MinIdealTemperature, 0.1f, 1f, ambientTemperature);
                }
                return 1f;
            }
        }
        
        public override void ExposeData()
        {
            base.ExposeData();
            Scribe_Values.Look(ref milkCount, "milkCount", 0);
            Scribe_Values.Look(ref progressInt, "progress", 0f);
        }

        public override void TickRare()
        {
            base.TickRare();
            if (!Empty)
            {
                Progress = Mathf.Min(Progress + 250f * ProgressPerTickAtCurrentTemp, 1f);
            }
        }

        public void AddMilk(int count)
        {
            GetComp<CompTemperatureRuinable>().Reset();
            if (Coagulated)
            {
                Log.Warning("Tried to add milk to a full cheese press. Colonists should take the cheese first.");
                return;
            }
            int num = Mathf.Min(count, MaxCapacity - milkCount);
            if (num > 0)
            {
                Progress = GenMath.WeightedAverage(0f, num, Progress, milkCount);
                milkCount += num;
            }
        }

        protected override void ReceiveCompSignal(string signal)
        {
            if (signal == "RuinedByTemperature")
            {
                Reset();
            }
        }

        private void Reset()
        {
            milkCount = 0;
            Progress = 0f;
        }

        public void AddMilk(Thing milk)
        {
            int num = Mathf.Min(milk.stackCount, MaxCapacity - milkCount);
            if (num > 0)
            {
                AddMilk(num);
                milk.SplitOff(num).Destroy();
            }
        }

        public override string GetInspectString()
        {
            StringBuilder stringBuilder = new StringBuilder();
            stringBuilder.Append(base.GetInspectString());
            if (stringBuilder.Length != 0)
            {
                stringBuilder.AppendLine();
            }
            CompTemperatureRuinable comp = GetComp<CompTemperatureRuinable>();
            if (!Empty && !comp.Ruined)
            {
                if (Coagulated)
                {
                    stringBuilder.AppendLine(ResourceBank.Strings.ContainsCheese.Translate(milkCount, MaxCapacity));
                }
                else
                {
                    stringBuilder.AppendLine(ResourceBank.Strings.ContainsMilk.Translate(milkCount, MaxCapacity));
                }
            }
            if (!Empty)
            {
                if (Coagulated)
                {
                    stringBuilder.AppendLine(ResourceBank.Strings.Coagulated.Translate());
                }
                else
                {
                    stringBuilder.AppendLine(ResourceBank.Strings.CoagulationProgress.Translate(Progress.ToStringPercent(), EstimatedTicksLeft.ToStringTicksToPeriod()));
                    if (CurrentTempProgressSpeedFactor != 1f)
                    {
                        stringBuilder.AppendLine(ResourceBank.Strings.CheesePressOutOfIdealTemperature.Translate(CurrentTempProgressSpeedFactor.ToStringPercent()));
                    }
                }
            }
            stringBuilder.AppendLine(ResourceBank.Strings.Temperature.Translate() + ": " + base.AmbientTemperature.ToStringTemperature("F0"));
            stringBuilder.AppendLine(ResourceBank.Strings.IdealCoagulationTemperature.Translate() + ": " + MinIdealTemperature.ToStringTemperature("F0") + " ~ " + comp.Props.maxSafeTemperature.ToStringTemperature("F0"));
            return stringBuilder.ToString().TrimEndNewlines();
        }

        public Thing TakeOutCheese()
        {
            if (!Coagulated)
            {
                Log.Warning("Tried to get cheese, but its not yet ready.");
                return null;
            }
            Thing thing = ThingMaker.MakeThing(ResourceBank.ThingDefOf.MedTimes_MealCheese);
            thing.stackCount = milkCount;
            Reset();
            return thing;
        }

        public override IEnumerable<Gizmo> GetGizmos()
        {
            foreach (Gizmo gizmo in base.GetGizmos())
            {
                yield return gizmo;
            }
            if (Prefs.DevMode && !Empty)
            {
                yield return new Command_Action
                {
                    defaultLabel = "Debug: Set progress to 1",
                    action = delegate
                    {
                        Progress = 1f;
                    }
                };
            }
        }
    }
}
