﻿using RimWorld;
using RimWorld.Planet;
using System;
using System.Collections.Generic;
using System.Text;
using UnityEngine;
using Verse;
using Verse.Sound;

namespace MedTimes
{
    class Building_TrapRearmable : Building
    {                                           
        protected static readonly FloatRange DamageRandomFactorRange = new FloatRange(0.8f, 1.2f);

        protected static readonly float DamageCount = 5f; 

        private List<Pawn> touchingPawns = new List<Pawn>();

        protected const float KnowerSpringChanceFactorSameFaction = 0.005f;     
        protected const float KnowerSpringChanceFactorWildAnimal = 0.2f;    
        protected const float KnowerSpringChanceFactorFactionlessHuman = 0.3f;  
        protected const float KnowerSpringChanceFactorOther = 0f;  
        protected const ushort KnowerPathFindCost = 800;      
        protected const ushort KnowerPathWalkCost = 40;

        private Graphic cachedGraphicSprungTrap;

        private bool armed;

        private bool autoRearm;      



        private bool CanSetAutoRearm => base.Faction == Faction.OfPlayer && def.blueprintDef != null && def.IsResearchFinished;

        public override Graphic Graphic {                                       
            get
            {
                if (!armed)
                {
                    if (def.building.trapUnarmedGraphicData == null)
                    {
                        return base.Graphic;
                    }
                    if (cachedGraphicSprungTrap == null)
                    {
                        cachedGraphicSprungTrap = def.building.trapUnarmedGraphicData.GraphicColoredFor(this);
                    }
                    return cachedGraphicSprungTrap;
                }
                return base.Graphic;
            }
        }

        public bool Armed
        {
            get => armed;
            set
            {
                if (armed != value)
                {
                    armed = value;
                    SoundDefOf.TrapArm.PlayOneShot(new TargetInfo(base.Position, this.Map));
                    this.Map.mapDrawer.MapMeshDirty(Position, MapMeshFlag.Buildings, regenAdjacentCells: false, regenAdjacentSections: false);
                }
            }
        }

        public override void ExposeData()
        {
            base.ExposeData();
            Scribe_Values.Look(ref autoRearm, "autoRearm", defaultValue: false);
            Scribe_Values.Look(ref armed, "IsTrapArmed", defaultValue: true);
            Scribe_Collections.Look(ref touchingPawns, "testees", LookMode.Reference);
        }

        public override void SpawnSetup(Map map, bool respawningAfterLoad)
        {
            base.SpawnSetup(map, respawningAfterLoad);   
            
            if (!respawningAfterLoad)
            {
                armed = true;
                autoRearm = (CanSetAutoRearm && map.areaManager.Home[base.Position]);
                SoundDefOf.TrapArm.PlayOneShot(new TargetInfo(base.Position, map));
            }
        }    

        public override void Tick()
        {
            if (base.Spawned)
            {
                List<Thing> thingList = base.Position.GetThingList(base.Map);
                for (int i = 0; i < thingList.Count; i++)
                {
                    if (thingList[i] is Pawn pawn && !touchingPawns.Contains(pawn))
                    {
                        touchingPawns.Add(pawn);
                        CheckSpring(pawn);
                    }
                }
                for (int j = 0; j < touchingPawns.Count; j++)
                {
                    Pawn pawn2 = touchingPawns[j];
                    if (!pawn2.Spawned || pawn2.Position != base.Position)
                    {
                        touchingPawns.Remove(pawn2);
                    }
                }
            }
            base.Tick();
        }

        private void CheckSpring(Pawn p)
        {
            if (Rand.Chance(SpringChance(p)))
            {
                Map map = base.Map;
                Spring(p);
                if (p.Faction == Faction.OfPlayer || p.HostFaction == Faction.OfPlayer)
                {
                    Find.LetterStack.ReceiveLetter("LetterFriendlyTrapSprungLabel".Translate(p.LabelShort, p), "LetterFriendlyTrapSprung".Translate(p.LabelShort, p), LetterDefOf.NegativeEvent, new TargetInfo(base.Position, map));
                }
            }
        }         

        protected virtual float SpringChance(Pawn p)
        {
            if (Armed)
            {
                float num = 1f;
                if (KnowsOfTrap(p))
                {
                    if (p.Faction != null)
                    {
                        num = ((p.Faction != base.Faction) ? KnowerSpringChanceFactorOther : KnowerSpringChanceFactorSameFaction);
                    }
                    else if (p.RaceProps.Animal)
                    {
                        num = KnowerSpringChanceFactorWildAnimal;
                        num *= def.building.trapPeacefulWildAnimalsSpringChanceFactor;
                    }
                    else
                    {
                        num = KnowerSpringChanceFactorFactionlessHuman;
                    }
                }
                num *= this.GetStatValue(StatDefOf.TrapSpringChance) * p.GetStatValue(StatDefOf.PawnTrapSpringChance);
                return Mathf.Clamp01(num);
            }
            return 0f;  
        }

        public virtual bool KnowsOfTrap(Pawn p)
        {
            if (p.Faction != null && !p.Faction.HostileTo(base.Faction))
            {
                return true;
            }
            if (p.Faction == null && p.RaceProps.Animal && !p.InAggroMentalState)
            {
                return true;
            }
            if (p.guest != null && p.guest.Released)
            {
                return true;
            }
            if (!p.IsPrisoner && base.Faction != null && p.HostFaction == base.Faction)
            {
                return true;
            }
            if (p.RaceProps.Humanlike && p.IsFormingCaravan())
            {
                return true;
            }
            if (p.IsPrisoner && p.guest.ShouldWaitInsteadOfEscaping && base.Faction == p.HostFaction)
            {
                return true;
            }
            if (p.Faction == null && p.RaceProps.Humanlike)
            {
                return true;
            }
            return false;
        }

        public override ushort PathFindCostFor(Pawn p)
        {
            if (!KnowsOfTrap(p))
            {
                return 0;
            }
            return KnowerPathFindCost;
        }

        public override ushort PathWalkCostFor(Pawn p)
        {
            if (!KnowsOfTrap(p))
            {
                return 0;
            }
            return KnowerPathWalkCost;
        }

        public override bool IsDangerousFor(Pawn p)
        {
            return KnowsOfTrap(p);
        }

        public void Spring(Pawn p)
        {
            bool spawned = base.Spawned;
            Map map = base.Map;
            SpringSub(p);

            Armed = false;    

            if (autoRearm)
            {    
                if (spawned)
                {
                    UpdateRearmDesignation();
                }
            }
        }

        protected virtual void SpringSub(Pawn p)
        {
            SoundDefOf.TrapSpring.PlayOneShot(new TargetInfo(base.Position, base.Map));
            if (p == null)
            {
                return;
            }
            float num = this.GetStatValue(StatDefOf.TrapMeleeDamage) * DamageRandomFactorRange.RandomInRange;
            float num2 = num / DamageCount;
            float armorPenetration = num2 * 0.015f;
            for (int i = 0; (float)i < DamageCount; i++)
            {
                DamageInfo dinfo = new DamageInfo(DamageDefOf.Stab, num2, armorPenetration, -1f, this);
                DamageWorker.DamageResult damageResult = p.TakeDamage(dinfo);
                if (i == 0)
                {
                    BattleLogEntry_DamageTaken battleLogEntry_DamageTaken = new BattleLogEntry_DamageTaken(p, RulePackDefOf.DamageEvent_TrapSpike);
                    Find.BattleLog.Add(battleLogEntry_DamageTaken);
                    damageResult.AssociateWithLog(battleLogEntry_DamageTaken);
                }
            }
        }    

        public override IEnumerable<Gizmo> GetGizmos()
        {
            foreach (Gizmo gizmo in base.GetGizmos())
            {
                yield return gizmo;
            }                  
            yield return new Command_Toggle
            {
                defaultLabel = ResourceBank.Strings.CommandAutoRearm.Translate(),
                defaultDesc = ResourceBank.Strings.CommandAutoRearmDesc.Translate(),
                hotKey = KeyBindingDefOf.Misc3,
                icon = TexCommand.RearmTrap,
                isActive = (() => autoRearm),
                toggleAction = delegate
                {
                    autoRearm = !autoRearm;
                    UpdateRearmDesignation();
                }
            };        
        }

        private void UpdateRearmDesignation()
        {
            if(autoRearm && !Armed && base.Map.designationManager.DesignationOn(this, ResourceBank.DesignationDefOf.Designtaion_RearmTrap) == null)
            {     
                base.Map.designationManager.AddDesignation(new Designation(this, ResourceBank.DesignationDefOf.Designtaion_RearmTrap));
            }
        }
    }
}
