module TTFCore

import TTFConfig.*
import TTFSettings.*



public class TTF extends ScriptableSystem {

	private let interfaceLanguage: CName;
	private let aspectValue: Int32;
	public let settings: ref<TTFSettings>;
	private let ttfData: ref<TTFData>;

	public static func GetInstance() -> ref<TTF> {
		return GameInstance.GetScriptableSystemsContainer(GetGameInstance()).Get(n"TTFCore.TTF") as TTF;
	}

	private func OnAttach() -> Void {
		this.RefreshSystemSettings();
		this.RefreshRuntimeSettings();
	}

	public final func RefreshRuntimeSettings() -> Void {
		this.settings = new TTFSettings();
	}

	public final func RefreshSystemSettings() -> Void {
		let userSettings: ref<UserSettings> = GameInstance.GetSettingsSystem(this.GetGameInstance());
		this.interfaceLanguage = (userSettings.GetVar(n"/language", n"OnScreen") as ConfigVarListName).GetValue();
		let screenResolution: String = (userSettings.GetVar(n"/video/display", n"Resolution") as ConfigVarListString).GetValue();
		this.aspectValue = this.CalculateAspectValue(screenResolution);
	}

	private final func CalculateAspectValue(resolution: String) -> Int32 {
		let arr: array<String> = StrSplit(resolution, "x", false);
		if ArraySize(arr) != 2 {
			return 0;
		}
		let w: Int32 = StringToInt(arr[0], 0);
		let h: Int32 = StringToInt(arr[1], 0);
		return (h != 0) ? (w * 100 / h) : 0;
	}

	private final func FlushTTFData() -> Void {
		this.ttfData = null;
	}

	public final func SetTooltipData(tooltipController: wref<AGenericTooltipController>) -> Void {
		let controller: wref<ItemTooltipCommonController> = tooltipController as ItemTooltipCommonController;

		if !IsDefined(controller) && !IsDefined(tooltipController as CyberdeckTooltip) {
			this.FlushTTFData();
			return;
		}

		let isWeapon: Bool = false;
		let isClothing: Bool = false;

		let mitd: ref<MinimalItemTooltipData> = controller.m_data;
		if IsDefined(mitd) {
			isWeapon = Equals(mitd.itemCategory, gamedataItemCategory.Weapon);
			isClothing = Equals(mitd.itemCategory, gamedataItemCategory.Clothing) && !Equals(mitd.equipmentArea, gamedataEquipmentArea.Invalid);
		}

		let uiii: ref<UIInventoryItem> = controller.m_itemData;
		if IsDefined(uiii) {
			isWeapon = uiii.IsWeapon();
			isClothing = uiii.IsClothing() && !Equals(uiii.GetEquipmentArea(), gamedataEquipmentArea.Invalid);
		}

		let isNewData: Bool = false;

		if !IsDefined(this.ttfData) {
			this.ttfData = new TTFData();
			isNewData = true;
		}

		this.ttfData.isWeapon = isWeapon;
		this.ttfData.isClothing = isClothing;

		if !(isWeapon || isClothing) {
			return;
		}

		let newTotalLinesCount: Int32 = 0;
		newTotalLinesCount += this.GetTotalLinesCount(controller.m_data);
		newTotalLinesCount += this.GetTotalLinesCount(controller.m_itemData);

		if isNewData || (!isNewData && (newTotalLinesCount > this.ttfData.totalLinesCount)) {
			this.ttfData.totalLinesCount = newTotalLinesCount;
		}
	}

	public final func UpdateTooltipsContainer(container: inkWidgetRef) -> Void {
		let newScale: Float;
		let newVShift: Float;

		if IsDefined(this.ttfData) {
			let scaleLevel: ScaleLevel = (this.ttfData.isWeapon || this.ttfData.isClothing) ? TTFConfig.GetScaleLevel(this.ttfData.totalLinesCount + this.settings.customLimitOffset) : ScaleLevel.Default;
			let scaleBase: Float = Cast<Float>(EnumInt(scaleLevel)) / 1000.0;
			let scaleMultiplier: Float = Equals(scaleLevel, ScaleLevel.Default) ? 1.0 : this.settings.customScaleMultiplier;
			let scaleOffset: Float = this.ttfData.isWeapon ? this.settings.customScaleOffset : 0.0;

			if this.settings.useCorrection && !this.settings.useRescale {
				scaleLevel = ScaleLevel.Default;
			}

			newScale = this.settings.useRescale ? (scaleBase * scaleMultiplier + scaleOffset) : 1.0;
			newVShift = this.settings.useCorrection ? TTFConfig.GetVShift(scaleLevel, this.aspectValue) : 0.0;

			this.FlushTTFData();
		} else {
			newScale = 1.0;
			newVShift = 0.0;
		}

		inkWidgetRef.SetScale(container, new Vector2(newScale, newScale));
		inkWidgetRef.SetPadding(container, new inkMargin(0.0, 0.0, 0.0, newVShift * -1.0));
	}

	private final func GetTotalLinesCount(mitd: ref<MinimalItemTooltipData>) -> Int32 {
		if !IsDefined(mitd) {
			return 0;
		}
		let totalLinesCount: Int32 = 0;
		totalLinesCount += this.GetEvoLinesCount(mitd.itemEvolution);
		totalLinesCount += this.GetMITDReqLinesCount(mitd.requirements);
		totalLinesCount += this.GetMITDStatsLinesCount(mitd.stats);
		totalLinesCount += this.GetMITDModsLinesCount(mitd.mods);
		totalLinesCount += this.GetMITDDedicatedModsLinesCount(mitd.dedicatedMods);
		totalLinesCount += this.GetDescLinesCount(mitd.description);
		return totalLinesCount;
	}

	private final func GetEvoLinesCount(evo: gamedataWeaponEvolution) -> Int32 {
		if this.settings.hideEvoDescription {
			return 0;
		}

		let evoLinesCharLimit: Int32 = TTFConfig.GetMagicNumber(this.interfaceLanguage, MagicIndex.EVO);
		let text: String;

		switch evo {
		  case gamedataWeaponEvolution.Power: text = "LocKey#54117"; break;
		  case gamedataWeaponEvolution.Smart: text = "LocKey#54120"; break;
		  case gamedataWeaponEvolution.Tech: text  = "LocKey#54122"; break;
		  case gamedataWeaponEvolution.Blunt: text = "LocKey#77969"; break;
		  case gamedataWeaponEvolution.Blade: text = "LocKey#77960"; break;
		  default: return 0;
		}
		return this.GetLocalizedLinesCount(text, evoLinesCharLimit);
	}

	private final func GetMITDReqLinesCount(req: ref<MinimalItemTooltipDataRequirements>) -> Int32 {
		let reqLinesCharLimit: Int32 = TTFConfig.GetMagicNumber(this.interfaceLanguage, MagicIndex.REQ);
		let reqLinesCount: Int32 = 0;
		
		if req.isLevelRequirementNotMet {
			reqLinesCount += 1;
		}
		if req.isSmartlinkRequirementNotMet {
			reqLinesCount += this.GetLocalizedLinesCount("LocKey#34995", reqLinesCharLimit);
		}
		if req.isStrengthRequirementNotMet || req.isReflexRequirementNotMet {
			reqLinesCount += 2;
		}
		if req.isPerkRequirementNotMet {
			reqLinesCount += 1;
		}
		if req.isRarityRequirementNotMet {
			reqLinesCount += this.GetLocalizedLinesCount("LocKey#79881", reqLinesCharLimit);
		}
		return reqLinesCount;
	}

	private final func GetMITDStatsLinesCount(stats: array<ref<MinimalItemTooltipStatData>>) -> Int32 {
		let statsLinesCharLimit: Int32 = TTFConfig.GetMagicNumber(this.interfaceLanguage, MagicIndex.STATS);
		let statsLinesCount: Int32 = 0;

		for stat in stats {
			statsLinesCount += this.GetLocalizedLinesCount(stat.statName, statsLinesCharLimit);
		}
		return statsLinesCount;
	}

	private final func GetMITDModsLinesCount(mods: array<ref<MinimalItemTooltipModData>>) -> Int32 {
		let modsLinesCharLimit: Int32 = TTFConfig.GetMagicNumber(this.interfaceLanguage, MagicIndex.MODS);
		let modsLinesCount: Int32 = 0;
		
		for mod in mods {
			let attachment: ref<MinimalItemTooltipModAttachmentData> = mod as MinimalItemTooltipModAttachmentData;
			if IsDefined(attachment) {
				if attachment.abilitiesSize == 0 {
					modsLinesCount += this.GetLocalizedLinesCount(attachment.slotName, modsLinesCharLimit);
				} else {
					modsLinesCount += this.GetMITDAbilitiesLinesCount(attachment.abilities, false);
				}
			}
		}
		return modsLinesCount;
	}

	private final func GetMITDDedicatedModsLinesCount(dMods: array<ref<MinimalItemTooltipModAttachmentData>>) -> Int32 {
		let dModsLinesCount: Int32 = 0;
		
		for attachment in dMods {
			dModsLinesCount += this.GetMITDAbilitiesLinesCount(attachment.abilities, true);
		}
		return dModsLinesCount;
	}

	private final func GetMITDAbilitiesLinesCount(abilities: array<InventoryItemAbility>, isDedicated: Bool) -> Int32 {
		let magicIndex: MagicIndex = isDedicated ? MagicIndex.DMODS : MagicIndex.MODS;
		let modsLinesCharLimit: Int32 = TTFConfig.GetMagicNumber(this.interfaceLanguage, magicIndex);
		let abilitiesLinesCount: Int32 = 0;

		for ability in abilities {
			abilitiesLinesCount += this.GetLocalizedLinesCount(ability.Description, modsLinesCharLimit);
		}
		return abilitiesLinesCount;
	}

	private final func GetDescLinesCount(text: String) -> Int32 {
		let descLinesCharLimit: Int32 = TTFConfig.GetMagicNumber(this.interfaceLanguage, MagicIndex.DESC);
		return this.GetLocalizedLinesCount(text, descLinesCharLimit);
	}

	private final func GetTotalLinesCount(uiii: ref<UIInventoryItem>) -> Int32 {
		if !IsDefined(uiii) {
			return 0;
		}
		let totalLinesCount: Int32 = 0;
		totalLinesCount += this.GetEvoLinesCount(uiii.GetWeaponEvolution());
		totalLinesCount += this.GetUIIIReqLinesCount(uiii);
		totalLinesCount += this.GetUIIIStatsLinesCount(uiii.GetStatsManager());
		totalLinesCount += this.GetUIIIModsLinesCount(uiii.GetModsManager());
		totalLinesCount += this.GetDescLinesCount(uiii.GetDescription());
		totalLinesCount += this.GetUIIICraftedLinesCount(uiii.IsCrafted());
		return totalLinesCount;
	}

	private final func GetUIIIReqLinesCount(uiii: ref<UIInventoryItem>) -> Int32 {
		let player: ref<PlayerPuppet> = (GameInstance.GetPlayerSystem(GetGameInstance()).GetLocalPlayerMainGameObject() as PlayerPuppet);

		if !IsDefined(player) {
			return 0;
		}

		let reqLinesCharLimit: Int32 = TTFConfig.GetMagicNumber(this.interfaceLanguage, MagicIndex.REQ);
		let reqLinesCount: Int32 = 0;
		let manager: ref<UIInventoryItemRequirementsManager> = uiii.GetRequirementsManager(player);
		
		if !manager.IsLevelRequirementMet() {
			reqLinesCount += 1;
		}
		if !manager.IsSmartlinkRequirementMet() {
			reqLinesCount += this.GetLocalizedLinesCount("LocKey#34995", reqLinesCharLimit);
		}
		if !manager.IsStrengthRequirementMet() || !manager.IsReflexRequirementMet() {
			reqLinesCount += 2;
		}
		if !manager.IsPerkRequirementMet() {
			reqLinesCount += 1;
		}
		if !manager.IsRarityRequirementMet(uiii) {
			reqLinesCount += this.GetLocalizedLinesCount("LocKey#79881", reqLinesCharLimit);
		}
		return reqLinesCount;
	}

	private final func GetUIIIStatsLinesCount(manager: ref<UIInventoryItemStatsManager>) -> Int32 {
		let statsLinesCharLimit: Int32 = TTFConfig.GetMagicNumber(this.interfaceLanguage, MagicIndex.STATS);
		let statsLinesCount: Int32 = 0;

		for stat in manager.Stats {
			statsLinesCount += this.GetLocalizedLinesCount(stat.Properties.GetName(), statsLinesCharLimit);
		}
		return statsLinesCount;
	}

	private final func GetUIIIModsLinesCount(manager: ref<UIInventoryItemModsManager>) -> Int32 {
		let modsLinesCount: Int32 = 0;
		
		modsLinesCount += this.GetUIIIAbilitiesLinesCount(manager, false);
		modsLinesCount += this.GetUIIIAbilitiesLinesCount(manager, true);
		return modsLinesCount;
	}

	private final func GetUIIIAbilitiesLinesCount(manager: ref<UIInventoryItemModsManager>, isDedicated: Bool) -> Int32 {
		let magicIndex: MagicIndex = isDedicated ? MagicIndex.DMODS : MagicIndex.MODS;
		let modsLinesCharLimit: Int32 = TTFConfig.GetMagicNumber(this.interfaceLanguage, magicIndex);
		let abilitiesLinesCount: Int32 = 0;

		let modsCount: Int32 = isDedicated ? manager.GetDedicatedModsSize() : manager.GetModsSize();
		let i: Int32 = 0;
		while i < modsCount {
			let mod: ref<UIInventoryItemModAttachment> = (isDedicated ? manager.GetDedicatedMod(i) : manager.GetMod(i)) as UIInventoryItemModAttachment;
			if IsDefined(mod) {
				let abilitiesCount: Int32 = mod.AbilitiesSize;
				if !isDedicated && mod.AbilitiesSize <= 0 {
					abilitiesLinesCount += this.GetLocalizedLinesCount(mod.SlotName, modsLinesCharLimit);
				}
				for ability in mod.Abilities {
					abilitiesLinesCount += this.GetLocalizedLinesCount(ability.Description, modsLinesCharLimit);
				}
			}
			i += 1;
		}
		return abilitiesLinesCount;
	}

	private final func GetUIIICraftedLinesCount(isCrafted: Bool) -> Int32 {
		return (this.settings.replaceCrafted || !isCrafted) ? 0 : 2;
	}

	private final func GetLocalizedLinesCount(key: String, limit: Int32) -> Int32 {
		let text: String = ToString(GetLocalizedText(key));
		let variablesCorrectionOffset: Int32 = 24;
		let precisionMultiplier: Int32 = 4;
		let textModLen: Int32 = StrLen(text) * precisionMultiplier;
		let modLimit: Int32 = limit + (StrContains(text, "{") ? variablesCorrectionOffset : 0);
		return (textModLen / modLimit) + 1;
	}
}

class TTFData {
	public let totalLinesCount: Int32 = 0;
	public let isWeapon: Bool = false;
	public let isClothing: Bool = false;
}

@wrapMethod(SettingsMainGameController)
private final func OnApplyButton() -> Void {
	wrappedMethod();
	(GameInstance.GetScriptableSystemsContainer(GetGameInstance()).Get(n"TTFCore.TTF") as TTF).RefreshSystemSettings();
}

@wrapMethod(PauseMenuBackgroundGameController)
protected cb func OnUninitialize() -> Bool {
	(GameInstance.GetScriptableSystemsContainer(GetGameInstance()).Get(n"TTFCore.TTF") as TTF).RefreshRuntimeSettings();
	wrappedMethod();
}
