local HigherCamera = { version = "1.0.0" }
local Config = require("Modules/Config")

local enabled = true
local isTiltEnabled = true

local isInVehicle = false

function TiltCamera()
    Game.GetPlayer():GetFPPCameraComponent():SetLocalOrientation(Quaternion.new((-0.06 * Config.data.tiltMult), 0.0, 0.0, 1.0))
end

function ResetTilt()
    Game.GetPlayer():GetFPPCameraComponent():SetLocalOrientation(Quaternion.new(0.00, 0.0, 0.0, 1.0))
end

function RaiseCamera()
    Game.GetPlayer():GetFPPCameraComponent():SetLocalPosition(Vector4.new(0.0, -(0.02 * Config.data.zMult), (0.09 * Config.data.yMult), 1.0))
end
function ResetCamera()
    Game.GetPlayer():GetFPPCameraComponent():SetLocalPosition(Vector4.new(0.0, 0, 0, 1.0))
end

function OnVehicleEnter()
    if not enabled then
        ResetCamera()
        ResetTilt()
        return
    end

    if isTiltEnabled then
        TiltCamera()
    end
    RaiseCamera()

    isInVehicle = true
end

function OnVehicleExit()
    ResetCamera()
    ResetTilt()

    isInVehicle = false
end

function RefreshCameraIfNeeded()
    Config.SaveConfig()
    if isInVehicle then
        OnVehicleEnter()
    end
end

local presets = {
    { 1.150, 0.9, -0.2 },
    { 1.050, 0.8, -3.410 },
    { 0.950, 0.610, -10 },
}

function HigherCamera:New()
    registerForEvent("onInit", function()
        Config.InitConfig()

        Observe('hudCarController', 'RegisterToVehicle', function(self, registered)
            print("registered", registered)
            if registered then
                OnVehicleEnter()
            else
                OnVehicleExit()
            end
        end)
        Observe('hudCarController', 'OnCameraModeChanged', function(mode, self)
            if mode then
                OnVehicleExit()
            else
                OnVehicleEnter()
            end
        end)

        -- Fires with loaded save file too
        Observe('hudCarController', 'OnPlayerAttach', function()
            -- TODO: remove?
        end)

        -- Fires when execting
        Observe('hudCarController', 'OnUnmountingEvent', function()
            OnVehicleExit()
        end)
        Observe('RadialWheelController', 'RegisterBlackboards', function(_, loaded)
            if not loaded then
                isInVehicle = false
                OnVehicleExit()
            end
        end)

    end)

    registerForEvent("onOverlayOpen", function() isOverlayOpen = true end)
    registerForEvent("onOverlayClose", function() isOverlayOpen = false end)

    registerHotkey("VehicleFPPCameraEnabled", "Toggle Enhanced Vehicle Camera Enabled", function()
        if isInVehicle then
            enabled = not enabled
            RefreshCameraIfNeeded()
        end
    end)

    function RestrictMult(val)
        return math.max(0, math.min(3, val))
    end

    registerForEvent("onDraw", function()
        if not isOverlayOpen or not Config or not Config.isReady then
            return
        end

        ImGui.PushStyleVar(ImGuiStyleVar.WindowMinSize, 300, 200)
        ImGui.PushStyleColor(ImGuiCol.Border, 0, 0, 0, 0)
        ImGui.PushStyleColor(ImGuiCol.TitleBg, 0, 0, 0, 0.8)
        ImGui.PushStyleColor(ImGuiCol.TitleBgActive, 0, 0, 0, 0.8)
        ImGui.PushStyleColor(ImGuiCol.WindowBg, 0, 0, 0, 0.8)
        ImGui.PushStyleColor(ImGuiCol.Button, 0.25, 0.35, 0.45, 0.8)
        ImGui.PushStyleColor(ImGuiCol.ButtonHovered, 0.35, 0.45, 0.55, 1.0)
        ImGui.PushStyleColor(ImGuiCol.ButtonActive, 0.25, 0.35, 0.45, 0.5)
        ImGui.PushStyleColor(ImGuiCol.FrameBg, 0.25, 0.35, 0.45, 0.8)

        ImGui.Begin("VehicleFPPCamera", ImGuiWindowFlags.AlwaysAutoResize)
        ImGui.SetWindowFontScale(1)


        -- toggle enabled
        enabled, toggleEnabled = ImGui.Checkbox("Enabled", enabled)
        if toggleEnabled then
            RefreshCameraIfNeeded()
        end

        -- Presets
        ImGui.Text("Presets: ")
        ImGui.SameLine()
        if ImGui.SmallButton(" 1 ") then
            Config.data.tiltMult = presets[1][1]
            Config.data.yMult = presets[1][2]
            Config.data.zMult = presets[1][3]
            RefreshCameraIfNeeded()
        end
        ImGui.SameLine()
        if ImGui.SmallButton(" 2 ") then
            Config.data.tiltMult = presets[2][1]
            Config.data.yMult = presets[2][2]
            Config.data.zMult = presets[2][3]
            RefreshCameraIfNeeded()
        end
        ImGui.SameLine()
        if ImGui.SmallButton(" 3 ") then
            Config.data.tiltMult = presets[3][1]
            Config.data.yMult = presets[3][2]
            Config.data.zMult = presets[3][3]
            RefreshCameraIfNeeded()
        end


        -- Tilt controll
        Config.data.tiltMult, tiltMultUsed = ImGui.DragFloat(" Tilt Multiplier ", Config.data.tiltMult, 0.01, 0, 5)
        if tiltMultUsed then
            RefreshCameraIfNeeded()
        end

        -- Raise controll
        Config.data.yMult, raiseMultUsed = ImGui.DragFloat(" Y Multiplier ", Config.data.yMult, 0.01, 0, 2)
        if raiseMultUsed then
            RefreshCameraIfNeeded()
        end

        -- Backoff controll
        Config.data.zMult, backoffMultUsed = ImGui.DragFloat(" Z Multiplier ", Config.data.zMult, 0.01, -15, 7)
        if backoffMultUsed then
            RefreshCameraIfNeeded()
        end

        local shouldReset = ImGui.Button(" Reset ")
        if shouldReset then
            Config.data.tiltMult = 1.150
            Config.data.yMult = 1.0
            Config.data.zMult = 0.5
            RefreshCameraIfNeeded()
        end

        ImGui.End()
        ImGui.PopStyleVar(1)
        ImGui.PopStyleColor(8)
    end)
end

return HigherCamera:New()
