local GameSettings = require("modules/external/GameSettings")
local Cron = require("modules/external/Cron")

observers = {
    isAds = false,
    stateContext = nil,
    scriptInterface = nil,
    transition = nil,
    currentLevel = 1,
    zoomAllowed = true
}

function observers.startInputObserver(bs)
    Observe('PlayerPuppet', 'OnGameAttached', function(this)
        observers.startListeners(this)
    end)

    Observe("ZoomTransition", "SendZoomAnimFeatureData", function (this, stateContext, scriptInterface)
        observers.stateContext = stateContext
        observers.scriptInterface = scriptInterface
        observers.transition = this
    end)

    Observe('PlayerPuppet', 'OnAction', function(_, action)
        local actionName = Game.NameToString(action:GetName(action))
        local actionType = action:GetType(action).value
        if actionName == 'RangedADS' then
            if actionType == 'BUTTON_PRESSED' then
                observers.isAds = true
                observers.currentLevel = 1
            elseif actionType == "BUTTON_RELEASED" then
                observers.isAds = false
            end
        elseif actionName == 'ToggleSprint' then
            if actionType == 'BUTTON_PRESSED' then
                if (not observers.isAds) or IsBound("betterScopesToggle") or not observers.zoomAllowed then return end
                observers.zoom(bs)
            end
        end
    end)
end

function observers.zoom(bs)
    local info = observers.getWeaponInfo()
    if not info.scope then return end

    local settings = observers.getSettingsByWeapoName(info.type, bs)
    if settings.levels == 1 then return end

    if GameSettings.Get("/gameplay/difficulty/SwayEffect") ~= 1 then
        GameSettings.Set("/gameplay/difficulty/SwayEffect", 1)
        GameSettings.Save()
    end

    observers.currentLevel = observers.currentLevel + 1
    if observers.currentLevel > settings.levels then observers.currentLevel = 1 end

    local zoomAmount = 0
    if observers.currentLevel ~= 1 then
        if observers.currentLevel == 2 then
            zoomAmount = settings.amount * 1.1
        elseif observers.currentLevel == 3 then
            zoomAmount = settings.amount * 3
        end
    end

    local animeData = observers.generateAnimFeatureData(observers.transition, observers.stateContext, observers.scriptInterface, zoomAmount)
    observers.scriptInterface:SetAnimationParameterFeature("ZoomAnimData", animeData)

    observers.zoomAllowed = false
    Cron.NextTick(function()
        observers.zoomAllowed = true
    end)
end

---@param this ZoomTransition
function observers.generateAnimFeatureData(this, stateContext, scriptInterface, zoom)
    local weapon = DefaultTransition.GetActiveWeapon(scriptInterface)
    ---@type gameStatsSystem
    local stats = scriptInterface:GetStatsSystem()
    local animFeatureData = AnimFeature_Zoom.new()
    animFeatureData.finalZoomLevel = this:GetCurrentZoomLevel(stateContext)
    if IsDefined(weapon) then
        animFeatureData.weaponZoomLevel = stats:GetStatValue(weapon:GetEntityID(), gamedataStatType.ZoomLevel) + zoom
        animFeatureData.weaponScopeFov = stats:GetStatValue(weapon:GetEntityID(), gamedataStatType.ScopeFOV)
        animFeatureData.weaponAimFOV = stats:GetStatValue(weapon:GetEntityID(), gamedataStatType.AimFOV)
    end
    animFeatureData.worldFOV = Game.GetCameraSystem():GetActiveCameraFOV()
    animFeatureData.zoomLevelNum = this:GetZoomLevelNumber(stateContext)
    animFeatureData.noWeaponAimInTime = this:GetStaticFloatParameterDefault("noWeaponAimInTime", 0.20)
    animFeatureData.noWeaponAimOutTime = this:GetStaticFloatParameterDefault("noWeaponAimOutTime", 0.20)
    animFeatureData.shouldUseWeaponZoomStats = this:GetShouldUseWeaponZoomData(stateContext)
    animFeatureData.focusModeActive = this:IsInVisionModeActiveState(stateContext, scriptInterface) or stateContext:IsStateActive("UpperBody", "temporaryUnequip")
    return animFeatureData
end

function observers.getWeaponInfo()
    local weapon = Game.GetTransactionSystem():GetItemInSlot(Game.GetPlayer(), TweakDBID.new('AttachmentSlots.WeaponRight'))
    local id = weapon:GetItemID()
    local itemRecord = gameRPGManager.GetItemRecord(id)
    local weaType = itemRecord:ItemType():Type().value
    local hasScope = false
    local itemParts = ItemModificationSystem.GetAllSlots(Game.GetPlayer(), id)
    for _, part in pairs(itemParts) do
        if part.installedPart:GetTDBID().hash ~= 0 then
            if (tostring(part.slotID) == tostring(TweakDBID.new("AttachmentSlots.Scope"))) then hasScope = true end
        end
    end

    return {scope = hasScope, type = weaType}
end

function observers.getSettingsByWeapoName(name, bs)
    if name == "Wea_SniperRifle" or name == "Wea_PrecisionRifle" then
        return bs.settings.sniper
    elseif name == "Wea_SubmachineGun" then
        return bs.settings.smg
    elseif name == "Wea_LightMachineGun" then
        return bs.settings.lmg
    elseif name == "Wea_Shotgun" or name == "Wea_ShotgunDual" then
        return bs.settings.shotgun
    elseif name == "Wea_Handgun" or name == "Wea_Revolver" then
        return bs.settings.handgun
    elseif name == "Wea_Rifle" then
        return bs.settings.ar
    end
end

function observers.startListeners(player)
    player:UnregisterInputListener(player, 'RangedADS')

    player:RegisterInputListener(player, 'RangedADS')
end

return observers