-- Dec 30, 2021 by anygoodname
modVer='v1.0.1b'
modName='fixVehicles'

--[[
This code uses (c)psiberx code libraries and extracts from his examples:
https://github.com/WolvenKit/cet-examples
Thank you very much psiberx!

This code uses code extracts from (c)keanuWheeze almostAutoLoot mod:
https://www.nexusmods.com/cyberpunk2077/mods/1886?tab=description
Thank you very much keanuWheeze!

And the code would not work without both psiberx and keanuWheeze active developer support :)
]]--

-- encapsulation:

local fixVehicles = {}
function fixVehicles:new()

-- prerequisites:

local GameUI = require('GameUI')

-- some basic data structures:

local vehiclesProcessed = {}
local autoMode = false
local isGameplayMode = false
local sumDelta = 0

local notification = {
	text = '',			-- text to show in a notification window. Calculated on a notification set up.
	isValid = false,	-- show/don't show the window. Calculated on a notification set up and during the show time.
	maxTTL = 3,			-- notification max show time. Calculated on a notification set up.
	TTL = 0,			-- notification remaining show time. Calculated on a notification set up and during the show time.
	fadeInLen = 0.25,	-- notification window fade in length in sec.
	fadeOutLen = 1,		-- notification window fade out length in sec.
	fadeInStop = 0,		-- when to stop the fade in. In seconds to the end. Calculated on a notification set up.
	fadeOutStart = 0,	-- when to start the fade out. In seconds to the end. Calculated on a notification set up.
	fadeInSpeed = 1,	-- this is actually a ratio to calculate alpha level for a given moment during the fade in. Calculated on a notification set up.
	fadeOutSpeed = 1,	-- this is actually a ratio to calculate alpha level for a given moment during the fade out. Calculated on a notification set up.
	xPos = 0,			-- notification window position. Calculated on a notification set up.
	yPos = 0,			-- notification window position. Calculated on a notification set up.
	windowFlags = ImGuiWindowFlags.NoTitleBar + ImGuiWindowFlags.NoMove + ImGuiWindowFlags.NoScrollbar + ImGuiWindowFlags.NoScrollWithMouse + ImGuiWindowFlags.AlwaysAutoResize + ImGuiWindowFlags.NoCollapse + ImGuiWindowFlags.NoFocusOnAppearing + ImGuiWindowFlags.NoBringToFrontOnFocus + ImGuiWindowFlags.NoMouseInputs
}

-- Hotkeys setup section:

registerHotkey("fixVehiclesPosition", "manualFixVehiclesPosition", function()
	--print('manualFixVehiclesPosition pressed')
	if isPreGame() then return end
	if isGamePaused() then return end
	if not isGameplayMode then return end
	setNotification(10, 5, 1, modName..": manualFixVehiclesPosition pressed")
	printL('manualFixVehiclesPosition pressed')
	Game.GetAudioSystem():PlayLootAllSound()
    resetVehiclesInRange(100)
end)

registerHotkey("autoVehiclesPosition", "autoFixVehiclesPosition", function()
	--print('autoFixVehiclesPosition pressed')
	autoMode = not autoMode
	if autoMode then
		setNotification(10, 5, 1, modName..": Automode enabled.")
		printL('Automode enabled')
	else
		setNotification(10, 5, 1, modName..": Automode disabled.")
		printL('Automode disabled')
	end
	Game.GetAudioSystem():PlayLootAllSound()
end)

-- init and key event handlers:

registerForEvent("onInit", function()
	autoMode = false
	isGameplayMode = false
	setObservers()
	print (modName, modVer, "initialized.")
end)

registerForEvent('onUpdate', function(delta)
	updateNotificationTTL(delta)
	if not autoMode then return end
	if isPreGame() then return end
	if isGamePaused() then return end
	if not isGameplayMode then return end
	
	sumDelta = sumDelta + delta	-- let's try to reduce the overhead here. No need to process the crosshair on every tick
	if sumDelta >= 0.5 then -- twice a second should be enough.
		sumDelta = 0
		resetVehiclesInRange(100)
	end
end)

registerForEvent("onDraw", function()
	showNotificationWindow(notification)
end)

-- Payload:

function isVehicleOfInterest(gameObj)	--https://nativedb.red4ext.com/vehicleBaseObject
	if not IsDefinedS(gameObj) then return false end
	if gameObj:IsCrowdVehicle() then return false end
	if gameObj:IsTurnedOn() then return false end
	if gameObj:IsEngineTurnedOn() then return false end
	if gameObj:IsPlayerVehicle() then return false end
	if gameObj:IsPlayerDriver() then return false end
	if gameObj:IsPlayerMounted() then return false end
	if gameObj:IsQuest() then return false end
	if gameObj:IsExecutingAnyCommand() then return false end
	if gameObj:IsDestroyed() then return false end
	if gameObj:IsVehicleUpsideDown() then return false end
	-- TODO: maybe more filtering needed:
	-- exclude vehicles with occupants?
	-- exclude vehicles with items inside, especially quest items?
	if vehiclesProcessed then
		local entityIdHashUInt64Str = tostring(gameObj:GetEntityID().hash) --force new data, not a reference
		--print ('isVehicleOfInterest: looking for', entityIdHashUInt64Str, 'in', vehiclesProcessed, vehiclesProcessed[entityIdHashUInt64Str])
		if vehiclesProcessed[entityIdHashUInt64Str] then
			--print('isVehicleOfInterest: vehicle', entityIdHashUInt64Str, 'already reset. Skipping.')
			return
		end
	end
	--print('isVehicleOfInterest: vehicle passed pre-filtering')
	return true
end

-- based on a code extract from almostAutoLoot by (c)keanuWheeze
--https://www.nexusmods.com/cyberpunk2077/mods/1886?tab=description
function resetVehiclesInRange(range, force)
	local objects = {}
	local searchQuery = Game["TSQ_ALL;"]()
	searchQuery.testedSet = gameTargetingSet.Visible
	searchQuery.maxDistance = range -- Set up search query 
	searchQuery.includeSecondaryTargets = false
	searchQuery.ignoreInstigator = true	
	local _, v, obj, objAt, entityIdHashUInt64Str
	local result, data
	
	objAt = Game.GetTargetingSystem():GetLookAtObject(Game.GetPlayer(), false, false)
	if objAt then
		obj = objAt
		if obj:IsVehicle() then
			--print ('found a vehicle in GetLookAtObject')
			if isVehicleOfInterest(obj) then resetVehiclePosition(obj) end
		end
	end
	
	_, objects = Game.GetTargetingSystem():GetTargetParts(Game.GetPlayer(), searchQuery) 
	for _, v in ipairs(objects) do  -- Iterate over all visible objects
		obj = v
		if obj then
			obj = obj:GetComponent(obj):GetEntity()
			if obj:IsVehicle() then
				--print ('found a vehicle in GetTargetParts')
				if isVehicleOfInterest(obj) then resetVehiclePosition(obj, force) end
			end
		end
	end
end

-- this is the core:
function resetVehiclePosition(gameObj, force)
	if not IsDefinedS(gameObj) then return false end
	local result, data
	
	result = false
	result, data = pcall(function() return gameObj:GetEntityID() end)
	if not result then return end
	if not data then return end
	local entityId = data
	local entityIdHashUInt64Str = tostring(entityId.hash) --force new data, not a reference
	
	if vehiclesProcessed then
		--print ('looking for', entityIdHashUInt64Str, 'in', vehiclesProcessed, vehiclesProcessed[entityIdHashUInt64Str])
		if vehiclesProcessed[entityIdHashUInt64Str] then
			--print('vehicle', entityIdHashUInt64Str, 'already reset. Skipping.')
			return
		end
	end
	
	local orgPos = gameObj:GetWorldPosition()
	local floorDistance = getFloorDistance(orgPos, 10)
	if floorDistance == nil then floorDistance = 1 end
	--print ('object position, yaw and orientation:', orgPos, gameObj:GetWorldYaw(), gameObj:GetWorldForward())
	
	-- get the floor slope pitch:
	local direction = gameObj:GetWorldForward()
	local newFrontPos = Vector4.new(orgPos.x + direction.x * 0.5, orgPos.y + direction.y * 0.5, orgPos.z + direction.z * 0.5, orgPos.w)
	local frontFloorDistance = getFloorDistance(newFrontPos, 10)
	if frontFloorDistance then newFrontPos.z = newFrontPos.z - frontFloorDistance end	
	
	local newRearPos = Vector4.new(orgPos.x - direction.x * 0.5, orgPos.y - direction.y * 0.5, orgPos.z - direction.z * 0.5, orgPos.w)
	local rearFloorDistance = getFloorDistance(newRearPos, 10)
	if rearFloorDistance then newRearPos.z = newRearPos.z - rearFloorDistance end
	local pitch = -Vector4.new(newFrontPos.x - newRearPos.x, newFrontPos.y - newRearPos.y, newFrontPos.z - newRearPos.z, newFrontPos.w - newRearPos.w):ToRotation().pitch

	-- get the floor slope roll:
	direction = gameObj:GetWorldRight()
	local newLeftPos = Vector4.new(orgPos.x + direction.x * 0.5, orgPos.y + direction.y * 0.5, orgPos.z + direction.z * 0.5, orgPos.w)
	local leftFloorDistance = getFloorDistance(newLeftPos, 10)
	if leftFloorDistance then newLeftPos.z = newLeftPos.z - leftFloorDistance end	
	
	local newRightPos = Vector4.new(orgPos.x - direction.x * 0.5, orgPos.y - direction.y * 0.5, orgPos.z - direction.z * 0.5, orgPos.w)
	local rightFloorDistance = getFloorDistance(newRightPos, 10)
	if rightFloorDistance then newRightPos.z = newRightPos.z - rightFloorDistance end
	local roll = Vector4.new(newLeftPos.x - newRightPos.x, newLeftPos.y - newRightPos.y, newLeftPos.z - newRightPos.z, newLeftPos.w - newRightPos.w):ToRotation().pitch
	
	-- don't ask me why I had to take the pitch for roll this way and then swap the two bellow - it just works ;-)
	--[[
	print('----------')
	print(orgPos)
	print(newFrontPos)
	print(newRearPos)
	print(pitch)
	print(roll)
	print('----------')
	]]--
	-- TODO: find a minimal distance for a given vehicle type to determine if it's on the ground.
	-- for the time being 0.40667724609375 is a normal limo distance on a horizontal floor. Say 0.40667 is the threshold
	-- TODO: need more filtering here: vehicles are different sizes and they're rarely perfectly positioned in the game. Sometimes above, sometimes bellow the floor. The floor may not be horizontal too.
	
	--print ('floorDistance', floorDistance, 'floorDistance < 0.40667', floorDistance < 0.40667)
	if not force then
		if floorDistance < 0.40667 then
			vehiclesProcessed[entityIdHashUInt64Str] = true
			--print('vehicle', entityIdHashUInt64Str, 'seems to be on the floor. No need to reset it')
			--print('vehicle', entityIdHashUInt64Str, 'table value', vehiclesProcessed[entityIdHashUInt64Str])
			return
		end
	end

	orgPos.z = orgPos.z - floorDistance + 0.4 -- for a horizontal floor 0.4 looks good. For slopes it may be not enough and moreover it may cause excessive bumping
	local orgYaw = gameObj:GetWorldYaw()
	local result, data
	result, data = pcall(function() Game.GetTeleportationFacility():Teleport(gameObj, orgPos, EulerAngles.new(roll, pitch, orgYaw)) end)
	if result then vehiclesProcessed[entityIdHashUInt64Str] = true end
	--print('vehicle', entityIdHashUInt64Str, 'reset result', result, data)
	--print('vehicle', entityIdHashUInt64Str, 'old position', orgPos)
	--print('vehicle', entityIdHashUInt64Str, 'new position', gameObj:GetWorldPosition(`))
	--print('vehicle', entityIdHashUInt64Str, 'table value', vehiclesProcessed[entityIdHashUInt64Str])
end

function setObservers()
	-- This is the core:
	GameUI.Listen(function(state)
		isGameplayMode = state.isLoaded and not state.isDetached and not state.isLoading and not state.isMenu and not state.isBraindance and not state.isShard and not state.isTutorial and not state.isPossessed and not state.isFlashback and not state.isCyberspace and not state.isScene
		--GameUI.PrintState(state)
	end)
		
	-- This is the core:
	Observe('PlayerPuppet', 'OnGameAttached', function(self)		--https://nativedb.red4ext.com/PlayerPuppet
		if not self:IsReplacer() then
			vehiclesProcessed = {}
		end
	end)
end

-- This is the core:
function isPreGame()
	-- meaning if it's the main menu
	return GetSingleton('inkMenuScenario'):GetSystemRequestsHandler():IsPreGame()
end

-- This is the core:
function isGamePaused()
	return GetSingleton('inkMenuScenario'):GetSystemRequestsHandler():IsGamePaused()
end

-- This is the core:
function IsDefinedS(gameObj)
	local result, val
	result, val = pcall(function() return IsDefined(gameObj) end)
	if result then return val else return false end
end

-- This is the core:
-- this part is a modified extract from TargetingHelper.lua example by (c)psiberx
--https://github.com/WolvenKit/cet-examples
function getFloorDistance(from, distance)
	if not distance then
		distance = 10 -- should be enough in most cases.
	end

	local to = Vector4.new(from.x, from.y, from.z - distance, from.w)	

	local filters = {
		'Dynamic', -- Movable Objects
		'Vehicle',
		'Static', -- Buildings, Concrete Roads, Crates, etc.
		'Water',
		'Terrain',
		'PlayerBlocker', -- Trees, Billboards, Barriers
	}

	local results = {}

	for _, filter in ipairs(filters) do
		local success, result = Game.GetSpatialQueriesSystem():SyncRaycastByCollisionGroup(from, to, filter, false, false)
		
		if success then
			table.insert(results, {
				distance = Vector4.Distance(from, ToVector4(result.position)) --,
				--position = ToVector4(result.position),
				--normal = result.normal,
				--material = result.material,
				--collision = CName.new(filter),
			})
		end
	end
	
	if not results then return nil end
	if #results == 0 then return nil end

	local nearest = results[1]
	local lowestDistance = 0
	local highestDistance = 0
	for i = 2, #results do
		if results[i].distance > 0 then
			if lowestDistance == 0 then lowestDistance = results[i].distance else if results[i].distance < lowestDistance then lowestDistance = results[i].distance end end
			if highestDistance == 0 then highestDistance = results[i].distance else if results[i].distance > highestDistance then highestDistance = results[i].distance end end
		end
	end
	--print('final distance selected', lowestDistance)
	return lowestDistance
end

function printL(str1)
	local currTimeStr=os.date("%x %X")
	print (currTimeStr..': '..modName..':',str1)
end

-- Notifications

function setNotification(xPos, yPos, maxTTL, text)
	notification.xPos = xPos
	notification.yPos = yPos

	if maxTTL < 1 then maxTTL = 1 end
	notification.maxTTL = maxTTL
	notification.maxTTL = notification.maxTTL + notification.fadeInLen + notification.fadeOutLen
	notification.TTL = notification.maxTTL
	
	notification.text = text
	notification.isValid = true
	
	if (notification.fadeInLen > 0) then
		notification.fadeInStop = notification.maxTTL - notification.fadeInLen
		notification.fadeInSpeed = 1 / notification.fadeInLen
	end
	if (notification.fadeOutLen > 0) then
		notification.fadeOutStart = notification.fadeOutLen
		notification.fadeOutSpeed = 1 / notification.fadeOutLen
	end
end

function showNotificationWindow(notification)
	if not notification.isValid then return end
	ImGui.SetNextWindowPos(notification.xPos, notification.yPos)
	local alpha = 1
	if (notification.fadeInLen > 0) then if (notification.TTL >= notification.fadeInStop) then alpha = 1 - (notification.TTL - notification.fadeInStop) * notification.fadeInSpeed end end
	if (notification.fadeOutLen > 0) then if (notification.TTL <= notification.fadeOutStart) then alpha = notification.TTL * notification.fadeOutSpeed end end
	ImGui.SetNextWindowBgAlpha(alpha)
	ImGui.Begin(modName..'Notification', true, notification.windowFlags)
	ImGui.Text(notification.text)
	ImGui.End()
end

function updateNotificationTTL(delta)
	if not notification.isValid then return end
	notification.TTL = notification.TTL - delta
	if (notification.TTL <= 0) then notification.isValid = false end
end

-- end of encapsulation:
end
return fixVehicles:new()