﻿using System;
using System.Collections.Generic;
using UnityEngine;
using Verse;
using Verse.AI;
using Verse.Sound;
using RimWorld;

namespace RecreationalSlotMachine
{
	public class RecSlotMachineSettings : ModSettings
	{
		public bool winsGiveMoodBoost = true;
		public bool winsGiveGold = true;
		public float goldSlotChance = 0.25f;
		public int payoutMin = 1;
		public int payoutMax = 1;
		public float moodBoostAmount = 5f;
		public float moodBoostDurationDays = 2f;
		public int moodBoostStackAmount = 1;
		public bool chargeGuests = false;
		public bool chargeNonGuests = false;
		public int gameCharge = 1;
		public bool winsGiveSilver = false;
		public int payoutMinSilver = 50;
		public int payoutMaxSilver = 150;
		public int pawnCollectsSilverAmount = 50;
		public int pawnKeepsSilverMin = 10;
		public int pawnInventoryHardCapForPayout = 300;
		public bool[] page = new bool[] { true, false, false };
		public bool[] pagePreviously = new bool[] { false, false, false };
		public string pawnInventoryHardCapForPayoutBuffer = "300";
		public string pawnCollectsSilverAmountBuffer = "50";
		public string pawnKeepsSilverMinBuffer = "10";
		public string payoutMinSilverBuffer = "50";
		public string payoutMaxSilverBuffer = "150";
		public string gameChargeBuffer = "1";
		public string moodBoostStackAmountBuffer = "1";
		public string payoutMinBuffer = "1";
		public string payoutMaxBuffer = "1";
		public string moodBoostAmountBuffer = "5";
		public string moodBoostDurationDaysBuffer = "2";
		public Vector2 scrollPosition = new Vector2(1f, 1f);
		public float gameLengthSeconds = 2f;
		public string gameLengthSecondsBuffer = "2";
		public int gameLengthTicks = 120;
		public float gameSettingsToDefaultSlider = 0f;
		public int toilDurationTicks = 1205;
		public float toilDurationSeconds = 20f;
		public string toilDurationSecondsBuffer = "20";

		// Gives radio button logic to a list of bools tied to a set of checkboxes.
		public void RadioSettingsController(bool[] radioButtonsBoolsCurrently, bool[] radioButtonsBoolsPreviously)
		{
			bool allRadiosDeactive = true;
			for(int i = 0; i < radioButtonsBoolsCurrently.Length; i++)
            {
				allRadiosDeactive = !radioButtonsBoolsCurrently[i] & allRadiosDeactive;
            }
			if (allRadiosDeactive)
			{
				radioButtonsBoolsCurrently[0] = true;
				radioButtonsBoolsPreviously[0] = false;
			}
			for (int i = 0; i < radioButtonsBoolsCurrently.Length; i++)
            {
				if (radioButtonsBoolsCurrently[i] && !radioButtonsBoolsPreviously[i])
                {
					radioButtonsBoolsPreviously[i] = true;
					for (int j = 0; j < radioButtonsBoolsCurrently.Length; j++)
                    {
						if (i == j)
							continue;
						radioButtonsBoolsCurrently[j] = false;
						radioButtonsBoolsPreviously[j] = false;
                    }
					break;
                }
            }
		}
		public void DefaultSettings()
        {
			winsGiveMoodBoost = true;
			winsGiveGold = true;
			goldSlotChance = 0.25f;
			payoutMin = 1;
			payoutMax = 1;
			moodBoostAmount = 5f;
			moodBoostDurationDays = 2f;
			moodBoostStackAmount = 1;
			chargeGuests = false;
			chargeNonGuests = false;
			gameCharge = 1;
			winsGiveSilver = false;
			payoutMinSilver = 50;
			payoutMaxSilver = 150;
			pawnCollectsSilverAmount = 50;
			pawnKeepsSilverMin = 10;
			pawnInventoryHardCapForPayout = 300;
			pawnInventoryHardCapForPayoutBuffer = "300";
			pawnCollectsSilverAmountBuffer = "50";
			pawnKeepsSilverMinBuffer = "10";
			payoutMinSilverBuffer = "50";
			payoutMaxSilverBuffer = "150";
			gameChargeBuffer = "1";
			moodBoostStackAmountBuffer = "1";
			payoutMinBuffer = "1";
			payoutMaxBuffer = "1";
			moodBoostAmountBuffer = "5";
			moodBoostDurationDaysBuffer = "2";
			scrollPosition = new Vector2(1f, 1f);
			gameLengthSeconds = 2f;
			gameLengthSecondsBuffer = "2";
			gameLengthTicks = 120;
			toilDurationTicks = 1205;
			toilDurationSeconds = 20f;
			toilDurationSecondsBuffer = "20";
		}
		public void PostUserInput()
		{
			int temp = payoutMax;
			payoutMax = Math.Max(payoutMin, payoutMax);
			payoutMin = Math.Min(payoutMin, temp);
			temp = payoutMaxSilver;
			payoutMaxSilver = Math.Max(payoutMinSilver, payoutMaxSilver);
			payoutMinSilver = Math.Min(payoutMinSilver, temp);
			payoutMaxBuffer = payoutMax.ToString();
			payoutMinBuffer = payoutMin.ToString();
			payoutMaxSilverBuffer = payoutMaxSilver.ToString();
			payoutMinSilverBuffer = payoutMinSilver.ToString();
			gameCharge = Math.Max(1, gameCharge);
			gameChargeBuffer = gameCharge.ToString();
			pawnCollectsSilverAmount = Math.Max(1, pawnCollectsSilverAmount);
			pawnCollectsSilverAmountBuffer = pawnCollectsSilverAmount.ToString();
			moodBoostStackAmount = Math.Max(1, moodBoostStackAmount);
			moodBoostStackAmountBuffer = moodBoostStackAmount.ToString();
			if (!moodBoostDurationDaysBuffer.EndsWith("."))
            {
				moodBoostDurationDays = Mathf.Max(0.1f, moodBoostDurationDays);
				moodBoostDurationDaysBuffer = moodBoostDurationDays.ToString();
            }
			if (!toilDurationSecondsBuffer.EndsWith("."))
			{
				toilDurationSeconds = Mathf.Max(0.1f, toilDurationSeconds);
				toilDurationSecondsBuffer = toilDurationSeconds.ToString();
				toilDurationTicks = (int)(toilDurationSeconds * 60f) + 5;
			}
			if (!moodBoostAmountBuffer.EndsWith("."))
			{
				moodBoostAmount = Mathf.Max(0.1f, moodBoostAmount);
				moodBoostAmountBuffer = moodBoostAmount.ToString();
			}
			if (!gameLengthSecondsBuffer.EndsWith("."))
			{
				gameLengthSeconds = Mathf.Max(0.1f, gameLengthSeconds);
				gameLengthTicks = (int)(gameLengthSeconds * 60f);
				gameLengthSecondsBuffer = gameLengthSeconds.ToString();
			}
		}
		public override void ExposeData()
		{
            Scribe_Values.Look(ref winsGiveMoodBoost, "winsGiveMoodBoost", true);
			Scribe_Values.Look(ref winsGiveGold, "winsGiveGold", true);
			Scribe_Values.Look(ref goldSlotChance, "goldSlotChance", 0.25f);
			Scribe_Values.Look(ref payoutMin, "payoutMin", 1);
			Scribe_Values.Look(ref payoutMax, "payoutMax", 1);
			Scribe_Values.Look(ref moodBoostAmount, "moodBoostAmount", 5f);
			Scribe_Values.Look(ref moodBoostDurationDays, "moodBoostDurationDays", 2f);
			Scribe_Values.Look(ref gameLengthSeconds, "gameLengthSeconds", 2f);
			Scribe_Values.Look(ref moodBoostStackAmount, "moodBoostStackAmount", 1);
			Scribe_Values.Look(ref gameCharge, "gameCharge", 1);
			Scribe_Values.Look(ref winsGiveSilver, "winsGiveSilver", false);
			Scribe_Values.Look(ref payoutMinSilver, "payoutMinSilver", 50);
			Scribe_Values.Look(ref payoutMaxSilver, "payoutMaxSilver", 150);
			Scribe_Values.Look(ref pawnKeepsSilverMin, "pawnKeepsSilverMin", 10);
			Scribe_Values.Look(ref pawnCollectsSilverAmount, "pawnCollectsSilverAmount", 50);
			Scribe_Values.Look(ref chargeGuests, "chargeGuests", false);
			Scribe_Values.Look(ref chargeNonGuests, "chargeNonGuests", false);
			Scribe_Values.Look(ref pawnInventoryHardCapForPayout, "pawnInventoryHardCapForPayout", 300);
			Scribe_Values.Look(ref toilDurationSeconds, "toilDurationSeconds", 20f);
			pawnInventoryHardCapForPayoutBuffer = pawnInventoryHardCapForPayout.ToString();
			pawnKeepsSilverMinBuffer = pawnKeepsSilverMin.ToString();
			PostUserInput();  // Also sets all the other string buffers
			base.ExposeData();
		}
	}

	[StaticConstructorOnStartup]
	public class RecreationalSlotMachine : Mod
	{
		public static RecSlotMachineSettings settings;
		public static ThingDef[] reelFrameThingDefs;
		public static Thing[] reelFrameThings;
		public static Mesh[] reelMeshes;
		public static Material[] reelMaterials;
		public static int blurFrameCount;
		public static bool initialised = false;
		public RecreationalSlotMachine(ModContentPack content) : base(content)
		{
			settings = GetSettings<RecSlotMachineSettings>();
		}
		public override void DoSettingsWindowContents(Rect inRect)
		{
			Listing_Standard listingStandard = new Listing_Standard();
            listingStandard.Begin(inRect);
			settings.RadioSettingsController(settings.page, settings.pagePreviously);
			string pageOneText = "Page 1 - general";
			string pageOneToolTip = "You are looking at page 1 of the settings menu. Click to change page.";
			string pageTwoText = "Page 2 - payouts";
			string pageTwoToolTip = "You are looking at page 2 of the settings menu. Click to change page.";
			string pageThreeText = "Page 3 - charges";
			string pageThreeToolTip = "You are looking at page 3 of the settings menu. Click to change page.";
			string winsGiveGoldText = "Winning gives gold";
			string winsGiveGoldToolTip = "Gold falls out of the machine. Who put that in there?";
			string winsGiveSilverText = "Winning gives silver";
			string winsGiveSilverToolTip = "Silver is paid directly into the pawn's inventory if the pawn has to pay to play.";
			string winsGiveMoodBoostText = "Winning gives mood boost";
			string winsGiveMoodBoostToolTip = "Winning feels good";
			string chargeGuestsText = "Charge visitors";
			string chargeGuestsToolTip = "Games will cost visitors silver. Visitors will be unable to attend this activity if they don't have silver in their inventory.";
			string chargeNonGuestsText = "Charge owners";
			string chargeNonGuestsToolTip = "Games will cost silver to your own pawns. Your own pawns will be unable to attend this activity if they don't have silver in their inventory.";
			string defaultSettingsText = "Use this \"button\" to move the slider";
			string defaultSettingsToolTip = "Yep.";
			int gamesCompletable = (int)((settings.toilDurationSeconds + 0.083f) / settings.gameLengthSeconds);
			string sessionLength = settings.toilDurationSecondsBuffer + "-second session";
			float winChanceNum = Mathf.Pow(settings.goldSlotChance, 3f) * 100f;
			float expectedSilverReturn = winChanceNum / 100f * ((settings.payoutMinSilver + settings.payoutMaxSilver) / 2) / settings.gameCharge;
			if (settings.page[0])
            {
				listingStandard.CheckboxLabeled(pageOneText, ref settings.page[1], pageOneToolTip);
				listingStandard.Label("");
				listingStandard.Label("Chance for 1 slot to land on gold: " + (settings.goldSlotChance * 100f).ToString() + "%");
				settings.goldSlotChance = listingStandard.Slider(settings.goldSlotChance, 0.012f, 1f);
				listingStandard.Label("");
				listingStandard.Label("Length of the activity in seconds. This governs how much recreation the activity will give as a result of how long the pawn spends doing it.");
				listingStandard.TextFieldNumeric(ref settings.toilDurationSeconds, ref settings.toilDurationSecondsBuffer, 0.1f);
				listingStandard.Label("");
				listingStandard.Label("Length of 1 game in seconds. Unfinishable games will not be started and during the time of which, the pawn will just stare at the machine but still gain recreation.");
				listingStandard.TextFieldNumeric(ref settings.gameLengthSeconds, ref settings.gameLengthSecondsBuffer, 0.1f);
				listingStandard.Label("");

				float overallWinChanceNum = winChanceNum * gamesCompletable;
				string winChance = winChanceNum.ToString();
				if (winChance.Length > 5)
					winChance = winChance.Substring(0, 5);
				string overallWinChance = overallWinChanceNum.ToString();
				if (overallWinChance.Length > 5)
					overallWinChance = overallWinChance.Substring(0, 5);
				listingStandard.Label("Pawns can complete " + gamesCompletable.ToString() + " games per " + sessionLength + ". Pawns have a " + winChance + "% chance per game (" + overallWinChance + "% overall) to win");
				listingStandard.Label("");
				listingStandard.Label("");
				listingStandard.Label("Use this slider minigame to reset this mod's settings to default. The slider needs to reach the end.");
				listingStandard.Slider(settings.gameSettingsToDefaultSlider, 0f, 1f);
				bool moveSlider = false;
				listingStandard.CheckboxLabeled(defaultSettingsText, ref moveSlider, defaultSettingsToolTip);
				if (moveSlider)
				{
					settings.gameSettingsToDefaultSlider += 0.175f;
					if (settings.gameSettingsToDefaultSlider >= 1f)
					{
						settings.gameSettingsToDefaultSlider = 1f;
						settings.DefaultSettings();
					}
				}
				//listingStandard.Label("");
				//listingStandard.Label("");
				//listingStandard.Label("For any text box that looks like you should be able to put a decimal point in but cannot; you can, it is just a little buggy. Try typing \"11\" and putting the decimal point between the two 1's.");
			}
			if (settings.page[1])
			{
				listingStandard.CheckboxLabeled(pageTwoText, ref settings.page[2], pageTwoToolTip);
				listingStandard.Label("");
				listingStandard.CheckboxLabeled(winsGiveSilverText, ref settings.winsGiveSilver, winsGiveSilverToolTip);
				listingStandard.Label("Minimum silver payout");
				listingStandard.IntEntry(ref settings.payoutMinSilver, ref settings.payoutMinSilverBuffer);
				listingStandard.Label("Maximum silver payout");
				listingStandard.IntEntry(ref settings.payoutMaxSilver, ref settings.payoutMaxSilverBuffer);
				listingStandard.Label("Expected silver return per 1 silver spent (if being charged to play): " + expectedSilverReturn.ToString());
				listingStandard.Label("");
				listingStandard.CheckboxLabeled(winsGiveGoldText, ref settings.winsGiveGold, winsGiveGoldToolTip);
				listingStandard.Label("Minimum gold payout");
				listingStandard.IntEntry(ref settings.payoutMin, ref settings.payoutMinBuffer);
				listingStandard.Label("Maximum gold payout");
				listingStandard.IntEntry(ref settings.payoutMax, ref settings.payoutMaxBuffer);
				listingStandard.Label("");
				listingStandard.CheckboxLabeled(winsGiveMoodBoostText, ref settings.winsGiveMoodBoost, winsGiveMoodBoostToolTip);
				listingStandard.Label("Mood boost stack limit");
				listingStandard.IntEntry(ref settings.moodBoostStackAmount, ref settings.moodBoostStackAmountBuffer);
				listingStandard.Label("Amount of mood per boost");
				listingStandard.TextFieldNumeric(ref settings.moodBoostAmount, ref settings.moodBoostAmountBuffer);
				listingStandard.Label("How many days the mood boost will last");
				listingStandard.TextFieldNumeric(ref settings.moodBoostDurationDays, ref settings.moodBoostDurationDaysBuffer);
			}
			if (settings.page[2])
			{
				listingStandard.CheckboxLabeled(pageThreeText, ref settings.page[0], pageThreeToolTip);
				listingStandard.Label("");
				string overallSilverCost = (gamesCompletable * settings.gameCharge).ToString();
				listingStandard.Label("Silver charge per game (Per " + sessionLength + ": " + overallSilverCost + ")");
				listingStandard.IntEntry(ref settings.gameCharge, ref settings.gameChargeBuffer);
				listingStandard.Label("Expected silver return per 1 silver spent (if being charged to play): " + expectedSilverReturn.ToString());
				listingStandard.Label("");
				listingStandard.CheckboxLabeled(chargeGuestsText, ref settings.chargeGuests, chargeGuestsToolTip);
				listingStandard.CheckboxLabeled(chargeNonGuestsText, ref settings.chargeNonGuests, chargeNonGuestsToolTip);
				listingStandard.Label("");
				listingStandard.Label("Minimum amount of silver in a pawn's inventory before the pawn will attempt to collect some from storage");
				listingStandard.IntEntry(ref settings.pawnKeepsSilverMin, ref settings.pawnKeepsSilverMinBuffer);
				listingStandard.Label("");
				listingStandard.Label("Amount of silver the pawns will collect when they feel they are running out");
				listingStandard.IntEntry(ref settings.pawnCollectsSilverAmount, ref settings.pawnCollectsSilverAmountBuffer);
				listingStandard.Label("");
				listingStandard.Label("How much silver the pawn can hold at once as a result of a payout. (extra silver will drop on the floor, does not apply to visitors)");
				listingStandard.IntEntry(ref settings.pawnInventoryHardCapForPayout, ref settings.pawnInventoryHardCapForPayoutBuffer);
				listingStandard.Label("");
				listingStandard.Label("Your own pawns will only collect and keep silver if \"Charge owners\" is enabled and only if they decide to use a slot machine (they won't pick up silver unless a slot machine exists).");
				listingStandard.Label("");
			}
			if (settings.gameSettingsToDefaultSlider > 0f)
            {
				settings.gameSettingsToDefaultSlider -= 0.0005f;
				if (settings.gameSettingsToDefaultSlider < 0f)
                {
					settings.gameSettingsToDefaultSlider = 0f;
                }
            }
			settings.PostUserInput();
			MyDefs.WonSlotMachineGame.durationDays = settings.moodBoostDurationDays;
			MyDefs.WonSlotMachineGame.stackLimit = settings.moodBoostStackAmount;
			listingStandard.End();
            base.DoSettingsWindowContents(inRect);
		}
		public override string SettingsCategory()
		{
			return "RecreationalSlotMachine";
		}
		public static void InitialiseTextures()
        {
			if (!initialised)
			{
				initialised = true;
				reelFrameThingDefs = new ThingDef[]
				{
					MyDefs.Mote_ReelOutcomeA,
					MyDefs.Mote_ReelOutcomeB,
					MyDefs.Mote_ReelBlurA,
					MyDefs.Mote_ReelBlurB,
					MyDefs.Mote_ReelBlurC,
					MyDefs.Mote_ReelBlurD,
					MyDefs.Mote_ReelBlurE,
					MyDefs.Mote_ReelBlurF,
					MyDefs.Mote_ReelBlurG,
					MyDefs.Mote_ReelBlurH,
					MyDefs.Mote_ReelBlurI,
					MyDefs.Mote_ReelBlurJ
				};
				blurFrameCount = reelFrameThingDefs.Length - 2;
				reelFrameThings = new Thing[reelFrameThingDefs.Length];
				reelMeshes = new Mesh[reelFrameThingDefs.Length];
				reelMaterials = new Material[reelFrameThingDefs.Length];
				for (int i = 0; i < reelFrameThings.Length; i++)
				{
					reelFrameThings[i] = ThingMaker.MakeThing(reelFrameThingDefs[i]);
					reelMeshes[i] = reelFrameThings[i].Graphic.MeshAt(Rot4.North);
					reelMaterials[i] = reelFrameThings[i].Graphic.MatSingle;
				}
				MyDefs.WonSlotMachineGame.durationDays = settings.moodBoostDurationDays;
				MyDefs.WonSlotMachineGame.stackLimit = settings.moodBoostStackAmount;
			}
        }
	}
	public class SlotMachineComp : ThingComp
    {
		public bool initialised = false;
		public SlotMachineGameEventManager eventManager = new SlotMachineGameEventManager();
		public bool justRespawned = false;
        public override void PostSpawnSetup(bool respawningAfterLoad)
        {
            base.PostSpawnSetup(respawningAfterLoad);
			justRespawned = respawningAfterLoad;
			for (int i = 0; i < 3; i++)
				eventManager.reels[i] = new SlotReel();
			Vector3 middle = parent.Position.ToVector3() + new Vector3(0.5045f, 0f, 0.5f);
			Vector3 offset = new Vector3(0.188f, 0f, 0f);
			middle.z += 0.445f;
			eventManager.reels[0].drawLocation = middle - offset;
			eventManager.reels[1].drawLocation = middle;
			eventManager.reels[2].drawLocation = middle + offset;
			eventManager.reels[0].drawLocation.y = 5.5f;
			eventManager.reels[1].drawLocation.y = 5.5f;
			eventManager.reels[2].drawLocation.y = 5.5f;
			eventManager.gameStarted = false;
			initialised = true;
		}
        public override void PostDeSpawn(Map map)
		{
			initialised = false;
			base.PostDeSpawn(map);
        }
        public override void CompTick()
        {
            base.CompTick();
			RecreationalSlotMachine.InitialiseTextures(); // Only happens once. This is the only safe place I know of right now to call it from.
			foreach(SlotReel reel in eventManager.reels)
            {
				if (reel.state == SlotReelState.Spinning)
				{
					if (reel.ticksRemainingForHardCutoff > 0)
						reel.ticksRemainingForHardCutoff--;
					if (reel.ticksRemainingForHardCutoff == 0)
					{
						reel.state = SlotReelState.Loss;
					}
				}
				reel.UpdateDrawState();
			}
        }
        public override void PostDraw()
        {
            base.PostDraw();
			if (initialised)
			{
				/*----------------------------  RimWorld v1.3 compatible rendering method ----------------------------*/
				Matrix4x4 matrix1 = default(Matrix4x4);
                matrix1.SetTRS(eventManager.reels[0].drawLocation, Rot4.North.AsQuat, Vector3.one);
                Graphics.DrawMesh(eventManager.reels[0].drawingMesh, matrix1, eventManager.reels[0].drawingMaterial, 0);
                Matrix4x4 matrix2 = default(Matrix4x4);
                matrix2.SetTRS(eventManager.reels[1].drawLocation, Rot4.North.AsQuat, Vector3.one);
                Graphics.DrawMesh(eventManager.reels[1].drawingMesh, matrix2, eventManager.reels[1].drawingMaterial, 0);
                Matrix4x4 matrix3 = default(Matrix4x4);
                matrix3.SetTRS(eventManager.reels[2].drawLocation, Rot4.North.AsQuat, Vector3.one);
                Graphics.DrawMesh(eventManager.reels[2].drawingMesh, matrix3, eventManager.reels[2].drawingMaterial, 0);
				/*----------------------------------------------------------------------------------------------------*/


				// The 1.2 assembly is compiled with this instead
				/*----------------------------------------------------  RimWorld v1.2 compatible rendering method ----------------------------------------------------*/
				//Graphics.DrawMesh(eventManager.reels[0].drawingMesh, eventManager.reels[0].drawLocation, Rot4.North.AsQuat, eventManager.reels[0].drawingMaterial, 0);
				//Graphics.DrawMesh(eventManager.reels[1].drawingMesh, eventManager.reels[1].drawLocation, Rot4.North.AsQuat, eventManager.reels[1].drawingMaterial, 0);
				//Graphics.DrawMesh(eventManager.reels[2].drawingMesh, eventManager.reels[2].drawLocation, Rot4.North.AsQuat, eventManager.reels[2].drawingMaterial, 0);
				/*----------------------------------------------------------------------------------------------------------------------------------------------------*/

				justRespawned = false;
			}
		}
	}
	public enum SlotReelState { Loss, Win, Spinning }
	public class SlotReel
	{
		public SlotReelState state = SlotReelState.Loss;
		public Vector3 drawLocation;
		public int ticksRemainingForHardCutoff = 0; // Hard cutoff occurs when a pawn leaves an unfinished game
		public Mesh drawingMesh;
		public Material drawingMaterial;
		public SlotReel()
        {

        }
		public void UpdateDrawState()
        {
			switch(state)
			{
				case SlotReelState.Loss:
					drawingMaterial = RecreationalSlotMachine.reelMaterials[0];
					drawingMesh = RecreationalSlotMachine.reelMeshes[0];
					break;
				case SlotReelState.Win:
					drawingMaterial = RecreationalSlotMachine.reelMaterials[1];
					drawingMesh = RecreationalSlotMachine.reelMeshes[1];
					break;
				case SlotReelState.Spinning:
					int thisFrame = Rand.Range(2, 2 + RecreationalSlotMachine.blurFrameCount);
					drawingMaterial = RecreationalSlotMachine.reelMaterials[thisFrame];
					drawingMesh = RecreationalSlotMachine.reelMeshes[thisFrame];
					break;
			}
        }
    }
	public class SlotMachineGameEventManager
    {
		public List<SlotMachineGameEvent> gameEvents = new List<SlotMachineGameEvent>();
		public SlotReel[] reels = new SlotReel[3];
		public bool gameStarted = false;
		public bool gameWon = false;
		public int goldReelCount;
		public bool NextEvent(int tickNow)
        {
			if (gameEvents.Count > 0)
            {
				if(gameEvents[0].AttemptFireEvent(tickNow))
                {
					if (gameEvents[0].eventType == SlotEventType.Start)
					{
						goldReelCount = 0;
						gameWon = false;
					}
					else if ((gameEvents[0].eventType == SlotEventType.Stop) && (gameEvents[0].reel.state == SlotReelState.Win))
                    {
						goldReelCount++;
                    }
					return true;
                }
            }
			return false;
        }
		public void EndGame()
        {
			gameEvents.Clear();
			foreach(SlotReel reel in reels)
            {
				reel.state = SlotReelState.Loss;
				reel.UpdateDrawState();
            }
        }
		public void CalculateNewEvents(int ticksLength, float gameTime)
        {
			ticksLength -= 5;
			if (gameTime < 0.1f) gameTime = 0.1f;
			gameEvents.Clear();
			float startInterval = gameTime * 0.95f;
			float reel1Interval = gameTime * 0.55f;
			float reel2Interval = gameTime * 0.35f;
			float reel3Interval = gameTime * 0.15f;
			float sessionLength = ticksLength / 60;
			while (sessionLength >= gameTime)
            {
				sessionLength -= gameTime;
				int thisStart = (int)((sessionLength + startInterval) * 60f) + 5;
				int thisReel1 = (int)((sessionLength + reel1Interval) * 60f) + 5;
				int thisReel2 = (int)((sessionLength + reel2Interval) * 60f) + 5;
				int thisReel3 = (int)((sessionLength + reel3Interval) * 60f) + 5;
				gameEvents.Add(new SlotMachineGameEvent(thisStart, reels[0], SlotEventType.Start));
				gameEvents.Add(new SlotMachineGameEvent(thisStart, reels[1], SlotEventType.Start));
				gameEvents.Add(new SlotMachineGameEvent(thisStart, reels[2], SlotEventType.Start));
				gameEvents.Add(new SlotMachineGameEvent(thisReel1, reels[0], SlotEventType.Stop));
				gameEvents.Add(new SlotMachineGameEvent(thisReel2, reels[1], SlotEventType.Stop));
				gameEvents.Add(new SlotMachineGameEvent(thisReel3, reels[2], SlotEventType.Stop));
			}
			gameStarted = true;
        }
    }
	public enum SlotEventType { Start, Stop }
	public class SlotMachineGameEvent
    {
		public int tickHappeningOn;
		public bool eventFiredYet = false;
		public SlotEventType eventType;
		public SlotReel reel;
		public SlotMachineGameEvent(int tickHappeningOn, SlotReel reel, SlotEventType type)
        {
			if (tickHappeningOn < 0) tickHappeningOn = 0;
			this.tickHappeningOn = tickHappeningOn;
			eventType = type;
			this.reel = reel;
        }
		public bool AttemptFireEvent(int tickNow)
        {
			if (tickNow <= tickHappeningOn || tickNow <= 5)
			{
				if (eventType == SlotEventType.Start)
                {
					reel.state = SlotReelState.Spinning;
					reel.ticksRemainingForHardCutoff = RecreationalSlotMachine.settings.gameLengthTicks * 2;
                }
				else
					reel.state = (Rand.Range(0f, 1f) <= RecreationalSlotMachine.settings.goldSlotChance) ? SlotReelState.Win : SlotReelState.Loss;
				reel.UpdateDrawState();
				return true;
			}
			return false;
        }
	}
	public class JobJoyHelper
	{

		public static bool CheckIfShouldPay(Pawn pawn, Thing slotMachine)
		{
			if (pawn.Faction != slotMachine.Faction)
			{
				if (RecreationalSlotMachine.settings.chargeGuests)
					return true;
			}
			else if (RecreationalSlotMachine.settings.chargeNonGuests)
				return true;
			return false;
		}
		public static int CountSilver(Pawn pawn)
		{
			int silverCount = 0;
			for (int i = 0; i < pawn.inventory.innerContainer.Count; i++)
			{
				if (pawn.inventory.innerContainer[i] == null)
					continue;
				if (pawn.inventory.innerContainer[i].def == ThingDefOf.Silver)
					silverCount += pawn.inventory.innerContainer[i].stackCount;
			}
			return silverCount;
		}
		public static bool CheckHasEnoughSilver(Pawn pawn)
		{
			int silverCount = CountSilver(pawn);
			return silverCount >= RecreationalSlotMachine.settings.gameCharge;
		}
		public static void GiveRewardToPawn(Pawn pawn, int amount, bool isVisitor, ThingDef rewardDef)
		{
			int amountThatCanBeGiven = 0;
			if (rewardDef == ThingDefOf.Silver)
				amountThatCanBeGiven = RecreationalSlotMachine.settings.pawnInventoryHardCapForPayout - CountSilver(pawn);
			if (amountThatCanBeGiven < 0)
				amountThatCanBeGiven = 0;
			if (isVisitor || !pawn.Faction.IsPlayer)
				amountThatCanBeGiven = amount;
			int diffOffset = 0;
			if (amount > amountThatCanBeGiven)
			{
				diffOffset = amount - amountThatCanBeGiven;
				amount = amountThatCanBeGiven;
			}
			Thing reward = ThingMaker.MakeThing(rewardDef);
			reward.stackCount = amount;
			bool couldGiveAll = false;
			if (amount > 0)
				couldGiveAll = pawn.inventory.innerContainer.TryAdd(reward, true);
			if (!couldGiveAll)
			{
				reward.stackCount += diffOffset;
				GenSpawn.Spawn(reward, pawn.Position, pawn.Map);
			}
			else if (diffOffset > 0)
			{
				Thing anotherReward = ThingMaker.MakeThing(rewardDef);
				anotherReward.stackCount = diffOffset;
				GenSpawn.Spawn(anotherReward, pawn.Position, pawn.Map);
			}
		}
	}

	public class JoyGiver_PlaySlotMachine : JoyGiver_InteractBuildingInteractionCell
	{
		protected override bool CanDoDuringGathering
		{
			get
			{
				return true;
			}
		}
		protected override Job TryGivePlayJob(Pawn pawn, Thing t)
		{
			if (JobJoyHelper.CheckIfShouldPay(pawn, t))
			{
				int silverCount = JobJoyHelper.CountSilver(pawn);
				if (pawn.Faction.IsPlayer && silverCount < RecreationalSlotMachine.settings.pawnKeepsSilverMin)
				{
					if (pawn.Map.resourceCounter.Silver > 0)
					{
						Thing foundSilver = GenClosest.ClosestThingReachable(
							pawn.Position, 
							pawn.Map, 
							ThingRequest.ForDef(ThingDefOf.Silver), 
							PathEndMode.ClosestTouch, 
							TraverseParms.For(pawn, Danger.Deadly, TraverseMode.ByPawn, false)
						);
						if (foundSilver != null)
						{
							if (pawn.CanReserveAndReach(foundSilver, PathEndMode.OnCell, Danger.Deadly))
							{
								Job grabSilver = JobMaker.MakeJob(JobDefOf.TakeInventory, foundSilver);
								if (grabSilver != null)
								{
									grabSilver.count = RecreationalSlotMachine.settings.pawnCollectsSilverAmount;
									return grabSilver;
								}
							}
						}
					}
				}
				if (silverCount < RecreationalSlotMachine.settings.gameCharge)
					return null;
			}
			return JobMaker.MakeJob(this.def.jobDef, t);
		}
	}


	public class JobDriver_PlaySlotMachine : JobDriver
	{
		public override bool TryMakePreToilReservations(bool errorOnFailed)
		{
			return this.pawn.Reserve(this.job.targetA, this.job, this.job.def.joyMaxParticipants, 0, null, errorOnFailed);
		}

        protected override IEnumerable<Toil> MakeNewToils()
		{
			this.EndOnDespawnedOrNull(TargetIndex.A, JobCondition.Incompletable);
			yield return Toils_Goto.GotoCell(TargetThingA.InteractionCell, PathEndMode.OnCell);
			Toil toil = new Toil();
			toil.initAction = delegate ()
			{
				this.job.locomotionUrgency = LocomotionUrgency.Walk;
				TargetThingA.TryGetComp<SlotMachineComp>().eventManager.gameStarted = false;
			};
			toil.tickAction = delegate ()
			{
				SlotMachineComp comp = TargetThingA.TryGetComp<SlotMachineComp>();
				if (comp.initialised)
				{
					if (comp.justRespawned)
                    {
						comp.justRespawned = false;
						base.EndJobWith(JobCondition.Succeeded);
						return;
					}
					if (!comp.eventManager.gameStarted)
					{
						comp.eventManager.CalculateNewEvents(RecreationalSlotMachine.settings.toilDurationTicks, RecreationalSlotMachine.settings.gameLengthSeconds);
					}
					bool playStart = false;
					while (comp.eventManager.NextEvent(ticksLeftThisToil))
					{
						if (comp.eventManager.gameEvents[0].eventType == SlotEventType.Start)
							playStart = true;
						else
						{
							MyDefs.MapOnlyDragSlider.PlayOneShot(new TargetInfo(pawn.Position, pawn.Map));
						}
						if (comp.eventManager.goldReelCount == 3 && !comp.eventManager.gameWon)
						{
							comp.eventManager.gameWon = true;
							MyDefs.MapOnlyTinyBell.PlayOneShot(new TargetInfo(pawn.Position, pawn.Map));
							if (RecreationalSlotMachine.settings.winsGiveMoodBoost)
							{
								pawn.needs.mood.thoughts.memories.TryGainMemory(MyDefs.WonSlotMachineGame);
							}
							if (RecreationalSlotMachine.settings.winsGiveGold)
							{
								int goldRewarded = Rand.RangeInclusive(RecreationalSlotMachine.settings.payoutMin, RecreationalSlotMachine.settings.payoutMax);
								if (goldRewarded > 0)
								{
									if (JobJoyHelper.CheckIfShouldPay(pawn, TargetThingA))
									{
										JobJoyHelper.GiveRewardToPawn(pawn, goldRewarded, pawn.Faction != TargetThingA.Faction, ThingDefOf.Gold);
									}
									else
									{
										Thing gold = ThingMaker.MakeThing(ThingDefOf.Gold);
										gold.stackCount = goldRewarded;
										GenSpawn.Spawn(gold, pawn.Position, pawn.Map);
									}
								}
							}
							if (RecreationalSlotMachine.settings.winsGiveSilver)
							{
								int silverRewarded = Rand.RangeInclusive(RecreationalSlotMachine.settings.payoutMinSilver, RecreationalSlotMachine.settings.payoutMaxSilver);
								if (silverRewarded > 0)
								{
									if (JobJoyHelper.CheckIfShouldPay(pawn, TargetThingA))
                                    {
										JobJoyHelper.GiveRewardToPawn(pawn, silverRewarded, pawn.Faction != TargetThingA.Faction, ThingDefOf.Silver);
									}
									else
                                    {
										Thing silver = ThingMaker.MakeThing(ThingDefOf.Silver);
										silver.stackCount = silverRewarded;
										GenSpawn.Spawn(silver, pawn.Position, pawn.Map);
                                    }
								}
							}
						}
						comp.eventManager.gameEvents.RemoveAt(0);
					}
					if (playStart)
					{
						if (JobJoyHelper.CheckIfShouldPay(pawn, TargetThingA))
                        {
							if(JobJoyHelper.CheckHasEnoughSilver(pawn))
                            {
								int owed = RecreationalSlotMachine.settings.gameCharge;
								for (int i = 0; i < pawn.inventory.innerContainer.Count; i++)
								{
									if (pawn.inventory.innerContainer[i] == null)
                                    {
										continue;
                                    }
									if (pawn.inventory.innerContainer[i].def == ThingDefOf.Silver)
									{
										if(pawn.inventory.innerContainer[i].stackCount > owed)
                                        {
											pawn.inventory.innerContainer[i].stackCount -= owed;
											break;
										}
										else
                                        {
											owed -= pawn.inventory.innerContainer[i].stackCount;
											Thing j = pawn.inventory.innerContainer[i];
											pawn.inventory.innerContainer.RemoveAt(i);
											j.Destroy();
											if (owed == 0)
												break;
											i--;
                                        }
									}
								}
							}
							else
							{
								comp.eventManager.EndGame();
								base.EndJobWith(JobCondition.Succeeded);
								return;
							}
						}
						SoundDefOf.FlickSwitch.PlayOneShot(new TargetInfo(pawn.Position, pawn.Map));
					}
				}
				if (Find.TickManager.TicksGame > this.startTick + this.job.def.joyDuration)
				{
					base.EndJobWith(JobCondition.Succeeded);
					return;
				}
				JoyUtility.JoyTickCheckEnd(this.pawn, JoyTickFullJoyAction.EndJob, 1f, (Building)base.TargetThingA);
			};
			
			toil.socialMode = RandomSocialMode.SuperActive;
			toil.defaultCompleteMode = ToilCompleteMode.Delay;
			toil.defaultDuration = RecreationalSlotMachine.settings.toilDurationTicks;
			toil.AddFinishAction(delegate
			{
				JoyUtility.TryGainRecRoomThought(this.pawn);
			});
			yield return toil;
			yield break;
		}

		public override object[] TaleParameters()
		{
			return new object[]
			{
				this.pawn,
				base.TargetA.Thing.def
			};
		}
	}
	[DefOf]
	public class MyDefs
	{
		static MyDefs()
		{
			DefOfHelper.EnsureInitializedInCtor(typeof(MyDefs));
		}

		public static ThoughtDef WonSlotMachineGame;
		public static SoundDef MapOnlyDragSlider;
		public static SoundDef MapOnlyTinyBell;
		public static ThingDef SlotMachine;
		public static ThingDef Mote_ReelBlurA;
		public static ThingDef Mote_ReelBlurB;
		public static ThingDef Mote_ReelBlurC;
		public static ThingDef Mote_ReelBlurD;
		public static ThingDef Mote_ReelBlurE;
		public static ThingDef Mote_ReelBlurF;
		public static ThingDef Mote_ReelBlurG;
		public static ThingDef Mote_ReelBlurH;
		public static ThingDef Mote_ReelBlurI;
		public static ThingDef Mote_ReelBlurJ;
		public static ThingDef Mote_ReelOutcomeA;
		public static ThingDef Mote_ReelOutcomeB;
	}
}