PrefabFiles = {
	"oceancurrent_tracker",
}

Assets = {
	Asset("IMAGE", "images/oceancurrentnoise.tex"),
	Asset("IMAGE", "images/mini_currentnoise.tex"),
	
	Asset("IMAGE", "images/oceancurrentedge.tex"),
	Asset("IMAGE", "images/oceancurrentcorner.tex"),
	Asset("IMAGE", "images/oceancurrentendcap.tex"),

	Asset("SHADER", "shaders/ocean_current.ksh"),
}

local debug = GLOBAL.debug
local ROAD_STRIPS = GLOBAL.ROAD_STRIPS
local resolvefilepath = GLOBAL.resolvefilepath
local TheSim = GLOBAL.TheSim
local TUNING = GLOBAL.TUNING

require("scheduler")

GLOBAL.OceanCurrents = nil

TUNING.OCEAN_CURRENTS = {
	MIN_SPACING = 20*20,
	CURRENT_SIZE = GetModConfigData("CURRENTSIZE"),
	CURRENT_STRENGTH = GetModConfigData("CURRENTSTRENGTH"),
}

GLOBAL.OCEAN_CURRENT_PARAMETERS =
{
	NUM_SUBDIVISIONS_PER_SEGMENT = 5,
	MIN_WIDTH = 2,
	MAX_WIDTH = 3,
	MIN_EDGE_WIDTH = TUNING.OCEAN_CURRENTS.CURRENT_SIZE,
	MAX_EDGE_WIDTH = TUNING.OCEAN_CURRENTS.CURRENT_SIZE + 5,
	WIDTH_JITTER_SCALE=1,
}

local function PositionTaken(currentstable, x, y)
	for k, v in pairs(currentstable) do
		local distsq = (v[1] - x)*(v[1] - x) + (v[2] - y)*(v[2] - y)
		if distsq <= TUNING.OCEAN_CURRENTS.MIN_SPACING then
			return true
		end
	end
	return false
end

local function findNearestIndex(thisPoint, listToSearch)
    local nearestDistSquared = math.huge
    local nearestIndex
    for i=1, #listToSearch, 1 do
        local point2 = listToSearch[i]
        local distsq = (thisPoint[1] - point2[1])*(thisPoint[1] - point2[1]) + (thisPoint[2] - point2[2])*(thisPoint[2] - point2[2])
        if (distsq < nearestDistSquared) then
            nearestDistSquared = distsq
            nearestIndex=i
        end
    end
    return nearestIndex, nearestDistSquared
end

local function GetUpValue(func, varname)
	local i = 1
	local n, v = debug.getupvalue(func, 1)
	while v ~= nil do
		if n == varname then
			return v
		end
		i = i + 1
		n, v = debug.getupvalue(func, i)
	end
end

local function ReplaceUpValue(func, varname, newvalue)
	local i = 1
	local n, v = debug.getupvalue(func, 1)
	while v ~= nil do
		if n == varname then
			debug.setupvalue(func, i, newvalue)
			return
		end
		i = i + 1
		n, v = debug.getupvalue(func, i)
	end
end

AddSimPostInit(function()
	local pref = GLOBAL.Prefabs["schoolherd_oceanfish_medium_1"]
	if pref ~= nil then
		local _fn = pref.fn
		local __fn = GetUpValue(_fn, "fn")
		if __fn then
			local _setupnavs = GetUpValue(__fn, "setupnavs")
			if _setupnavs then
				local function setupnavs(inst, ...)
					local current = GLOBAL.GetClosestInstWithTag("oceancurrent", inst, 30)
					if current ~= nil and current.key ~= nil then
						if not inst.components.knownlocations:GetLocation("nav1") then        
							local origin = current:GetPosition()
							inst.components.knownlocations:RememberLocation("nav1", origin)

							for i = 2, 6 do
								local nextcurrent = current.currentgroup[current.key - (i - 1)]
								
								if nextcurrent ~= nil then
									local swim_x, swim_y, swim_z = nextcurrent.Transform:GetWorldPosition()
									inst.components.knownlocations:RememberLocation("nav"..i, GLOBAL.Vector3(swim_x, 0, swim_z))
								end
							end
					
							inst.currentnav = 2
						end
					end
					
					_setupnavs(inst, ...)
				end
		
				ReplaceUpValue(__fn, "setupnavs", setupnavs)
			end
		end
	end
	
	local squidpref = GLOBAL.Prefabs["squidherd"]
	if squidpref ~= nil then
		local _fn = squidpref.fn
		local _setupnavs = GetUpValue(_fn, "setupnavs")
		if _setupnavs then
			local function setupnavs(inst, ...)
				local current = GLOBAL.GetClosestInstWithTag("oceancurrent", inst, 30)
				if current ~= nil and current.key ~= nil then
					if not inst.components.knownlocations:GetLocation("nav1") then        
						local origin = current:GetPosition()
						inst.components.knownlocations:RememberLocation("nav1", origin)

						for i = 2, 6 do
							local nextcurrent = current.currentgroup[current.key - (i - 1)]
							
							if nextcurrent ~= nil then
								local swim_x, swim_y, swim_z = nextcurrent.Transform:GetWorldPosition()
								inst.components.knownlocations:RememberLocation("nav"..i, GLOBAL.Vector3(swim_x, 0, swim_z))
							end
						end
				
						inst.currentnav = 1
					end
				end
				
				_setupnavs(inst, ...)
			end
		
			ReplaceUpValue(_fn, "setupnavs", setupnavs)
		end
	end
	--
	local function GetSaveData(success, str)
        if success and str ~= nil and #str > 0 then
            local success, savedata = GLOBAL.RunInSandbox(str)
            if success and savedata ~= nil and savedata.map.ocean_currents and GLOBAL.GetTableSize(savedata) > 0 then
                GLOBAL.OceanCurrents = savedata.map.ocean_currents
            end
        end
    end

    local slot = GLOBAL.ShardGameIndex:GetSlot()
    local shard = GLOBAL.ShardGameIndex:GetShard()
    local session_id = GLOBAL.ShardGameIndex:GetSession()

    if session_id then
        if slot and shard and not GLOBAL.ShardGameIndex:GetServerData().use_legacy_session_path then
            local file = GLOBAL.TheNet:GetWorldSessionFileInClusterSlot(slot, shard, session_id)
            if file ~= nil then
                TheSim:GetPersistentStringInClusterSlot(slot, shard, file, function(success, str)
                    GetSaveData(success, str)
                end)
            end
        else
            local file = GLOBAL.TheNet:GetWorldSessionFile(session_id)
            if file ~= nil then
                TheSim:GetPersistentString(file, function(success, str)
                    GetSaveData(success, str)
                end)
            end
        end
    end

	if GLOBAL.OceanCurrents == nil and GLOBAL.TheWorld and GLOBAL.TheWorld:HasTag("forest") then
		GLOBAL.OceanCurrents = {}
		GLOBAL.OceanCurrents[1] = {}
	
		local currentstable = {}
	
		local offs =
		{
				{-1,-1},
			{-1, 0},	{1, 0},
				{0, 1},
		}
		local width, height = GLOBAL.TheWorld.Map:GetSize()
		for i = 0, width, 1 do
			for j = 0, height, 1 do
				for k = 1, #offs, 1 do
					local tile_x = i + offs[k][1]
					local tile_y = j + offs[k][2]
					local x, y, z = GLOBAL.TheWorld.Map:GetTileCenterPoint(tile_x, tile_y)
					local x2, y2, z2 = GLOBAL.TheWorld.Map:GetTileCenterPoint(i, j)
					if x ~= nil and y ~= nil and z ~= nil and x2 ~= nil and y2 ~= nil and z2 ~= nil then
						local ocean = GLOBAL.TheWorld.Map:GetTileAtPoint(x, y, z)
						local ground = GLOBAL.TheWorld.Map:GetTileAtPoint(x2, y2, z2)
						if ground == GROUND.OCEAN_SWELL and ocean == GROUND.OCEAN_COASTAL then
							if GLOBAL.TheWorld.Map:IsOceanAtPoint(x, 0, z) and (not PositionTaken(currentstable, x, z)) then
								table.insert(currentstable, {x, z})
							end
						end
					end
				end
			end
		end
	
		local orderedList = {}
	
		local firstpos = currentstable[1]
		table.insert(orderedList, {firstpos[1], firstpos[2]})
		table.remove(currentstable, 1)
		
		while (#currentstable > 0) do
			local nearestIndex, nearestDistSquared = findNearestIndex(orderedList[#orderedList], currentstable)
			local pos = currentstable[nearestIndex]
			if nearestDistSquared < 250*250 then
				table.insert(orderedList, {pos[1], pos[2]})
			end
			table.remove(currentstable, nearestIndex)
		end

		GLOBAL.OceanCurrents[1] = orderedList
	end
	
	if GLOBAL.OceanCurrents ~= nil then
		for k, road_data in pairs( GLOBAL.OceanCurrents ) do
			GLOBAL.RoadManager:BeginRoad()
			
			for i = 2, #road_data do
				local ctrl_pt = road_data[i]
				GLOBAL.RoadManager:AddControlPoint( ctrl_pt[1], ctrl_pt[2] )
				GLOBAL.scheduler:ExecuteInTime(i/30, function()
					local oceancurrent_tracker = GLOBAL.SpawnPrefab("oceancurrent_tracker")
					oceancurrent_tracker.Transform:SetPosition(ctrl_pt[1], 0, ctrl_pt[2])
					
					if GLOBAL.TheWorld ~= nil then
						if GLOBAL.TheWorld.oceancurrents == nil then
							GLOBAL.TheWorld.oceancurrents = {}
						end
						if GLOBAL.TheWorld.oceancurrents[k] == nil then
							GLOBAL.TheWorld.oceancurrents[k] = {}
						end
						table.insert(GLOBAL.TheWorld.oceancurrents[k], oceancurrent_tracker)
						oceancurrent_tracker.currentgroup = GLOBAL.TheWorld.oceancurrents[k]
					end
				end)
			end
			
			for k, v in pairs( ROAD_STRIPS ) do
				GLOBAL.RoadManager:SetStripEffect( v, resolvefilepath("shaders/ocean_current.ksh") )
			end
			GLOBAL.RoadManager:SetStripTextures( ROAD_STRIPS.EDGES,	resolvefilepath("images/oceancurrentedge.tex"),		resolvefilepath("images/oceancurrentnoise.tex") ,		resolvefilepath("images/mini_currentnoise.tex") )
			GLOBAL.RoadManager:SetStripTextures( ROAD_STRIPS.CENTER,	resolvefilepath("images/square.tex"),		resolvefilepath("images/oceancurrentnoise.tex") ,		resolvefilepath("images/mini_currentnoise.tex") )
			GLOBAL.RoadManager:SetStripTextures( ROAD_STRIPS.CORNERS,	resolvefilepath("images/oceancurrentcorner.tex"),	resolvefilepath("images/oceancurrentnoise.tex") ,		resolvefilepath("images/mini_currentnoise.tex") )
			GLOBAL.RoadManager:SetStripTextures( ROAD_STRIPS.ENDS,		resolvefilepath("images/oceancurrentendcap.tex"),	resolvefilepath("images/oceancurrentnoise.tex"),		resolvefilepath("images/mini_currentnoise.tex")  )

			GLOBAL.RoadManager:GenerateVB(
				GLOBAL.OCEAN_CURRENT_PARAMETERS.NUM_SUBDIVISIONS_PER_SEGMENT,
				0, 0,
				GLOBAL.OCEAN_CURRENT_PARAMETERS.MIN_EDGE_WIDTH, GLOBAL.OCEAN_CURRENT_PARAMETERS.MAX_EDGE_WIDTH,
				0, false )
		end
		GLOBAL.RoadManager:GenerateQuadTree()
	end
end)

local _DataDumper = GLOBAL.DataDumper
function GLOBAL.DataDumper(value, varname, fastmode, ident, ...)
	if value ~= nil and value.map ~= nil then
		value.map.ocean_currents = GLOBAL.OceanCurrents --Hornet: Save the ocean currents
	end
	
	return _DataDumper(value, varname, fastmode, ident, ...)
end

AddComponentPostInit("locomotor", function(self) --Hornet: slightly hacky way to make sure ocean current doesn't give speed boost to players, since it's considered a road (duh)
	local _FasterOnRoad = self.FasterOnRoad
	
	self.FasterOnRoad = function(self, ...)
		local x, y, z = self.inst.Transform:GetWorldPosition()
		
		if GLOBAL.TheWorld.Map:IsOceanTileAtPoint(x, 0, z) then
			return false
		end

		if _FasterOnRoad ~= nil then
			return _FasterOnRoad(self, ...)
		end
	end
end)

local offsets = {
	{0, 0},
	{0, -2},
	{0, 2},
	{-2, 0},
	{2, 0},
}

local function IsOnCurrent(inst)
	local x, y, z = inst.Transform:GetWorldPosition()
	
	for k, v in pairs(offsets) do
		if GLOBAL.RoadManager:IsOnRoad(x+v[1], 0, z+v[2]) then
			return true
		end
	end
end

AddComponentPostInit("boatphysics", function(self)
	local _GetMaxVelocity = self.GetMaxVelocity
	
	function self:GetMaxVelocity(...)
		local max_vel = _GetMaxVelocity(self, ...)

		if IsOnCurrent(self.inst) then
			max_vel = max_vel * 1.34
		end
		
		return max_vel
	end
end)

AddPrefabPostInit("burnable_locator_medium", function(inst)
	inst:AddTag("burnable_locator")
end)

function GLOBAL.SetControlPoint()
	local x, y, z = GLOBAL.ThePlayer.Transform:GetWorldPosition()
	
	local oceancurrent_tracker = GLOBAL.SpawnPrefab("oceancurrent_tracker")
	oceancurrent_tracker:DoTaskInTime(0, function()
		oceancurrent_tracker.Transform:SetPosition(x, 0, z)
	end)
	
	if GLOBAL.TheWorld ~= nil then
		if GLOBAL.TheWorld.oceancurrents == nil then
			GLOBAL.TheWorld.oceancurrents = {}
		end
		if GLOBAL.TheWorld.oceancurrents[69] == nil then
			GLOBAL.TheWorld.oceancurrents[69] = {}
		end
		table.insert(GLOBAL.TheWorld.oceancurrents[69], oceancurrent_tracker)
		oceancurrent_tracker.currentgroup = GLOBAL.TheWorld.oceancurrents[69]
	end
	
	GLOBAL.RoadManager:AddControlPoint(x, z)
end

function GLOBAL.SpawnRoad()
	for k, v in pairs( ROAD_STRIPS ) do
		GLOBAL.RoadManager:SetStripEffect( v, resolvefilepath("shaders/ocean_current.ksh") )
	end
	
	GLOBAL.RoadManager:SetStripTextures( ROAD_STRIPS.EDGES,	resolvefilepath("images/oceancurrentedge.tex"),		resolvefilepath("images/oceancurrentnoise.tex") ,		resolvefilepath("images/mini_currentnoise.tex") )
	GLOBAL.RoadManager:SetStripTextures( ROAD_STRIPS.CENTER,	resolvefilepath("images/square.tex"),		resolvefilepath("images/oceancurrentnoise.tex") ,		resolvefilepath("images/mini_currentnoise.tex") )
	GLOBAL.RoadManager:SetStripTextures( ROAD_STRIPS.CORNERS,	resolvefilepath("images/oceancurrentcorner.tex"),	resolvefilepath("images/oceancurrentnoise.tex") ,		resolvefilepath("images/mini_currentnoise.tex") )
	GLOBAL.RoadManager:SetStripTextures( ROAD_STRIPS.ENDS,		resolvefilepath("images/oceancurrentendcap.tex"),	resolvefilepath("images/oceancurrentnoise.tex") ,		resolvefilepath("images/mini_currentnoise.tex") )

	GLOBAL.RoadManager:GenerateVB(
			GLOBAL.OCEAN_CURRENT_PARAMETERS.NUM_SUBDIVISIONS_PER_SEGMENT,
			0, 0,
			GLOBAL.OCEAN_CURRENT_PARAMETERS.MIN_EDGE_WIDTH, GLOBAL.OCEAN_CURRENT_PARAMETERS.MAX_EDGE_WIDTH,
			0, false )
end