'''
Copyright (C) 2022 vfxguide
realvfxguide@gmail.com

Created by VFXGuide

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
'''
bl_info = {
    "name": "OCD",
    "author": "VFXGuide",
    "version": (1, 7, 2),
    "blender": (2, 93, 0),
    "location": "View3D -> N-Panel -> OCD",
    "description": "One Click Damage",
    "warning": "",
    "doc_url": "",
    "category": "Object",
}

import bpy
import random
from bpy import context
from bpy.types import Operator, Panel, PropertyGroup, Object
from bpy.props import FloatProperty, PointerProperty, BoolProperty, EnumProperty
from bpy.utils import register_class, unregister_class
from typing import List
import math
import time
from numpy import interp

def assign_materials(mat_names):
    mat_1 = bpy.data.materials.get(mat_names[0]) or bpy.data.materials.new(name=mat_names[0])
    mat_2 = bpy.data.materials.get(mat_names[1]) or bpy.data.materials.new(name=mat_names[1])

    mat_1.use_nodes = True
    mat_2.use_nodes = True

    obj = bpy.context.object
    if obj.active_material is None:
        if obj.data.materials:
            obj.data.materials[0] = mat_1
            obj.data.materials.append(mat_2)
        else:
            obj.data.materials.append(mat_1)
            obj.data.materials.append(mat_2)
        bpy.ops.object.editmode_toggle()
        obj.active_material_index = 0
        bpy.ops.object.material_slot_assign()
        bpy.ops.mesh.select_all(action='INVERT')
        obj.active_material_index = 1
        bpy.ops.object.material_slot_assign()
    else:
        bpy.ops.object.material_slot_add()
        idx = obj.active_material_index
        obj.material_slots[idx].material = mat_2
        bpy.ops.object.editmode_toggle()
        bpy.ops.mesh.select_all(action='INVERT')
        bpy.ops.object.material_slot_assign()

    bpy.ops.mesh.region_to_loop()
    bpy.ops.mesh.mark_sharp()
    bpy.ops.object.editmode_toggle()

def update_func(self, context):
    name = context.object.name + "_temp"
    props = context.scene.dmgamnt
    scale_amount = props.scale_amount
    sourceName = bpy.context.object.name
    sourceObj = bpy.data.objects[sourceName]
    #sourceObj dimensions
    sourceX = sourceObj.dimensions.x
    sourceY = sourceObj.dimensions.y
    sourceZ = sourceObj.dimensions.z
    x = props.damage_amount

    sourceDim = [sourceX, sourceY, sourceZ]
    #sourceObj multiplier
    multiplier = max(sourceDim)
    
    #strength graph sin
    minS = 0.475
    maxS = 1 
    xSq = math.sqrt(x)/4.2
    sin = math.sin(xSq) * math.sin(xSq)
    amnt = sin*100
    StrengthGraph = interp(amnt, [0, 100], [minS, maxS]) * multiplier
										
    #exponential
    minM = 0.25
    maxM = 0.94
    exp1 = (-x+100)/21.8
    exp2 = math.exp(exp1) * -1 + 100
    MidGraph = interp(exp2, [0, 100], [minM, maxM]) 
    
    #smooth graph sin
    minSm = 0.45 
    maxSm = 0.75
    ins2 = xSq * 0.66
    sinus2 = math.sin(ins2) * math.sin(ins2)
    amount2 = sinus2*100
    SmoothGraph = interp(amount2, [0, 100], [minSm, maxSm]) 

    sc = scale_amount
    minSc = 0.01 * multiplier
    maxSc = 0.5 * multiplier
    xSc = (maxSc - minSc) / 100
    #props.noise_type
    bpy.data.objects[name].modifiers["OCD_Displace"].mid_level = MidGraph
    bpy.data.objects[name].modifiers["OCD_Displace"].strength = StrengthGraph / 10
    bpy.data.objects[name].modifiers["OCD_Smooth"].factor = SmoothGraph
    bpy.data.textures["OCD_texture"].noise_scale = xSc * sc + minSc
    bpy.data.textures["OCD_texture"].type = props.noise_type

    has_noise_basis = hasattr(bpy.data.textures["OCD_texture"], "noise_basis")

    if bpy.data.textures["OCD_texture"].type == 'MUSGRAVE':
        bpy.data.textures["OCD_texture"].contrast = 1
        bpy.data.textures["OCD_texture"].intensity = 1
        bpy.data.textures["OCD_texture"].noise_basis = 'VORONOI_CRACKLE'

    if bpy.data.textures["OCD_texture"].type == 'CLOUDS':
        bpy.data.textures["OCD_texture"].contrast = 1.5
        bpy.data.textures["OCD_texture"].intensity = 1
        bpy.data.textures["OCD_texture"].noise_basis = 'VORONOI_CRACKLE'

    if bpy.data.textures["OCD_texture"].type == 'MARBLE':
        bpy.data.textures["OCD_texture"].contrast = 1.5
        bpy.data.textures["OCD_texture"].intensity = 1.5
        bpy.data.textures["OCD_texture"].turbulence = 5
        bpy.data.textures["OCD_texture"].noise_basis = 'IMPROVED_PERLIN'

    if bpy.data.textures["OCD_texture"].type == 'VORONOI':
        bpy.data.textures["OCD_texture"].contrast = 0.5
        bpy.data.textures["OCD_texture"].intensity = 1.4
        if has_noise_basis:
            bpy.data.textures["OCD_texture"].noise_basis = 'VORONOI_CRACKLE'
        else:
            pass
    if bpy.data.textures["OCD_texture"].type == 'WOOD':
        bpy.data.textures["OCD_texture"].contrast = 1.5
        bpy.data.textures["OCD_texture"].intensity = 2
        bpy.data.textures["OCD_texture"].turbulence = 15
        bpy.data.textures["OCD_texture"].noise_basis = 'IMPROVED_PERLIN'
    
def damage_on(context, self):
    props = context.scene.dmgamnt
    scale_amount = props.scale_amount
    sc = scale_amount
    damage_amount = props.damage_amount

    sourceName = bpy.context.object.name
    sourceObj = bpy.data.objects[sourceName]
    meshName = sourceObj.data.name 

    if meshName == sourceName:
        pass
    else:
        if meshName in bpy.data.objects:
            bpy.data.objects[sourceName].data.name = bpy.context.object.name
        else:
            bpy.context.object.name = bpy.data.objects[sourceName].data.name
    
    sourceName = bpy.context.object.name
    sourceObj = bpy.data.objects[sourceName]
    x = props.damage_amount
    #sourceObj dimensions
    sourceX = sourceObj.dimensions.x
    sourceY = sourceObj.dimensions.y
    sourceZ = sourceObj.dimensions.z
    sourceDim = [sourceX, sourceY, sourceZ]
    #sourceObj multiplier
    multiplier = max(sourceDim)

    #strength graph sin
    minS = 0.475
    maxS = 1 
    xSq = math.sqrt(x)/4.2
    sin = math.sin(xSq) * math.sin(xSq)
    amnt = sin*100
    StrengthGraph = interp(amnt, [0, 100], [minS, maxS]) * multiplier

    #exponential
    minM = 0.25
    maxM = 0.94
    exp1 = (-x+100)/21.8
    exp2 = math.exp(exp1) * -1 + 100
    MidGraph = interp(exp2, [0, 100], [minM, maxM]) 
    
    #smooth graph sin
    minSm = 0.45 
    maxSm = 0.75
    ins2 = xSq * 0.66
    sinus2 = math.sin(ins2) * math.sin(ins2)
    amount2 = sinus2*100
    SmoothGraph = interp(amount2, [0, 100], [minSm, maxSm]) 

    sc = scale_amount
    minSc = 0.01 * multiplier
    maxSc = 0.5 * multiplier
    xSc = (maxSc - minSc) / 100
    
    voxel_size = context.preferences.addons[__name__].preferences.ocd_voxel_size / 1000
    damaged_col = "OCD_temp"
    if damaged_col not in bpy.data.collections:      
        damaged_collection = bpy.context.blend_data.collections.new(name=damaged_col)
        bpy.context.collection.children.link(damaged_collection)
    else:
        damaged_collection = bpy.data.collections[damaged_col]      
    damaged_collection.hide_viewport = True
    damaged_collection.hide_render = True
    
    sourceName = bpy.context.object.name
    sourceMeshName = bpy.data.objects[sourceName].data.name
    bpy.ops.object.transform_apply(location = False, rotation = False, scale = True)
    bpy.ops.object.duplicate_move(OBJECT_OT_duplicate={"linked":False, "mode":'TRANSLATION'})
    bpy.context.object.name = sourceName + "_dmg"
    bpy.ops.object.editmode_toggle()
    bpy.ops.mesh.select_all(action='DESELECT')
    bpy.ops.mesh.select_mode(use_extend=False, use_expand=False, type='EDGE')
    bpy.ops.mesh.edges_select_sharp(sharpness=0.436332)
    bpy.ops.mesh.mark_sharp()
    bpy.ops.object.editmode_toggle()
    damageName = bpy.context.object.name
    damageObj = bpy.data.objects[damageName]
    
    bpy.context.object.name = damageName
    bpy.data.objects[damageName].data.name = damageName
    bpy.ops.object.select_all(action='DESELECT')
    bpy.data.objects[sourceName].select_set(True)
    bpy.context.view_layer.objects.active = sourceObj
    bpy.ops.object.delete(use_global=False)

    bpy.data.objects[damageName].select_set(True)
    bpy.context.view_layer.objects.active = damageObj

    bpy.ops.object.duplicate_move(OBJECT_OT_duplicate={"linked":False, "mode":'TRANSLATION'})
    bpy.context.object.name = damageName + "_temp"

    targetName = bpy.context.object.name
    targetObj = bpy.data.objects[targetName]
    
    bpy.ops.object.editmode_toggle()
    bpy.ops.mesh.select_mode(use_extend=False, use_expand=False, type='FACE')

    bpy.ops.mesh.select_all(action='DESELECT')
    bpy.ops.mesh.select_all(action='SELECT')
    
    bpy.ops.mesh.normals_make_consistent(inside=False)

    bpy.ops.mesh.select_all(action='DESELECT')
    bpy.ops.mesh.select_all(action='SELECT')
    
    bpy.ops.object.editmode_toggle()

    for ob in bpy.context.object.users_collection:
        ob.objects.unlink(targetObj)
        
    bpy.data.collections[damaged_col].objects.link(targetObj)

    bpy.context.view_layer.objects.active = targetObj
     
    bpy.context.object.modifiers.new("OCD_remesh", type='REMESH')
    bpy.context.object.modifiers["OCD_remesh"].voxel_size = voxel_size * multiplier
    bpy.context.object.modifiers["OCD_remesh"].use_smooth_shade = True

    bpy.data.objects[targetName].modifiers.new("OCD_Smooth", type='SMOOTH')
    bpy.data.objects[targetName].modifiers["OCD_Smooth"].factor = SmoothGraph
    bpy.data.objects[targetName].modifiers["OCD_Smooth"].iterations = 30
    
    bpy.context.object.modifiers.new("OCD_Displace", type='DISPLACE')
    bpy.context.object.modifiers["OCD_Displace"].strength = StrengthGraph / 10
    bpy.context.object.modifiers["OCD_Displace"].mid_level = MidGraph
    
    if "OCD_texture" not in bpy.data.textures:
        OCD_texture = bpy.data.textures.new(name = "OCD_texture", type = 'MUSGRAVE')
        has_noise_basis = hasattr(bpy.data.textures["OCD_texture"], "noise_basis")
        if has_noise_basis:
            bpy.data.textures["OCD_texture"].noise_basis = 'VORONOI_CRACKLE'
        else:
            pass
        bpy.data.textures["OCD_texture"].musgrave_type = 'HYBRID_MULTIFRACTAL'
        bpy.data.textures["OCD_texture"].type = props.noise_type
        bpy.data.textures["OCD_texture"].noise_scale = xSc * sc + minSc
        
    else:     
        OCD_texture = bpy.data.textures["OCD_texture"]

        bpy.data.textures["OCD_texture"].type = props.noise_type
        has_noise_basis = hasattr(bpy.data.textures["OCD_texture"], "noise_basis")

        if bpy.data.textures["OCD_texture"].type == 'MUSGRAVE':
            bpy.data.textures["OCD_texture"].contrast = 1
            bpy.data.textures["OCD_texture"].intensity = 1
            bpy.data.textures["OCD_texture"].noise_basis = 'VORONOI_CRACKLE'
        if bpy.data.textures["OCD_texture"].type == 'CLOUDS':
            bpy.data.textures["OCD_texture"].contrast = 1.5
            bpy.data.textures["OCD_texture"].intensity = 1
            bpy.data.textures["OCD_texture"].noise_basis = 'VORONOI_CRACKLE'
        if bpy.data.textures["OCD_texture"].type == 'MARBLE':
            bpy.data.textures["OCD_texture"].contrast = 1.5
            bpy.data.textures["OCD_texture"].intensity = 1.5
            bpy.data.textures["OCD_texture"].turbulence = 5
            bpy.data.textures["OCD_texture"].noise_basis = 'IMPROVED_PERLIN'
        if bpy.data.textures["OCD_texture"].type == 'VORONOI':
            bpy.data.textures["OCD_texture"].contrast = 0.5
            bpy.data.textures["OCD_texture"].intensity = 1.4
            if has_noise_basis:
                bpy.data.textures["OCD_texture"].noise_basis = 'VORONOI_CRACKLE'
            else:
                pass
        if bpy.data.textures["OCD_texture"].type == 'WOOD':
            bpy.data.textures["OCD_texture"].contrast = 1.5
            bpy.data.textures["OCD_texture"].intensity = 2
            bpy.data.textures["OCD_texture"].turbulence = 15
            bpy.data.textures["OCD_texture"].noise_basis = 'IMPROVED_PERLIN'

        bpy.data.textures["OCD_texture"].noise_scale = xSc * sc + minSc
        
    bpy.context.object.modifiers.new("OCD_remesh_02", type='REMESH')
    bpy.context.object.modifiers["OCD_remesh_02"].voxel_size = voxel_size * multiplier
    bpy.context.object.modifiers["OCD_remesh_02"].use_smooth_shade = True
    
    bpy.context.object.modifiers["OCD_Displace"].texture = bpy.data.textures['OCD_texture']
    bpy.context.object.modifiers["OCD_Displace"].texture_coords = 'GLOBAL'
    
    bpy.context.view_layer.objects.active = damageObj
    bpy.context.active_object.select_set(state=True)
    
    props2 = bpy.context.object.stored_value
    props2.ocd_strength = bpy.data.collections[damaged_col].objects[targetName].modifiers["OCD_Displace"].strength
    props2.ocd_mid_level = bpy.data.collections[damaged_col].objects[targetName].modifiers["OCD_Displace"].mid_level
    props2.ocd_smooth = bpy.data.collections[damaged_col].objects[targetName].modifiers["OCD_Smooth"].factor
    props2.ocd_noise_scale = bpy.data.textures["OCD_texture"].noise_scale
    props2.ocd_noise_type = bpy.data.textures["OCD_texture"].type
    if bpy.data.textures["OCD_texture"].type in ["MUSGRAVE", "CLOUDS", "WOOD", "MARBLE"]:
        props2.ocd_noise_basis = bpy.data.textures["OCD_texture"].noise_basis
    else:
        pass
    if bpy.data.textures["OCD_texture"].type not in ["MUSGRAVE", "CLOUDS", "VORONOI"]:
        props2.ocd_turbulence = bpy.data.textures["OCD_texture"].turbulence
    else:
        pass
    props2.ocd_contrast = bpy.data.textures["OCD_texture"].contrast
    props2.ocd_brightness = bpy.data.textures["OCD_texture"].intensity
    props2.ocd_voxel_size = bpy.data.collections[damaged_col].objects[targetName].modifiers["OCD_remesh"].voxel_size
    props2.ocd_voxel_size_02 = bpy.data.collections[damaged_col].objects[targetName].modifiers["OCD_remesh_02"].voxel_size

    bpy.ops.object.editmode_toggle()
    bpy.ops.mesh.select_mode(use_extend=False, use_expand=False, type='FACE')

    bpy.ops.mesh.select_all(action='DESELECT')
    bpy.ops.mesh.select_all(action='SELECT')
   
    bpy.ops.mesh.normals_make_consistent(inside=False)
    bpy.ops.object.editmode_toggle()

    bpy.context.object.modifiers.new("OCD_Boolean", type='BOOLEAN')
    bpy.context.object.modifiers["OCD_Boolean"].solver = 'FAST'
    bpy.context.object.modifiers["OCD_Boolean"].object = targetObj
    bpy.context.object.modifiers["OCD_Boolean"].operation = 'INTERSECT'
    bpy.context.object.modifiers.new("OCD_Preview", type='EDGE_SPLIT')
    bpy.context.object.modifiers["OCD_Preview"].split_angle = 0.261799

    bpy.context.object.data.use_auto_smooth = True
    bpy.context.object.data.auto_smooth_angle = 1.0472
    
    obj = context.active_object
    if len(obj.modifiers) == 0:
        self.report({'WARNING'}, "Not a single modifier to Expand/Collapse")
        return {'CANCELLED'}
    is_close = any(mod.show_expanded for mod in obj.modifiers)
    for mod in obj.modifiers:
        mod.show_expanded = not is_close
    return {'FINISHED'}
 
def ctrl_damage(context, self):
    props = context.scene.dmgamnt
    x = props.damage_amount
    scale_amount = props.scale_amount
    sc = scale_amount
    sourceName = bpy.context.object.name
    sourceObj = bpy.data.objects[sourceName]

    #sourceObj dimensions
    sourceX = sourceObj.dimensions.x
    sourceY = sourceObj.dimensions.y
    sourceZ = sourceObj.dimensions.z

    sourceDim = [sourceX, sourceY, sourceZ]
    #sourceObj multiplier
    multiplier = max(sourceDim)

    #strength graph sin
    minS = 0.475
    maxS = 1 
    xSq = math.sqrt(x)/4.2
    sin = math.sin(xSq) * math.sin(xSq)
    amnt = sin*100
    StrengthGraph = interp(amnt, [0, 100], [minS, maxS]) * multiplier

    #exponential
    minM = 0.25
    maxM = 0.94
    exp1 = (-x+100)/21.8
    exp2 = math.exp(exp1) * -1 + 100
    MidGraph = interp(exp2, [0, 100], [minM, maxM]) 
    
    #smooth graph sin
    minSm = 0.45 
    maxSm = 0.75
    ins2 = xSq * 0.66
    sinus2 = math.sin(ins2) * math.sin(ins2)
    amount2 = sinus2*100
    SmoothGraph = interp(amount2, [0, 100], [minSm, maxSm]) 

    minSc = 0.01 * multiplier
    maxSc = 0.50 * multiplier
    xSc = (maxSc - minSc) / 100
    
    voxel_size = context.preferences.addons[__name__].preferences.ocd_voxel_size / 1000 

    damaged_col = "OCD_temp"
        
    if damaged_col not in bpy.data.collections:      
        damaged_collection = bpy.context.blend_data.collections.new(name=damaged_col)
        bpy.context.collection.children.link(damaged_collection)
    else:
        damaged_collection = bpy.data.collections[damaged_col]  
    damaged_collection.hide_viewport = True
    damaged_collection.hide_render = True
    
    sourceName = bpy.context.object.name
    sourceMeshName = bpy.data.objects[sourceName].data.name

    bpy.context.object.name = sourceMeshName
    bpy.context.object.name = sourceName
    
    bpy.data.objects[sourceName].data.name = sourceName 
    
    bpy.ops.object.transform_apply(location = False, rotation = False, scale = True)
    
    bpy.ops.object.duplicate_move(OBJECT_OT_duplicate={"linked":False, "mode":'TRANSLATION'})
    bpy.context.object.name = sourceName + "_dmg"
    
    bpy.ops.object.editmode_toggle()
    bpy.ops.mesh.select_all(action='DESELECT')
    bpy.ops.mesh.select_mode(use_extend=False, use_expand=False, type='EDGE')
    bpy.ops.mesh.edges_select_sharp(sharpness=0.436332)
    bpy.ops.mesh.mark_sharp()
    bpy.ops.object.editmode_toggle()

    damageName = bpy.context.object.name
    damageObj = bpy.data.objects[damageName]
    
    bpy.context.object.name = damageName
    bpy.data.objects[damageName].data.name = damageName
    
    bpy.ops.object.select_all(action='DESELECT')
    bpy.data.objects[sourceName].select_set(True)
    bpy.context.view_layer.objects.active = sourceObj
    bpy.ops.object.delete(use_global=False)

    bpy.data.objects[damageName].select_set(True)
    bpy.context.view_layer.objects.active = damageObj

    bpy.ops.object.duplicate_move(OBJECT_OT_duplicate={"linked":False, "mode":'TRANSLATION'})
    bpy.context.object.name = damageName + "_temp"

    targetName = bpy.context.object.name
    targetObj = bpy.data.objects[targetName]
    
    bpy.ops.object.editmode_toggle()
    bpy.ops.mesh.select_mode(use_extend=False, use_expand=False, type='FACE')

    bpy.ops.mesh.select_all(action='DESELECT')
    bpy.ops.mesh.select_all(action='SELECT')
    
    bpy.ops.mesh.normals_make_consistent(inside=False)

    bpy.ops.mesh.select_all(action='DESELECT')
    bpy.ops.mesh.select_all(action='SELECT')
    
    bpy.ops.object.editmode_toggle()

    for ob in bpy.context.object.users_collection:
        ob.objects.unlink(targetObj)
    
    bpy.data.collections[damaged_col].objects.link(targetObj)

    bpy.context.view_layer.objects.active = targetObj
    
    bpy.context.object.modifiers.new("OCD_remesh", type='REMESH')
    bpy.context.object.modifiers["OCD_remesh"].voxel_size = voxel_size * multiplier
    bpy.context.object.modifiers["OCD_remesh"].use_smooth_shade = True

    bpy.data.objects[targetName].modifiers.new("OCD_Smooth", type='SMOOTH')
    bpy.data.objects[targetName].modifiers["OCD_Smooth"].factor = SmoothGraph
    bpy.data.objects[targetName].modifiers["OCD_Smooth"].iterations = 30
    
    bpy.context.object.modifiers.new("OCD_Displace", type='DISPLACE')
    bpy.context.object.modifiers["OCD_Displace"].strength = StrengthGraph / 10
    bpy.context.object.modifiers["OCD_Displace"].mid_level = MidGraph
    
    if "OCD_texture" not in bpy.data.textures:
        OCD_texture = bpy.data.textures.new(name = "OCD_texture", type = 'MUSGRAVE')
        has_noise_basis = hasattr(bpy.data.textures["OCD_texture"], "noise_basis")
        if has_noise_basis:
            bpy.data.textures["OCD_texture"].noise_basis = 'VORONOI_CRACKLE'
        else:
            pass
        bpy.data.textures["OCD_texture"].musgrave_type = 'HYBRID_MULTIFRACTAL'
        bpy.data.textures["OCD_texture"].type = props.noise_type
        bpy.data.textures["OCD_texture"].noise_scale = xSc * sc + minSc
        
    else:     
        OCD_texture = bpy.data.textures["OCD_texture"]

        bpy.data.textures["OCD_texture"].type = props.noise_type
        has_noise_basis = hasattr(bpy.data.textures["OCD_texture"], "noise_basis")

        if bpy.data.textures["OCD_texture"].type == 'MUSGRAVE':
            bpy.data.textures["OCD_texture"].contrast = 1
            bpy.data.textures["OCD_texture"].intensity = 1
            bpy.data.textures["OCD_texture"].noise_basis = 'VORONOI_CRACKLE'
        if bpy.data.textures["OCD_texture"].type == 'CLOUDS':
            bpy.data.textures["OCD_texture"].contrast = 1.5
            bpy.data.textures["OCD_texture"].intensity = 1
            bpy.data.textures["OCD_texture"].noise_basis = 'VORONOI_CRACKLE'
        if bpy.data.textures["OCD_texture"].type == 'MARBLE':
            bpy.data.textures["OCD_texture"].contrast = 1.5
            bpy.data.textures["OCD_texture"].intensity = 1.5
            bpy.data.textures["OCD_texture"].turbulence = 5
            bpy.data.textures["OCD_texture"].noise_basis = 'IMPROVED_PERLIN'
        if bpy.data.textures["OCD_texture"].type == 'VORONOI':
            bpy.data.textures["OCD_texture"].contrast = 0.5
            bpy.data.textures["OCD_texture"].intensity = 1.4
            if has_noise_basis:
                bpy.data.textures["OCD_texture"].noise_basis = 'VORONOI_CRACKLE'
            else:
                pass
        if bpy.data.textures["OCD_texture"].type == 'WOOD':
            bpy.data.textures["OCD_texture"].contrast = 1.5
            bpy.data.textures["OCD_texture"].intensity = 2
            bpy.data.textures["OCD_texture"].turbulence = 15
            bpy.data.textures["OCD_texture"].noise_basis = 'IMPROVED_PERLIN'

        bpy.data.textures["OCD_texture"].noise_scale = xSc * sc + minSc

    bpy.context.object.modifiers.new("OCD_remesh_02", type='REMESH')
    bpy.context.object.modifiers["OCD_remesh_02"].voxel_size = voxel_size * multiplier
    bpy.context.object.modifiers["OCD_remesh_02"].use_smooth_shade = True
   
    bpy.context.object.modifiers["OCD_Displace"].texture = bpy.data.textures['OCD_texture']
    bpy.context.object.modifiers["OCD_Displace"].texture_coords = 'GLOBAL'
    
    bpy.context.view_layer.objects.active = damageObj
    bpy.context.active_object.select_set(state=True)
    
    props2 = bpy.context.object.stored_value
    props2.ocd_strength = bpy.data.collections[damaged_col].objects[targetName].modifiers["OCD_Displace"].strength
    props2.ocd_mid_level = bpy.data.collections[damaged_col].objects[targetName].modifiers["OCD_Displace"].mid_level
    props2.ocd_smooth = bpy.data.collections[damaged_col].objects[targetName].modifiers["OCD_Smooth"].factor
    props2.ocd_noise_scale = bpy.data.textures["OCD_texture"].noise_scale
    props2.ocd_noise_type = bpy.data.textures["OCD_texture"].type
    if bpy.data.textures["OCD_texture"].type in ["MUSGRAVE", "CLOUDS", "WOOD", "MARBLE"]:
        props2.ocd_noise_basis = bpy.data.textures["OCD_texture"].noise_basis
    else:
        pass
    if bpy.data.textures["OCD_texture"].type not in ["MUSGRAVE", "CLOUDS", "VORONOI"]:
        props2.ocd_turbulence = bpy.data.textures["OCD_texture"].turbulence
    else:
        pass
    props2.ocd_contrast = bpy.data.textures["OCD_texture"].contrast
    props2.ocd_brightness = bpy.data.textures["OCD_texture"].intensity
    props2.ocd_voxel_size = bpy.data.collections[damaged_col].objects[targetName].modifiers["OCD_remesh"].voxel_size
    props2.ocd_voxel_size_02 = bpy.data.collections[damaged_col].objects[targetName].modifiers["OCD_remesh_02"].voxel_size

    bpy.ops.object.editmode_toggle()
    bpy.ops.mesh.select_mode(use_extend=False, use_expand=False, type='FACE')

    bpy.ops.mesh.select_all(action='DESELECT')
    bpy.ops.mesh.select_all(action='SELECT')
    
    bpy.ops.mesh.normals_make_consistent(inside=False)
    bpy.ops.object.editmode_toggle()

    bpy.context.object.modifiers.new("OCD_Boolean", type='BOOLEAN')
    bpy.context.object.modifiers["OCD_Boolean"].solver = 'FAST'
    bpy.context.object.modifiers["OCD_Boolean"].object = targetObj
    bpy.context.object.modifiers["OCD_Boolean"].operation = 'INTERSECT'
    bpy.context.object.modifiers.new("OCD_Preview", type='EDGE_SPLIT')
    bpy.context.object.modifiers["OCD_Preview"].split_angle = 0.261799

    bpy.context.object.data.use_auto_smooth = True
    bpy.context.object.data.auto_smooth_angle = 1.0472
    
    obj = context.active_object
    if len(obj.modifiers) == 0:
        self.report({'WARNING'}, "Not a single modifier to Expand/Collapse")
        return {'CANCELLED'}
    is_close = any(mod.show_expanded for mod in obj.modifiers)
    for mod in obj.modifiers:
        mod.show_expanded = not is_close
    
    active_obj = context.view_layer.objects.active
    
    for mod in active_obj.modifiers:
        if mod.name == "OCD_Boolean":
            bpy.ops.object.modifier_apply(modifier=mod.name)
    for mod in active_obj.modifiers:
        if mod.type == "EDGE_SPLIT":
            bpy.ops.object.modifier_remove(modifier=mod.name)   
    
    bpy.context.object.modifiers.new("OCD_WNormal", type='WEIGHTED_NORMAL')
    bpy.context.object.modifiers["OCD_WNormal"].keep_sharp = True
    
    #materials
    mat_names = ["OUTSIDE", "INSIDE"]
    assign_materials(mat_names)
    
    damageName = bpy.context.object.name
    sourceName = damageName[:-4]
    
    bpy.data.meshes[sourceName].use_fake_user = True
    
    collection_name = "OCD_temp"
    collection = bpy.data.collections[collection_name]

    meshes = set()
    for obj in [o for o in collection.objects if o.type == 'MESH']:
        meshes.add( obj.data )
        bpy.data.objects.remove( obj )
    for mesh in [m for m in meshes]:
        bpy.data.meshes.remove( mesh )
    collection = bpy.data.collections.get(collection_name)
    bpy.data.collections.remove(collection)
    return {'FINISHED'}   
    
def multobj_damage(context):
    props = context.scene.dmgamnt
    x = props.damage_amount
    scale_amount = props.scale_amount
    sc = scale_amount
    sourceName = bpy.context.object.name
    sourceObj = bpy.data.objects[sourceName]
    sourceX = sourceObj.dimensions.x
    sourceY = sourceObj.dimensions.y
    sourceZ = sourceObj.dimensions.z
    sourceDim = [sourceX, sourceY, sourceZ]

    #sourceObj multiplier
    multiplier = max(sourceDim)

    #strength graph sin
    minS = 0.475
    maxS = 1 
    xSq = math.sqrt(x)/4.2
    sin = math.sin(xSq) * math.sin(xSq)
    amnt = sin*100
    StrengthGraph = interp(amnt, [0, 100], [minS, maxS]) * multiplier

    #exponential
    minM = 0.25
    maxM = 0.94
    exp1 = (-x+100)/21.8
    exp2 = math.exp(exp1) * -1 + 100
    MidGraph = interp(exp2, [0, 100], [minM, maxM]) 
    
    #smooth graph sin
    minSm = 0.45 
    maxSm = 0.75
    ins2 = xSq * 0.66
    sinus2 = math.sin(ins2) * math.sin(ins2)
    amount2 = sinus2*100
    SmoothGraph = interp(amount2, [0, 100], [minSm, maxSm])
    
    minSc = 0.01 * multiplier
    maxSc = 0.50 * multiplier
    xSc = (maxSc - minSc) / 100
   
    voxel_size = context.preferences.addons[__name__].preferences.ocd_voxel_size / 1000

    for obj in bpy.context.selected_objects:
        bpy.context.view_layer.objects.active = obj
        damaged_col = "OCD_temp"
        if damaged_col not in bpy.data.collections:      
            damaged_collection = bpy.context.blend_data.collections.new(name=damaged_col)
            bpy.context.collection.children.link(damaged_collection)
        else:
            damaged_collection = bpy.data.collections[damaged_col]    
        damaged_collection.hide_viewport = True
        damaged_collection.hide_render = True

        sourceName = bpy.context.object.name

        meshName = bpy.data.objects[sourceName].data.name
        
        if meshName == sourceName:
            pass
        else:
            if meshName in bpy.data.objects:
                bpy.data.objects[sourceName].data.name = bpy.context.object.name
            else:
                bpy.context.object.name = bpy.data.objects[sourceName].data.name
        
        sourceName = bpy.context.object.name
        sourceObj = bpy.data.objects[sourceName]
        bpy.ops.object.select_all(action='DESELECT')
        
        bpy.data.objects[sourceName].select_set(True)
        bpy.context.view_layer.objects.active = sourceObj
        
        bpy.ops.object.transform_apply(location = False, rotation = False, scale = True)
        
        bpy.ops.object.duplicate_move(OBJECT_OT_duplicate={"linked":False, "mode":'TRANSLATION'})
        
        bpy.context.object.name = sourceName + "_dmg"
        
        bpy.ops.object.editmode_toggle()
        bpy.ops.mesh.select_all(action='DESELECT')
        bpy.ops.mesh.select_mode(use_extend=False, use_expand=False, type='EDGE')
        bpy.ops.mesh.edges_select_sharp(sharpness=0.436332)
        bpy.ops.mesh.mark_sharp()
        bpy.ops.object.editmode_toggle()

        damageName = bpy.context.object.name
        damageObj = bpy.data.objects[damageName]
        
        bpy.context.object.name = damageName
        bpy.data.objects[damageName].data.name = damageName
        
        bpy.ops.object.select_all(action='DESELECT')
        bpy.data.objects[sourceName].select_set(True)
        bpy.context.view_layer.objects.active = sourceObj
        bpy.ops.object.delete(use_global=False)
        
        bpy.data.objects[damageName].select_set(True)
        bpy.context.view_layer.objects.active = damageObj

        bpy.ops.object.duplicate_move(OBJECT_OT_duplicate={"linked":False, "mode":'TRANSLATION'})
        bpy.context.object.name = damageName + "_temp"
        
        targetName = bpy.context.object.name
        targetObj = bpy.data.objects[targetName]
        
        for ob in bpy.context.object.users_collection:
            ob.objects.unlink(targetObj)
        bpy.data.collections[damaged_col].objects.link(targetObj)
        bpy.context.view_layer.objects.active = targetObj
        
        bpy.context.object.modifiers.new("OCD_remesh", type='REMESH')
        bpy.context.object.modifiers["OCD_remesh"].voxel_size = voxel_size * multiplier
        bpy.context.object.modifiers["OCD_remesh"].use_smooth_shade = True

        bpy.data.objects[targetName].modifiers.new("OCD_Smooth", type='SMOOTH')
        bpy.data.objects[targetName].modifiers["OCD_Smooth"].factor = SmoothGraph
        bpy.data.objects[targetName].modifiers["OCD_Smooth"].iterations = 30
        
        bpy.context.object.modifiers.new("OCD_Displace", type='DISPLACE')
        bpy.context.object.modifiers["OCD_Displace"].strength = StrengthGraph / 10
        bpy.context.object.modifiers["OCD_Displace"].mid_level = MidGraph
       
        if "OCD_texture" not in bpy.data.textures:
            OCD_texture = bpy.data.textures.new(name = "OCD_texture", type = 'MUSGRAVE')
            has_noise_basis = hasattr(bpy.data.textures["OCD_texture"], "noise_basis")
            if has_noise_basis:
                bpy.data.textures["OCD_texture"].noise_basis = 'VORONOI_CRACKLE'
            else:
                pass
            bpy.data.textures["OCD_texture"].musgrave_type = 'HYBRID_MULTIFRACTAL'
            bpy.data.textures["OCD_texture"].type = props.noise_type
            bpy.data.textures["OCD_texture"].noise_scale = xSc * sc + minSc
        else:     
            OCD_texture = bpy.data.textures["OCD_texture"]

            bpy.data.textures["OCD_texture"].type = props.noise_type
            has_noise_basis = hasattr(bpy.data.textures["OCD_texture"], "noise_basis")

            if bpy.data.textures["OCD_texture"].type == 'MUSGRAVE':
                bpy.data.textures["OCD_texture"].contrast = 1
                bpy.data.textures["OCD_texture"].intensity = 1
                bpy.data.textures["OCD_texture"].noise_basis = 'VORONOI_CRACKLE'
            if bpy.data.textures["OCD_texture"].type == 'CLOUDS':
                bpy.data.textures["OCD_texture"].contrast = 1.5
                bpy.data.textures["OCD_texture"].intensity = 1
                bpy.data.textures["OCD_texture"].noise_basis = 'VORONOI_CRACKLE'
            if bpy.data.textures["OCD_texture"].type == 'MARBLE':
                bpy.data.textures["OCD_texture"].contrast = 1.5
                bpy.data.textures["OCD_texture"].intensity = 1.5
                bpy.data.textures["OCD_texture"].turbulence = 5
                bpy.data.textures["OCD_texture"].noise_basis = 'IMPROVED_PERLIN'
            if bpy.data.textures["OCD_texture"].type == 'VORONOI':
                bpy.data.textures["OCD_texture"].contrast = 0.5
                bpy.data.textures["OCD_texture"].intensity = 1.4
                if has_noise_basis:
                    bpy.data.textures["OCD_texture"].noise_basis = 'VORONOI_CRACKLE'
                else:
                    pass
            if bpy.data.textures["OCD_texture"].type == 'WOOD':
                bpy.data.textures["OCD_texture"].contrast = 1.5
                bpy.data.textures["OCD_texture"].intensity = 2
                bpy.data.textures["OCD_texture"].turbulence = 15
                bpy.data.textures["OCD_texture"].noise_basis = 'IMPROVED_PERLIN'

            bpy.data.textures["OCD_texture"].noise_scale = xSc * sc + minSc

        bpy.context.object.modifiers.new("OCD_remesh_02", type='REMESH')
        bpy.context.object.modifiers["OCD_remesh_02"].voxel_size = voxel_size * multiplier
        bpy.context.object.modifiers["OCD_remesh_02"].use_smooth_shade = True
        bpy.context.object.modifiers["OCD_Displace"].texture = bpy.data.textures['OCD_texture']
        bpy.context.object.modifiers["OCD_Displace"].texture_coords = 'GLOBAL'
        
        bpy.context.view_layer.objects.active = damageObj
        bpy.context.active_object.select_set(state=True)

        bpy.ops.object.editmode_toggle()
        bpy.ops.mesh.select_mode(use_extend=False, use_expand=False, type='FACE')

        props2 = bpy.context.object.stored_value
        props2.ocd_strength = bpy.data.collections[damaged_col].objects[targetName].modifiers["OCD_Displace"].strength
        props2.ocd_mid_level = bpy.data.collections[damaged_col].objects[targetName].modifiers["OCD_Displace"].mid_level
        props2.ocd_smooth = bpy.data.collections[damaged_col].objects[targetName].modifiers["OCD_Smooth"].factor
        props2.ocd_noise_scale = bpy.data.textures["OCD_texture"].noise_scale
        props2.ocd_noise_type = bpy.data.textures["OCD_texture"].type
        if bpy.data.textures["OCD_texture"].type in ["MUSGRAVE", "CLOUDS", "WOOD", "MARBLE"]:
            props2.ocd_noise_basis = bpy.data.textures["OCD_texture"].noise_basis
        else:
            pass
        if bpy.data.textures["OCD_texture"].type not in ["MUSGRAVE", "CLOUDS", "VORONOI"]:
            props2.ocd_turbulence = bpy.data.textures["OCD_texture"].turbulence
        else:
            pass
        props2.ocd_contrast = bpy.data.textures["OCD_texture"].contrast
        props2.ocd_brightness = bpy.data.textures["OCD_texture"].intensity
        props2.ocd_voxel_size = bpy.data.collections[damaged_col].objects[targetName].modifiers["OCD_remesh"].voxel_size
        props2.ocd_voxel_size_02 = bpy.data.collections[damaged_col].objects[targetName].modifiers["OCD_remesh_02"].voxel_size

        bpy.ops.mesh.select_all(action='DESELECT')
        bpy.ops.mesh.select_all(action='SELECT')
        
        bpy.ops.mesh.normals_make_consistent(inside=False)
        bpy.ops.object.editmode_toggle()
        
        bpy.context.object.modifiers.new("OCD_Boolean", type='BOOLEAN')
        bpy.context.object.modifiers["OCD_Boolean"].solver = 'FAST'
        bpy.context.object.modifiers["OCD_Boolean"].object = targetObj
        bpy.context.object.modifiers["OCD_Boolean"].operation = 'INTERSECT'
        bpy.context.object.modifiers.new("OCD_Preview", type='EDGE_SPLIT')
        bpy.context.object.modifiers["OCD_Preview"].split_angle = 0.261799

        bpy.context.object.data.use_auto_smooth = True
        bpy.context.object.data.auto_smooth_angle = 1.0472
        
        bpy.ops.object.select_all(action='DESELECT')
        
        bpy.data.objects[damageName].select_set(True)
        bpy.context.view_layer.objects.active = damageObj 
        
        active_obj = bpy.context.view_layer.objects.active
        
        for mod in active_obj.modifiers:
            if mod.type == "EDGE_SPLIT":
                bpy.ops.object.modifier_remove(modifier=mod.name)  
        
        for mod in active_obj.modifiers:
            if mod.type == "BOOLEAN":
                bpy.ops.object.modifier_apply(modifier=mod.name)
              
        bpy.context.object.modifiers.new("OCD_WNormal", type='WEIGHTED_NORMAL')
        bpy.context.object.modifiers["OCD_WNormal"].keep_sharp = True
        
        #materials
        mat_names = ["OUTSIDE", "INSIDE"]
        assign_materials(mat_names)
              
        damageName = bpy.context.object.name
        sourceName = damageName[:-4]
        
        bpy.data.meshes[sourceName].use_fake_user = True
        
        collection_name = "OCD_temp"
        collection = bpy.data.collections[collection_name]
        meshes = set()
        for obj in [o for o in collection.objects if o.type == 'MESH']:
            meshes.add( obj.data )
            bpy.data.objects.remove( obj )

        for mesh in [m for m in meshes]:
            bpy.data.meshes.remove( mesh )
        
        collection = bpy.data.collections.get(collection_name)
        bpy.data.collections.remove(collection)  
        #redraw the viewport
        bpy.ops.wm.redraw_timer(type='DRAW_SWAP', iterations=1)

def damage_off(context):
    obj = context.active_object
    
    damageName = bpy.context.object.name
    sourceName = damageName[:-4]
    
    bpy.context.object.name = sourceName
    
    if sourceName in bpy.data.meshes:
            mesh = bpy.data.meshes[sourceName]
            bpy.data.meshes.remove(mesh)
            bpy.data.objects[sourceName].data.name = sourceName

    for mod in obj.modifiers:
        if mod.type == "EDGE_SPLIT":
            bpy.ops.object.modifier_remove(modifier=mod.name)
    for mod in obj.modifiers:
        if mod.type == "BOOLEAN":
            bpy.ops.object.modifier_remove(modifier=mod.name)
            
        collection_name = "OCD_temp"
        
        if collection_name not in bpy.data.collections:
            pass
        else:
            collection = bpy.data.collections[collection_name]
            meshes = set()
            for obj in [o for o in collection.objects if o.type == 'MESH']:
                meshes.add( obj.data )
                bpy.data.objects.remove( obj )
            for mesh in [m for m in meshes]:
                bpy.data.meshes.remove( mesh )
        if collection_name not in bpy.data.collections:      
            pass
        else:
            collection = bpy.data.collections.get(collection_name)
            bpy.data.collections.remove(collection)
        
    return {'FINISHED'}

def simple_apply(context):
    props2 = bpy.context.object.stored_value
    damaged_col = "OCD_temp"
    sourceName = bpy.context.object.name
    targetName = sourceName + "_temp"
    props2.ocd_strength = bpy.data.collections[damaged_col].objects[targetName].modifiers["OCD_Displace"].strength
    props2.ocd_mid_level = bpy.data.collections[damaged_col].objects[targetName].modifiers["OCD_Displace"].mid_level
    props2.ocd_smooth = bpy.data.collections[damaged_col].objects[targetName].modifiers["OCD_Smooth"].factor
    props2.ocd_noise_scale = bpy.data.textures["OCD_texture"].noise_scale
    props2.ocd_noise_type = bpy.data.textures["OCD_texture"].type
    has_noise_basis = hasattr(bpy.data.textures["OCD_texture"], "noise_basis")
    if has_noise_basis:
        props2.ocd_noise_basis = bpy.data.textures["OCD_texture"].noise_basis
    else:
        pass
    if bpy.data.textures["OCD_texture"].type not in ["MUSGRAVE", "CLOUDS", "VORONOI"]:
        props2.ocd_turbulence = bpy.data.textures["OCD_texture"].turbulence
    else:
        pass
    props2.ocd_contrast = bpy.data.textures["OCD_texture"].contrast
    props2.ocd_brightness = bpy.data.textures["OCD_texture"].intensity
    props2.ocd_voxel_size = bpy.data.collections[damaged_col].objects[targetName].modifiers["OCD_remesh"].voxel_size
    props2.ocd_voxel_size_02 = bpy.data.collections[damaged_col].objects[targetName].modifiers["OCD_remesh_02"].voxel_size

    active_obj = context.view_layer.objects.active

    bpy.context.object.modifiers["OCD_Boolean"].solver = 'FAST'
    
    for mod in active_obj.modifiers:
        if mod.name == "OCD_Boolean":
            bpy.ops.object.modifier_apply(modifier=mod.name)
    for mod in active_obj.modifiers:
        if mod.type == "EDGE_SPLIT":
            bpy.ops.object.modifier_remove(modifier=mod.name) 
    
    bpy.context.object.modifiers.new("OCD_WNormal", type='WEIGHTED_NORMAL')
    bpy.context.object.modifiers["OCD_WNormal"].keep_sharp = True

    #materials
    mat_names = ["OUTSIDE", "INSIDE"]
    assign_materials(mat_names)

    bpy.data.meshes[sourceName].use_fake_user = True
    
    collection_name = "OCD_temp"

    collection = bpy.data.collections[collection_name]
    meshes = set()
    for obj in [o for o in collection.objects if o.type == 'MESH']:
        meshes.add( obj.data )
        bpy.data.objects.remove( obj )
    for mesh in [m for m in meshes]:
        bpy.data.meshes.remove( mesh )
    collection = bpy.data.collections.get(collection_name)
    bpy.data.collections.remove(collection)
    return {'FINISHED'}

def ctrl_apply(context):
    obj = bpy.context.object
    name = context.object.name + "_temp"
    bpy.data.objects[name].modifiers.new("OCD_Subd", type='SUBSURF')
    
    if "OCD_texture_02" not in bpy.data.textures:
        OCD_texture_02 = bpy.data.textures.new(name = "OCD_texture_02", type = 'CLOUDS')
        bpy.data.textures["OCD_texture_02"].noise_basis = 'BLENDER_ORIGINAL'
        bpy.data.textures["OCD_texture_02"].noise_scale = 0.025
    else:     
        OCD_texture_02 = bpy.data.textures["OCD_texture_02"]
    
    bpy.data.objects[name].modifiers.new("OCD_Displace_02", type='DISPLACE')
    bpy.data.objects[name].modifiers["OCD_Displace_02"].strength = 0.02
    bpy.data.objects[name].modifiers["OCD_Displace_02"].texture = bpy.data.textures['OCD_texture_02']
    bpy.data.objects[name].modifiers["OCD_Displace_02"].texture_coords = 'GLOBAL'
    active_obj = context.view_layer.objects.active
    bpy.context.object.modifiers["OCD_Boolean"].solver = 'FAST'
    
    for mod in active_obj.modifiers:
        if mod.name == "OCD_Boolean":
            bpy.ops.object.modifier_apply(modifier=mod.name)
    for mod in active_obj.modifiers:
        if mod.type == "EDGE_SPLIT":
            bpy.ops.object.modifier_remove(modifier=mod.name) 
    
    bpy.context.object.modifiers.new("OCD_WNormal", type='WEIGHTED_NORMAL')
    bpy.context.object.modifiers["OCD_WNormal"].keep_sharp = True
    
    #materials
    mat_names = ["OUTSIDE", "INSIDE"]
    assign_materials(mat_names)
    
    damageName = bpy.context.object.name
    sourceName = damageName[:-4]
    
    bpy.data.meshes[sourceName].use_fake_user = True
    
    collection_name = "OCD_temp"
    collection = bpy.data.collections[collection_name]
    meshes = set()
    for obj in [o for o in collection.objects if o.type == 'MESH']:
        meshes.add( obj.data )
        bpy.data.objects.remove( obj )
    for mesh in [m for m in meshes]:
        bpy.data.meshes.remove( mesh )
    collection = bpy.data.collections.get(collection_name)
    bpy.data.collections.remove(collection)
    return {'FINISHED'}    

def recall(context,self):
    damageName = bpy.context.object.name
    objlocation = bpy.data.objects[damageName].location
    objrotation = bpy.data.objects[damageName].rotation_euler
    objscale = bpy.data.objects[damageName].scale

    bpy.ops.mesh.primitive_plane_add(size=0.1, enter_editmode=False, align='WORLD', location=objlocation, scale=(1, 1, 1))
    
    ob = bpy.context.object
    me = ob.data
    ob.name = 'donor'
    me.name = 'donor'   
    bpy.context.object.rotation_euler = objrotation
    
    donorName = bpy.context.object.name
    donorObj = bpy.data.objects[donorName]
    
    area = bpy.context.area
    area.type = 'OUTLINER'
                    
    sourceName = damageName[:-4] 
    bpy.context.view_layer.objects.active = donorObj
    bpy.ops.outliner.id_remap(id_type='MESH', old_id='donor', new_id=sourceName)
    area.type = 'VIEW_3D'
    
    for mesh in bpy.data.meshes:
        if mesh.name == 'donor':
            bpy.data.meshes.remove( mesh )
            
    bpy.ops.object.duplicate_move(OBJECT_OT_duplicate={"linked":False, "mode":'TRANSLATION'})
    bpy.context.object.name = sourceName + ".001"
    bpy.context.object.scale = objscale
    
    ob = bpy.data.objects.get("donor") 
    bpy.data.objects.remove(ob)
    return {'FINISHED'}
    
def ctrl_recall(context,self):
    for o in bpy.context.selected_objects:
        bpy.context.view_layer.objects.active = o
        damageName = bpy.context.object.name
        bpy.context.object.name = damageName
        bpy.data.objects[damageName].data.name = damageName
        area = bpy.context.area
        area.type = 'OUTLINER'
        sourceName = damageName[:-4] 
        if sourceName not in bpy.data.meshes:
            self.report({'ERROR'}, "No source mesh found")
            pass
        else:
            bpy.ops.outliner.id_remap(id_type='MESH', old_id=damageName, new_id=sourceName)
        
        area.type = 'VIEW_3D'
        bpy.context.object.name = damageName[:-4]
        
        sourceName = damageName[:-4]
        bpy.data.objects[sourceName].data.name = sourceName
        
        if damageName in bpy.data.meshes:
            mesh = bpy.data.meshes[damageName]
            bpy.data.meshes.remove(mesh)
        damageName = sourceName
        obj = context.active_object
        for mod in obj.modifiers:
            if mod.name == "OCD_WNormal":
                bpy.ops.object.modifier_remove(modifier=mod.name)

def OCD_random(context, self, random):
    
    props2 = bpy.context.object.stored_value
    voxel_1 = props2.ocd_voxel_size
    voxel_2 = props2.ocd_voxel_size_02
    mid = props2.ocd_mid_level
    strength = props2.ocd_strength
    noiseScale = props2.ocd_noise_scale
    noiseType = props2.ocd_noise_type
    noise_basis = props2.ocd_noise_basis
    ocd_brightness = props2.ocd_brightness
    ocd_contrast = props2.ocd_contrast
    ocd_turbulence = props2.ocd_turbulence
    smooth = props2.ocd_smooth
    
    damageName = bpy.context.object.name
    bpy.context.object.name = damageName
    bpy.data.objects[damageName].data.name = damageName
    area = bpy.context.area

    area.type = 'OUTLINER'
    sourceName2 = damageName[:-4] 
    bpy.ops.outliner.id_remap(id_type='MESH', old_id=damageName, new_id=sourceName2)
    area.type = 'VIEW_3D'
    bpy.context.object.name = damageName[:-4]
    
    sourceName = damageName[:-4]
    bpy.data.objects[sourceName2].data.name = sourceName2
    
    if damageName in bpy.data.meshes:
        mesh = bpy.data.meshes[damageName]
        bpy.data.meshes.remove(mesh)
    damageName = sourceName2
    obj = bpy.context.active_object
    for mod in obj.modifiers:
        if mod.name == "OCD_WNormal":
            bpy.ops.object.modifier_remove(modifier=mod.name)

    sourceName = bpy.context.object.name
    sourceObj = bpy.data.objects[sourceName]
    damaged_col = "OCD_temp"
         
    if damaged_col not in bpy.data.collections:      
        damaged_collection = bpy.context.blend_data.collections.new(name=damaged_col)
        bpy.context.collection.children.link(damaged_collection)
    else:
        damaged_collection = bpy.data.collections[damaged_col]      

    damaged_collection.hide_viewport = True
    damaged_collection.hide_render = True
    
    sourceName = bpy.context.object.name
    sourceMeshName = bpy.data.objects[sourceName].data.name
    bpy.context.object.name = sourceMeshName
    bpy.context.object.name = sourceName
    
    bpy.data.objects[sourceName].data.name = sourceName 
    
    sourceMeshName = bpy.data.objects[sourceName].data.name
    
    bpy.ops.object.transform_apply(location = False, rotation = False, scale = True)
    
    bpy.ops.object.duplicate_move(OBJECT_OT_duplicate={"linked":False, "mode":'TRANSLATION'})
    bpy.context.object.name = sourceName + "_dmg"
    
    bpy.ops.object.editmode_toggle()
    bpy.ops.mesh.select_all(action='DESELECT')
    bpy.ops.mesh.select_mode(use_extend=False, use_expand=False, type='EDGE')
    bpy.ops.mesh.edges_select_sharp(sharpness=0.436332)
    bpy.ops.mesh.mark_sharp()
    bpy.ops.object.editmode_toggle()

    damageName = bpy.context.object.name
    damageObj = bpy.data.objects[damageName]
    
    bpy.context.object.name = damageName
    bpy.data.objects[damageName].data.name = damageName
    
    bpy.ops.object.select_all(action='DESELECT')
    bpy.data.objects[sourceName].select_set(True)
    bpy.context.view_layer.objects.active = sourceObj
    bpy.ops.object.delete(use_global=False)

    bpy.data.objects[damageName].select_set(True)
    bpy.context.view_layer.objects.active = damageObj

    bpy.ops.object.duplicate_move(OBJECT_OT_duplicate={"linked":False, "mode":'TRANSLATION'})
    bpy.context.object.name = damageName + "_temp"

    targetName = bpy.context.object.name
    targetObj = bpy.data.objects[targetName]
    
    bpy.ops.object.editmode_toggle()
    bpy.ops.mesh.select_mode(use_extend=False, use_expand=False, type='FACE')

    bpy.ops.mesh.select_all(action='DESELECT')
    bpy.ops.mesh.select_all(action='SELECT')
    
    bpy.ops.mesh.normals_make_consistent(inside=False)

    bpy.ops.mesh.select_all(action='DESELECT')
    bpy.ops.mesh.select_all(action='SELECT')
    
    bpy.ops.object.editmode_toggle()

    for ob in bpy.context.object.users_collection:
        ob.objects.unlink(targetObj)
    
    bpy.data.collections[damaged_col].objects.link(targetObj)

    #place an Empty named OCD_Empty at the random location
    random_location = (random.uniform(-100, 100), random.uniform(-100, 100), random.uniform(-100, 100))
    bpy.ops.object.empty_add(type='PLAIN_AXES', align='WORLD', location=random_location)
    bpy.context.object.name = "OCD_Empty"
    
    bpy.data.objects[targetName].modifiers.new("OCD_remesh", type='REMESH')
    bpy.data.objects[targetName].modifiers["OCD_remesh"].mode = 'VOXEL'
    bpy.data.objects[targetName].modifiers["OCD_remesh"].voxel_size = voxel_1
    bpy.data.objects[targetName].modifiers["OCD_remesh"].use_smooth_shade = True
    
    bpy.data.objects[targetName].modifiers.new("OCD_Smooth", type='SMOOTH')
    bpy.data.objects[targetName].modifiers["OCD_Smooth"].iterations = 30
    bpy.data.objects[targetName].modifiers["OCD_Smooth"].factor = smooth

    if "OCD_texture" not in bpy.data.textures:
        OCD_texture = bpy.data.textures.new("OCD_texture", type = noiseType)
        bpy.data.textures["OCD_texture"].noise_basis = 'VORONOI_CRACKLE'
    else:
        OCD_texture = bpy.data.textures["OCD_texture"]
    bpy.data.textures["OCD_texture"].noise_scale = noiseScale
    bpy.data.textures["OCD_texture"].type = noiseType
    has_noise_basis = hasattr(bpy.data.textures["OCD_texture"], "noise_basis")
    if has_noise_basis:
        bpy.data.textures["OCD_texture"].noise_basis = noise_basis
    else:
        pass
    bpy.data.textures["OCD_texture"].contrast = ocd_contrast
    bpy.data.textures["OCD_texture"].intensity = ocd_brightness
    has_turbulence = hasattr(bpy.data.textures["OCD_texture"], "turbulence")
    if has_turbulence:
        bpy.data.textures["OCD_texture"].turbulence = ocd_turbulence
    else:
        pass
    has_noise_basis = hasattr(bpy.data.textures["OCD_texture"], "noise_basis")
    if has_noise_basis:
        bpy.data.textures["OCD_texture"].noise_basis = noise_basis
    else:
        pass
    
    bpy.data.objects[targetName].modifiers.new("OCD_Displace", type='DISPLACE')
    bpy.data.objects[targetName].modifiers["OCD_Displace"].texture = bpy.data.textures["OCD_texture"]
    bpy.data.objects[targetName].modifiers["OCD_Displace"].strength = strength
    bpy.data.objects[targetName].modifiers["OCD_Displace"].mid_level = mid

    bpy.data.objects[targetName].modifiers.new("OCD_remesh_02", type='REMESH')
    bpy.data.objects[targetName].modifiers["OCD_remesh_02"].voxel_size = voxel_2
    bpy.data.objects[targetName].modifiers["OCD_remesh_02"].use_smooth_shade = True
    
    bpy.data.objects[targetName].modifiers["OCD_Displace"].texture = OCD_texture
    bpy.data.objects[targetName].modifiers["OCD_Displace"].texture_coords = 'OBJECT'
    bpy.data.objects[targetName].modifiers["OCD_Displace"].texture_coords_object = bpy.data.objects["OCD_Empty"]
    
    bpy.context.view_layer.objects.active = damageObj
    bpy.context.active_object.select_set(state=True)
    
    bpy.ops.object.editmode_toggle()
    bpy.ops.mesh.select_mode(use_extend=False, use_expand=False, type='FACE')

    bpy.ops.mesh.select_all(action='DESELECT')
    bpy.ops.mesh.select_all(action='SELECT')
    
    bpy.ops.mesh.normals_make_consistent(inside=False)
    bpy.ops.object.editmode_toggle()

    bpy.context.object.modifiers.new("OCD_Boolean", type='BOOLEAN')
    bpy.context.object.modifiers["OCD_Boolean"].solver = 'FAST'
    bpy.context.object.modifiers["OCD_Boolean"].object = targetObj
    bpy.context.object.modifiers["OCD_Boolean"].operation = 'INTERSECT'
    bpy.context.object.modifiers.new("OCD_Preview", type='EDGE_SPLIT')
    bpy.context.object.modifiers["OCD_Preview"].split_angle = 0.261799

    bpy.context.object.data.use_auto_smooth = True
    bpy.context.object.data.auto_smooth_angle = 1.0472
    
    bpy.data.objects[damageName].modifiers["OCD_Boolean"].solver = 'FAST'

    active_obj = bpy.context.view_layer.objects.active
    
    for mod in active_obj.modifiers:
        if mod.type == "BOOLEAN":
            bpy.ops.object.modifier_apply(modifier=mod.name)
    for mod in active_obj.modifiers:
        if mod.type == "EDGE_SPLIT":
            bpy.ops.object.modifier_remove(modifier=mod.name) 
     
    bpy.data.objects[damageName].modifiers.new("OCD_WNormal", type='WEIGHTED_NORMAL')
    bpy.data.objects[damageName].modifiers["OCD_WNormal"].keep_sharp = True
    
    #materials
    mat_names = ["OUTSIDE", "INSIDE"]
    assign_materials(mat_names)
    
    damageName = bpy.context.object.name
    sourceName = damageName[:-4]
    
    bpy.data.meshes[sourceName].use_fake_user = True
    
    collection_name = "OCD_temp"
    collection = bpy.data.collections[collection_name]

    meshes = set()
    for obj in [o for o in collection.objects if o.type == 'MESH']:
        meshes.add( obj.data )
        bpy.data.objects.remove( obj )
    for mesh in [m for m in meshes]:
        bpy.data.meshes.remove( mesh )
    collection = bpy.data.collections.get(collection_name)
    bpy.data.collections.remove(collection)
    #delete OCD_Empty:
    bpy.data.objects.remove(bpy.data.objects["OCD_Empty"])
    return {'FINISHED'}   
    
class DamageProps(PropertyGroup):
    damage_amount : FloatProperty(
        name = "Amount",
        default = 25,
        min = 0,
        max = 100,
        subtype = "PERCENTAGE",
        update=update_func,
    )
    
    scale_amount : FloatProperty(
        name = "Scale",
        default = 35,
        min = 0,
        soft_max = 100,
        subtype = "PERCENTAGE",
        update=update_func,
    )

    smooth_amount : FloatProperty(
        name = "Smooth",
        default = 0.25,
        min = 0.0,
        max = 0.5,
        subtype = "FACTOR",
    )
    
    noise_type : EnumProperty( 
        name= "Noise Selection", 
        description= "Noise Selection", 
        items= [('CLOUDS', "CLOUDS", ""),
                ('MUSGRAVE', "MUSGRAVE", ""),
                ('MARBLE',"MARBLE",""),
                ('VORONOI', "VORONOI",""),
                ('WOOD', "WOOD","")],
        update=update_func,       
    )
class StoredValueProp(PropertyGroup):
    ocd_strength : FloatProperty(
        name = "Strength",
    )
    ocd_voxel_size : FloatProperty(
        name = "Voxel Size",
    )
    ocd_voxel_size_02 : FloatProperty(
        name = "Voxel Size 02",
    )
    ocd_smooth : FloatProperty(
        name = "Smooth",
    )
    ocd_noise_scale : FloatProperty(
        name = "Noise Scale",
    )
    ocd_mid_level : FloatProperty(
        name = "Mid Level",
    )
    ocd_noise_type : bpy.props.StringProperty(
        name = "Noise Type",
    )
    ocd_noise_basis : bpy.props.StringProperty(
        name = "Noise Basis",
    )
    ocd_turbulence : FloatProperty(
        name = "Noise Turbulence",
    )
    ocd_brightness : FloatProperty(
        name = "Noise Brightness",
    )
    ocd_contrast : FloatProperty(
        name = "Noise Contrast",
    )
class OCD_Preferences(bpy.types.AddonPreferences):
    bl_idname = __name__

    ocd_voxel_size : FloatProperty(
        name = "Voxel Size (less is more)",
        default = 24,
        min = 10,
        soft_max = 30,
    )

    def draw(self, context):
        layout = self.layout
        layout.label(text="OCD Resolution:")
        row = layout.row()
        row.prop(self, "ocd_voxel_size")

class OBJECT_OT_damageON(Operator):
    bl_label = "OCD"
    bl_idname = "object.ocd_on"
    bl_description = "Hold ctrl for instant damage"
    bl_space_type = "VIEW_3D"
    bl_region_type = "UI"
    bl_options = {'REGISTER', 'UNDO'}

    @classmethod
    def poll(cls, context):
        obj = context.object
        return obj and obj.select_get() and obj.type == 'MESH' and obj.mode == 'OBJECT'

    def invoke(self, context, event):
        objs = context.selected_objects
        if event.ctrl or event.oskey:
            ctrl_damage(context, self)
        elif len(objs) > 1:
            multobj_damage(context)
        else:
            damage_on(context, self)
        return {'FINISHED'}

class OBJECT_OT_damageOFF(Operator):
    bl_label = "OCD"
    bl_idname = "object.ocd_off"
    bl_description = "Cancel the damage"
    bl_space_type = "VIEW_3D"
    bl_region_type = "UI"
    bl_options = {'REGISTER', 'UNDO'}

    @classmethod
    def poll(cls, context):
        obj = context.object
        return obj and obj.select_get() and obj.type == 'MESH' and obj.mode == 'OBJECT'
    
    def execute(self, context):
        damage_off(context)
        self.report({'INFO'}, "Damage cancelled!")
        return {'FINISHED'} 

class OBJECT_OT_apply(Operator):
    bl_label = "Apply all"
    bl_idname = "object.apply_all_mods"
    bl_description = "Hold ctrl for more detail"
    bl_space_type = "VIEW_3D"
    bl_region_type = "UI"
    bl_options = {'REGISTER', 'UNDO'}
    
    @classmethod
    def poll(cls, context):
        obj = context.active_object
        if not obj or obj.mode != 'OBJECT':
            return False
        for mod in obj.modifiers:
            if mod.type == "BOOLEAN": 
                return True
        return False
    
    def invoke(self, context, event):
        obj = context.active_object
        if event.ctrl or event.oskey:
            ctrl_apply(context)
        else:
            simple_apply(context)
            self.report({'INFO'}, "Damage applied!")
        return {'FINISHED'}

class OBJECT_OT_recall(Operator):
    bl_label = "Recall Original"
    bl_idname = "object.recall_original"
    bl_description = "Hold ctrl to recall the original object as a duplicate"
    bl_space_type = "VIEW_3D"
    bl_region_type = "UI"
    bl_options = {'REGISTER', 'UNDO'}

    def invoke(self, context, event):
        if event.ctrl or event.oskey:
            recall(context, self)
            self.report({'INFO'}, "Original object recalled as a duplicate")
        else:
            ctrl_recall(context,self)
            self.report({'INFO'}, "Original object recalled")
        return {'FINISHED'}

class OBJECT_OT_random(Operator):
    bl_label = "Randomize"
    bl_idname = "object.randomize"
    bl_description = "Change the OCD pattern. Size and Resolution will be kept"
    bl_space_type = "VIEW_3D"
    bl_region_type = "UI"
    bl_options = {'REGISTER', 'UNDO'}

    @classmethod
    def poll(cls, context):
        obj = context.object
        return obj and obj.select_get() and obj.type == 'MESH' and obj.mode == 'OBJECT'

    def execute(self, context):
        obj = context.object
        props2 = obj.stored_value
        if not props2.ocd_voxel_size:
            self.report({'WARNING'}, "No data found")
            return {'CANCELLED'}
        OCD_random(self, context, random)
        self.report({'INFO'}, "Randomized damage pattern applied")
        return {'FINISHED'}

class OBJECT_PT_OCD_panel(Panel):
    bl_label = 'One Click Damage v1.7'
    bl_space_type = 'VIEW_3D'
    bl_region_type = 'UI'
    bl_category = 'OCD'
    
    try:
        objName = bpy.context.object.name
    except: 
        pass

    def draw(self, context):
        layout = self.layout
        obj = context.active_object
        props = context.scene.dmgamnt
        damage_amount = props.damage_amount
        scale_amount = props.scale_amount
        dmgamnt = context.scene.dmgamnt

        if obj is None:
            return  # exit function early if no object is selected

        targetNameProp = obj.name + '_temp'
        row = layout.row()
        row.scale_y = 2.0
        if '_dmg' not in obj.name:
            row.operator("object.ocd_on", text='Make Damage', depress=True)
        else:
            pass

        boolean_mod = next((mod for mod in obj.modifiers if mod.name == 'OCD_Boolean'), None)
        if boolean_mod:
            row.operator("object.ocd_off", text='Cancel', depress=False)
            col = layout.column()
            col.prop(props, "scale_amount", text='Scale')
            col.prop(props, "damage_amount")
            col.label(text="Noise type")
            col.prop(dmgamnt, "noise_type", text="")
            row = layout.row()
            col.operator("object.apply_all_mods", text="Apply")

        wnormal_mod = next((mod for mod in obj.modifiers if mod.name == 'OCD_WNormal'), None)
        if wnormal_mod:
            row.enabled = '_dmg.' not in obj.name
            row.operator("object.recall_original", text="Recall")
            row = layout.row()
            row.scale_y = 2.0
            row.operator("object.randomize", text="Change Pattern")


# ------------------------------------------------------------------------
#    Registration
# ------------------------------------------------------------------------

#---Classes register list
classes = [
    DamageProps,
    StoredValueProp,
    OCD_Preferences,
    OBJECT_OT_damageON,
    OBJECT_OT_damageOFF,
    OBJECT_OT_apply,
    OBJECT_OT_recall,
    OBJECT_PT_OCD_panel,
    OBJECT_OT_random,
]

def register():
    for cl in classes:
       register_class(cl)
    bpy.types.Scene.dmgamnt = PointerProperty(type = DamageProps) 
    bpy.types.Object.stored_value = PointerProperty(type = StoredValueProp)
    
       
def unregister():
    for cl in reversed(classes):
        unregister_class(cl)
    del bpy.types.Scene.dmgamnt
    del bpy.types.Object.stored_value