=begin
#-------------------------------------------------------------------------------------------------
#*************************************************************************************************
# Copyright © 2009-2011 Fredo6 - Designed and written July 2009 and August 2011 by Fredo6
#
# Permission to use this software for any purpose and without fee is hereby granted
# Distribution of this software for commercial purpose is subject to:
#  - the expressed, written consent of the author
#  - the inclusion of the present copyright notice in all copies.

# THIS SOFTWARE IS PROVIDED "AS IS" AND WITHOUT ANY EXPRESS OR
# IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
# WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
#-----------------------------------------------------------------------------
# Name			:   !AdditionalPluginFolders.rb
# Original Date	:   04 July 2009
# Description	:   Define additional directories to load plugins
# Menu Item		:   Windows > Plugins Additional Directory
#-------------------------------------------------------------------------------------------------
#*************************************************************************************************
=end

require "sketchup.rb"

#==============================================================
# Dialog box for Configuration of Additional Plugin Folders
#==============================================================

module AdditionalPluginFolders6

#For LibFredo6 Check for Update
def self.register_plugin_for_LibFredo6 
	@copyright = "© 2008-2015"
	@creator = "Fredo6"
	{	
		:name => @myname,
		:author => "Fredo6",
		:version => "5.0a",
		:date => "07 Nov 15",	
		:description => @tooltip,
		:lst_obsolete_files => ["000_RedirectPlugins.rb", "000_AdditionalPluginFolders.rb"],
		:link_info => "http://sketchucation.com/forums/viewtopic.php?f=323&t=39073#p345150"
	}
end

#Initialize the texts
def self.init_text
	case Sketchup.get_locale
	when /\AFR/i
		@title = "Dossiers supplémentaires pour Plugins"
		@fullname = "Entrez le Chemin COMPLET en respectant la casse [/ et \\ acceptés]"
		@tooltip = "Spécifie un ou plusieurs dossiers supplémentaires pour le chargement des Plugins"
		@msg_directory = "Répertoire"
		@msg_load_at_startup = "Voulez-vous charger les plugins de ce répertoire ?"
		@msg_exist = "Les Dossiers suivants n'existent pas"
		@msg_confirm = "Confirmez-vous ?"
		@lst_status = ["Toujours", "Demander", "Jamais"]
		@msg_next = "Prise en compte au prochain démarrage de Sketchup"
		@msg_error = "%1 ERREUR(S) : Les plugins suivants sont en erreur et n'ont pu etre chargés complétement :"
		@msg_plugin = "PLUGIN :"
		@msg_folder = "REPERTOIRE :"
		@msg_trace = "ERREUR :"
		@dlg_save = "Enregistrer"
		@dlg_cancel = "Annuler"
		@dlg_print = "Imprimer"
		@dlg_precede_qum = "Précéder le chemin avec un ? pour demander confirmation au démarrage de Sketchup"
		@dlg_precede_exm = "Précéder le chemin avec un ! pour ignorer au démarrage de Sketchup"
		@title_dir = "Sélectionner un répertoire Plugins alternatif"
		@title_clear = "Effacer le champ"
		
	when /\AES/i
		@title = "Carpetas Adicionales para Plugins"
		@tooltip = "Especifica Uno o Mas Directorios Adicionales para la Carga de Plugins"
		@fullname = "Introduce la UBICACION Completa en Relacion al Caso [/ y \ Aceptar]"
		@msg_directory = "Directorio"
		@msg_load_at_startup = "Deseas Cargar los Plugins Contenidos en este Directorio ?"
		@msg_exist = "Los Siguientes Directorios No Existen !!"
		@msg_confirm = "Deseas Emplearlos de Todos Modos ?"
		@lst_status = ["Siempre", "Preguntar", "Nunca"]
		@msg_next = "Estas se Tomaran en Cuenta a Partir del Siguiente Inicio de SketchUp !!"
		@msg_error = "%1 ERROR(ES): Los Siguientes Plugins Tienen Errores y No se Pueden Cargar por Completo:"
		@msg_plugin = "PLUGIN:"
		@msg_folder = "CARPETA:"
		@msg_trace = "ERROR:"
		@dlg_save = "Guardar"
		@dlg_cancel = "Cancelar"
		@dlg_print = "Imprimir"
		@dlg_precede_qum = "Teclea ? antes de la Ubicacion para Pedir una Confirmacion al Inicio de SketchUp"
		@dlg_precede_exm = "Teclea ! antes de la Ubicacion para Omitirla al Inicio de SketchUp"
		@title_dir = "Carpetas Adicionales para Plugins"
		@title_clear = "Clear the field"
	
	else
		@title = "加载指定插件路径"
		@tooltip = "Define one or several additional Directories for loading Plugins"
		@fullname = "Enter FULL path with respect of case [/ and \\ accepted]"
		@msg_directory = "目录"
		@msg_load_at_startup = "Do you want to load the plugins in this directory?"
		@msg_exist = "The following Directories do not exist"
		@msg_confirm = "Do you confirm?"
		@lst_status = ["Always", "Ask", "Never"]
		@msg_next = "This will taken into account at next Sketchup startup"
		@msg_error = "%1 ERROR(S): The following plugins had errors and may not be fully loaded:"
		@msg_plugin = "PLUGIN:"
		@msg_folder = "FOLDER:"
		@msg_trace = "ERROR:"
		@dlg_save = "保存"
		@dlg_cancel = "取消"
		@dlg_print = "打印"
		@dlg_precede_qum = "Put a ? before the path to Ask confirmation at Sketchup startup"
		@dlg_precede_exm = "Put a ! before the path to Skip at Sketchup startup"
		@title_dir = "选择备用插件目录"
		@title_clear = "清除路径"
	end
end

#Registering as an extension
def self.register_execute
	#Account for non-ascii characters in the __FILE__path
	file__ = __FILE__
	file__ = file__.force_encoding("UTF-8") if defined?(Encoding)
	f6__file__ = file__.gsub(/\\/, '/')
	f6__sudir = File.dirname(f6__file__)
	f6__base = File.basename(f6__file__, ".rb")
	
	#Calculating the loader
	floader = File.join f6__sudir, f6__base, "#{f6__base}_loader"	
	
	#Creating the extension
	hsh = register_plugin_for_LibFredo6
	date = hsh[:date]
	init_text
	ext = SketchupExtension.new "Fredo6 AdditionalPluginFolders", floader
	ext.creator = @creator
	ext.description = @tooltip
	ext.version = "v" + hsh[:version] + " - " + date
	ext.copyright = @creator + " - " + @copyright
	
	#Registering the extension for execution
	status = Sketchup.register_extension ext, true
end

#Initialization
def self.startup
	@myname = "AdditionalPluginFolders"
	@regkey = "Alternate_dir"	#Registry key in section LibFredo6
	@priority_libs = ["!SketchUcation_loader.rb", "Fredo6_!LibFredo6.rb", "LibFredo6.rb"]
	@run_on_mac = (RUBY_PLATFORM =~ /darwin/i) ? true : false
	@su_capa_ask_dir = defined?(UI.select_directory)

	init_text
	load_menu
	load_additional_folders
end

#Load the single menu
def self.load_menu
	sumenu = UI.menu "Windows"
	sumenu.add_separator
	cmd = UI::Command.new(@title) { ask_dir_dialog }
	cmd.status_bar_text = @tooltip
	sumenu.add_item cmd
end

#Check validity and existence of folders
def self.check_dir(sdir, ldir, lst_not_exist)
	sdir.split(/\s*;+\s*/).each do |d|
		d = d.strip
		c = d[0..0]
		if c == '?' || c == '!'
			d = d[1..-1].strip
		else
			c = ''
		end	
		next if d == ''
		d = File.expand_path d
		lst_not_exist.push d unless FileTest.directory?(d)
		ldir.push(c + d)
	end	
	ldir
end

def self.erase_registry
	Sketchup.write_default "LibFredo6", @regkey, nil
end

#Load Plugins in Additional Directories
def self.load_additional_folders
	#Load Alternate directories from Registry 
	alt_dirs = Sketchup.read_default "LibFredo6", @regkey
	alt_dirs = "" unless alt_dirs && alt_dirs.strip.length > 0
	alt_dirs = alt_dirs.force_encoding("UTF-8") if defined?(Encoding)
	alt_dirs = alt_dirs.split(/\s*;+\s*/)
	
	#Filtering for status at load time
	ldir = []
	lask = []
	alt_dirs.each do |sdir|
		c = sdir[0..0]
		next if c == '!'
		if c == '?'
			sdir = sdir[1..-1]
		else
			c = ''
		end	
		next unless FileTest.directory?(sdir)
		next if $LOAD_PATH.include?(sdir)
		ldir.push [c, sdir]
	end
	
	#Asking confirmation for loading of some directories
	lst_dir = []
	ldir.each do |a|
		c, dir = a
		if c == '?'
			status = UI.messagebox @msg_load_at_startup + "\n\n" + dir, MB_YESNO
			next if status != 6		
		end
		lst_dir.push dir
	end	
	lst_dir.uniq!
	
	#Initialization of tracking parameters
	@nb_errors = 0
	@lst_plugins_loaded = []
	@lext = (RUBY_PLATFORM =~ /darwin/i) ? ['rb', 'rbs', 'bundle'] : ['rb', 'rbs', 'so']
	
	#Complementing the Load Path for Ruby scripts - New directories are put before SU ones
	$LOAD_PATH[0,0] = lst_dir
	
	#Priority libs - Finding most recents
	load_path = lst_dir + $LOAD_PATH.find_all { |d|  d =~ /Sketchup..\/Plugins\Z/i }
	lst_libs = []
	@priority_libs.each do |lib|
		ls = []
		load_path.each do |d|
			path = File.join(d, lib)
			ls.push [d, lib, File.mtime(path)] if FileTest.exist?(path)
		end	
		ls.sort! { |a, b| b[2] <=> a[2] } if ls.length > 1
		lst_libs.push ls[0] if ls.length > 0
	end
	
	#Loading Priority libs
	hsh_lib_path = {}
	lst_libs.each do |a|
		hsh_lib_path[a[1]] = true
		load_single_file a[0], a[1]
	end
	
	#Loading all other scripts from SU Standard and Alternate directories
	t0 = Time.now
	load_path.each do |d|
		lsf = Dir[File.join(d, '*.*')]
		lsf.each { |f| load_single_file(d, f) unless hsh_lib_path[File.basename(f)] }
	end

	#Logging the loading to LibFredo6 if present
	if defined?(LibFredo6.log)
		LibFredo6.log t0, "#{@myname}: Beginning of loading cycle***+"
		@lst_plugins_loaded.each do |a| 
			pname = a[0]
			pname = $' if pname =~ /\A!+/
			if a[2]
				LibFredo6.log a[3], "?#{pname}: ERROR - #{a[2]}"
			else
				d = a[1]
				sd = ".../#{File.basename(File.dirname(d))}/#{File.basename(d)}"
				LibFredo6.log a[3], "#{pname}: loaded from #{sd}   [#{a[4]} ms]"
			end
		end	
		LibFredo6.log Time.now, "#{@myname}: End of loading cycle***-"
	end
	
	#Message box for errors if any
	if @nb_errors > 0
		n = 0
		text = ""
		@lst_plugins_loaded.each do |a| 
			if a[2]
				n += 1
				text += "\n\n#{@msg_plugin} #{a[0]}\n#{@msg_folder} #{a[1]}\n#{@msg_trace} #{a[2]}" 
				break if n > 5
			end
		end	
		text = "Message from #{@myname}\n\n" + @msg_error.sub("%1", "#{@nb_errors}") + text
		@timer_id = UI.start_timer(0, false) { signal_error text }
	end	
end

def self.locate_error(e)
	return "#{e.backtrace[0]}" unless e.backtrace[0] =~ /(\d+):in\s`(.*)'/
	line = $1
	function = $2
	file = $`
	"#{e.message} [in #{File.basename(file)} - ##{function}(): line #{line}]"
end

#Load a single file, with check of errors
def self.load_single_file(d, f)
	t0 = Time.now
	basename = File.basename(f)
	ext =  basename.match(/.+\.(.+)\Z/)[1]
	return unless ext
	return unless @lext.include?(ext.downcase)
	status = false
	begin
		if (ext =~ /rbs\Z/)
			LibFredo6.log "True load: #{basename}" if defined?(LibFredo6.log)
			rbs_file = basename
			eval( "Sketchup.load(\"#{rbs_file}\")", TOPLEVEL_BINDING)		#top level namespace required
			status = true
		else	
			status = Sketchup.require basename
		end	
		if status
			delta = ((Time.now - t0) * 1000).to_i
			@lst_plugins_loaded.push [basename, d, nil, Time.now, delta]
		end	
	rescue Exception => e
		delta = ((Time.now - t0) * 1000).to_i
		if (ext =~ /so\Z/)
			@lst_plugins_loaded.push [basename, d, nil, Time.now, delta]
		else	
			@lst_plugins_loaded.push [basename, d, locate_error(e), Time.now, delta]
			@nb_errors += 1
		end	
	end
end

#Display the dialog box for Errors if any
def self.signal_error(text)
	UI.stop_timer @timer_id if @timer_id
	UI.messagebox text
end

#Return information on Plugins loaded
def self.plugins_loaded_info
	@lst_plugins_loaded
end

#--------------------------------------------------------------------------------------------------------------
#--------------------------------------------------------------------------------------------------------------
# Dialog box for Asking Directories
#--------------------------------------------------------------------------------------------------------------			 
#--------------------------------------------------------------------------------------------------------------			 

#Create the dialog box
def self.ask_dir_dialog
	#Dialog box already displayed
	return @wdlg.bring_to_front if @wdlg
	
	#Finding the position
	xpos = 200
	ypos = 200
	
	#Computing the size of dialog box
	@sreg0 = Object::Sketchup.read_default("LibFredo6", @regkey)
	@lst_not_exist = []
	@ldir = (@sreg0 && @sreg0.strip != "") ? @sreg0.split(';') : [""]
	@ldir.push "" unless @ldir[-1] == ""
	@sy = compute_height @ldir
	
	@nbchar = 50
	@size_field = 550
	@sx = 250 + @size_field
	
	#Creating the dialog box
	title = encode_to_ruby @title

	@wdlg = UI::WebDialog.new title, false, nil, @sx, @sy, xpos, ypos, false
	@wdlg.add_action_callback("top_callback") { |dialog, params| callback(params) }
	@wdlg.set_background_color 'cornsilk'
	@wdlg.navigation_buttons_enabled = false if Object::Sketchup.version.to_i > 6
	@wdlg.set_on_close { on_close_top }
	@wdlg.set_position xpos, ypos
	@wdlg.set_size @sx, @sy
	@wdlg.set_html format_html(@ldir)
	
	(@run_on_mac) ? @wdlg.show_modal : @wdlg.show
	@wdlg
end

#Adjust the size of the dialog box based on number of directory
def self.compute_height(ldir)
	nfields = ldir.length 
	syextra = (Object::Sketchup.version.to_i == 7 && !@run_on_mac) ? 20 : 0
	sy = 210 + nfields * 30 + syextra
	sy
end

#Notification of window closure
def self.on_close_top
	@wdlg = nil
end

#Call back for Statistics Dialog
def self.callback(params)
	case params
		
	#Command buttons
	when /print/i
		@wdlg.execute_script "window.print() ;"
	when /cancel/i
		@wdlg.close
	when /ButtonDir_(\d)/
		query_directory $1.to_i
	when /ButtonClear_(\d)/
		update_field($1, '')
		
	when /save/i
		if check_and_save
			@wdlg.close
		else
			sy = compute_height(@ldir)
			if sy != @sy
				@sy = sy
				@wdlg.set_size @sx, @sy
				@wdlg.set_html format_html(@ldir)
			end	
		end
	end
end

#Asking the directory graphically (SU version >= 15)
def self.query_directory(ifield)
	val = @wdlg.get_element_value "Field#{ifield}"
	if val && File.directory?(val)
		defdir = File.dirname val
	else
		iprev = [ifield - 1, 0].max
		val1 = @wdlg.get_element_value "Field#{iprev}"
		if val1 && File.directory?(val1)
			defdir = File.dirname val1
		else
			defdir = nil
		end	
	end
	hsh = { :title => @title_dir, :directory => defdir }
	d = UI.select_directory hsh
	
	#Updating the directory field unless user cancelled query
	update_field(ifield, d) if d
end

def self.update_field(ifield, value)
	@wdlg.execute_script "obj = document.getElementById ('Field#{ifield}') ; obj.value = \"#{value}\" ;"
end

#Verify and Save the directories
def self.check_and_save
	
	#Getting the directory value
	nfields = @ldir.length - 1
	ldir = []
	for i in 0..nfields
		val = @wdlg.get_element_value "Field#{i}"
		val = encode_to_ruby val
		ldir.push val
	end
	
	#Parsing the results
	@lst_not_exist = []
	lnew_dir = []
	for i in 0..nfields
		check_dir(ldir[i].strip, lnew_dir, @lst_not_exist)
	end
				
	#Some directories do not exist - Asking for confirmation
	if @lst_not_exist.length > 0
		status = UI.messagebox @msg_exist + "\n" + @lst_not_exist.join("\n") + "\n" + @msg_confirm, MB_YESNO
		if status == 7
			@ldir = lnew_dir
			@ldir.push "" unless @ldir[-1] == ""
			return nil
		end	
	end	
	
	#Storing the directory string in the Registry
	sreg = lnew_dir.join ';'
	return lnew_dir if sreg == @sreg0
	Object::Sketchup.write_default "LibFredo6", @regkey, sreg
	
	#Summary if there are changes
	lst_dir = []
	lnew_dir.each do |sdir|
		d = sdir[0..0]
		if d == '?'
			lst_dir.push "#{sdir[1..-1]} --> #{@lst_status[1]}"
		elsif d == '!'	
			lst_dir.push "#{sdir[1..-1]} --> #{@lst_status[2]}"
		else
			lst_dir.push "#{sdir} --> #{@lst_status[0]}"
		end
	end		
	UI.messagebox @title + "\n\n" + lst_dir.join("\n") + "\n\n" + @msg_next
	lnew_dir
end

#Encoding ISO and UTF
def self.encode_to_ruby(text)
	return text unless text.class == String
	begin
		return text.unpack("U*").pack("U*")
	rescue
		return text.unpack("C*").pack("U*")
	end
end 

#Format the HTML for the dialog box
def self.format_button(name, text, tip=nil, color=nil)
	scolor = (color) ? " ; color:#{color}" : ""
	"<input type='button' style='cursor:pointer #{scolor}' id='#{name}' title='#{tip}' value='#{text}' onclick='action(\"#{name}\")'/>"
end

#Javascript for capturing Return and Cancel
def self.script_for_keys
	text = %Q~
	
function CaptureKeyDown(e, event) { CaptureKey(e, "onKeyDown") }
function CaptureKeyUp(e, event) { CaptureKey(e, "onKeyUp") }	
function CaptureKey(e, event) {
	if (! e) e = window.event ;
	obj = (e.target) ? e.target : e.srcElement ;
	if (obj == null) return ;
	if ((event == "onKeyDown") && (e.keyCode == 13)) action("ButtonSave") ;
	if ((event == "onKeyUp") && (e.keyCode == 27)) action("ButtonCancel") ;
	return true ;
}
	~
	text
end

#HTML formatting for the whole page
def self.format_html(ldir)
	#HTML Headers
	text = ""
	text += %q(<!DOCTYPE HTML PUBLIC "-//W3C//DTD HTML 4.01 Transitional//EN" "http://www.w3.org/TR/html4/strict.dtd">)
	text += %q(<META http-equiv='Content-Type' Content='text/html; charset=UTF-8'> )
	text += %q(<META http-equiv='Content-Script-Type' content='text/javascript'>)
	
	#Scripts
	text += "<SCRIPT>"
	text += "function action(msg) {	window.location = 'skp:top_callback@' + msg ; }"	
	text += script_for_keys
	text += "</SCRIPT>"
	text += "<BODY onkeyup='CaptureKeyUp()' onkeydown='CaptureKeyDown()'>"

	#Some styles
	text += "<style type='text/css' media='print'>"
	text += ".NOPRINT_Style { display: none }"
	text += "</style>"
	
	#Information Message
	message = encode_to_ruby(@title)
	style = "color: navy ; font-size: 22px ; font-style: bold"
	text += "<div><table width='100%' cellpadding='0' cellspacing='0'>"
	text += "<tr align='center'><td><span style='#{style}'>#{message}</span><td><tr>"
	text += "</table></div>"
	text += "<br>"

	#Input Fields
	label = encode_to_ruby(@msg_directory)
	style_label = "color: blue ; font-size: 15px"
	style_field = "color: %3 ; font-size: 13px ; width: #{@size_field}px"
	tfield = "<input type='text' value=\"%1\" id='Field%2' style='#{style_field}'/>"
	text += "<div><table width='100%' cellpadding='1px' cellspacing='5px' style='border:1px solid gray'>"
	ldir.each_with_index do |dir, i|
		butdot = format_button("ButtonDir_#{i}", "...", @title_dir, 'blue')
		butclear = format_button("ButtonClear_#{i}", "X", @title_clear, 'red')
		color = (@lst_not_exist.include?(dir)) ? 'red' : 'black'
		text += "<tr border>"
		text += "<td align='right'><span style='#{style_label}'>#{label} ##{i+1}</span><td>"
		field = tfield.sub("%1", dir).sub("%2", i.to_s).sub("%3", color)
		text += "<td width='#{@size_field}px' align='left'>#{field}<td>"
		text += "<td width='#{6}px' align='center'>#{butclear}<td>"
		text += "<td width='#{10}px' align='center'>#{butdot}<td>" if @su_capa_ask_dir
		text += "</tr>"
	end	
	text += "</table></div>"	
	
	#Information Message
	message = "<br>- " + encode_to_ruby(@fullname)
	message += "<br>- " + encode_to_ruby(@dlg_precede_qum)
	message += "<br>- " + encode_to_ruby(@dlg_precede_exm)
	style = "color: brown ; font-size: 12px ; font-style: italic"
	sspan = "<span style='#{style}'>#{message}</span>"
	text += "<div><table width='100%' cellpadding='0' cellspacing='0'>"
	text += "<tr><td><span style='#{style}'>#{message}</span><td><tr>"
	text += "</table></div>"
	
	#Buttons
	scancel = "Cancel"
	ssave = "Cancel"
	butprint = format_button("ButtonPrint", @dlg_print)
	butcancel = format_button("ButtonCancel", @dlg_cancel)
	butsave = format_button("ButtonSave", @dlg_save)
	text += "<br><table class='NOPRINT_Style' width='100%' cellpadding='0' cellspacing='0'><tr>"
	text += "<td width='50%' align='left'>#{butprint}</td>"
	text += "<td width='25%' align='right'>#{butcancel}</td>"
	text += "<td align='right'>#{butsave}</td>"
	text += "</tr></table>"
	
	#End of body
	text += "</BODY>"

	text
end

#==============================================================
# Methods to be Executed Once at SU Startup
#==============================================================

#Menu and load for  Alternate Directory (called once)
unless @redirect_plugins_loaded
	@redirect_plugins_loaded = true
	self.startup
	self.register_execute
end

end	#module AdditionalPluginFolders6