--
-- CharLi v1 (aka Character Lighting)
-- 2023 by FreakaZ (+FlowerMan)
--
-- https://rootpunk.com/mod/
--
local CORE = {}
local SELF = {}

--// CORE:Initialize()
function CORE:Initialize()
	local o = {}
	setmetatable(o, self)
	self.__index = self

	-- public
	CORE.isReady = true
	CORE.isPaint = false
	CORE.isPhoto = false
	CORE.isModel = nil

	-- init me
	SELF:Initialize()

	return o
end

--// CORE:Interface()
function CORE:Interface()

	-- always update
	SELF:UpdateScreen()

	-- pre styles
	if SELF:DrawStyles(true)
	then
		-- start window
		if SELF:DrawWindow(true, false)
		then
			-- always update
			SELF:UpdateWindow()

			-- update circle count
			SELF.Counter = SELF:TableLength(SELF.Circles)

			-- start tabbar
			if SELF:DrawTabbar(true)
			then
				-- start tabitem (static)
				if SELF:DrawTabitem(true, "@")
				then
					-- make the tabbar line thicker, before the child
					SELF:DrawSeparator(SELF:WindowScale(2), "tabbar/bottomline")

					-- start tabchild
					if SELF:DrawTabchild(true, true)
					then
						-- draw title
						SELF:DrawSeparator(SELF:WindowScale(5))
						SELF:DrawSpacer(1,SELF:WindowScale(6))
						SELF:DrawTextCenter("SPAWNABLE", "category/header")
						SELF:DrawTextCenter(SELF:ConvertGlyph("LIGHTS"), "category/text")
						SELF:DrawSpacer(1,SELF:WindowScale(5))
						SELF:DrawSeparator(SELF:WindowScale(5))

						-- start light loop
						for SpawnID=0,SELF:TableLength(SELF.Element.Lights)-1
						do
							if SpawnID > 0 then SELF:DrawSeparator(SELF:WindowScale(1)) end
							SELF:DrawSpawnable(SELF.Element.Lights[SpawnID])
						end

						-- draw title
						SELF:DrawSeparator(SELF:WindowScale(5))
						SELF:DrawSpacer(1,SELF:WindowScale(6))
						SELF:DrawTextCenter("SPAWNABLE", "category/header")
						SELF:DrawTextCenter(SELF:ConvertGlyph("OBJECTS"), "category/text")
						SELF:DrawSpacer(1,SELF:WindowScale(5))
						SELF:DrawSeparator(SELF:WindowScale(5))

						-- start other loop
						for SpawnID=0,SELF:TableLength(SELF.Element.Others)-1
						do
							if SpawnID > 0 then SELF:DrawSeparator(SELF:WindowScale(1)) end
							SELF:DrawSpawnable(SELF.Element.Others[SpawnID])
						end

					-- end tabchild
					SELF:DrawTabchild(false)
					end

				-- end tabitem
				SELF:DrawTabitem(false)
				end

				-- start tabloop
				for GroupID,_ in pairs(SELF.Circles)
				do
					-- start tabitem (dynamic)
					if SELF:DrawTabitem(true, GroupID)
					then
						-- make the tabbar line thicker, before the child
						SELF:DrawSeparator(SELF:WindowScale(2), "tabbar/bottomline")

						-- start tabchild
						if SELF:DrawTabchild(true, true)
						then
							-- draw title
							SELF:DrawSeparator(SELF:WindowScale(5))
							SELF:DrawSpacer(1,SELF:WindowScale(6))
							SELF:DrawTextCenter("GROUP", "category/header")
							SELF:DrawTextCenter(SELF:ConvertGlyph("CONTROLS"), "category/text")
							SELF:DrawSpacer(1,SELF:WindowScale(5))
							SELF:DrawSeparator(SELF:WindowScale(5))

							-- stick to char (not yet ready)
							--local trigger, value = SELF:DrawQuickshift(GroupID,SELF.Circles[GroupID].Sticky,"Stick to Character")
							--if trigger then
							--	SELF.Circles[GroupID].Sticky = value
							--	if value then SELF:GroupUpdate() end
							--end

							local trigger = SELF:DrawDespawnbutton(GroupID, "Destroy Group")
							if trigger then SELF:GroupDestroy(GroupID) end


							SELF:DrawSeparator(SELF:WindowScale(2))

							-- create slider
							local trigger, value = SELF:DrawQuicktune("Circle Radius", SELF.Circles[GroupID].Radius, 0.5, 5, "%.2f")
							if trigger then
								SELF.Circles[GroupID].Radius = value
								SELF:GroupUpdate(GroupID)
							end

							-- create slider
							local trigger, value = SELF:DrawQuicktune("Radius Multiplier", SELF.Circles[GroupID].Expand, 1, 10, "%.1f")
							if trigger then
								SELF.Circles[GroupID].Expand = value
								SELF:GroupUpdate(GroupID)
							end

							-- create slider
							local trigger, value = SELF:DrawQuicktune("Circle Rotation", SELF.Circles[GroupID].Rotate, -180, 180, "%.0f")
							if trigger then
								SELF.Circles[GroupID].Rotate = value
								SELF:GroupUpdate(GroupID)
							end

							-- draw light controls if it has some
							if SELF.Circles[GroupID].Power ~= nil then
								-- draw title
								SELF:DrawSeparator(SELF:WindowScale(5))
								SELF:DrawSpacer(1,SELF:WindowScale(6))
								SELF:DrawTextCenter("LIGHT", "category/header")
								SELF:DrawTextCenter(SELF:ConvertGlyph("CONTROLS"), "category/text")
								SELF:DrawSpacer(1,SELF:WindowScale(5))
								SELF:DrawSeparator(SELF:WindowScale(5))

								-- create slider
								local trigger, value = SELF:DrawQuicktune("Light Intensity", SELF.Circles[GroupID].Power, 10, 1000, "%.0f")
								if trigger then
									SELF.Circles[GroupID].Power = value
									SELF:GroupColors(GroupID)
								end

								-- create slider
								local trigger, value = SELF:DrawQuicktune("Intensity Boost", SELF.Circles[GroupID].Boost, 1, 100, "%.0f")
								if trigger then
									SELF.Circles[GroupID].Boost = value
									SELF:GroupColors(GroupID)
								end

								SELF:DrawSeparator(SELF:WindowScale(2))

								-- create slider
								local trigger, value = SELF:DrawQuicktune("Color: Red", SELF.Circles[GroupID].Color.R, 0, 255, "%.0f")
								if trigger then
									SELF.Circles[GroupID].Color.R = value
									SELF:GroupColors(GroupID)
								end

								-- create slider
								local trigger, value = SELF:DrawQuicktune("Color: Green", SELF.Circles[GroupID].Color.G, 0, 255, "%.0f")
								if trigger then
									SELF.Circles[GroupID].Color.G = value
									SELF:GroupColors(GroupID)
								end

								-- create slider
								local trigger, value = SELF:DrawQuicktune("Color: Blue", SELF.Circles[GroupID].Color.B, 0, 255, "%.0f")
								if trigger then
									SELF.Circles[GroupID].Color.B = value
									SELF:GroupColors(GroupID)
								end
							end


							-- draw title
							SELF:DrawSeparator(SELF:WindowScale(5))
							SELF:DrawSpacer(1,SELF:WindowScale(6))
							SELF:DrawTextCenter("ADDITIONAL", "category/header")
							SELF:DrawTextCenter(SELF:ConvertGlyph("CONTROLS"), "category/text")
							SELF:DrawSpacer(1,SELF:WindowScale(5))
							SELF:DrawSeparator(SELF:WindowScale(5))


							-- create slider
							local trigger, value = SELF:DrawQuicktune("Offset: X Position", SELF.Circles[GroupID].PosX, -2.5, 2.5, "%.2f")
							if trigger then
								SELF.Circles[GroupID].PosX = value
								SELF:GroupUpdate(GroupID)
							end

							-- create slider
							local trigger, value = SELF:DrawQuicktune("Offset: Y Position", SELF.Circles[GroupID].PosY, -2.5, 2.5, "%.2f")
							if trigger then
								SELF.Circles[GroupID].PosY = value
								SELF:GroupUpdate(GroupID)
							end

							-- create slider
							local trigger, value = SELF:DrawQuicktune("Offset: Z Position", SELF.Circles[GroupID].PosZ, -1, 9, "%.2f")
							if trigger then
								SELF.Circles[GroupID].PosZ = value
								SELF:GroupUpdate(GroupID)
							end

							SELF:DrawSeparator(SELF:WindowScale(2))

							-- create slider
							local trigger, value = SELF:DrawQuicktune("Object Rotation", SELF.Circles[GroupID].RotR, -90, 90, "%.0f")
							if trigger then
								SELF.Circles[GroupID].RotR = value
								SELF:GroupUpdate(GroupID)
							end

							-- create slider
							local trigger, value = SELF:DrawQuicktune("Object Pitch", SELF.Circles[GroupID].RotP, -90, 90, "%.0f")
							if trigger then
								SELF.Circles[GroupID].RotP = value
								SELF:GroupUpdate(GroupID)
							end


							-- create slider
							local trigger, value = SELF:DrawQuicktune("Object Yaw", SELF.Circles[GroupID].RotY, -90, 90, "%.0f")
							if trigger then
								SELF.Circles[GroupID].RotY = value
								SELF:GroupUpdate(GroupID)
							end




						-- end tabchild
						SELF:DrawTabchild(false)
						end

					-- end tabitem
					SELF:DrawTabitem(false)
					end
				-- end tabloop
				end

			-- end tabbar
			end
			SELF:DrawTabbar(false)

			-- make the tabbar line thicker, before the child
			SELF:DrawSeparator(SELF:WindowScale(2), "tabbar/bottomline")

			-- draw who
			SELF:DrawSpacer(1,SELF:WindowScale(5))
			SELF:DrawSpacing(SELF:WindowScale(1 + SELF.Window.Border),1)
			SELF:DrawAlphabutton(SELF:ConvertGlyph("?"), "window/bottom/icon", 25, 21)
			ImGui.SameLine()
			ImGui.PushStyleColor(ImGuiCol.Text, SELF:GetColor("window/bottom/who"))
			ImGui.Text(SELF.Authors)
			ImGui.PopStyleColor(1)

		-- end window
		end
		SELF:DrawWindow(false)

	-- reset styles
	end
	SELF:DrawStyles(false)
end

--// CORE:UpdatePlayer()
function CORE:UpdatePlayer()

	-- ingame prefered!
	if not SELF:isPreGame()
	then
		-- predefine
		local pos = {}
		local yaw = 0

		-- fetch current position
		if CORE.isPhoto then
			pos = CORE.isModel:GetWorldPosition()
			yaw = CORE.isModel:GetWorldYaw()
		else
			pos = Game.GetPlayer():GetWorldPosition()
			yaw = Game.GetPlayer():GetWorldYaw()
		end

		-- update only if puppet moved
		if SELF.PuppetX ~= pos.x or SELF.PuppetY ~= pos.y or SELF.PuppetZ ~= pos.z or SELF.PuppYaw ~= yaw
		then
			SELF.PuppetX = pos.x
			SELF.PuppetY = pos.y
			SELF.PuppetZ = pos.z
			SELF.PuppYaw = yaw
			SELF:GroupUpdate()
			print(tostring(yaw))
		end
	end
end

function CORE:DestroyEverything()
	for id,_ in pairs(SELF.Circles) do
		SELF:GroupDestroy(id)
	end
end

--
--
--//////////////////// INTERNAL ////////////////////
--
--

--// SELF:Initialize()
function SELF:Initialize()
	local o = {}
	setmetatable(o, self)
	self.__index = self

	-- project
	SELF.Project = "CharLi v1.0"
	SELF.Authors = "FreakaZ + FlowerD"

	-- scaling
	SELF.Screen = {Width=0,Height=0,Factor=1,Usable=0,Border=50}
	SELF.Window = {Width=456,Height=600,Factor=1,Usable=0,Border=2}
	SELF.Legacy = false
	SELF.Compat = false
	SELF.Glyphs = false
	SELF.Scroll = true

	-- private
	SELF.Circles = {}
	SELF.Counter = 0
	SELF.PuppetX = 0
	SELF.PuppetY = 0
	SELF.PuppetZ = 0
	SELF.PuppYaw = 0

	-- elements
	SELF.Element = {
		Lights = {
			[0] = {name="Entropy Lamp", path="base\\charli\\nomesh\\entropy.ent", tags="Set Color, Set Intensity", light="Light5050", pos={z=1}, rot={r=-90} },
			[1] = {name="Entropy Lamp (Visible)", path="base\\charli\\mesh\\entropy.ent", tags="Set Color, Set Intensity", light="Light5050", pos={z=1}, rot={r=-90} },
			[2] = {name="Large Industrial", path="base\\charli\\nomesh\\industrial.ent", tags="Set Color, Set Intensity", light="Light6234", pos={z=1}, rot={r=-90,y=90} },
			[3] = {name="Large Industrial (Visible)", path="base\\charli\\mesh\\industrial.ent", tags="Collides with other Objects, Set Color, Set Intensity", light="Light6234", pos={z=1}, rot={r=-90,y=90} },
			[4] = {name="Neon Device", path="base\\charli\\nomesh\\neondevice.ent", tags="Set Color, Set Intensity", light="Light2103", pos={z=1} }
		},
		Others = {
			[0] = {name="Animated Fog", path="base\\open_world\\street_stories\\city_center\\downtown\\sts_cct_dtn_02\\entites\\sts_cct_dtn_02_dance_fog.ent", tags=nil, light=nil, pos={z=1} },
			[1] = {name="Race Checkpoint", path="base\\gameplay\\devices\\street_signs\\race_checkpoint\\race_checkpoint.ent", tags=nil, light=nil, pos={z=-1} },
			[2] = {name="Afterlife Sign", path="base\\environment\\quests\\q115_afterlife\\q115_bar_sign.ent", tags=nil, light=nil, pos={z=1} }
		}
	}

	-- legacy cet (no scaling)
	if tonumber(SELF:FilterNumbers(GetVersion())) <= 1200 then
		SELF.Legacy = true
	end

	-- legacy imgui (has no disable)
	if tonumber(SELF:FilterNumbers(GetVersion())) <= 1163 then
		SELF.Compat = true
	end

	-- has new icons (has shiny icons)
	if tonumber(SELF:FilterNumbers(GetVersion())) >= 1220 then
		SELF.Glyphs = true
	end

	return o
end

--
--
--//////////////////// INTERFACE ////////////////////
--
--

--// SELF:GetColor(<STRING>)
function SELF:GetColor(_color, _alpha)

	-- catch unset
	local color = _color or false
	local alpha = _alpha or 1

	-- no color = transparency
	if not color then					return ImGui.GetColorU32(0, 0, 0, 0) end

	-- window
	if color == "window/text"				then return ImGui.GetColorU32(0.8, 0.8, 0.85, 1) end
	if color == "window/border"				then return ImGui.GetColorU32(0.2, 0.2, 0.23, 0.3) end
	if color == "window/background"				then return ImGui.GetColorU32(0.01, 0.01, 0.03, 0.8) end
	if color == "window/resize"				then return ImGui.GetColorU32(0.3, 0.3, 0.33, 0.7) end
	if color == "window/resize/active"			then return ImGui.GetColorU32(1, 0.56, 0.13, 0.85) end
	if color == "window/resize/hovered"			then return ImGui.GetColorU32(1, 0.56, 0.13, 1) end
	if color == "window/scrollbar"				then return ImGui.GetColorU32(0.3, 0.3, 0.33, 0.7) end
	if color == "window/scrollbar/active"			then return ImGui.GetColorU32(1, 0.56, 0.13, 0.85) end
	if color == "window/scrollbar/hovered"			then return ImGui.GetColorU32(1, 0.56, 0.13, 1) end
	if color == "window/scrollbar/background"		then return ImGui.GetColorU32(0, 0, 0, 0) end
	if color == "window/separator"				then return ImGui.GetColorU32(0.2, 0.2, 0.23, 0.7) end
	if color == "window/bottom/who"				then return ImGui.GetColorU32(0.5, 0.5, 0.53, 0.7) end
	if color == "window/bottom/icon"			then return ImGui.GetColorU32(0.4, 0.4, 0.43, 0.7) end

	-- tabbar
	if color == "tabbar/text"				then return ImGui.GetColorU32(0.8, 0.8, 0.85, 1) end
	if color == "tabbar/tab"				then return ImGui.GetColorU32(0.2, 0.2, 0.23, 0.7) end
	if color == "tabbar/tab/active"				then return ImGui.GetColorU32(1, 0.56, 0.13, 0.85) end
	if color == "tabbar/tab/hovered"			then return ImGui.GetColorU32(1, 0.56, 0.13, 1) end
	if color == "tabbar/bottomline"				then return ImGui.GetColorU32(1, 0.56, 0.13, 0.85) end

	-- customs
	if color == "category/text"				then return ImGui.GetColorU32(1, 0.56, 0.13, 0.9) end
	if color == "category/header"				then return ImGui.GetColorU32(0.7, 0.7, 0.73, 0.9) end

	-- spawnable
	if color == "spawnable/glyph"				then return ImGui.GetColorU32(1, 0.56, 0.13, 0.8) end
	if color == "spawnable/text"				then return ImGui.GetColorU32(0.9, 0.9, 1, 0.9) end
	if color == "spawnable/button"				then return ImGui.GetColorU32(1, 0.56, 0.13, 0.85) end
	if color == "spawnable/button/active"			then return ImGui.GetColorU32(1, 0.56, 0.13, 0.7) end
	if color == "spawnable/button/hovered"			then return ImGui.GetColorU32(1, 0.56, 0.13, 1) end
	if color == "spawnable/infoglyph"			then return ImGui.GetColorU32(0.4, 0.4, 0.43, 0.7) end
	if color == "spawnable/infotext"			then return ImGui.GetColorU32(0.7, 0.7, 0.73, 0.9) end

	-- quickshift
	if color == "quickshift/title"				then return ImGui.GetColorU32(0.8, 0.8, 0.85, 0.9) end
	if color == "quickshift/title/active"			then return ImGui.GetColorU32(1, 0.56, 0.13, 0.85) end
	if color == "quickshift/border"				then return ImGui.GetColorU32(0.3, 0.3, 0.33, 0.7) end
	if color == "quickshift/background"			then return ImGui.GetColorU32(0.2, 0.2, 0.23, 0.7) end
	if color == "quickshift/background/active"		then return ImGui.GetColorU32(0.2, 0.2, 0.23, 0.7) end
	if color == "quickshift/background/hovered"		then return ImGui.GetColorU32(0.2, 0.2, 0.23, 0.7) end
	if color == "quickshift/on/grab/normal"			then return ImGui.GetColorU32(1, 0.56, 0.13, 0.85) end
	if color == "quickshift/on/grab/active"			then return ImGui.GetColorU32(1, 0.56, 0.13, 1) end
	if color == "quickshift/off/grab/normal"		then return ImGui.GetColorU32(0.5, 0.5, 0.53, 0.7) end
	if color == "quickshift/off/grab/active"		then return ImGui.GetColorU32(0.6, 0.6, 0.63, 0.8) end

	-- quicktune
	if color == "quicktune/title"				then return ImGui.GetColorU32(0.8, 0.8, 0.85, 0.9) end
	if color == "quicktune/text"				then return ImGui.GetColorU32(0.9, 0.9, 1, 0.9) end
	if color == "quicktune/grab"				then return ImGui.GetColorU32(0.8, 0.8, 0.85, 0.9) end
	if color == "quicktune/border"				then return ImGui.GetColorU32(1, 0.56, 0.13, 0.7) end
	if color == "quicktune/progress"			then return ImGui.GetColorU32(0.9, 0.9, 1, 0.125) end
	if color == "quicktune/background"			then return ImGui.GetColorU32(0.15, 0.15, 0.18, 1) end
	if color == "quicktune/decoration"			then return ImGui.GetColorU32(0.4, 0.4, 0.43, 0.7) end







	-- fallback
	-- (50% red to see missing stuff)
	return ImGui.GetColorU32(1, 0, 0, 0.5)
end

--// SELF:DrawStyles(<BOOL>)
function SELF:DrawStyles(_start)
	if CORE.isReady and _start then
		-- apply styles
		ImGui.PushStyleVar(ImGuiStyleVar.ScrollbarSize, 0)
		ImGui.PushStyleVar(ImGuiStyleVar.ScrollbarRounding, 0)
		ImGui.PushStyleVar(ImGuiStyleVar.FramePadding, 0, 0)
		ImGui.PushStyleVar(ImGuiStyleVar.FrameRounding, 0)
		ImGui.PushStyleVar(ImGuiStyleVar.FrameBorderSize, 0)
		ImGui.PushStyleVar(ImGuiStyleVar.ItemSpacing, 0, 0)
		ImGui.PushStyleVar(ImGuiStyleVar.ItemInnerSpacing, SELF:WindowScale(2), 0)
		ImGui.PushStyleVar(ImGuiStyleVar.CellPadding, 0, 0)
		ImGui.PushStyleVar(ImGuiStyleVar.GrabRounding, SELF:WindowScale(2))
		ImGui.PushStyleVar(ImGuiStyleVar.WindowPadding, 0, 0)
		ImGui.PushStyleVar(ImGuiStyleVar.ChildBorderSize, 0)
		ImGui.PushStyleVar(ImGuiStyleVar.GrabMinSize, SELF:WindowScale(10))
		ImGui.PushStyleVar(ImGuiStyleVar.DisabledAlpha, 0.70)
		return true
	elseif CORE.isReady and not _start then
		-- drop styles
		ImGui.PopStyleVar(13)
		return true
	else
		return false
	end
end

--// SELF:DrawWindow(<BOOL>,<BOOL>)
function SELF:DrawWindow(_start, _force)
	if CORE.isReady and _start
	then
		-- initial position
		ImGui.SetNextWindowPos(100, 100, ImGuiCond.FirstUseEver)

		-- force resize
		if _force then
			ImGui.SetNextWindowSizeConstraints(SELF:ScreenScale(456), SELF:ScreenScale(600), SELF.Screen.Usable, SELF.Screen.Height - 50)
		else
			ImGui.SetNextWindowSizeConstraints(SELF:ScreenScale(456), SELF:ScreenScale(600), SELF:ScreenScale(456), SELF.Screen.Height - 50)
		end

		-- apply colors
		ImGui.PushStyleColor(ImGuiCol.Text, SELF:GetColor("window/text"))
		ImGui.PushStyleColor(ImGuiCol.Border, SELF:GetColor("window/border"))
		ImGui.PushStyleColor(ImGuiCol.WindowBg, SELF:GetColor("window/background"))
		ImGui.PushStyleColor(ImGuiCol.TitleBg, SELF:GetColor("window/background"))
		ImGui.PushStyleColor(ImGuiCol.TitleBgActive, SELF:GetColor("window/background"))
		ImGui.PushStyleColor(ImGuiCol.TitleBgCollapsed, SELF:GetColor("window/background"))
		ImGui.PushStyleColor(ImGuiCol.ResizeGrip, SELF:GetColor("window/resize"))
		ImGui.PushStyleColor(ImGuiCol.ResizeGripActive, SELF:GetColor("window/resize/active"))
		ImGui.PushStyleColor(ImGuiCol.ResizeGripHovered, SELF:GetColor("window/resize/hovered"))
		ImGui.PushStyleColor(ImGuiCol.Button, SELF:GetColor())
		ImGui.PushStyleColor(ImGuiCol.ButtonActive, SELF:GetColor())
		ImGui.PushStyleColor(ImGuiCol.ButtonHovered, SELF:GetColor())

		-- apply styles
		ImGui.PushStyleVar(ImGuiStyleVar.WindowBorderSize, SELF:BorderScale(SELF.Window.Border))
		ImGui.PushStyleVar(ImGuiStyleVar.WindowRounding, SELF:ScreenScale(6))
		ImGui.PushStyleVar(ImGuiStyleVar.FramePadding, SELF:WindowScale(10), SELF:WindowScale(10))
		ImGui.PushStyleVar(ImGuiStyleVar.ItemInnerSpacing, SELF:WindowScale(7), 0)

		-- predefine
		local _trigger

		-- has glyphs
		if SELF.Glyphs
		then
			-- apply style (right, with arrow)
			ImGui.PushStyleVar(ImGuiStyleVar.WindowTitleAlign, 1, 0.55)

			-- create window (flashy glyphs)
			_trigger = ImGui.Begin("\u{f0af0}\u{f0af5}\u{f0aee}\u{f0aff} \u{f0c0d}\u{f0c04}", ImGuiWindowFlags.NoScrollbar)

			-- drop style
			ImGui.PopStyleVar(1)

		-- no glyphs :(
		else
			-- apply style (left, with arrow)
			ImGui.PushStyleVar(ImGuiStyleVar.WindowTitleAlign, 1, 0.45)

			-- create window (real oldschool)
			_trigger = ImGui.Begin(SELF.Project, ImGuiWindowFlags.NoScrollbar)

			-- drop style
			ImGui.PopStyleVar(1)
		end

		-- set font scale
		ImGui.SetWindowFontScale(SELF.Window.Factor)

		-- drop styles
		ImGui.PopStyleVar(4)

		-- drop colors
		ImGui.PopStyleColor(12)

		-- window trigger
		return _trigger

	elseif CORE.isReady and not _start
	then
		ImGui.End()
	else
		return false
	end
end

--// SELF:Tabbar(<BOOL>)
function SELF:DrawTabbar(_start)
	if CORE.isReady and _start
	then
		-- fix left border size
		if SELF:BorderScale(SELF.Window.Border) > 0 then
			SELF:DrawSpacing(SELF:BorderScale(SELF.Window.Border),1)
		end

		-- apply styles
		--ImGui.PushStyleColor(ImGuiCol.TabActive, SELF:GetColor("tabbar/bottomline"))
		ImGui.PushStyleColor(ImGuiCol.TabActive, SELF:GetColor())
		ImGui.PushStyleVar(ImGuiStyleVar.FramePadding, SELF:WindowScale(10), SELF:WindowScale(5))

		-- create tabbar
		local _trigger = ImGui.BeginTabBar("DE_TB", ImGuiTabBarFlags.AutoSelectNewTabs)

		-- drop styles
		ImGui.PopStyleVar(1)
		ImGui.PopStyleColor(1)

		-- tabbar trigger
		return _trigger

	elseif CORE.isReady and not _start
	then
		ImGui.EndTabBar()
	else
		return false
	end
end

--// SELF:DrawTabitem(<BOOL>,<STRING>)
function SELF:DrawTabitem(_start, _title)
	if CORE.isReady and _start
	then
		-- apply styles
		ImGui.PushStyleColor(ImGuiCol.Text, SELF:GetColor("tabbar/text"))
		ImGui.PushStyleColor(ImGuiCol.Tab, SELF:GetColor("tabbar/tab"))
		ImGui.PushStyleColor(ImGuiCol.TabActive, SELF:GetColor("tabbar/tab/active"))
		ImGui.PushStyleColor(ImGuiCol.TabHovered, SELF:GetColor("tabbar/tab/hovered"))

		if _title == "@" then
			ImGui.PushStyleVar(ImGuiStyleVar.FramePadding, SELF:WindowScale(10), SELF:WindowScale(5))
		else
			ImGui.PushStyleVar(ImGuiStyleVar.FramePadding, SELF:WindowScale(9), SELF:WindowScale(5))
		end

		-- create tabitem
		local _trigger = ImGui.BeginTabItem(tostring(SELF:ConvertGlyph(tostring(_title))))

		-- drop styles
		ImGui.PopStyleVar(1)
		ImGui.PopStyleColor(4)

		-- tabitem trigger
		return _trigger

	elseif CORE.isReady and not _start
	then
		ImGui.EndTabItem()
	else
		return false
	end
end

--// SELF:DrawTabchild(<BOOl>)
function SELF:DrawTabchild(_start)
	if CORE.isReady and _start
	then
		-- apply styles
		ImGui.PushStyleColor(ImGuiCol.ScrollbarBg, SELF:GetColor("window/scrollbar/background"))
		ImGui.PushStyleColor(ImGuiCol.ScrollbarGrab, SELF:GetColor("window/scrollbar"))
		ImGui.PushStyleColor(ImGuiCol.ScrollbarGrabActive, SELF:GetColor("window/scrollbar/active"))
		ImGui.PushStyleColor(ImGuiCol.ScrollbarGrabHovered, SELF:GetColor("window/scrollbar/hovered"))
		if SELF.Scroll then ImGui.PushStyleVar(ImGuiStyleVar.ScrollbarSize, SELF:WindowScale(10)) end

		-- create tabchild
		local _trigger = ImGui.BeginChild("DE_TC", SELF.Window.Width, SELF.Window.Height - SELF:WindowScale(100), false, ImGuiWindowFlags.AlwaysVerticalScrollbar + ImGuiWindowFlags.NoBackground)

		-- drop styles
		if SELF.Scroll then ImGui.PopStyleVar(1) end
		ImGui.PopStyleColor(4)

		-- tabchild trigger
		return _trigger

	elseif CORE.isReady and not _start
	then
		ImGui.EndChild()
	else
		return false
	end
end

--// SELF:DrawQuickshift(<INT>)
function SELF:DrawQuickshift(_id, _value, _title)

	-- top spacer
	SELF:DrawSpacer(1,SELF:WindowScale(8))

	-- left space
	SELF:DrawSpacing(SELF:WindowScale(16 + SELF.Window.Border),1)

	-- draw title
	if _value then
		ImGui.PushStyleColor(ImGuiCol.Text, SELF:GetColor("quickshift/title/active"))
	else
		ImGui.PushStyleColor(ImGuiCol.Text, SELF:GetColor("quickshift/title"))
	end
	ImGui.Text(_title)
	ImGui.PopStyleColor(1)
	ImGui.SameLine()

	-- fill space between
	SELF:DrawSpacing(SELF.Window.Usable - (SELF:TextWidth(_title) + SELF:WindowScale(70)),1)

	-- fixed width to keep aspect ratio
	ImGui.SetNextItemWidth(SELF:WindowScale(32))

	-- add stacks
	if _value then
		ImGui.PushStyleColor(ImGuiCol.SliderGrab, SELF:GetColor("quickshift/on/grab/normal"))
		ImGui.PushStyleColor(ImGuiCol.SliderGrabActive, SELF:GetColor("quickshift/on/grab/active"))
	else
		ImGui.PushStyleColor(ImGuiCol.SliderGrab, SELF:GetColor("quickshift/off/grab/normal"))
		ImGui.PushStyleColor(ImGuiCol.SliderGrabActive, SELF:GetColor("quickshift/off/grab/active"))
	end

	ImGui.PushStyleColor(ImGuiCol.Text, SELF:GetColor())
	ImGui.PushStyleColor(ImGuiCol.Border, SELF:GetColor("quickshift/border"))
	ImGui.PushStyleColor(ImGuiCol.FrameBg, SELF:GetColor("quickshift/background"))
	ImGui.PushStyleColor(ImGuiCol.FrameBgActive, SELF:GetColor("quickshift/background/active"))
	ImGui.PushStyleColor(ImGuiCol.FrameBgHovered, SELF:GetColor("quickshift/background/hovered"))

	ImGui.PushStyleVar(ImGuiStyleVar.GrabRounding, SELF:WindowScale(10))
	ImGui.PushStyleVar(ImGuiStyleVar.FrameRounding, SELF:WindowScale(10))
	ImGui.PushStyleVar(ImGuiStyleVar.FramePadding, 0, 0)
	ImGui.PushStyleVar(ImGuiStyleVar.FrameBorderSize, SELF:WindowScale(2))

	-- draw minified slider
	ImGui.PushID("CL_QS")
	local _return, _trigger = ImGui.SliderInt("", SELF:BoolToInt(_value), 1, 0)
	ImGui.PopID()

	-- drop stacks
	ImGui.PopStyleVar(4)
	ImGui.PopStyleColor(7)

	-- bottom spacer
	SELF:DrawSpacer(1,SELF:WindowScale(9))

	-- result
	return _trigger, SELF:IntToBool(_return)
end

--// SELF:DrawQuicktune(<STRING>,<FLOAT>,<FLOAT>,<FLOAT>,<STRING>)
function SELF:DrawQuicktune(title, value, min, max, res)

	-- default spacing
	local _left = SELF:WindowScale(6)
	local _right = SELF:WindowScale(40)

	-- calculate slider width
	local _width = SELF.Window.Width - (_left + _right)

	-- top deco
	SELF:DrawSpacer(1,SELF:WindowScale(5))
	SELF:DrawSpacing(_left,1)

	-- paint decoration
	ImGui.PushStyleColor(ImGuiCol.Text, SELF:GetColor("quicktune/decoration"))
	ImGui.Text("»")
	ImGui.PopStyleColor(1)

	-- paint title
	ImGui.SameLine()
	SELF:DrawSpacing(SELF:WindowScale(5),1)
	ImGui.PushStyleColor(ImGuiCol.Text, SELF:GetColor("quicktune/title"))
	ImGui.Text(title)
	ImGui.PopStyleColor(1)

	-- spacer between title and slider
	SELF:DrawSpacer(1,SELF:WindowScale(6))

	-- left deco border
	SELF:DrawSpacing(_left + SELF:WindowScale(13),1)
	ImGui.PushStyleColor(ImGuiCol.FrameBg, SELF:GetColor())
	ImGui.PushStyleColor(ImGuiCol.PlotHistogram, SELF:GetColor("quicktune/border"))
	ImGui.PushStyleVar(ImGuiStyleVar.FrameRounding, SELF:WindowScale(3))
	ImGui.ProgressBar(1, SELF:WindowScale(6), SELF:WindowScale(26), "")
	ImGui.PopStyleVar(1)
	ImGui.PopStyleColor(2)
	ImGui.SameLine()

	-- right deco border
	SELF:DrawSpacing(_width - SELF:WindowScale(10),1)
	ImGui.PushStyleColor(ImGuiCol.FrameBg, SELF:GetColor())
	ImGui.PushStyleColor(ImGuiCol.PlotHistogram, SELF:GetColor("quicktune/border"))
	ImGui.PushStyleVar(ImGuiStyleVar.FrameRounding, SELF:WindowScale(3))
	ImGui.ProgressBar(1, SELF:WindowScale(6), SELF:WindowScale(26), "")
	ImGui.PopStyleVar(1)
	ImGui.PopStyleColor(2)
	ImGui.SameLine()

	-- overwriting progress background
	SELF:DrawSpacing(-(_width - SELF:WindowScale(2)),1)
	ImGui.PushStyleColor(ImGuiCol.FrameBg, SELF:GetColor("quicktune/background"))
	ImGui.PushStyleColor(ImGuiCol.PlotHistogram, SELF:GetColor("quicktune/progress"))
	ImGui.ProgressBar(SELF:ValueToProgress(min, max, value), _width - SELF:WindowScale(6), SELF:WindowScale(26), "")
	ImGui.PopStyleColor(2)
	ImGui.SameLine()

	-- position the slider above the deco
	SELF:DrawSpacing(-(_width - SELF:WindowScale(3)),1)

	-- make sure it has the right width
	ImGui.SetNextItemWidth(_width)

	-- add stacks
	ImGui.PushStyleColor(ImGuiCol.Text, SELF:GetColor("quicktune/text"))
	ImGui.PushStyleColor(ImGuiCol.SliderGrab, SELF:GetColor("quicktune/grab"))
	ImGui.PushStyleColor(ImGuiCol.SliderGrabActive, SELF:GetColor("quicktune/grab"))
	ImGui.PushStyleColor(ImGuiCol.FrameBg, SELF:GetColor())
	ImGui.PushStyleColor(ImGuiCol.FrameBgActive, SELF:GetColor())
	ImGui.PushStyleColor(ImGuiCol.FrameBgHovered, SELF:GetColor())
	ImGui.PushStyleVar(ImGuiStyleVar.FramePadding, SELF:WindowScale(2), SELF:WindowScale(4))
	ImGui.PushStyleVar(ImGuiStyleVar.GrabMinSize, SELF:WindowScale(4))

	-- make them local before
	ImGui.PushID("DE_QT"..title)
	_return, _trigger = ImGui.SliderFloat("", value, min, max, res)
	ImGui.PopID()

	-- drop stacks
	ImGui.PopStyleVar(2)
	ImGui.PopStyleColor(6)

	-- bottom
	SELF:DrawSpacer(1,SELF:WindowScale(8))

	-- result
	return _trigger, _return
end

--// SELF:DrawSpacer(<INT>,<INT>)
function SELF:DrawSpacer(_width, _height)

	-- catch unset
	local width = _width or 1
	local height = _height or 1

	ImGui.Dummy(width, height)
end


--// SELF:DrawSpacing(<INT>,<INT>)
function SELF:DrawSpacing(_width, _height)

	-- catch unset
	local width = _width or 1
	local height = _height or 1

	ImGui.Dummy(width, height)
	ImGui.SameLine()
end

--// SELF:DrawSeparator()
function SELF:DrawSeparator(_lines, _color)

	-- catch unset
	local lines = _lines or 1
	local color = _color or "window/separator"

	-- apply styles
	ImGui.PushStyleColor(ImGuiCol.Separator, SELF:GetColor(color))
	ImGui.PushStyleVar(ImGuiStyleVar.ItemSpacing, 0, 1)

	-- multiple if wanted
	if lines > 0 then
		for i=1, lines do ImGui.Separator() end
	end

	-- drop styles
	ImGui.PopStyleVar(1)
	ImGui.PopStyleColor(1)
end


--// SELF:DrawTextCenter(<STRING>)
function SELF:DrawTextCenter(_text, _color)

	-- catch unset
	local text = _text or ""
	local color = _color or "window/text"

	SELF:DrawSpacing(math.floor((SELF.Window.Width - SELF:TextWidth(text)) / 2))

	-- apply styles
	ImGui.PushStyleColor(ImGuiCol.Text, SELF:GetColor(color))

	ImGui.Text(text)

	-- drop styles
	ImGui.PopStyleColor(1)
end

--// SELF:DrawSpawnable(<TABLE>)
function SELF:DrawSpawnable(_object)

	-- top spacer
	SELF:DrawSpacer(1,SELF:WindowScale(9))

	-- draw header
	SELF:DrawSpacing(SELF:WindowScale(7 + SELF.Window.Border),1)
	SELF:DrawAlphabutton(SELF:ConvertGlyph("@"), "spawnable/glyph", 25, 21)
	ImGui.SameLine()
	ImGui.Text(_object.name)

	-- draw spawnbuttons
	SELF:DrawSpacer(1,SELF:WindowScale(5))
	SELF:DrawSpacing(SELF:WindowScale(32 + SELF.Window.Border),1)

	-- use disable
	if SELF:isPreGame() or SELF.Counter >= 10 then ImGui.BeginDisabled() end

	if SELF:DrawSpawnbutton(_object.path..tostring(1), "Spawn x1") then
		SELF:GroupCreate(_object, 1)
	end
	ImGui.SameLine()
	SELF:DrawSpacing(SELF:WindowScale(9),1)
	if SELF:DrawSpawnbutton(_object.path..tostring(2), "Spawn x2") then
		SELF:GroupCreate(_object, 2)
	end
	ImGui.SameLine()
	SELF:DrawSpacing(SELF:WindowScale(9),1)
	if SELF:DrawSpawnbutton(_object.path..tostring(4), "Spawn x4") then
		SELF:GroupCreate(_object, 4)
	end
	ImGui.SameLine()
	SELF:DrawSpacing(SELF:WindowScale(9),1)
	if SELF:DrawSpawnbutton(_object.path..tostring(8), "Spawn x8") then
		SELF:GroupCreate(_object, 8)
	end
	ImGui.SameLine()
	SELF:DrawSpacing(SELF:WindowScale(9),1)
	if SELF:DrawSpawnbutton(_object.path..tostring(16), "Spawn x16") then
		SELF:GroupCreate(_object, 16)
	end

	-- use disable
	if SELF:isPreGame() or SELF.Counter >= 10 then ImGui.EndDisabled() end

	-- draw tags
	if _object.tags ~= nil then
		SELF:DrawSpacer(1,SELF:WindowScale(5))
		SELF:DrawSpacing(SELF:WindowScale(27 + SELF.Window.Border),1)
		SELF:DrawAlphabutton(SELF:ConvertGlyph("!"), "spawnable/infoglyph", 25, 23)
		ImGui.SameLine()
		ImGui.PushStyleColor(ImGuiCol.Text, SELF:GetColor("spawnable/infotext"))
		ImGui.Text(_object.tags)
		ImGui.PopStyleColor(1)
	else
		SELF:DrawSpacer(1,SELF:WindowScale(6))
	end

	-- bottom spacer
	SELF:DrawSpacer(1,SELF:WindowScale(8))
end

--// SELF:DrawAlphabutton(<STRING>,<STRING>,<INT>,<INT>)
function SELF:DrawAlphabutton(_text, _color, _width, _height)

	-- catch unset
	local text = _text or ""
	local color = _color or "window/text"
	local width = _width or 1
	local height = _height or 1

	-- apply styles
	ImGui.PushStyleColor(ImGuiCol.Text, SELF:GetColor(color))
	ImGui.PushStyleColor(ImGuiCol.Button, SELF:GetColor())
	ImGui.PushStyleColor(ImGuiCol.ButtonActive, SELF:GetColor())
	ImGui.PushStyleColor(ImGuiCol.ButtonHovered, SELF:GetColor())

	-- element id
	ImGui.PushID("CL_"..color..text)
	local _blind = ImGui.Button(text, SELF:WindowScale(width), SELF:WindowScale(height))
	ImGui.PopID()

	-- drop styles
	ImGui.PopStyleColor(4)
end

--// SELF:DrawSpawnbutton
function SELF:DrawSpawnbutton(_id, _text)

	-- catch unset
	local id = _id or "idontknow"
	local text = _text or ""

	-- apply styles
	ImGui.PushStyleColor(ImGuiCol.Text, SELF:GetColor("spawnable/text"))
	ImGui.PushStyleColor(ImGuiCol.Button, SELF:GetColor("spawnable/button"))
	ImGui.PushStyleColor(ImGuiCol.ButtonActive, SELF:GetColor("spawnable/button/active"))
	ImGui.PushStyleColor(ImGuiCol.ButtonHovered, SELF:GetColor("spawnable/button/hovered"))
	ImGui.PushStyleVar(ImGuiStyleVar.FrameRounding, SELF:WindowScale(3))

	-- element id
	ImGui.PushID("CL_"..id..text)
	local _trigger = ImGui.Button(text, SELF:TextWidth(text) + SELF:WindowScale(10), SELF:TextHeight(text) + SELF:WindowScale(5))
	ImGui.PopID()

	-- drop styles
	ImGui.PopStyleVar(1)
	ImGui.PopStyleColor(4)

	-- spawn trigger
	return _trigger
end

--// SELF:DrawSpawnbutton
function SELF:DrawDespawnbutton(_id, _text)

	-- catch unset
	local id = _id or "idontknow"
	local text = _text or ""

	-- default spacing
	local _left = SELF:WindowScale(19)
	local _right = SELF:WindowScale(23)

	-- calculate slider width
	local _width = SELF.Window.Width - (SELF.Window.Border * 2) - (_left + _right)

	SELF:DrawSpacer(1, SELF:WindowScale(8))

	SELF:DrawSpacing(_left)

	-- apply styles
	ImGui.PushStyleColor(ImGuiCol.Text, SELF:GetColor("spawnable/text"))
	ImGui.PushStyleColor(ImGuiCol.Button, SELF:GetColor("spawnable/button"))
	ImGui.PushStyleColor(ImGuiCol.ButtonActive, SELF:GetColor("spawnable/button/active"))
	ImGui.PushStyleColor(ImGuiCol.ButtonHovered, SELF:GetColor("spawnable/button/hovered"))
	ImGui.PushStyleVar(ImGuiStyleVar.FrameRounding, SELF:WindowScale(3))

	-- element id
	ImGui.PushID("CL_"..id..text)
	local _trigger = ImGui.Button(text, _width, SELF:TextHeight(text) + SELF:WindowScale(5))
	ImGui.PopID()

	-- drop styles
	ImGui.PopStyleVar(1)
	ImGui.PopStyleColor(4)

	SELF:DrawSpacer(1, SELF:WindowScale(8))

	-- spawn trigger
	return _trigger
end




--
--
--//////////////////// SCALING ////////////////////
--
--

--// SELF:UpdateScreen()
function SELF:UpdateScreen()

	-- get dimension
	local width, height = GetDisplayResolution()

	-- update only if something has changed
	if SELF.Screen.Width ~= width or SELF.Screen.Height ~= height
	then
		-- update dimension
		SELF.Screen.Width = width
		SELF.Screen.Height = height

		-- update factor & usable
		SELF.Screen.Usable = math.floor((height / 9 * 16) / 2.5)
		SELF.Screen.Factor = SELF:ShortenFloat((height / 9 * 16) / 1920) -- we use the screen height to keep the aspect ratio
	end
end

--// SELF:UpdateWindow()
function SELF:UpdateWindow()

	-- get dimension
	local width = ImGui.GetWindowWidth()
	local height = ImGui.GetWindowHeight()

	-- update only if something has changed
	if SELF.Window.Width ~= width or SELF.Window.Height ~= height
	then
		-- update dimension
		SELF.Window.Width = width
		SELF.Window.Height = height

		-- update factor & usable
		SELF.Window.Usable = width - (SELF.Window.Border * 2)
		SELF.Window.Factor = SELF:ShortenFloat(SELF.Window.Width / (456 * SELF.Screen.Factor))
	end
end

--// SELF:ScreenScale(<INT>,<FLOAT>)
function SELF:ScreenScale(_pixels, _factor)

	-- catch non set
	local pixels = _pixels or 0
	local factor = _factor or SELF.Screen.Factor

	if not SELF.Legacy and pixels > 0 then
		return SELF:ShortenFloat(factor * pixels)
	end
	return pixels
end

--// SELF:WindowScale(<INT>,<FLOAT>)
function SELF:WindowScale(_pixels, _factor)

	-- catch non set
	local pixels = _pixels or 0
	local factor = _factor or SELF.Window.Factor

	if not SELF.Legacy and pixels > 0 then
		return SELF:ShortenFloat((SELF.Screen.Factor * factor) * pixels)
	end
	return pixels
end

--// SELF:BorderScale(<INT>,<FLOAT>)
function SELF:BorderScale(_pixels, _factor)

	-- catch non set
	local pixels = _pixels or 0
	local factor = _factor or SELF.Screen.Factor

	if not SELF.Legacy and pixels > 0 then
		return math.ceil(factor * pixels)
	end
	return pixels
end

--
--
--//////////////////// CIRCLES ////////////////////
--
--

--// SELF:GroupCreate(<TABLE>,<INT>)
function SELF:GroupCreate(object,number)

	-- search free spot
	local function FindSpot()
		for i=1,10,1 do
			if SELF.Circles[i] == nil then
				return i
			end
		end
		return nil
	end

	-- get spot in tabbar
	local GroupID = FindSpot()

	-- define group
	SELF.Circles[GroupID] = {}
	SELF.Circles[GroupID].Sticky = true		-- stick to player
	SELF.Circles[GroupID].Radius = 1.5 + (0.2 * number)	-- default radius
	SELF.Circles[GroupID].Expand = 1		-- distance multiplier
	SELF.Circles[GroupID].Number = number		-- number of lights
	SELF.Circles[GroupID].Lights = {}		-- list of lights
	SELF.Circles[GroupID].Rotate = 0		-- group rotation

	--SELF.Circles[GroupID].Grades = 0		-- base group rotation
	--SELF.Circles[GroupID].Pitch = 0

	-- is changeable lightsource
	if object.light ~= nil then
		SELF.Circles[GroupID].Codes = object.light
		SELF.Circles[GroupID].Power = 100.0
		SELF.Circles[GroupID].Boost = 1
		SELF.Circles[GroupID].Color = {}
		SELF.Circles[GroupID].Color.R = 255
		SELF.Circles[GroupID].Color.G = 255
		SELF.Circles[GroupID].Color.B = 255
	end

	-- default offsets
	SELF.Circles[GroupID].PosX = 0
	SELF.Circles[GroupID].PosY = 0
	SELF.Circles[GroupID].PosZ = 0
	SELF.Circles[GroupID].RotR = 0
	SELF.Circles[GroupID].RotP = 0
	SELF.Circles[GroupID].RotY = 0

	-- has defined offsets
	if object.pos ~= nil then
		if object.pos.x ~= nil then SELF.Circles[GroupID].PosX = object.pos.x end
		if object.pos.y ~= nil then SELF.Circles[GroupID].PosY = object.pos.y end
		if object.pos.z ~= nil then SELF.Circles[GroupID].PosZ = object.pos.z end
	end
	if object.rot ~= nil then
		if object.rot.r ~= nil then SELF.Circles[GroupID].RotR = object.rot.r end
		if object.rot.p ~= nil then SELF.Circles[GroupID].RotP = object.rot.p end
		if object.rot.y ~= nil then SELF.Circles[GroupID].RotY = object.rot.y end
	end

	-- calculate & spawn
	local yaw = 0
	for LightID=0,SELF.Circles[GroupID].Number -1,1
	do
		local alpha = math.rad((360 / SELF.Circles[GroupID].Number) * LightID)
		local x = math.sin(alpha) * (SELF.Circles[GroupID].Radius * SELF.Circles[GroupID].Expand)
		local y = math.cos(alpha) * (SELF.Circles[GroupID].Radius * SELF.Circles[GroupID].Expand)
		if LightID == 0 then yaw = math.deg(alpha) + 180 else yaw = yaw - (360 / SELF.Circles[GroupID].Number) end

		local pos = Vector4.new(SELF.PuppetX + SELF.Circles[GroupID].PosX + x, SELF.PuppetY + SELF.Circles[GroupID].PosY + y, SELF.PuppetZ + SELF.Circles[GroupID].PosZ, 1)
		local rot = EulerAngles.new(SELF.Circles[GroupID].RotR, SELF.Circles[GroupID].RotP, SELF.Circles[GroupID].RotY + yaw)
		SELF.Circles[GroupID].Lights[LightID] = SELF:LightCreate(object.path, pos, rot)
	end
end

--// SELF:GroupColors()
function SELF:GroupColors(GroupID)

	for LightID,_ in pairs(SELF.Circles[GroupID].Lights)
	do
		-- grab light
		local ent = Game.FindEntityByID(SELF.Circles[GroupID].Lights[LightID])
		local com = ent:FindComponentByName(SELF.Circles[GroupID].Codes)

		-- define color
		local clr = NewObject('Color')
		clr.Red = SELF.Circles[GroupID].Color.R
		clr.Green = SELF.Circles[GroupID].Color.G
		clr.Blue = SELF.Circles[GroupID].Color.B
		clr.Alpha = 255

		-- set light color
		com:SetColor(clr)
		com:SetIntensity(SELF.Circles[GroupID].Power * SELF.Circles[GroupID].Boost)
	end
end

--// SELF:GroupUpdate()
function SELF:GroupUpdate()

	for GroupID,_ in pairs(SELF.Circles)
	do
		if SELF.Circles[GroupID].Sticky
		then
			-- calculate & teleport
			local yaw = 0
			for LightID,_ in pairs(SELF.Circles[GroupID].Lights)
			do
				local alpha = math.rad((360 / SELF.Circles[GroupID].Number) * LightID) + math.rad(SELF.Circles[GroupID].Rotate)
				local x = math.sin(alpha) * (SELF.Circles[GroupID].Radius * SELF.Circles[GroupID].Expand)
				local y = math.cos(alpha) * (SELF.Circles[GroupID].Radius * SELF.Circles[GroupID].Expand)
				if LightID == 0 then yaw = math.deg(alpha) + 180 - (SELF.Circles[GroupID].Rotate * 2) else yaw = yaw - (360 / SELF.Circles[GroupID].Number) end

				local pos = Vector4.new(SELF.PuppetX + SELF.Circles[GroupID].PosX + x, SELF.PuppetY + SELF.Circles[GroupID].PosY + y, SELF.PuppetZ + SELF.Circles[GroupID].PosZ, 1)
				local rot = EulerAngles.new(SELF.Circles[GroupID].RotR, SELF.Circles[GroupID].RotP, SELF.Circles[GroupID].RotY + yaw)
				SELF:LightUpdate(SELF.Circles[GroupID].Lights[LightID], pos, rot)
			end
		end
	end
end

--// SELF:GroupDestroy(<INT>)
function SELF:GroupDestroy(GroupID)
	for _,LightID in pairs(SELF.Circles[GroupID].Lights) do
		SELF:LightDestroy(LightID)
	end
	SELF.Circles[GroupID] = nil
end

--// SELF:LightCreate(<STRING>,<VECTOR4>,<EULER>)
function SELF:LightCreate(path,pos,rot)

	-- get puppet
	local puppet = nil
	if CORE.isPhoto then
		puppet = CORE.isModel:GetWorldTransform()
	else
		puppet = Game.GetPlayer():GetWorldTransform()
	end
	puppet:SetPosition(pos)
	puppet:SetOrientation(GetSingleton('EulerAngles'):ToQuat(rot))
	return exEntitySpawner.Spawn(path, puppet, 2)
end

--// SELF:LightCreate(<ID>)
function SELF:LightDestroy(LightID)
	Game.FindEntityByID(LightID):GetEntity():Destroy()
end

--// SELF:LightUpdate(<ID>,<VECTOR4>,<EULER>)
function SELF:LightUpdate(LightID,pos,rot)
	Game.GetTeleportationFacility():Teleport(Game.FindEntityByID(LightID),pos,rot)
end



























--
--
--//////////////////// HELPER ////////////////////
--
--

--// SELF:ConvertGlyph(<STRING>)
function SELF:ConvertGlyph(data)
	if SELF.Glyphs then
		if data == "@" then return "\u{f1253}" end
		if data == "!" then return "\u{f02fd}" end
		if data == "?" then return "\u{f0a8c}" end
		if data == "1" then return "\u{f0ca0}" end
		if data == "2" then return "\u{f0ca2}" end
		if data == "3" then return "\u{f0ca4}" end
		if data == "4" then return "\u{f0ca6}" end
		if data == "5" then return "\u{f0ca8}" end
		if data == "6" then return "\u{f0caa}" end
		if data == "7" then return "\u{f0cac}" end
		if data == "8" then return "\u{f0cae}" end
		if data == "9" then return "\u{f0cb0}" end
		if data == "10" then return "\u{f0fec}" end
		if data == "LIGHTS" then return "\u{f0af9}\u{f0af6}\u{f0af4}\u{f0af5}\u{f0b01}\u{f0b00}" end
		if data == "OBJECTS" then return "\u{f0afc}\u{f0aef}\u{f0af7}\u{f0af2}\u{f0af0}\u{f0b01}\u{f0b00}" end
		if data == "CONTROLS" then return "\u{f0af0}\u{f0afc}\u{f0afb}\u{f0b01}\u{f0aff}\u{f0afc}\u{f0af9}\u{f0b00}" end
	end
	return data
end

--// SELF:TextWidth(<STRING>)
function SELF:TextWidth(text)
	local x, y = ImGui.CalcTextSize(tostring(text))
	return x
end

--// SELF:TextHeight(<STRING>)
function SELF:TextHeight(text)
	local x, y = ImGui.CalcTextSize(tostring(text))
	return y
end

--// SELF:IntToBool(<INT>)
function SELF:IntToBool(data)
	if data == 0 then return true end
	return false
end

--// SELF:BoolToInt(<BOOL>)
function SELF:BoolToInt(data)
	if data == true then return 0 end
	return 1
end

--// SELF:ShortenFloat(<MIXED>)
function SELF:ShortenFloat(data)
	return tonumber(string.format("%.3f", data))
end

--// SELF:FilterNumbers(<STRING>)
function SELF:FilterNumbers(data)
	local result = ""
	string.gsub(data,"%d+",function(found)
		result = result..found
	end)
	return result
end

--// SELF:TableLength(<TABLE>)
function SELF:TableLength(table)
	local count = 0
	for _ in pairs(table) do
		count = count + 1
	end
	return count
end

--// SELF:isPreGame()
function SELF:isPreGame()
	return GetSingleton("inkMenuScenario"):GetSystemRequestsHandler():IsPreGame()
end

--// SELF:ValueToProgress(<INT/FLOAT>, <INT/FLOAT>, <INT/FLOAT>)
-- Formula: NewValue = (((OldValue - OldMin) * (NewMax - NewMin)) / (OldMax - OldMin)) + NewMin
function SELF:ValueToProgress(min, max, value)
	return SELF:ShortenFloat((((value - min) * (1.0 - 0.0)) / (max - min)) + 0.0)
end







-- debugging only
function SELF:DebugDump(data)
	local tablecache = {}
	local buffer = ""
    local padder = "    "
 
    local function _dumpvar(d, depth)
        local t = type(d)
        local str = tostring(d)
        if (t == "table") then
            if (tablecache[str]) then
                -- table already dumped before, so we dont
                -- dump it again, just mention it
                buffer = buffer.."<"..str..">\n"
            else
                tablecache[str] = (tablecache[str] or 0) + 1
                buffer = buffer.."("..str..") {\n"
                for k, v in pairs(d) do
                    buffer = buffer..string.rep(padder, depth+1).."["..k.."] => "
                    _dumpvar(v, depth+1)
                end
                buffer = buffer..string.rep(padder, depth).."}\n"
            end
        elseif (t == "number") then
            buffer = buffer.."("..t..") "..str.."\n"
        else
            buffer = buffer.."("..t..") \""..str.."\"\n"
        end
    end
    _dumpvar(data, 0)
    return buffer
end




-- EOF
return CORE