import bpy
import rna_keymap_ui

from .copy_paste_bone_relative import WS_OT_copy_bone_relative_matrix, WS_OT_paste_bone_relative_matrix
from .copy_paste_object_relative import WS_OT_copy_object_relative_matrix, WS_OT_paste_object_relative_matrix
from .copy_paste_world_matrix import (WS_OT_copy_world_matrix, WS_OT_paste_world_matrix,
                                      WS_OT_paste_bone_world_matrix, WS_OT_paste_object_world_matrix)
from .clear_clipboards import WS_OT_copy_world_space_clear
from .ui import WS_PT_copy_paste_space, WS_MT_pie_menu

bl_info = {
    'name': 'Copy World Space',
    'description': 'Copy and paste "World Space" transforms',
    'author': 'Fin',
    'version': (0, 0, 2, 0),
    'blender': (2, 80, 0),
    'location': 'View3D > Properties  > Item',
    'warning': '',
    'wiki_url': '',
    'tracker_url': '',
    'category': 'Animation'
    }


class WS_AddonPreferences(bpy.types.AddonPreferences):
    bl_idname = __name__

    show_panel: bpy.props.BoolProperty(
        name="Show Panel",
        description="Add 'Copy World Space' panel to 'Item' tab",
        default=True)

    def draw(self, context):

        wm = context.window_manager

        layout = self.layout

        col = layout.column()
        col.prop(self, "show_panel", text="Enable Addon Panel")
        col.separator()

        col = layout.column()
        col.label(text=" Keymap/Pie:")
        AddonKeymaps.draw_keymap_items(wm, layout)


class clipboard_bones(bpy.types.PropertyGroup):
    arm_data: bpy.props.PointerProperty(name="Associated armature", description="", type=bpy.types.Object)
    bone_matrix: bpy.props.FloatVectorProperty(name="Matrix", size=16, subtype="MATRIX")
    # tail_matrix: bpy.props.FloatVectorProperty(name="Tail matrix", size=16, subtype="MATRIX")


class AddonKeymaps:
    _addon_keymaps = []
    _keymaps = {}

    @classmethod
    def new_keymap(cls, name, kmi_name, kmi_value=None, km_name='3D View',
                   space_type="VIEW_3D", region_type="WINDOW",
                   event_type=None, event_value=None,
                   ctrl=False, shift=False, alt=False, key_modifier="NONE"):

        cls._keymaps.update({name: [kmi_name, kmi_value, km_name, space_type,
                                    region_type, event_type, event_value,
                                    ctrl, shift, alt, key_modifier]
                             })

    @classmethod
    def add_hotkey(cls, kc, keymap_name):

        items = cls._keymaps.get(keymap_name)
        if not items:
            return

        kmi_name, kmi_value, km_name, space_type, region_type = items[:5]
        event_type, event_value, ctrl, shift, alt, key_modifier = items[5:]
        km = kc.keymaps.new(name=km_name, space_type=space_type,
                            region_type=region_type)

        kmi = km.keymap_items.new(kmi_name, event_type, event_value,
                                  ctrl=ctrl,
                                  shift=shift, alt=alt,
                                  key_modifier=key_modifier
                                  )
        if kmi_value:
            kmi.properties.name = kmi_value

        kmi.active = True

        cls._addon_keymaps.append((km, kmi))

    @staticmethod
    def register_keymaps():
        wm = bpy.context.window_manager
        kc = wm.keyconfigs.addon
        # In background mode, there's no such thing has keyconfigs.user,
        # because headless mode doesn't need key combos.
        # So, to avoid error message in background mode, we need to check if
        # keyconfigs is loaded.
        if not kc:
            return

        for keymap_name in AddonKeymaps._keymaps.keys():
            AddonKeymaps.add_hotkey(kc, keymap_name)

    @classmethod
    def unregister_keymaps(cls):
        kmi_values = [item[1] for item in cls._keymaps.values() if item]
        kmi_names = [item[0] for item in cls._keymaps.values() if
                     item not in ['wm.call_menu', 'wm.call_menu_pie']]

        for km, kmi in cls._addon_keymaps:
            # remove addon keymap for menu and pie menu
            if hasattr(kmi.properties, 'name'):
                if kmi_values:
                    if kmi.properties.name in kmi_values:
                        km.keymap_items.remove(kmi)

            # remove addon_keymap for operators
            else:
                if kmi_names:
                    if kmi.idname in kmi_names:
                        km.keymap_items.remove(kmi)

        cls._addon_keymaps.clear()

    @staticmethod
    def get_hotkey_entry_item(name, kc, km, kmi_name, kmi_value, col):

        # for menus and pie_menu
        if kmi_value:
            for km_item in km.keymap_items:
                if km_item.idname == kmi_name and km_item.properties.name == kmi_value:
                    col.context_pointer_set('keymap', km)
                    rna_keymap_ui.draw_kmi([], kc, km, km_item, col, 0)
                    return

            col.label(text=f"No hotkey entry found for {name}")
            col.operator(WS_OT_restore_hotkey.bl_idname,
                         text="Restore keymap",
                         icon='ADD').km_name = km.name

        # for operators
        else:
            if km.keymap_items.get(kmi_name):
                col.context_pointer_set('keymap', km)
                rna_keymap_ui.draw_kmi([], kc, km, km.keymap_items[kmi_name],
                                       col, 0)

            else:
                col.label(text=f"No hotkey entry found for {name}")
                col.operator(WS_OT_restore_hotkey.bl_idname,
                             text="Restore keymap",
                             icon='ADD').km_name = km.name

    @staticmethod
    def draw_keymap_items(wm, layout):
        kc = wm.keyconfigs.user

        for name, items in AddonKeymaps._keymaps.items():
            kmi_name, kmi_value, km_name = items[:3]
            box = layout.box()
            split = box.split()
            col = split.column()
            col.label(text=name)
            col.separator()
            km = kc.keymaps[km_name]
            AddonKeymaps.get_hotkey_entry_item(name, kc, km, kmi_name, kmi_value, col)


class WS_OT_restore_hotkey(bpy.types.Operator):
    bl_idname = "copy_world_space.restore_hotkey"
    bl_label = "Restore hotkeys"
    bl_options = {'REGISTER', 'INTERNAL'}

    km_name: bpy.props.StringProperty()

    def execute(self, context):
        context.preferences.active_section = 'KEYMAP'
        wm = context.window_manager
        kc = wm.keyconfigs.addon
        km = kc.keymaps.get(self.km_name)
        if km:
            km.restore_to_default()
            context.preferences.is_dirty = True
        context.preferences.active_section = 'ADDONS'
        return {'FINISHED'}


classes = (WS_AddonPreferences, WS_OT_copy_bone_relative_matrix, WS_OT_paste_bone_relative_matrix,
           WS_OT_copy_object_relative_matrix, WS_OT_paste_object_relative_matrix,
           WS_OT_copy_world_matrix, WS_OT_paste_world_matrix, WS_OT_paste_bone_world_matrix, WS_OT_copy_world_space_clear,
           WS_OT_paste_object_world_matrix, WS_PT_copy_paste_space, WS_MT_pie_menu, clipboard_bones, WS_OT_restore_hotkey,
           )


def register():
    from bpy.utils import register_class
    for cls in classes:
        register_class(cls)

    bpy.types.WindowManager.obj_ws_clipboard_data = bpy.props.BoolProperty(default=False)
    bpy.types.WindowManager.obj_rel_clipboard_data = bpy.props.BoolProperty(default=False)
    bpy.types.WindowManager.bone_ws_clipboard_data = bpy.props.BoolProperty(default=False)
    bpy.types.WindowManager.bone_rel_clipboard_data = bpy.props.BoolProperty(default=False)
    bpy.types.WindowManager.single_clipboard_data = bpy.props.BoolProperty(default=False)
    bpy.types.WindowManager.bone_data = bpy.props.CollectionProperty(type=clipboard_bones)

    AddonKeymaps.new_keymap('Copy World Space Pie Menu',
                            'wm.call_menu_pie', 'WS_MT_pie_menu',
                            '3D View', 'VIEW_3D', 'WINDOW',
                            'T', 'PRESS', True, False, False, 'NONE'
                            )

    AddonKeymaps.register_keymaps()


def unregister():

    AddonKeymaps.unregister_keymaps()

    from bpy.utils import unregister_class
    for cls in reversed(classes):
        unregister_class(cls)

    del bpy.types.WindowManager.obj_ws_clipboard_data
    del bpy.types.WindowManager.obj_rel_clipboard_data
    del bpy.types.WindowManager.bone_ws_clipboard_data
    del bpy.types.WindowManager.bone_rel_clipboard_data
    del bpy.types.WindowManager.single_clipboard_data
    del bpy.types.WindowManager.bone_data


if __name__ == "__main__":
    register()
