import bpy
from mathutils import Matrix, Vector
from .copy_paste_utils import quick_sort, translateScale


class WS_OT_copy_object_relative_matrix(bpy.types.Operator):
    bl_idname = "object_relative.copy"
    bl_label = "Copy Relative Transform"
    bl_options = {'REGISTER', 'UNDO'}
    bl_description = "Copy transforms relative to Active object"

    @classmethod
    def poll(cls, context):
        return context.object is not None

    def execute(self, context):
        relative_obj = bpy.context.active_object
        bpy.context.window_manager['relative_obj'] = relative_obj
        bpy.context.window_manager['source_relative_matrix'] = relative_obj.matrix_world.copy()
        bpy.context.window_manager['obj_rel_clipboard_data'] = {}

        for obj in bpy.context.selected_objects:
            if obj != relative_obj:
                bpy.context.window_manager['obj_rel_clipboard_data'][obj.name] = obj.matrix_world.copy()
                bpy.types.WindowManager.obj_rel_clipboard_data = True

        return {'FINISHED'}


class WS_OT_paste_object_relative_matrix(bpy.types.Operator):
    bl_idname = "object_relative.paste"
    bl_label = "Paste Relative Transforms"
    bl_options = {'REGISTER', 'UNDO'}
    bl_description = "Copy transforms relative to Active object"

    include_loc: bpy.props.BoolProperty(name="Location",
                                        description="Paste location",
                                        default=True)

    include_rot: bpy.props.BoolProperty(name="Rotation",
                                        description="Paste rotation",
                                        default=True)

    include_scale: bpy.props.BoolProperty(name="Scale",
                                          description="Paste scale",
                                          default=True)

    set_inverse: bpy.props.BoolProperty(name="Set inverse",
                                        description="'Set Inverse' for 'Child Of' constraints",
                                        default=False)

    @classmethod
    def poll(cls, context):
        if bpy.types.WindowManager.obj_rel_clipboard_data is True and context.object is not None:
            return True

    def draw(self, context):
        layout = self.layout
        layout.use_property_split = True
        layout.use_property_decorate = False
        row = layout.row()
        row.label(text='Include :')
        row = layout.row()
        row.prop(self, "include_loc")
        row = layout.row()
        row.prop(self, "include_rot")
        row = layout.row()
        row.prop(self, "include_scale")
        layout.row().separator()
        row = layout.row()
        row.prop(self, "set_inverse")

    def execute(self, context):

        selected = bpy.context.selected_objects
        qsort = quick_sort(selected)

        relative_obj = bpy.context.window_manager['relative_obj']

        source_matrix = Matrix(bpy.context.window_manager['source_relative_matrix'])
        source_loc, source_rot, source_scale = source_matrix.decompose()
        source_loc_mat = Matrix.Translation(source_loc)
        source_rot_mat = source_rot.to_matrix().to_4x4()
        source_scale_mat = translateScale(source_scale)

        current_matrix = relative_obj.matrix_world
        current_loc, current_rot, current_scale = current_matrix.decompose()
        current_loc_mat = Matrix.Translation(current_loc)
        current_rot_mat = current_rot.to_matrix().to_4x4()
        current_scale_mat = translateScale(current_scale)

        offset_matrix = source_matrix @ relative_obj.matrix_world.inverted()
        offset_loc, offset_rot, offset_scale = offset_matrix.inverted().decompose()
        offset_loc_mat = Matrix.Translation(offset_loc)
        offset_rot_mat = offset_rot.to_matrix().to_4x4()
        offset_scale_mat = translateScale(offset_scale)

        if qsort is not None:
            for obj in qsort:

                # only supporting child_of constraints with influence of 1 for now
                valid_childof_cons = [con for con in obj.constraints
                                      if con.type == "CHILD_OF" and con.target and con.influence == 1.0 and not con.mute]

                if obj.name in bpy.context.window_manager['obj_rel_clipboard_data']:

                    stored_matrix = Matrix(bpy.context.window_manager['obj_rel_clipboard_data'][obj.name])
                    stored_loc, stored_rot, stored_scale = stored_matrix.decompose()
                    stored_loc_mat = Matrix.Translation(stored_loc)
                    stored_rot_mat = stored_rot.to_matrix().to_4x4()
                    stored_scale_mat = translateScale(stored_scale)

                    relative_vec = Vector(stored_loc - source_loc)

                    relative_vec.rotate(offset_rot)

                    rot_final_mat = offset_rot_mat

                    scale_rel_mat = source_scale_mat.inverted() @ current_scale_mat

                    if self.include_scale:
                        scale_final_mat = stored_scale_mat @ scale_rel_mat

                    else:
                        scale_final_mat = stored_scale_mat

                    if self.include_loc:
                        loc_final_mat = Matrix.Translation(current_loc + relative_vec)

                    else:
                        loc_final_mat = stored_loc_mat

                    if self.include_scale:
                        scale_final_mat = stored_scale_mat @ scale_rel_mat

                    else:
                        scale_final_mat = stored_scale_mat

                    if self.include_rot:
                        matrix_final = loc_final_mat @ rot_final_mat @ stored_rot_mat @ scale_final_mat

                    else:
                        matrix_final = loc_final_mat @ stored_rot_mat @ scale_final_mat

                    bpy.context.view_layer.update()

                    # use the last valid childof constraint
                    if valid_childof_cons:
                        con = valid_childof_cons[-1]
                        con_name = con.name
                        target = con.target

                        if self.set_inverse:
                            context.view_layer.objects.active = obj
                            override = bpy.context.copy()
                            override["constraint"] = obj.constraints[con_name]
                            bpy.ops.constraint.childof_set_inverse(override, constraint=con_name, owner='OBJECT')
                            bpy.context.view_layer.update()

                        con_matrix = con.inverse_matrix.inverted()
                        target_matrix = target.matrix_world
                        offset_matrix = target_matrix @ con_matrix.inverted()
                        matrix_final = offset_matrix.inverted() @ matrix_final
                        obj.matrix_world = matrix_final
                        bpy.context.view_layer.update()

                    else:
                        obj.matrix_world = matrix_final

                bpy.context.view_layer.update()

                context.view_layer.objects.active = obj

                if bpy.context.scene.tool_settings.use_keyframe_insert_auto:
                    try:
                        bpy.ops.anim.keyframe_insert_menu(type='Available')
                    except TypeError:
                        self.report({'WARNING'}, f'{obj.name} has no active keyframes')
                        pass

        return {'FINISHED'}
