import bpy
from mathutils import Matrix, Vector
from collections import defaultdict
from .copy_paste_utils import quick_sort, translateScale
import numpy as np


class WS_OT_paste_world_matrix(bpy.types.Operator):
    bl_idname = "world_matrix.paste"
    bl_label = "Paste World Space Transforms"
    bl_options = {'REGISTER', 'UNDO'}
    bl_description = "Paste World Space Transforms to Active"

    include_loc: bpy.props.BoolProperty(name="Location",
                                        description="Paste location",
                                        default=True)

    include_rot: bpy.props.BoolProperty(name="Rotation",
                                        description="Paste rotation",
                                        default=True)

    include_scale: bpy.props.BoolProperty(name="Scale",
                                          description="Paste scale",
                                          default=True)

    set_inverse: bpy.props.BoolProperty(name="Set inverse",
                                        description="'Set Inverse' for 'Child Of' constraints",
                                        default=False)

    @classmethod
    def poll(cls, context):
        if bpy.types.WindowManager.single_clipboard_data is True and context.object is not None:
            return True

    def draw(self, context):
        layout = self.layout
        layout.use_property_split = True
        layout.use_property_decorate = False
        row = layout.row()
        row.label(text='Include :')
        row = layout.row()
        row.prop(self, "include_loc")
        row = layout.row()
        row.prop(self, "include_rot")
        row = layout.row()
        row.prop(self, "include_scale")
        layout.row().separator()
        row = layout.row()
        row.prop(self, "set_inverse")

    def execute(self, context):
        stored_matrix = Matrix(bpy.context.window_manager['ws_clipboard_data'])
        stored_loc, stored_rot, stored_scale = stored_matrix.decompose()
        stored_loc_mat = Matrix.Translation(stored_loc)
        stored_rot_mat = stored_rot.to_matrix().to_4x4()
        stored_scale_mat = translateScale(stored_scale)

        if context.active_pose_bone is not None:
            selected = bpy.context.selected_pose_bones
            qsort = quick_sort(selected)

            if qsort is not None:
                for pbone in qsort:
                    arm = pbone.id_data
                    valid_childof_cons = [con for con in pbone.constraints
                                          if con.type == "CHILD_OF" and con.target and con.influence == 1.0 and not con.mute]
                    current_matrix = arm.matrix_world @ pbone.matrix
                    current_loc, current_rot, current_scale = current_matrix.decompose()
                    current_loc_mat = Matrix.Translation(current_loc)
                    current_rot_mat = current_rot.to_matrix().to_4x4()
                    current_scale_mat = translateScale(current_scale)

                    if self.include_loc:
                        loc_final_mat = stored_loc_mat

                    else:
                        loc_final_mat = current_loc_mat

                    if self.include_scale:
                        scale_final_mat = stored_scale_mat

                    else:
                        scale_final_mat = current_scale_mat

                    if self.include_rot:
                        rot_final_mat = stored_rot_mat

                    else:
                        rot_final_mat = current_rot_mat

                    matrix_final = loc_final_mat @ rot_final_mat @ scale_final_mat

                    if valid_childof_cons:
                        con = valid_childof_cons[-1]
                        target_bone = con.target.pose.bones[con.subtarget]

                        if self.set_inverse:
                            con = valid_childof_cons[-1]
                            con_name = con.name
                            context.view_layer.objects.active = arm
                            bpy.context.active_object.data.bones.active = pbone.bone
                            override = bpy.context.copy()
                            override["constraint"] = pbone.constraints[con_name]
                            bpy.ops.constraint.childof_set_inverse(override, constraint=con_name, owner='BONE')

                        con_matrix = arm.matrix_world @ con.inverse_matrix.inverted()
                        target_matrix = arm.matrix_world @ target_bone.matrix

                        offset_matrix = target_matrix @ con_matrix.inverted()

                        matrix_final = offset_matrix.inverted() @ matrix_final

                        pbone.matrix = arm.convert_space(pose_bone=pbone,
                                                         matrix=matrix_final,
                                                         from_space='WORLD',
                                                         to_space='POSE')

                    else:
                        pbone.matrix = arm.convert_space(pose_bone=pbone,
                                                         matrix=matrix_final,
                                                         from_space='WORLD',
                                                         to_space='POSE')

                    bpy.context.view_layer.update()
                    arm.data.bones.active = pbone.bone

                    if bpy.context.scene.tool_settings.use_keyframe_insert_auto:
                        try:
                            bpy.ops.anim.keyframe_insert_menu(type='Available')
                        except RuntimeError:
                            self.report({'WARNING'}, f'{pbone.name} has no active keyframes')
                            pass

        else:
            selected = bpy.context.selected_objects
            qsort = quick_sort(selected)
            if qsort is not None:
                for obj in qsort:
                    # only supporting child_of constraints with influence of 1 for now
                    valid_childof_cons = [con for con in obj.constraints
                                          if con.type == "CHILD_OF" and con.target and con.influence == 1.0 and not con.mute]

                    current_matrix = obj.matrix_world
                    current_loc, current_rot, current_scale = current_matrix.decompose()
                    current_loc_mat = Matrix.Translation(current_loc)
                    current_rot_mat = current_rot.to_matrix().to_4x4()
                    current_scale_mat = translateScale(current_scale)

                    if self.include_loc:
                        loc_final_mat = stored_loc_mat

                    else:
                        loc_final_mat = current_loc_mat

                    if self.include_scale:
                        scale_final_mat = stored_scale_mat

                    else:
                        scale_final_mat = current_scale_mat

                    if self.include_rot:
                        rot_final_mat = stored_rot_mat

                    else:
                        rot_final_mat = current_rot_mat

                    matrix_final = loc_final_mat @ rot_final_mat @ scale_final_mat

                    if valid_childof_cons:
                        con = valid_childof_cons[-1]
                        con_name = con.name
                        target = con.target

                        if self.set_inverse:
                            context.view_layer.objects.active = obj
                            override = bpy.context.copy()
                            override["constraint"] = obj.constraints[con_name]
                            bpy.ops.constraint.childof_set_inverse(override, constraint=con_name, owner='OBJECT')
                            bpy.context.view_layer.update()

                        con_matrix = con.inverse_matrix.inverted()
                        target_matrix = target.matrix_world
                        offset_matrix = target_matrix @ con_matrix.inverted()
                        matrix_final = offset_matrix.inverted() @ matrix_final
                        obj.matrix_world = matrix_final
                        bpy.context.view_layer.update()

                    else:
                        obj.matrix_world = matrix_final

                    bpy.context.view_layer.update()

                    context.view_layer.objects.active = obj

                    if bpy.context.scene.tool_settings.use_keyframe_insert_auto:
                        try:
                            bpy.ops.anim.keyframe_insert_menu(type='Available')
                        except TypeError:
                            self.report({'WARNING'}, f'{obj.name} has no active keyframes')
                            pass

        return {'FINISHED'}


class WS_OT_copy_world_matrix(bpy.types.Operator):
    bl_idname = "world_matrix.copy"
    bl_label = "Copy World Space Transform"
    bl_options = {'REGISTER', 'UNDO'}
    bl_description = "Copy World Space Transforms"

    @classmethod
    def poll(cls, context):
        return context.object is not None

    def execute(self, context):

        bpy.context.window_manager['ws_clipboard_data'] = {}

        if context.active_pose_bone is not None:
            if len(bpy.context.selected_pose_bones) == 1:
                bpy.types.WindowManager.obj_ws_clipboard_data = False
                bpy.types.WindowManager.bone_ws_clipboard_data = False
                bpy.types.WindowManager.single_clipboard_data = True
                pbone = context.active_pose_bone
                arm = pbone.id_data
                matrix_final = arm.matrix_world @ pbone.matrix
                bpy.context.window_manager['ws_clipboard_data'] = matrix_final.copy()
            else:
                bpy.types.WindowManager.obj_ws_clipboard_data = False
                bpy.types.WindowManager.bone_ws_clipboard_data = True
                bpy.types.WindowManager.single_clipboard_data = False

                # clear clipboard property group
                bpy.context.window_manager.bone_data.clear()

                for pbone in bpy.context.selected_pose_bones:
                    arm = pbone.id_data
                    matrix_final = arm.matrix_world @ pbone.matrix
                    cbone = bpy.context.window_manager.bone_data.add()
                    cbone.arm_data = pbone.id_data
                    cbone.name = pbone.name
                    flat_mat = np.matrix(matrix_final.copy())
                    # flatten matrix for FloatVector storage
                    cbone.bone_matrix = np.ravel(flat_mat, order='F')

        else:
            if len(bpy.context.selected_objects) == 1:
                obj = context.active_object
                bpy.context.window_manager['ws_clipboard_data'] = obj.matrix_world.copy()
                bpy.types.WindowManager.obj_ws_clipboard_data = False
                bpy.types.WindowManager.bone_ws_clipboard_data = False
                bpy.types.WindowManager.single_clipboard_data = True

            else:
                bpy.types.WindowManager.obj_ws_clipboard_data = True
                bpy.types.WindowManager.bone_ws_clipboard_data = False
                bpy.types.WindowManager.single_clipboard_data = False
                for obj in bpy.context.selected_objects:
                    bpy.context.window_manager['ws_clipboard_data'][obj.name] = obj.matrix_world.copy()

        return {'FINISHED'}


class WS_OT_paste_bone_world_matrix(bpy.types.Operator):
    bl_idname = "bone_world_matrix.paste"
    bl_label = "Paste World Space Transforms"
    bl_options = {'REGISTER', 'UNDO'}
    bl_description = "Paste World Space Transforms"

    include_loc: bpy.props.BoolProperty(name="Location",
                                        description="Paste location",
                                        default=True)

    include_rot: bpy.props.BoolProperty(name="Rotation",
                                        description="Paste rotation",
                                        default=True)

    include_scale: bpy.props.BoolProperty(name="Scale",
                                          description="Paste scale",
                                          default=True)

    set_inverse: bpy.props.BoolProperty(name="Set inverse",
                                        description="'Set Inverse' for 'Child Of' constraints",
                                        default=False)

    @classmethod
    def poll(cls, context):
        if bpy.types.WindowManager.bone_ws_clipboard_data is True and context.active_pose_bone is not None:
            return True

    def draw(self, context):
        layout = self.layout
        layout.use_property_split = True
        layout.use_property_decorate = False
        row = layout.row()
        row.label(text='Include :')
        row = layout.row()
        row.prop(self, "include_loc")
        row = layout.row()
        row.prop(self, "include_rot")
        row = layout.row()
        row.prop(self, "include_scale")
        layout.row().separator()
        row = layout.row()
        row.prop(self, "set_inverse")

    def execute(self, context):
        bone_data = bpy.context.window_manager.bone_data
        selected = bpy.context.selected_pose_bones
        qsort = quick_sort(selected)

        if qsort is not None:
            for pbone in qsort:
                arm = pbone.id_data

                # only supporting child_of constraints with influence of 1 for now
                valid_childof_cons = [con for con in pbone.constraints
                                      if con.type == "CHILD_OF" and con.target and con.influence == 1.0 and not con.mute]

                for i, item in enumerate(bone_data.items()):
                    # check for matching name and armature
                    if bone_data[i].name == pbone.name and bone_data[i].arm_data == arm:
                        stored_matrix = Matrix(bone_data[i].bone_matrix)
                        stored_loc, stored_rot, stored_scale = stored_matrix.decompose()
                        stored_loc_mat = Matrix.Translation(stored_loc)
                        stored_rot_mat = stored_rot.to_matrix().to_4x4()
                        stored_scale_mat = translateScale(stored_scale)

                        current_matrix = arm.matrix_world @ pbone.matrix
                        current_loc, current_rot, current_scale = current_matrix.decompose()
                        current_loc_mat = Matrix.Translation(current_loc)
                        current_rot_mat = current_rot.to_matrix().to_4x4()
                        current_scale_mat = translateScale(current_scale)

                        if self.include_loc:
                            loc_final_mat = stored_loc_mat

                        else:
                            loc_final_mat = current_loc_mat

                        if self.include_scale:
                            scale_final_mat = stored_scale_mat

                        else:
                            scale_final_mat = current_scale_mat

                        if self.include_rot:
                            rot_final_mat = stored_rot_mat

                        else:
                            rot_final_mat = current_rot_mat

                        matrix_final = loc_final_mat @ rot_final_mat @ scale_final_mat

                        if valid_childof_cons:
                            con = valid_childof_cons[-1]
                            target_bone = con.target.pose.bones[con.subtarget]

                            if self.set_inverse:
                                con = valid_childof_cons[-1]
                                con_name = con.name
                                context.view_layer.objects.active = arm
                                bpy.context.active_object.data.bones.active = pbone.bone
                                override = bpy.context.copy()
                                override["constraint"] = pbone.constraints[con_name]
                                bpy.ops.constraint.childof_set_inverse(override, constraint=con_name, owner='BONE')

                            con_matrix = arm.matrix_world @ con.inverse_matrix.inverted()
                            target_matrix = arm.matrix_world @ target_bone.matrix

                            offset_matrix = target_matrix @ con_matrix.inverted()

                            matrix_final = offset_matrix.inverted() @ matrix_final

                            pbone.matrix = arm.convert_space(pose_bone=pbone,
                                                             matrix=matrix_final,
                                                             from_space='WORLD',
                                                             to_space='POSE')
                        else:
                            pbone.matrix = arm.convert_space(pose_bone=pbone,
                                                             matrix=matrix_final,
                                                             from_space='WORLD',
                                                             to_space='POSE')

                        bpy.context.view_layer.update()

                        arm.data.bones.active = pbone.bone

                        if bpy.context.scene.tool_settings.use_keyframe_insert_auto:
                            try:
                                bpy.ops.anim.keyframe_insert_menu(type='Available')
                            except RuntimeError:
                                self.report({'WARNING'}, f'{pbone.name} has no active keyframes')
                                pass

        return {'FINISHED'}


class WS_OT_paste_object_world_matrix(bpy.types.Operator):
    bl_idname = "object_world_matrix.paste"
    bl_label = "Paste World Space Transforms"
    bl_options = {'REGISTER', 'UNDO'}
    bl_description = "Paste World Space Transforms"

    include_loc: bpy.props.BoolProperty(name="Location",
                                        description="Paste location",
                                        default=True)

    include_rot: bpy.props.BoolProperty(name="Rotation",
                                        description="Paste rotation",
                                        default=True)

    include_scale: bpy.props.BoolProperty(name="Scale",
                                          description="Paste scale",
                                          default=True)

    set_inverse: bpy.props.BoolProperty(name="Set inverse",
                                        description="'Set Inverse' for 'Child Of' constraints",
                                        default=True)

    @classmethod
    def poll(cls, context):
        if bpy.types.WindowManager.obj_ws_clipboard_data is True and context.active_object.mode == 'OBJECT':
            return True

    def draw(self, context):
        layout = self.layout
        layout.use_property_split = True
        layout.use_property_decorate = False
        row = layout.row()
        row.label(text='Include :')
        row = layout.row()
        row.prop(self, "include_loc")
        row = layout.row()
        row.prop(self, "include_rot")
        row = layout.row()
        row.prop(self, "include_scale")
        layout.row().separator()
        row = layout.row()
        row.prop(self, "set_inverse")

    def execute(self, context):

        selected = bpy.context.selected_objects
        qsort = quick_sort(selected)

        if qsort is not None:
            for obj in qsort:
                # only supporting child_of constraints with influence of 1 for now
                valid_childof_cons = [con for con in obj.constraints
                                      if con.type == "CHILD_OF" and con.target and con.influence == 1.0 and not con.mute]

                if obj.name in bpy.context.window_manager['ws_clipboard_data']:
                    stored_matrix = Matrix(bpy.context.window_manager['ws_clipboard_data'][obj.name])
                    stored_loc, stored_rot, stored_scale = stored_matrix.decompose()
                    stored_loc_mat = Matrix.Translation(stored_loc)
                    stored_rot_mat = stored_rot.to_matrix().to_4x4()
                    stored_scale_mat = translateScale(stored_scale)

                    current_matrix = obj.matrix_world
                    current_loc, current_rot, current_scale = current_matrix.decompose()
                    current_loc_mat = Matrix.Translation(current_loc)
                    current_rot_mat = current_rot.to_matrix().to_4x4()
                    current_scale_mat = translateScale(current_scale)

                    if self.include_loc:
                        loc_final_mat = stored_loc_mat

                    else:
                        loc_final_mat = current_loc_mat

                    if self.include_scale:
                        scale_final_mat = stored_scale_mat

                    else:
                        scale_final_mat = current_scale_mat

                    if self.include_rot:
                        rot_final_mat = stored_rot_mat

                    else:
                        rot_final_mat = current_rot_mat

                    matrix_final = loc_final_mat @ rot_final_mat @ scale_final_mat

                    if valid_childof_cons:
                        con = valid_childof_cons[-1]
                        con_name = con.name
                        target = con.target

                        if self.set_inverse:
                            context.view_layer.objects.active = obj
                            override = bpy.context.copy()
                            override["constraint"] = obj.constraints[con_name]
                            bpy.ops.constraint.childof_set_inverse(override, constraint=con_name, owner='OBJECT')
                            bpy.context.view_layer.update()

                        con_matrix = con.inverse_matrix.inverted()
                        target_matrix = target.matrix_world
                        offset_matrix = target_matrix @ con_matrix.inverted()
                        matrix_final = offset_matrix.inverted() @ matrix_final
                        obj.matrix_world = matrix_final
                        bpy.context.view_layer.update()

                    else:
                        obj.matrix_world = matrix_final

                    bpy.context.view_layer.update()

                    context.view_layer.objects.active = obj

                    if bpy.context.scene.tool_settings.use_keyframe_insert_auto:
                        try:
                            bpy.ops.anim.keyframe_insert_menu(type='Available')
                        except TypeError:
                            self.report({'WARNING'}, f'{obj.name} has no active keyframes')
                            pass

        return {'FINISHED'}
