--https://www.lua.org/pil/contents.html
--https://wiki.cybermods.net/cyber-engine-tweaks/console/functions
--https://github.com/yamashi/CyberEngineTweaks/blob/eb0f7daf2971ed480abf04355458cbe326a0e922/src/sol_imgui/README.md

require "ghost_forward/lib/gameobj_accessor"
require "ghost_forward/lib/ghosting"
require "ghost_forward/lib/input_processing"
require "ghost_forward/lib/mathutils"
require "ghost_forward/lib/peeking"
require "ghost_forward/lib/util"

--------------------------------------------------------------------

local jumpDistance = 1.8

local peekTime = 1.5      -- seconds
local peekDistance = jumpDistance * 2.5

--------------------------------------------------------------------

local isShutdown = true

local o     -- This is a class that wraps access to Game.xxx

local keys = { ghostForward=false, peekForward=false }

local timer = 0

local isPeeking = false
local peekingStartTime = 0

--------------------------------------------------------------------

registerForEvent("onInit", function()
    isShutdown = false

    local wrappers = {}
    function wrappers.GetPlayer() return Game.GetPlayer() end
    function wrappers.Player_GetPos(player) return player:GetWorldPosition() end
    function wrappers.Player_GetYaw(player) return player:GetWorldYaw() end
    function wrappers.GetWorkspotSystem() return Game.GetWorkspotSystem() end
    function wrappers.Workspot_InWorkspot(workspot, player) return workspot:IsActorInWorkspot(player) end
    function wrappers.GetCameraSystem() return Game.GetCameraSystem() end
    function wrappers.Camera_GetForward(camera) return camera:GetActiveCameraForward() end
    function wrappers.GetTeleportationFacility() return Game.GetTeleportationFacility() end
    function wrappers.Teleport(teleport, player, pos, yaw) return teleport:Teleport(player, pos, EulerAngles.new(0, 0, yaw)) end

    function wrappers.GetFPPCamera(player) return player:GetFPPCameraComponent() end
    function wrappers.SetLocalCamPosition(playerCam, pos) playerCam:SetLocalPosition(pos) end

    o = GameObjectAccessor:new(wrappers)
end)

registerForEvent("onShutdown", function()
    isShutdown = true
end)

registerForEvent("onUpdate", function(deltaTime)
    if isShutdown then
        do return end
    end

    timer = timer + deltaTime

    if (not keys.ghostForward) and (not keys.peekForward) and (not isPeeking) then
        do return end
    end

    o:GetPlayerInfo(timer)      -- very important to use : and not . (colon is a syntax shortcut that passes self as a hidden first param)
    if not o.player then
        do return end
    end

    if not IsStandingStill(o.vel) then
        o:GetInWorkspot(timer)       -- this crashes soon after loading a save.  So don't call if velocity is near zero.  Still got a crash when reloading after dying in a car shootout.  Hopefully this looser method keeps from crashing
        if o.isInWorkspot then
            do return end
        end
    end

    -- Ghost Forward
    if keys.ghostForward then
        GhostForward(jumpDistance, true, o, timer)
    end

    -- Peek Forward
    if isPeeking then
        if timer - peekingStartTime < peekTime then
            local offset = peekDistance * GetPeekDistPercent(timer - peekingStartTime, peekTime)

            o:SetLocalCamPosition(Vector4.new(0, offset, 0, 1), timer)
        else
            isPeeking = false

            o:SetLocalCamPosition(Vector4.new(0, 0, 0, 1), timer)
        end
    elseif keys.peekForward then
        isPeeking = true
        peekingStartTime = timer
    end

    ResetKeys(keys)
end)

registerHotkey("shouldGhostForward", "Ghost Forward", function()
    keys.ghostForward = true
end)

registerHotkey("shouldPeekForward", "Peek Forward", function()
    keys.peekForward = true
end)
