/* global _ */

$(function() {
    'use strict';

    /**
     * init-focus
     * head-tail-cycle
     * row-cycle='inrow/outrow'
     * col-cycle='false'
     */
    var $currFocus = null;

    // 移动方向上的偏移量
    var OFFSET_MAP = {
        'left': -1,
        'right': 1,
        'up': -1,
        'down': 1
    };

    var parent = function($el, selector) {
        var $parent = $el.parentsUntil(selector).parent();
        if ($parent.length) {
            return $parent;
        }
        return $el.parent(selector);
    };

    var trigger = function($el, name) {
        var $child = $el.children().first();
        if ($child.length) {
            $child.trigger(name);
        } else {
            $el.trigger(name);
        }
    };

    var switchFocus = function($nextFocus, key) {
        if ($currFocus.is($nextFocus)) {
            return;
        }

        if ($nextFocus && $nextFocus.length) {
            $nextFocus.addClass('focused');
            $currFocus.removeClass('focused');

            $currFocus.trigger('xblur');
            $nextFocus.trigger('xfocus');
        } else {
            trigger($currFocus, 'stop' + key);
        }

        $currFocus = null;
    };

    var oldFocusFunc = $.fn.focus;
    $.fn.focus = function() {
        if (!arguments.length) {
            var $container = parent($(this), '[focus-table]');

            if (!$container.length) {
                $container = parent($(this), '[focus-panel]');
            }

            if (!$container.length) {
                $container = $('body');
            }

            $currFocus = $container.find('.focused');
            switchFocus($(this));
        }

        return oldFocusFunc.apply(this, arguments);
    };


    /**
     * @param {Object} options
     * @param {jQuery.element} options.$el
     */
    var Table = function(options) {
        this.isFocused = false;
        _.extend(this, options);

        if (!this.$el.is(['trChecked'])) {
            _.each(this.$el.find('.tr'), function(tr, index) {
                _.each($(tr).find('.td'), function(td) {
                    if (!$(td).is('[tr]')) {
                        $(td).attr('tr', '' + index);
                    }
                });
            });
            this.$el.attr('trChecked', 'true');
        }

    };
    Table.prototype = {
        sendKey: function(key) {
            if (!this.isFocused) {
                return this.$el.find('.td:first');
            }

            return this[key](key);
        },

        leftOrRight: function(key) {
            var offset = OFFSET_MAP[key];
            var rowIndexes = this.getRowIndexes($currFocus);


            var flows = ['findNextInRow', 'findNextInRowCycle', 'findNextOutRowCycle', 'findNextHeadTailCycle'];

            var nextFocus;
            _.find(flows, function(name) {
                nextFocus = this[name](rowIndexes, offset);
                return nextFocus;
            }, this);

            if (nextFocus) {
                return $(nextFocus) || $();
            }
        },

        getRowIndexes: function($td) {
            return _.map($td.attr('tr').split(/\s+/), function(str) {
                return +str;
            });
        },


        findNextInRow: function(rowIndexes, offset) {
            var self = this;
            var nextFocus;

            var finder = function(rowIndex) {
                var $tds = self.getTdsByRowIndex(rowIndex);
                var focusIndex = _.indexOf($tds, $currFocus[0]);
                return $tds[focusIndex + offset];
            };

            _.find(rowIndexes, function(rowIndex) {
                nextFocus = finder(rowIndex);
                return nextFocus;
            }, this);

            return nextFocus;
        },

        findNextInRowCycle: function(rowIndexes, offset) {
            var isSupportInRowCycle = this.$el.is('[row-cycle=inrow]');

            // 是否支持行内循环
            if (isSupportInRowCycle) {
                var $tds = this.getTdsByRowIndex(_.first(rowIndexes));
                return this.getAgainstIndexByOffset($tds, offset);
            }
        },

        findNextOutRowCycle: function(rowIndexes, offset) {
            var isSupportOutRowCycle = this.$el.is('[row-cycle=outrow]');

            // 是否支持行间循环
            if (isSupportOutRowCycle) {
                var rowIndex = offset > 0 ? _.last(rowIndexes) : _.first(rowIndexes);
                var $tds = this.getTdsByRowIndex(rowIndex + offset);
                return this.getAgainstIndexByOffset($tds, offset);
            }
        },

        findNextHeadTailCycle: function(rowIndexes, offset) {
            var isSupportHeadTailCycle = this.$el.is('[head-tail-cycle]');

            // 是否支持头尾循环
            if (isSupportHeadTailCycle) {
                var $tds = this.$el.find('.td');
                return this.getAgainstIndexByOffset($tds, offset);
            }
        },


        /**
         * 同一行中的元素
         */
        getTdsByRowIndex: function(rowIndex) {
            return this.$el.find('[tr~="' + rowIndex + '"].td');
        },

        /**
         * 根据Offset获取另一端的值
         */
        getAgainstIndexByOffset: function(list, offset) {
            return list[(list.length - 1) * Math.max(-offset, 0)];
        },

        /**
         * 根据offset获取两端值
         */
        getEndIndexByOffset: function(list, offset) {
            if (offset > 0) {
                return _.last(list);
            } else {
                return _.first(list);
            }
        },

        'upOrDown': function(key) {
            var offset = OFFSET_MAP[key];
            var rowIndexes = this.getRowIndexes($currFocus);
            var currentRowIndex = this.getEndIndexByOffset(rowIndexes, offset);
            var nextRownIndex = currentRowIndex + offset;

            var $tds = this.getTdsByRowIndex(currentRowIndex);
            var tdIndex = _.indexOf($tds, $currFocus[0]);

            var $nextRowTds = this.getTdsByRowIndex(nextRownIndex);

            // 是否支持col-cycle
            if (!$nextRowTds.length) {
                var isSupportColCycle = this.$el.is('[col-cycle]');
                if (isSupportColCycle) {

                    // 找出两端的行号
                    var allRowIndexes = [];
                    allRowIndexes = _.union(allRowIndexes, this.getRowIndexes(this.$el.find('.td:last')));
                    allRowIndexes = _.union(allRowIndexes, this.getRowIndexes(this.$el.find('.td:first')));
                    allRowIndexes.sort();

                    nextRownIndex = this.getAgainstIndexByOffset(allRowIndexes, offset);
                    $nextRowTds = this.getTdsByRowIndex(nextRownIndex);
                }
            }

            // 计算下一行的搜索起始点
            var nextTdIndex = tdIndex;
            if (!$nextRowTds[nextTdIndex]) {
                nextTdIndex = $nextRowTds.length - 1;
            }

            // 找出所有相交的元素
            var isFind = false;
            var direct = NaN;

            while ($nextRowTds[nextTdIndex]) {

                // 比较是否相交，或在偏左、偏右
                var result = this.isOverlaped($currFocus, $($nextRowTds[nextTdIndex]));

                // 搜索只能是一个方向
                if (_.isNaN(direct)) {
                    direct = result;
                }

                isFind = (0 === result);
                if (isFind) {
                    break;
                }

                nextTdIndex = nextTdIndex + direct;
            }

            if (isFind) {
                // 用已搜索到的TD，加上同方向的另一个TD，与当前焦点比较，看哪个重合的多
                var resultTd = this.tryToFindMoreSuitableTd($currFocus, [$nextRowTds[nextTdIndex], $nextRowTds[nextTdIndex + 1], $nextRowTds[nextTdIndex - 1]]);
                return $(resultTd);
            }
        },

        'isOverlaped': function($a, $b) {
            var aBegin = $a.offset().left;
            var aEnd = aBegin + $a.width();

            var bBegin = $b.offset().left;
            var bEnd = bBegin + $b.width();

            if (bEnd < aBegin) {
                return 1;
            }

            if (aEnd < bBegin) {
                return -1;
            }

            return 0;
        },

        getOverlapedLength: function($a, $b) {
            if (0 === this.isOverlaped($a, $b)) {
                var aBegin = $a.offset().left;
                var aEnd = aBegin + $a.width();

                var bBegin = $b.offset().left;
                var bEnd = bBegin + $b.width();

                var aLength = (aEnd - aBegin);
                var bLength = (bEnd - bBegin);
                var totalDistance = Math.max(aEnd, bEnd) - Math.min(aBegin, bBegin);
                var overlapedLength = (aLength + bLength - totalDistance) / 2;

                return overlapedLength > 0 ? overlapedLength : 0;
            }

            return 0;
        },

        tryToFindMoreSuitableTd: function($currFocus, candidateTds) {
            candidateTds = _.compact(candidateTds);

            var distances = [];
            var map = _.groupBy(candidateTds, function(td) {
                var distance = this.getOverlapedLength($currFocus, $(td));
                distances.push(distance);
                return distance;
            }, this);

            var maxOverlapedLength = _.max(distances);

            return _.first(map[maxOverlapedLength]);
        },

        'left': function(key) {
            return this.leftOrRight(key);
        },

        'right': function(key) {
            return this.leftOrRight(key);
        },

        'up': function(key) {
            return this.upOrDown(key);
        },
        'down': function(key) {
            return this.upOrDown(key);
        }
    };

    /**
     * @param {Object} options
     * @param {jQuery.element} options.$el
     */
    var Panel = function(options) {
        _.extend(this, options);
        var $tables = this.$el.find('.table');
        if (!$tables.length) {
            $tables = this.$el;
        }

        this.tables = _.map($tables, function(tableEl) {
            return new Table({
                '$el': $(tableEl)
            });
        });
    };

    Panel.prototype = {
        sendKey: function(key) {

            var currTable = _.find(this.tables, function(table) {
                return table.$el.has('.focused').length;
            });

            if (currTable) {
                currTable.isFocused = true;
            } else {
                currTable = _.first(this.tables);
            }

            var $nextFocus = currTable.sendKey(key);

            if (!$nextFocus) {

                // 相邻的Table
                var adjacentTable = _.find(this.tables, function(table) {
                    return table.$el.is(currTable.$el.attr('key'));
                });

                if (adjacentTable) {
                    $nextFocus = adjacentTable.sendKey(key);
                }
            }

            return $nextFocus;
        }
    };

    /**
     *
     * @param {String} key 按键值
     * - ok
     * - left
     * - up
     * - right
     * - down
     *
     */
    $.fn.sendKey = function(key, suppressEvent) {
        var $this = $(this);
        $currFocus = $this.find('.focused');
        if (OFFSET_MAP[key]) {
            var panel = new Panel({
                $el: $this
            });

            var $nextFocus = $();

            // 尝试定位直接指定的下一元素
            var directSelector = $currFocus.attr(key);
            if (directSelector) {
                $nextFocus = $this.find(directSelector);
            }

            if (!$nextFocus.length) {
                $nextFocus = panel.sendKey(key);
            }

            switchFocus($nextFocus, key, suppressEvent);

        } else {
            if (key === 'ok') {
                trigger($currFocus, 'click');
            } else {
                trigger($currFocus, '' + key);
            }
        }
    };

});
