bl_info = {
    "name": "AI Denoise",
    "author": "Mathieu Menuet",
    "version": (1, 2, 0),
    "blender": (2, 80, 0),
    "location": "Render Settings > Performance",
    "description": ("Create node tree for denoising"),
    "wiki_url": "https://www.gumroad.com/l/E-Cycles",
    "category": "Render"}

import bpy
from bpy.types import (
        Operator,
        PropertyGroup,
        )
from bpy.props import (
        IntProperty,
        FloatProperty,
        BoolProperty,
        PointerProperty,
        )

# bpy.context.scene["quality"] = 1
class AISettings(bpy.types.PropertyGroup):
    @classmethod
    def register(cls):
        bpy.types.Scene.ai_settings = PointerProperty(
            name="AI Settings",
            description="AI Denoise settings",
            type=cls,
        )
        # cls.quality= IntProperty(
            # name="Denoising Quality",
            # description="1 is fast, 2 medium, 3 is high quality (requires lot of memory)",
            # min=1, max=3,
            # default=1,
        # )
        # cls.sss= BoolProperty(
            # name="SSS",
            # description="you can uncheck if you have no SSS in your frame",
            # default=True,
        # )
        cls.trans= BoolProperty(
            name="Transmission",
            description="you can uncheck if you have no transparency in your frame",
            default=True,
        )
        cls.vol= BoolProperty(
            name="Volumetrics",
            description="you can uncheck if you have no volumes in your frame",
            default=True,
        )
        # cls.memory= BoolProperty(
            # name="Low Mem",
            # description="uses less memory, but is also slower and requires 2 steps",
            # default=False,
        # )
        cls.strength= FloatProperty(
            name="Strength",
            description="Denoising strength",
            min=0.0, max=1.0,
            default=1.0,
        )
        cls.slow_pass= BoolProperty(
            name="Slow Pass",
            description="Use denoise data which is slower but in some case also offer better quality",
            default=False,
        )
        cls.multi_pass= BoolProperty(
            name="Multi Pass",
            description="High quality denoising, requires rendering more passes, which may be slightly slower",
            default=True,
        )
        cls.multi_denoiser= BoolProperty(
            name="Keep old denoiser",
            description="Allows to use both denoiser (slow, may require manual masking)",
            default=False,
        )
        
    
bpy.utils.register_class(AISettings)
# bpy.types.Scene.ai_settings = bpy.props.PointerProperty(type=AISettings)

class AIDenoise(bpy.types.Operator):
    """Tooltip"""
    bl_idname = "render.setnodes"
    bl_label = "Create Node Tree for AI Denoise"
    ai_settings: bpy.props.PointerProperty(type=AISettings)

    @classmethod
    def poll(cls, context):
        return context.scene.render.engine == 'CYCLES'

    def execute(self, context):
        # level = context.scene.ai_settings.quality

        # set to False if not needed, make it auto by going through all materials? But what if material not visible = wasted memory and perf?
        link_comp = False
        color = True
        glossy = True
        trans = context.scene.ai_settings.trans
        vol = context.scene.ai_settings.vol
        # sss = context.scene.ai_settings.sss
        multi_pass = context.scene.ai_settings.multi_pass
        slow_pass = context.scene.ai_settings.slow_pass
        multi_denoiser = context.scene.ai_settings.multi_denoiser
        # mem = context.scene.ai_settings.memory

        scene = bpy.context.scene
        scene.use_nodes = True
        scene.render.use_compositing = True
        scene.cycles.pixel_filter_type = 'GAUSSIAN'
        if scene.cycles.filter_width > 2 or scene.cycles.filter_width < 1:
            scene.cycles.filter_width = 1.5
        
        if bpy.app.version[1] < 80:
            name = bpy.context.scene.render.layers.active.name
            if not multi_denoiser:
                for layer in scene.render.layers:
                    layer.cycles.use_denoising = False
        else:
            name = bpy.context.window.view_layer.name
            if not multi_denoiser:
                for layer in scene.view_layers:
                    layer.cycles.use_denoising = False
        for node in bpy.context.scene.node_tree.nodes:
            if node.type == 'R_LAYERS' and node.layer == name:
                rend_layers = node
            if node.type == 'COMPOSITE':
                comp = node
                # print('comp')
            if node.type == 'OUTPUT_FILE':
                file_out = node
                # print('out')
        try:
            if bpy.context.scene.node_tree.nodes.active.type == 'R_LAYERS':
                rend_layers = bpy.context.scene.node_tree.nodes.active
        except:
            r = 1
        if "rend_layers" not in locals():
            rend_layers = scene.node_tree.nodes.new(type="CompositorNodeRLayers")
            bpy.context.scene.node_tree.nodes.active = rend_layers
            rend_layers.layer = name
            link_comp = True
        if "comp" not in locals() and "file_out" not in locals():
            comp = scene.node_tree.nodes.new(type="CompositorNodeComposite")
                    
        # if "comp" in locals():
            # comp.location.x += 100
        # if "file_out" in locals():
            # file_out.location.x += 100
        if bpy.app.version[1] < 80:
            layers = bpy.context.scene.render.layers.active
        else:
            if bpy.context.scene.node_tree.nodes.active == rend_layers:
                layers = bpy.context.scene.view_layers[rend_layers.layer]
            else:
                layers = bpy.context.window.view_layer
        x_min = 5000

        if not slow_pass and not multi_pass:
            layers.use_pass_diffuse_color = True
            layers.use_pass_normal = True
            denoise = scene.node_tree.nodes.new(type="CompositorNodeDenoise")
            strength = scene.node_tree.nodes.new(type="CompositorNodeMixRGB")
            denoise.name = "denoising_"
            strength.name = "denoising_strength"
            if link_comp:
                scene.node_tree.links.new(strength.outputs[0], comp.inputs[0])
                comp.location.x = strength.location.x + 250
            else:
                for node in scene.node_tree.nodes:
                    for input in node.inputs:
                        if input.is_linked and input.links[0].from_socket == rend_layers.outputs[0]:
                            scene.node_tree.links.new(strength.outputs[0], input)
                            x_min = min(node.location.x,x_min)
                rend_layers.location.x = x_min - 700
            denoise.location = (rend_layers.location.x + 300, rend_layers.location.y)
            strength.location = (rend_layers.location.x + 500, rend_layers.location.y)
            scene.node_tree.links.new(denoise.outputs[0], strength.inputs[2])
            scene.node_tree.links.new(rend_layers.outputs[0], strength.inputs[1])
            scene.node_tree.links.new(rend_layers.outputs[0], denoise.inputs[0])
            scene.node_tree.links.new(rend_layers.outputs['Normal'], denoise.inputs['Normal'])
            scene.node_tree.links.new(rend_layers.outputs['DiffCol'],denoise.inputs['Albedo'])
            
        if slow_pass and not multi_pass:
            # rend_layers.location.x -= 300
            layers.cycles.denoising_store_passes = True
            denoise = scene.node_tree.nodes.new(type="CompositorNodeDenoise")
            strength = scene.node_tree.nodes.new(type="CompositorNodeMixRGB")
            denoise.name = "denoising_"
            strength.name = "denoising_strength"
            if link_comp:
                scene.node_tree.links.new(strength.outputs[0], comp.inputs[0])
                comp.location.x = strength.location.x + 250
            else:
                for node in scene.node_tree.nodes:
                    for input in node.inputs:
                        if input.is_linked and input.links[0].from_socket == rend_layers.outputs[0]:
                            scene.node_tree.links.new(strength.outputs[0], input)
                            x_min = min(node.location.x,x_min)
                rend_layers.location.x = x_min - 700
            denoise.location = (rend_layers.location.x + 300, rend_layers.location.y)
            strength.location = (rend_layers.location.x + 500, rend_layers.location.y)
            scene.node_tree.links.new(denoise.outputs[0], strength.inputs[2])
            scene.node_tree.links.new(rend_layers.outputs[0], strength.inputs[1])
            scene.node_tree.links.new(rend_layers.outputs[0], denoise.inputs[0])
            scene.node_tree.links.new(rend_layers.outputs['Denoising Normal'], denoise.inputs['Normal'])
            scene.node_tree.links.new(rend_layers.outputs['Denoising Albedo'], denoise.inputs['Albedo'])
            
        if multi_pass:
            # if mem:
                #tile the image in small chuncks, deactivate compositing. 
                #Later, the render.threads and render.threads_mode = 'FIXED' options will be set and compositing started
                # scene.render.use_compositing = False
            
            # warn if an option is missing compared to available materials.
            trans_case = ['VOL', 'TRANS', 'GLASS']
            trans_present = False
            # sss_present = False
            vol_present = False
            for mat in bpy.data.materials:
                if mat.use_nodes:
                    for node in mat.node_tree.nodes:
                        if node.type == 'BSDF_PRINCIPLED':
                            # if (node.inputs[1].is_linked or ((not node.inputs[1].is_linked) and node.inputs[1].default_value > 0)):
                                # sss_present = True
                            if (node.inputs[15].is_linked or ((not node.inputs[15].is_linked) and node.inputs[15].default_value > 0)):
                                trans_present = True
                        # if node.type == 'SUBSURFACE_SCATTERING':
                            # sss_present = True
                        for case in trans_case:
                            if case in node.type:
                                trans_present = True
                        if 'VOL' in node.type:
                            vol_present = True
            for group in bpy.data.node_groups:
                for node in group.nodes:
                    if node.type == 'BSDF_PRINCIPLED':
                        # if (node.inputs[1].is_linked or ((not node.inputs[1].is_linked) and node.inputs[1].default_value > 0)):
                            # sss_present = True
                        if (node.inputs[15].is_linked or ((not node.inputs[15].is_linked) and node.inputs[15].default_value > 0)):
                            trans_present = True
                    # if node.type == 'SUBSURFACE_SCATTERING':
                        # sss_present = True
                    for case in trans_case:
                        if case in node.type:
                            trans_present = True
                    if 'VOL' in node.type:
                        vol_present = True
            try:
                for node in bpy.context.scene.world.node_tree.nodes:
                    if 'VOL' in node.type:
                        vol_present = True
            except:
                r=1
            
            # if sss_present and not sss:
                # self.report({'WARNING'}, "SSS present in the scene, but not activated for denoising, activate SSS option if any is visible")
            if trans_present and not trans:
                self.report({'WARNING'}, "Transmitive material present in the scene, but not activated for denoising, activate Transmission option if any is visible")
            if vol_present and not vol:
                self.report({'WARNING'}, "Volumetrics present in the scene, but not activated for denoising, activate Transmission option if any is visible")
            
            # rend_layers.location.x -= 700
            if slow_pass:
                layers.cycles.denoising_store_passes = True
            else:
                layers.use_pass_diffuse_color = True
                layers.use_pass_normal = True
                layers.cycles.denoising_store_passes = False

            #create a new denoising node group
            group = bpy.data.node_groups.new(type="CompositorNodeTree", name="DenoiseLayer")
            group.inputs.new("NodeSocketColor", "Direct")
            group.inputs.new("NodeSocketColor", "Indirect")
            group.inputs.new("NodeSocketColor", "Color")
            group.inputs.new("NodeSocketColor", "Normal")
            group.inputs.new("NodeSocketColor", "Albedo")
            group.outputs.new("NodeSocketColor", "Image")

            input_node = group.nodes.new("NodeGroupInput")
            input_node.location = (0, 0)
            output_node = group.nodes.new("NodeGroupOutput")
            output_node.location = (600, 0)

            #create new nodes in the group
            #add_diff = scene.node_tree.nodes.new(type="CompositorNodeMixRGB")
            add = group.nodes.new(type="CompositorNodeMixRGB")
            add.blend_type = "ADD"
            multiply = group.nodes.new(type="CompositorNodeMixRGB")
            multiply.blend_type = "MULTIPLY"
            denoise = group.nodes.new(type="CompositorNodeDenoise")

            #internally link the nodes in the group
            group.links.new(input_node.outputs["Direct"], add.inputs[1])
            group.links.new(input_node.outputs["Indirect"], add.inputs[2])
            group.links.new(input_node.outputs["Color"], multiply.inputs[1])
            group.links.new(add.outputs[0], denoise.inputs[0])
            group.links.new(input_node.outputs["Albedo"], denoise.inputs["Albedo"])
            group.links.new(input_node.outputs["Normal"], denoise.inputs["Normal"])
            group.links.new(denoise.outputs[0], multiply.inputs[2])
            group.links.new(multiply.outputs[0], output_node.inputs[0])

            #create a new denoising node group
            group_vol = bpy.data.node_groups.new(type="CompositorNodeTree", name="DenoiseVol")
            group_vol.inputs.new("NodeSocketColor", "Direct")
            group_vol.inputs.new("NodeSocketColor", "Indirect")
            str = group_vol.inputs.new("NodeSocketFloat", "Strength")
            str.default_value = 1.00
            group_vol.outputs.new("NodeSocketColor", "Image")

            input_node = group_vol.nodes.new("NodeGroupInput")
            input_node.location = (0, 0)
            output_node = group_vol.nodes.new("NodeGroupOutput")
            output_node.location = (600, 0)

            #create new nodes in the group
            #add_diff = scene.node_tree.nodes.new(type="CompositorNodeMixRGB")
            add = group_vol.nodes.new(type="CompositorNodeMixRGB")
            add.blend_type = "ADD"
            vol_strength = group_vol.nodes.new(type="CompositorNodeMixRGB")
            vol_strength.blend_type = "MIX"
            denoise = group_vol.nodes.new(type="CompositorNodeDenoise")

            #internally link the nodes in the group
            group_vol.links.new(input_node.outputs["Direct"], add.inputs[1])
            group_vol.links.new(input_node.outputs["Indirect"], add.inputs[2])
            group_vol.links.new(add.outputs[0], denoise.inputs[0])
            group_vol.links.new(input_node.outputs["Strength"], vol_strength.inputs[0])
            group_vol.links.new(add.outputs[0], vol_strength.inputs[1])
            group_vol.links.new(denoise.outputs[0], vol_strength.inputs[2])
            group_vol.links.new(vol_strength.outputs[0], output_node.inputs[0])

            #create a new merge node group
            merge = bpy.data.node_groups.new(type="CompositorNodeTree", name="merge")
            merge.inputs.new("NodeSocketColor", "Diffuse")
            merge.inputs.new("NodeSocketColor", "Glossy")
            merge.inputs.new("NodeSocketColor", "Trans")
            # merge.inputs.new("NodeSocketColor", "SSS")
            merge.inputs.new("NodeSocketColor", "Volume")
            merge.inputs.new("NodeSocketColor", "Emit")
            merge.inputs.new("NodeSocketColor", "Env")
            merge.outputs.new("NodeSocketColor", "Image")

            input_node = merge.nodes.new("NodeGroupInput")
            input_node.location = (0, 0)
            output_node = merge.nodes.new("NodeGroupOutput")
            output_node.location = (600, 0)

            #create new nodes in the group
            #add_diff = scene.node_tree.nodes.new(type="CompositorNodeMixRGB")
            add1 = merge.nodes.new(type="CompositorNodeMixRGB")
            add1.blend_type = "ADD"
            # add2 = merge.nodes.new(type="CompositorNodeMixRGB")
            # add2.blend_type = "ADD"
            add3 = merge.nodes.new(type="CompositorNodeMixRGB")
            add3.blend_type = "ADD"
            add4 = merge.nodes.new(type="CompositorNodeMixRGB")
            add4.blend_type = "ADD"
            add5 = merge.nodes.new(type="CompositorNodeMixRGB")
            add5.blend_type = "ADD"
            add6 = merge.nodes.new(type="CompositorNodeMixRGB")
            add6.blend_type = "ADD"

            #internally link the nodes in the group
            merge.links.new(input_node.outputs["Diffuse"], add1.inputs[1])
            merge.links.new(input_node.outputs["Glossy"], add1.inputs[2])
            merge.links.new(input_node.outputs["Trans"], add4.inputs[2])
            # merge.links.new(input_node.outputs["SSS"], add2.inputs[2])
            merge.links.new(input_node.outputs["Emit"], add3.inputs[1])
            merge.links.new(input_node.outputs["Env"], add3.inputs[2])
            merge.links.new(add1.outputs[0], add4.inputs[1])
            # merge.links.new(add2.outputs[0], add4.inputs[2])
            merge.links.new(add3.outputs[0], add5.inputs[1])
            merge.links.new(add4.outputs[0], add5.inputs[2])
            merge.links.new(add5.outputs[0], add6.inputs[1])
            merge.links.new(input_node.outputs["Volume"], add6.inputs[2])
            merge.links.new(add6.outputs[0], output_node.inputs[0])

            #connect layers and the corresponding nodes   

            #activate the needed layers for denoising
            layers.use_pass_emit = True
            layers.use_pass_environment = True

            channel = 0

            merge_node = scene.node_tree.nodes.new("CompositorNodeGroup")
            merge_node.node_tree = merge
            merge_node.name = "denoising_merge"
            strength = scene.node_tree.nodes.new(type="CompositorNodeMixRGB")
            strength.name = "denoising_strength"
            if link_comp:
                scene.node_tree.links.new(strength.outputs[0], comp.inputs[0])
                comp.location.x = strength.location.x + 250
            else:
                for node in scene.node_tree.nodes:
                    for input in node.inputs:
                        if input.is_linked and input.links[0].from_socket == rend_layers.outputs[0]:
                            scene.node_tree.links.new(strength.outputs[0], input)
                            x_min = min(node.location.x,x_min)
                rend_layers.location.x = x_min - 1050
            merge_node.location = (rend_layers.location.x + 600, rend_layers.location.y)
            strength.location = (merge_node.location.x + 200, merge_node.location.y)
            scene.node_tree.links.new(merge_node.outputs[0], strength.inputs[2])
            scene.node_tree.links.new(rend_layers.outputs[0], strength.inputs[1])

            if color:
                layers.use_pass_diffuse_direct = True
                layers.use_pass_diffuse_indirect = True
                layers.use_pass_diffuse_color = True
                #add the denoising group to the compositing node tree
                diff = scene.node_tree.nodes.new("CompositorNodeGroup")
                diff.node_tree = group
                diff.location = (rend_layers.location.x + 400, rend_layers.location.y)
                diff.name = "denoising_diff"
                #link the layers with the node group
                scene.node_tree.links.new(rend_layers.outputs['DiffDir'],diff.inputs[0])
                scene.node_tree.links.new(rend_layers.outputs['DiffInd'],diff.inputs[1])
                scene.node_tree.links.new(rend_layers.outputs['DiffCol'],diff.inputs[2])
                if slow_pass:
                    scene.node_tree.links.new(rend_layers.outputs['Denoising Normal'],diff.inputs[3])
                    scene.node_tree.links.new(rend_layers.outputs['Denoising Albedo'],diff.inputs[4])
                else:
                    scene.node_tree.links.new(rend_layers.outputs['Normal'],diff.inputs[3])
                    scene.node_tree.links.new(rend_layers.outputs['DiffCol'],diff.inputs[4])
                
                scene.node_tree.links.new(diff.outputs[0], merge_node.inputs["Diffuse"])
                
            if glossy:
                layers.use_pass_glossy_direct = True
                layers.use_pass_glossy_indirect = True
                layers.use_pass_glossy_color = True
                #add the denoising group to the compositing node tree
                gloss = scene.node_tree.nodes.new("CompositorNodeGroup")
                gloss.node_tree = group
                gloss.location = (rend_layers.location.x + 400, rend_layers.location.y-200)
                gloss.name = "denoising_gloss"
                #link the layers with the node group
                scene.node_tree.links.new(rend_layers.outputs['GlossDir'],gloss.inputs[0])
                scene.node_tree.links.new(rend_layers.outputs['GlossInd'],gloss.inputs[1])
                scene.node_tree.links.new(rend_layers.outputs['GlossCol'],gloss.inputs[2])
                if slow_pass:
                    scene.node_tree.links.new(rend_layers.outputs['Denoising Normal'],gloss.inputs[3])
                    scene.node_tree.links.new(rend_layers.outputs['Denoising Albedo'],gloss.inputs[4])
                else:
                    scene.node_tree.links.new(rend_layers.outputs['Normal'],gloss.inputs[3])
                    scene.node_tree.links.new(rend_layers.outputs['DiffCol'],gloss.inputs[4])
                
                scene.node_tree.links.new(gloss.outputs[0], merge_node.inputs["Glossy"])
                    
            if trans and trans_present:
                layers.use_pass_transmission_direct = True
                layers.use_pass_transmission_indirect = True
                layers.use_pass_transmission_color = True
                #add the denoising group to the compositing node tree
                trans = scene.node_tree.nodes.new("CompositorNodeGroup")
                trans.node_tree = group
                trans.location = (rend_layers.location.x + 400, rend_layers.location.y-400)
                trans.name = "denoising_trans"
                #link the layers with the node group
                scene.node_tree.links.new(rend_layers.outputs['TransDir'],trans.inputs[0])
                scene.node_tree.links.new(rend_layers.outputs['TransInd'],trans.inputs[1])
                scene.node_tree.links.new(rend_layers.outputs['TransCol'],trans.inputs[2])
                if slow_pass:
                    scene.node_tree.links.new(rend_layers.outputs['Denoising Normal'],trans.inputs[3])
                    scene.node_tree.links.new(rend_layers.outputs['Denoising Albedo'],trans.inputs[4])
                else:
                    scene.node_tree.links.new(rend_layers.outputs['Normal'],trans.inputs[3])
                    scene.node_tree.links.new(rend_layers.outputs['DiffCol'],trans.inputs[4])
                
                scene.node_tree.links.new(trans.outputs[0], merge_node.inputs["Trans"])
            else:
                layers.use_pass_transmission_direct = False
                layers.use_pass_transmission_indirect = False
                layers.use_pass_transmission_color = False
                
            # if sss and sss_present:
                # layers.use_pass_subsurface_direct = True
                # layers.use_pass_subsurface_indirect = True
                # layers.use_pass_subsurface_color = True
                
                # sss = scene.node_tree.nodes.new("CompositorNodeGroup")
                # sss.node_tree = group
                # sss.location = (rend_layers.location.x + 400, rend_layers.location.y-600)
                # sss.name = "denoising_sss"
                
                # scene.node_tree.links.new(rend_layers.outputs['SubsurfaceDir'],sss.inputs[0])
                # scene.node_tree.links.new(rend_layers.outputs['SubsurfaceInd'],sss.inputs[1])
                # scene.node_tree.links.new(rend_layers.outputs['SubsurfaceCol'],sss.inputs[2])
                # if slow_pass:
                    # scene.node_tree.links.new(rend_layers.outputs['Denoising Normal'],sss.inputs[3])
                    # scene.node_tree.links.new(rend_layers.outputs['Denoising Albedo'],sss.inputs[4])
                # else:
                    # scene.node_tree.links.new(rend_layers.outputs['Normal'],sss.inputs[3])
                    # scene.node_tree.links.new(rend_layers.outputs['DiffCol'],sss.inputs[4])
                
                # scene.node_tree.links.new(sss.outputs[0], merge_node.inputs["SSS"])
            # else:
                # layers.use_pass_subsurface_direct = False
                # layers.use_pass_subsurface_indirect = False
                # layers.use_pass_subsurface_color = False
                
            if vol and vol_present:
                layers.cycles.use_pass_volume_direct = True
                layers.cycles.use_pass_volume_indirect = True
                #add the denoising group to the compositing node tree
                vol = scene.node_tree.nodes.new("CompositorNodeGroup")
                vol.node_tree = group_vol
                vol.location = (rend_layers.location.x + 400, rend_layers.location.y-800)
                vol.name = "denoising_vol"
                #link the layers with the node group
                scene.node_tree.links.new(rend_layers.outputs['VolumeDir'],vol.inputs[0])
                scene.node_tree.links.new(rend_layers.outputs['VolumeInd'],vol.inputs[1])
                
                scene.node_tree.links.new(vol.outputs[0], merge_node.inputs["Volume"])
            else:
                layers.cycles.use_pass_volume_direct = False
                layers.cycles.use_pass_volume_indirect = False
                
            scene.node_tree.links.new(rend_layers.outputs['Env'], merge_node.inputs["Env"])
            scene.node_tree.links.new(rend_layers.outputs['Emit'], merge_node.inputs["Emit"])

        
        return {'FINISHED'}
        
class AIDenoise_del(bpy.types.Operator):
    """Delete AI nodes"""
    bl_idname = "render.delnodes"
    bl_label = "Remove AI Denoise nodes"

    @classmethod
    def poll(cls, context):
        return context.scene.render.engine == 'CYCLES'

    def execute(self, context):
        win      = bpy.context.window
        scr      = win.screen
        if bpy.app.version[1] < 80:
            areas3d  = [area for area in scr.areas if area.type == 'NODE_EDITOR' and area.spaces[0].tree_type == 'CompositorNodeTree']
        else:
            areas3d  = [area for area in scr.areas if area.type == 'NODE_EDITOR' and area.ui_type == 'CompositorNodeTree']
        if len(areas3d) == 0:
            self.report({'ERROR'}, "A compositor editor must be opened for this operator to work")
            return {'FINISHED'}
        region   = [region for region in areas3d[0].regions if region.type == 'WINDOW']

        override = {'window':win, 'screen':scr, 'area'  :areas3d[0], 'region':region, 'scene' :bpy.context.scene, }
        for node in bpy.context.scene.node_tree.nodes:
            node.select = False
            if "denoising_" in node.name:
                node.select = True
        bpy.ops.node.delete_reconnect(override, 'INVOKE_DEFAULT')
        return {'FINISHED'}

class AIDenoise_regen(bpy.types.Operator):
    """Regenerate AI nodes"""
    bl_idname = "render.regennodes"
    bl_label = "Regenerate AI Denoise nodes"

    @classmethod
    def poll(cls, context):
        return context.scene.render.engine == 'CYCLES'

    def execute(self, context):
        bpy.ops.render.delnodes()
        bpy.ops.render.setnodes()
        return {'FINISHED'}

# class AIDenoise_denoise(bpy.types.Operator):
    # """Regenerate AI nodes"""
    # bl_idname = "render.lowmemdenoise"
    # bl_label = "Prepare low mem denoise"

    # @classmethod
    # def poll(cls, context):
        # return context.scene.render.engine == 'CYCLES'

    # def execute(self, context):
        # scene = bpy.context.scene
        # scene.render.use_compositing = True
        # scene.render.threads = 1
        # scene.render.threads_mode = 'FIXED' #TODO: Restore both after compositing
        # return {'FINISHED'}
    
def register():
    bpy.utils.register_class(AIDenoise)
    #bpy.utils.register_class(AISettings)
    bpy.utils.register_class(AIDenoise_del)
    bpy.utils.register_class(AIDenoise_regen)
    # bpy.utils.register_class(AIDenoise_denoise)

def unregister():
    bpy.utils.unregister_class(AIDenoise)
    bpy.utils.unregister_class(AISettings)
    bpy.utils.unregister_class(AIDenoise_del)
    bpy.utils.unregister_class(AIDenoise_regen)
    # bpy.utils.unregister_class(AIDenoise_denoise)

if __name__ == "__main__":
    register()
