/*
 * Copyright 2011-2013 Blender Foundation, Modifications Copyright Mathieu Menuet. 
 *
 * Licensed under the GPL License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.

 *
 * You should have received a copy of the GNU General Public License along with this program; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

CCL_NAMESPACE_BEGIN

/* Attribute Node */

ccl_device AttributeDescriptor svm_node_attr_init(
    KernelGlobals *kg, ShaderData *sd, uint4 node, NodeAttributeType *type, uint *out_offset)
{
  *out_offset = node.z;
  *type = (NodeAttributeType)node.w;

  AttributeDescriptor desc;

  if (sd->object != OBJECT_NONE) {
    desc = find_attribute(kg, sd, node.y);
    if (desc.offset == ATTR_STD_NOT_FOUND) {
      desc = attribute_not_found();
      desc.offset = 0;
      desc.type = (NodeAttributeType)node.w;
    }
  }
  else {
    /* background */
    desc = attribute_not_found();
    desc.offset = 0;
    desc.type = (NodeAttributeType)node.w;
  }

  return desc;
}

ccl_device void svm_node_attr(KernelGlobals *kg, ShaderData *sd, float *stack, uint4 node)
{
  NodeAttributeType type = NODE_ATTR_FLOAT;
  uint out_offset = 0;
  AttributeDescriptor desc = svm_node_attr_init(kg, sd, node, &type, &out_offset);

  /* fetch and store attribute */
  if (desc.type == NODE_ATTR_FLOAT) {
    float f = primitive_attribute_float(kg, sd, desc, NULL, NULL);
    if (type == NODE_ATTR_FLOAT) {
      stack_store_float(stack, out_offset, f);
    }
    else {
      stack_store_float3(stack, out_offset, make_float3(f, f, f));
    }
  }
  else if (desc.type == NODE_ATTR_FLOAT2) {
    float2 f = primitive_attribute_float2(kg, sd, desc, NULL, NULL);
    if (type == NODE_ATTR_FLOAT) {
      stack_store_float(stack, out_offset, f.x);
    }
    else {
      stack_store_float3(stack, out_offset, make_float3(f.x, f.y, 0.0f));
    }
  }
  else if (desc.type == NODE_ATTR_RGBA) {
    float4 f = primitive_attribute_float4(kg, sd, desc, NULL, NULL);
    if (type == NODE_ATTR_FLOAT) {
      stack_store_float(stack, out_offset, average(float4_to_float3(f)));
    }
    else {
      stack_store_float3(stack, out_offset, float4_to_float3(f));
    }
  }
  else {
    float3 f = primitive_attribute_float3(kg, sd, desc, NULL, NULL);
    if (type == NODE_ATTR_FLOAT) {
      stack_store_float(stack, out_offset, average(f));
    }
    else {
      stack_store_float3(stack, out_offset, f);
    }
  }
}

ccl_device void svm_node_attr_bump_dx(KernelGlobals *kg, ShaderData *sd, float *stack, uint4 node)
{
  NodeAttributeType type = NODE_ATTR_FLOAT;
  uint out_offset = 0;
  AttributeDescriptor desc = svm_node_attr_init(kg, sd, node, &type, &out_offset);

  /* fetch and store attribute */
  if (desc.type == NODE_ATTR_FLOAT) {
    float dx;
    float f = primitive_surface_attribute_float(kg, sd, desc, &dx, NULL);
    if (type == NODE_ATTR_FLOAT) {
      stack_store_float(stack, out_offset, f + dx);
    }
    else {
      stack_store_float3(stack, out_offset, make_float3(f + dx, f + dx, f + dx));
    }
  }
  else if (desc.type == NODE_ATTR_FLOAT2) {
    float2 dx;
    float2 f = primitive_attribute_float2(kg, sd, desc, &dx, NULL);
    if (type == NODE_ATTR_FLOAT) {
      stack_store_float(stack, out_offset, f.x + dx.x);
    }
    else {
      stack_store_float3(stack, out_offset, make_float3(f.x + dx.x, f.y + dx.y, 0.0f));
    }
  }
  else if (desc.type == NODE_ATTR_RGBA) {
    float4 dx;
    float4 f = primitive_attribute_float4(kg, sd, desc, &dx, NULL);
    if (type == NODE_ATTR_FLOAT) {
      stack_store_float(stack, out_offset, average(float4_to_float3(f + dx)));
    }
    else {
      stack_store_float3(stack, out_offset, float4_to_float3(f + dx));
    }
  }
  else {
    float3 dx;
    float3 f = primitive_surface_attribute_float3(kg, sd, desc, &dx, NULL);
    if (type == NODE_ATTR_FLOAT) {
      stack_store_float(stack, out_offset, average(f + dx));
    }
    else {
      stack_store_float3(stack, out_offset, f + dx);
    }
  }
}

ccl_device void svm_node_attr_bump_dy(KernelGlobals *kg, ShaderData *sd, float *stack, uint4 node)
{
  NodeAttributeType type = NODE_ATTR_FLOAT;
  uint out_offset = 0;
  AttributeDescriptor desc = svm_node_attr_init(kg, sd, node, &type, &out_offset);

  /* fetch and store attribute */
  if (desc.type == NODE_ATTR_FLOAT) {
    float dy;
    float f = primitive_surface_attribute_float(kg, sd, desc, NULL, &dy);
    if (type == NODE_ATTR_FLOAT) {
      stack_store_float(stack, out_offset, f + dy);
    }
    else {
      stack_store_float3(stack, out_offset, make_float3(f + dy, f + dy, f + dy));
    }
  }
  else if (desc.type == NODE_ATTR_FLOAT2) {
    float2 dy;
    float2 f = primitive_attribute_float2(kg, sd, desc, NULL, &dy);
    if (type == NODE_ATTR_FLOAT) {
      stack_store_float(stack, out_offset, f.x + dy.x);
    }
    else {
      stack_store_float3(stack, out_offset, make_float3(f.x + dy.x, f.y + dy.y, 0.0f));
    }
  }
  else if (desc.type == NODE_ATTR_RGBA) {
    float4 dy;
    float4 f = primitive_attribute_float4(kg, sd, desc, NULL, &dy);
    if (type == NODE_ATTR_FLOAT) {
      stack_store_float(stack, out_offset, average(float4_to_float3(f + dy)));
    }
    else {
      stack_store_float3(stack, out_offset, float4_to_float3(f + dy));
    }
  }
  else {
    float3 dy;
    float3 f = primitive_surface_attribute_float3(kg, sd, desc, NULL, &dy);
    if (type == NODE_ATTR_FLOAT) {
      stack_store_float(stack, out_offset, average(f + dy));
    }
    else {
      stack_store_float3(stack, out_offset, f + dy);
    }
  }
}

CCL_NAMESPACE_END
