﻿# This script updates (if this script is run from a folder where Echosync is already present) or downloads and extracts Echosync in the current folder
# Version 2.7 (2022-07-07)
# Copyright Sylvain Rougeaux, 2018-2022, www.luminescence-software.org

param([string] $scriptFolder = "?")
$ErrorActionPreference = "Stop"

$scriptBlock = 
{
    param([string] $echosyncFolder)

    $DownloadingEchosync = "Downloading Echosync from the Luminescence Software website..."
    $ScriptRootNotIdentified = "The folder in which Echosync should be installed or updated could not be identified."
    $WaitingForEchosyncClosing = "Waiting for Echosync closing..."
    $EchosyncClosed = "Echosync has been closed."
    $EchosyncDownloaded = "Echosync has been successfully downloaded."
    $InvalidZipFile = "The Echosync zip archive is not valid because it doesn't contain any root folder."
    $EchosyncUpdating = "Echosync is ready to be updated."
    $EchosyncExtracting = "Extracting data from the Echosync zip archive..."
    $EchosyncUpdated = "Echosync has been successfully updated in '{0}'."
    $PreviousInstallationLocation = "Previous installation has been kept in '{0}'. You can delete this folder if you are satisfied with the latest version."
    $EchosyncInstalling = "Echosync is ready to be installed."
    $EchosyncFolderAlreadyExists = "The Echosync zip archive can not be extracted in '{0}' because a folder named '{1}' is already there."
    $EchosyncInstalled = "Echosync has been successfully installed in '{0}'."
    $FolderAccessDenied = "Make sure you have closed all processes using this path, and have sufficient rights to write to this path, before trying again."
    $AbortRetryQuestion = "Press 'A' to abort or any other key to retry"
    $FileHashMismatch = "The Echosync zip archive is not valid because its hash ({0}) doesn't match the expected one ({1})."

    $culture = (Get-UICulture).TwoLetterISOLanguageName
    if ($culture -eq "fr")
    {
        $DownloadingEchosync = "Téléchargement d'Echosync depuis le site web Luminescence Software..."
        $ScriptRootNotIdentified = "Le répertoire dans lequel Echosync doit être installé ou mis à jour n'a pas pu être identifié."
        $WaitingForEchosyncClosing = "En attente de la fermeture d'Echosync..."
        $EchosyncClosed = "Echosync a été fermé."
        $EchosyncDownloaded = "Echosync a été téléchargé avec succès."
        $InvalidZipFile = "L'archive zip d'Echosync n'est pas valide car elle ne contient pas de répertoire racine."
        $EchosyncUpdating = "Echosync est prêt à être mis à jour."
        $EchosyncExtracting = "Décompression des données de l'archive zip d'Echosync..."
        $EchosyncUpdated = "Echosync a été mis à jour avec succès dans '{0}'."
        $PreviousInstallationLocation = "L'installation précédente a été conservée dans '{0}'. Vous pourrez supprimer ce répertoire si vous êtes satisfait de la dernière version."
        $EchosyncInstalling = "Echosync est prêt à être installé."
        $EchosyncFolderAlreadyExists = "L'archive zip d'Echosync ne peut être extraite dans '{0}' car un dossier nommé '{1}' y est déjà présent."
        $EchosyncInstalled = "Echosync a été installé avec succès dans '{0}'."
        $FolderAccessDenied = "Assurez-vous d'avoir fermé tous les processus utilisant ce chemin, et de disposer des droits suffisants pour écrire dans ce chemin, avant de réessayer."
        $AbortRetryQuestion = "Appuyez sur 'A' pour abandonner ou toute autre touche pour réessayer"
        $FileHashMismatch = "L'archive zip d'Echosync n'est pas valide car son hash ({0}) ne correspond pas à celui attendu ({1})."
    }
    elseif ($culture -eq "it")
    {
        # Italian translations provided by Fabio Biocchetti
        $DownloadingEchosync = "Download di Echosync dal sito di Luminescence Software in corso..."
        $ScriptRootNotIdentified = "La cartella di installazione di Echosync non può essere trovata."
        $WaitingForEchosyncClosing = "Chiusura di Echosync in corso..."
        $EchosyncClosed = "	Echosync è stato chiuso."
        $EchosyncDownloaded = "Echosync è stato scaricato con successo."
        $InvalidZipFile = "L'archivio zip di Echosync non è valido perché non contiene alcuna cartella di root."
        $EchosyncUpdating = "Echosync è pronto per essere aggiornato."
        $EchosyncExtracting = "Estrazione dei dati dall'archivio zip di Echosync..."
        $EchosyncUpdated = "Echosync è stato aggiornato con successo in '{0}'."
        $PreviousInstallationLocation = "Una precedente installazione è stata conservata in '{0}'. Puoi eliminare questa cartella se ti stai trovando bene con l'ultima versione."
        $EchosyncInstalling = "Echosync è pronto per essere installato."
        $EchosyncFolderAlreadyExists = "L'archivio zip di Echosync non può essere estratto in '{0}' perché una cartella chiamata '{1}' già esiste in quel percorso."
        $EchosyncInstalled = "Echosync è stato installato con successo in '{0}'."
        $FolderAccessDenied = "Assicurati di aver chiuso tutti i processi che usano questo percorso, e di avere sufficienti diritti di scrittura, prima di riprovare."
        $AbortRetryQuestion = "Premi 'A' per annullare o qualsiasi altro tasto per riprovare"
        $FileHashMismatch = "L'archivio zip Echosync non è valido perché il suo hash ({0}) non corrisponde a quello atteso ({1})."
    }
    elseif ($culture -eq "es")
    {
        # Spanish translations provided by Javier Tapia Torres
        $DownloadingEchosync = "Descargando Echosync desde el sitio web de Luminescence Software..."
        $ScriptRootNotIdentified = "La carpeta en la cual Echosync debería ser instalado o actualizado puede no estar identificada."
        $WaitingForEchosyncClosing = "Esperando que Echosync se cierre..."
        $EchosyncClosed = "Echosync se ha cerrado."
        $EchosyncDownloaded = "Echosync ha sido descargado correctamente."
        $InvalidZipFile = "El archivo zip de Echosync no es válido porque no contiene ninguna carpeta raíz."
        $EchosyncUpdating = "Echosync está listo para ser actualizado."
        $EchosyncExtracting = "Extrayendo datos desde el archivo zip de Echosync..."
        $EchosyncUpdated = "Echosync ha sido actualizado con éxito en '{0}'."
        $PreviousInstallationLocation = "La instalación anterior se ha mantenido en '{0}'. Puede eliminar esta carpeta si está satisfecho con la última versión."
        $EchosyncInstalling = "Echosync está listo para ser instalado."
        $EchosyncFolderAlreadyExists = "El archivo zip de Echosync no se puede extraer en '{0}' porque ya hay una carpeta llamada '{1}'."
        $EchosyncInstalled = "Echosync se ha instalado con éxito en '{0}'."
        $FolderAccessDenied = "Asegúrese de haber cerrado todos los procesos que utilizan esta ruta y de tener derechos insuficientes para escribir en esta ruta, antes de intentarlo de nuevo."
        $AbortRetryQuestion = "Pulse 'A' para anular o cualquier otra tecla para reintentar"
        $FileHashMismatch = "El archivo zip de Echosync no es válido porque su hash ({0}) no coincide con el esperado ({1})."
    }
    elseif ($culture -eq "de")
    {
        # German translations provided by Roger Taouss-Schirmer
        $DownloadingEchosync = "Echosync wird von Luminescence Software heruntergeladen..."
        $ScriptRootNotIdentified = "Der Installationsordner für Echosync wurde nicht gefunden."
        $WaitingForEchosyncClosing = "Warten auf Beenden von Echosync..."
        $EchosyncClosed = "Echosync wurde beendet."
        $EchosyncDownloaded = "Echosync wurde erfolgreich heruntergeladen."
        $InvalidZipFile = "Das Echosync zip Archiv ist ungültig, da es keinen Quellordner enthält."
        $EchosyncUpdating = "Echosync kann upgedated werden."
        $EchosyncExtracting = "Extraktion des Echosync zip Archivs..."
        $EchosyncUpdated = "Echosync wurde in '{0}' erfolgreich upgedated."
        $PreviousInstallationLocation = "Die vorherige Installation wurde in '{0}' belassen. Sie können diesen Ordner löschen, wenn Sie mit der neuesten Version zufrieden sind."
        $EchosyncInstalling = "Echosync kann nun installiert werden."
        $EchosyncFolderAlreadyExists = "Echosync kann nicht nach '{0}' extrahiert werden, da dort bereits ein Ordner '{1}' existiert."
        $EchosyncInstalled = "Echosync wurde erfolgreich in '{0}' installiert."
        $FolderAccessDenied = "Verlassen Sie alle Prozesse, die diesen Weg benutzen und vergewissern Sie sich, dass Sie autorisiert sind diesen Weg zu benutzen bevor Sie es erneut versuchen."
        $AbortRetryQuestion = "Geben Sie 'A' ein um abzubrechen oder drücken Sie irgendeine andere Taste um erneut zu versuchen"
        $FileHashMismatch = "Das Echosync zip-Archiv ist ungültig da die digitale Signatur {0}  nicht mit der erwarteten Signatur {1} übereinstimmt."
    }
    elseif ($culture -eq "ru")
    {
        # Russian translations provided by Jyrkka Teterev
        $DownloadingEchosync = "Скачивание Echosync с веб-сайта Luminescence Software..."
        $ScriptRootNotIdentified = "Папка, в которой Echosync должен быть установлен или обновлён, не может быть идентифицирована."
        $WaitingForEchosyncClosing = "Ожидание закрытия Echosync..."
        $EchosyncClosed = "Echosync был закрыт."
        $EchosyncDownloaded = "Echosync был успешно скачан."
        $InvalidZipFile = "ZIP-архив Echosync недействителен, поскольку в нём нет корневой папки."
        $EchosyncUpdating = "Echosync готов к обновлению."
        $EchosyncExtracting = "Извлечение данных из ZIP-архива Echosync..."
        $EchosyncUpdated = "Echosync был успешно обнавлён в '{0}'."
        $PreviousInstallationLocation = "Предыдущая установка была сохранена в '{0}'. Вы можете удалить эту папку, если вас устраивает последняя версия."
        $EchosyncInstalling = "Echosync готов к установке."
        $EchosyncFolderAlreadyExists = "Не удалось распаковать ZIP-архив Echosync в '{0}', поскольку папка с именем '{1}' уже существует."
        $EchosyncInstalled = "Echosync был успешно установлен в '{0}'."
        $FolderAccessDenied = "Перед повторной попыткой убедитесь, что вы закрыли все процессы, использующие этот путь, и имеете достаточные права для записи по этому пути."
        $AbortRetryQuestion = "Нажмите 'A' для отмены или любую другую клавишу для повторной попытки"
        $FileHashMismatch = "Архив Echosync, вероятно, повреждён, поскольку его контрольная сумма({0}) не совпадает с ожидаемой({1})."
    }
    elseif ($culture -eq "zh")
    {
        # Chinese translations provided by 王晨旭
        $DownloadingEchosync = "从 Luminescence 网站下载 Echosync..."
        $ScriptRootNotIdentified = "无法确定应在其中安装或更新 Echosync 的文件夹。"
        $WaitingForEchosyncClosing = "等待 Echosync 关闭..."
        $EchosyncClosed = "Echosync 已关闭。"
        $EchosyncDownloaded = "Echosync 下载已成功。"
        $InvalidZipFile = "Echosync zip 存档无效，因为它不包含任何根文件夹。"
        $EchosyncUpdating = "Echosync 已准备好进行更新。"
        $EchosyncExtracting = "从 Echosync zip 存档中提取数据..."
        $EchosyncUpdated = "Echosync 已成功在 '{0}' 中更新。"
        $PreviousInstallationLocation = "以前的安装已保存在 '{0}' 中。 如果您对最新版本满意，可以删除此文件夹。"
        $EchosyncInstalling = "Echosync 已准备好安装。"
        $EchosyncFolderAlreadyExists = "无法在 '{0}' 中提取 Echosync zip 存档，因为已经存在名为 '{1}' 的文件夹。"
        $EchosyncInstalled = "Echosync 已成功安装在 '{0}' 中。"
        $FolderAccessDenied = "在重试之前，请确保已关闭使用此路径的所有进程，并具有足够的权限来写入此路径。"
        $AbortRetryQuestion = "按 'A' 放弃或按其他任何键重试"
        $FileHashMismatch = "Echosync zip 存档无效，因为其哈希 ({0}) 与预期的 ({1}) 不匹配。"
    }
    elseif ($culture -eq "nl")
    {
        # Dutch translations provided by Jeroen Westera
        $DownloadingEchosync = "Download Echosync van de Luminescence Software website..."
        $ScriptRootNotIdentified = "De map waarin Echosync moet worden geïnstalleerd of moet worden bijgewerkt kan niet gevonden worden."
        $WaitingForEchosyncClosing = "Bezig met wachten op het sluiten van Echosync..."
        $EchosyncClosed = "Echosync is afgesloten."
        $EchosyncDownloaded = "Echosync is succesvol gedownload."
        $InvalidZipFile = "Het zip-archief van Echosync is niet geldig omdat het geen hoofdmap bevat."
        $EchosyncUpdating = "Echosync is klaar om te worden bijgewerkt."
        $EchosyncExtracting = "Gegevens uit het Echosync zip-archief halen..."
        $EchosyncUpdated = "Echosync is succesvol bijgewerkt in '{0}'."
        $PreviousInstallationLocation = "De vorige installatie is in de map '{0}' bewaard. U kunt deze map verwijderen als u tevreden bent met de nieuwste versie."
        $EchosyncInstalling = "Echosync is klaar om te worden geïnstalleerd."
        $EchosyncFolderAlreadyExists = "Het zip-archief van Echosync kan niet worden uitgepakt in '{0}' omdat er al een map met de naam '{1}' is."
        $EchosyncInstalled = "Echosync is succesvol geïnstalleerd in '{0}'."
        $FolderAccessDenied = "Zorg ervoor dat u alle processen die dit pad gebruiken, hebt gesloten en voldoende rechten hebt om naar dit pad te schrijven, voordat u het opnieuw probeert."
        $AbortRetryQuestion = "Druk op 'A' om af te breken of een andere toets om het opnieuw te proberen"
        $FileHashMismatch = "Het zip-archief van Echosync is niet geldig omdat de hash ({0}) niet overeenkomt met de verwachte ({1})."
    }

    if (-not (Test-Path -Path $echosyncFolder -PathType Container))
    {
        Write-Host $ScriptRootNotIdentified -ForegroundColor Red
        exit
    }

    Add-Type -Assembly "System.IO.Compression.FileSystem"
    Function GetUniqueRootFolder([string] $zipPath)
    {
        $zip = [IO.Compression.ZipFile]::OpenRead($zipPath)
        try 
        {
            if ($zip.Entries.Count -eq 0) { return $null }

            $index = $zip.Entries[0].FullName.IndexOf('/')
            if ($index -eq -1) { return $null }

            $firstEntry = $zip.Entries[0].FullName.Substring(0, $index + 1)
            foreach ($entry in $zip.Entries) {
                if (-not $entry.FullName.StartsWith($firstEntry)) { return $null }
            }

            return $firstEntry.TrimEnd('/')
        }
        finally 
        {
            $zip.Dispose()
        }
    }

    Function WaitForEchosyncToExit()
    {
        $echosyncProcess = Get-Process -Name "Echosync" -ErrorAction SilentlyContinue
        if ($null -ne $echosyncProcess)
        {
            Write-Host $WaitingForEchosyncClosing -ForegroundColor Yellow
            $echosyncProcess | Foreach-Object { $_.CloseMainWindow() | Out-Null }
            Wait-Process -InputObject $echosyncProcess
            Write-Host $EchosyncClosed
        }
    }

    Function TryRenameFolder([string] $path, [string] $newPath)
    {
        while ($true)
        {
            try
            {
                Rename-Item -Path $path -NewName $newPath -Force
                return $true
            }
            catch
            {
                Write-Host $_.Exception.Message -ForegroundColor Red
                Write-Host $FolderAccessDenied -ForegroundColor Yellow
            
                $response = Read-Host -Prompt $AbortRetryQuestion
                if ($response -eq "a") { return $false }
            }
        }
    }

    Function TryExtractZip([string] $zipPath, [string] $destinationPath)
    {
        try 
        {
            [IO.Compression.ZipFile]::ExtractToDirectory($zipPath, $destinationPath)
            return $true
        }
        catch 
        {
            Write-Host $_.Exception.GetBaseException().Message -ForegroundColor Red
            return $false
        }
    }

    Function Quit()
    {
        Set-Location -Path $echosyncFolder
        exit
    }

    Write-Host $DownloadingEchosync
    $installerZip = Join-Path -Path $env:TEMP -ChildPath "Echosync.zip"
    Invoke-WebRequest -Uri "https://www.luminescence-software.org/download/echosync/Echosync.zip" -OutFile $installerZip -UseBasicParsing # -UseBasicParsing is required for compatibility reason when PowerShell < 6.0
    Write-Host $EchosyncDownloaded

    $hash = Get-FileHash -Algorithm "SHA256" -Path $installerZip
    $expected_hash = Invoke-WebRequest -Uri "https://www.luminescence-software.org/webservices/echosync_hash.php" -UseBasicParsing # -UseBasicParsing is required for compatibility reason when PowerShell < 6.0
    if ($hash.Hash -ne $expected_hash.Content)
    {    
        Write-Host ($FileHashMismatch -f $hash.Hash, $expected_hash.Content) -ForegroundColor Red
        Quit
    }

    $rootFolder = GetUniqueRootFolder $installerZip
    if ($null -eq $rootFolder)
    {
        Write-Host $InvalidZipFile -ForegroundColor Red
        Quit
    }

    $echosyncExe = Join-Path -Path $echosyncFolder -ChildPath "Echosync.exe"
    if (Test-Path -Path $echosyncExe -PathType Leaf)
    {
        Write-Host $EchosyncUpdating

        $oldProfilesPath = Join-Path -Path $echosyncFolder -ChildPath "Profiles"
        $hasProfiles = Test-Path -Path $oldProfilesPath -PathType Container
        if ($hasProfiles) { WaitForEchosyncToExit }

        $echosyncParentFolder = Split-Path $echosyncFolder -Parent
        $newEchosyncFolder = Join-Path -Path $echosyncParentFolder -ChildPath $rootFolder
        $oldEchosyncFolder = $echosyncFolder

        if (Test-Path -Path $newEchosyncFolder -PathType Container)
        {
            $oldEchosyncFolder = "${newEchosyncFolder}_old_$((Get-Date).Ticks)"
            if ((TryRenameFolder $newEchosyncFolder $oldEchosyncFolder) -ne $true) { Quit }
            $oldProfilesPath = Join-Path -Path $oldEchosyncFolder -ChildPath "Profiles"
        }
    
        Write-Host $EchosyncExtracting
        if ((TryExtractZip $installerZip $echosyncParentFolder) -ne $true) { Quit }

        if ($hasProfiles) { Copy-Item -Path $oldProfilesPath -Recurse -Destination $newEchosyncFolder -Container }

        Write-Host ($EchosyncUpdated -f $newEchosyncFolder) -ForegroundColor Green
        Write-Host ($PreviousInstallationLocation -f $oldEchosyncFolder)
    }
    else 
    {
        Write-Host $EchosyncInstalling

        $newEchosyncFolder = Join-Path -Path $echosyncFolder -ChildPath $rootFolder
        if (Test-Path -Path $newEchosyncFolder -PathType Container)
        {
            Write-Host ($EchosyncFolderAlreadyExists -f $echosyncFolder, $rootFolder) -ForegroundColor Red
            Quit
        }

        Write-Host $EchosyncExtracting
        if ((TryExtractZip $installerZip $echosyncFolder) -ne $true) { Quit }
    
        Write-Host ($EchosyncInstalled -f $newEchosyncFolder) -ForegroundColor Green
    }

    Remove-Item $installerZip -ErrorAction SilentlyContinue

    Set-Location -Path $newEchosyncFolder
    Start-Process -FilePath $newEchosyncFolder
}

if (!(Test-Path variable:PSVersionTable) -or $PSVersionTable.PSVersion.Major -lt 4) # PowerShell v4 is required for Get-FileHash command
{
    Write-Host "This script requires PowerShell v4 or later." -ForegroundColor Red
    Read-Host -Prompt "Press Enter to continue..." # "Pause" is not supported in PowerShell v2
    exit
}

$programFilesPath = [Environment]::GetFolderPath([Environment+SpecialFolder]::ProgramFiles)
if ($PSScriptRoot.StartsWith($programFilesPath) -eq $true)
{
    Write-Host "This script can only install or update portable version of Echosync. It cannot be run from the Program Files folder." -ForegroundColor Red
    Pause
    exit
}

$tempPath = [IO.Path]::GetTempPath().TrimEnd('\')
if ($PSScriptRoot.StartsWith($tempPath) -eq $false)
{
    Set-Location -Path "C:\"
    Copy-Item -Path $PSCommandPath -Destination $tempPath -Force
    $scriptPath = Join-Path -Path $tempPath -ChildPath (Split-Path $PSCommandPath -Leaf)
    Start-Process -FilePath "powershell.exe" -ArgumentList "-NoExit -NoLogo -ExecutionPolicy Bypass -File `"$scriptPath`" `"$PSScriptRoot`""
    [Environment]::Exit(0)
}

Invoke-Command -ScriptBlock $scriptBlock -ArgumentList $scriptFolder
# SIG # Begin signature block
# MIIm2gYJKoZIhvcNAQcCoIImyzCCJscCAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCAmFCg7cwDts37/
# 7vyV4WsoHRJPLHrEd1yQWi8uBgp6q6CCH8UwggVvMIIEV6ADAgECAhBI/JO0YFWU
# jTanyYqJ1pQWMA0GCSqGSIb3DQEBDAUAMHsxCzAJBgNVBAYTAkdCMRswGQYDVQQI
# DBJHcmVhdGVyIE1hbmNoZXN0ZXIxEDAOBgNVBAcMB1NhbGZvcmQxGjAYBgNVBAoM
# EUNvbW9kbyBDQSBMaW1pdGVkMSEwHwYDVQQDDBhBQUEgQ2VydGlmaWNhdGUgU2Vy
# dmljZXMwHhcNMjEwNTI1MDAwMDAwWhcNMjgxMjMxMjM1OTU5WjBWMQswCQYDVQQG
# EwJHQjEYMBYGA1UEChMPU2VjdGlnbyBMaW1pdGVkMS0wKwYDVQQDEyRTZWN0aWdv
# IFB1YmxpYyBDb2RlIFNpZ25pbmcgUm9vdCBSNDYwggIiMA0GCSqGSIb3DQEBAQUA
# A4ICDwAwggIKAoICAQCN55QSIgQkdC7/FiMCkoq2rjaFrEfUI5ErPtx94jGgUW+s
# hJHjUoq14pbe0IdjJImK/+8Skzt9u7aKvb0Ffyeba2XTpQxpsbxJOZrxbW6q5KCD
# J9qaDStQ6Utbs7hkNqR+Sj2pcaths3OzPAsM79szV+W+NDfjlxtd/R8SPYIDdub7
# P2bSlDFp+m2zNKzBenjcklDyZMeqLQSrw2rq4C+np9xu1+j/2iGrQL+57g2extme
# me/G3h+pDHazJyCh1rr9gOcB0u/rgimVcI3/uxXP/tEPNqIuTzKQdEZrRzUTdwUz
# T2MuuC3hv2WnBGsY2HH6zAjybYmZELGt2z4s5KoYsMYHAXVn3m3pY2MeNn9pib6q
# RT5uWl+PoVvLnTCGMOgDs0DGDQ84zWeoU4j6uDBl+m/H5x2xg3RpPqzEaDux5mcz
# mrYI4IAFSEDu9oJkRqj1c7AGlfJsZZ+/VVscnFcax3hGfHCqlBuCF6yH6bbJDoEc
# QNYWFyn8XJwYK+pF9e+91WdPKF4F7pBMeufG9ND8+s0+MkYTIDaKBOq3qgdGnA2T
# OglmmVhcKaO5DKYwODzQRjY1fJy67sPV+Qp2+n4FG0DKkjXp1XrRtX8ArqmQqsV/
# AZwQsRb8zG4Y3G9i/qZQp7h7uJ0VP/4gDHXIIloTlRmQAOka1cKG8eOO7F/05QID
# AQABo4IBEjCCAQ4wHwYDVR0jBBgwFoAUoBEKIz6W8Qfs4q8p74Klf9AwpLQwHQYD
# VR0OBBYEFDLrkpr/NZZILyhAQnAgNpFcF4XmMA4GA1UdDwEB/wQEAwIBhjAPBgNV
# HRMBAf8EBTADAQH/MBMGA1UdJQQMMAoGCCsGAQUFBwMDMBsGA1UdIAQUMBIwBgYE
# VR0gADAIBgZngQwBBAEwQwYDVR0fBDwwOjA4oDagNIYyaHR0cDovL2NybC5jb21v
# ZG9jYS5jb20vQUFBQ2VydGlmaWNhdGVTZXJ2aWNlcy5jcmwwNAYIKwYBBQUHAQEE
# KDAmMCQGCCsGAQUFBzABhhhodHRwOi8vb2NzcC5jb21vZG9jYS5jb20wDQYJKoZI
# hvcNAQEMBQADggEBABK/oe+LdJqYRLhpRrWrJAoMpIpnuDqBv0WKfVIHqI0fTiGF
# OaNrXi0ghr8QuK55O1PNtPvYRL4G2VxjZ9RAFodEhnIq1jIV9RKDwvnhXRFAZ/ZC
# J3LFI+ICOBpMIOLbAffNRk8monxmwFE2tokCVMf8WPtsAO7+mKYulaEMUykfb9gZ
# pk+e96wJ6l2CxouvgKe9gUhShDHaMuwV5KZMPWw5c9QLhTkg4IUaaOGnSDip0TYl
# d8GNGRbFiExmfS9jzpjoad+sPKhdnckcW67Y8y90z7h+9teDnRGWYpquRRPaf9xH
# +9/DUp/mBlXpnYzyOmJRvOwkDynUWICE5EV7WtgwggYaMIIEAqADAgECAhBiHW0M
# UgGeO5B5FSCJIRwKMA0GCSqGSIb3DQEBDAUAMFYxCzAJBgNVBAYTAkdCMRgwFgYD
# VQQKEw9TZWN0aWdvIExpbWl0ZWQxLTArBgNVBAMTJFNlY3RpZ28gUHVibGljIENv
# ZGUgU2lnbmluZyBSb290IFI0NjAeFw0yMTAzMjIwMDAwMDBaFw0zNjAzMjEyMzU5
# NTlaMFQxCzAJBgNVBAYTAkdCMRgwFgYDVQQKEw9TZWN0aWdvIExpbWl0ZWQxKzAp
# BgNVBAMTIlNlY3RpZ28gUHVibGljIENvZGUgU2lnbmluZyBDQSBSMzYwggGiMA0G
# CSqGSIb3DQEBAQUAA4IBjwAwggGKAoIBgQCbK51T+jU/jmAGQ2rAz/V/9shTUxjI
# ztNsfvxYB5UXeWUzCxEeAEZGbEN4QMgCsJLZUKhWThj/yPqy0iSZhXkZ6Pg2A2NV
# DgFigOMYzB2OKhdqfWGVoYW3haT29PSTahYkwmMv0b/83nbeECbiMXhSOtbam+/3
# 6F09fy1tsB8je/RV0mIk8XL/tfCK6cPuYHE215wzrK0h1SWHTxPbPuYkRdkP05Zw
# mRmTnAO5/arnY83jeNzhP06ShdnRqtZlV59+8yv+KIhE5ILMqgOZYAENHNX9SJDm
# +qxp4VqpB3MV/h53yl41aHU5pledi9lCBbH9JeIkNFICiVHNkRmq4TpxtwfvjsUe
# dyz8rNyfQJy/aOs5b4s+ac7IH60B+Ja7TVM+EKv1WuTGwcLmoU3FpOFMbmPj8pz4
# 4MPZ1f9+YEQIQty/NQd/2yGgW+ufflcZ/ZE9o1M7a5Jnqf2i2/uMSWymR8r2oQBM
# dlyh2n5HirY4jKnFH/9gRvd+QOfdRrJZb1sCAwEAAaOCAWQwggFgMB8GA1UdIwQY
# MBaAFDLrkpr/NZZILyhAQnAgNpFcF4XmMB0GA1UdDgQWBBQPKssghyi47G9IritU
# pimqF6TNDDAOBgNVHQ8BAf8EBAMCAYYwEgYDVR0TAQH/BAgwBgEB/wIBADATBgNV
# HSUEDDAKBggrBgEFBQcDAzAbBgNVHSAEFDASMAYGBFUdIAAwCAYGZ4EMAQQBMEsG
# A1UdHwREMEIwQKA+oDyGOmh0dHA6Ly9jcmwuc2VjdGlnby5jb20vU2VjdGlnb1B1
# YmxpY0NvZGVTaWduaW5nUm9vdFI0Ni5jcmwwewYIKwYBBQUHAQEEbzBtMEYGCCsG
# AQUFBzAChjpodHRwOi8vY3J0LnNlY3RpZ28uY29tL1NlY3RpZ29QdWJsaWNDb2Rl
# U2lnbmluZ1Jvb3RSNDYucDdjMCMGCCsGAQUFBzABhhdodHRwOi8vb2NzcC5zZWN0
# aWdvLmNvbTANBgkqhkiG9w0BAQwFAAOCAgEABv+C4XdjNm57oRUgmxP/BP6YdURh
# w1aVcdGRP4Wh60BAscjW4HL9hcpkOTz5jUug2oeunbYAowbFC2AKK+cMcXIBD0Zd
# OaWTsyNyBBsMLHqafvIhrCymlaS98+QpoBCyKppP0OcxYEdU0hpsaqBBIZOtBajj
# cw5+w/KeFvPYfLF/ldYpmlG+vd0xqlqd099iChnyIMvY5HexjO2AmtsbpVn0OhNc
# WbWDRF/3sBp6fWXhz7DcML4iTAWS+MVXeNLj1lJziVKEoroGs9Mlizg0bUMbOalO
# hOfCipnx8CaLZeVme5yELg09Jlo8BMe80jO37PU8ejfkP9/uPak7VLwELKxAMcJs
# zkyeiaerlphwoKx1uHRzNyE6bxuSKcutisqmKL5OTunAvtONEoteSiabkPVSZ2z7
# 6mKnzAfZxCl/3dq3dUNw4rg3sTCggkHSRqTqlLMS7gjrhTqBmzu1L90Y1KWN/Y5J
# KdGvspbOrTfOXyXvmPL6E52z1NZJ6ctuMFBQZH3pwWvqURR8AgQdULUvrxjUYbHH
# j95Ejza63zdrEcxWLDX6xWls/GDnVNueKjWUH3fTv1Y8Wdho698YADR7TNx8X8z2
# Bev6SivBBOHY+uqiirZtg0y9ShQoPzmCcn63Syatatvx157YK9hlcPmVoa1oDE5/
# L9Uo2bC5a4CH2RwwggZGMIIErqADAgECAhEAym8XT/g/27pUS8j3ZAgpGTANBgkq
# hkiG9w0BAQwFADBUMQswCQYDVQQGEwJHQjEYMBYGA1UEChMPU2VjdGlnbyBMaW1p
# dGVkMSswKQYDVQQDEyJTZWN0aWdvIFB1YmxpYyBDb2RlIFNpZ25pbmcgQ0EgUjM2
# MB4XDTIyMDYyOTAwMDAwMFoXDTIzMDYyOTIzNTk1OVowXDELMAkGA1UEBhMCRlIx
# FzAVBgNVBAgMDsOObGUtZGUtRnJhbmNlMRkwFwYDVQQKDBBTeWx2YWluIFJvdWdl
# YXV4MRkwFwYDVQQDDBBTeWx2YWluIFJvdWdlYXV4MIICIjANBgkqhkiG9w0BAQEF
# AAOCAg8AMIICCgKCAgEAtSmbyoi2DpufmuQ6DnGfdPOrcpu0aec0Wyr3t5fNpdO/
# HGgLUm6fRhcxY6RV7f/FbtOCMc2jzxqLYfoxdFS8i3Gu/i2UILihjhLrxECNl2+V
# pA1UV8cu9G/Jdt+5zW4jJtaNQKZqXmP2aeE5TLL/GG81wQTGAoSiCZMfoNd2kbMl
# NEM65yzBdGcBsY+TJYlh2/h+uFJlgbdjObx+MMSSF4sXpjhOZ7eZYOG8w5IZnOMV
# 86QpbcKSrZ2U25WA6kv+/SkWRRUKXz9/YvuBm0Kr46HWmB/F/ePSmisNHvybquBE
# gLuwRfROfVsExoBzB9t2O48Ohdw1Y0cbNGMElO2IndufKYHXo2xowzZgGObaF3P4
# sXYStLxd9tuLTbgJdUG0idLHg3jA/OtG0N7ps9I5haDUGIL5qiB0jC6W9ACIeMGw
# GbRo6B3QtgIL2qj4kHn5ftQ9LtIU3Ks9VlepeglmjxGcUSZUVgHZnhFWQFa7ZV6y
# i5WKXTBzLL5xXuk0uTTRZuk0XVzuJPZyvtXKJZsAI3LPAaaqhYWtBR1o5Cvs34GC
# bWJ9BBRTteuloPL3wsmJ54oF88ZU5XTxN9j7mrH3sZXL7Xqo+6417W+MkAp4+tgT
# RIHWMBPkZQHIDz3HTwFmGBkWp7fKB5kMDWfhEBSkvoGP6rI98Xt6XQYRsdrolmkC
# AwEAAaOCAYkwggGFMB8GA1UdIwQYMBaAFA8qyyCHKLjsb0iuK1SmKaoXpM0MMB0G
# A1UdDgQWBBSRuEwqvjaMe8Lzm2SQLPbtV4m7NjAOBgNVHQ8BAf8EBAMCB4AwDAYD
# VR0TAQH/BAIwADATBgNVHSUEDDAKBggrBgEFBQcDAzBKBgNVHSAEQzBBMDUGDCsG
# AQQBsjEBAgEDAjAlMCMGCCsGAQUFBwIBFhdodHRwczovL3NlY3RpZ28uY29tL0NQ
# UzAIBgZngQwBBAEwSQYDVR0fBEIwQDA+oDygOoY4aHR0cDovL2NybC5zZWN0aWdv
# LmNvbS9TZWN0aWdvUHVibGljQ29kZVNpZ25pbmdDQVIzNi5jcmwweQYIKwYBBQUH
# AQEEbTBrMEQGCCsGAQUFBzAChjhodHRwOi8vY3J0LnNlY3RpZ28uY29tL1NlY3Rp
# Z29QdWJsaWNDb2RlU2lnbmluZ0NBUjM2LmNydDAjBggrBgEFBQcwAYYXaHR0cDov
# L29jc3Auc2VjdGlnby5jb20wDQYJKoZIhvcNAQEMBQADggGBAG+vlDmHRxAQ51JC
# qxxPfCPrG8vCF89eobldW4mfO4D3jihxVAyMfckO1qmqZZKgvv4jJW5iI4QGR83Y
# /FZmKjjIXoPU2XxAcPP/NtZ8VeDTRTGeGSXF2VA+LxuEI7BO5agqDtfWp6SJnZ75
# OxlPGkgBy6MmkokcIUEi/67d46zXJ1Zr8x13Pi8BuDZhkWal6y+uZWzR+FHIQU4a
# e+ISqRdRqd9E0e3KCZ23WrdAt1FC2Gw4aL68fqEjFvhl1kuRJqSm+JMR1u70YwFb
# ebo8+uNmCqe1KTVgwb7aG23ulh7v5RVVf+jSDpYU0dBxTLNMgcksl0iEoNzrQern
# jRQfotNm/mCsWsWIyhZHWUcxvoNTgEg7FmAk6lhufKJ9vqfQyT7fabAYznb4X86U
# 53LZg4I3sF/01orRnb4lkSXbHsWVPcphvUZuaECIecMcXS1OKYhTigvGyiTCSr8E
# Y1jc56KqvWCeCyl+2k6I2wDu/JGTFg1KWzrUo3NRv9UCAeYTkTCCBuwwggTUoAMC
# AQICEDAPb6zdZph0fKlGNqd4LbkwDQYJKoZIhvcNAQEMBQAwgYgxCzAJBgNVBAYT
# AlVTMRMwEQYDVQQIEwpOZXcgSmVyc2V5MRQwEgYDVQQHEwtKZXJzZXkgQ2l0eTEe
# MBwGA1UEChMVVGhlIFVTRVJUUlVTVCBOZXR3b3JrMS4wLAYDVQQDEyVVU0VSVHJ1
# c3QgUlNBIENlcnRpZmljYXRpb24gQXV0aG9yaXR5MB4XDTE5MDUwMjAwMDAwMFoX
# DTM4MDExODIzNTk1OVowfTELMAkGA1UEBhMCR0IxGzAZBgNVBAgTEkdyZWF0ZXIg
# TWFuY2hlc3RlcjEQMA4GA1UEBxMHU2FsZm9yZDEYMBYGA1UEChMPU2VjdGlnbyBM
# aW1pdGVkMSUwIwYDVQQDExxTZWN0aWdvIFJTQSBUaW1lIFN0YW1waW5nIENBMIIC
# IjANBgkqhkiG9w0BAQEFAAOCAg8AMIICCgKCAgEAyBsBr9ksfoiZfQGYPyCQvZyA
# IVSTuc+gPlPvs1rAdtYaBKXOR4O168TMSTTL80VlufmnZBYmCfvVMlJ5LsljwhOb
# toY/AQWSZm8hq9VxEHmH9EYqzcRaydvXXUlNclYP3MnjU5g6Kh78zlhJ07/zObu5
# pCNCrNAVw3+eolzXOPEWsnDTo8Tfs8VyrC4Kd/wNlFK3/B+VcyQ9ASi8Dw1Ps5EB
# jm6dJ3VV0Rc7NCF7lwGUr3+Az9ERCleEyX9W4L1GnIK+lJ2/tCCwYH64TfUNP9vQ
# 6oWMilZx0S2UTMiMPNMUopy9Jv/TUyDHYGmbWApU9AXn/TGs+ciFF8e4KRmkKS9G
# 493bkV+fPzY+DjBnK0a3Na+WvtpMYMyou58NFNQYxDCYdIIhz2JWtSFzEh79qsoI
# WId3pBXrGVX/0DlULSbuRRo6b83XhPDX8CjFT2SDAtT74t7xvAIo9G3aJ4oG0paH
# 3uhrDvBbfel2aZMgHEqXLHcZK5OVmJyXnuuOwXhWxkQl3wYSmgYtnwNe/YOiU2fK
# sfqNoWTJiJJZy6hGwMnypv99V9sSdvqKQSTUG/xypRSi1K1DHKRJi0E5FAMeKfob
# pSKupcNNgtCN2mu32/cYQFdz8HGj+0p9RTbB942C+rnJDVOAffq2OVgy728YUInX
# T50zvRq1naHelUF6p4MCAwEAAaOCAVowggFWMB8GA1UdIwQYMBaAFFN5v1qqK0rP
# VIDh2JvAnfKyA2bLMB0GA1UdDgQWBBQaofhhGSAPw0F3RSiO0TVfBhIEVTAOBgNV
# HQ8BAf8EBAMCAYYwEgYDVR0TAQH/BAgwBgEB/wIBADATBgNVHSUEDDAKBggrBgEF
# BQcDCDARBgNVHSAECjAIMAYGBFUdIAAwUAYDVR0fBEkwRzBFoEOgQYY/aHR0cDov
# L2NybC51c2VydHJ1c3QuY29tL1VTRVJUcnVzdFJTQUNlcnRpZmljYXRpb25BdXRo
# b3JpdHkuY3JsMHYGCCsGAQUFBwEBBGowaDA/BggrBgEFBQcwAoYzaHR0cDovL2Ny
# dC51c2VydHJ1c3QuY29tL1VTRVJUcnVzdFJTQUFkZFRydXN0Q0EuY3J0MCUGCCsG
# AQUFBzABhhlodHRwOi8vb2NzcC51c2VydHJ1c3QuY29tMA0GCSqGSIb3DQEBDAUA
# A4ICAQBtVIGlM10W4bVTgZF13wN6MgstJYQRsrDbKn0qBfW8Oyf0WqC5SVmQKWxh
# y7VQ2+J9+Z8A70DDrdPi5Fb5WEHP8ULlEH3/sHQfj8ZcCfkzXuqgHCZYXPO0EQ/V
# 1cPivNVYeL9IduFEZ22PsEMQD43k+ThivxMBxYWjTMXMslMwlaTW9JZWCLjNXH8B
# lr5yUmo7Qjd8Fng5k5OUm7Hcsm1BbWfNyW+QPX9FcsEbI9bCVYRm5LPFZgb289ZL
# Xq2jK0KKIZL+qG9aJXBigXNjXqC72NzXStM9r4MGOBIdJIct5PwC1j53BLwENrXn
# d8ucLo0jGLmjwkcd8F3WoXNXBWiap8k3ZR2+6rzYQoNDBaWLpgn/0aGUpk6qPQn1
# BWy30mRa2Coiwkud8TleTN5IPZs0lpoJX47997FSkc4/ifYcobWpdR9xv1tDXWU9
# UIFuq/DQ0/yysx+2mZYm9Dx5i1xkzM3uJ5rloMAMcofBbk1a0x7q8ETmMm8c6xdO
# lMN4ZSA7D0GqH+mhQZ3+sbigZSo04N6o+TzmwTC7wKBjLPxcFgCo0MR/6hGdHgbG
# pm0yXbQ4CStJB6r97DDa8acvz7f9+tCjhNknnvsBZne5VhDhIG7GrrH5trrINV0z
# do7xfCAMKneutaIChrop7rRaALGMq+P5CslUXdS5anSevUiumDCCBvYwggTeoAMC
# AQICEQCQOX+a0ko6E/K9kV8IOKlDMA0GCSqGSIb3DQEBDAUAMH0xCzAJBgNVBAYT
# AkdCMRswGQYDVQQIExJHcmVhdGVyIE1hbmNoZXN0ZXIxEDAOBgNVBAcTB1NhbGZv
# cmQxGDAWBgNVBAoTD1NlY3RpZ28gTGltaXRlZDElMCMGA1UEAxMcU2VjdGlnbyBS
# U0EgVGltZSBTdGFtcGluZyBDQTAeFw0yMjA1MTEwMDAwMDBaFw0zMzA4MTAyMzU5
# NTlaMGoxCzAJBgNVBAYTAkdCMRMwEQYDVQQIEwpNYW5jaGVzdGVyMRgwFgYDVQQK
# Ew9TZWN0aWdvIExpbWl0ZWQxLDAqBgNVBAMMI1NlY3RpZ28gUlNBIFRpbWUgU3Rh
# bXBpbmcgU2lnbmVyICMzMIICIjANBgkqhkiG9w0BAQEFAAOCAg8AMIICCgKCAgEA
# kLJxP3nh1LmKF8zDl8KQlHLtWjpvAUN/c1oonyR8oDVABvqUrwqhg7YT5EsVBl5q
# iiA0cXu7Ja0/WwqkHy9sfS5hUdCMWTc+pl3xHl2AttgfYOPNEmqIH8b+GMuTQ1Z6
# x84D1gBkKFYisUsZ0vCWyUQfOV2csJbtWkmNfnLkQ2t/yaA/bEqt1QBPvQq4g8W9
# mCwHdgFwRd7D8EJp6v8mzANEHxYo4Wp0tpxF+rY6zpTRH72MZar9/MM86A2cOGbV
# /H0em1mMkVpCV1VQFg1LdHLuoCox/CYCNPlkG1n94zrU6LhBKXQBPw3gE3crETz7
# Pc3Q5+GXW1X3KgNt1c1i2s6cHvzqcH3mfUtozlopYdOgXCWzpSdoo1j99S1ryl9k
# x2soDNqseEHeku8Pxeyr3y1vGlRRbDOzjVlg59/oFyKjeUFiz/x785LaruA8Tw9a
# zG7fH7wir7c4EJo0pwv//h1epPPuFjgrP6x2lEGdZB36gP0A4f74OtTDXrtpTXKZ
# 5fEyLVH6Ya1N6iaObfypSJg+8kYNabG3bvQF20EFxhjAUOT4rf6sY2FHkbxGtUZT
# bMX04YYnk4Q5bHXgHQx6WYsuy/RkLEJH9FRYhTflx2mn0iWLlr/GreC9sTf3H99C
# e6rrHOnrPVrd+NKQ1UmaOh2DGld/HAHCzhx9zPuWFcUCAwEAAaOCAYIwggF+MB8G
# A1UdIwQYMBaAFBqh+GEZIA/DQXdFKI7RNV8GEgRVMB0GA1UdDgQWBBQlLmg8a5or
# JBSpH6LfJjrPFKbx4DAOBgNVHQ8BAf8EBAMCBsAwDAYDVR0TAQH/BAIwADAWBgNV
# HSUBAf8EDDAKBggrBgEFBQcDCDBKBgNVHSAEQzBBMDUGDCsGAQQBsjEBAgEDCDAl
# MCMGCCsGAQUFBwIBFhdodHRwczovL3NlY3RpZ28uY29tL0NQUzAIBgZngQwBBAIw
# RAYDVR0fBD0wOzA5oDegNYYzaHR0cDovL2NybC5zZWN0aWdvLmNvbS9TZWN0aWdv
# UlNBVGltZVN0YW1waW5nQ0EuY3JsMHQGCCsGAQUFBwEBBGgwZjA/BggrBgEFBQcw
# AoYzaHR0cDovL2NydC5zZWN0aWdvLmNvbS9TZWN0aWdvUlNBVGltZVN0YW1waW5n
# Q0EuY3J0MCMGCCsGAQUFBzABhhdodHRwOi8vb2NzcC5zZWN0aWdvLmNvbTANBgkq
# hkiG9w0BAQwFAAOCAgEAc9rtaHLLwrlAoTG7tAOjLRR7JOe0WxV9qOn9rdGSDXw9
# NqBp2fOaMNqsadZ0VyQ/fg882fXDeSVsJuiNaJPO8XeJOX+oBAXaNMMU6p8IVKv/
# xH6WbCvTlOu0bOBFTSyy9zs7WrXB+9eJdW2YcnL29wco89Oy0OsZvhUseO/NRaAA
# 5PgEdrtXxZC+d1SQdJ4LT03EqhOPl68BNSvLmxF46fL5iQQ8TuOCEmLrtEQMdUHC
# DzS4iJ3IIvETatsYL254rcQFtOiECJMH+X2D/miYNOR35bHOjJRs2wNtKAVHfpsu
# 8GT726QDMRB8Gvs8GYDRC3C5VV9HvjlkzrfaI1Qy40ayMtjSKYbJFV2Ala8C+7TR
# Lp04fDXgDxztG0dInCJqVYLZ8roIZQPl8SnzSIoJAUymefKithqZlOuXKOG+fRuh
# fO1WgKb0IjOQ5IRT/Cr6wKeXqOq1jXrO5OBLoTOrC3ag1WkWt45mv1/6H8Sof6eh
# SBSRDYL8vU2Z7cnmbDb+d0OZuGktfGEv7aOwSf5bvmkkkf+T/FdpkkvZBT9thnLT
# otDAZNI6QsEaA/vQ7ZohuD+vprJRVNVMxcofEo1XxjntXP/snyZ2rWRmZ+iqMODS
# rbd9sWpBJ24DiqN04IoJgm6/4/a3vJ4LKRhogaGcP24WWUsUCQma5q6/YBXdhvUx
# ggZrMIIGZwIBATBpMFQxCzAJBgNVBAYTAkdCMRgwFgYDVQQKEw9TZWN0aWdvIExp
# bWl0ZWQxKzApBgNVBAMTIlNlY3RpZ28gUHVibGljIENvZGUgU2lnbmluZyBDQSBS
# MzYCEQDKbxdP+D/bulRLyPdkCCkZMA0GCWCGSAFlAwQCAQUAoIGEMBgGCisGAQQB
# gjcCAQwxCjAIoAKAAKECgAAwGQYJKoZIhvcNAQkDMQwGCisGAQQBgjcCAQQwHAYK
# KwYBBAGCNwIBCzEOMAwGCisGAQQBgjcCARUwLwYJKoZIhvcNAQkEMSIEIKzsfg8j
# QR5gdRdHZwRE0V2YISCnM0Net+UR48zcuXdUMA0GCSqGSIb3DQEBAQUABIICAIHu
# f1NK6ytmg5wP0LCaw1DGG5uZXJLfsm3XE+7sBRYhMTQZArCgd2TvZ+yxbKFeEeQu
# tBM/AGDO5CyD45G4cCmA65RghgufBhDsl8KnPBXap4ktyNQ59Y+weBkR6iYEiq5y
# 9DBX2gGH5zM5/hfkfB/ymxuQjNjMr91A7TKVsPg4YICwHKYSWM4PiNyJ9d+NxQ06
# a30K/FGCIczT07JSHY9MOC3uFmz0GpiKmDDQwwF6DcSAdRV0aluPq6g+2mi2mRDK
# pq/LNvD8dHBsQkUcCa82YgYs4oowwGGZfUyn5xIsrB/nQ90PUCNDj+HoKUEpHEQq
# jHvVHzsj8vExoUXX0zxn6GK8i/LKGTDQr2CMuaUBzNT+ucgDtMZDsOwN6V3/seqY
# sMlPnHuc7MLN0ENPMfWMzEzyePQ4RaHfdSrILXIerUKFpIkOT7R5rjgrCU2VSRxN
# ItMOlnVpu2VEVKVahUyT6tKA586b1tl7gfAOvfqDFRCFFWh1gct8PYEh5QYs/GdZ
# FUYqthDMSJKvWWu5CFV7DiSh9KzLXxdDB2WOHds2uJcDY0zg/XQj0kQlhzs4eBs9
# VnxsMZnHN2Vumxz5BJOhgC+rJ7KgI9yFLrsG8KQg2dwnJbYFHxxVybJ8MnFVgn3/
# ogFFoNxy5E+11K9Pt5Li2+FqC6tdVYnwVKI69xyIoYIDTDCCA0gGCSqGSIb3DQEJ
# BjGCAzkwggM1AgEBMIGSMH0xCzAJBgNVBAYTAkdCMRswGQYDVQQIExJHcmVhdGVy
# IE1hbmNoZXN0ZXIxEDAOBgNVBAcTB1NhbGZvcmQxGDAWBgNVBAoTD1NlY3RpZ28g
# TGltaXRlZDElMCMGA1UEAxMcU2VjdGlnbyBSU0EgVGltZSBTdGFtcGluZyBDQQIR
# AJA5f5rSSjoT8r2RXwg4qUMwDQYJYIZIAWUDBAICBQCgeTAYBgkqhkiG9w0BCQMx
# CwYJKoZIhvcNAQcBMBwGCSqGSIb3DQEJBTEPFw0yMjExMjMwODE5MTNaMD8GCSqG
# SIb3DQEJBDEyBDAqCIiMkaaOP3yrZKfwUmwX2y+6z3odbV8DjV4IeNhSyLx7jksG
# HP42exAJOv2t33cwDQYJKoZIhvcNAQEBBQAEggIAUsSgDueVRM3f3fMLrR1DLX44
# FOIowoCIXXK6OTWpA4oF4O77BoT8XwZ2727kea6w8QBDZWgeRr/qk/f09Vq0HGbw
# jUFru9k/Ur2xQXpANSOldgYVs3a5TZ6tG2LkEya3vr1fdCX7BZsPxZ9/ggghBzIm
# SvpMIVXbTZgKb8L4lwsFWnQWDBTXyPDQjezJhgVsOROFNYzumjwbUqCyck6KJ5eB
# LumyUoDTAExoXilMPlMG6QmPJTgL7c+vNM1lZsK3t0EyvqO/Esi/z0eoXC172rKB
# otsQlKgZGpP8au3bQMHQNsYdSiwP7zpB61hnOqVm4kfSN2BKZ3uq+ZlhwZz9KR8x
# mRX2SR2eag68Ejt+3yJqV3rbiOhF7+BmZVfSM4BLwf9ON1nmu9ck4yBNVo7Iho2r
# UrXbrW5wiJqI+QZjKj02p12ObmTmysQwtkitsJCb0QlCTpz9HbDD6d6GVCLs+Kt6
# MYOOxP3myoSR01eflq5ehVwQGRkKuesx6SgzQwHja83xjOW239eau+RT3jGeSSFt
# bAiqW0GcrEB2JD8jdyG7PBGmuFtGW2L/ZmmHgbU3gg/0e3SO1d7IRKcyEfq7I6KO
# fDfH/tSKvDOZVPNUzSswFQV3GCciZdbz57BzkykUdjMssP/n3eN0sXcDU1kpE5xC
# WvWPPwjiLxhdGEszMsI=
# SIG # End signature block
