--[[

TidyShop: Buy Used Equipment - Adds the option to buy vehicles & implements as used/second hand

Author:     w33zl / WZL Modding
Version:    1.1.0
Modified:   2020-11-22

Changelog:
v1.1.0      Rewritten core logic
v1.0.5.1    Hotfix: Fix for specific equipment like Magsi TH fork
v1.0.5      Hotfix: Error when buying pallets
v1.0.4      Hotfix: fixed issue with workshop/customize
v1.0.3      Hotfix: fixed issue with repair/fuel & removing mod
v1.0.2      Hotfix: fixed issue with buying in MP
v1.0.1      Hotfix: fixed issue with hand tools
v1.0        Initial public release
v0.9.2      Tweaked both vanilla & seasons
v0.9.1      Seasons support
v0.9        Private beta

]]

UsedEquipment = Mod:init()

-- UsedEquipment:trySource("UEDebug.lua", true) -- Conditional load of debug script (will only run locally)

-- UsedEquipment.debugMode = true

local CONFIGTYPE_NAME = "vehicleBuyCondition"
local CONFIG_NEW_NAME = "new"
local CONFIG_USED_NAME = "used"

local CONDITION_TIERS = {
    [1] = { discount = 0,       opTimeFactor = 0,       opTimeFactorTool = 0,       wearFactor = 0,     wearFactorTool = 0,     age = 0,    ageTool = 0 }, -- NEW
    [2] = { discount = 0.32,    opTimeFactor = 0.2,     opTimeFactorTool = 0.061,   wearFactor = 0.7,   wearFactorTool = 0.7,   age = 2,    ageTool = 3 }, -- USED T1
}

function UsedEquipment:calculateDiscount(originalPrice, tier)
    local currentTier = tier or CONDITION_TIERS[2]
    local discount = 0

    if self.isSeasons then
        discount = originalPrice * currentTier.discount
    else
        discount = originalPrice * 0.2
    end

    return math.floor(discount)
end

local function addConfigOption(name, price, index, isDefault)
    return {
        -- name = g_i18n:getText("configuration_" .. name),
        name = name,
        index = index or 1,
        isDefault = isDefault or false,
        price = price or 0,
        dailyUpkeep = 0,
        desc = ""
    }
end

function UsedEquipment:canBeBoughtUsed(storeItem)

    if storeItem == nil then return false end


    if storeItem.canBeBoughtUsed == nil then
        storeItem.canBeBoughtUsed = false

        if storeItem.showInStore == false then
            return storeItem.canBeBoughtUsed -- is false
        end

        local species = storeItem.species
        
        if species ~= nil and type(species) == "string" then
            species = species:upper()
        end

        if species == "PLACEABLE" or species == "HANDTOOL" then 
            return storeItem.canBeBoughtUsed -- is false
        end

        TidyShopHelper:extendShopItem(storeItem)

        storeItem.canBeBoughtUsed = storeItem.canBeBoughtUsed or storeItem:query():isCategoryType(StoreManager.CATEGORY_TYPE.VEHICLE):matches()
        storeItem.canBeBoughtUsed = storeItem.canBeBoughtUsed or storeItem:query():isCategoryType(StoreManager.CATEGORY_TYPE.TOOL):matches()
    end

    return storeItem.canBeBoughtUsed
end

function UsedEquipment:loadMap()
    self.isServer = g_currentMission:getIsServer() == true
    self.isClient = g_currentMission:getIsClient() == true
    self.isMultiplayer = g_currentMission.missionDynamicInfo ~=nil and g_currentMission.missionDynamicInfo.isMultiplayer
    self.isHostMP = self.isMultiplayer and self.isServer and self.isClient
    self.isDediServerMP = self.isMultiplayer and self.isServer and not self.isClient
    self.isPlayerMP = self.isMultiplayer and not self.isServer and self.isClient
    self.isSeasons = (g_seasons or getfenv(0)["g_seasons"]) ~= nil
    self.isMapReady = false


    for key, newStoreItem in pairs(g_storeManager:getItems()) do
        
        if newStoreItem.canBeBoughtUsed and newStoreItem.configurations ~= nil and newStoreItem.configurations[CONFIGTYPE_NAME] ~= nil then
            local discount = UsedEquipment:calculateDiscount(newStoreItem.price)

            -- newStoreItem.configurations[CONFIGTYPE_NAME][2].price = 0
            -- newStoreItem.configurations[CONFIGTYPE_NAME][1].price = discount
            -- newStoreItem.price = newStoreItem.price - discount

            newStoreItem.configurations[CONFIGTYPE_NAME][2].price = -discount
            newStoreItem.configurations[CONFIGTYPE_NAME][1].price = 0
            

        end
    end

    self.isStoreItemsReady = true
end

FSBaseMission.onStartMission = Utils.prependedFunction(FSBaseMission.onStartMission, function(...)
    if UsedEquipment.onStartMission ~= nil then
        UsedEquipment:onStartMission(...)
    end

    UsedEquipment.isMapReady = true
end);

ShopConfigScreen.setStoreItem = Utils.overwrittenFunction(ShopConfigScreen.setStoreItem, function(self, superFunc, storeItem, vehicle, ...)
    UsedEquipment:printDebug("SetStoreItem")
    if storeItem ~= nil and storeItem.configurations ~= nil and UsedEquipment:canBeBoughtUsed(storeItem) then
        if vehicle == nil then -- I.e. buying (not using workshop)
            storeItem.configurations["vehicleBuyCondition"][1].name = UsedEquipment.l10n.texts.NEW
        else
            -- vehicle.configurations[CONFIGTYPE_NAME] = 1
            -- storeItem.configurations["vehicleBuyCondition"] = nil
            -- storeItem.configurations["vehicleBuyCondition"] = storeItem.configurations["vehicleBuyCondition"] or {}
            storeItem.configurations["vehicleBuyCondition"][1].name = UsedEquipment.l10n.texts.USED
        end
    end

    return superFunc(self, storeItem, vehicle, ...)
end)

-- addVehicle event (override hook)
FSBaseMission.addVehicle = Utils.overwrittenFunction(FSBaseMission.addVehicle, function(currentMission, superFunc, newVehicle)

    if newVehicle ~= nil and newVehicle.configurations ~= nil then

        local isBuying = g_currentMission.shopController.isBuying == true
        local boughtConfigurations = g_currentMission.shopController.buyItemConfigurations  
        local hasSelectedUsedCondition = boughtConfigurations ~= nil and boughtConfigurations[CONFIGTYPE_NAME] ~= nil and (boughtConfigurations[CONFIGTYPE_NAME] > 1)
        local isBuyingUsedItem = isBuying and hasSelectedUsedCondition
        local vehicleConditionIndex = newVehicle.configurations[CONFIGTYPE_NAME] or 1

        -- Only update vehicle stats if not during initial load - then only update when client is buying used or 
        local shouldVehicleBeUpdated = UsedEquipment.isMapReady and ( (isBuying and hasSelectedUsedCondition) or (vehicleConditionIndex > 1) )
        -- local shouldVehicleBeUpdated = isBuyingUsedItem and hasSelectedUsedCondition and not isVehicleAlreadyAdded

        newVehicle.configurations[CONFIGTYPE_NAME] = 1 -- reset flag - all vehicles should be "used"

        UsedEquipment:printDebugVar("hasSelectedUsedContidion", hasSelectedUsedCondition)
        UsedEquipment:printDebugVar("boughtConfigurations", boughtConfigurations)
        UsedEquipment:printDebugVar("buyConditionIndex", vehicleConditionIndex)  
        UsedEquipment:printDebugVar("isBuyingUsedItem", isBuyingUsedItem)
        UsedEquipment:printDebugVar("shouldVehicleBeUpdated", shouldVehicleBeUpdated)

        if shouldVehicleBeUpdated and newVehicle.operatingTime == 0 then -- Only update vehicles with the updates flag AND that is not already used (for whatever reason)
            local storeItem = g_storeManager:getItemByXMLFilename(newVehicle.configFileName)
            local lifetime = storeItem.lifetime or 600
            local optime = 18720000
            local currentTier = CONDITION_TIERS[2]
            local isMotorized = newVehicle.spec_motorized ~= nil          
            local wearFactor = (isMotorized and currentTier.wearFactor) or currentTier.wearFactorTool
            local isSeasons =  UsedEquipment.isSeasons -- (newVehicle["spec_FS19_RM_Seasons.seasonsVehicle"] ~= nil)
            local age = (isMotorized == false and isSeasons == true and currentTier.ageTool) or currentTier.age
            local opTimeFactor = (isMotorized == false and isSeasons == true and currentTier.opTimeFactorTool) or currentTier.opTimeFactor
            local dirtFactor = (isSeasons and 0.5) or 0.65

            local discount = storeItem.configurations[CONFIGTYPE_NAME][2].price -- Get the actual discount for this specific item bought

            UsedEquipment:printDebugVar("isSeasons", isSeasons)

            if isSeasons == true then
                local seasonsVehicle = newVehicle["spec_FS19_RM_Seasons.seasonsVehicle"]

                UsedEquipment:printDebug("Seasons vehicle specialization found")

                if seasonsVehicle ~= nil then
                    seasonsVehicle.years = age
                    local x = lifetime * opTimeFactor * 3600

                    optime = x * 1000

                    seasonsVehicle.nextRepair = x + (108000 * (1 - wearFactor))
                end

                newVehicle.age = (g_seasons.environment.daysPerSeason or 9) * age
            else

                newVehicle.age = age * 6

                if newVehicle.spec_wearable ~= nil then
                    UsedEquipment:printDebug("Setting wear levels")
    
                    local spec = newVehicle.spec_wearable
                    for _, data in ipairs(spec.wearableNodes) do
                        newVehicle:setNodeWearAmount(data, 0.89, true)
                    end
                    newVehicle:raiseDirtyFlags(spec.dirtyFlag)
    
                end
            end

            newVehicle.price = newVehicle.price + math.abs(  discount )

            if newVehicle.setOperatingTime ~= nil then
                newVehicle:setOperatingTime(optime, true)
            end

            if newVehicle.addDirtAmount ~= nil then
                newVehicle:addDirtAmount(dirtFactor)
                -- newVehicle:addDirtAmount(0.45)
            end


            if newVehicle.spec_fillUnit ~= nil then
                local fillUnits = newVehicle.spec_fillUnit.fillUnits

                --TODO: add DEF/ADBLUE also
                if fillUnits ~= nil then
                    for _, fillUnit in ipairs(fillUnits) do
                        if fillUnit.fillType == g_fillTypeManager.nameToIndex["DIESEL"] then
                            fillUnit.fillLevel = fillUnit.capacity * 0.37
                        end
                    end
                end
            end
        end

        
    end

    return superFunc(currentMission, newVehicle)

end)

Vehicle.saveToXMLFile = Utils.overwrittenFunction(Vehicle.saveToXMLFile, function(self, superFunc, ...)
    local success = pcall(function() 
        if self ~= nil and self.configurations ~= nil and self.configurations["vehicleBuyCondition"] ~= nil then
            self.configurations["vehicleBuyCondition"] = nil -- Reset condition flag before save - this is required to be able to remove/disable the mod
        end
    end)

    if not success then
        UsedEquipment:printWarning("Failed to reset used condition flag for vehicle!")
    end

    return superFunc(self, ...)
end)

-- Add new config type
g_configurationManager:addConfigurationType(CONFIGTYPE_NAME, g_i18n:getText("configuration_title"), nil, nil, nil, nil, ConfigurationUtil.SELECTOR_MULTIOPTION)

UsedEquipment.l10n = 
{
    texts = 
    {
        NEW = g_i18n:getText("configuration_" .. CONFIG_NEW_NAME),
        USED = g_i18n:getText("configuration_" .. CONFIG_USED_NAME),
    }
}


-- Injecty new config type to all machines via getConfigurationsFromXML event
StoreItemUtil.getConfigurationsFromXML = Utils.overwrittenFunction(StoreItemUtil.getConfigurationsFromXML, function(xmlFile, superFunc, baseXMLName, baseDir, customEnvironment, isMod, storeItem)
	local configurations = superFunc(xmlFile, baseXMLName, baseDir, customEnvironment, isMod, storeItem)

    if UsedEquipment:canBeBoughtUsed(storeItem) then
        configurations = configurations or {}

        configurations[CONFIGTYPE_NAME] = {
            addConfigOption(UsedEquipment.l10n.texts.USED, 0, 1, false),
            addConfigOption(UsedEquipment.l10n.texts.USED, 0, 2, false),
        }
    end

    return configurations
end)


