--[[--------------------------------------------------------------------------------------------------
MAIN
------------------------------------------------------------------------------------------------------
file: ExtendedDriving
author:	Team LTW
start project: 07.04.2019 
------------------------------------------------------------------------------------------------------
Farming Simulator 19
V1.0.0.0 - 22.04.2019
V1.1.0.0 - 16.03.2020	- fix for reverse driving and cruiseControl
------------------------------------------------------------------------------------------------------
copyright (c) Team LTW
----------------------------------------------------------------------------------------------------]]

extendedDriving = {}
extendedDriving.modDir = g_currentModDirectory


function extendedDriving.prerequisitesPresent(specializations)
    return SpecializationUtil.hasSpecialization(Drivable, specializations)
end

--[[--------------------------------------------------------------------------------------------------
Register functions
----------------------------------------------------------------------------------------------------]]
function extendedDriving.registerFunctions(vehicleType)
    SpecializationUtil.registerFunction(vehicleType, "getExtendedDirection", extendedDriving.getExtendedDirection)
	SpecializationUtil.registerFunction(vehicleType, "ToggelDirection", extendedDriving.ToggelDirection)
	SpecializationUtil.registerFunction(vehicleType, "setForwardDirection", extendedDriving.setForwardDirection)
	SpecializationUtil.registerFunction(vehicleType, "setBackwardDirection", extendedDriving.setBackwardDirection)
	SpecializationUtil.registerFunction(vehicleType, "setDirection", extendedDriving.setDirection)
	SpecializationUtil.registerFunction(vehicleType, "setChangeDirection", extendedDriving.setChangeDirection)
	SpecializationUtil.registerFunction(vehicleType, "getChangeDirection", extendedDriving.getChangeDirection)
	SpecializationUtil.registerFunction(vehicleType, "setStopGoBreak", extendedDriving.setStopGoBreak)
	SpecializationUtil.registerFunction(vehicleType, "getStopGoBreak", extendedDriving.getStopGoBreak)
end

--[[--------------------------------------------------------------------------------------------------
Register event listeners
----------------------------------------------------------------------------------------------------]]
function extendedDriving.registerEventListeners(vehicleType)
    SpecializationUtil.registerEventListener(vehicleType, "onLoad", extendedDriving)
	SpecializationUtil.registerEventListener(vehicleType, "onReadStream", extendedDriving)
	SpecializationUtil.registerEventListener(vehicleType, "onWriteStream", extendedDriving)
	SpecializationUtil.registerEventListener(vehicleType, "onReadUpdateStream", extendedDriving)
	SpecializationUtil.registerEventListener(vehicleType, "onWriteUpdateStream", extendedDriving)
	SpecializationUtil.registerEventListener(vehicleType, "onUpdateTick", extendedDriving)
	SpecializationUtil.registerEventListener(vehicleType, "onDraw", extendedDriving)
	SpecializationUtil.registerEventListener(vehicleType, "onLeaveVehicle", extendedDriving)
	SpecializationUtil.registerEventListener(vehicleType, "onRegisterActionEvents", extendedDriving)																								 
end

--[[--------------------------------------------------------------------------------------------------
Load
----------------------------------------------------------------------------------------------------]]
function extendedDriving:onLoad(savegame)
	local spec = self.extendedDriving

	self.extendedDriving = {}
	self.extendedDriving.direction = 0
	self.extendedDriving.directionSend = 0
	self.extendedDriving.blinking = 0
	self.extendedDriving.state = false
	self.extendedDriving.toff = 0
	self.extendedDriving.StopGoBreak = false
	self.extendedDriving.ChangeDirection = false
	
	self.extendedDriving.ToggelText = g_i18n:getText("CHANGE_DIRECTION")
	self.extendedDriving.ForwardText = g_i18n:getText("FORWARD")
	self.extendedDriving.BackwardText = g_i18n:getText("BACKWARD")
	
	self.extendedDriving.arrow_green_for = createImageOverlay(extendedDriving.modDir .. "arrow_green_for.png")
	self.extendedDriving.arrow_green_back = createImageOverlay(extendedDriving.modDir .. "arrow_green_back.png")
	self.extendedDriving.arrow_white_for = createImageOverlay(extendedDriving.modDir .. "arrow_white_for.png")
	self.extendedDriving.arrow_white_back = createImageOverlay(extendedDriving.modDir .. "arrow_white_back.png")
	
	self.extendedDriving.actionEvents = {}
	
	self.extendedDriving.dirtyFlag = self:getNextDirtyFlag()
end;

--[[--------------------------------------------------------------------------------------------------
onReadStream
----------------------------------------------------------------------------------------------------]]
function extendedDriving:onReadStream(streamId, connection)
	local spec = self.extendedDriving
	self:setDirection(streamReadInt8(streamId))
	self:setChangeDirection(streamReadBool(streamId))
	self:setStopGoBreak(streamReadBool(streamId))
end

--[[--------------------------------------------------------------------------------------------------
onWriteStream
----------------------------------------------------------------------------------------------------]]
function extendedDriving:onWriteStream(streamId, connection)
	local spec = self.extendedDriving
	streamWriteInt8(streamId, spec.direction)
	streamWriteBool(streamId, spec.ChangeDirection)
	streamWriteBool(streamId, spec.StopGoBreak)
end

--[[--------------------------------------------------------------------------------------------------
onReadUpdateStream
----------------------------------------------------------------------------------------------------]]
function extendedDriving:onReadUpdateStream(streamId, timestamp, connection)
	local spec = self.extendedDriving
		if streamReadBool(streamId) then
			self:setDirection(streamReadInt8(streamId))
			self:setChangeDirection(streamReadBool(streamId))
			self:setStopGoBreak(streamReadBool(streamId))
    end
end

--[[--------------------------------------------------------------------------------------------------
onWriteUpdateStream
----------------------------------------------------------------------------------------------------]]
function extendedDriving:onWriteUpdateStream(streamId, connection, dirtyMask)
	local spec = self.extendedDriving
		if streamWriteBool(streamId, bitAND(dirtyMask, spec.dirtyFlag) ~= 0) then
			streamWriteInt8(streamId, spec.direction)
			streamWriteBool(streamId, spec.ChangeDirection)
			streamWriteBool(streamId, spec.StopGoBreak)
    end
end

--[[--------------------------------------------------------------------------------------------------
Update
----------------------------------------------------------------------------------------------------]]
function extendedDriving:onUpdateTick(dt)
	local spec = self.extendedDriving
	if self.isClient then
		if spec.directionSend ~= spec.direction then
			self:raiseDirtyFlags(spec.dirtyFlag)
			spec.directionSend = spec.direction
		end
	end
	if spec.direction == 0 then
		spec.blinking = spec.blinking + dt
		if spec.blinking > 500 then
			spec.state = not spec.state
			spec.blinking = 0
		end
	end
	if spec.direction ~= 0 and self.lastSpeedReal < 0.0001 then
		spec.toff = spec.toff + dt
		if spec.toff > 60000 then
			spec.direction = 0
			spec.toff = 0
		end
	else
		spec.toff = 0
	end
	if not self.spec_motorized.isMotorStarted then
		spec.direction = 0
	end
end

--[[--------------------------------------------------------------------------------------------------
Hud overlay
----------------------------------------------------------------------------------------------------]]
function extendedDriving:onDraw()
	if self.isClient and self:getIsActive() then
		local spec = self.extendedDriving
		local uiScale = g_gameSettings:getValue("uiScale")

		local iconWidth = 0.0176 * uiScale
		local iconHeight = iconWidth * g_screenAspectRatio

		local cruiseOverlay = g_currentMission.inGameMenu.hud.speedMeter.overlay;
		local startX_1 = cruiseOverlay.x + cruiseOverlay.width * 0.5
		local startX_2 = startX_1 - iconWidth
		
		local startY = cruiseOverlay.y + cruiseOverlay.height * 0.85
		if not self:getIsAIActive() then
			if spec.direction > 0 and self.spec_motorized.isMotorStarted then
				renderOverlay(spec.arrow_green_for, startX_2, startY, iconWidth, iconHeight)
			end
			if spec.direction < 0 and self.spec_motorized.isMotorStarted then
				renderOverlay(spec.arrow_green_back, startX_1, startY, iconWidth, iconHeight)
			end
			if spec.direction == 0 and spec.state and self.spec_motorized.isMotorStarted then
				renderOverlay(spec.arrow_green_for, startX_2, startY, iconWidth, iconHeight)
				renderOverlay(spec.arrow_green_back, startX_1, startY, iconWidth, iconHeight)
			end
			renderOverlay(spec.arrow_white_for, startX_2, startY, iconWidth, iconHeight)
			renderOverlay(spec.arrow_white_back, startX_1, startY, iconWidth, iconHeight)
		end	
	end
end

--[[--------------------------------------------------------------------------------------------------
Fuctions
----------------------------------------------------------------------------------------------------]]
function extendedDriving:onLeaveVehicle()
    local spec = self.extendedDriving
	spec.direction = 0
end

function extendedDriving:getExtendedDirection()
    local spec = self.extendedDriving
    return spec.direction
end

function extendedDriving:setDirection(dir)
	local spec = self.extendedDriving
	if spec.direction ~= dir then
		spec.direction = dir
		self:setChangeDirection(true)
	end
	return nil
end

function extendedDriving:setChangeDirection(bool)
	local spec = self.extendedDriving
	spec.ChangeDirection = bool
	return nil
end

function extendedDriving:getChangeDirection()
	local spec = self.extendedDriving
	return spec.ChangeDirection
end

function extendedDriving:setStopGoBreak(bool)
	local spec = self.extendedDriving
	spec.StopGoBreak = bool
	return nil
end

function extendedDriving:getStopGoBreak()
	local spec = self.extendedDriving
	return spec.StopGoBreak
end

--[[--------------------------------------------------------------------------------------------------
Set directions
----------------------------------------------------------------------------------------------------]]
function extendedDriving:ToggelDirection()
	local spec = self.extendedDriving
	if spec.direction == 0 then
		self:setDirection(1)
	else
		self:setDirection(-spec.direction)
	end
    return nil
end

function extendedDriving:setForwardDirection()
	self:setDirection(1)
    return nil
end

function extendedDriving:setBackwardDirection()
	self:setDirection(-1)
    return nil
end

--[[--------------------------------------------------------------------------------------------------
Action Events
----------------------------------------------------------------------------------------------------]]
function extendedDriving:onRegisterActionEvents(isActiveForInput, isActiveForInputIgnoreSelection)
    if self.isClient then
        local spec = self.extendedDriving

        self:clearActionEventsTable(spec.actionEvents)
        if isActiveForInputIgnoreSelection then
            local _, actionEventId = self:addActionEvent(spec.actionEvents, InputAction.TOGGLE_DIRECTION_STATE, self, extendedDriving.ToggelDirection, false, true, false, true, nil)
            g_inputBinding:setActionEventTextPriority(actionEventId, GS_PRIO_VERY_LOW)
            g_inputBinding:setActionEventText(actionEventId, spec.ToggelText)
			
			local _, actionEventId = self:addActionEvent(spec.actionEvents, InputAction.FORWARD_DIRECTION, self, extendedDriving.setForwardDirection, false, true, false, true, nil)
            g_inputBinding:setActionEventTextPriority(actionEventId, GS_PRIO_VERY_LOW)
            g_inputBinding:setActionEventText(actionEventId, spec.ForwardText)
			
			local _, actionEventId = self:addActionEvent(spec.actionEvents, InputAction.BACKWARD_DIRECTION, self, extendedDriving.setBackwardDirection, false, true, false, true, nil)
            g_inputBinding:setActionEventTextPriority(actionEventId, GS_PRIO_VERY_LOW)
            g_inputBinding:setActionEventText(actionEventId,spec.BackwardText)
        end
    end
end

--[[--------------------------------------------------------------------------------------------------
overwrite "Drivable:onUpdate"
----------------------------------------------------------------------------------------------------]]
function Drivable:onUpdate(dt, isActiveForInput, isActiveForInputIgnoreSelection, isSelected)
    local spec = self.spec_drivable
    -- update inputs on client side for basic controls
    if self.isClient then
        if self.getIsEntered ~= nil and self:getIsEntered() then
            if self:getIsActiveForInput(true, true) then
                if self:getIsVehicleControlledByPlayer() then
                    spec.doHandbrake = false
                    -- gas and brake pedal
                    local axisForward = MathUtil.clamp((spec.lastInputValues.axisAccelerate - spec.lastInputValues.axisBrake), -1, 1)
                    spec.axisForward = axisForward
					
					-- new code for extendedDriving spec
					if self:getExtendedDirection() < 0 then
						spec.axisForward = -spec.axisForward
					end
					
					if self.lastSpeedReal > 0.0003 and self:getChangeDirection() then
						self:setStopGoBreak(true)
						self:setChangeDirection(false)
					elseif self.lastSpeedReal == 0.0 or self.movingDirection == 0 then
						self:setStopGoBreak(false)
						self:setChangeDirection(false)
					end

					if spec.reverserDirection > 0 then
						if self:getExtendedDirection() > 0 then
							if self.movingDirection > 0 then
								spec.axisForward = MathUtil.clamp(spec.axisForward,-1,1)
							else
								spec.axisForward = MathUtil.clamp(spec.axisForward,0,1)
							end					
						elseif self:getExtendedDirection() < 0 then
							if self.movingDirection < 0 then
								spec.axisForward = MathUtil.clamp(spec.axisForward,-1,1)					
							else
								spec.axisForward = MathUtil.clamp(spec.axisForward,-1,0)
							end	
						else
							spec.axisForward = 0
						end 
					elseif spec.reverserDirection < 0 then
						if self:getExtendedDirection() > 0 then
							if self.movingDirection < 0 then
								spec.axisForward = MathUtil.clamp(spec.axisForward,-1,1)
							else
								spec.axisForward = MathUtil.clamp(spec.axisForward,0,1)
							end					
						elseif self:getExtendedDirection() < 0 then
							if self.movingDirection > 0 then
								spec.axisForward = MathUtil.clamp(spec.axisForward,-1,1)					
							else
								spec.axisForward = MathUtil.clamp(spec.axisForward,-1,0)
							end	
						else
							spec.axisForward = 0
						end 
					end
					
                    -- steering
                    local speedFactor = 1.0
                    local sensitivitySetting = g_gameSettings:getValue(GameSettings.SETTING.STEERING_SENSITIVITY)
                    if spec.lastInputValues.axisSteerIsAnalog then
                        local isArticulatedSteering = self.spec_articulatedAxis ~= nil and self.spec_articulatedAxis.componentJoint ~= nil
                        if isArticulatedSteering then
                            speedFactor = 1.5
                        else
                            speedFactor = 2.5
                        end
                        -- only use steering speed for gamepads
                        if spec.lastInputValues.axisSteerDeviceCategory == InputDevice.CATEGORY.GAMEPAD then
                            speedFactor = speedFactor * sensitivitySetting
                        end
                    else
                        if spec.lastInputValues.axisSteer == 0 then
                            local rotateBackSpeedSetting = g_gameSettings:getValue(GameSettings.SETTING.STEERING_BACK_SPEED) / 10
                            -- if the setting is '1' we use the same speed as we use for steering
                            -- if the setting is smaller '1' we reduce the steering angle depending on the driving speed
                            if rotateBackSpeedSetting < 1 and self.speedDependentRotateBack then
                                local speed = self:getLastSpeed() / 36
                                local setting = rotateBackSpeedSetting / 0.5
                                speedFactor = speedFactor * math.min(speed * setting, 1)
                            end
                            speedFactor = speedFactor * (self.autoRotateBackSpeed or 1) / 1.5
                        else
                            speedFactor = math.min(1.0/(self.lastSpeed*spec.speedRotScale+spec.speedRotScaleOffset), 1);
                            speedFactor = speedFactor * sensitivitySetting
                        end
                    end
                    local steeringDuration = (self.wheelSteeringDuration or 1) * 1000
                    local rotDelta = (dt / steeringDuration) * speedFactor
                    if spec.lastInputValues.axisSteer > spec.axisSide then
                        spec.axisSide = math.min(spec.lastInputValues.axisSteer, spec.axisSide + rotDelta)
                    elseif spec.lastInputValues.axisSteer < spec.axisSide then
                        spec.axisSide = math.max(spec.lastInputValues.axisSteer, spec.axisSide - rotDelta)
                    end
                else
                    spec.axisForward = 0
                    if self.rotatedTime < 0 then
                        spec.axisSide = self.rotatedTime / -self.maxRotTime / self:getSteeringDirection()
                    else
                        spec.axisSide = self.rotatedTime / self.minRotTime / self:getSteeringDirection()
                    end
                end
            else
                spec.doHandbrake = true
                spec.axisForward = 0
            end
            -- prepare for next frame
            spec.lastInputValues.axisAccelerate = 0
            spec.lastInputValues.axisBrake = 0
            spec.lastInputValues.axisSteer = 0
            -- prepare network update
            if spec.axisForward ~= spec.axisForwardSend or spec.axisSide ~= spec.axisSideSend or spec.doHandbrake ~= spec.doHandbrakeSend then
                spec.axisForwardSend = spec.axisForward
                spec.axisSideSend = spec.axisSide
                spec.doHandbrakeSend = spec.doHandbrake
                self:raiseDirtyFlags(spec.dirtyFlag)
            end
        end
    end
    -- update inputs on client side for cruise control
    if self.isClient then
        if self.getIsEntered ~= nil and self:getIsEntered() then
            -- cruise control state
            local inputValue = spec.lastInputValues.cruiseControlState
            spec.lastInputValues.cruiseControlState = 0
            if inputValue == 1 then
                if spec.cruiseControl.topSpeedTime == Drivable.CRUISECONTROL_FULL_TOGGLE_TIME then
                    if spec.cruiseControl.state == Drivable.CRUISECONTROL_STATE_OFF then
                        self:setCruiseControlState(Drivable.CRUISECONTROL_STATE_ACTIVE)
                    else
                        self:setCruiseControlState(Drivable.CRUISECONTROL_STATE_OFF)
                    end
                end
                if spec.cruiseControl.topSpeedTime > 0 then
                    spec.cruiseControl.topSpeedTime = spec.cruiseControl.topSpeedTime - dt
                    if spec.cruiseControl.topSpeedTime < 0 then
                        self:setCruiseControlState(Drivable.CRUISECONTROL_STATE_FULL)
                    end
                end
            else
                spec.cruiseControl.topSpeedTime = Drivable.CRUISECONTROL_FULL_TOGGLE_TIME
            end
            -- cruise control value
            local lastCruiseControlValue = spec.lastInputValues.cruiseControlValue
            spec.lastInputValues.cruiseControlValue = 0
            if lastCruiseControlValue ~= 0 then
                spec.cruiseControl.changeCurrentDelay = spec.cruiseControl.changeCurrentDelay - (dt * spec.cruiseControl.changeMultiplier)
                spec.cruiseControl.changeMultiplier = math.min(spec.cruiseControl.changeMultiplier + (dt * 0.003), 10)
                if spec.cruiseControl.changeCurrentDelay < 0 then
                    spec.cruiseControl.changeCurrentDelay = spec.cruiseControl.changeDelay
                    local dir = MathUtil.sign(lastCruiseControlValue)
                    local speed = spec.cruiseControl.speed + dir
                    self:setCruiseControlMaxSpeed(speed)
                    if spec.cruiseControl.speed ~= spec.cruiseControl.speedSent then
                        if g_server ~= nil then
                            g_server:broadcastEvent(SetCruiseControlSpeedEvent:new(self, spec.cruiseControl.speed), nil, nil, self)
                        else
                            g_client:getServerConnection():sendEvent(SetCruiseControlSpeedEvent:new(self, spec.cruiseControl.speed))
                        end
                        spec.cruiseControl.speedSent = spec.cruiseControl.speed
                    end
                end
            else
                spec.cruiseControl.changeCurrentDelay = 0
                spec.cruiseControl.changeMultiplier = 1
            end
        end
    end
    local isControlled = self.getIsControlled ~= nil and self:getIsControlled()
    -- update vehicle physics on server side
    if self:getIsVehicleControlledByPlayer() then
        if self.isServer then
            if isControlled then
                -- lock max speed to working tool
                local speed,_ = self:getSpeedLimit(true)
                if spec.cruiseControl.state == Drivable.CRUISECONTROL_STATE_ACTIVE then
                    speed = math.min(speed, spec.cruiseControl.speed)
				end
                self:getMotor():setSpeedLimit(speed)
                self:updateVehiclePhysics(spec.axisForward, spec.axisSide, spec.doHandbrake, dt)
            end
        end
    end
    -- just a visual update of the steering wheel
    if self.isClient and isControlled then
        self:updateSteeringWheel(spec.steeringWheel, dt, 1)
    end
end

--[[--------------------------------------------------------------------------------------------------
overwrite "Drivable:updateVehiclePhysics"
----------------------------------------------------------------------------------------------------]]
function Drivable:updateVehiclePhysics(axisForward, axisSide, doHandbrake, dt)
    local spec = self.spec_drivable
	
    axisForward = axisForward
    axisSide = self:getSteeringDirection() * axisSide
	
    local acceleration = 0
	
    if self:getIsMotorStarted() and self:getMotorStartTime() <= g_currentMission.time then
        acceleration = axisForward
        if math.abs(acceleration) > 0 then
            self:setCruiseControlState(Drivable.CRUISECONTROL_STATE_OFF)
        end
        if spec.cruiseControl.state ~= Drivable.CRUISECONTROL_STATE_OFF then
			acceleration = 1.0 * self:getExtendedDirection() -- fix for cruiseControl
        end
    end
    if not self:getCanMotorRun() then
        acceleration = 0
        if self:getIsMotorStarted() then
            self:stopMotor()
        end
    end
	
    -- only update steering if a player is in the vehicle
    if self.getIsControlled ~= nil and self:getIsControlled() then
        local targetRotatedTime = 0
        if self.maxRotTime ~= nil and self.minRotTime ~= nil then
            if axisSide < 0 then
                -- 0 to maxRotTime
                targetRotatedTime = math.min(-self.maxRotTime * axisSide, self.maxRotTime)
            else
                -- 0 to minRotTime
                targetRotatedTime = math.max(self.minRotTime * axisSide, self.minRotTime)
            end
        end
        self.rotatedTime = targetRotatedTime
    end
	
    if self.firstTimeRun then
        if self.spec_wheels ~= nil then
            WheelsUtil.updateWheelsPhysics(self, dt, self.lastSpeedReal*self.movingDirection, acceleration, doHandbrake, self:getStopGoBreak()) -- g_currentMission.missionInfo.stopAndGoBraking)
        end
    end
	
    return acceleration
end

