/*=============================================================================

    Copyright (c) Pascal Gilcher. All rights reserved.

 * Unauthorized copying of this file, via any medium is strictly prohibited
 * Proprietary and confidential

=============================================================================*/

#pragma once

/*===========================================================================*/

namespace Colorspace
{

float3 rgb_to_hcv(in float3 RGB)
{
    RGB = saturate(RGB);
    float Epsilon = 1e-10;
        // Based on work by Sam Hocevar and Emil Persson
    float4 P = (RGB.g < RGB.b) ? float4(RGB.bg, -1.0, 2.0/3.0) : float4(RGB.gb, 0.0, -1.0/3.0);
    float4 Q = (RGB.r < P.x) ? float4(P.xyw, RGB.r) : float4(RGB.r, P.yzx);
    float C = Q.x - min(Q.w, Q.y);
    float H = abs((Q.w - Q.y) / (6 * C + Epsilon) + Q.z);
    return float3(H, C, Q.x);
}

float3 rgb_to_hsl(in float3 RGB)
{
    float3 HCV = rgb_to_hcv(RGB);
    float L = HCV.z - HCV.y * 0.5;
    float S = HCV.y / (1.0000001 - abs(L * 2 - 1));
    return float3(HCV.x, S, L);
}

float3 hsl_to_rgb(in float3 HSL)
{
    HSL = saturate(HSL);
    float3 RGB = saturate(float3(abs(HSL.x * 6.0 - 3.0) - 1.0,2.0 - abs(HSL.x * 6.0 - 2.0),2.0 - abs(HSL.x * 6.0 - 4.0)));
    float C = (1 - abs(2 * HSL.z - 1)) * HSL.y;
    return (RGB - 0.5) * C + HSL.z;
}

float3 rgb_to_xyz(float3 RGB)
{
    static const float3x3 m =   float3x3( 0.4124564, 0.3575761, 0.1804375,        
                                          0.2126729, 0.7151522, 0.0721750,
                                          0.0193339, 0.1191920, 0.9503041);
    return mul(m, RGB);
}


float3 xyz_to_rgb(float3 XYZ)
{
    static const float3x3 m = float3x3(  3.2404542, -1.5371385, -0.4985314,
                                        -0.9692660,  1.8760108,  0.0415560,
                                         0.0556434, -0.2040259,  1.0572252);
    return mul(m, XYZ);
}

float3 xyz_to_cielab(float3 xyz)
{
    xyz *= float3(1.05211, 1.0, 0.91842); // reciprocal of °2 D65 reference values
    xyz = xyz > 0.008856 ? pow(xyz, 1.0/3.0) : xyz * 7.787037 + 4.0/29.0;
    float L = (116.0 * xyz.y) - 16.0;
    float a = 500.0 * (xyz.x - xyz.y);
    float b = 200.0 * (xyz.y - xyz.z);
    return float3(L, a, b);
}

float3 cielab_to_xyz(float3 lab)
{
    float3 xyz;
    xyz.y = (lab.x + 16.0) / 116.0;
    xyz.x = xyz.y + lab.y / 500.0;
    xyz.z = xyz.y - lab.z / 200.0;
    xyz = xyz > 0.206897 ? xyz * xyz * xyz : 0.128418 * (xyz - 4.0/29.0);
    return max(0.0, xyz) * float3(0.95047, 1.0, 1.08883); // °2 D65 reference values
}

float3 rgb_to_cielab(float3 rgb) 
{ 
    return xyz_to_cielab(rgb_to_xyz(rgb)); 
}

float3 cielab_to_rgb(float3 lab) 
{ 
    return xyz_to_rgb(cielab_to_xyz(lab)); 
}

float3 xyz_to_lms(float3 xyz)
{
    return mul(xyz, float3x3(0.7328, 0.4296,-0.1624,           
                             -0.7036, 1.6975, 0.0061,
                            0.0030, 0.0136, 0.9834));
}

} //Namespace
