local tablex = require("core.tablex")
local cardComponent = require("ui.cardComponent")
local colors = require("ui.colors")
local frame = require("ui.frame")
local util = require("ui.util")
local UI = game.UI
local Map = game.Map
local GetTextHandle = util.GetTextHandle
local SetText = UI.SetText
local CardComponent_MapSummaryList = {}
setmetatable(CardComponent_MapSummaryList, {
  __index = cardComponent.CardComponent
})
function CardComponent_MapSummaryList.New(goParent)
  assert(goParent ~= nil, "GameObject parent reference entered into CardComponent_MapSummaryList constructor was nil.")
  local newTable = {
    _name = "MapSummaryList",
    _goParent = goParent,
    _gameObject = nil,
    _children = {},
    _rows = {}
  }
  local mt = {__index = CardComponent_MapSummaryList}
  setmetatable(newTable, mt)
  return newTable
end
local maxNumOfRows = 12
function CardComponent_MapSummaryList:Init()
  local baseString = "CardComponent_MapSummaryList_Row"
  for i = 1, maxNumOfRows do
    local objName = baseString .. tostring(i)
    local goCurrentRow = self._goParent:FindSingleGOByName(objName)
    tablex.FastInsert(self._rows, goCurrentRow, i)
    goCurrentRow:Show()
    goCurrentRow.Child:Show()
  end
  local summaryType = self:GetProperty("SummaryType")
  if summaryType == "Region" then
    self._updateFunc = self.Update_Region
  elseif summaryType == "Realm" then
    self._updateFunc = self.Update_Realm
  else
    assert(false, "No proper summary type was assigned to CardComponent_MapSummaryFooter. Value was: " .. tostring(summaryType))
  end
  local goBacking = self._gameObject:FindSingleGOByName("backing")
  goBacking:Show()
  self._goBlur = self._goParent:FindSingleGOByName("MapSummary_Blur")
  self._goBlur:Show()
  self._blurFrame = frame.Frame.New(self._goBlur)
  self._backingframe = frame.Frame.New(goBacking)
  local initialHeight = 4
  self._backingframe:Init({height = initialHeight})
  self._blurFrame:Init({height = initialHeight})
  self._frame:Init({height = initialHeight})
end
local summaryCategorySortOrder = {
  SideQuests = 1,
  Dungeons = 2,
  Artifacts = 3,
  FastTravelLocation = 4,
  VendorLocation = 5,
  LoreMarker = 6,
  Ravens = 7,
  Valkyrie = 8,
  ValkyrieQueen = 9,
  RunicChest = 10,
  LegendaryChest = 11,
  PocketRift = 12
}
function CardComponent_MapSummaryList:Update_Region()
  local regionSummaryInfo = self:GetProperty("RegionSummaryInfo")
  local rowsTable = self:GetProperty("Rows")
  local totalRegionProgress = 0
  local totalRegionGoal = 0
  local numberOfInformationRows = regionSummaryInfo ~= nil and #regionSummaryInfo or 0
  if 1 < numberOfInformationRows then
    table.sort(regionSummaryInfo, function(a, b)
      return summaryCategorySortOrder[a.CategoryStr] < summaryCategorySortOrder[b.CategoryStr]
    end)
  end
  for i, goRow in ipairs(self._rows) do
    local goChild = goRow.Child
    local thName = GetTextHandle(goChild, "Text_Name")
    local thAmount = GetTextHandle(goChild, "Text_Amount")
    if i <= numberOfInformationRows then
      local extraInfoTable = regionSummaryInfo[i]
      local name = util.GetLAMSMsg(extraInfoTable.LamsName)
      local progress = extraInfoTable.Progress
      local goal = extraInfoTable.Goal
      totalRegionProgress = totalRegionProgress + progress
      totalRegionGoal = totalRegionGoal + goal
      if extraInfoTable.Discovered then
        SetText(thName, name)
        SetText(thAmount, tostring(progress) .. "/" .. tostring(goal))
        util.SetGameObjectColor(goChild, colors.WHITE, "small_text", "LayerX", "cst_EmissiveTint")
        local progressPercent = progress / goal
        local materialSwapName = 1 <= progressPercent and "" or "bulletPoint_empty"
        goChild:SetMaterialSwap(materialSwapName)
      else
        SetText(thName, util.GetLAMSMsg(8266))
        SetText(thAmount, "")
        goChild:SetMaterialSwap("bulletPoint_empty")
        util.SetGameObjectColor(goChild, colors.GRAY, "small_text", "LayerX", "cst_EmissiveTint")
      end
      goRow:Show()
    else
      goRow:Hide()
    end
  end
  self:SetProperty("TotalRegionProgress", totalRegionProgress)
  self:SetProperty("TotalRegionGoal", totalRegionGoal)
  return numberOfInformationRows
end
function CardComponent_MapSummaryList:Update_Realm()
  local realmInfo = self:GetProperty("RealmInfo")
  local realmSummary, realmTotalProgress, realmTotalGoal = Map.GetRealmSummary(realmInfo.Id)
  self:SetProperty("RealmTotalProgress", realmTotalProgress)
  self:SetProperty("RealmTotalGoal", realmTotalGoal)
  local numActiveRows = 0
  for i, goRow in ipairs(self._rows) do
    local category = realmSummary[i]
    if category ~= nil then
      local goChild = goRow.Child
      local thName = GetTextHandle(goChild, "Text_Name")
      local thAmount = GetTextHandle(goChild, "Text_Amount")
      if category.Discovered then
        SetText(thName, util.GetLAMSMsg(category.LamsName))
        SetText(thAmount, tostring(category.Progress) .. "/" .. tostring(category.Goal))
        util.SetGameObjectColor(goChild, colors.WHITE, "small_text", "LayerX", "cst_EmissiveTint")
        local progressPercent = category.Progress / category.Goal
        local materialSwapName = 1 <= progressPercent and "" or "bulletPoint_empty"
        goChild:SetMaterialSwap(materialSwapName)
      else
        SetText(thName, util.GetLAMSMsg(8266))
        SetText(thAmount, "")
        goChild:SetMaterialSwap("bulletPoint_empty")
        util.SetGameObjectColor(goChild, colors.GRAY, "small_text", "LayerX", "cst_EmissiveTint")
      end
      goRow:Show()
      numActiveRows = numActiveRows + 1
    else
      goRow:Hide()
    end
  end
  return numActiveRows
end
function CardComponent_MapSummaryList:Update()
  local numActiveRows = self:_updateFunc()
  self:SetProperty("numActiveRows", numActiveRows)
  if 0 < numActiveRows then
    local bufferSpace = 0.25
    local heightOfSingleRow = 0.5
    local newHeight = numActiveRows * heightOfSingleRow + bufferSpace * 2
    self._backingframe:SetGOHeight(newHeight)
    self._blurFrame:SetGOHeight(newHeight)
    self._frame:SetHeight(newHeight)
  end
end
return {CardComponent_MapSummaryList = CardComponent_MapSummaryList}
