local tablex = require("core.tablex")
local attributeUtil = require("ui.attributeUtil")
local animationUtil = require("ui.animationUtil")
local card = require("ui.card")
local cardComponent_Header = require("ui.cardComponent_Header")
local cardComponent_Separator = require("ui.cardComponent_Separator")
local cardComponent_Weapon_Slot = require("ui.cardComponent_Weapon_Slot")
local cardComponent_Attributes_Weapon = require("ui.cardComponent_Attributes_Weapon")
local consts = require("ui.consts")
local gemUtil = require("ui.gemUtil")
local lamsConsts = require("ui.lamsConsts")
local pickupConsts = require("ui.pickupConsts")
local pickupUtil = require("ui.pickupUtil")
local recipeUtil = require("ui.recipeUtil")
local util = require("ui.util")
local UI = game.UI
local SetText = game.UI.SetText
local GetTextHandle = util.GetTextHandle
local Pickup = game.Pickup
local WeaponCard = {}
setmetatable(WeaponCard, {
  __index = card.Card
})
function WeaponCard.New(gameObjectReference)
  assert(gameObjectReference ~= nil, "GameObject reference entered into WeaponCard constructor was nil.")
  local newTable = {
    _gameObject = gameObjectReference,
    _components = {},
    _textHandles = {},
    _properties = {}
  }
  local mt = {__index = WeaponCard}
  setmetatable(newTable, mt)
  return newTable
end
function WeaponCard:Init()
  local goRefnode = self._gameObject
  local goRoot = goRefnode.Child
  self._children = {}
  self._children.goDescSeparator = goRoot:FindSingleGOByName("descriptionSeparator")
  self._children.goAttrSlotSeparator = goRoot:FindSingleGOByName("runeSeparator_Diamond")
  self._children.goAnimatedBackings = goRoot:FindSingleGOByName("weaponBackings")
  self._children.goLevelNum = goRoot:FindSingleGOByName("LevelNum")
  self.jid_backing_axe = self._children.goAnimatedBackings:GetJointIndex("Backing_Axe")
  self.jid_backing_blades = self._children.goAnimatedBackings:GetJointIndex("Backing_Blades")
  self.jid_backing_bow = self._children.goAnimatedBackings:GetJointIndex("Backing_Bow")
  self.jid_slotBacking_axe = self._children.goAnimatedBackings:GetJointIndex("SlotBacking_Axe")
  self.jid_slotBacking_blades = self._children.goAnimatedBackings:GetJointIndex("SlotBacking_Blades")
  self._thDescription = util.GetTextHandle(goRoot, "descriptionText")
  self._thLevelNum = util.GetTextHandle(self._children.goLevelNum)
  goRoot:Show()
  self._children.goLevelNum:Show()
  local cc_header = cardComponent_Header.CardComponent_Header.New(goRefnode)
  local cc_descSeparator = cardComponent_Separator.CardComponent_Separator.New(goRefnode, self._children.goDescSeparator)
  local cc_weaponAttributes = cardComponent_Attributes_Weapon.CardComponent_Attributes_Weapon.New(goRefnode)
  local cc_attrSlot_Separator = cardComponent_Separator.CardComponent_Separator.New(goRefnode, self._children.goAttrSlotSeparator)
  cc_header:SetHeight(6)
  cc_descSeparator:SetAssociatedComponent(cc_weaponAttributes)
  self:AddComponent(cc_header)
  self:AddComponent(cc_descSeparator)
  self:AddComponent(cc_weaponAttributes)
  self:AddComponent(cc_attrSlot_Separator)
  local maxSlots = 3
  for i = 1, maxSlots do
    local cc_currentSlot = cardComponent_Weapon_Slot.CardComponent_Weapon_Slot.New(goRefnode, i)
    self:AddComponent(cc_currentSlot)
  end
  self:HideCard()
  local useFrameGroup = true
  self:BaseInit(useFrameGroup)
end
function WeaponCard:Update(doTransitionAnim)
  if doTransitionAnim then
    self:DoDefaultTransitionAnim()
  end
  local weaponName = self:GetProperty("Item")
  local character = weaponName == pickupConsts.SonBow and pickupConsts.TAG_PICKUP_SON or pickupConsts.TAG_PICKUP_KRATOS
  self:SetProperty("Character", character)
  local recipe = self:GetProperty("Recipe")
  local isReinforceRecipe = recipe ~= nil and recipeUtil.IsReinforcementRecipe(recipe)
  self:SetProperty("IsReinforcementRecipe", isReinforceRecipe)
  local stage = pickupUtil.GetProfileStage(weaponName)
  self:SetProperty("Stage", stage)
  self:SetHeader(weaponName, stage)
  self:SetDescription(weaponName, stage)
  self:SetPowerLevel(weaponName, stage, isReinforceRecipe)
  self:SetSlots(weaponName, character)
  self:SetBackings(weaponName)
  local posStart = 0
  local posEnd = 1
  local animRate = 1
  UI.Anim(self._children.goAnimatedBackings, consts.AS_Forward, "", animRate, posStart, posEnd)
  self:BaseUpdate()
end
function WeaponCard:SetBackings(weaponName)
  self._children.goAnimatedBackings:HideJoint(self.jid_backing_axe)
  self._children.goAnimatedBackings:HideJoint(self.jid_slotBacking_axe)
  self._children.goAnimatedBackings:HideJoint(self.jid_backing_blades)
  self._children.goAnimatedBackings:HideJoint(self.jid_slotBacking_blades)
  self._children.goAnimatedBackings:HideJoint(self.jid_backing_bow)
  if weaponName == pickupConsts.Axe then
    self._children.goAnimatedBackings:ShowJoint(self.jid_backing_axe)
    self._children.goAnimatedBackings:ShowJoint(self.jid_slotBacking_axe)
  elseif weaponName == pickupConsts.Blades then
    self._children.goAnimatedBackings:ShowJoint(self.jid_backing_blades)
    self._children.goAnimatedBackings:ShowJoint(self.jid_slotBacking_blades)
  elseif weaponName == pickupConsts.SonBow then
    self._children.goAnimatedBackings:ShowJoint(self.jid_backing_bow)
  end
end
function WeaponCard:SetHeader(weaponName, stage)
  local headerText = pickupUtil.GetDisplayName(weaponName)
  if 0 < stage then
  end
  self:SetProperty("Header", headerText)
end
function WeaponCard:SetDescription(weaponName, stage)
  local descriptionText
  if weaponName == pickupConsts.SonBow then
    descriptionText = util.GetLAMSMsg(lamsConsts.SonBowDescriptions[stage])
  else
    descriptionText = pickupUtil.GetDescription(weaponName)
  end
  UI.SetText(self._thDescription, descriptionText)
end
local powerLevelNumberOffsets = {
  [4] = {3.3, -3.186}
}
function WeaponCard:SetPowerLevel(weaponName, stage, isReinforcementRecipe)
  local powerLevel = attributeUtil.GetPowerLevel(weaponName, stage)
  self:SetProperty("PowerLevel", powerLevel)
  UI.SetText(self._thLevelNum, powerLevel)
  local xOffset = powerLevelNumberOffsets[powerLevel] and powerLevelNumberOffsets[powerLevel][1] or 3.5
  local yOffset = powerLevelNumberOffsets[powerLevel] and powerLevelNumberOffsets[powerLevel][2] or -3.186
  local vTargetPos = engine.Vector.New(xOffset, yOffset, 0)
  local useWorldSpace = false
  UI.SetGOTransformInterpolated(self._children.goLevelNum, vTargetPos, 0, useWorldSpace)
  if isReinforcementRecipe then
    local powerLevelPreview = attributeUtil.GetPowerLevel(weaponName, stage + 1)
    self:SetProperty("PowerLevel_Preview", powerLevelPreview)
  end
  local weaponSpecificAttribute = self:GetWeaponAttributeName(weaponName)
  local attributeValue, previewAttributeValue
  if weaponSpecificAttribute ~= nil then
    attributeValue = pickupUtil.GetAttribute(weaponName, weaponSpecificAttribute, stage)
    if isReinforcementRecipe then
      previewAttributeValue = pickupUtil.GetAttribute(weaponName, weaponSpecificAttribute, stage + 1)
    end
  end
  self:SetProperty("ShortAttr_Name", weaponSpecificAttribute)
  self:SetProperty("ShortAttr_Value", attributeValue)
  if isReinforcementRecipe then
    self:SetProperty("ShortAttr_Value_Preview", previewAttributeValue)
  end
end
function WeaponCard:SetSlots(weaponName, character)
  local pickupSlot = pickupUtil.GetSlotName(weaponName)
  local upgradeSlotNames = pickupUtil.GetWeaponUpgradeSlotNames(character, pickupSlot)
  self:SetProperty("UpgradeSlots", upgradeSlotNames)
end
function WeaponCard:GetWeaponAttributeName(weaponName)
  if weaponName == pickupConsts.Axe then
    return "Strength"
  elseif weaponName == pickupConsts.Blades then
    return "Runic"
  end
  return nil
end
function WeaponCard:ShowCard(showAll)
  self._gameObject:Show()
  if showAll then
    for _, obj in pairs(self.children) do
      if type(obj) == "table" then
        self:ShowCard_Recursive(obj)
      else
        obj:Show()
      end
    end
  end
  self:DoDefaultTransitionAnim()
end
function WeaponCard:SetOnScreen(onScreen, instant)
  local time = instant and 0 or 0.2
  local fadeAlpha = onScreen and 1 or 0
  UI.AlphaFade(self._gameObject, fadeAlpha, time)
end
return {WeaponCard = WeaponCard}
