local classlib = require("core.class")
local colors = require("ui.colors")
local consts = require("ui.consts")
local pickupUtil = require("ui.pickupUtil")
local resourceConsts = require("ui.resourceConsts")
local resourceUtil = require("ui.resourceUtil")
local runeUtil = require("ui.runeUtil")
local util = require("ui.util")
local UI = game.UI
local MessageRow = classlib.Class("MessageRow")
function MessageRow:init(state, args)
  assert(state ~= nil, "Attempted to create new MessageRow without state!")
  assert(args ~= nil, "Attempted to create new MessageRow without args!")
  self._state = state
  self._resourceName = args.ResourceName
  self._resourceType = args.ResourceType
  self._count = args.Count
  self._isMaxed = args.IsMaxed
  self._animInRate = util.GetValueWithDefault(args.AnimInRate, 2)
  self._animOutRate = util.GetValueWithDefault(args.AnimOutRate, 2)
  self._rowTextFadeTime = util.GetValueWithDefault(args.RowTextFadeTime, 0.2)
  self._material = util.GetValueWithDefault(args.Material, "body_text")
  self._layer = util.GetValueWithDefault(args.Layer, "LayerX")
  self._attribute = util.GetValueWithDefault(args.Attribute, "cst_EmissiveTint")
  self._textIndexNames = args.TextIndexNames
end
function MessageRow:get_resourceName()
  return self._resourceName
end
function MessageRow:get_resourceType()
  return self._resourceType
end
function MessageRow:get_count()
  return self._count
end
function MessageRow:get_isMaxed()
  return self._isMaxed
end
function MessageRow:get_rowTextFadeTime()
  return self._rowTextFadeTime
end
function MessageRow:get_textIndexNames()
  return self._textIndexNames
end
function MessageRow:set_count(count)
  self._count = count
end
function MessageRow:GetDisplayName()
  local displayName
  if self._resourceType == resourceConsts.RESOURCE_FLAG_RUNE then
    displayName = runeUtil.GetRuneDisplayNameFromID(self._resourceName)
  elseif self._resourceType == resourceConsts.RESOURCE_FLAG_PICKUP then
    displayName = pickupUtil.GetDisplayName(self._resourceName)
  else
    displayName = resourceUtil.GetDisplayName(self._resourceName)
  end
  return displayName
end
function MessageRow:GetResourceNameColor()
  local rarityColor
  if self._resourceType == resourceConsts.RESOURCE_FLAG_RUNE then
    rarityColor = runeUtil.GetRuneColorFromID(self._resourceName)
  elseif self._resourceType == resourceConsts.RESOURCE_FLAG_PICKUP then
    rarityColor = colors.WHITE
  else
    rarityColor = resourceUtil.GetRarityColor(self._resourceName)
  end
  return rarityColor
end
function MessageRow:ShouldUseLargeRow()
  local useLargeRow = false
  if self._resourceType == resourceConsts.RESOURCE_FLAG_RESOURCE then
    useLargeRow = resourceUtil.ShouldUseLargeRow(self._resourceName)
  end
  return useLargeRow
end
function MessageRow:TakeCount(otherRow)
  local countTaken = false
  if self._resourceName == otherRow:get_resourceName() then
    self._count = self._count + otherRow:get_count()
    otherRow:set_count(-1)
    if otherRow:get_isMaxed() == true then
      self._isMaxed = true
    end
    countTaken = true
  end
  return countTaken
end
function MessageRow:AnimateIn(rowChildObject, animRate)
  if animRate == nil then
    animRate = self._animOutRate
  end
  if animRate == 0 then
    UI.Anim(rowChildObject, consts.AS_Forward, "", 0, 1)
  else
    UI.Anim(rowChildObject, consts.AS_Forward, "", self._animInRate, 0, 1)
  end
end
function MessageRow:AnimateOut(rowChildObject, animRate)
  if animRate == nil then
    animRate = self._animOutRate
  end
  if animRate == 0 then
    UI.Anim(rowChildObject, consts.AS_Forward, "", 0, 0)
  else
    UI.Anim(rowChildObject, consts.AS_Reverse, "", animRate, 1, 0)
  end
end
function MessageRow:DisplayRowIcon(goRow)
  assert(goRow ~= nil, "MessageRow attempted to call DisplayRowIcon without a game object")
  local iconText = ""
  if self._resourceType == resourceConsts.RESOURCE_FLAG_RUNE then
    iconText = runeUtil.GetRuneIconFromID(self._resourceName)
  elseif self._resourceType == resourceConsts.RESOURCE_FLAG_PICKUP then
    iconText = resourceUtil.GetIconText(self._resourceName)
  end
  local thIconText = util.GetTextHandle(goRow, "icon_text")
  if thIconText ~= nil then
    game.UI.SetText(thIconText, iconText)
  end
end
function MessageRow:DisplayRowText(goRow, rowText, rowIndex, color, animRate)
  assert(goRow ~= nil, "MessageRow attempted to call DisplayRowText without a game object")
  if rowText ~= nil and rowIndex ~= nil then
    if animRate == nil then
      animRate = self._rowTextFadeTime
    end
    util.SetGameObjectColor(goRow, color, self._material, self._layer, self._attribute)
    self:SetOneItemVariable(self._textIndexNames.RowText, rowIndex, rowText)
    UI.AlphaFade(goRow, 1, animRate)
  end
end
function MessageRow:DisplayRowCount(rowIndex)
  assert(rowIndex ~= nil, "MessageRow attempted to call DisplayRowCount without a row index!")
  local countText = self:get_isMaxed() == true and "MAX" or "+" .. self:get_count()
  self:SetOneItemVariable(self._textIndexNames.RowCount, rowIndex, countText)
end
function MessageRow:Start(currMessage, rowFinishedCallback)
  local displayRowIndex = currMessage:GetDisplayRowIndex()
  local goRow = currMessage:GetRowGOByIndex(displayRowIndex)
  local bodyColor = self:GetResourceNameColor()
  local oldDisplayTime = currMessage:GetDisplayTime(consts.LOG_MODE_BODY)
  self:AnimateIn(goRow, nil)
  self:DisplayRowIcon(goRow)
  self:DisplayRowText(goRow, self:GetDisplayName(), displayRowIndex, bodyColor, nil)
  self:DisplayRowCount(displayRowIndex)
  currMessage:SetDisplayRowIndex(displayRowIndex + 1)
  currMessage:UpdateTotalDisplayTime(consts.LOG_MODE_BODY)
  self._state:StartTimer(currMessage:GetTimerName(), oldDisplayTime, function()
    rowFinishedCallback(self, currMessage:GetNextMode())
  end)
end
function MessageRow:SetOneItemVariable(var, index, value)
  if index == nil then
    self._state.shared[var] = value
  else
    self._state.shared[var][index] = value
  end
end
return {MessageRow = MessageRow}
