local tablex = require("core.tablex")
local animationUtil = require("ui.animationUtil")
local attributeUtil = require("ui.attributeUtil")
local card = require("ui.card")
local cardComponent_AttributeBonus = require("ui.cardComponent_AttributeBonus")
local cardComponent_Video = require("ui.cardComponent_Video")
local consts = require("ui.consts")
local frame = require("ui.frame")
local frameGroup = require("ui.frameGroup")
local lamsConsts = require("ui.lamsConsts")
local pickupConsts = require("ui.pickupConsts")
local recipeUtil = require("ui.recipeUtil")
local resourceConsts = require("ui.resourceConsts")
local resourceUtil = require("ui.resourceUtil")
local skillTreeConsts = require("ui.skillTreeConsts")
local skillUtil = require("ui.skillUtil")
local util = require("ui.util")
local GetTextHandle = util.GetTextHandle
local SetText = game.UI.SetText
local Pickup = game.Pickup
local UI = game.UI
local Wallets = game.Wallets
local FRAMEINDEX = {header = 0, description = 1}
local SkillInfoCard = {}
setmetatable(SkillInfoCard, {
  __index = card.Card
})
function SkillInfoCard.New(state, gameObject)
  assert(gameObject ~= nil, "GameObject reference entered into SkillInfoCard constructor was nil.")
  assert(state ~= nil, "FSM State entered into SkillInfoCard constructor was nil.")
  local newTable = {
    _gameObject = gameObject,
    _state = state,
    _skillTreeType = nil,
    _backFrame = nil,
    _components = {},
    _children = {
      goBackFrame = nil,
      goBlurFrame = nil,
      goHeader = nil,
      goHeaderText = nil,
      goHeaderBacking = nil,
      goDescriptionGroup = nil,
      goDescriptionText = nil,
      goDescriptionBacking = nil,
      goLockedText = nil,
      goMainRequirementsGroup = nil,
      goMainPurchasedText = nil,
      goMainRequirements = {
        nil,
        nil,
        nil
      }
    },
    _textHandles = {},
    _properties = {}
  }
  local mt = {__index = SkillInfoCard}
  setmetatable(newTable, mt)
  return newTable
end
function GetSkillLevelRequirementGOTable(goLevelSlot, index, gettingMainRequirements)
  local newTable = {
    goRefnode = nil,
    goRefnodeGroup = nil,
    goNumerator = nil,
    goDenominator = nil,
    goName = nil
  }
  if gettingMainRequirements then
    newTable.goRefnode = goLevelSlot:FindSingleGOByName("skillinfocard_main_requirement" .. index)
  else
    newTable.goRefnode = goLevelSlot:FindSingleGOByName("Requirement" .. index)
  end
  newTable.goRefnodeGroup = newTable.goRefnode:FindSingleGOByName("Requirement")
  newTable.goNumerator = newTable.goRefnodeGroup:FindSingleGOByName("Text_Numerator")
  newTable.goDenominator = newTable.goRefnodeGroup:FindSingleGOByName("Text_Denominator")
  newTable.goName = newTable.goRefnodeGroup:FindSingleGOByName("Text_Name")
  return newTable
end
function GetSkillLevelSlotGOTable(cardObj, index)
  local newTable = {
    goRefnode = nil,
    goRefnodeGroup = nil,
    goBackdropGroup = nil,
    goPurchased = nil,
    goTextGroup = nil,
    goTextDescription = nil,
    goTextLVL = nil,
    goRequirementSlots = {
      nil,
      nil,
      nil
    }
  }
  newTable.goRefnode = cardObj:FindSingleGOByName("skillinfocard_LevelSlot" .. index)
  newTable.goRefnodeGroup = newTable.goRefnode:FindSingleGOByName("levelSlot")
  newTable.goBackdropGroup = newTable.goRefnodeGroup:FindSingleGOByName("backdropGroup")
  newTable.goPurchased = newTable.goRefnodeGroup:FindSingleGOByName("Text_Purchased")
  newTable.goTextGroup = newTable.goRefnodeGroup:FindSingleGOByName("TextObjects")
  newTable.goTextDescription = newTable.goRefnodeGroup:FindSingleGOByName("Text_Description")
  newTable.goTextLVL = newTable.goRefnodeGroup:FindSingleGOByName("Text_LVL")
  newTable.goRequirementSlots[1] = GetSkillLevelRequirementGOTable(newTable.goRefnode, 1, false)
  newTable.goRequirementSlots[2] = GetSkillLevelRequirementGOTable(newTable.goRefnode, 2, false)
  newTable.goRequirementSlots[3] = GetSkillLevelRequirementGOTable(newTable.goRefnode, 3, false)
  return newTable
end
function SkillInfoCard:Init(state)
  self._state.shared.SkillInfo_XPNum = {
    n = skillTreeConsts.MAX_LEVEL_COUNT
  }
  self._state.shared.SkillInfo_Description = {
    n = skillTreeConsts.MAX_LEVEL_COUNT
  }
  self._state.shared.SkillInfo_Requirement_Icon_Main = {
    n = skillTreeConsts.MAX_SKILL_CONSUMABLE_REQUIREMENTS_COUNT
  }
  self._state.shared.SkillInfo_RequirementName_Main = {
    n = skillTreeConsts.MAX_SKILL_CONSUMABLE_REQUIREMENTS_COUNT
  }
  self._state.shared.SkillInfo_Numerator_Main = {
    n = skillTreeConsts.MAX_SKILL_CONSUMABLE_REQUIREMENTS_COUNT
  }
  self._state.shared.SkillInfo_Denominator_Main = {
    n = skillTreeConsts.MAX_SKILL_CONSUMABLE_REQUIREMENTS_COUNT
  }
  for levelIndex = 1, skillTreeConsts.MAX_LEVEL_COUNT do
    self._state.shared["SkillInfo_RequirementName_Level" .. levelIndex] = {
      n = skillTreeConsts.MAX_SKILL_CONSUMABLE_REQUIREMENTS_COUNT
    }
    self._state.shared["SkillInfo_Requirement_Icon_Level" .. levelIndex] = {
      n = skillTreeConsts.MAX_SKILL_CONSUMABLE_REQUIREMENTS_COUNT
    }
    self._state.shared["SkillInfo_Numerator_Level" .. levelIndex] = {
      n = skillTreeConsts.MAX_SKILL_CONSUMABLE_REQUIREMENTS_COUNT
    }
    self._state.shared["SkillInfo_Denominator_Level" .. levelIndex] = {
      n = skillTreeConsts.MAX_SKILL_CONSUMABLE_REQUIREMENTS_COUNT
    }
  end
  self._children.rootObj = self._gameObject:FindSingleGOByName("statGroup")
  self._children.goBackFrame = self._gameObject:FindSingleGOByName("BackFrame")
  self._children.goHeader = self._gameObject:FindSingleGOByName("headerGroup")
  self._children.goHeaderText = self._gameObject:FindSingleGOByName("headerText")
  self._children.goHeaderBacking = self._gameObject:FindSingleGOByName("headerBacking")
  self._children.goLevelTextHeader = self._gameObject:FindSingleGOByName("headerText_Level")
  self._children.goDescriptionGroup = self._gameObject:FindSingleGOByName("descriptionGroup")
  self._children.goDescriptionText = self._gameObject:FindSingleGOByName("descriptionText")
  self._children.goDescriptionBacking = self._gameObject:FindSingleGOByName("descriptionBacking")
  self._children.goPurchaseBacking = self._gameObject:FindSingleGOByName("purchaseBacking")
  self._children.goLockedText = self._gameObject:FindSingleGOByName("LockedText")
  self._children.goMainRequirementsGroup = self._gameObject:FindSingleGOByName("FirstLevelRequirementsGroup")
  self._children.goMainPurchasedText = self._gameObject:FindSingleGOByName("MainPurchasedText")
  self._children.goTestBacking = self._gameObject:FindSingleGOByName("descriptionBacking1r")
  for requirementIndex = 1, skillTreeConsts.MAX_SKILL_CONSUMABLE_REQUIREMENTS_COUNT do
    self._children.goMainRequirements[requirementIndex] = GetSkillLevelRequirementGOTable(self._children.goDescriptionGroup, requirementIndex, true)
  end
  self:ShowCard(true)
  self:HideCard()
  self._backFrame = frame.Frame.New(self._children.goBackFrame)
  self._frameGroup = frameGroup.FrameGroup.New()
  self._frameGroup:SetPadding(0.25)
  self._frameGroup:SetBorderPadding(0.75)
  self._backFrame:SetInitialHeight(41)
  self._frameGroup:AddRoot(frame.Frame.New(self._children.goHeaderBacking))
  self._frameGroup:AddNode(frame.Frame.New(self._children.goDescriptionGroup), FRAMEINDEX.description)
  self._frameGroup:GetRoot():Init({height = 3.75})
  self._frameGroup:GetNode(FRAMEINDEX.description):SetHeight(10.75)
  local targetTimelinePos = 1
  local animRate = 0
  UI.Anim(self._children.goHeaderBacking, consts.AS_Forward, "", animRate, targetTimelinePos)
  UI.Anim(self._children.goDescriptionBacking, consts.AS_Forward, "", animRate, targetTimelinePos)
  local color = {
    0.2,
    0.2,
    1
  }
  util.SetGameObjectColor(self._children.goLevelTextHeader, color, "body_text", "LayerX", "cst_EmissiveTint")
  self:AddComponent(cardComponent_AttributeBonus.CardComponent_AttributeBonus.New(self._gameObject))
  self:AddComponent(cardComponent_Video.CardComponent_Video.New(self._gameObject, state))
  self:InitComponents()
end
function SkillInfoCard:ClearCard()
  self._state.shared.SkillInfo_Header = ""
  self._state.shared.SkillInfo_DisplayName = ""
  self._state.shared.SkillInfo_DescriptionMain = ""
  self:ClearLevels()
end
function SkillInfoCard:ClearLevels()
  for levelIndex = 1, skillTreeConsts.MAX_LEVEL_COUNT do
    for requirementIndex = 1, skillTreeConsts.MAX_SKILL_CONSUMABLE_REQUIREMENTS_COUNT do
      self:ClearRequirementSlot(levelIndex, requirementIndex)
    end
  end
end
function SkillInfoCard:ClearRequirementSlot(levelIndex, requirementIndex)
  local clearString = ""
  if levelIndex == 0 then
    self._state.shared.SkillInfo_Requirement_Icon_Main[requirementIndex] = clearString
    self._state.shared.SkillInfo_RequirementName_Main[requirementIndex] = clearString
    self._state.shared.SkillInfo_Numerator_Main[requirementIndex] = clearString
    self._state.shared.SkillInfo_Denominator_Main[requirementIndex] = clearString
  else
    self._state.shared["SkillInfo_Requirement_Icon_Level" .. levelIndex][requirementIndex] = clearString
    self._state.shared["SkillInfo_RequirementName_Level" .. levelIndex][requirementIndex] = clearString
    self._state.shared["SkillInfo_Numerator_Level" .. levelIndex][requirementIndex] = clearString
    self._state.shared["SkillInfo_Denominator_Level" .. levelIndex][requirementIndex] = clearString
  end
end
function SkillInfoCard:SetSkill(skillName, character)
  assert(skillName ~= nil and type(skillName) == "string", "Invalid skillName passed into SkillInfoCard")
  self:ClearCard()
  local pickupName = skillUtil.GetSkillType(skillName)
  local skillIsLocked = false
  local skillIsPurchased = skillUtil.IsSkillPurchased(skillName)
  local skillIsLockedByReinforcement, requiredReinforcementLevel = skillUtil.IsSkillLockedByReinforcement(pickupName, skillName)
  local skillIsLockedByResource = skillUtil.IsSkillLockedByResourceRequirement(skillName)
  local skillIsLockedByPrerequisite = skillUtil.IsSkillLockedByPrerequisiteSkill(skillName)
  local skillIsAffordable = skillUtil.HasEnoughResourcesToPurchase(skillName)
  if not skillIsPurchased and (skillIsLockedByReinforcement or skillIsLockedByResource or skillIsLockedByPrerequisite) then
    skillIsLocked = true
  end
  local hideSkillInfo = skillIsLockedByResource
  if hideSkillInfo then
    self:SetProperty("VideoNameToPlay", animationUtil.GetMysteryVideoName(pickupName))
  else
    self:SetProperty("VideoNameToPlay", skillName)
  end
  self:SetLocked(skillName, pickupName, skillIsLocked, skillIsPurchased, requiredReinforcementLevel, skillIsLockedByReinforcement, skillIsLockedByPrerequisite, skillIsLockedByResource, skillIsAffordable)
  self:SetHeader(skillName, hideSkillInfo)
  self:SetDescription(skillName, hideSkillInfo)
  self:SetMainRequirements(skillName, skillIsLockedByReinforcement, skillIsLockedByPrerequisite, hideSkillInfo)
  self:SetAttributeBonus(skillName, character, hideSkillInfo)
  self:SetIcon(skillName)
end
function SkillInfoCard:SetLocked(skillName, pickupName, skillIsLocked, skillIsPurchased, requiredReinforcementLevel, skillIsLockedByReinforcement, skillIsLockedByPrerequisite, skillIsLockedByResource, skillIsAffordable)
  local lockedReasonString = util.GetLAMSMsg(lamsConsts.skillLocked_unavailable)
  local thLockedText = util.GetTextHandle(self._children.goLockedText, "Text_LockedText")
  UI.SetTextStyle(thLockedText, "NewBaseHeader4_Left")
  if skillIsLocked or not skillIsPurchased and not skillUtil.HasEnoughResourcesToPurchase(skillName) then
    self._children.goLockedText:Show()
    if skillIsLockedByReinforcement then
      UI.SetTextStyle(thLockedText, "NewBaseHeader4")
      lockedReasonString = "[IconLock] " .. util.GetLAMSMsg(pickupConsts.SkillTreeLockedMessages[pickupName]) .. " " .. tostring(requiredReinforcementLevel + 1)
    elseif skillIsLockedByPrerequisite then
      UI.SetTextStyle(thLockedText, "NewBaseHeader4")
      lockedReasonString = util.GetLAMSMsg(lamsConsts.skillLocked_prevSkillRequired)
    elseif not skillIsPurchased and skillIsLockedByResource then
      UI.SetTextStyle(thLockedText, "NewBaseHeader4")
      lockedReasonString = util.GetLAMSMsg(lamsConsts.skillLocked_hiddenSkill)
    elseif not skillIsPurchased and not skillIsAffordable then
      lockedReasonString = util.GetLAMSMsg(lamsConsts.skillLocked_notEnough)
    elseif not skillIsPurchased then
      lockedReasonString = util.GetLAMSMsg(lamsConsts.NotPurchased)
    end
  else
    self._children.goLockedText:Hide()
    if not skillIsPurchased then
      self._children.goMainRequirementsGroup:Show()
    end
  end
  self._state.shared.SkillInfo_Locked = lockedReasonString
  local cardBackingColors, purchaseBackingColors
  if skillIsPurchased then
    cardBackingColors = skillTreeConsts.SkillInfoCardLockedColors.Purchased
    purchaseBackingColors = skillTreeConsts.SkillInfoCardLockedColors.Purchased
  else
    cardBackingColors = skillTreeConsts.SkillInfoCardLockedColors.Purchased
    if skillIsAffordable then
      purchaseBackingColors = skillTreeConsts.SkillInfoCardLockedColors.Purchased
    else
      purchaseBackingColors = skillTreeConsts.SkillInfoCardLockedColors.Purchased
    end
  end
  if skillIsLockedByReinforcement and not skillIsPurchased and not skillIsLockedByResource then
    cardBackingColors = skillTreeConsts.SkillInfoCardLockedColors.Locked
    purchaseBackingColors = skillTreeConsts.SkillInfoCardLockedColors.Locked
  end
  local children = self._children
  util.SetGameObjectColor(children.goDescriptionBacking, cardBackingColors.Image, "skillCard_Backing", "Layer0", "cst_EmissiveTint")
  util.SetGameObjectColor(children.goDescriptionBacking, cardBackingColors.Bars, "skillTree_Divider", "LayerX", "cst_EmissiveTint")
  util.SetGameObjectColor(children.goDescriptionBacking, cardBackingColors.Backing, "skillCard_Backing", "Layer2", "cst_EmissiveTint")
  util.SetGameObjectColor(children.goDescriptionBacking, cardBackingColors.Bars, "skillTree_Divider", "LayerX", "cst_EmissiveTint")
  util.SetGameObjectColor(children.goPurchaseBacking, purchaseBackingColors.Image, "skillCard_Backing", "Layer0", "cst_EmissiveTint")
  util.SetGameObjectColor(children.goPurchaseBacking, purchaseBackingColors.Bars, "skillTree_Divider", "LayerX", "cst_EmissiveTint")
  util.SetGameObjectColor(children.goPurchaseBacking, purchaseBackingColors.Backing, "skillCard_Backing", "Layer2", "cst_EmissiveTint")
  util.SetGameObjectColor(children.goPurchaseBacking, purchaseBackingColors.BarsR, "skillTree_Divider", "LayerX", "cst_EmissiveTint")
end
function SkillInfoCard:UpdateFrames(backframe, backframeBlur)
  self._frameGroup:Update()
  self._frameGroup:SetBackframeScale(backframe)
  self._frameGroup:SetBackframeScale(backframeBlur)
end
function SkillInfoCard:SetHeader(skillName, hideSkillInfo)
  if hideSkillInfo then
    self._state.shared.SkillInfo_Header = util.GetLAMSMsg(lamsConsts.HiddenSkill_Header)
  else
    self._state.shared.SkillInfo_Header = recipeUtil.GetDisplayName(skillName)
  end
  local numOfMainTextLines = 2
  local padding = 0.25
  local mainTextSingleLineHeight = 1.25
  local blueLevelTextHeight = 0.75
  local totalHeight = padding * 2 + mainTextSingleLineHeight * numOfMainTextLines + blueLevelTextHeight
  self._frameGroup:GetRoot():SetGOHeight(totalHeight)
end
function SkillInfoCard:SetDescription(skillName, hideSkillInfo)
  if hideSkillInfo then
    self._state.shared.SkillInfo_DescriptionMain = util.GetLAMSMsg(lamsConsts.HiddenSkill_Description)
  else
    self._state.shared.SkillInfo_DescriptionMain = recipeUtil.GetDescription(skillName)
  end
end
function SkillInfoCard:SetMainRequirements(skillName, skillIsLockedByReinforcement, skillIsLockedByPrerequisite, hideSkillInfo)
  local purchased = skillUtil.IsSkillPurchased(skillName)
  if purchased then
    self._children.goMainRequirementsGroup:Hide()
    self._children.goMainPurchasedText:Show()
  else
    if hideSkillInfo or skillIsLockedByReinforcement or skillIsLockedByPrerequisite then
      self._children.goMainRequirementsGroup:Hide()
    else
      self._children.goMainRequirementsGroup:Show()
    end
    self._children.goMainPurchasedText:Hide()
    self:SetLevelRequirements(skillName, 0)
  end
end
function SkillInfoCard:SetAttributeBonus(skillName, character, hideSkillInfo)
  if hideSkillInfo then
    self:SetProperty("BonusTable", {})
    self:SetProperty("Character", nil)
  else
    local bonusTable = skillUtil.GetAttributeBonusesForSkill(skillName)
    self:SetProperty("BonusTable", bonusTable)
    self:SetProperty("Character", character)
  end
end
function SkillInfoCard:SetIcon(skillName)
  local iconTextMacro = "icon"
  self._state.shared.SkillInfo_Icon = "[" .. skillName .. "_Icon" .. "]"
end
function SkillInfoCard:SetRequirementSlotsVisible(levelIndex, value)
  if value then
    for requirementSlotIndex = 1, skillTreeConsts.MAX_SKILL_CONSUMABLE_REQUIREMENTS_COUNT do
      self._children.goLevelSlots[levelIndex].goRequirementSlots[requirementSlotIndex].goRefnode:Show()
    end
  else
    for requirementSlotIndex = 1, skillTreeConsts.MAX_SKILL_CONSUMABLE_REQUIREMENTS_COUNT do
      self._children.goLevelSlots[levelIndex].goRequirementSlots[requirementSlotIndex].goRefnode:Hide()
    end
  end
end
function SkillInfoCard:SetSingleLevel(levelIndex, skillName)
  self._state.shared.SkillInfo_Description[levelIndex] = recipeUtil.GetDescription(skillName)
  local purchased = skillUtil.IsSkillPurchased(skillName)
  self:SetLevelPurchased(levelIndex, purchased)
  if purchased then
    self:SetRequirementSlotsVisible(levelIndex, false)
  else
    self:SetRequirementSlotsVisible(levelIndex, true)
    self:SetLevelRequirements(skillName, levelIndex)
    local pickupName = pickupConsts.Axe
    local locked = skillUtil.IsSkillLockedByReinforcement(pickupName, skillName)
    self:SetLevelLocked(levelIndex, locked)
    if locked then
      self:SetRequirementSlotsVisible(levelIndex, false)
    end
  end
end
function SkillInfoCard:SetLevelLocked(levelIndex, locked)
  local animRate = 0
  local targetTimelinePos = 0
  if locked then
    targetTimelinePos = 1
  end
  local goBackdrop = self._children.goLevelSlots[levelIndex].goBackdropGroup
  UI.Anim(goBackdrop, consts.AS_Forward, "", animRate, targetTimelinePos)
  local scalarColorValue = 0.5
  local color = {
    scalarColorValue,
    scalarColorValue,
    scalarColorValue
  }
  util.SetGameObjectColor(self._children.goLevelSlots[levelIndex].goTextGroup, color, "body_text", "LayerX", "cst_EmissiveTint")
end
function SkillInfoCard:IsMainSlot(levelIndex)
  return levelIndex == 0
end
function SkillInfoCard:SetLevelRequirements(skillName, levelIndex)
  if skillName == nil then
    return
  end
  local inputItems = recipeUtil.GetInputRecipeItems(skillName)
  local outputItems = recipeUtil.GetOutputRecipeItems(skillName)
  local numOfConsumableItems = 0
  for requirementIndex, item in ipairs(inputItems) do
    local isConsumable = item.Consume
    if isConsumable then
      numOfConsumableItems = numOfConsumableItems + 1
      if numOfConsumableItems <= skillTreeConsts.MAX_SKILL_CONSUMABLE_REQUIREMENTS_COUNT then
        local currResourceAmount = resourceUtil.GetAmount(item.Name)
        local requiredAmount = item.Amount
        local goRequirementSlotRefnode
        if levelIndex == 0 then
          goRequirementSlotRefnode = self._children.goMainRequirements[requirementIndex].goRefnode
        else
          goRequirementSlotRefnode = self._children.goLevelSlots[levelIndex].goRequirementSlots[requirementIndex].goRefnode
        end
        goRequirementSlotRefnode:Show()
        self:SetAcquiredAndRequiredIngredients(levelIndex, requirementIndex, currResourceAmount, requiredAmount)
        self:SetRequirementIcon(levelIndex, requirementIndex, item.Name)
        self:SetRequirementDisplayName(levelIndex, requirementIndex, resourceUtil.GetDisplayName(item.Name))
      end
    end
  end
end
function SkillInfoCard:SetRequirementRowAlpha(levelIndex, requirementIndex, alpha)
  local objsToSet = {
    numerator = nil,
    denominator = nil,
    name = nil
  }
  if levelIndex == 0 then
    objsToSet.numerator = self._children.goMainRequirements[requirementIndex].goNumerator
    objsToSet.denominator = self._children.goMainRequirements[requirementIndex].goDenominator
    objsToSet.name = self._children.goMainRequirements[requirementIndex].goName
  else
    objsToSet.numerator = self._children.goLevelSlots[levelIndex].goRequirementSlots[requirementIndex].goNumerator
    objsToSet.denominator = self._children.goLevelSlots[levelIndex].goRequirementSlots[requirementIndex].goDenominator
    objsToSet.name = self._children.goLevelSlots[levelIndex].goRequirementSlots[requirementIndex].goName
  end
  for _, obj in pairs(objsToSet) do
    util.SetGameObjectAlpha(obj, alpha, "body_text", "LayerX", "cst_AlphaBlendAmount")
  end
end
function SkillInfoCard:SetAcquiredAndRequiredIngredients(levelIndex, requirementIndex, numeratorValue, denominatorValue)
  assert(levelIndex ~= nil and requirementIndex ~= nil and numeratorValue ~= nil and denominatorValue ~= nil, "Nil values passed into SkillInfoCard:SetAcquiredAndRequiredIngredients")
  local Color = util.StyleText
  if levelIndex == 0 then
    local haveEnough = denominatorValue <= numeratorValue
    local color = haveEnough and "Color_White" or "Color_Red"
    local numeratorString = Color(tostring(numeratorValue), color)
    self._state.shared.SkillInfo_Numerator_Main[requirementIndex] = "[EconomyXP]" .. " " .. numeratorString .. " / " .. tostring(denominatorValue)
  else
    self._state.shared["SkillInfo_Numerator_Level" .. levelIndex][requirementIndex] = tostring(numeratorValue)
    self._state.shared["SkillInfo_Denominator_Level" .. levelIndex][requirementIndex] = tostring(denominatorValue)
  end
end
function SkillInfoCard:SetRequirementIcon(levelIndex, requirementIndex, requirementName)
  if levelIndex == 0 then
    self._state.shared.SkillInfo_Requirement_Icon_Main[requirementIndex] = "[" .. requirementName .. "]"
  else
    self._state.shared["SkillInfo_Requirement_Icon_Level" .. levelIndex][requirementIndex] = "[" .. requirementName .. "]"
  end
end
function SkillInfoCard:SetRequirementDisplayName(levelIndex, requirementIndex, requirementName)
  if levelIndex == 0 then
    self._state.shared.SkillInfo_RequirementName_Main[requirementIndex] = requirementName
  else
    self._state.shared["SkillInfo_RequirementName_Level" .. levelIndex][requirementIndex] = requirementName
  end
end
function SkillInfoCard:SetLevelPurchased(levelIndex, purchased)
  local goBackdrop = self._children.goLevelSlots[levelIndex].goBackdropGroup
  local alpha
  if purchased then
    alpha = 100
    self._children.goLevelSlots[levelIndex].goPurchased:Show()
  else
    alpha = 30
    self._children.goLevelSlots[levelIndex].goPurchased:Hide()
  end
  util.SetGameObjectAlpha(self._children.goLevelSlots[levelIndex].goTextLVL, alpha, "body_text", "LayerX", "cst_AlphaBlendAmount")
  util.SetGameObjectAlpha(self._children.goLevelSlots[levelIndex].goTextDescription, alpha, "body_text", "LayerX", "cst_AlphaBlendAmount")
  util.SetGameObjectAlpha(self._children.goLevelSlots[levelIndex].goTextGroup, alpha, "body_text", "LayerX", "cst_AlphaBlendAmount")
end
function SkillInfoCard:ShowCard(showAll)
  self._gameObject:Show()
  if showAll then
    for _, obj in pairs(self._children) do
      if type(obj) == "table" then
        self:ShowCard_Recursive(obj)
      else
        obj:Show()
      end
    end
  end
end
function SkillInfoCard:ShowCard_Recursive(obj)
  if type(obj) == "table" then
    for _, childObj in pairs(obj) do
      self:ShowCard_Recursive(childObj)
    end
  else
    obj:Show()
  end
end
function SkillInfoCard:HideCard()
  self._gameObject:Hide()
end
return {SkillInfoCard = SkillInfoCard}
