local classlib = require("core.class")
local tablex = require("core.tablex")
local colors = require("ui.colors")
local consts = require("ui.consts")
local fsm = require("ui.fsm")
local lamsIDs = require("ui.lamsConsts")
local list = require("ui.list")
local menu = require("ui.menu")
local mpicon = require("ui.mpicon")
local slider = require("ui.slider")
local toggle = require("ui.toggle")
local util = require("ui.util")
local uiCalls = require("ui.uicalls")
local Audio = game.Audio
local UI = game.UI
local PhotoMode = game.PhotoMode
local PhotoModeHUD = classlib.Class("PhotoModeHUD", fsm.UIState)
local PHOff = PhotoModeHUD:StateClass("PHOff", fsm.UIState)
local PHOn = PhotoModeHUD:StateClass("PHOn", fsm.UIState)
local GeneralCamera = PhotoModeHUD:StateClass("GeneralCamera", fsm.UIState)
local Shutter = PhotoModeHUD:StateClass("Shutter", fsm.UIState)
local Brightness = PhotoModeHUD:StateClass("Brightness", fsm.UIState)
local Vignette = PhotoModeHUD:StateClass("Vignette", fsm.UIState)
local Character = PhotoModeHUD:StateClass("Character", fsm.UIState)
local HideUI = PhotoModeHUD:StateClass("HideUI", fsm.UIState)
local photoModeHUD = PhotoModeHUD.New("photoModeHUD", {
  PHOff,
  PHOn,
  {
    GeneralCamera,
    Shutter,
    Brightness,
    Vignette,
    Character,
    HideUI
  }
})
function PhotoModeHUD:Enter()
  self:WantPadEvents(true)
  self:Goto("PHOff")
end
PhotoModeHUD.EVT_Restart = PhotoModeHUD.Enter
PhotoModeHUD.EVT_GAME_START = PhotoModeHUD.Enter
function PHOff:Enter()
  util.Hide("photoModeHUD", "UI_Elements", "PM_Borders")
end
function PHOff:Exit()
end
function PHOff:EVT_TURN_ON_PHOTO_MODE()
  self:Goto("PHOn")
end
function PHOn:Setup()
  self.tab_animation_rate = 0
  self.total_num_buttons = 5
  self.brightness = self:GetState("Brightness")
  self.generalCamera = self:GetState("GeneralCamera")
  self.vignette = self:GetState("Vignette")
  self.character = self:GetState("Character")
  self.shutter = self:GetState("Shutter")
  self.hideUI = self:GetState("HideUI")
  self.go_photoModeHUD = util.GetUiObjByName("photoModeHUD")
  self.go_UI_Elements = util.GetUiObjByName("UI_Elements")
  self.go_PM_Borders = util.GetUiObjByName("PM_Borders")
  self.go_RestrictedCamera = util.GetUiObjByName("RestrictedCamera")
  self.goGlobalMenu = util.GetUiObjByName("globalMenu")
  self.goMainHUD = util.GetUiObjByName("mainHUD")
  self.goMessageHUD = util.GetUiObjByName("messageHUD")
  self.globalMenuPosition = self.goGlobalMenu:GetWorldPosition()
  self.mainHUDPosition = self.goMainHUD:GetWorldPosition()
  self.th_footerButtons = util.GetTextHandle(self.go_UI_Elements, "PhotomodeFooterButtons")
  UI.SetTextIsClickable(self.th_footerButtons)
  self.camera_footer = "[L2Button] " .. util.GetLAMSMsg(lamsIDs.CamDown) .. consts.Footer_Spacer .. "[R2Button] " .. util.GetLAMSMsg(lamsIDs.CamUp) .. consts.Footer_Spacer
  self.camera_partial_footer = "[SquareButton] " .. util.GetLAMSMsg(lamsIDs.ResetCamera) .. consts.Footer_Spacer
  self.ui_footer = "[TriangleButton] " .. util.GetLAMSMsg(lamsIDs.ResetSettings) .. consts.Footer_Spacer .. "[AdvanceButton] " .. util.GetLAMSMsg(lamsIDs.HideUI) .. consts.Footer_Spacer .. "[UpButton][LeftButton][DownButton][RightButton]" .. util.GetLAMSMsg(45126) .. consts.Footer_Spacer .. "[BackButton] " .. util.GetLAMSMsg(lamsIDs.Back) .. consts.Footer_Spacer .. "[R3]" .. util.GetLAMSMsg(lamsIDs.TakePhoto)
  self.player = nil
  self:SetAllHandlersFalse()
  self.handling_event = false
  self.takingPicture = false
  self.currentMenu = 1
end
function PHOn:Enter()
  self.entering = true
  self.entering_counter = 0
  self.handling_event = true
  self.takingPicture = false
  self:SetAllHandlersFalse()
  self.son = game.AI.FindSon()
  self.player = game.Player.FindPlayer()
  self.player.Pad:UIDisableHack(true, consts.UIBUTTONDISABLE_PHOTOMODE)
  if self.player.Pad.PhotoModeButtonMappingHack then
    self.player.Pad:PhotoModeButtonMappingHack(true)
  end
  self.currentMenu = 1
  UI.PushCamera("HUDMenuCam")
  UI.WorldUIRender("")
  engine.PhotoModeActivate()
  self:SendEventToUIFsm("mainHUD", "EVT_TURN_PHOTO_MODE_ON")
  self:SendEventToUIFsm("globalMenu", "EVT_TURN_PHOTO_MODE_ON")
end
function PHOn:Exit()
  self:ResetValuesOnExit()
  self.go_photoModeHUD:Hide()
  self.go_UI_Elements:Hide()
  self.go_PM_Borders:Hide()
  self.player.Pad:UIDisableHack(false, consts.UIBUTTONDISABLE_PHOTOMODE)
  if self.player.Pad.PhotoModeButtonMappingHack then
    self.player.Pad:PhotoModeButtonMappingHack(false)
  end
  self:SendEventToUIFsm("mainHUD", "EVT_TURN_PHOTO_MODE_OFF")
  self:SendEventToUIFsm("globalMenu", "EVT_TURN_PHOTO_MODE_OFF")
  Audio.PlaySound("SND_UX_Photo_Mode_Bottom_Menu_Back")
  engine.PhotoModeDeactivate()
  UI.PopCamera("HUDMenuCam")
  self.goMessageHUD:SetWorldPosition(self.mainHUDPosition)
end
function PHOn:DeferredEnter()
  self.inCinematic = self:IsGameInCinematic()
  self.isCameraPartiallyLocked = self:CheckIfPartiallyLocked()
  self.isSonAvailable = self:FindIfSonIsAvailable()
  self.faceChangeEnabled = self:IsFaceChangeEnabled()
  self:SetupTabList()
  self:FindAllDefaultValues()
  self.go_photoModeHUD:Show()
  self.go_UI_Elements:Show()
  self.go_PM_Borders:Show()
  self:ShowHeaderTabs()
  self:UpdateFooterText()
  self.handling_event = false
  if self.inCinematic then
    self.go_RestrictedCamera:Show()
    self:Goto("Brightness")
  else
    self.go_RestrictedCamera:Hide()
    self:Goto("GeneralCamera")
  end
end
function PHOn:UpdateFooterText()
  local footer = self.ui_footer
  if not self.inCinematic then
    footer = self.camera_footer .. self.camera_partial_footer .. footer
  elseif self.inCinematic and self.isCameraPartiallyLocked then
    footer = self.camera_partial_footer .. footer
  end
  UI.SetText(self.th_footerButtons, footer)
end
function PHOn:IsGameInCinematic()
  if PhotoMode.IsCameraMotionLocked ~= nil then
    return PhotoMode.IsCameraMotionLocked()
  end
  return false
end
function PHOn:CheckIfPartiallyLocked()
  if PhotoMode.IsCameraMotionPartiallyLocked ~= nil and self.inCinematic then
    return PhotoMode.IsCameraMotionPartiallyLocked()
  end
  return false
end
function PHOn:IsFaceChangeEnabled()
  if PhotoMode.GetFaceChangeEnabled ~= nil then
    return PhotoMode.GetFaceChangeEnabled()
  end
  return true
end
function PHOn:FindIfSonIsAvailable()
  local son = game.AI.FindSon()
  if son ~= nil then
    return son:IsAvailableInLevel()
  end
  return true
end
function PHOn:EnableTab(tab_name, tab_material)
  util.SetColor(tab_name, colors.WHITE, tab_material, "LayerX", "cst_EmissiveTint")
end
function PHOn:DisableTab(tab_name, tab_material)
  util.SetColor(tab_name, colors.Text_Settings_Disabled, tab_material, "LayerX", "cst_EmissiveTint")
end
function PHOn:SetupTabList()
  if self.inCinematic then
    if self.faceChangeEnabled then
      self.tab_list = {
        "Brightness",
        "Vignette",
        "Character"
      }
      self:EnableTab("CharacterTab", "Characters")
    else
      self.tab_list = {"Brightness", "Vignette"}
      self:DisableTab("CharacterTab", "Characters")
    end
    self:DisableTab("GeneralCameraTab", "Camera")
    self:DisableTab("ShutterTab", "Aperture")
  else
    self.tab_list = {
      "GeneralCamera",
      "Shutter",
      "Brightness",
      "Vignette",
      "Character"
    }
    self:EnableTab("GeneralCameraTab", "Camera")
    self:EnableTab("ShutterTab", "Aperture")
    self:EnableTab("CharacterTab", "Characters")
  end
end
function PHOn:ResetValuesOnExit()
  self:ForceAllCharactersOn()
  self.vignette:ShowSelectedBorderGO(1, self.vignette.borders)
  self.vignette:ShowSelectedBorderGO(1, self.vignette.gow_logos)
  self.brightness:SetLutValue(-1)
  self.brightness:SetLutIntensity(10)
  self.character:SetCharacterFace(self.son, -1)
  self.character:SetCharacterFace(self.player, -1)
  self.shutter:DestroyCreatureIcon()
end
function PHOn:FindAllDefaultValues()
  self.currentBorder = 1
  self.currentGOWBorder = 1
  self.currentLUT = -1
  self.currentLUTIntensity = 10
  self.currentCameraRoll = 0
  self.currentKratosFace = -1
  self.currentAtreusFace = -1
  self.original_fov = PhotoMode.GetFOV()
  self.original_focal = PhotoMode.GetFocalLength()
  self.original_dofDist = PhotoMode.GetDOFFocusDistance()
  self.original_dofFStop = PhotoMode.GetDOFFStop()
  self:SetupOriginalFilmGrainAmount()
  self.brightness.first_time = true
  self.original_vignette = PhotoMode.GetVignette()
  self.original_vignetteBrightness = PhotoMode.GetVignetteBrightness()
  self.original_vignetteFalloff = PhotoMode.GetVignetteFalloff()
end
function PHOn:SetupOriginalFilmGrainAmount()
  local fg = PhotoMode.GetFilmGrainAmount()
  self.original_filmgrain = fg
  PhotoMode.SetFilmGrainAmount(self.original_filmgrain)
end
function PHOn:ForceAllCharactersOn()
  self.character:ForceAllCharactersOn()
end
function PHOn:RefreshLists(photoModeSettingsList, newItemArray)
  local showList = true
  local useOnGainFocus = true
  local itemDetermineFocusabilityFunc = function(settingsInfo)
    return (not self.inCinematic or settingsInfo.DisabledInCine == nil or not settingsInfo.DisabledInCine) and (self.faceChangeEnabled or settingsInfo.DisabledFaces == nil or not settingsInfo.DisabledFaces) and (self.isSonAvailable or settingsInfo.DisableSon == nil or not settingsInfo.DisableSon)
  end
  local itemGetTextFunc = function(setting)
    return setting.Text
  end
  photoModeSettingsList:Refresh(newItemArray, showList, useOnGainFocus, itemDetermineFocusabilityFunc, itemGetTextFunc)
end
function PHOn:EVT_PHOTOMODE_TAKINGPICTURE_START()
  self:WantPadEvents(false)
  self.go_UI_Elements:Hide()
  if self.shutter.inShutter then
    self.shutter:DestroyCreatureIcon()
  end
end
function PHOn:EVT_PHOTOMODE_TAKINGPICTURE_DONE()
  if not self.hideUI.inState then
    self.go_UI_Elements:Show()
    if self.shutter.inShutter then
      self.shutter:ShouldShowIconAfterPicture()
    end
  end
  self:WantPadEvents(true)
end
function PHOn:EVT_TURN_OFF_PHOTO_MODE()
  self:Goto("PHOff")
end
function PHOn:ShowHeaderTabs()
  util.Show("GeneralCameraTab", "ShutterTab", "BrightnessTab", "VignetteTab", "CharacterTab")
  UI.Anim("GeneralCameraTab", consts.AS_Forward, "", 0, 0)
  UI.Anim("ShutterTab", consts.AS_Forward, "", 0, 0)
  UI.Anim("BrightnessTab", consts.AS_Forward, "", 0, 0)
  UI.Anim("VignetteTab", consts.AS_Forward, "", 0, 0)
  UI.Anim("CharacterTab", consts.AS_Forward, "", 0, 0)
end
function PHOn:SetNewMenu()
  self:Goto(self.tab_list[self.currentMenu])
end
function PHOn:SetColorOnButtonAndAllChildren(setting_button, text_color, alpha)
  if setting_button == nil then
    return
  end
  setting_button:SetTextColor(text_color, "medium_text_settings1")
  local button_go = setting_button:GetGameObject()
  if button_go.Descendants ~= nil then
    for _, childGO in ipairs(button_go.Descendants) do
      local th = util.GetTextHandle(childGO)
      if th ~= nil then
        util.SetGameObjectColor(childGO, text_color, "medium_text_settings1", "LayerX", "cst_EmissiveTint")
      end
    end
  end
  local activationList = setting_button:get_activationList()
  for _, activationSet in ipairs(activationList) do
    local settingsItem = activationSet.Item
    if settingsItem ~= nil and settingsItem.CLASSNAME == "Slider" then
      local goChild = settingsItem:GetInstancedChildObject()
      UI.AlphaFade(goChild, alpha, 0)
    end
  end
end
function PHOn:DetermineIfItemsAreDisabled(setting_list, itemArray, disabledBoolean, disabledKey)
  if not disabledBoolean or setting_list == nil then
    return
  end
  for i, item in ipairs(itemArray) do
    if item[disabledKey] then
      local setting_button = setting_list:GetButton(i)
      self:SetColorOnButtonAndAllChildren(setting_button, colors.Text_Settings_Disabled, 0.5)
    end
  end
end
function PHOn:EnableAllValuesOnExit(setting_list, itemArray)
  if setting_list == nil then
    return
  end
  for i, _ in ipairs(itemArray) do
    local setting_button = setting_list:GetButton(i)
    self:SetColorOnButtonAndAllChildren(setting_button, colors.Text_Settings1, 1)
  end
end
function PHOn:DynamicPrompt(settingsList, menu, end_function, header, body, do_function)
  if settingsList ~= nil then
    self:LockCameraMovement()
    self:SendEventUI("EVT_SHOW_SETTINGS_DIALOG_MESSAGE", {
      Header = header,
      Body = body,
      ActiveMenus = {menu},
      ActiveLists = {settingsList},
      Events = {
        Advance = {
          Text = util.GetLAMSMsg(lamsIDs.Confirm),
          Handler = function()
            self:UnlockCameraMovement()
            end_function()
            do_function()
            self.handling_event = false
          end
        },
        Back = {
          Text = util.GetLAMSMsg(lamsIDs.Back),
          Handler = function()
            self:UnlockCameraMovement()
            end_function()
            self.handling_event = false
          end
        }
      },
      PhotoMode = true
    })
  end
end
function PHOn:SetDefaultValues(settingsList, menu, endFunction)
  self:DynamicPrompt(settingsList, menu, endFunction, util.GetLAMSMsg(lamsIDs.ResetSettings), util.GetLAMSMsg(lamsIDs.AreYouSure_SettingsReset), function()
    settingsList:SetDefaultButtonValues()
  end)
end
function PHOn:SetDefaultCameraPrompt(settingsList, menu, endFunction)
  self:DynamicPrompt(settingsList, menu, endFunction, util.GetLAMSMsg(lamsIDs.ResetCamera), util.GetLAMSMsg(lamsIDs.AreYouSure_SettingsReset), function()
    self:ResetCameras()
  end)
end
function PHOn:ExitPhotoModePrompt(settingsList, menu, endFunction)
  self:DynamicPrompt(settingsList, menu, endFunction, util.GetLAMSMsg(lamsIDs.PhotoModeExit), util.GetLAMSMsg(lamsIDs.AreYouSure_PhotoModeExit), function()
    self:ExitPhotomode()
  end)
end
function PHOn:ResetCameras()
  PhotoMode.ResetCamera()
  if self.generalCamera.cameraRollSlider ~= nil then
    self.generalCamera.cameraRollSlider:Update()
  end
end
function PHOn:ExitPhotomode()
  self:Goto("PHOff")
end
function PHOn:LockCameraMovement()
  if PhotoMode.LockCameraMotion ~= nil then
    PhotoMode.LockCameraMotion(true)
  end
end
function PHOn:UnlockCameraMovement()
  if PhotoMode.LockCameraMotion ~= nil then
    PhotoMode.LockCameraMotion(false)
  end
end
function PHOn:BackHandler()
  self:GetState(self.tab_list[self.currentMenu]):LeavePhotomode()
end
function PHOn:ResetCamHandler()
  self:GetState(self.tab_list[self.currentMenu]):ResetCamera()
end
function PHOn:ResetSettingsHandler()
  self:GetState(self.tab_list[self.currentMenu]):SetDefaultValues()
end
function PHOn:TabLeftHandler()
  self.currentMenu = self.currentMenu - 1
  if self.currentMenu < 1 then
    self.currentMenu = #self.tab_list
  end
  self:SetNewMenu()
  self.handling_event = false
end
function PHOn:TabRightHandler()
  self.currentMenu = self.currentMenu + 1
  if self.currentMenu > #self.tab_list then
    self.currentMenu = 1
  end
  self:SetNewMenu()
  self.handling_event = false
end
function PHOn:HideHandler()
  self.handling_event = false
  self:Goto("HideUI")
end
function PHOn:TakePictureHandler()
  self.handling_event = false
  if not self.displayTime then
    self:EVT_PHOTOMODE_TAKINGPICTURE_START()
    uiCalls.UI_Event_ClearAllDesignerMessages()
    self.takingPicture = true
    game.StartScreenCapture()
  end
end
function PHOn:FinishPicture()
  if self.displayTime then
    self.displayTime = self.displayTime - UI.GetFrameTime()
    if self.displayTime <= 0 then
      uiCalls.UI_Event_ClearAllDesignerMessages()
      self.takingPicture = false
      self.displayTime = nil
    end
    return
  end
  local path = game.FinishScreenCapture()
  if path then
    self:EVT_PHOTOMODE_TAKINGPICTURE_DONE()
    self.displayTime = 2.3
    self.goMessageHUD:SetWorldPosition(self.globalMenuPosition)
    util.Show("messageHUD")
    uiCalls.SendSidebarDesignerMessage({
      Text = "<FilenameHeader>" .. util.GetLAMSMsg(lamsIDs.PhotoSaved) .. [[
</>
<FilenameBody>]] .. string.gsub(path, "\\", "\\\\") .. "</>",
      MessageVariant = uiCalls.msgParam.SIDEBAR_SMALL,
      DisplayTime = self.displayTime
    })
  end
end
function PHOn:EVT_Back_Release()
  if not self.handling_event then
    self.back = true
  end
end
PHOn.EVT_Options_Release = PHOn.EVT_Back_Release
function PHOn:EVT_Advance_Release()
  if not self.handling_event then
    self.hide = true
  end
end
function PHOn:EVT_Square_Release()
  if not self.handling_event and (not self.inCinematic or self.isCameraPartiallyLocked) then
    self.reset_cam = true
  end
end
function PHOn:EVT_Triangle_Release()
  if not self.handling_event then
    self.reset_settings = true
  end
end
function PHOn:EVT_L1_Release()
  if not self.handling_event then
    self.tab_left = true
    Audio.PlaySound("SND_UX_Photo_Mode_Menu_Change_Tick_Horiztonal")
  end
end
function PHOn:EVT_R1_Release()
  if not self.handling_event then
    self.tab_right = true
    Audio.PlaySound("SND_UX_Photo_Mode_Menu_Change_Tick_Horiztonal")
  end
end
function PHOn:EVT_R3_Release()
  if not self.handling_event then
    self.take_picture = true
  end
end
function PHOn:EVT_UPDATE_WINDOW_SIZE()
  self.vignette:ShowSelectedBorderGO(self.currentBorder, self.vignette.borders)
end
function PHOn:SetAllHandlersFalse()
  self.back = false
  self.options_back = false
  self.reset_cam = false
  self.reset_settings = false
  self.tab_right = false
  self.tab_left = false
  self.hide = false
  self.take_picture = false
end
function PHOn:Update()
  if self.takingPicture then
    self:FinishPicture()
  end
  if self.entering and self.entering_counter < 2 then
    self.entering_counter = self.entering_counter + 1
  elseif self.entering then
    self.entering = false
    self:DeferredEnter()
    print(game.InteractZone.IsSonInteractStarting())
  elseif not self.handling_event then
    if self.back and not self.displayTime then
      self.handling_event = true
      self:SetAllHandlersFalse()
      self:BackHandler()
    elseif self.reset_cam then
      self.handling_event = true
      self:SetAllHandlersFalse()
      self:ResetCamHandler()
    elseif self.reset_settings then
      self.handling_event = true
      self:SetAllHandlersFalse()
      self:ResetSettingsHandler()
    elseif self.tab_right then
      self.handling_event = true
      self:SetAllHandlersFalse()
      self:TabRightHandler()
    elseif self.tab_left then
      self.handling_event = true
      self:SetAllHandlersFalse()
      self:TabLeftHandler()
    elseif self.hide then
      self.handling_event = true
      self:SetAllHandlersFalse()
      self:HideHandler()
    elseif self.take_picture then
      self.handling_event = true
      self:SetAllHandlersFalse()
      self:TakePictureHandler()
    end
  end
end
function GeneralCamera:Setup()
  self.PHOn = self:GetState("PHOn")
  self.menu = menu.Menu.New(self, {})
  local generalList = list.List.New(self, {
    MaxFocusableObjectCount = self.PHOn.total_num_buttons,
    ListObjectName = "PhotoModeSettingsList",
    EmptyTextLamsID = lamsIDs.NoSettingsAvailable,
    PreviousEvents = {
      "EVT_Up_Press"
    },
    NextEvents = {
      "EVT_Down_Press"
    }
  })
  generalList:SetSelectedButton(1, false)
  self.menu:SetList("GeneralCameraList", generalList)
  self.fovSlider = slider.Slider.New(self, {
    Name = "FOV",
    TextFormat = "%0.0f\176",
    Sound = "SND_UX_Photo_Mode_Setting_Change_Tick_Horizontal",
    Get = PhotoMode.GetFOV,
    Set = function(fov)
      PhotoMode.SetFOV(fov)
    end,
    MinValue = 17,
    MaxValue = 100,
    Delta = 1,
    DecrementEvents = {
      "EVT_Left_Press"
    },
    IncrementEvents = {
      "EVT_Right_Press"
    },
    Default = function()
      return self.PHOn.original_fov
    end
  })
  self.focalLengthSlider = slider.Slider.New(self, {
    Name = "FocalLengthChange",
    TextFormat = "%0.1fmm",
    Sound = "SND_UX_Photo_Mode_Setting_Change_Tick_Horizontal",
    Get = PhotoMode.GetFocalLength,
    Set = function(fov)
      if PhotoMode.SetFocalLength ~= nil then
        PhotoMode.SetFocalLength(fov)
      end
    end,
    MinValue = 15,
    MaxValue = 120,
    Delta = 1,
    DecrementEvents = {
      "EVT_Left_Press"
    },
    IncrementEvents = {
      "EVT_Right_Press"
    },
    Default = function()
      return self.PHOn.original_focal
    end
  })
  self.cameraRollSlider = slider.Slider.New(self, {
    Name = "cameraRollSlider",
    TextFormat = "%0.0f",
    Sound = "SND_UX_Photo_Mode_Setting_Change_Tick_Horizontal",
    Get = function()
      return PhotoMode.GetRoll()
    end,
    Set = function(v)
      PhotoMode.SetRoll(v)
    end,
    MinValue = -180,
    MaxValue = 180,
    Delta = 1,
    DecrementEvents = {
      "EVT_Left_Press"
    },
    IncrementEvents = {
      "EVT_Right_Press"
    },
    Default = function()
      return 0
    end
  })
  self.settingsInfo = {
    {
      Item = "fovSlider",
      Text = util.GetLAMSMsg(lamsIDs.FieldOfView),
      ActivationList = {
        {
          Item = self.fovSlider
        }
      }
    },
    {
      Item = "focalLengthSlider",
      Text = util.GetLAMSMsg(lamsIDs.FocalLength),
      ActivationList = {
        {
          Item = self.focalLengthSlider
        }
      }
    },
    {
      Item = "cameraRollSlider",
      Text = util.GetLAMSMsg(lamsIDs.CameraRoll),
      ActivationList = {
        {
          Item = self.cameraRollSlider
        }
      }
    }
  }
end
function GeneralCamera:Enter()
  UI.Anim("GeneralCameraTab", consts.AS_Forward, "", self.PHOn.tab_animation_rate, 1)
  self.menu:Activate()
  local photoModeSettingsList = self.menu:GetList("GeneralCameraList")
  local newItemArray = self.settingsInfo
  self.PHOn:RefreshLists(photoModeSettingsList, newItemArray)
  self.fovSlider:Update()
end
function GeneralCamera:Exit()
  UI.Anim("GeneralCameraTab", consts.AS_Forward, "", self.PHOn.tab_animation_rate, 0)
  self.menu:Deactivate(true)
end
function GeneralCamera:LeavePhotomode()
  local settingsList = self.menu:GetList("GeneralCameraList")
  self.PHOn:ExitPhotoModePrompt(settingsList, self.menu, function()
    self:Enter()
  end)
end
function GeneralCamera:ResetCamera()
  local settingsList = self.menu:GetList("GeneralCameraList")
  self.PHOn:SetDefaultCameraPrompt(settingsList, self.menu, function()
    self:Enter()
  end)
end
function GeneralCamera:SetDefaultValues()
  local settingsList = self.menu:GetList("GeneralCameraList")
  self.PHOn:SetDefaultValues(settingsList, self.menu, function()
    self:Enter()
  end)
end
function GeneralCamera:Update()
  self.cameraRollSlider:Update()
  self.focalLengthSlider:Update()
  self.fovSlider:Update()
end
function Shutter:Setup()
  self.inShutter = false
  self.PHOn = self:GetState("PHOn")
  self.menu = menu.Menu.New(self, {})
  self.PHOTO_MODE_TARGETING_INDICATOR = "PHOTO_MODE_TARGETING_INDICATOR"
  self.HEAD_JOINT = "johead1"
  self.localizedDOF = {
    util.GetLAMSMsg(lamsIDs.Off),
    util.GetLAMSMsg(lamsIDs.Custom),
    util.GetLAMSMsg(lamsIDs.Kratos),
    util.GetLAMSMsg(lamsIDs.Atreus),
    util.GetLAMSMsg(lamsIDs.Character)
  }
  local generalList = list.List.New(self, {
    MaxFocusableObjectCount = self.PHOn.total_num_buttons,
    ListObjectName = "PhotoModeSettingsList",
    EmptyTextLamsID = lamsIDs.NoSettingsAvailable,
    PreviousEvents = {
      "EVT_Up_Press"
    },
    NextEvents = {
      "EVT_Down_Press"
    },
    Button_OnGainFocus = function(button)
      self:ToggleShowTheDoFIcons(button)
    end
  })
  generalList:SetSelectedButton(1, false)
  self.menu:SetList("ShutterList", generalList)
  self.dofToggle = toggle.Toggle.New(self, {
    Name = "DoF",
    Settings = function()
      return self:SetupDOFValues()
    end,
    Get = PhotoMode.GetDOF,
    Set = function(mode)
      self:SetCreatureIcon(mode)
      PhotoMode.SetDOF(mode)
      if self.dofFocusDistanceSlider ~= nil then
        self.dofFocusDistanceSlider:Update()
      end
    end,
    PreviousEvents = {
      "EVT_Left_Press"
    },
    NextEvents = {
      "EVT_Right_Press"
    },
    Default = function()
      return 1
    end
  })
  self.dofFocusDistanceSlider = slider.Slider.New(self, {
    Name = "DoFFocusDistance",
    TextFormat = "%0.2f",
    Sound = "SND_UX_Photo_Mode_Setting_Change_Tick_Horizontal",
    Get = PhotoMode.GetDOFFocusDistance,
    Set = function(dist)
      PhotoMode.SetDOFFocusDistance(dist)
      if self.dofToggle ~= nil then
        self.dofToggle:ForceSetValue(1)
      end
    end,
    MinValue = PhotoMode.GetDOFFocusDistanceMin(),
    MaxValue = 100,
    Delta = 0.01,
    MaxDeltaMultiplier = 70,
    DecrementEvents = {
      "EVT_Left_Press"
    },
    IncrementEvents = {
      "EVT_Right_Press"
    },
    Default = function()
      return self.PHOn.original_dofDist
    end
  })
  self.dofFStopSlider = slider.Slider.New(self, {
    Name = "DoFFStop",
    TextFormat = "f/%0.1f",
    Sound = "SND_UX_Photo_Mode_Setting_Change_Tick_Horizontal",
    Get = PhotoMode.GetDOFFStop,
    Set = function(num)
      PhotoMode.SetDOFFStop(num)
    end,
    MinValue = PhotoMode.GetDOFFStopMin(),
    MaxValue = PhotoMode.GetDOFFStopMax(),
    Delta = function(increment)
      local curr = self.dofFStopSlider:get_value()
      local curr_idx = 3 * math.log(math.pow(curr, 2) / 2) / math.log(2)
      curr_idx = math.floor(curr_idx + 0.5)
      local new_idx = curr_idx - 1
      if increment then
        new_idx = curr_idx + 1
      end
      local fstop = math.sqrt(math.pow(2, new_idx / 3 + 1))
      return math.abs(fstop - curr)
    end,
    DecrementEvents = {
      "EVT_Left_Press"
    },
    IncrementEvents = {
      "EVT_Right_Press"
    },
    Default = function()
      return self.PHOn.original_dofFStop
    end
  })
  self.settingsInfo = {
    {
      Item = "dofToggle",
      Text = util.GetLAMSMsg(lamsIDs.DepthOfField),
      ActivationList = {
        {
          Item = self.dofToggle
        }
      }
    },
    {
      Item = "dofFocusDistanceSlider",
      Text = util.GetLAMSMsg(lamsIDs.dofDistance),
      ActivationList = {
        {
          Item = self.dofFocusDistanceSlider
        }
      }
    },
    {
      Item = "dofFStopSlider",
      Text = util.GetLAMSMsg(lamsIDs.dofFstop),
      ActivationList = {
        {
          Item = self.dofFStopSlider
        }
      }
    }
  }
end
function Shutter:Enter()
  self.inShutter = true
  UI.Anim("ShutterTab", consts.AS_Forward, "", self.PHOn.tab_animation_rate, 1)
  self.menu:Activate()
  local photoModeSettingsList = self.menu:GetList("ShutterList")
  local newItemArray = self.settingsInfo
  self.PHOn:RefreshLists(photoModeSettingsList, newItemArray)
  if self.dofFocusDistanceSlider ~= nil then
    self.dofFocusDistanceSlider:SetRange(PhotoMode.GetDOFFocusDistanceMin(), 100)
  end
end
function Shutter:Exit()
  self.inShutter = false
  self:DestroyCreatureIcon()
  UI.Anim("ShutterTab", consts.AS_Forward, "", self.PHOn.tab_animation_rate, 0)
  self.menu:Deactivate(true)
end
function Shutter:ToggleShowTheDoFIcons(button)
  local item = button:get_item()
  if item.Item == "dofToggle" then
    self:ShowCreatureIconAfterHide()
  else
    self:DestroyCreatureIcon()
  end
end
function Shutter:ShouldShowIconAfterPicture()
  local shutterList = self.menu:GetList("ShutterList")
  local selectedItem = shutterList:GetSelectedItem()
  if selectedItem ~= nil and selectedItem.Item == "dofToggle" then
    self:ShowCreatureIconAfterHide()
  end
end
function Shutter:SetupDOFValues()
  local t = {}
  local monster_counter = 0
  local insert_idx = 1
  for _, item in ipairs(PhotoMode.GetDOFSettings()) do
    local idx = item.Type
    local visible = self:IsCreatureVisible(idx)
    if visible then
      local text = self.localizedDOF[idx + 1]
      if idx == 4 then
        monster_counter = monster_counter + 1
        text = text .. " " .. monster_counter
      end
      local new_t = {
        Value = item.Value,
        Text = text,
        Creature = item.Creature
      }
      tablex.FastInsert(t, new_t, insert_idx)
      insert_idx = insert_idx + 1
    end
  end
  return t
end
function Shutter:IsCreatureVisible(creature_type)
  if creature_type < 2 then
    return true
  elseif creature_type == 2 then
    return PhotoMode.GetHeroVisibility()
  elseif creature_type == 3 then
    return PhotoMode.GetSonVisibility() and self.PHOn.isSonAvailable
  elseif creature_type == 4 then
    return PhotoMode.GetEnemyVisibility()
  end
  return false
end
function Shutter:SetAndDestroyCreatureIcon(add, destroy)
  for _, item in ipairs(PhotoMode.GetDOFSettings()) do
    local creature = item.Creature
    if creature ~= nil then
      if add == item.Value and self:IsCreatureVisible(item.Type) then
        mpicon.level.Create(creature, self.PHOTO_MODE_TARGETING_INDICATOR, self.HEAD_JOINT)
      elseif destroy == item.Value then
        mpicon.level.Off(creature, self.PHOTO_MODE_TARGETING_INDICATOR)
      end
    end
  end
end
function Shutter:SetCreatureIcon(value)
  self:SetAndDestroyCreatureIcon(value, PhotoMode.GetDOF())
end
function Shutter:DestroyCreatureIcon()
  self:SetAndDestroyCreatureIcon(nil, PhotoMode.GetDOF())
end
function Shutter:ShowCreatureIconAfterHide()
  self:SetAndDestroyCreatureIcon(PhotoMode.GetDOF())
end
function Shutter:LeavePhotomode()
  local settingsList = self.menu:GetList("ShutterList")
  self.PHOn:ExitPhotoModePrompt(settingsList, self.menu, function()
    self:Enter()
  end)
end
function Shutter:ResetCamera()
  local settingsList = self.menu:GetList("ShutterList")
  self.PHOn:SetDefaultCameraPrompt(settingsList, self.menu, function()
    self:Enter()
  end)
end
function Shutter:SetDefaultValues()
  local settingsList = self.menu:GetList("ShutterList")
  self.PHOn:SetDefaultValues(settingsList, self.menu, function()
    self:Enter()
  end)
end
function Shutter:Update()
  self.dofFStopSlider:Update()
  self.dofFocusDistanceSlider:Update()
end
function Brightness:Setup()
  self.first_time = true
  self.defer_a_frame = true
  self.PHOn = self:GetState("PHOn")
  self.menu = menu.Menu.New(self, {})
  local generalList = list.List.New(self, {
    MaxFocusableObjectCount = self.PHOn.total_num_buttons,
    ListObjectName = "PhotoModeSettingsList",
    EmptyTextLamsID = lamsIDs.NoSettingsAvailable,
    PreviousEvents = {
      "EVT_Up_Press"
    },
    NextEvents = {
      "EVT_Down_Press"
    }
  })
  generalList:SetSelectedButton(1, false)
  self.menu:SetList("BrightnessList", generalList)
  self.filmGrainAmountSlider = slider.Slider.New(self, {
    Name = "FilmGrainAmount",
    TextFormat = "%0.1f",
    Sound = "SND_UX_Photo_Mode_Setting_Change_Tick_Horizontal",
    Get = function()
      return PhotoMode.GetFilmGrainAmount()
    end,
    Set = function(num)
      PhotoMode.SetFilmGrainAmount(num)
    end,
    MinValue = PhotoMode.GetFilmGrainAmountMin(),
    MaxValue = PhotoMode.GetFilmGrainAmountMax(),
    Delta = PhotoMode.GetFilmGrainAmountDelta(),
    DecrementEvents = {
      "EVT_Left_Press"
    },
    IncrementEvents = {
      "EVT_Right_Press"
    },
    Default = function()
      return self.PHOn.original_filmgrain
    end
  })
  self.exposureSlider = slider.Slider.New(self, {
    Name = "exposureSlider",
    TextFormat = "%0.0f",
    Sound = "SND_UX_Photo_Mode_Setting_Change_Tick_Horizontal",
    Get = function()
      return PhotoMode.GetExposure() * 10 / 1
    end,
    Set = function(num)
      local r = num * 1 / 10
      PhotoMode.SetExposure(r)
    end,
    MinValue = -10,
    MaxValue = 10,
    Delta = 1,
    DecrementEvents = {
      "EVT_Left_Press"
    },
    IncrementEvents = {
      "EVT_Right_Press"
    },
    Default = function()
      return 0
    end
  })
  self.lutItems = {
    {
      Value = -1,
      Text = util.GetLAMSMsg(lamsIDs.None)
    },
    {
      Value = 0,
      Text = util.GetLAMSMsg(lamsIDs.Atreus)
    },
    {
      Value = 1,
      Text = util.GetLAMSMsg(lamsIDs.BW)
    },
    {
      Value = 2,
      Text = util.GetLAMSMsg(lamsIDs.Cool)
    },
    {
      Value = 3,
      Text = util.GetLAMSMsg(lamsIDs.Cross)
    },
    {
      Value = 4,
      Text = util.GetLAMSMsg(lamsIDs.Kratos)
    },
    {
      Value = 5,
      Text = util.GetLAMSMsg(lamsIDs.Sepia)
    },
    {
      Value = 6,
      Text = util.GetLAMSMsg(lamsIDs.Vibrant)
    },
    {
      Value = 7,
      Text = util.GetLAMSMsg(lamsIDs.Warm)
    },
    {
      Value = 8,
      Text = util.GetLAMSMsg(lamsIDs.Blurange)
    },
    {
      Value = 9,
      Text = util.GetLAMSMsg(lamsIDs.Old)
    },
    {
      Value = 10,
      Text = util.GetLAMSMsg(lamsIDs.Pastel)
    },
    {
      Value = 11,
      Text = util.GetLAMSMsg(lamsIDs.Silhouette)
    }
  }
  self.LUTToggle = toggle.Toggle.New(self, {
    Name = "LUTToggle",
    Get = function()
      return self.PHOn.currentLUT
    end,
    Set = function(num)
      self:SetLutValue(num)
    end,
    Settings = self.lutItems,
    PreviousEvents = {
      "EVT_Left_Press"
    },
    NextEvents = {
      "EVT_Right_Press"
    },
    Default = function()
      return -1
    end
  })
  self.LUTIntensitySlider = slider.Slider.New(self, {
    Name = "LutIntensity",
    TextFormat = "%0.1f",
    Sound = "SND_UX_Photo_Mode_Setting_Change_Tick_Horizontal",
    Get = function()
      return self.PHOn.currentLUTIntensity
    end,
    Set = function(num)
      self:SetLutIntensity(num)
    end,
    MinValue = 0,
    MaxValue = 10,
    Delta = 0.5,
    DecrementEvents = {
      "EVT_Left_Press"
    },
    IncrementEvents = {
      "EVT_Right_Press"
    },
    Default = function()
      return 10
    end
  })
  self.settingsInfo = {
    {
      Item = "filmGrainAmountSlider",
      Text = util.GetLAMSMsg(lamsIDs.FilmGrain),
      ActivationList = {
        {
          Item = self.filmGrainAmountSlider
        }
      }
    },
    {
      Item = "exposureSlider",
      Text = util.GetLAMSMsg(lamsIDs.Exposure),
      ActivationList = {
        {
          Item = self.exposureSlider
        }
      },
      DisabledInCine = true
    },
    {
      Item = "LUTToggle",
      Text = util.GetLAMSMsg(lamsIDs.Filter),
      ActivationList = {
        {
          Item = self.LUTToggle
        }
      }
    },
    {
      Item = "LutIntensity",
      Text = util.GetLAMSMsg(lamsIDs.FilterIntensity),
      ActivationList = {
        {
          Item = self.LUTIntensitySlider
        }
      }
    }
  }
end
function Brightness:Enter()
  self.defer_a_frame = true
  UI.Anim("BrightnessTab", consts.AS_Forward, "", self.PHOn.tab_animation_rate, 1)
  self.menu:Activate()
  local photoModeSettingsList = self.menu:GetList("BrightnessList")
  local newItemArray = self.settingsInfo
  self.PHOn:RefreshLists(photoModeSettingsList, newItemArray)
  self.PHOn:DetermineIfItemsAreDisabled(photoModeSettingsList, newItemArray, self.PHOn.inCinematic, "DisabledInCine")
end
function Brightness:Exit()
  self.PHOn:EnableAllValuesOnExit(self.menu:GetList("BrightnessList"), self.settingsInfo)
  UI.Anim("BrightnessTab", consts.AS_Forward, "", self.PHOn.tab_animation_rate, 0)
  self.menu:Deactivate(true)
end
function Brightness:SetLutValue(num)
  if UI.SetLutIndex ~= nil then
    UI.SetLutIndex(num)
  end
  self.PHOn.currentLUT = num
end
function Brightness:SetLutIntensity(v)
  local r = v * 1 / 10
  if UI.SetLutWeight ~= nil then
    UI.SetLutWeight(r)
  end
  self.PHOn.currentLUTIntensity = v
end
function Brightness:LeavePhotomode()
  local settingsList = self.menu:GetList("BrightnessList")
  self.PHOn:ExitPhotoModePrompt(settingsList, self.menu, function()
    self:Enter()
  end)
end
function Brightness:ResetCamera()
  local settingsList = self.menu:GetList("BrightnessList")
  self.PHOn:SetDefaultCameraPrompt(settingsList, self.menu, function()
    self:Enter()
  end)
end
function Brightness:SetDefaultValues()
  local settingsList = self.menu:GetList("BrightnessList")
  self.PHOn:SetDefaultValues(settingsList, self.menu, function()
    self:Enter()
  end)
end
function Brightness:Update()
  if self.defer_a_frame then
    self.defer_a_frame = false
  elseif self.first_time then
    self.PHOn.original_filmgrain = PhotoMode.GetFilmGrainAmount()
    self.first_time = false
  end
  self.filmGrainAmountSlider:Update()
  self.exposureSlider:Update()
end
function Vignette:Setup()
  self.PHOn = self:GetState("PHOn")
  self.menu = menu.Menu.New(self, {})
  local generalList = list.List.New(self, {
    MaxFocusableObjectCount = self.PHOn.total_num_buttons,
    ListObjectName = "PhotoModeSettingsList",
    EmptyTextLamsID = lamsIDs.NoSettingsAvailable,
    PreviousEvents = {
      "EVT_Up_Press"
    },
    NextEvents = {
      "EVT_Down_Press"
    }
  })
  generalList:SetSelectedButton(1, false)
  self.menu:SetList("VignetteList", generalList)
  self.vignetteToggle = toggle.Toggle.New(self, {
    Name = "Vignette",
    Get = PhotoMode.GetVignette,
    Set = function(num)
      PhotoMode.SetVignette(num)
    end,
    PreviousEvents = {
      "EVT_Left_Press"
    },
    NextEvents = {
      "EVT_Right_Press"
    },
    Default = function()
      return self.PHOn.original_vignette
    end
  })
  self.vignetteBrightnessSlider = slider.Slider.New(self, {
    Name = "VignetteBrightness",
    TextFormat = "%0.1f",
    Sound = "SND_UX_Photo_Mode_Setting_Change_Tick_Horizontal",
    Get = function()
      return (1 - PhotoMode.GetVignetteBrightness() / 0.7) * 10
    end,
    Set = function(num)
      local r = (num / 10 - 1) * -0.7
      PhotoMode.SetVignetteBrightness(r)
    end,
    MinValue = 0,
    MaxValue = 10,
    Delta = 0.5,
    DecrementEvents = {
      "EVT_Left_Press"
    },
    IncrementEvents = {
      "EVT_Right_Press"
    },
    Default = function()
      return (1 - self.PHOn.original_vignetteBrightness / 0.7) * 10
    end
  })
  self.vignetteFalloffSlider = slider.Slider.New(self, {
    Name = "VignetteBrightness",
    TextFormat = "%0.1f",
    Sound = "SND_UX_Photo_Mode_Setting_Change_Tick_Horizontal",
    Get = function()
      local r = (PhotoMode.GetVignetteFalloff() - 0.1) * 10 / 0.9
      return r
    end,
    Set = function(num)
      local r = num * 0.9 / 10 + 0.1
      PhotoMode.SetVignetteFalloff(r)
    end,
    MinValue = 0,
    MaxValue = 10,
    Delta = 0.5,
    DecrementEvents = {
      "EVT_Left_Press"
    },
    IncrementEvents = {
      "EVT_Right_Press"
    },
    Default = function()
      return (self.PHOn.original_vignetteFalloff - 0.1) * 10 / 0.9
    end
  })
  self.borders = {
    {
      Value = 1,
      Text = util.GetLAMSMsg(lamsIDs.None)
    },
    {
      Value = 2,
      Text = util.GetLAMSMsg(lamsIDs.FilmGate2_1),
      GO = "FilmGate2_1"
    },
    {
      Value = 3,
      Text = util.GetLAMSMsg(lamsIDs.FilmGate_Cine),
      GO = "FilmGateCine"
    },
    {
      Value = 4,
      Text = util.GetLAMSMsg(lamsIDs.FilmGate4_3),
      GO = "FilmGate4_3"
    },
    {
      Value = 5,
      Text = util.GetLAMSMsg(lamsIDs.FilmGateSquare),
      GO = "FilmGateSquare"
    },
    {
      Value = 6,
      Text = util.GetLAMSMsg(lamsIDs.FilmGate9_16),
      GO = "FilmGate9_16"
    },
    {
      Value = 7,
      Text = util.GetLAMSMsg(lamsIDs.NordicBraidBlack),
      GO16x10 = "NordicBraidBlack16x10",
      GO16x9 = "NordicBraidBlack16x9",
      GO21x9 = "NordicBraidBlack21x9"
    },
    {
      Value = 8,
      Text = util.GetLAMSMsg(lamsIDs.NordicBraidWhite),
      GO16x10 = "NordicBraidWhite16x10",
      GO16x9 = "NordicBraidWhite16x9",
      GO21x9 = "NordicBraidWhite21x9"
    },
    {
      Value = 9,
      Text = util.GetLAMSMsg(lamsIDs.PinstripeBlack),
      GO16x10 = "PinstripeBlack16x10",
      GO16x9 = "PinstripeBlack16x9",
      GO21x9 = "PinstripeBlack21x9"
    },
    {
      Value = 10,
      Text = util.GetLAMSMsg(lamsIDs.Pinstripe),
      GO16x10 = "Pinstripe16x10",
      GO16x9 = "Pinstripe16x9",
      GO21x9 = "Pinstripe21x9"
    },
    {
      Value = 11,
      Text = util.GetLAMSMsg(lamsIDs.PinstripeClear),
      GO16x10 = "PinstripeClear16x10",
      GO16x9 = "PinstripeClear16x9",
      GO21x9 = "PinstripeClear21x9"
    }
  }
  self.borderToggle = toggle.Toggle.New(self, {
    Name = "BorderToggle",
    Get = function()
      return self.PHOn.currentBorder
    end,
    Set = function(v)
      self.PHOn.currentBorder = v
      self:ShowSelectedBorderGO(v, self.borders)
    end,
    Settings = self.borders,
    Default = function()
      return 1
    end,
    PreviousEvents = {
      "EVT_Left_Press"
    },
    NextEvents = {
      "EVT_Right_Press"
    }
  })
  self.gow_logos = {
    {
      Value = 1,
      Text = util.GetLAMSMsg(lamsIDs.None)
    },
    {
      Value = 2,
      Text = util.GetLAMSMsg(lamsIDs.TopLeft),
      GO = "GOW_logo_TopLeft"
    },
    {
      Value = 3,
      Text = util.GetLAMSMsg(lamsIDs.TopCenter),
      GO = "GOW_logo_TopCenter"
    },
    {
      Value = 4,
      Text = util.GetLAMSMsg(lamsIDs.TopRight),
      GO = "GOW_logo_TopRight"
    },
    {
      Value = 5,
      Text = util.GetLAMSMsg(lamsIDs.BottomLeft),
      GO = "GOW_logo_BotLeft"
    },
    {
      Value = 6,
      Text = util.GetLAMSMsg(lamsIDs.BottomCenter),
      GO = "GOW_logo_BotCenter"
    },
    {
      Value = 7,
      Text = util.GetLAMSMsg(lamsIDs.BottomRight),
      GO = "GOW_logo_BotRight"
    }
  }
  self.gowLogoToggle = toggle.Toggle.New(self, {
    Name = "BorderToggle",
    Get = function()
      return self.PHOn.currentGOWBorder
    end,
    Set = function(v)
      self.PHOn.currentGOWBorder = v
      self:ShowSelectedBorderGO(v, self.gow_logos)
    end,
    Settings = self.gow_logos,
    Default = function()
      return 1
    end,
    PreviousEvents = {
      "EVT_Left_Press"
    },
    NextEvents = {
      "EVT_Right_Press"
    }
  })
  self.settingsInfo = {
    {
      Item = "vignetteToggle",
      Text = util.GetLAMSMsg(lamsIDs.Vignette),
      ActivationList = {
        {
          Item = self.vignetteToggle
        }
      }
    },
    {
      Item = "vignetteBrightnessSlider",
      Text = util.GetLAMSMsg(lamsIDs.VignetteIntensity),
      ActivationList = {
        {
          Item = self.vignetteBrightnessSlider
        }
      }
    },
    {
      Item = "vignetteFalloffSlider",
      Text = util.GetLAMSMsg(lamsIDs.VignetteFalloff),
      ActivationList = {
        {
          Item = self.vignetteFalloffSlider
        }
      }
    },
    {
      Item = "BorderToggle",
      Text = util.GetLAMSMsg(lamsIDs.Border),
      ActivationList = {
        {
          Item = self.borderToggle
        }
      }
    },
    {
      Item = "logoToggle",
      Text = util.GetLAMSMsg(lamsIDs.GOW_Logo),
      ActivationList = {
        {
          Item = self.gowLogoToggle
        }
      }
    }
  }
end
function Vignette:Enter()
  UI.Anim("VignetteTab", consts.AS_Forward, "", self.PHOn.tab_animation_rate, 1)
  self.menu:Activate()
  local photoModeSettingsList = self.menu:GetList("VignetteList")
  local newItemArray = self.settingsInfo
  self.PHOn:RefreshLists(photoModeSettingsList, newItemArray)
end
function Vignette:Exit()
  UI.Anim("VignetteTab", consts.AS_Forward, "", self.PHOn.tab_animation_rate, 0)
  self.menu:Deactivate(true)
end
function Vignette:ShowSelectedBorderGO(value, border_list)
  local aspect = game.GetRenderWidth() / game.GetRenderHeight()
  for i, item in ipairs(border_list) do
    local go = item.GO
    if go == nil then
      if 2.3 < aspect then
        go = item.GO21x9
        if go ~= nil then
          util.Hide(item.GO16x9)
          util.Hide(item.GO16x10)
        end
      elseif 1.7 < aspect then
        go = item.GO16x9
        if go ~= nil then
          util.Hide(item.GO21x9)
          util.Hide(item.GO16x10)
        end
      else
        go = item.GO16x10
        if go ~= nil then
          util.Hide(item.GO21x9)
          util.Hide(item.GO16x9)
        end
      end
    end
    if go ~= nil then
      if i == value then
        util.Show(go)
      else
        util.Hide(go)
      end
    end
  end
end
function Vignette:LeavePhotomode()
  local settingsList = self.menu:GetList("VignetteList")
  self.PHOn:ExitPhotoModePrompt(settingsList, self.menu, function()
    self:Enter()
  end)
end
function Vignette:ResetCamera()
  local settingsList = self.menu:GetList("VignetteList")
  self.PHOn:SetDefaultCameraPrompt(settingsList, self.menu, function()
    self:Enter()
  end)
end
function Vignette:SetDefaultValues()
  local settingsList = self.menu:GetList("VignetteList")
  self.PHOn:SetDefaultValues(settingsList, self.menu, function()
    self:Enter()
  end)
end
function Vignette:Update()
  self.vignetteBrightnessSlider:Update()
  self.vignetteFalloffSlider:Update()
end
function Character:Setup()
  self.PHOn = self:GetState("PHOn")
  self.menu = menu.Menu.New(self, {})
  local generalList = list.List.New(self, {
    MaxFocusableObjectCount = self.PHOn.total_num_buttons,
    ListObjectName = "PhotoModeSettingsList",
    EmptyTextLamsID = lamsIDs.NoSettingsAvailable,
    PreviousEvents = {
      "EVT_Up_Press"
    },
    NextEvents = {
      "EVT_Down_Press"
    }
  })
  generalList:SetSelectedButton(1, false)
  self.menu:SetList("CharacterList", generalList)
  self.kratosShowToggle = toggle.Toggle.New(self, {
    Name = "KratosToggle",
    Get = function()
      if PhotoMode.GetHeroVisibility ~= nil then
        return PhotoMode.GetHeroVisibility()
      end
      return true
    end,
    Set = function(v)
      self:ShowKratos(v)
    end,
    Settings = {
      {
        Value = false,
        Text = util.GetLAMSMsg(lamsIDs.Hide)
      },
      {
        Value = true,
        Text = util.GetLAMSMsg(lamsIDs.Show)
      }
    },
    Default = function()
      return true
    end,
    PreviousEvents = {
      "EVT_Left_Press"
    },
    NextEvents = {
      "EVT_Right_Press"
    }
  })
  self.sonShowToggle = toggle.Toggle.New(self, {
    Name = "sonShowToggle",
    Get = function()
      if PhotoMode.GetSonVisibility ~= nil then
        return PhotoMode.GetSonVisibility()
      end
      return true
    end,
    Set = function(v)
      self:ShowSon(v)
    end,
    Settings = {
      {
        Value = false,
        Text = util.GetLAMSMsg(lamsIDs.Hide)
      },
      {
        Value = true,
        Text = util.GetLAMSMsg(lamsIDs.Show)
      }
    },
    Default = function()
      return true
    end,
    PreviousEvents = {
      "EVT_Left_Press"
    },
    NextEvents = {
      "EVT_Right_Press"
    }
  })
  self.characterShowToggle = toggle.Toggle.New(self, {
    Name = "characterShowToggle",
    Get = function()
      if PhotoMode.GetEnemyVisibility ~= nil then
        return PhotoMode.GetEnemyVisibility()
      end
      return true
    end,
    Set = function(v)
      self:ShowCharacters(v)
    end,
    Settings = {
      {
        Value = false,
        Text = util.GetLAMSMsg(lamsIDs.Hide)
      },
      {
        Value = true,
        Text = util.GetLAMSMsg(lamsIDs.Show)
      }
    },
    Default = function()
      return true
    end,
    PreviousEvents = {
      "EVT_Left_Press"
    },
    NextEvents = {
      "EVT_Right_Press"
    }
  })
  local face_table = self:BuildFacesTable()
  self.kratosFaceToggle = toggle.Toggle.New(self, {
    Name = "kratosFaceToggle",
    Get = function()
      return self.PHOn.currentKratosFace
    end,
    Set = function(v)
      self.PHOn.currentKratosFace = v
      self:SetCharacterFace(self.PHOn.player, v)
    end,
    Settings = face_table,
    Default = function()
      return -1
    end,
    PreviousEvents = {
      "EVT_Left_Press"
    },
    NextEvents = {
      "EVT_Right_Press"
    }
  })
  self.atreusFaceToggle = toggle.Toggle.New(self, {
    Name = "atreusFaceToggle",
    Get = function()
      return self.PHOn.currentAtreusFace
    end,
    Set = function(v)
      self.PHOn.currentAtreusFace = v
      self:SetCharacterFace(self.PHOn.son, v)
    end,
    Settings = face_table,
    Default = function()
      return -1
    end,
    PreviousEvents = {
      "EVT_Left_Press"
    },
    NextEvents = {
      "EVT_Right_Press"
    }
  })
  self.settingsInfo = {
    {
      Item = "KratosToggle",
      Text = util.GetLAMSMsg(lamsIDs.Kratos),
      ActivationList = {
        {
          Item = self.kratosShowToggle
        }
      },
      DisabledInCine = true
    },
    {
      Item = "sonShowToggle",
      Text = util.GetLAMSMsg(lamsIDs.Atreus),
      ActivationList = {
        {
          Item = self.sonShowToggle
        }
      },
      DisabledInCine = true,
      DisableSon = true
    },
    {
      Item = "characterShowToggle",
      Text = util.GetLAMSMsg(lamsIDs.OtherCharacters),
      ActivationList = {
        {
          Item = self.characterShowToggle
        }
      },
      DisabledInCine = true
    },
    {
      Item = "kratosFaceToggle",
      Text = util.GetLAMSMsg(lamsIDs.KratosFace),
      ActivationList = {
        {
          Item = self.kratosFaceToggle
        }
      },
      DisabledFaces = true
    },
    {
      Item = "atreusFaceToggle",
      Text = util.GetLAMSMsg(lamsIDs.AtreusFace),
      ActivationList = {
        {
          Item = self.atreusFaceToggle
        }
      },
      DisabledFaces = true,
      DisableSon = true
    }
  }
end
function Character:Enter()
  UI.Anim("CharacterTab", consts.AS_Forward, "", self.PHOn.tab_animation_rate, 1)
  self.menu:Activate()
  local photoModeSettingsList = self.menu:GetList("CharacterList")
  local newItemArray = self.settingsInfo
  self.PHOn:RefreshLists(photoModeSettingsList, newItemArray)
  self.PHOn:DetermineIfItemsAreDisabled(photoModeSettingsList, newItemArray, self.PHOn.inCinematic, "DisabledInCine")
  self.PHOn:DetermineIfItemsAreDisabled(photoModeSettingsList, newItemArray, not self.PHOn.faceChangeEnabled, "DisabledFaces")
  self.PHOn:DetermineIfItemsAreDisabled(photoModeSettingsList, newItemArray, not self.PHOn.isSonAvailable, "DisableSon")
end
function Character:Exit()
  UI.Anim("CharacterTab", consts.AS_Forward, "", self.PHOn.tab_animation_rate, 0)
  self.PHOn:EnableAllValuesOnExit(self.menu:GetList("CharacterList"), self.settingsInfo)
  self.menu:Deactivate(true)
end
function Character:SetCharacterFace(char, face)
  if char.SetPhotoModeFace ~= nil then
    char:SetPhotoModeFace(face)
  end
end
function Character:BuildFacesTable()
  local t = {
    {
      Value = -1,
      Text = util.GetLAMSMsg(lamsIDs.Default)
    }
  }
  local face = util.GetLAMSMsg(lamsIDs.Face) .. " "
  for i = 1, 16 do
    local f = {
      Value = i - 1,
      Text = face .. i
    }
    tablex.FastInsert(t, f, i + 1)
  end
  return t
end
function Character:ForceAllCharactersOn()
  self:ShowKratos(true)
  self:ShowSon(true)
  self:ShowCharacters(true)
end
function Character:ShowKratos(value)
  if not value and PhotoMode.GetDOF() == 2 then
    PhotoMode.SetDOF(1)
  end
  if PhotoMode.SetHeroVisibility ~= nil then
    PhotoMode.SetHeroVisibility(value)
  end
end
function Character:ShowSon(value)
  if not value and PhotoMode.GetDOF() == 3 then
    PhotoMode.SetDOF(1)
  end
  if PhotoMode.SetSonVisibility ~= nil then
    PhotoMode.SetSonVisibility(value)
  end
end
function Character:ShowCharacters(value)
  if not value and PhotoMode.GetDOF() > 3 then
    PhotoMode.SetDOF(1)
  end
  if PhotoMode.SetEnemyVisibility ~= nil then
    PhotoMode.SetEnemyVisibility(value)
  end
end
function Character:LeavePhotomode()
  local settingsList = self.menu:GetList("CharacterList")
  self.PHOn:ExitPhotoModePrompt(settingsList, self.menu, function()
    self:Enter()
  end)
end
function Character:ResetCamera()
  local settingsList = self.menu:GetList("CharacterList")
  self.PHOn:SetDefaultCameraPrompt(settingsList, self.menu, function()
    self:Enter()
  end)
end
function Character:SetDefaultValues()
  local settingsList = self.menu:GetList("CharacterList")
  self.PHOn:SetDefaultValues(settingsList, self.menu, function()
    self:Enter()
  end)
end
function HideUI:Setup()
  self.PHOn = self:GetState("PHOn")
  self.shutter = self:GetState("Shutter")
  self.go_UI_Elements = util.GetUiObjByName("UI_Elements")
  self.inState = false
  self.button_pressed = false
  self.exiting = false
end
function HideUI:Enter()
  self.button_pressed = false
  self.exiting = false
  self.inState = true
  Audio.PlaySound("SND_UX_Photo_Mode_Bottom_Menu_Hide_UI")
  self.go_UI_Elements:Hide()
end
function HideUI:Exit()
  self.inState = false
  Audio.PlaySound("SND_UX_Photo_Mode_Bottom_Menu_Hide_UI")
  self.go_UI_Elements:Show()
end
function HideUI:Update()
  if self.button_pressed and not self.exiting then
    self.exiting = true
    self.PHOn:SetNewMenu()
  end
end
function HideUI:EVT_Back_Release()
  if not self.exiting then
    self.button_pressed = true
  end
end
HideUI.EVT_Advance_Release = HideUI.EVT_Back_Release
HideUI.EVT_Options_Release = HideUI.EVT_Back_Release
HideUI.EVT_Square_Release = HideUI.EVT_Back_Release
HideUI.EVT_Triangle_Release = HideUI.EVT_Back_Release
HideUI.EVT_Up_Release = HideUI.EVT_Back_Release
HideUI.EVT_Right_Release = HideUI.EVT_Back_Release
HideUI.EVT_Down_Release = HideUI.EVT_Back_Release
HideUI.EVT_Left_Release = HideUI.EVT_Back_Release
HideUI.EVT_L1_Release = HideUI.EVT_Back_Release
HideUI.EVT_R1_Release = HideUI.EVT_Back_Release
function PhotoModeHUD:OnSaveCheckpoint(tab)
end
function PhotoModeHUD:OnRestoreCheckpoint(tab)
end
