local classlib = require("core.class")
local tablex = require("core.tablex")
local animationUtil = require("ui.animationUtil")
local bestiaryUtil = require("ui.bestiaryUtil")
local buttonUtil = require("ui.buttonUtil")
local consts = require("ui.consts")
local fsm = require("ui.fsm")
local lamsIDs = require("ui.lamsConsts")
local list = require("ui.list")
local menu = require("ui.menu")
local pickupConsts = require("ui.pickupConsts")
local questUtil = require("ui.questUtil")
local resourceConsts = require("ui.resourceConsts")
local resourceUtil = require("ui.resourceUtil")
local util = require("ui.util")
local UI = game.UI
local QuestManager = game.QuestManager
local SetText = UI.SetText
local GetTextHandle = util.GetTextHandle
local Audio = game.Audio
local JournalMenu = classlib.Class("JournalMenu", fsm.UIState)
local JournalOff = JournalMenu:StateClass("JournalOff", fsm.UIState)
local JournalOn = JournalMenu:StateClass("JournalOn", fsm.UIState)
local Lore = JournalMenu:StateClass("Lore", fsm.UIState)
local Bestiary = JournalMenu:StateClass("Bestiary", fsm.UIState)
local journalMenu = JournalMenu.New("journalMenu", {
  JournalOff,
  JournalOn,
  {Lore, Bestiary}
})
function JournalMenu:Setup()
  self.pauseMenuOpened = false
end
function JournalMenu:Enter()
  self:WantPadEvents(true)
  self:TurnOff()
end
function JournalMenu:TurnOff()
  self:Goto("JournalOff")
end
function JournalMenu:EVT_PAUSE_MENU_OPEN()
  self.pauseMenuOpened = true
end
JournalMenu.EVT_GAME_OVER = JournalMenu.TurnOff
JournalMenu.EVT_Restart = JournalMenu.TurnOff
function JournalOff:Enter()
  util.Hide("sonJournal_Main")
  self.journalOn = self:GetState("JournalOn")
end
function JournalOff:Exit()
  util.Hide("sonJournal_Main")
end
function JournalOff:EVT_TURN_ON_JOURNAL_MENU(instructionEntries, instructionArgs)
  self.journalOn.menu:set_instructionEntries(instructionEntries)
  self.journalOn.menu:set_instructionArgs(instructionArgs)
  self:Goto("JournalOn")
end
function JournalOn:Setup()
  self.menu = menu.Menu.New(self, {})
  self.journalMenu = self:GetState("journalMenu")
  self.menu:SetupSubmenuList(consts.inworldMenu_SubmenuList, nil, nil)
  self.goSonJournal_Main = util.GetUiObjByName("sonJournal_Main")
  self.goPageDecals = self.goSonJournal_Main:FindSingleGOByName("pageDecals")
  self.goJournalLightGroup = util.GetUiObjByName("JournalLight_Grp")
  self.goJournalGroup = util.GetUiObjByName("Journal_Group")
  self.goSharedInfo = self.goJournalGroup:FindSingleGOByName("SharedInfo")
  self.thSharedInfoTitle = util.GetTextHandle(self.goSharedInfo, "Title")
  self.thSharedInfoDesc = util.GetTextHandle(self.goSharedInfo, "Description")
  UI.Anim(self.goSonJournal_Main, consts.AS_Forward, "pageTurn03", 0, 1)
  self.goSonJournal_Main:Hide()
  self.goJournalLightGroup:Hide()
end
function JournalOn:Enter()
  self.menu:Activate()
  util.Show("Journal")
  self.goSonJournal_Main:Show()
  self.goJournalLightGroup:Show()
  self.goJournalGroup:Show()
  UI.WorldUIRender("JournalMenuCamera")
  local submenuList = self.menu:GetList(consts.inworldMenu_SubmenuList)
  if self.journalMenu.pauseMenuOpened == true then
    local init_useOnGainFocus = false
    submenuList:SetSelectedButton(1, init_useOnGainFocus)
  end
  local newItemArray = self:GetSubStateNames()
  local showList = true
  local useOnGainFocus = not self.menu:HasInstructionEntryForMenuState()
  local itemDetermineFocusabilityFunc, getDisplayNameFunc
  self.menu:RefreshSubmenuList(submenuList, newItemArray, showList, useOnGainFocus, itemDetermineFocusabilityFunc, getDisplayNameFunc)
  self.menu:ExecuteInstructions()
  local instant = true
  animationUtil.ResetCamera(pickupConsts.TAG_PICKUP_KRATOS, instant)
  animationUtil.MoveWorldUIMainList("ListJoint_Journal")
  if self:HaveTimer("setShowing") then
    self:StopTimer("setShowing")
  end
  if self:HaveTimer("hideDecal") then
    self:StopTimer("hideDecal")
  end
  self.showing = false
  self.goPageDecals:Show()
  self.goPageDecals:SetMaterialSwap("")
  UI.Anim(self.goPageDecals, consts.AS_Reset, "", 0, 1)
end
function JournalOn:Exit()
  self.goSonJournal_Main:Hide()
  self.goJournalLightGroup:Hide()
  self.goJournalGroup:Hide()
  self.menu:Deactivate(true)
  Audio.StopSound("SND_UX_Pause_Menu_Page_Write")
  if self:HaveTimer("setShowing") then
    self:StopTimer("setShowing")
  end
  if self:HaveTimer("delayedFadeIn") then
    self:StopTimer("delayedFadeIn")
  end
  if self:HaveTimer("fadeIn") then
    self:StopTimer("fadeIn")
  end
  UI.Anim(self.goPageDecals, consts.AS_Reverse_NoReset, "", 2, 0, 1)
  self.showing = false
  self.previousMaterial = nil
  self:StartTimer("hideDecal", 0.5, function()
    self.goPageDecals:Hide()
  end)
end
function JournalOn:UpdateAccessibilityScaling()
  if UI.GetAccessibilityScaling == nil then
    return
  end
  self.goSonJournal_Main:SetLocalPosition(engine.Vector.New(-2.156 + UI.GetAccessibilityScaling(0, 0.05), 1.29, 6.41))
  self.goSonJournal_Main:SetScale(UI.GetAccessibilityScaling(5.87, 6.5))
end
function JournalOn:GetSubStateNames()
  return consts.SUB_STATES.Journal
end
function JournalOn:SubmenuList_Button_Update(button)
  local alphaValue = 1
  local fadeTime = 0
  button:AlphaFade(alphaValue, fadeTime)
  button:SetIcon(button:get_item())
  button:UpdateNewIcon(nil)
end
function JournalOn:SubmenuList_Button_OnGainFocus(button)
  local currentItem = button:get_item()
  local subState = self:GetState(currentItem)
  local mainList = subState.menu:GetList("MainList")
  self.menu:SetSubmenuListLabelText(consts.inworldMenu_SubmenuList, "")
  mainList:SetHeaderText(util.GetLAMSMsg(lamsIDs[currentItem]))
  subState.menu:set_instructionEntries(self.menu:get_instructionEntries())
  self:Goto(currentItem)
end
function JournalOn:SubmenuList_AfterDelay(list, delayStartButtonIndex, currButtonIndex)
end
function JournalOn:UpdateFooterButtonPrompt(currMenu)
  local showTouchPadMap = game.IsMapAvailable() and game.build.GOLD_VERSION ~= 0
  currMenu:UpdateFooterButton("Map", showTouchPadMap)
  currMenu:UpdateFooterButton("Weapon", true)
  currMenu:UpdateFooterButton("Skill", true)
  currMenu:UpdateFooterButton("Quest", true)
  currMenu:UpdateFooterButton("Settings", true)
  currMenu:UpdateFooterButton("Exit", true)
  currMenu:UpdateFooterButtonText()
end
function JournalOn:EVT_TURN_OFF_JOURNAL_MENU()
  self:Goto("JournalOff")
end
function JournalOn:delayedFadeIn()
  if self.goPageDecals:CheckMaterialFullyStreamed() then
    self.goPageDecals:Show()
    Audio.PlaySound("SND_UX_Pause_Menu_Page_Write")
    UI.Anim(self.goPageDecals, consts.AS_Forward, "", 1.5, 1, 0)
    self:StartTimer("setShowing", 0.6666667, function()
      self.showing = true
    end)
  else
    self.goPageDecals:RequestLoadHighMips(true)
    self:RestartTimer("delayedFadeIn")
  end
end
function JournalOn:JournalSetMaterialSwap(baseSwapMaterial, alwaysSwitch)
  if self.previousMaterial == baseSwapMaterial and not alwaysSwitch then
    return
  end
  if baseSwapMaterial == nil then
    baseSwapMaterial = ""
  end
  self.previousMaterial = baseSwapMaterial
  local fadeIn = function()
    self.goPageDecals:Hide()
    self.goPageDecals:SetMaterialSwap(baseSwapMaterial)
    self.goPageDecals:RequestLoadHighMips(true)
    if not self:HaveTimer("delayedFadeIn") then
      self:StartTimer("delayedFadeIn", 0, function()
        self:delayedFadeIn()
      end)
    else
      self:RestartTimer("delayedFadeIn")
    end
  end
  if self:HaveTimer("setShowing") then
    self:StopTimer("setShowing")
  end
  if self:HaveTimer("delayedFadeIn") then
    self:StopTimer("delayedFadeIn")
  end
  if self:HaveTimer("fadeIn") then
    self:StopTimer("fadeIn")
  end
  UI.Anim(self.goPageDecals, consts.AS_Reverse_NoReset, "", 2, 0, 1)
  if self.showing then
    if not self:HaveTimer("clearShowing") then
      self:StartTimer("clearShowing", 0.5, function()
        self.showing = false
      end)
    end
    self:StartTimer("fadeIn", 0.5, fadeIn)
  else
    self:StartTimer("fadeIn", 0.2, fadeIn)
  end
end
function Lore:Setup()
  self.journalOn = self:GetState("JournalOn")
  self.menu = menu.Menu.New(self, {
    FooterButtonInfo = {
      {
        Item = "Map",
        Text = "[TouchPad] " .. util.GetLAMSMsg(lamsIDs.Map),
        EventHandlers = {
          {
            Events = {
              "EVT_TouchPad_Release"
            },
            Handler = function()
              self:SendEventToUIFsm("globalMenu", "EVT_GO_TO_MAP")
            end
          }
        }
      },
      {
        Item = "Weapon",
        Text = "",
        EventHandlers = {
          {
            Events = {
              "EVT_GO_TO_WEAPON_MENU"
            },
            Handler = function()
              self:SendEventToUIFsm("globalMenu", "EVT_GO_TO_WEAPON")
            end
          }
        }
      },
      {
        Item = "Skill",
        Text = "",
        EventHandlers = {
          {
            Events = {
              "EVT_GO_TO_SKILL_TREE_MENU"
            },
            Handler = function()
              self:SendEventToUIFsm("globalMenu", "EVT_GO_TO_SKILL_TREE")
            end
          }
        }
      },
      {
        Item = "Quest",
        Text = "",
        EventHandlers = {
          {
            Events = {
              "EVT_GO_TO_QUEST_MENU"
            },
            Handler = function()
              self:SendEventToUIFsm("globalMenu", "EVT_GO_TO_QUEST")
            end
          }
        }
      },
      {
        Item = "Settings",
        Text = "[TriangleButton] " .. util.GetLAMSMsg(lamsIDs.Options),
        EventHandlers = {
          {
            Events = {
              "EVT_Triangle_Release"
            },
            Handler = function()
              self:SendEventToUIFsm("globalMenu", "EVT_OPEN_SETTINGS_MENU")
            end
          }
        }
      },
      {
        Item = "Exit",
        Text = "[BackButton] " .. util.GetLAMSMsg(lamsIDs.Exit),
        EventHandlers = {
          {
            Events = {
              "EVT_Back_Release",
              "EVT_Options_Release"
            },
            Handler = function()
              self:SendEventToUIFsm("globalMenu", "EVT_TURN_OFF_GLOBAL_MENU")
            end
          }
        }
      }
    }
  })
  local storyList = list.List.New(self, {
    MaxFocusableObjectCount = 10,
    ListObjectName = "PickupList",
    EmptyTextLamsID = lamsIDs.NoLoreAvailable,
    HoverToSelect = true,
    DelayTime = 0.33,
    AfterDelay = function(list, delayStartButtonIndex, currButtonIndex)
    end,
    Button_Update = function(button)
      self:MainList_Button_Update(button)
    end,
    Button_OnGainFocus = function(button)
      self:MainList_Button_OnGainFocus(button)
    end,
    Button_ItemCompare = function(item, otherItem)
      local a = type(item) == "string" and item or item.Name
      local b = type(otherItem) == "string" and otherItem or otherItem.Name
      return a == b
    end
  })
  self.menu:SetList("MainList", storyList)
  storyList:SetSelectedButton(1, false)
end
function Lore:UpdateAccessibilityScaling()
  if UI.GetAccessibilityScaling == nil then
    return
  end
  self.journalOn:UpdateAccessibilityScaling()
  self.journalOn.goSharedInfo:SetLocalPosition(engine.Vector.New(-10 + UI.GetAccessibilityScaling(0, -1.95), UI.GetAccessibilityScaling(0, 1), 0))
  self.journalOn.goSharedInfo:SetScale(UI.GetAccessibilityScaling(1, 1.15))
end
function Lore:Enter()
  self.menu:Activate()
  self:UpdateAccessibilityScaling()
  UI.SetText(self.journalOn.thSharedInfoTitle, util.GetLAMSMsg(lamsIDs.Lore))
  UI.SetText(self.journalOn.thSharedInfoDesc, "")
  local resources = {}
  local index = 1
  for i = 1, #resourceConsts.RESOURCE_LORE_CATEGORY_FLAGS do
    local category = resourceConsts.RESOURCE_LORE_CATEGORY_FLAGS[i]
    if category == "Tryptich" then
      local items = resourceUtil.GetResources(resourceConsts.WALLET_KRATOS, {category}, {})
      if 0 < #items then
        resources[index] = {Name = category, ButtonType = "category"}
        index = index + 1
        local postboat = {}
        for j = 1, #items do
          local loreName = items[j]
          if string.sub(loreName, -9) == "_PostBoat" then
            postboat[string.sub(loreName, 1, string.len(loreName) - 9)] = loreName
          end
        end
        for j = 1, #items do
          local loreName = items[j]
          if string.sub(loreName, -9) ~= "_PostBoat" then
            resources[index] = {
              Name = loreName,
              ButtonType = "button",
              Image = loreName,
              Name_PostBoat = postboat[loreName]
            }
            index = index + 1
          end
        end
      end
    else
      local items = resourceUtil.GetResources(resourceConsts.WALLET_KRATOS, {category}, {})
      if 0 < #items then
        resources[index] = {Name = category, ButtonType = "category"}
        index = index + 1
        for j = 1, #items do
          resources[index] = {
            Name = items[j],
            ButtonType = "button",
            Image = category
          }
          index = index + 1
        end
      end
    end
  end
  local storyList = self.menu:GetList("MainList")
  if self.journalOn.journalMenu.pauseMenuOpened == true then
    local init_useOnGainFocus = false
    storyList:SetSelectedButton(1, init_useOnGainFocus)
    self.journalOn.journalMenu.pauseMenuOpened = false
  end
  local showList = true
  local useOnGainFocus = true
  local itemDetermineFocusabilityFunc = function(resource)
    return resource.ButtonType ~= "category"
  end
  local itemGetTextFunc = function(resource)
    if resource.ButtonType == "category" then
      return util.GetLAMSMsg(lamsIDs[resource.Name], resource.Name)
    end
    return resourceUtil.GetDisplayName(resource.Name)
  end
  storyList:Refresh(resources, showList, useOnGainFocus, itemDetermineFocusabilityFunc, itemGetTextFunc)
  if #resources == 0 then
    self.journalOn:JournalSetMaterialSwap(nil)
  end
  self.journalOn:UpdateFooterButtonPrompt(self.menu)
  self.menu:ExecuteInstructions()
end
function Lore:Exit()
  self.menu:Deactivate(true)
  UI.SetText(self.journalOn.thSharedInfoTitle, "")
  UI.SetText(self.journalOn.thSharedInfoDesc, "")
  util.Hide("SharedInfo")
end
function Lore:MainList_Button_Update(button)
  assert(button ~= nil, "Attempted to call MainList_Button_Update without a Button!")
  local fadeAlpha = 1
  local fadeTime = 0
  local currentItem = button:get_item()
  if currentItem == nil then
    fadeAlpha = 0.45
  end
  button:AlphaFade(fadeAlpha, fadeTime)
  buttonUtil.UpdateColorQuad(button, nil)
end
function Lore:MainList_Button_OnGainFocus(button)
  local resource = button:get_item()
  local titleText = ""
  local descText = ""
  if resource ~= nil then
    self.journalOn:JournalSetMaterialSwap(resource.Image)
    titleText = resourceUtil.GetDisplayName(resource.Name)
    descText = resourceUtil.GetDescription(resource.Name)
    if resource.Name_PostBoat ~= nil then
      descText = descText .. [[


]] .. resourceUtil.GetDescription(resource.Name_PostBoat)
    end
    if resource.Name == "BackAttack_Tutorial" then
      if game.GetQuickTurnOptionNumber() == 0 then
        descText = util.GetLAMSMsg(40908)
      elseif game.GetQuickTurnOptionNumber() == 1 then
        descText = util.GetLAMSMsg(40906)
      elseif game.GetQuickTurnOptionNumber() == 2 then
        descText = util.GetLAMSMsg(18025)
      end
    end
  end
  UI.SetText(self.journalOn.thSharedInfoTitle, titleText)
  UI.SetText(self.journalOn.thSharedInfoDesc, descText)
  if resource == nil then
    util.Hide("SharedInfo")
  else
    util.Show("SharedInfo")
  end
end
function Bestiary:Setup()
  self.journalOn = self:GetState("JournalOn")
  self.menu = menu.Menu.New(self, {
    FooterButtonInfo = {
      {
        Item = "Map",
        Text = "[TouchPad] " .. util.GetLAMSMsg(lamsIDs.Map),
        EventHandlers = {
          {
            Events = {
              "EVT_TouchPad_Release"
            },
            Handler = function()
              self:SendEventToUIFsm("globalMenu", "EVT_GO_TO_MAP")
            end
          }
        }
      },
      {
        Item = "Weapon",
        Text = "",
        EventHandlers = {
          {
            Events = {
              "EVT_GO_TO_WEAPON_MENU"
            },
            Handler = function()
              self:SendEventToUIFsm("globalMenu", "EVT_GO_TO_WEAPON")
            end
          }
        }
      },
      {
        Item = "Skill",
        Text = "",
        EventHandlers = {
          {
            Events = {
              "EVT_GO_TO_SKILL_TREE_MENU"
            },
            Handler = function()
              self:SendEventToUIFsm("globalMenu", "EVT_GO_TO_SKILL_TREE")
            end
          }
        }
      },
      {
        Item = "Quest",
        Text = "",
        EventHandlers = {
          {
            Events = {
              "EVT_GO_TO_QUEST_MENU"
            },
            Handler = function()
              self:SendEventToUIFsm("globalMenu", "EVT_GO_TO_QUEST")
            end
          }
        }
      },
      {
        Item = "Settings",
        Text = "[TriangleButton] " .. util.GetLAMSMsg(lamsIDs.Options),
        EventHandlers = {
          {
            Events = {
              "EVT_Triangle_Release"
            },
            Handler = function()
              self:SendEventToUIFsm("globalMenu", "EVT_OPEN_SETTINGS_MENU")
            end
          }
        }
      },
      {
        Item = "Exit",
        Text = "[BackButton] " .. util.GetLAMSMsg(lamsIDs.Exit),
        EventHandlers = {
          {
            Events = {
              "EVT_Back_Release",
              "EVT_Options_Release"
            },
            Handler = function()
              self:SendEventToUIFsm("globalMenu", "EVT_TURN_OFF_GLOBAL_MENU")
            end
          }
        }
      }
    }
  })
  local bestiaryList = list.List.New(self, {
    MaxFocusableObjectCount = 10,
    ListObjectName = "PickupList",
    EmptyTextLamsID = lamsIDs.NoCategoriesAvailable,
    HoverToSelect = true,
    Button_Update = function(button)
      self:MainList_Button_Update(button)
    end,
    Button_OnGainFocus = function(button)
      self:MainList_Button_OnGainFocus(button)
    end,
    Button_ItemCompare = function(item, otherItem)
      return UI.BestiaryCompare and UI.BestiaryCompare(item, otherItem) or false
    end
  })
  bestiaryList:SetSelectedButton(1, false)
  self.menu:SetList("MainList", bestiaryList)
  self.goBestiaryGroup = util.GetUiObjByName("Bestiary_Group")
  self.goBestiaryInfo = self.journalOn.goJournalGroup:FindSingleGOByName("BestiaryInfo")
  self.thBestiaryInfoName = util.GetTextHandle(self.goBestiaryInfo, "Name")
  self.thBestiaryInfoDesc = util.GetTextHandle(self.goBestiaryInfo, "Description")
  self.weaknessTextHandles = {
    util.GetTextHandle(self.goBestiaryInfo, "Weakness1"),
    util.GetTextHandle(self.goBestiaryInfo, "Weakness2")
  }
  self.weaknessPips = {
    self.goBestiaryInfo:FindSingleGOByName("WeaknessPip1"),
    self.goBestiaryInfo:FindSingleGOByName("WeaknessPip2")
  }
end
function Bestiary:UpdateAccessibilityScaling()
  if UI.GetAccessibilityScaling == nil then
    return
  end
  self.journalOn:UpdateAccessibilityScaling()
  self.goBestiaryInfo:SetLocalPosition(engine.Vector.New(UI.GetAccessibilityScaling(0, -1.95), UI.GetAccessibilityScaling(0, 1), 0))
  self.goBestiaryInfo:SetScale(UI.GetAccessibilityScaling(1, 1.15))
end
function Bestiary:Enter()
  self.menu:Activate()
  self:UpdateAccessibilityScaling()
  local bestiaryList = self.menu:GetList("MainList")
  if self.journalOn.journalMenu.pauseMenuOpened == true then
    local init_useOnGainFocus = false
    bestiaryList:SetSelectedButton(1, init_useOnGainFocus)
    self.journalOn.journalMenu.pauseMenuOpened = false
  end
  local newItemArray = UI.GetBestiaryList()
  local showList = true
  local useOnGainFocus = true
  local itemDetermineFocusabilityFunc = function(category)
    return category.ButtonType ~= "category"
  end
  local itemGetTextFunc = function(category)
    return util.GetLAMSMsg(category.lams)
  end
  bestiaryList:Refresh(newItemArray, showList, useOnGainFocus, itemDetermineFocusabilityFunc, itemGetTextFunc)
  self.journalOn:UpdateFooterButtonPrompt(self.menu)
  if 0 < #newItemArray then
    self.goBestiaryGroup:Show()
    self.goBestiaryInfo:Show()
  else
    self.journalOn:JournalSetMaterialSwap(nil)
    self.goBestiaryGroup:Hide()
    self.goBestiaryInfo:Hide()
  end
  self.menu:ExecuteInstructions()
end
function Bestiary:Exit()
  self.goBestiaryGroup:Hide()
  self.goBestiaryInfo:Hide()
  self.menu:Deactivate(true)
end
function Bestiary:ClearBestiaryInfo()
  UI.SetText(self.thBestiaryInfoName, "")
  UI.SetText(self.thBestiaryInfoDesc, "")
  for weaknessIndex, thWeakness in ipairs(self.weaknessTextHandles) do
    self.weaknessPips[weaknessIndex]:Hide()
    UI.SetText(thWeakness, "")
  end
end
function Bestiary:JournalSetMaterialSwap(bestiaryEntry)
  if bestiaryEntry ~= nil then
    self.journalOn:JournalSetMaterialSwap(bestiaryUtil.BestiaryEntry_GetMaterialSwap(bestiaryEntry), true)
  else
    self.journalOn:JournalSetMaterialSwap(nil)
  end
end
function Bestiary:UpdateWeaknesses(bestiaryEntry)
  if bestiaryEntry ~= nil then
    for weaknessIndex, thWeakness in ipairs(self.weaknessTextHandles) do
      local weaknessName = ""
      local weaknessDescLamsID = bestiaryUtil.BestiaryEntryWeakness_GetDescriptionLamsID(bestiaryEntry, weaknessIndex)
      if weaknessDescLamsID ~= nil then
        self.weaknessPips[weaknessIndex]:Show()
        weaknessName = util.GetLAMSMsg(weaknessDescLamsID)
      end
      UI.SetText(thWeakness, weaknessName)
    end
  end
end
function Bestiary:UpdateBestiaryInfo(nameHash)
  self:ClearBestiaryInfo()
  if nameHash ~= nil then
    local bestiaryEntry = bestiaryUtil.GetBestiaryEntry(nameHash)
    if bestiaryEntry ~= nil then
      self:JournalSetMaterialSwap(bestiaryEntry)
      local displayNameLamsID = bestiaryUtil.BestiaryEntry_GetDisplayNameLamsID(bestiaryEntry)
      local descriptionLamsID = bestiaryUtil.BestiaryEntry_GetDescriptionLamsID(bestiaryEntry)
      UI.SetText(self.thBestiaryInfoName, util.GetLAMSMsg(displayNameLamsID))
      UI.SetText(self.thBestiaryInfoDesc, util.GetLAMSMsg(descriptionLamsID))
      self:UpdateWeaknesses(bestiaryEntry)
    end
  end
  self:UpdateFooterButtonPrompt()
end
function Bestiary:UpdateFooterButtonPrompt()
  local showTouchPadMap = game.IsMapAvailable() and game.build.GOLD_VERSION ~= 0
  self.menu:UpdateFooterButton("Map", showTouchPadMap)
  self.menu:UpdateFooterButton("Weapon", true)
  self.menu:UpdateFooterButton("Skill", true)
  self.menu:UpdateFooterButton("Quest", true)
  self.menu:UpdateFooterButton("Settings", true)
  self.menu:UpdateFooterButton("Exit", true)
  self.menu:UpdateFooterButtonText()
end
function Bestiary:MainList_Button_Update(button)
  assert(button ~= nil, "Attempted to call MainList_Button_Update without a Button!")
  local fadeAlpha = 1
  local fadeTime = 0
  local currItem = button:get_item()
  if currItem == nil then
    fadeAlpha = 0.45
  end
  button:AlphaFade(fadeAlpha, fadeTime)
  buttonUtil.UpdateColorQuad(button, nil)
end
function Bestiary:MainList_Button_OnGainFocus(button)
  local currItem = button:get_item()
  if currItem == nil or type(currItem) == "number" then
    return
  end
  Audio.StopSound("SND_UX_Pause_Menu_Page_Write")
  self:UpdateBestiaryInfo(currItem.hash)
end
function Bestiary:Bestiary_Button_Update(button)
  assert(button ~= nil, "Attempted to call Bestiary_Button_Update without a Button!")
  local fadeAlpha = 1
  local fadeTime = 0
  button:AlphaFade(fadeAlpha, fadeTime)
end
function JournalMenu:OnSaveCheckpoint(tab)
end
function JournalMenu:OnRestoreCheckpoint(tab)
end
