local classlib = require("core.class")
local tablex = require("core.tablex")
local animationUtil = require("ui.animationUtil")
local attributeUtil = require("ui.attributeUtil")
local buttonUtil = require("ui.buttonUtil")
local characterUtil = require("ui.characterUtil")
local colors = require("ui.colors")
local consts = require("ui.consts")
local emptyCard = require("ui.emptyCard")
local fsm = require("ui.fsm")
local lamsIDs = require("ui.lamsConsts")
local list = require("ui.list")
local menu = require("ui.menu")
local meter = require("ui.meter")
local gemInfoCard = require("ui.gemInfoCard")
local skillUtil = require("ui.skillUtil")
local pickupConsts = require("ui.pickupConsts")
local pickupUtil = require("ui.pickupUtil")
local recipeUtil = require("ui.recipeUtil")
local resourceConsts = require("ui.resourceConsts")
local resourceUtil = require("ui.resourceUtil")
local runeUtil = require("ui.runeUtil")
local statsCard = require("ui.statsCard")
local toggle = require("ui.toggle")
local tutorialUtil = require("ui.tutorialUtil")
local util = require("ui.util")
local iconConsts = require("ui.iconConsts")
local AI = game.AI
local Audio = game.Audio
local Player = game.Player
local UI = game.UI
local maxFocusablePickupSlots = 4
local StatsMenu = classlib.Class("StatsMenu", fsm.UIState)
local StatsOff = StatsMenu:StateClass("StatsOff", fsm.UIState)
local StatsOn = StatsMenu:StateClass("StatsOn", fsm.UIState)
local StatsKratos = StatsMenu:StateClass("StatsKratos", fsm.UIState)
local StatsSon = StatsMenu:StateClass("StatsSon", fsm.UIState)
local Upgrade = StatsMenu:StateClass("Upgrade", fsm.UIState)
local statsMenu = StatsMenu.New("statsMenu", {
  StatsOff,
  StatsOn,
  {
    StatsKratos,
    StatsSon,
    Upgrade
  }
})
function StatsMenu:Setup()
  self.pauseMenuOpened = false
end
function StatsMenu:Enter()
  self:WantPadEvents(true)
  self:TurnOff()
end
function StatsMenu:TurnOff()
  self:Goto("StatsOff")
end
function StatsMenu:EVT_PAUSE_MENU_OPEN()
  self.pauseMenuOpened = true
end
StatsMenu.EVT_GAME_OVER = StatsMenu.TurnOff
StatsMenu.EVT_Restart = StatsMenu.TurnOff
function StatsOff:Setup()
  self.statsOn = self:GetState("StatsOn")
end
function StatsOff:Enter()
end
function StatsOff:Exit()
end
function StatsOff:EVT_TURN_ON_STATS_MENU(trinketAvailability, instructionEntries, instructionArgs)
  self.statsOn.trinketAvailability = trinketAvailability
  self.statsOn.menu:set_instructionEntries(instructionEntries)
  self.statsOn.menu:set_instructionArgs(instructionArgs)
  self:Goto("StatsOn")
end
function StatsOn:Setup()
  self.menu = menu.Menu.New(self, {})
  self.submenus = {}
  self.statsMenu = self:GetState("statsMenu")
  self.showDesc = true
  self.itemComparisonCardOn = false
  self.pickupBeingUpgraded = nil
  self.trinketAvailability = true
  self.currSortingGroupIndex = nil
  self.streamedElementOutwards = 0
  self.maxStreamedElements = 6
  util.Show("InfoCards", "pauseCameraParent")
  local goStatsCard = util.GetUiObjByName("statsdetail_card_statsMenu")
  self.statsCard = statsCard.StatsCard.New(goStatsCard)
  local goStatsCardComparison = util.GetUiObjByName("statsdetail_card_statsMenu_comparison")
  self.statsCard_comparison = statsCard.StatsCard.New(goStatsCardComparison)
  local isVendorCard = false
  local isComparisonCard = false
  self.statsCard:Init(isVendorCard, isComparisonCard, {
    ComparisonCard = self.statsCard_comparison
  })
  isVendorCard = false
  isComparisonCard = true
  self.statsCard_comparison:Init(isVendorCard, isComparisonCard, {
    DoDefaultTransitionAnim = false,
    NonComparisonCard = self.statsCard
  })
  self.statsCard_comparison:SetButtonStates("", "")
  local goEmptyCard = util.GetUiObjByName("EmptyCard")
  self._emptyCard = emptyCard.EmptyCard.New(goEmptyCard)
  self._emptyCard:Init()
  self.currentCharacter = pickupConsts.TAG_PICKUP_KRATOS
  self.attributesCardRefNode = util.GetUiObjByName("InWorldMenu_Attributes_Card")
  self.menu:SetupSubmenuList(consts.inworldMenu_SubmenuList, {
    "EVT_Left_Release"
  }, {
    "EVT_Right_Release"
  })
  local goSublist = util.GetUiObjByName("PickupList_EquippedItem")
  local goSortList = goSublist:FindSingleGOByName("list_sort_nav")
  self.menu:SetupSortList("SortList", goSortList)
  tutorialUtil.RegisterDesaturationObject("Stats_Stats_Card", goStatsCard)
  tutorialUtil.RegisterDesaturationObject("Stats_Stats_Card_socket1", self.statsCard.upgradeSlots[1].gameObjects.goRoot)
  tutorialUtil.RegisterDesaturationObject("Stats_Stats_Card_buttonPromptGrp", self.statsCard.children.goButtonPromptGrp)
  tutorialUtil.RegisterDesaturationObject("Stats_Stats_Card_buttonPrompt_Left", self.statsCard.children.goButtonPrompt_Left)
  tutorialUtil.RegisterDesaturationObject("Stats_Stats_Card_buttonPrompt_Middle", self.statsCard.children.goButtonPrompt_Middle)
  tutorialUtil.RegisterDesaturationObject("Stats_Stats_Card_buttonPrompt_Right", self.statsCard.children.goButtonPrompt_Right)
  self.deferredSelectPickupTable = {}
end
function StatsOn:Enter()
  self.menu:Activate()
  self.itemComparisonCardOn = false
  self.pickupBeingUpgraded = nil
  self._needCameraInit = true
  UI.SetUIGORenderLayer(self.statsCard._gameObject, consts.RenderLayer_WorldUI)
  UI.SetUIGORenderLayer(self.statsCard_comparison._gameObject, consts.RenderLayer_WorldUI)
  UI.WorldUIRender("cameraJoint")
  util.Show("Stats")
  self:SendEventToUIFsm("inWorldMenu", "EVT_SHOW_ATTRIBUTE_CARD")
  self:SendEventToUIFsm("inWorldMenu", "EVT_REFRESH_ATTRIBUTES", self.currentCharacter)
  self:SendEventToUIFsm("inWorldMenu", "EVT_SHOW_ATTRIBUTE_DESCRIPTIONS")
  local submenuList = self.menu:GetList(consts.inworldMenu_SubmenuList)
  if self.statsMenu.pauseMenuOpened == true then
    local init_useOnGainFocus = false
    submenuList:SetSelectedButton(1, init_useOnGainFocus)
  end
  local newItemArray = resourceUtil.StatsMenu_GetSubStateNames()
  local showList = true
  local useOnGainFocus = false
  local itemDetermineFocusabilityFunc = function(name)
    return self:IsSubstateFocusable(name)
  end
  local getDisplayNameFunc
  self.menu:RefreshSubmenuList(submenuList, newItemArray, showList, useOnGainFocus, itemDetermineFocusabilityFunc, getDisplayNameFunc)
  if not self.menu:HasInstructionEntryForMenuState() then
    submenuList:SetSelectedButton(1, true)
  end
  self:SendEventToUIFsm("inWorldMenu", "EVT_REFRESH_NOTIFICATIONS")
  local currentStateName = self.menu:GetList(consts.inworldMenu_SubmenuList):GetSelectedItem()
  local currentCharacter = self:GetCharacterFromState(currentStateName)
  animationUtil.ResetCamera(currentCharacter, false)
  animationUtil.MoveWorldUIMainList("ListJoint_Pickups")
  self.currSortingGroupIndex = resourceConsts.SORTING_GROUP_RARITY_STATS
  local sortListName = self.menu:get_sortListName()
  local sortList = self.menu:GetList(sortListName)
  local sort_newItemArray = resourceUtil.GetSortGroups(self.currSortingGroupIndex)
  local sort_showList = false
  local sort_useOnGainFocus = false
  local sort_itemDetermineFocusabilityFunc
  local sort_getDisplayNameFunc = resourceUtil.GetSortGroupDisplayName
  sortList:Refresh(sort_newItemArray, sort_showList, sort_useOnGainFocus, sort_itemDetermineFocusabilityFunc, sort_getDisplayNameFunc)
  sortList:SetSelectedButton(1, false)
  buttonUtil.InitSlotButtons("PickupSlotList")
  self._refreshingListOnEnter = false
  self.menu:ExecuteInstructions()
end
function StatsOn:Exit()
  self.menu:Deactivate(true)
  util.Hide("Stats")
  self.statsCard:ClearCard()
  self.statsCard:HideCard()
  self.statsCard_comparison:ClearCard()
  self.statsCard_comparison:HideCard()
  self._emptyCard:HideCard()
  self._exitedFromSublist = false
  self:SendEventToUIFsm("inWorldMenu", "EVT_CLEAR_ATTRIBUTES_PREVIEW")
  self:SendEventToUIFsm("inWorldMenu", "EVT_HIDE_ATTRIBUTE_CARD")
  characterUtil.ApplyAllPickupChanges()
  if self:HaveTimer("StatsMenuCharacterSwitchCardTimer") then
    self:StopTimer("StatsMenuCharacterSwitchCardTimer")
  end
  local currentStateName = self.menu:GetList(consts.inworldMenu_SubmenuList):GetSelectedItem()
  local currentCharacter = self:GetCharacterFromState(currentStateName)
  animationUtil.ResetCamera(currentCharacter, false)
end
function StatsOn:GetCharacterFromState(substateName)
  local currentCharacter = pickupConsts.TAG_PICKUP_KRATOS
  if substateName == "StatsSon" then
    currentCharacter = pickupConsts.TAG_PICKUP_SON
  end
  return currentCharacter
end
function StatsOn:UpdateFooterButtonPrompt(currMenu, isSubListOpen, canOpenSubList, isEquipped, canEquip)
  local canSelect = not isSubListOpen and canOpenSubList
  local showTouchPadMap = game.IsMapAvailable() and game.build.GOLD_VERSION ~= 0
  local equipUnavailable = not canEquip and isSubListOpen and not isEquipped
  currMenu:UpdateFooterButton("Equip", isSubListOpen and not isEquipped and canEquip)
  currMenu:UpdateFooterButton("Select", canSelect)
  currMenu:UpdateFooterButton("CooldownActive", equipUnavailable)
  currMenu:UpdateFooterButton("Map", showTouchPadMap)
  currMenu:UpdateFooterButton("Weapon", true)
  currMenu:UpdateFooterButton("Skill", true)
  currMenu:UpdateFooterButton("Quest", true)
  currMenu:UpdateFooterButton("Settings", true)
  currMenu:UpdateFooterButton("Back", isSubListOpen)
  currMenu:UpdateFooterButton("Exit", not isSubListOpen)
  currMenu:UpdateFooterButton("Close", true)
  currMenu:UpdateFooterButtonText()
end
function StatsOn:UpdateStatsCard(character, item, resourceFlag, itemIsEquipped, currentlyEquippedPickup, pickupSlot, isSublistOpen)
  self.statsCard:ClearCard()
  self.statsCard_comparison:ClearCard()
  local isValidRune = resourceFlag == resourceConsts.RESOURCE_FLAG_RUNE and type(item) == "table"
  local isValidPickup = resourceFlag ~= resourceConsts.RESOURCE_FLAG_RUNE and pickupUtil.IsValidName(item)
  if isValidRune or isValidPickup then
    if currentlyEquippedPickup ~= nil and item ~= currentlyEquippedPickup and self.itemComparisonCardOn then
      self.statsCard_comparison:SetItem(currentlyEquippedPickup, resourceFlag, character, true)
      self.statsCard_comparison:ShowCard()
      self:SendEventToUIFsm("inWorldMenu", "EVT_SMALL_ATTRIBUTE_CARD")
    else
      self.statsCard_comparison:HideCard()
      self:SendEventToUIFsm("inWorldMenu", "EVT_LARGE_ATTRIBUTE_CARD")
    end
    self.statsCard:SetItem(item, resourceFlag, character, itemIsEquipped)
    self.statsCard:ShowCard()
    self._emptyCard:HideCard()
    local leftButtonMsg = ""
    local rightButtonMsg = ""
    self.statsCard:SetButtonState_Middle(nil)
    self.canCompare = currentlyEquippedPickup ~= nil and resourceUtil.DetermineIfCanCompare(item, itemIsEquipped, resourceFlag)
    if self.canCompare then
      rightButtonMsg = "[R3]" .. (self.itemComparisonCardOn and util.GetLAMSMsg(lamsIDs.Hide) or util.GetLAMSMsg(lamsIDs.Compare))
    end
    local isCurrItemEquippedItem = characterUtil.ItemIsEquipped(item)
    local currItemHasSockets = pickupUtil.IsUpgradeAllowed(item)
    if currItemHasSockets and isCurrItemEquippedItem then
      leftButtonMsg = "[SquareButton] " .. util.GetLAMSMsg(lamsIDs.SocketEnchantmentPrompt)
    elseif not isCurrItemEquippedItem and currItemHasSockets and pickupUtil.IsUpgradeAllowed(currentlyEquippedPickup) then
      local currItemRuneCount = #runeUtil.GetRunesInPickup(item)
      local equippedItemRuneCount = #runeUtil.GetRunesInPickup(currentlyEquippedPickup)
      if 0 < currItemRuneCount or 0 < equippedItemRuneCount then
        if not self.itemComparisonCardOn then
          rightButtonMsg = "[R3]" .. util.GetLAMSMsg(lamsIDs.Compare) .. " / " .. util.GetLAMSMsg(lamsIDs.SwapEnchantments)
        else
          leftButtonMsg = "[SquareButton] " .. util.GetLAMSMsg(lamsIDs.SwapEnchantments)
        end
      end
    end
    self.statsCard:SetButtonState_Middle(nil)
    self.statsCard:SetButtonStates(leftButtonMsg, rightButtonMsg)
  else
    self._emptyCard:Update(pickupSlot)
    self._emptyCard:ShowCard()
    self.statsCard:HideCard()
    self.statsCard_comparison:HideCard()
    self:SendEventToUIFsm("inWorldMenu", "EVT_LARGE_ATTRIBUTE_CARD")
  end
end
function StatsOn:SubmenuList_Button_Update(button)
  local isFocusable = button:get_focusable()
  local isUsable = isFocusable == true and self.pickupBeingUpgraded == nil
  local alphaValue = isUsable and consts.FOCUSABLE_ALPHA or consts.NON_FOCUSABLE_ALPHA
  local fadeTime = 0
  button:AlphaFade(alphaValue, fadeTime)
  button:SetIcon(button:get_item())
  button:UpdateNewIcon(function(button)
    local currItem = button:GetSortItem()
    return resourceUtil.AnyNewArmorBySubState(currItem)
  end)
end
function StatsOn:SortList_Button_OnGainFocus(button)
  local submenuListName = self.menu:get_submenuListName()
  local submenuList = self.menu:GetList(submenuListName)
  local currSubstateName = submenuList:GetSelectedItem()
  local currSubstate = self:GetState(currSubstateName)
  local pickupSlotsList = currSubstate.menu:GetList("MainList")
  local currPickupSlotName = pickupSlotsList:GetSelectedItem()
  if pickupUtil.IsValidSlotName(currPickupSlotName) then
    local subList = currSubstate.menu:GetList(currPickupSlotName)
    local character = self:GetCharacterFromState(currSubstateName)
    local creature = pickupUtil.GetCreatureFromCharacter(character)
    local useOnGainFocus = false
    local selectEquippedItem = false
    local isOnEquip = false
    self:UpdatePickupListBySlot(currSubstate.menu, currPickupSlotName, creature, character, useOnGainFocus, selectEquippedItem, isOnEquip)
    local labelText = resourceUtil.GetSortGroupLabelDisplayText(button:get_item())
    self.menu:SetMenuListLabelText("SortList", labelText)
  else
    local sortListName = self.menu:get_sortListName()
    local sortList = self.menu:GetList(sortListName)
    local sortList_hideList = true
    local sortList_clearButtons = false
    sortList:Deactivate(sortList_hideList, sortList_clearButtons)
  end
  if not self:EquipAllowed(currPickupSlotName) then
    self:SendEventToUIFsm("globalMenu", "EVT_SET_FOOTER_TOOL_TIP", util.GetLAMSMsg(lamsIDs.TalismanCooldownTooltip))
  end
end
function StatsOn:MainList_Button_Update(button, creature, character)
  local pickupSlot = button:get_item()
  local equippedPickupName
  local equippedPickupColor = colors.WHITE
  local pickupSlots = pickupUtil.GetAvailablePickupSlotNames(character)
  local goList = util.GetUiObjByName("PickupSlotList")
  if pickupSlot ~= nil then
    equippedPickupName = self:GetCurrentlyEquippedPickupInSlot(creature, pickupSlot)
    if not util.IsStringNilOrEmpty(equippedPickupName) then
      equippedPickupColor = resourceUtil.GetRarityColor(equippedPickupName)
    end
    local isTalismanButton = button:get_item() == pickupConsts.Slot_ArmorTrinket
    buttonUtil.UpdateSlotButtonColor(button, equippedPickupName, equippedPickupColor, isTalismanButton)
    button:SetText("text_level", "")
    local goButtonChild = button:GetInstancedChildObject()
    goButtonChild:Show()
    if not isTalismanButton then
      local iconMaterial = resourceUtil.GetMaterialSwapName(equippedPickupName)
      button:SetIcon(iconMaterial)
    else
      local icon_text = iconConsts.GetIcon(equippedPickupName)
      if icon_text == nil then
        icon_text = ""
      end
      button:SetText("icon_text", icon_text)
    end
    local alpha = 1
    local time = 0
    UI.AlphaFade(goButtonChild, alpha, time)
    buttonUtil.UpdateSlotButtonUpgradeSlots(button, equippedPickupName, character)
    local currItem = button:GetSortItem()
    local goNewIcon
    for i, v in ipairs(pickupSlots) do
      if v == currItem then
        goNewIcon = goList:FindSingleGOByName("newIcon" .. tostring(i))
        break
      end
    end
    if resourceUtil.AnyNewPickupsInSlotByCharacter(currItem, character) then
      goNewIcon:Show()
      UI.Anim(goNewIcon, consts.AS_ForwardCycle_NoReset, "", 1)
    else
      goNewIcon:Hide()
    end
  end
  if character == pickupConsts.TAG_PICKUP_SON then
    local goList = util.GetUiObjByName("PickupSlotList")
    for i = 2, maxFocusablePickupSlots do
      goList:FindSingleGOByName("newIcon" .. tostring(i)):Hide()
    end
  end
end
function StatsOn:SubList_Button_Update(button, creature, currPickupSlotName)
  local currPickupName = button:get_item()
  local equippedPickupName = self:GetCurrentlyEquippedPickupInSlot(creature, currPickupSlotName)
  local canEquip = self:EquipAllowed(currPickupSlotName)
  button:UpdateEquippedIcon(function(button)
    return pickupUtil.IsValidName(currPickupName) and currPickupName == equippedPickupName
  end)
  local pickupNameColor = colors.WHITE
  if not canEquip then
    pickupNameColor = colors.GRAY
  end
  button:SetTextColor(pickupNameColor)
  local rarity = resourceUtil.GetRarity(currPickupName)
  local rarityColor = pickupUtil.GetRarityColor(rarity)
  buttonUtil.UpdateColorQuad(button, rarityColor)
  local effectFlag = resourceUtil.GetEffectFlag(currPickupSlotName, currPickupName)
  local effectFlagIcon = ""
  if effectFlag ~= nil then
    effectFlagIcon = iconConsts[effectFlag]
  end
  button:SetText("icon_text", effectFlagIcon)
  button:UpdateNewIcon(function(button)
    return buttonUtil.ShowNotification(button, consts.NotificationTypes.Resource)
  end)
end
function StatsOn:SubList_Button_OnLoseFocus(button)
  if self._refreshingListOnEnter == false then
    local currPickupName = button:get_item()
    if pickupUtil.IsValidName(currPickupName) then
      resourceUtil.ClearNew(currPickupName)
      button:UpdateNewIcon(function(button)
        return buttonUtil.ShowNotification(button, consts.NotificationTypes.Resource)
      end)
    end
    self:SendEventToUIFsm("inWorldMenu", "EVT_REFRESH_NOTIFICATIONS")
  end
end
function StatsOn:UpdateCamera(character, cameraState)
  if self._needCameraInit then
    self:SendEventToUIFsm("inWorldMenu", "EVT_INIT_CAMERA", character, cameraState)
    self._needCameraInit = false
  else
    self:SendEventToUIFsm("inWorldMenu", "EVT_UPDATE_CAMERA", character, cameraState)
  end
end
function StatsOn:SubmenuList_Button_OnGainFocus(button)
  local currentStateName = button:get_item()
  self.menu:SetSubmenuListLabelText(consts.inworldMenu_SubmenuList, util.GetLAMSMsg(lamsIDs[currentStateName]))
  local currentCharacter = self:GetCharacterFromState(currentStateName)
  local cameraState = currentStateName
  self:UpdateCamera(currentCharacter, cameraState)
  local subState = self:GetState(currentStateName)
  subState.menu:set_instructionEntries(self.menu:get_instructionEntries())
  self:Goto(currentStateName)
end
function StatsOn:GetCurrentlyEquippedPickupInSlot(creature, curSlotName)
  local character = pickupUtil.GetCharacterFromCreature(creature)
  return characterUtil.GetPickupInSlot(character, curSlotName)
end
function StatsOn:SelectPickup(currMenu, currPickupSlotName, currPickupName, creature, character, isSubListOpen, canOpenSubList, canEquip)
  self.deferredSelectPickupTable.currPickupSlotName = currPickupSlotName
  self.deferredSelectPickupTable.currPickupName = currPickupName
  self.deferredSelectPickupTable.creature = creature
  self.deferredSelectPickupTable.character = character
  self.deferredSelectPickupTable.isSubListOpen = isSubListOpen
  local equippedPickupName = self:GetCurrentlyEquippedPickupInSlot(creature, currPickupSlotName)
  local isEquipped = currPickupName == equippedPickupName
  local resourceFlag = resourceConsts.RESOURCE_FLAG_PICKUP
  if resourceUtil.IsResourceAGem(currPickupName) then
    resourceFlag = resourceConsts.INFO_CARD_ID_GEM
  end
  if not characterUtil.ItemIsEquipped(currPickupName) and pickupUtil.IsUpgradeAllowed(currPickupName) then
    self:SendEventToUIFsm("globalMenu", "EVT_SET_FOOTER_TOOL_TIP", util.GetLAMSMsg(lamsIDs.EquipItemToSocketEnchantment))
  else
    self:SendEventToUIFsm("globalMenu", "EVT_SET_FOOTER_TOOL_TIP", "")
  end
  self:UpdateStatsCard(character, currPickupName, resourceFlag, isEquipped, equippedPickupName, currPickupSlotName, isSubListOpen)
  self:UpdateFooterButtonPrompt(currMenu, isSubListOpen, canOpenSubList, isEquipped, canEquip)
end
function StatsOn:SelectPickupFinishedStreaming(currPickupSlotName, currPickupName, creature, character, isSubListOpen)
  local pickupStage = pickupUtil.GetStage(currPickupName, creature)
  pickupUtil.UpdateSlotOnUICreature(currPickupSlotName, currPickupName, pickupStage, creature, character)
  if isSubListOpen then
    if character == pickupConsts.TAG_PICKUP_KRATOS then
      self:SendEventToUIFsm("inWorldMenu", "EVT_UPDATE_ATTRIBUTES_PREVIEW", character)
    end
    local fromMenu = true
    pickupUtil.PickupAcquire(creature, currPickupName, pickupStage, fromMenu)
  end
end
function StatsOn:SelectPickupSlot(currMenu, currPickupSlotName, creature, character)
  local currPickupName = self:GetCurrentlyEquippedPickupInSlot(creature, currPickupSlotName)
  local isSubListOpen = false
  local canOpenSubList = self:CanOpenSubList(currPickupSlotName)
  local canEquip = self:EquipAllowed(currPickupSlotName)
  self:SelectPickup(currMenu, currPickupSlotName, currPickupName, creature, character, isSubListOpen, canOpenSubList, canEquip)
end
function StatsOn:IsSubstateFocusable(substateName)
  local isBladesSelection = false
  local hasBlades = false
  local isSonSelection = substateName == "StatsSon"
  return util.IsMenuSelectionFocusable(isBladesSelection, hasBlades, isSonSelection)
end
function StatsOn:CanOpenSubList(pickupSlotName)
  if pickupSlotName == pickupConsts.Slot_PrimaryWeapon or pickupSlotName == pickupConsts.Slot_Blades then
    return false
  end
  return true
end
function StatsOn:EquipAllowed(pickupSlotName)
  if pickupSlotName == pickupConsts.Slot_ArmorTrinket then
    return self.trinketAvailability
  end
  return true
end
function StatsOn:StreamPickup(character, subList, buttonIndex)
  local creature = pickupUtil.GetUICreature(character)
  local pickupName = subList:GetItemByButtonIndex(buttonIndex)
  if creature ~= nil and pickupName ~= nil then
    local pickupStage = game.Pickup.GetProfileStage(pickupName)
    return game.Pickup.RequestLoadHighMips(creature, pickupName, pickupStage)
  end
  return true
end
function StatsOn:StreamSubmenu()
  local currState = self:GetCurrentState()
  if currState and currState.menu and currState.menu:get_active() then
    local pickupSlotsList = currState.menu:GetList("MainList")
    local currPickupSlotName = pickupSlotsList:GetSelectedItem()
    if currPickupSlotName ~= nil then
      local subList = currState.menu:GetList(currPickupSlotName)
      self.streamedElementOutwards = pickupUtil.StreamSubList(self.currentCharacter, subList, self.streamedElementOutwards, self.maxStreamedElements, function(character, subList, buttonIndex)
        return self:StreamPickup(character, subList, buttonIndex)
      end)
    end
  end
end
function StatsOn:Update()
  if self.deferredSelectPickupTable.currPickupName ~= nil then
    local fullyStreamed = game.Pickup.RequestLoadHighMips(self.deferredSelectPickupTable.creature, self.deferredSelectPickupTable.currPickupName, self.deferredSelectPickupTable.currPickupStage)
    if fullyStreamed and attributeUtil.AttributesInitialized() then
      self:SelectPickupFinishedStreaming(self.deferredSelectPickupTable.currPickupSlotName, self.deferredSelectPickupTable.currPickupName, self.deferredSelectPickupTable.creature, self.deferredSelectPickupTable.character, self.deferredSelectPickupTable.isSubListOpen)
      tablex.Clear(self.deferredSelectPickupTable)
    end
  end
  self:StreamSubmenu()
end
function StatsOn:ToggleItemComparison(currState)
  local pickupSlotsList = currState.menu:GetList("MainList")
  local currPickupSlotName = pickupSlotsList:GetSelectedItem()
  local currPickupList = currState.menu:GetList(currPickupSlotName)
  local selectedPickupName = currPickupList:GetSelectedItem()
  local isSubListOpen = true
  local canOpenSubList = false
  local canEquip = self:EquipAllowed(currPickupSlotName)
  local currCreature = self:GetCurrentCreature()
  self.itemComparisonCardOn = selectedPickupName ~= nil and self.canCompare and not self.itemComparisonCardOn
  self:SelectPickup(currState.menu, currPickupSlotName, selectedPickupName, currCreature, self.currentCharacter, isSubListOpen, canOpenSubList, canEquip)
  if self.itemComparisonCardOn then
    Audio.PlaySound("SND_UX_Pause_Menu_Compare_Show")
  else
    Audio.PlaySound("SND_UX_Pause_Menu_Compare_Hide")
  end
  self.statsCard_comparison:UpdateEquipBarBottom(self.itemComparisonCardOn)
end
function StatsOn:GetCurrentCreature()
  local creature
  if self.currentCharacter == pickupConsts.TAG_PICKUP_KRATOS then
    creature = Player.FindPlayer()
  elseif self.currentCharacter == pickupConsts.TAG_PICKUP_SON then
    creature = AI.FindSon()
  end
  assert(creature ~= nil, "Creature was not setup properly when parsing pickup info.")
  return creature
end
function StatsOn:Submenu_Setup(currState, character)
  currState.menu = menu.Menu.New(currState, {
    FooterButtonInfo = {
      {
        Item = "Select",
        Text = "[AdvanceButton] " .. util.GetLAMSMsg(lamsIDs.Select)
      },
      {
        Item = "Equip",
        Text = "[AdvanceButton] " .. util.GetLAMSMsg(lamsIDs.Equip)
      },
      {
        Item = "Map",
        Text = "[TouchPad] " .. util.GetLAMSMsg(lamsIDs.Map),
        EventHandlers = {
          {
            Events = {
              "EVT_TouchPad_Release"
            },
            Handler = function()
              self:SendEventToUIFsm("globalMenu", "EVT_GO_TO_MAP")
            end
          }
        }
      },
      {
        Item = "Weapon",
        Text = "",
        EventHandlers = {
          {
            Events = {
              "EVT_GO_TO_WEAPON_MENU"
            },
            Handler = function()
              self:SendEventToUIFsm("globalMenu", "EVT_GO_TO_WEAPON")
            end
          }
        }
      },
      {
        Item = "Skill",
        Text = "",
        EventHandlers = {
          {
            Events = {
              "EVT_GO_TO_SKILL_TREE_MENU"
            },
            Handler = function()
              self:SendEventToUIFsm("globalMenu", "EVT_GO_TO_SKILL_TREE")
            end
          }
        }
      },
      {
        Item = "Quest",
        Text = "",
        EventHandlers = {
          {
            Events = {
              "EVT_GO_TO_QUEST_MENU"
            },
            Handler = function()
              self:SendEventToUIFsm("globalMenu", "EVT_GO_TO_QUEST")
            end
          }
        }
      },
      {
        Item = "Settings",
        Text = "[TriangleButton] " .. util.GetLAMSMsg(lamsIDs.Options),
        EventHandlers = {
          {
            Events = {
              "EVT_Triangle_Release"
            },
            Handler = function()
              self:SendEventToUIFsm("globalMenu", "EVT_OPEN_SETTINGS_MENU")
            end
          }
        }
      },
      {
        Item = "Back",
        Text = "[BackButton] " .. util.GetLAMSMsg(lamsIDs.Back)
      },
      {
        Item = "Exit",
        Text = "[BackButton] " .. util.GetLAMSMsg(lamsIDs.Exit)
      },
      {
        Item = "Close",
        Text = "",
        EventHandlers = {
          {
            Events = {
              "EVT_Options_Release"
            },
            Handler = function()
              currState:SendEventToUIFsm("globalMenu", "EVT_TURN_OFF_GLOBAL_MENU")
            end
          }
        }
      }
    },
    Name = currState.name
  })
  local pickupSlotListName = currState.menu:get_name() .. "_MainList"
  local pickupSlotsList = list.List.New(currState, {
    MaxFocusableObjectCount = maxFocusablePickupSlots,
    Name = pickupSlotListName,
    ListObjectName = "PickupSlotList",
    HoverToSelect = true,
    EventHandlers = {
      {
        Events = {
          "EVT_Advance_Release"
        },
        Handler = function()
          currState:MainList_AdvanceReleaseHandler()
        end
      },
      {
        Events = {
          "EVT_MOUSE_RELEASED"
        },
        Handler = function()
          local selected = UI.GetEventSenderGameObject()
          local list = currState.menu:GetList("MainList")
          local button = list:GetSelectedButton()
          if button ~= nil and selected == button:GetInstancedChildObject() then
            currState:MainList_AdvanceReleaseHandler()
          end
        end
      },
      {
        Events = {
          "EVT_Back_Release"
        },
        Handler = function()
          currState:MainList_BackReleaseHandler()
        end
      },
      {
        Events = {
          "EVT_Square_Release"
        },
        Handler = function()
          currState:MainList_SquareReleaseHandler()
        end
      }
    },
    Button_Update = function(button)
      currState:MainList_Button_Update(button)
    end,
    Button_OnGainFocus = function(button)
      currState:MainList_Button_OnGainFocus(button)
    end,
    Button_OnLoseFocus = function(button)
      currState:MainList_Button_OnLoseFocus(button)
    end,
    Button_HighlightOn = function(button, animateImmediately)
      buttonUtil.StatsMenuPickupSlot_HighlightOn(button, animateImmediately)
    end,
    Button_HighlightOff = function(button, animateImmediately)
      buttonUtil.StatsMenuPickupSlot_HighlightOff(button, animateImmediately)
    end
  })
  currState.menu:SetList("MainList", pickupSlotsList)
  tutorialUtil.RegisterDesaturationList(pickupSlotListName, pickupSlotsList)
  currState.menu:SetupSubLists("PickupList_EquippedItem", pickupUtil.GetAvailablePickupSlotNames(character), nil, nil, nil)
  self.submenus[character] = currState.menu
  currState.waitingForEquippedAttributes = false
end
function StatsOn:Submenu_Enter(currState, character)
  currState.menu:Activate()
  self.currentCharacter = character
  currState.waitingForEquippedAttributes = false
  if character == pickupConsts.TAG_PICKUP_SON then
    self:SendEventToUIFsm("inWorldMenu", "EVT_HIDE_ATTRIBUTE_CARD")
  else
    self:SendEventToUIFsm("inWorldMenu", "EVT_SHOW_ATTRIBUTE_CARD")
  end
  local pickupSlotsList = currState.menu:GetList("MainList")
  if self.statsMenu.pauseMenuOpened == true then
    local init_useOnGainFocus = false
    pickupSlotsList:SetSelectedButton(1, init_useOnGainFocus)
    self.statsMenu.pauseMenuOpened = false
  end
  local newItemArray = pickupUtil.GetAvailablePickupSlotNames(character)
  local showList = true
  local useOnGainFocus = false
  local itemDetermineFocusabilityFunc
  local itemGetTextFunc = pickupUtil.GetSlotDisplayName
  pickupSlotsList:Refresh(newItemArray, showList, useOnGainFocus, itemDetermineFocusabilityFunc, itemGetTextFunc)
  animationUtil.ResetPickupSlotPositions(pickupSlotsList)
  local reselect_useOnGainFocus = not currState.menu:HasInstructionEntryForMenuState()
  pickupSlotsList:ReselectSelectedItem(reselect_useOnGainFocus)
  if self.pickupBeingUpgraded ~= nil then
    self.pickupBeingUpgraded = nil
    self.selectedSubstate = nil
  end
  currState.menu:ExecuteInstructions()
end
function StatsOn:UpdatePickupListBySlot(currMenu, currPickupSlotName, creature, character, useOnGainFocus, selectEquippedItem, isOnEquip)
  local subListName = currPickupSlotName
  local headerText = pickupUtil.GetSlotDisplayName(currPickupSlotName)
  local getSubListItemsFunc = function(pickupSlotName)
    local sortingGroup = resourceUtil.GetSortGroupFromMenu(self.menu)
    return resourceUtil.GetPickupsInSlotByCharacter(pickupSlotName, character, sortingGroup)
  end
  local itemToSelect = self:GetCurrentlyEquippedPickupInSlot(creature, currPickupSlotName)
  if self._itemSelectOverride ~= nil then
    itemToSelect = self._itemSelectOverride
    self._itemSelectOverride = nil
  end
  local itemDetermineFocusabilityFunc
  local itemGetTextFunc = pickupUtil.GetDisplayName
  local deactivateSublist = isOnEquip
  currMenu:RefreshEquippedItemSublist(subListName, headerText, getSubListItemsFunc, itemToSelect, itemDetermineFocusabilityFunc, itemGetTextFunc, useOnGainFocus, selectEquippedItem, isOnEquip, deactivateSublist)
  local sortListName = self.menu:get_sortListName()
  local sortList = self.menu:GetList(sortListName)
  local sortingGroupIndex = recipeUtil.GetSortingGroupIndex(currPickupSlotName)
  if sortList:get_active() == false and sortingGroupIndex ~= nil then
    local subList = currMenu:GetList(subListName)
    if subList:GetButtonCount() > 1 then
      if self.currSortingGroupIndex ~= sortingGroupIndex then
        self.currSortingGroupIndex = sortingGroupIndex
        local sort_newItemArray = resourceUtil.GetSortGroups(self.currSortingGroupIndex)
        local sort_showList = false
        local sort_useOnGainFocus = false
        local sort_itemDetermineFocusabilityFunc
        local sort_getDisplayNameFunc = resourceUtil.GetSortGroupDisplayName
        sortList:Refresh(sort_newItemArray, sort_showList, sort_useOnGainFocus, sort_itemDetermineFocusabilityFunc, sort_getDisplayNameFunc)
      end
      sortList:SetSelectedButton(1, false)
      local sortList_showList = true
      local sortList_useOnGainFocus = true
      sortList:Activate(sortList_showList, sortList_useOnGainFocus)
      self.menu:AnimateSortList()
    end
  end
end
function StatsOn:Submenu_EquippedAttributesUpdatedHandler(currState)
  local pickupSlotsList = currState.menu:GetList("MainList")
  local currPickupSlotName = pickupSlotsList:GetSelectedItem()
  local subList = currState.menu:GetList(currPickupSlotName)
  if currState.waitingForEquippedAttributes == true then
    subList:Activate(false, false)
    currState.waitingForEquippedAttributes = false
  end
end
function StatsOn:MainList_AdvanceReleaseHandler(currMenu, creature, character)
  local pickupSlotsList = currMenu:GetList("MainList")
  local currPickupSlotName = pickupSlotsList:GetSelectedItem()
  if self:CanOpenSubList(currPickupSlotName) == true then
    local notUsingOverride = self._itemSelectOverride == nil
    self._exitedFromSublist = true
    self._refreshingListOnEnter = true
    local cameraState
    self:UpdateCamera(character, currPickupSlotName)
    local submenuList = self.menu:GetList(consts.inworldMenu_SubmenuList)
    local submenuList_hideList = true
    local submenuList_clearButtons = false
    submenuList:Deactivate(submenuList_hideList, submenuList_clearButtons)
    local pickupSlotsList_hideList = false
    local pickupSlotsList_clearButtons = false
    pickupSlotsList:Deactivate(pickupSlotsList_hideList, pickupSlotsList_clearButtons)
    animationUtil.AnimatePickupSlotPositions(pickupSlotsList, "PickupSlot_Selected_EquippedItem")
    pickupSlotsList:HideHeader()
    local selectedButton = pickupSlotsList:GetSelectedButton()
    selectedButton:UpdateNewIcon(nil)
    selectedButton:HighlightOff(true)
    local useOnGainFocus = true
    local selectEquippedItem = true
    local isOnEquip = false
    self:UpdatePickupListBySlot(currMenu, currPickupSlotName, creature, character, useOnGainFocus, selectEquippedItem, isOnEquip)
    Audio.PlaySound("SND_UX_Menu_Camera_Pan_Slow")
    local subList = currMenu:GetList(currPickupSlotName)
    if subList:GetButtonCount() > 0 and notUsingOverride == true then
      Audio.PlaySound("SND_UX_Menu_Tick_Confirm")
    else
      Audio.PlaySound("SND_UX_Menu_Tick_Fail")
    end
    self._refreshingListOnEnter = false
  else
    Audio.PlaySound("SND_UX_Menu_Tick_Fail")
  end
end
function StatsOn:MainList_SquareReleaseHandler(currMenu, currPickupSlotName, creature, character)
  local currPickupList = currMenu:GetList(currPickupSlotName)
  local equippedPickupName = self:GetCurrentlyEquippedPickupInSlot(creature, currPickupSlotName)
  self:OpenUpgradeMenu(equippedPickupName)
end
function StatsOn:OpenUpgradeMenu(equippedPickupName)
  if pickupUtil.IsUpgradeAllowed(equippedPickupName) then
    self.pickupBeingUpgraded = equippedPickupName
    self.menu:SetSubmenuListLabelText(consts.inworldMenu_SubmenuList, util.GetLAMSMsg(lamsIDs.Upgrade))
    self:Goto("Upgrade")
    game.Audio.PlaySound("SND_UX_Menu_Tick_Confirm")
  end
end
function StatsOn:SubList_AdvanceReleaseHandler(currState, currPickupSlotName, creature, character)
  local currPickupList = currState.menu:GetList(currPickupSlotName)
  local currPickupName = currPickupList:GetSelectedItem()
  local equippedPickupName = self:GetCurrentlyEquippedPickupInSlot(creature, currPickupSlotName)
  local equipAllowed = self:EquipAllowed(currPickupSlotName)
  if not game.Cinematics.IsInCinematicMode() and pickupUtil.IsValidName(currPickupName) and currPickupName ~= equippedPickupName and equipAllowed then
    if currPickupSlotName == pickupConsts.Slot_ArmorTrinket then
      Audio.PlaySound("SND_UX_Equip_Talisman")
    else
      Audio.PlaySound("SND_UX_Equip_Armor")
    end
    currState.waitingForEquippedAttributes = true
    local acquirePickup = true
    characterUtil.SetPickupSlot(character, currPickupSlotName, currPickupName, acquirePickup)
    self:SendEventToUIFsm("inWorldMenu", "EVT_UPDATE_EQUIPPED_ATTRIBUTES", character)
    self:SendEventToUIFsm("inWorldMenu", "EVT_CLEAR_ATTRIBUTES_PREVIEW")
    self:StartTimer("ClearPowerLevelOnEquipDelay", consts.SingleFrameDelayTime, function()
      self:SendEventToUIFsm("inWorldMenu", "EVT_CLEAR_ATTRIBUTES_PREVIEW")
    end)
    self.statsCard:PlayFlourishOnEquip()
    local slotToFXName = {
      [pickupConsts.Slot_ArmorChest] = {
        FX = "equipchest_flourish",
        Joint = "JOBack1"
      },
      [pickupConsts.Slot_ArmorLegs] = {
        FX = "equipwaist_flourish",
        Joint = "JOPelvis1"
      },
      [pickupConsts.Slot_ArmorWrist] = {
        FX = "equiparm_flourish",
        Joint = "JORightLowerArm1"
      },
      [pickupConsts.Slot_ArmorTrinket] = nil,
      [pickupConsts.Slot_SonArmor] = {
        FX = "equipson_flourish",
        Joint = "JOPelvis1"
      }
    }
    local goKratos = pickupUtil.GetUICreature(character)
    local fxGroup = slotToFXName[currPickupSlotName]
    if fxGroup ~= nil then
      local particleName = fxGroup.FX
      local jointName = fxGroup.Joint
      animationUtil.SpawnFX(particleName, goKratos, jointName)
    end
    local goEquipFlourishJoint = util.GetUiObjByName("EquipFlourishGameObject")
    animationUtil.SpawnFX("equip_pickupslot_flourish", goEquipFlourishJoint, "")
    local pickupSlotsList = currState.menu:GetList("MainList")
    local selectedButton = pickupSlotsList:GetSelectedButton()
    selectedButton:Update()
    selectedButton:UpdateNewIcon(nil)
    local useOnGainFocus = true
    local selectEquippedItem = true
    local isOnEquip = true
    self:UpdatePickupListBySlot(currState.menu, currPickupSlotName, creature, character, useOnGainFocus, selectEquippedItem, isOnEquip)
  end
end
function StatsOn:SubList_BackReleaseHandler(currMenu, currPickupSlotName)
  self._exitedFromSublist = false
  self.itemComparisonCardOn = false
  local pickupList = currMenu:GetList(currPickupSlotName)
  local pickupList_hideList = true
  local pickupList_clearButtons = false
  pickupList:Deactivate(pickupList_hideList, pickupList_clearButtons)
  local sortListName = self.menu:get_sortListName()
  local sortList = self.menu:GetList(sortListName)
  local sortList_hideList = true
  local sortList_clearButtons = false
  sortList:Deactivate(sortList_hideList, sortList_clearButtons)
  local pickupSlotsList = currMenu:GetList("MainList")
  local pickupSlotsList_showList = true
  local pickupSlotsList_useOnGainFocus = true
  pickupSlotsList:Activate(pickupSlotsList_showList, pickupSlotsList_useOnGainFocus)
  pickupSlotsList:ShowHeader()
  local submenuList = self.menu:GetList(consts.inworldMenu_SubmenuList)
  local submenuList_showList = true
  local submenuList_useOnGainFocus = true
  submenuList:Activate(submenuList_showList, submenuList_useOnGainFocus)
  local currSubmenu = submenuList:GetSelectedItem()
  local currentCharacter = self:GetCharacterFromState(currSubmenu)
  animationUtil.ResetPickupSlotPositions(pickupSlotsList)
  self:UpdateCamera(currentCharacter, nil)
  self:SendEventToUIFsm("inWorldMenu", "EVT_CLEAR_ATTRIBUTES_PREVIEW")
  characterUtil.ApplyAllPickupChanges()
  Audio.PlaySound("SND_UX_Menu_Camera_Pan_Medium")
  Audio.PlaySound("SND_UX_Menu_Tick_Back")
end
function StatsOn:SubList_SquareReleaseHandler(currMenu, currPickupSlotName, creature, character)
  local currPickupList = currMenu:GetList(currPickupSlotName)
  local currPickupName = currPickupList:GetSelectedItem()
  if pickupUtil.IsUpgradeAllowed(currPickupName) then
    if characterUtil.ItemIsEquipped(currPickupName) then
      local sortListName = self.menu:get_sortListName()
      local sortList = self.menu:GetList(sortListName)
      local sortList_hideList = true
      local sortList_clearButtons = false
      sortList:Deactivate(sortList_hideList, sortList_clearButtons)
      self.pickupBeingUpgraded = currPickupName
      self:Goto("Upgrade")
      game.Audio.PlaySound("SND_UX_Menu_Tick_Confirm")
    elseif not currMenu._state.waitingForEquippedAttributes then
      local equippedPickupName = self:GetCurrentlyEquippedPickupInSlot(creature, currPickupSlotName)
      if self.itemComparisonCardOn and pickupUtil.IsUpgradeAllowed(equippedPickupName) then
        local currItemRuneCount = #runeUtil.GetRunesInPickup(currPickupName)
        local equippedItemRuneCount = #runeUtil.GetRunesInPickup(equippedPickupName)
        if 0 < currItemRuneCount or 0 < equippedItemRuneCount then
          currMenu._state.waitingForEquippedAttributes = true
          local deltaTable = runeUtil.SwapRunes(currPickupName, equippedPickupName, self.statsCard, self.statsCard_comparison)
          self:SendEventToUIFsm("inWorldMenu", "EVT_UPDATE_EQUIPPED_ATTRIBUTES_WITH_DELTA_RUNESWAP", pickupConsts.TAG_PICKUP_KRATOS, deltaTable)
          local isSubListOpen = true
          local canOpenSubList = false
          local canEquip = self:EquipAllowed(currPickupSlotName)
          self:SelectPickup(currMenu, currPickupSlotName, currPickupName, creature, self.currentCharacter, isSubListOpen, canOpenSubList, canEquip)
          local pickupSlotsList = currMenu:GetList("MainList")
          pickupSlotsList:UpdateSelectedButton()
          game.Audio.PlaySound("SND_UX_Menu_Tick_Confirm")
          game.Audio.PlaySound("SND_UX_Equip_Enchantment")
        end
      end
    end
  end
end
function StatsOn:EVT_HIDE_SUB_MENU_LIST()
  local submenuList = self.menu:GetList(consts.inworldMenu_SubmenuList)
  submenuList:Hide()
end
function StatsOn:EVT_SHOW_SUB_MENU_LIST()
  local submenuList = self.menu:GetList(consts.inworldMenu_SubmenuList)
  submenuList:Show()
end
function StatsOn:EVT_START_RETURN_FROM_SUB_MENU(pickup)
  if pickup ~= nil then
    local pickupSlot = pickupUtil.GetSlotName(pickup)
    local menu = self.submenus[pickupConsts.TAG_PICKUP_KRATOS]
    local mainList = menu:GetList("MainList")
    self.menu:AddInstructionEntry({
      StateName = "StatsKratos",
      ListName = "MainList",
      Item = pickupSlot,
      EventName = "EVT_Advance_Release"
    })
    self._itemSelectOverride = pickup
  end
end
function StatsOn:EVT_PREP_IMMEDIATE_ITEM_SELECT(pickup)
  if pickup ~= nil then
    self._itemSelectOverride = pickup
  end
end
function StatsOn:EVT_RETURN_FROM_SUB_MENU()
  self.pickupBeingUpgraded = nil
  local submenuList = self.menu:GetList(consts.inworldMenu_SubmenuList)
  local showList = false
  local useOnGainFocus = true
  submenuList:Activate(showList, useOnGainFocus)
  submenuList:Update()
end
function StatsOn:EVT_UPDATE_SELECTED_PICKUP_INFO(pickupSlotName, pickupName)
  local creature = self:GetCurrentCreature()
  self:SendEventToUIFsm("inWorldMenu", "EVT_UPDATE_ATTRIBUTES_PREVIEW", self.currentCharacter)
  local currentlyEquippedPickup = self:GetCurrentlyEquippedPickupInSlot(creature, pickupSlotName)
  local itemIsEquipped = pickupName == currentlyEquippedPickup
  local resourceFlag = resourceConsts.RESOURCE_FLAG_PICKUP
  if resourceUtil.IsResourceAGem(pickupName) then
    resourceFlag = resourceConsts.INFO_CARD_ID_GEM
  end
  if resourceFlag == resourceConsts.INFO_CARD_ID_GEM then
    self:UpdateGemCard(pickupName, itemIsEquipped)
  else
    local isSublistOpen = true
    self:UpdateStatsCard(self.currentCharacter, pickupName, resourceFlag, itemIsEquipped, currentlyEquippedPickup, pickupSlotName, isSublistOpen)
  end
end
function StatsOn:EVT_UPDATE_STATSMENU_NOTIFICATIONS()
  self:SendEventToUIFsm("inWorldMenu", "EVT_REFRESH_NOTIFICATIONS")
end
function StatsOn:EVT_TURN_OFF_STATS_MENU()
  self:Goto("StatsOff")
end
function StatsKratos:Setup()
  self.statsOn = self:GetState("StatsOn")
  self.statsOn:Submenu_Setup(self, pickupConsts.TAG_PICKUP_KRATOS)
end
function StatsKratos:Enter()
  self.statsOn:Submenu_Enter(self, pickupConsts.TAG_PICKUP_KRATOS)
end
function StatsKratos:Exit()
  self.menu:Deactivate(true)
end
function StatsKratos:MainList_Button_OnGainFocus(button)
  local currSlotName = button:get_item()
  local currSlotList = button:get_list()
  currSlotList:SetHeaderText(pickupUtil.GetSlotDisplayName(currSlotName))
  if currSlotName ~= nil then
    self.statsOn:SelectPickupSlot(self.menu, currSlotName, Player.FindPlayer(), pickupConsts.TAG_PICKUP_KRATOS)
    if not game.GetNewGamePlus() and not resourceUtil.HasResource("EnchantmentTutorial") then
      local creature = pickupUtil.GetCreatureFromCharacter(pickupConsts.TAG_PICKUP_KRATOS)
      local currPickup = self.statsOn:GetCurrentlyEquippedPickupInSlot(creature, currSlotName)
      if currPickup ~= nil then
        local stage = pickupUtil.GetProfileStage(currPickup)
        local upgradeSocketCount = pickupUtil.GetRuneSocketCount(currPickup, stage)
        if 0 < upgradeSocketCount then
          self:SendEventToUIFsm("inWorldMenu", "EVT_ATTEMPT_TUTORIAL", "Stats", "EnchantmentTutorial", "Pause_Enchantments")
        end
      end
    end
  end
end
function StatsKratos:MainList_Button_OnLoseFocus(button)
  local currSlotName = button:get_item()
  if currSlotName ~= nil then
    local Kratos = Player.FindPlayer()
    local pickupInPrevSlot = self.statsOn:GetCurrentlyEquippedPickupInSlot(Kratos, currSlotName)
    local pickupStage = pickupUtil.GetStage(pickupInPrevSlot, Kratos)
    pickupUtil.UpdateSlotOnUICreature(currSlotName, pickupInPrevSlot, pickupStage, Kratos, pickupConsts.TAG_PICKUP_KRATOS)
  end
end
function StatsKratos:MainList_Button_Update(button)
  self.statsOn:MainList_Button_Update(button, Player.FindPlayer(), pickupConsts.TAG_PICKUP_KRATOS)
end
function StatsKratos:SubList_Button_Update(button)
  local Kratos = Player.FindPlayer()
  local pickupSlotsList = self.menu:GetList("MainList")
  local currPickupSlotName = pickupSlotsList:GetSelectedItem()
  self.statsOn:SubList_Button_Update(button, Kratos, currPickupSlotName)
end
function StatsKratos:SubList_Button_OnGainFocus(button)
  local currPickupName = button:get_item()
  local pickupSlotsList = self.menu:GetList("MainList")
  local currPickupSlotName = pickupSlotsList:GetSelectedItem()
  local isSubListOpen = true
  local canOpenSubList = false
  local canEquip = self.statsOn:EquipAllowed(currPickupSlotName)
  local creature = Player.FindPlayer()
  self.statsOn:SelectPickup(self.menu, currPickupSlotName, currPickupName, creature, pickupConsts.TAG_PICKUP_KRATOS, isSubListOpen, canOpenSubList, canEquip)
  if not game.GetNewGamePlus() and not resourceUtil.HasResource("EnchantmentTutorial") and currPickupName ~= nil then
    local stage = pickupUtil.GetProfileStage(currPickupName)
    local upgradeSocketCount = pickupUtil.GetRuneSocketCount(currPickupName, stage)
    local itemIsEquipped = currPickupName == self.statsOn:GetCurrentlyEquippedPickupInSlot(creature, currPickupSlotName)
    if 0 < upgradeSocketCount and itemIsEquipped then
      self:SendEventToUIFsm("inWorldMenu", "EVT_ATTEMPT_TUTORIAL", "Stats", "EnchantmentTutorial", "Pause_Enchantments_From_List")
    end
  end
  if not self.statsOn:EquipAllowed(currPickupSlotName) then
    self:SendEventToUIFsm("globalMenu", "EVT_SET_FOOTER_TOOL_TIP", util.GetLAMSMsg(lamsIDs.TalismanCooldownTooltip))
  end
end
function StatsKratos:SubList_Button_OnLoseFocus(button)
  self.statsOn:SubList_Button_OnLoseFocus(button)
end
function StatsKratos:MainList_AdvanceReleaseHandler()
  self.statsOn:MainList_AdvanceReleaseHandler(self.menu, Player.FindPlayer(), pickupConsts.TAG_PICKUP_KRATOS)
end
function StatsKratos:MainList_BackReleaseHandler()
  self:SendEventToUIFsm("globalMenu", "EVT_TURN_OFF_GLOBAL_MENU")
end
function StatsKratos:MainList_SquareReleaseHandler()
  local pickupSlotsList = self.menu:GetList("MainList")
  local currPickupSlotName = pickupSlotsList:GetSelectedItem()
  self.statsOn:MainList_SquareReleaseHandler(self.menu, currPickupSlotName, Player.FindPlayer(), pickupConsts.TAG_PICKUP_KRATOS)
end
function StatsKratos:SubList_AdvanceReleaseHandler(pickupSlotName)
  self.statsOn:SubList_AdvanceReleaseHandler(self, pickupSlotName, Player.FindPlayer(), pickupConsts.TAG_PICKUP_KRATOS)
end
function StatsKratos:SubList_BackReleaseHandler(pickupSlotName)
  self.statsOn:SubList_BackReleaseHandler(self.menu, pickupSlotName)
end
function StatsKratos:SubList_SquareReleaseHandler(pickupSlotName)
  self.statsOn:SubList_SquareReleaseHandler(self.menu, pickupSlotName, Player.FindPlayer(), pickupConsts.TAG_PICKUP_KRATOS)
end
function StatsKratos:SubList_R3ReleaseHandler(pickupSlotName)
  self.statsOn:ToggleItemComparison(self)
end
function StatsKratos:EVT_EQUIPPED_ATTRIBUTES_UPDATED()
  self.statsOn:Submenu_EquippedAttributesUpdatedHandler(self)
end
function StatsSon:Setup()
  self.statsOn = self:GetState("StatsOn")
  self.statsOn:Submenu_Setup(self, pickupConsts.TAG_PICKUP_SON)
end
function StatsSon:Enter()
  if not util.SonUI_ShouldShow() or util.SonUI_ShouldDisable() then
    return
  end
  self.statsOn:Submenu_Enter(self, pickupConsts.TAG_PICKUP_SON)
  Audio.PlaySound("SND_UX_Stats_Menu_StatsSon_Enter")
end
function StatsSon:Exit()
  self.menu:Deactivate(true)
  Audio.PlaySound("SND_UX_Stats_Menu_StatsSon_Exit")
end
function StatsSon:MainList_Button_OnGainFocus(button)
  local currSlotName = button:get_item()
  local currSlotList = button:get_list()
  currSlotList:SetHeaderText(pickupUtil.GetSlotDisplayName(currSlotName))
  if currSlotName ~= nil then
    self.statsOn:SelectPickupSlot(self.menu, currSlotName, AI.FindSon(), pickupConsts.TAG_PICKUP_SON)
  end
end
function StatsSon:MainList_Button_OnLoseFocus(button)
  local currSlotName = button:get_item()
  if currSlotName ~= nil then
    local son = AI.FindSon()
    if son ~= nil then
      local pickupInSlot = self.statsOn:GetCurrentlyEquippedPickupInSlot(son, currSlotName)
      local pickupStage = pickupUtil.GetStage(pickupInSlot, son)
      pickupUtil.UpdateSlotOnUICreature(currSlotName, pickupInSlot, pickupStage, son, pickupConsts.TAG_PICKUP_SON)
    end
  end
end
function StatsSon:MainList_Button_Update(button)
  self.statsOn:MainList_Button_Update(button, AI.FindSon(), pickupConsts.TAG_PICKUP_SON)
end
function StatsSon:SubList_Button_Update(button)
  local son = AI.FindSon()
  local pickupSlotsList = self.menu:GetList("MainList")
  local currPickupSlotName = pickupSlotsList:GetSelectedItem()
  self.statsOn:SubList_Button_Update(button, son, currPickupSlotName)
end
function StatsSon:SubList_Button_OnGainFocus(button)
  local currPickupName = button:get_item()
  local pickupSlotsList = self.menu:GetList("MainList")
  local currPickupSlotName = pickupSlotsList:GetSelectedItem()
  local isSubListOpen = true
  local canOpenSubList = false
  local canEquip = true
  self.statsOn:SelectPickup(self.menu, currPickupSlotName, currPickupName, AI.FindSon(), pickupConsts.TAG_PICKUP_SON, isSubListOpen, canOpenSubList, canEquip)
end
function StatsSon:SubList_Button_OnLoseFocus(button)
  local currItem = button:get_item()
  if currItem ~= nil then
    local currSlotName = pickupUtil.GetSlotName(currItem)
    if currSlotName ~= nil then
      local Kratos = Player.FindPlayer()
      local pickupInPrevSlot = self.statsOn:GetCurrentlyEquippedPickupInSlot(Kratos, currSlotName)
      local pickupStage = pickupUtil.GetStage(pickupInPrevSlot, Kratos)
      pickupUtil.UpdateSlotOnUICreature(currSlotName, pickupInPrevSlot, pickupStage, Kratos, pickupConsts.TAG_PICKUP_KRATOS)
    end
  end
  self.statsOn:SubList_Button_OnLoseFocus(button)
end
function StatsSon:MainList_AdvanceReleaseHandler()
  self.statsOn:MainList_AdvanceReleaseHandler(self.menu, AI.FindSon(), pickupConsts.TAG_PICKUP_SON)
end
function StatsSon:MainList_BackReleaseHandler()
  self:SendEventToUIFsm("globalMenu", "EVT_TURN_OFF_GLOBAL_MENU")
end
function StatsSon:MainList_SquareReleaseHandler()
  local pickupSlotsList = self.menu:GetList("MainList")
  local currPickupSlotName = pickupSlotsList:GetSelectedItem()
  self.statsOn:MainList_SquareReleaseHandler(self.menu, currPickupSlotName, AI.FindSon(), pickupConsts.TAG_PICKUP_SON)
end
function StatsSon:SubList_AdvanceReleaseHandler(pickupSlotName)
  self.statsOn:SubList_AdvanceReleaseHandler(self, pickupSlotName, AI.FindSon(), pickupConsts.TAG_PICKUP_SON)
end
function StatsSon:SubList_BackReleaseHandler(pickupSlotName)
  self.statsOn:SubList_BackReleaseHandler(self.menu, pickupSlotName)
end
function StatsSon:SubList_SquareReleaseHandler(pickupSlotName)
  self.statsOn:SubList_SquareReleaseHandler(self.menu, pickupSlotName, AI.FindSon(), pickupConsts.TAG_PICKUP_SON)
end
function StatsSon:SubList_R3ReleaseHandler(pickupSlotName)
  self.statsOn:ToggleItemComparison(self)
end
function StatsSon:EVT_EQUIPPED_ATTRIBUTES_UPDATED()
  self.statsOn:Submenu_EquippedAttributesUpdatedHandler(self)
end
function Upgrade:Setup()
  self.statsOn = self:GetState("StatsOn")
  self.statsOnSubmenuList = self.statsOn.menu:GetList(consts.inworldMenu_SubmenuList)
end
function Upgrade:Enter()
  local hideList = false
  local clearButtons = false
  self.statsOnSubmenuList:Deactivate(hideList, clearButtons)
  self.statsOnSubmenuList:Update()
  self:SendEventToUIFsm("upgradeMenu", "EVT_TURN_ON_UPGRADE_MENU", self.statsOn.pickupBeingUpgraded, self.statsOn._exitedFromSublist, self.statsOn.menu:get_instructionEntries(), self.statsOn.menu:get_instructionArgs())
  self.statsOn._exitedFromSublist = false
end
function Upgrade:Exit()
  self:SendEventToUIFsm("upgradeMenu", "EVT_TURN_OFF_UPGRADE_MENU")
end
function StatsMenu:OnSaveCheckpoint(tab)
end
function StatsMenu:OnRestoreCheckpoint(tab)
end
