local classlib = require("core.class")
local tablex = require("core.tablex")
local animationUtil = require("ui.animationUtil")
local attributeUtil = require("ui.attributeUtil")
local characterUtil = require("ui.characterUtil")
local colors = require("ui.colors")
local consts = require("ui.consts")
local buttonUtil = require("ui.buttonUtil")
local emptyCard = require("ui.emptyCard")
local fsm = require("ui.fsm")
local gemInfoCard = require("ui.gemInfoCard")
local gemUtil = require("ui.gemUtil")
local iconConsts = require("ui.iconConsts")
local lamsIDs = require("ui.lamsConsts")
local list = require("ui.list")
local menu = require("ui.menu")
local pickupConsts = require("ui.pickupConsts")
local pickupUtil = require("ui.pickupUtil")
local resourceConsts = require("ui.resourceConsts")
local resourceUtil = require("ui.resourceUtil")
local runeUtil = require("ui.runeUtil")
local statsCard = require("ui.statsCard")
local tutorialUtil = require("ui.tutorialUtil")
local util = require("ui.util")
local weaponSpecialPreviewCard = require("ui.weaponSpecialPreviewCard")
local weaponCard = require("ui.weaponCard")
local AI = game.AI
local Audio = game.Audio
local Player = game.Player
local UI = game.UI
local WeaponMenu = classlib.Class("WeaponMenu", fsm.UIState)
local WeaponOff = WeaponMenu:StateClass("WeaponOff", fsm.UIState)
local WeaponOn = WeaponMenu:StateClass("WeaponOn", fsm.UIState)
local Axe = WeaponMenu:StateClass("Axe", fsm.UIState)
local Shield = WeaponMenu:StateClass("Shield", fsm.UIState)
local Blades = WeaponMenu:StateClass("Blades", fsm.UIState)
local SonBow = WeaponMenu:StateClass("SonBow", fsm.UIState)
local weaponMenu = WeaponMenu.New("weaponMenu", {
  WeaponOff,
  WeaponOn,
  {
    Axe,
    Shield,
    Blades,
    SonBow
  }
})
function WeaponMenu:Setup()
  self.pauseMenuOpened = false
end
function WeaponMenu:Enter()
  self:WantPadEvents(true)
  self:TurnOff()
end
function WeaponMenu:TurnOff()
  self:Goto("WeaponOff")
end
function WeaponMenu:EVT_PAUSE_MENU_OPEN()
  self.pauseMenuOpened = true
end
WeaponMenu.EVT_GAME_OVER = WeaponMenu.TurnOff
WeaponMenu.EVT_Restart = WeaponMenu.TurnOff
function WeaponOff:Setup()
  self.weaponOn = self:GetState("WeaponOn")
end
function WeaponOff:Enter()
end
function WeaponOff:Exit()
end
function WeaponOff:EVT_TURN_ON_WEAPON_MENU(weaponSpecialsAvailability, instructionEntries, instructionArgs)
  self.weaponOn.weaponSpecialsAvailability = weaponSpecialsAvailability
  self.weaponOn.menu:set_instructionEntries(instructionEntries)
  self.weaponOn.menu:set_instructionArgs(instructionArgs)
  self:Goto("WeaponOn")
end
function WeaponOn:Setup()
  self.menu = menu.Menu.New(self, {})
  self.weaponMenu = self:GetState("weaponMenu")
  self.showDesc = true
  self.itemComparisonCardOn = false
  self.weaponSpecialsAvailability = {}
  local goStatsCard = util.GetUiObjByName("statsdetail_card_statsMenu")
  local goStatsCardComparison = util.GetUiObjByName("statsdetail_card_statsMenu_comparison")
  self.statsCard = statsCard.StatsCard.New(goStatsCard)
  self.statsCard_comparison = statsCard.StatsCard.New(goStatsCardComparison)
  local isVendorCard = false
  local isComparisonCard = false
  self.statsCard:Init(isVendorCard, isComparisonCard, {
    ComparisonCard = self.statsCard_comparison
  })
  isComparisonCard = true
  self.statsCard_comparison:Init(isVendorCard, isComparisonCard, {
    NonComparisonCard = self.statsCard
  })
  self.statsCard_comparison:SetButtonStates("", "")
  local goEmptyCard = util.GetUiObjByName("EmptyCard")
  self._emptyCard = emptyCard.EmptyCard.New(goEmptyCard)
  self._emptyCard:Init()
  local goWeaponSpecialPreviewCard = util.GetUiObjByName("weaponSpecialPreview_card")
  self._weaponSpecialPreviewCard = weaponSpecialPreviewCard.WeaponSpecialPreviewCard.New(goWeaponSpecialPreviewCard)
  self._weaponSpecialPreviewCard:Init()
  local goGemInfoCard = util.GetUiObjByName("Weapon_Gem_Info_Card")
  self._gemInfoCard = gemInfoCard.GemInfoCard.New(self, goGemInfoCard)
  self._gemInfoCard:Init(self)
  self._gemInfoCard:SetOnHoldFillComplete_Left_Callback(function()
    self:OnHoldFillComplete()
  end)
  self.menu:SetupSubmenuList(consts.inworldMenu_SubmenuList, {
    "EVT_Left_Release"
  }, {
    "EVT_Right_Release"
  })
  local goSublist = util.GetUiObjByName("PickupList_EquippedItem")
  local goSortList = goSublist:FindSingleGOByName("list_sort_nav")
  self.menu:SetupSortList("SortList", goSortList)
  tutorialUtil.RegisterDesaturationObject("Weapon_Gem_Info_Card", goGemInfoCard)
  tutorialUtil.RegisterDesaturationObject("Weapon_Gem_Info_Card_LeftGroup", goGemInfoCard:FindSingleGOByName("LeftGroup"))
  tutorialUtil.RegisterDesaturationObject("Weapon_Gem_Info_Card_RightGroup", goGemInfoCard:FindSingleGOByName("RightGroup"))
  local goWeaponCardRefnode = util.GetUiObjByName("pause_weaponcard")
  self._weaponCard = weaponCard.WeaponCard.New(goWeaponCardRefnode)
  self._weaponCard:Init()
  self.deferredSelectPickupTable = {}
end
function WeaponOn:Enter()
  self.menu:Activate()
  self._refreshingListOnEnter = false
  self.itemComparisonCardOn = false
  self._needCameraInit = true
  UI.WorldUIRender("cameraJoint")
  util.Show("Weapon")
  local submenuList = self.menu:GetList(consts.inworldMenu_SubmenuList)
  if self.weaponMenu.pauseMenuOpened == true then
    local init_useOnGainFocus = false
    submenuList:SetSelectedButton(1, init_useOnGainFocus)
  end
  local newItemArray = resourceUtil.WeaponMenu_GetSubStateNames()
  local showList = true
  local useOnGainFocus = not self.menu:HasInstructionEntryForMenuState()
  local itemDetermineFocusabilityFunc = function(name)
    return self:IsSubstateFocusable(name)
  end
  local getDisplayNameFunc
  self.menu:RefreshSubmenuList(submenuList, newItemArray, showList, useOnGainFocus, itemDetermineFocusabilityFunc, getDisplayNameFunc)
  self:SendEventToUIFsm("inWorldMenu", "EVT_REFRESH_ATTRIBUTES", pickupConsts.TAG_PICKUP_KRATOS)
  self:SendEventToUIFsm("inWorldMenu", "EVT_REFRESH_NOTIFICATIONS")
  self:SendEventToUIFsm("inWorldMenu", "EVT_CLEAR_ATTRIBUTES_PREVIEW")
  animationUtil.MoveWorldUIMainList("ListJoint_Pickups")
  local sortListName = self.menu:get_sortListName()
  local sortList = self.menu:GetList(sortListName)
  local sort_newItemArray = resourceUtil.GetSortGroups(resourceConsts.SORTING_GROUP_RARITY_STATS_EFFECTS)
  local sort_showList = false
  local sort_useOnGainFocus = false
  local sort_itemDetermineFocusabilityFunc
  local sort_getDisplayNameFunc = resourceUtil.GetSortGroupDisplayName
  sortList:Refresh(sort_newItemArray, sort_showList, sort_useOnGainFocus, sort_itemDetermineFocusabilityFunc, sort_getDisplayNameFunc)
  sortList:SetSelectedButton(1, false)
  buttonUtil.InitSlotButtons("WeaponUpgradeSlotList")
  self.menu:ExecuteInstructions()
end
function WeaponOn:Exit()
  self.menu:Deactivate(true)
  self._gemInfoCard:DeactivateHoldFills()
  self._gemInfoCard:HideCard()
  self._weaponSpecialPreviewCard:HideCard()
  self:ClearCards()
  util.Hide("Weapon")
  self.statsCard:ClearCard()
  self.statsCard:HideCard()
  self.statsCard_comparison:ClearCard()
  self.statsCard_comparison:HideCard()
  self._emptyCard:HideCard()
  self:SendEventToUIFsm("inWorldMenu", "EVT_CLEAR_ATTRIBUTES_PREVIEW")
  self:SendEventToUIFsm("inWorldMenu", "EVT_HIDE_ATTRIBUTE_CARD")
  self._previousSlotSelected = nil
  animationUtil.ClearVideo()
end
function WeaponOn:IsSubstateFocusable(substateName)
  local isBladesSelection = substateName == pickupConsts.Blades
  local hasBlades = resourceUtil.HasResource(pickupConsts.Blades)
  local isSonSelection = substateName == pickupConsts.SonBow
  return util.IsMenuSelectionFocusable(isBladesSelection, hasBlades, isSonSelection)
end
function WeaponOn:UpdateCamera(character, currentSlotName)
  if self._needCameraInit then
    self:SendEventToUIFsm("inWorldMenu", "EVT_INIT_CAMERA", character, currentSlotName)
    self._needCameraInit = false
  else
    self:SendEventToUIFsm("inWorldMenu", "EVT_UPDATE_CAMERA", character, currentSlotName)
  end
end
function WeaponOn:CanOpenSubList(pickupSlotName)
  if self:IsHeaderSlot(pickupSlotName) then
    return pickupSlotName == pickupConsts.Slot_ArmorShield
  end
  return true
end
function WeaponOn:EquipAllowed(pickupSlotName)
  local canEquip = true
  if self.weaponSpecialsAvailability[pickupSlotName] == false then
    canEquip = false
  end
  return canEquip
end
function WeaponOn:GetCreature(substateCharacter, forceGetKratos)
  local creature
  if forceGetKratos == true then
    creature = Player.FindPlayer()
  else
    creature = pickupUtil.GetCreatureFromCharacter(substateCharacter)
  end
  assert(creature ~= nil, "WeaponOn:GetCreature was not able to find a creature!")
  return creature
end
function WeaponOn:IsHeaderSlot(pickupSlotName)
  return pickupUtil.IsValidSlotName(pickupSlotName) and (pickupSlotName == pickupConsts.Slot_PrimaryWeapon or pickupSlotName == pickupConsts.Slot_Blades or pickupSlotName == pickupConsts.Slot_ArmorShield)
end
function WeaponOn:IsWeaponComponentSlot(pickupSlotName)
  return pickupUtil.IsValidSlotName(pickupSlotName) and (pickupSlotName == pickupConsts.Slot_WeaponComponent or pickupSlotName == pickupConsts.Slot_WeaponComponent_Blades)
end
function WeaponOn:GetWeaponUpgradeSlotsListNames(character, pickupSlotName)
  local buttonListSlotNames = {pickupSlotName}
  local weaponUpgradeSlots = pickupUtil.GetWeaponUpgradeSlotNames(character, pickupSlotName)
  for _, weaponUpgradeSlot in ipairs(weaponUpgradeSlots) do
    tablex.FastInsert(buttonListSlotNames, weaponUpgradeSlot, #buttonListSlotNames + 1)
  end
  return buttonListSlotNames
end
function WeaponOn:OnHoldFillComplete()
  local submenuList = self.menu:GetList(consts.inworldMenu_SubmenuList)
  local currentSubmenuItem = submenuList:GetSelectedItem()
  local currState = self:GetState(currentSubmenuItem)
  local weaponUpgradeSlotsList = currState.menu:GetList("MainList")
  weaponUpgradeSlotsList:UpdateSelectedButton()
  local currWeaponUpgradeSlotName = weaponUpgradeSlotsList:GetSelectedItem()
  local currSubList = currState.menu:GetList(currWeaponUpgradeSlotName)
  currSubList:Update()
end
function WeaponOn:SubmenuList_Button_Update(button)
  local subStateName = button:get_item()
  local isFocusable = button:get_focusable()
  local isUsable = isFocusable == true and self.substatePickup == nil
  local alphaValue = isUsable and consts.FOCUSABLE_ALPHA or consts.NON_FOCUSABLE_ALPHA
  local fadeTime = 0
  button:AlphaFade(alphaValue, fadeTime)
  button:SetIcon(resourceUtil.GetMaterialSwapName(subStateName))
  button:UpdateNewIcon(function(button)
    local currItem = button:GetSortItem()
    return resourceUtil.AnyNewPickupsByWeapon(currItem)
  end)
end
function WeaponOn:SubmenuList_Button_OnGainFocus(button)
  local currentItem = button:get_item()
  local subState = self:GetState(currentItem)
  self.menu:SetSubmenuListLabelText(consts.inworldMenu_SubmenuList, pickupUtil.GetDisplayName(currentItem))
  subState.menu:set_instructionEntries(self.menu:get_instructionEntries())
  self:Goto(currentItem)
end
function WeaponOn:SortList_Button_OnGainFocus(button)
  local submenuListName = self.menu:get_submenuListName()
  local submenuList = self.menu:GetList(submenuListName)
  local currSubstateName = submenuList:GetSelectedItem()
  local currSubstate = self:GetState(currSubstateName)
  local pickupSlotsList = currSubstate.menu:GetList("MainList")
  local currPickupSlotName = pickupSlotsList:GetSelectedItem()
  if pickupUtil.IsValidSlotName(currPickupSlotName) then
    local subList = currSubstate.menu:GetList(currPickupSlotName)
    local useOnGainFocus = false
    local selectEquippedItem = false
    local isOnEquip = false
    self:UpdatePickupListBySlot(currSubstate, currPickupSlotName, useOnGainFocus, selectEquippedItem, isOnEquip)
    local labelText = resourceUtil.GetSortGroupLabelDisplayText(button:get_item())
    self.menu:SetMenuListLabelText("SortList", labelText)
  else
    local sortListName = self.menu:get_sortListName()
    local sortList = self.menu:GetList(sortListName)
    local sortList_hideList = true
    local sortList_clearButtons = false
    sortList:Deactivate(sortList_hideList, sortList_clearButtons)
  end
end
function WeaponOn:SelectWeaponUpgradeSlot(currState, currSlotName)
  local character = self.currentCharacter
  local creature = self:GetCreature(character, false)
  local equippedPickupName = pickupUtil.GetPickupNameInSlot(creature, currSlotName)
  local isItemEquipped = equippedPickupName ~= nil
  local isSubListOpen = false
  local canOpenSubList = self:CanOpenSubList(currSlotName)
  local equipAllowed = self:EquipAllowed(currSlotName)
  self:ClearCards()
  if equippedPickupName ~= nil and resourceUtil.IsResourceAGem(equippedPickupName) then
    self._weaponSpecialPreviewCard:SetProperty("Item", equippedPickupName)
    self._weaponSpecialPreviewCard:Update()
    self._weaponSpecialPreviewCard:ShowCard(false)
    self.statsCard:HideCard()
    local weaponUpgradeSlotsList = currState.menu:GetList("MainList")
    local currWeaponUpgradeSlotName = weaponUpgradeSlotsList:GetSelectedItem()
    local equippedWeaponUpgradeName = pickupUtil.GetPickupNameInSlot(creature, currWeaponUpgradeSlotName)
    self:UpdateFooterButtonPrompt(currState.menu, isItemEquipped, isSubListOpen, canOpenSubList, equipAllowed)
  else
    self._weaponSpecialPreviewCard:HideCard()
    self:SelectPickup(currState, currSlotName, equippedPickupName, creature, character, isSubListOpen, canOpenSubList)
  end
end
function WeaponOn:SelectPickup(currState, currUpgradeSlotName, currWeaponUpgradeName, creature, character, isSubListOpen, canOpenSubList)
  self.deferredSelectPickupTable.currPickupSlotName = currUpgradeSlotName
  self.deferredSelectPickupTable.currPickupName = currWeaponUpgradeName
  self.deferredSelectPickupTable.creature = creature
  self.deferredSelectPickupTable.character = character
  local weaponUpgradeSlotsList = currState.menu:GetList("MainList")
  local currWeaponUpgradeSlotName = weaponUpgradeSlotsList:GetSelectedItem()
  self:UpdateSelectedPickupInfo(currState, currWeaponUpgradeSlotName, currWeaponUpgradeName, creature, isSubListOpen)
  local equippedWeaponUpgradeName = pickupUtil.GetPickupNameInSlot(creature, currWeaponUpgradeSlotName)
  local isSelectedWeaponUpgradeEquipped = currWeaponUpgradeName == equippedWeaponUpgradeName
  local equipAllowed = self:EquipAllowed(currWeaponUpgradeSlotName)
  if currWeaponUpgradeName == nil then
    local animateDefaultTransition = self._previousSlotSelected ~= currUpgradeSlotName
    self._emptyCard:Update(currUpgradeSlotName, animateDefaultTransition)
    self._emptyCard:ShowCard()
  end
  self._previousSlotSelected = currUpgradeSlotName
  self:UpdateFooterButtonPrompt(currState.menu, isSelectedWeaponUpgradeEquipped, isSubListOpen, canOpenSubList, equipAllowed)
end
function WeaponOn:SelectPickupFinishedStreaming(currUpgradeSlotName, currWeaponUpgradeName, creature, character)
  local weaponUpgradeStage = pickupUtil.GetStage(currWeaponUpgradeName, creature)
  pickupUtil.UpdateSlotOnUICreature(currUpgradeSlotName, currWeaponUpgradeName, weaponUpgradeStage, creature, character)
  local submenuList = self.menu:GetList(consts.inworldMenu_SubmenuList)
  local currentSubmenuItem = submenuList:GetSelectedItem()
  local currState = self:GetState(currentSubmenuItem)
  local mainList = currState.menu:GetList("MainList")
  if not mainList:get_active() then
    self:SendEventToUIFsm("inWorldMenu", "EVT_UPDATE_ATTRIBUTES_PREVIEW", pickupConsts.TAG_PICKUP_KRATOS)
  end
end
function WeaponOn:ClearCards()
  self._emptyCard:HideCard()
  self._gemInfoCard:HideCard()
  self._weaponCard:HideCard()
  self.statsCard_comparison:HideCard()
  self.statsCard:HideCard()
  self._weaponSpecialPreviewCard:HideCard()
end
function WeaponOn:UpdateSelectedPickupInfo(currState, pickupSlotName, pickupName, creature, isSubListOpen)
  local currentlyEquippedPickup = pickupUtil.GetPickupNameInSlot(creature, pickupSlotName)
  local itemIsEquipped = pickupName == currentlyEquippedPickup
  local resourceFlag = resourceConsts.RESOURCE_FLAG_PICKUP
  if resourceUtil.IsResourceAGem(pickupName) then
    resourceFlag = resourceConsts.INFO_CARD_ID_GEM
  elseif resourceUtil.HasFlag(pickupName, resourceConsts.RESOURCE_FLAG_PRIMARY_WEAPON) then
    resourceFlag = resourceConsts.INFO_CARD_ID_WEAPON
  end
  if resourceFlag == resourceConsts.INFO_CARD_ID_GEM then
    local equipAllowed = self:EquipAllowed(pickupSlotName)
    local gemUpgradeAvailable = true
    if itemIsEquipped and not equipAllowed then
      gemUpgradeAvailable = Player.FindPlayer():PickupIsInCooldown(pickupName)
    end
    self:UpdateGemCard(pickupName, isSubListOpen, gemUpgradeAvailable)
  elseif resourceFlag == resourceConsts.INFO_CARD_ID_WEAPON then
    self:UpdateWeaponCard(pickupName)
  else
    self:UpdateStatsCard(currState.substateCharacter, pickupName, resourceFlag, itemIsEquipped, currentlyEquippedPickup, isSubListOpen)
  end
end
function WeaponOn:UpdateWeaponCard(pickupName)
  self._weaponCard:SetProperty("Item", pickupName)
  self._weaponCard:ShowCard()
  self._weaponCard:Update(false)
end
function WeaponOn:UpdateGemCard(pickupName, rightGroupVisible, gemUpgradeAvailable)
  self._gemInfoCard:SetGem(pickupName, gemUpgradeAvailable)
  self._gemInfoCard:ShowCard(false, rightGroupVisible)
  self.activeInfoCard = self._gemInfoCard
end
function WeaponOn:UpdateStatsCard(character, pickupName, resourceFlag, itemIsEquipped, currentlyEquippedPickup, isSubListOpen)
  self.statsCard:ClearCard()
  self.statsCard_comparison:ClearCard()
  local canEquip = not itemIsEquipped and isSubListOpen
  local leftButtonMsg = ""
  local rightButtonMsg = ""
  if pickupUtil.IsValidName(pickupName) then
    self.statsCard:SetItem(pickupName, resourceFlag, character, itemIsEquipped)
    self.statsCard:ShowCard(false, false)
    self.activeInfoCard = self.statsCard
    if currentlyEquippedPickup ~= nil and pickupName ~= currentlyEquippedPickup and self.itemComparisonCardOn then
      self.statsCard_comparison:SetItem(currentlyEquippedPickup, resourceFlag, character, true)
      self.statsCard_comparison:ShowCard(false, false)
      self:SendEventToUIFsm("inWorldMenu", "EVT_SMALL_ATTRIBUTE_CARD")
    else
      self.statsCard_comparison:HideCard()
      self:SendEventToUIFsm("inWorldMenu", "EVT_LARGE_ATTRIBUTE_CARD")
    end
  end
  self.canCompare = currentlyEquippedPickup ~= nil and resourceUtil.DetermineIfCanCompare(pickupName, itemIsEquipped, resourceFlag)
  if self.canCompare then
    rightButtonMsg = "[R3]" .. util.GetLAMSMsg(lamsIDs.Compare)
  end
  self.statsCard:SetButtonStates(leftButtonMsg, rightButtonMsg)
end
function WeaponOn:EnableComparisonState(currState, value)
  local valueIsNew = self.itemComparisonCardOn ~= value
  if valueIsNew then
    self.itemComparisonCardOn = value
    local mainList = currState.menu:GetList("MainList")
    local currWeaponUpgradeSlotName = mainList:GetSelectedItem()
    local currSubList = currState.menu:GetList(currWeaponUpgradeSlotName)
    local useOnGainFocus = true
    currSubList:ReselectSelectedItem(useOnGainFocus)
  end
end
function WeaponOn:ToggleItemComparison(currState)
  if self.canCompare then
    self:EnableComparisonState(currState, not self.itemComparisonCardOn)
    if self.itemComparisonCardOn then
      Audio.PlaySound("SND_UX_Pause_Menu_Compare_Show")
    else
      Audio.PlaySound("SND_UX_Pause_Menu_Compare_Hide")
    end
  end
end
function WeaponOn:UpdateFooterButtonPrompt(currMenu, isItemEquipped, isSubListOpen, canOpenSubList, equipAllowed)
  local canSelect = not isSubListOpen and canOpenSubList
  local canEquip = not isItemEquipped and isSubListOpen and equipAllowed
  local showTouchPadMap = game.IsMapAvailable() and game.build.GOLD_VERSION ~= 0
  currMenu:UpdateFooterButton("Equip", canEquip)
  currMenu:UpdateFooterButton("Select", canSelect)
  currMenu:UpdateFooterButton("Map", showTouchPadMap)
  currMenu:UpdateFooterButton("Skill", true)
  currMenu:UpdateFooterButton("Quest", true)
  currMenu:UpdateFooterButton("Settings", true)
  currMenu:UpdateFooterButton("Back", isSubListOpen)
  currMenu:UpdateFooterButton("Exit", not isSubListOpen)
  currMenu:UpdateFooterButton("Close", true)
  currMenu:UpdateFooterButtonText()
end
function WeaponOn:Update()
  if self.deferredSelectPickupTable.currPickupName ~= nil then
    local weaponToStreamFor = pickupUtil.GetPickupGO(self.deferredSelectPickupTable.character, self.deferredSelectPickupTable.creature, self.deferredSelectPickupTable.currPickupSlotName)
    local fullyStreamed = weaponToStreamFor == nil or game.Pickup.RequestLoadHighMips(weaponToStreamFor, self.deferredSelectPickupTable.currPickupName, self.deferredSelectPickupTable.currPickupStage)
    if fullyStreamed and attributeUtil.AttributesInitialized() then
      self:SelectPickupFinishedStreaming(self.deferredSelectPickupTable.currPickupSlotName, self.deferredSelectPickupTable.currPickupName, self.deferredSelectPickupTable.creature, self.deferredSelectPickupTable.character)
      tablex.Clear(self.deferredSelectPickupTable)
    end
  end
end
function WeaponOn:SubmenuSetup(currState)
  currState.menu = menu.Menu.New(currState, {
    FooterButtonInfo = {
      {
        Item = "Equip",
        Text = "[AdvanceButton] " .. util.GetLAMSMsg(lamsIDs.Equip)
      },
      {
        Item = "Select",
        Text = "[AdvanceButton] " .. util.GetLAMSMsg(lamsIDs.Select)
      },
      {
        Item = "Map",
        Text = "[TouchPad] " .. util.GetLAMSMsg(lamsIDs.Map),
        EventHandlers = {
          {
            Events = {
              "EVT_TouchPad_Release"
            },
            Handler = function()
              self:SendEventToUIFsm("globalMenu", "EVT_GO_TO_MAP")
            end
          }
        }
      },
      {
        Item = "Skill",
        Text = "",
        EventHandlers = {
          {
            Events = {
              "EVT_GO_TO_SKILL_TREE_MENU"
            },
            Handler = function()
              self:SendEventToUIFsm("globalMenu", "EVT_GO_TO_SKILL_TREE")
            end
          }
        }
      },
      {
        Item = "Quest",
        Text = "",
        EventHandlers = {
          {
            Events = {
              "EVT_GO_TO_QUEST_MENU"
            },
            Handler = function()
              self:SendEventToUIFsm("globalMenu", "EVT_GO_TO_QUEST")
            end
          }
        }
      },
      {
        Item = "Settings",
        Text = "[TriangleButton] " .. util.GetLAMSMsg(lamsIDs.Options),
        EventHandlers = {
          {
            Events = {
              "EVT_Triangle_Release"
            },
            Handler = function()
              self:Menu_Triangle_ReleaseHandler()
            end
          }
        }
      },
      {
        Item = "Back",
        Text = "[BackButton] " .. util.GetLAMSMsg(lamsIDs.Back)
      },
      {
        Item = "Exit",
        Text = "[BackButton] " .. util.GetLAMSMsg(lamsIDs.Exit)
      },
      {
        Item = "Close",
        Text = "",
        EventHandlers = {
          {
            Events = {
              "EVT_Options_Release",
              "EVT_GO_TO_WEAPON_MENU"
            },
            Handler = function()
              self:Menu_Options_ReleaseHandler()
            end
          }
        }
      }
    }
  })
  local weaponUpgradeSlotsList = list.List.New(currState, {
    MaxFocusableObjectCount = 4,
    ListObjectName = "WeaponUpgradeSlotList",
    HoverToSelect = true,
    EventHandlers = {
      {
        Events = {
          "EVT_Advance_Release"
        },
        Handler = function()
          currState:MainList_AdvanceReleaseHandler()
        end
      },
      {
        Events = {
          "EVT_MOUSE_RELEASED"
        },
        Handler = function()
          local selected = UI.GetEventSenderGameObject()
          local list = currState.menu:GetList("MainList")
          local button = list:GetSelectedButton()
          if button ~= nil and selected == button:GetInstancedChildObject() then
            currState:MainList_AdvanceReleaseHandler()
          end
        end
      },
      {
        Events = {
          "EVT_Back_Release"
        },
        Handler = function()
          currState:MainList_BackReleaseHandler()
        end
      }
    },
    Button_Update = function(button)
      self:MainList_Button_Update(currState, button)
    end,
    Button_OnGainFocus = function(button)
      self:MainList_Button_OnGainFocus(currState, button)
    end,
    Button_HighlightOn = function(button, animateImmediately)
      buttonUtil.StatsMenuPickupSlot_HighlightOn(button, animateImmediately)
    end,
    Button_HighlightOff = function(button, animateImmediately)
      buttonUtil.StatsMenuPickupSlot_HighlightOff(button, animateImmediately)
    end
  })
  currState.menu:SetList("MainList", weaponUpgradeSlotsList)
  tutorialUtil.RegisterDesaturationList(currState.menu:get_name() .. "_MainList", weaponUpgradeSlotsList)
  currState.menu:SetupSubLists("PickupList_EquippedItem", self:GetWeaponUpgradeSlotsListNames(currState.substateCharacter, currState.substatePickupSlot), nil, nil, nil)
  currState.waitingForEquippedAttributes = false
end
function WeaponOn:SubmenuEnter(currState)
  currState.menu:Activate()
  self.currentCharacter = currState.substateCharacter
  currState.waitingForEquippedAttributes = false
  if currState.substateCharacter == pickupConsts.TAG_PICKUP_SON or currState.substatePickupSlot == pickupConsts.Slot_ArmorShield then
    self:SendEventToUIFsm("inWorldMenu", "EVT_HIDE_ATTRIBUTE_CARD")
  else
    self:SendEventToUIFsm("inWorldMenu", "EVT_SHOW_ATTRIBUTE_CARD")
  end
  local weaponUpgradeSlotsList = currState.menu:GetList("MainList")
  if self.weaponMenu.pauseMenuOpened == true then
    local init_useOnGainFocus = false
    weaponUpgradeSlotsList:SetSelectedButton(1, init_useOnGainFocus)
    self.weaponMenu.pauseMenuOpened = false
  end
  local newItemArray = self:GetWeaponUpgradeSlotsListNames(currState.substateCharacter, currState.substatePickupSlot)
  local showList = true
  local useOnGainFocus = false
  local itemDetermineFocusabilityFunc
  local itemGetTextFunc = pickupUtil.GetSlotDisplayName
  weaponUpgradeSlotsList:Refresh(newItemArray, showList, useOnGainFocus, itemDetermineFocusabilityFunc, itemGetTextFunc)
  animationUtil.ResetPickupSlotPositions(weaponUpgradeSlotsList)
  self:UpdateCamera(currState.substateCharacter, currState.substatePickupSlot)
  local reselect_useOnGainFocus = not currState.menu:HasInstructionEntryForMenuState()
  weaponUpgradeSlotsList:ReselectSelectedItem(reselect_useOnGainFocus)
  currState.menu:ExecuteInstructions()
end
function WeaponOn:SubmenuExit(currState)
  currState.menu:Deactivate(true)
end
function WeaponOn:EVT_PREP_IMMEDIATE_ITEM_SELECT(pickup)
  if pickup ~= nil then
    self._itemSelectOverride = pickup
  end
end
function WeaponOn:UpdatePickupListBySlot(currState, currPickupSlotName, useOnGainFocus, selectEquippedItem, isOnEquip)
  local subListName = currPickupSlotName
  local headerText = pickupUtil.GetSlotDisplayName(currPickupSlotName)
  local getSubListItemsFunc = function(pickupSlotName)
    local sortingGroup = resourceUtil.GetSortGroupFromMenu(self.menu)
    return resourceUtil.GetPickupsInSlotByCharacter(pickupSlotName, currState.substateCharacter, sortingGroup)
  end
  local creature = self:GetCreature(currState.substateCharacter, true)
  local equippedItem = pickupUtil.GetPickupNameInSlot(creature, currPickupSlotName)
  if self._itemSelectOverride ~= nil then
    equippedItem = self._itemSelectOverride
    self._itemSelectOverride = nil
  end
  local itemDetermineFocusabilityFunc
  local itemGetTextFunc = pickupUtil.GetDisplayName
  local deactivateSublist = isOnEquip and self:IsWeaponComponentSlot(currPickupSlotName)
  currState.menu:RefreshEquippedItemSublist(subListName, headerText, getSubListItemsFunc, equippedItem, itemDetermineFocusabilityFunc, itemGetTextFunc, useOnGainFocus, selectEquippedItem, isOnEquip, deactivateSublist)
  if not pickupUtil.IsWeaponSpecialSlot(currPickupSlotName) and currPickupSlotName ~= pickupConsts.Slot_ArmorShield then
    local sortListName = self.menu:get_sortListName()
    local sortList = self.menu:GetList(sortListName)
    if sortList:get_active() == false then
      local subList = currState.menu:GetList(subListName)
      if subList:GetButtonCount() > 1 then
        local sortList_showList = true
        local sortList_useOnGainFocus = true
        sortList:Activate(sortList_showList, sortList_useOnGainFocus)
        self.menu:AnimateSortList()
      end
    end
  end
end
function WeaponOn:Submenu_SquarePressHandler(currState)
  local weaponUpgradeSlotsList = currState.menu:GetList("MainList")
  local currWeaponUpgradeSlotName = weaponUpgradeSlotsList:GetSelectedItem()
  local currSubList = currState.menu:GetList(currWeaponUpgradeSlotName)
  if self._gemInfoCard ~= nil and currSubList:get_active() and currSubList:GetButtonCount() > 0 then
    local selectedPickup = currSubList:GetSelectedItem()
    if resourceUtil.GetMax(selectedPickup) > resourceUtil.GetAmount(selectedPickup) then
      self._gemInfoCard.holdFill_Left:PlayUnavailableSound()
    end
  end
end
function WeaponOn:Submenu_EquippedAttributesUpdatedHandler(currState)
  local weaponUpgradeSlotsList = currState.menu:GetList("MainList")
  local currWeaponUpgradeSlotName = weaponUpgradeSlotsList:GetSelectedItem()
  local currSubList = currState.menu:GetList(currWeaponUpgradeSlotName)
  if currState.waitingForEquippedAttributes == true then
    currSubList:Activate(false, false)
    currState.waitingForEquippedAttributes = false
  end
end
function WeaponOn:MainList_Button_Update(currState, button)
  local pickupSlotName = button:get_item()
  local equippedPickupName
  local equippedPickupColor = colors.WHITE
  if pickupSlotName ~= nil then
    local isHeaderSlot = self:IsHeaderSlot(pickupSlotName)
    local fadeAlpha = 1
    local fadeTime = 0
    button:AlphaFade(fadeAlpha, fadeTime)
    local goButton = button:GetGameObject()
    local goButtonChild = goButton.Child
    local creature = self:GetCreature(currState.substateCharacter, true)
    equippedPickupName = pickupUtil.GetPickupNameInSlot(creature, pickupSlotName)
    if not util.IsStringNilOrEmpty(equippedPickupName) then
      equippedPickupColor = resourceUtil.GetRarityColor(equippedPickupName)
    end
    if isHeaderSlot then
      buttonUtil.UpdateSlotButtonColor(button, equippedPickupName, equippedPickupColor)
    elseif self:IsWeaponComponentSlot(pickupSlotName) then
      util.SetGameObjectColor(goButtonChild, equippedPickupColor, "weaponButtonPommel_Glass_WorldUI", "Layer1", "cst_Tint")
      util.SetGameObjectColor(goButtonChild, equippedPickupColor, "weaponButtonPommel_Glass_WorldUI", "Layer1", "cst_EmissiveTint")
    else
      if not self:CanOpenSubList(pickupSlotName) then
        local fadeAlpha = 0.5
        local fadeTime = 0.25
        button:AlphaFade(fadeAlpha, fadeTime)
      end
      local axeColor = {
        0,
        0.8,
        1
      }
      local bladesColor = {
        1,
        0.8,
        0
      }
      local bowColor = {
        1,
        0.8,
        1
      }
      local weaponSpecialColor = colors.WEAPON_SPECIALS[currState.name]
      util.SetGameObjectColor(goButtonChild, weaponSpecialColor, "weaponButtonRune_Glass_WorldUI", "Layer1", "cst_Tint")
      util.SetGameObjectColor(goButtonChild, weaponSpecialColor, "weaponButtonRune_Glass_WorldUI", "Layer1", "cst_EmissiveTint")
    end
    if isHeaderSlot then
      local headerButtonCreature = self:GetCreature(currState.substateCharacter, false)
      local headerButtonEquippedPickupName = pickupUtil.GetPickupNameInSlot(headerButtonCreature, pickupSlotName)
      local iconMaterial = resourceUtil.GetMaterialSwapName(headerButtonEquippedPickupName)
      button:SetIcon(iconMaterial)
      button:UpdateNewIcon(function(button)
        if pickupSlotName == pickupConsts.Slot_ArmorShield then
          return resourceUtil.AnyNewPickupsInSlotByCharacter(pickupSlotName, currState.substateCharacter)
        else
          return false
        end
      end)
    else
      local thIconText = util.GetTextHandle(goButtonChild, "icon_text")
      if thIconText ~= nil then
        local icon_text = iconConsts.GetIcon(equippedPickupName)
        if icon_text == nil then
          icon_text = ""
        end
        UI.SetText(thIconText, icon_text)
      end
      button:UpdateNewIcon(function(button)
        local currItem = button:GetSortItem()
        return resourceUtil.AnyNewPickupsInSlotByCharacter(currItem, currState.substateCharacter)
      end)
    end
    button:SetText("text_level", "")
    local jidBacking = goButtonChild:GetJointIndex("backing")
    if equippedPickupName ~= nil then
      goButtonChild:ShowJoint(jidBacking)
    else
      goButtonChild:HideJoint(jidBacking)
    end
  end
end
function WeaponOn:MainList_Button_OnGainFocus(currState, button)
  local currSlotName = button:get_item()
  local currSlotList = button:get_list()
  local isSonBow = currState.substateCharacter == pickupConsts.TAG_PICKUP_SON and currSlotName == pickupConsts.Slot_PrimaryWeapon
  currSlotList:SetHeaderText(pickupUtil.GetSlotDisplayName(currSlotName))
  if currSlotName ~= nil then
    self:SelectWeaponUpgradeSlot(currState, currSlotName)
  end
  if not isSonBow and not self:CanOpenSubList(currSlotName) then
    self:SendEventToUIFsm("globalMenu", "EVT_SET_FOOTER_TOOL_TIP", util.GetLAMSMsg(lamsIDs.WeaponMenuTooltip))
  end
end
function WeaponOn:MainList_AdvanceReleaseHandler(currState)
  local weaponUpgradeSlotsList = currState.menu:GetList("MainList")
  local currPickupSlotName = weaponUpgradeSlotsList:GetSelectedItem()
  local currPickupList = currState.menu:GetList(currPickupSlotName)
  self:SendEventUI("EVT_WeaponMenu_OnAdvanceRelease", currPickupSlotName)
  if self:CanOpenSubList(currPickupSlotName) == true then
    self._refreshingListOnEnter = true
    local cameraState = currPickupSlotName
    if cameraState == "ArmorShield" then
      cameraState = "ArmorShieldFocused"
    end
    self:UpdateCamera(currState.substateCharacter, cameraState)
    local submenuList = self.menu:GetList(consts.inworldMenu_SubmenuList)
    local submenuList_hideList = true
    local submenuList_clearButtons = false
    submenuList:Deactivate(submenuList_hideList, submenuList_clearButtons)
    local hideList = false
    local clearButtons = false
    weaponUpgradeSlotsList:Deactivate(hideList, clearButtons)
    animationUtil.AnimatePickupSlotPositions(weaponUpgradeSlotsList, "PickupSlot_Selected_EquippedItem")
    weaponUpgradeSlotsList:HideHeader()
    local selectedButton = weaponUpgradeSlotsList:GetSelectedButton()
    selectedButton:UpdateNewIcon(nil)
    selectedButton:HighlightOff(true)
    if pickupUtil.IsWeaponSpecialSlot(currPickupSlotName) then
      self:SendEventToUIFsm("inWorldMenu", "EVT_HIDE_ATTRIBUTE_CARD")
    end
    self:ClearCards()
    local useOnGainFocus = true
    local selectEquippedItem = true
    local isOnEquip = false
    self:UpdatePickupListBySlot(currState, currPickupSlotName, useOnGainFocus, selectEquippedItem, isOnEquip)
    Audio.PlaySound("SND_UX_Menu_Camera_Pan_Slow")
    local subList = currState.menu:GetList(currPickupSlotName)
    if subList:GetButtonCount() > 0 then
      Audio.PlaySound("SND_UX_Menu_Tick_Confirm")
    else
      Audio.PlaySound("SND_UX_Menu_Tick_Fail")
    end
    self._refreshingListOnEnter = false
  elseif currPickupSlotName ~= pickupConsts.Slot_PrimaryWeapon and currPickupSlotName ~= pickupConsts.Slot_Blades then
    Audio.PlaySound("SND_UX_Menu_Tick_Fail")
  end
end
function WeaponOn:MainList_BackReleaseHandler()
  self:Menu_Back_ReleaseHandler()
end
function WeaponOn:SubList_Button_Update(currState, button)
  local creature = self:GetCreature(currState.substateCharacter, true)
  local currWeaponUpgradeName = button:get_item()
  local weaponUpgradeSlotsList = currState.menu:GetList("MainList")
  local currWeaponUpgradeSlotName = weaponUpgradeSlotsList:GetSelectedItem()
  local equippedWeaponUpgradeName = pickupUtil.GetPickupNameInSlot(creature, currWeaponUpgradeSlotName)
  local equipAllowed = self:EquipAllowed(currWeaponUpgradeSlotName)
  button:UpdateEquippedIcon(function(button)
    return pickupUtil.IsValidName(currWeaponUpgradeName) and currWeaponUpgradeName == equippedWeaponUpgradeName
  end)
  local textColor = colors.WHITE
  if not equipAllowed then
    textColor = colors.GRAY
  end
  button:SetTextColor(textColor)
  if pickupUtil.IsWeaponSpecialSlot(currWeaponUpgradeSlotName) then
    buttonUtil.UpdateColorQuad(button, nil)
  else
    local rarityColor = colors.WHITE
    if pickupUtil.IsValidName(currWeaponUpgradeName) == true then
      local currWeaponRarity = resourceUtil.GetRarity(currWeaponUpgradeName)
      rarityColor = pickupUtil.GetRarityColor(currWeaponRarity)
    end
    buttonUtil.UpdateColorQuad(button, rarityColor)
  end
  local effectFlag = resourceUtil.GetEffectFlag(currWeaponUpgradeSlotName, currWeaponUpgradeName)
  local effectFlagIcon = ""
  if effectFlag ~= nil then
    effectFlagIcon = iconConsts[effectFlag]
  end
  button:SetText("icon_text", effectFlagIcon)
  button:UpdateNewIcon(function(button)
    return buttonUtil.ShowNotification(button, consts.NotificationTypes.Resource)
  end)
end
function WeaponOn:SubList_Button_OnGainFocus(currState, button)
  local currWeaponUpgradeName = button:get_item()
  local weaponUpgradeSlotsList = currState.menu:GetList("MainList")
  local substatePickupSlotName = weaponUpgradeSlotsList:GetSelectedItem()
  local character = currState.substateCharacter
  local creature = self:GetCreature(character, false)
  local isSubListOpen = true
  local canOpenSubList = false
  local equipAllowed = self:EquipAllowed(substatePickupSlotName)
  self:SelectPickup(currState, substatePickupSlotName, currWeaponUpgradeName, creature, character, isSubListOpen, canOpenSubList)
  self:SendEventToUIFsm("inWorldMenu", "EVT_UPDATE_ATTRIBUTES_PREVIEW", character)
  if not equipAllowed then
    self:SendEventToUIFsm("globalMenu", "EVT_SET_FOOTER_TOOL_TIP", util.GetLAMSMsg(lamsIDs.RunicAttackCooldownTooltip))
  end
end
function WeaponOn:SubList_Button_OnLoseFocus(currState, button)
  if self._refreshingListOnEnter == false then
    local currWeaponUpgradeName = button:get_item()
    if currWeaponUpgradeName ~= nil then
      resourceUtil.ClearNew(currWeaponUpgradeName)
      button:UpdateNewIcon(function(button)
        return buttonUtil.ShowNotification(button, consts.NotificationTypes.Resource)
      end)
    end
    self:SendEventToUIFsm("inWorldMenu", "EVT_REFRESH_NOTIFICATIONS")
  end
end
local name_Axe = "axe00"
local name_Blades = "explosive00"
local name_Shield = "kratosShield00"
local name_Bow = "bow00"
local slotToFXName = {
  [pickupConsts.Slot_WeaponSpecial_Light] = {
    FX = "axegem_flourish",
    Joint = "JOTopGem1",
    Weapon = name_Axe
  },
  [pickupConsts.Slot_WeaponSpecial_Heavy] = {
    FX = "axegem_flourish",
    Joint = "JOBottomGem3",
    Weapon = name_Axe
  },
  [pickupConsts.Slot_WeaponComponent] = {
    FX = "pommel_flourish",
    Joint = "JOPelvis1",
    Weapon = name_Axe
  },
  [pickupConsts.Slot_WeaponSpecial_Blades_Light] = {
    FX = "bladegem_flourish",
    Joint = "JOTriangleGem1",
    Weapon = name_Blades
  },
  [pickupConsts.Slot_WeaponSpecial_Blades_Heavy] = {
    FX = "bladegem_flourish",
    Joint = "JOTriangleGem1",
    Weapon = name_Blades
  },
  [pickupConsts.Slot_WeaponComponent_Blades] = {
    FX = "blades_pommel_flourish",
    Joint = "JOPelvis1",
    Weapon = name_Blades
  },
  [pickupConsts.Slot_SonSpecial] = {
    FX = "bowgem_flourish",
    Joint = "JOCircleGem2",
    Weapon = name_Bow
  },
  [pickupConsts.Slot_ArmorShield] = {
    FX = "shield_flourish",
    Joint = "JOtopCircle1",
    Weapon = name_Shield
  }
}
function WeaponOn:SubList_AdvanceReleaseHandler(currState, currPickupSlotName)
  local creature = self:GetCreature(currState.substateCharacter, true)
  local currPickupList = currState.menu:GetList(currPickupSlotName)
  local currPickupName = currPickupList:GetSelectedItem()
  local equippedPickupName = pickupUtil.GetPickupNameInSlot(creature, currPickupSlotName)
  local equipAllowed = self:EquipAllowed(currPickupSlotName)
  if pickupUtil.IsValidName(currPickupName) and currPickupName ~= equippedPickupName and equipAllowed then
    if pickupUtil.IsWeaponSpecialSlot(currPickupSlotName) then
      Audio.PlaySound("SND_UX_Equip_Rune_Stone")
    elseif currPickupSlotName == pickupConsts.Slot_ArmorShield then
      Audio.PlaySound("SND_UX_Equip_Shield")
    else
      Audio.PlaySound("SND_UX_Equip_Weapon")
    end
    if pickupUtil.IsWeaponComponent(currPickupName) then
      currState.waitingForEquippedAttributes = true
    end
    local character = pickupConsts.TAG_PICKUP_KRATOS
    local acquirePickup = true
    local fromMenu = true
    characterUtil.SetPickupSlot(character, currPickupSlotName, currPickupName, acquirePickup, fromMenu)
    self:SendEventToUIFsm("inWorldMenu", "EVT_UPDATE_EQUIPPED_ATTRIBUTES", currState.substateCharacter)
    self:SendEventToUIFsm("inWorldMenu", "EVT_CLEAR_ATTRIBUTES_PREVIEW")
    self.statsCard:PlayFlourishOnEquip()
    local goUICreature = pickupUtil.GetUICreature(currState.substateCharacter)
    local particleName = slotToFXName[currPickupSlotName].FX
    local jointName = slotToFXName[currPickupSlotName].Joint
    local targetWeapon = slotToFXName[currPickupSlotName].Weapon
    local gameobj
    local firstBladeHit = false
    if goUICreature.IterateActiveWeapons ~= nil then
      for activeWeapon in goUICreature:IterateActiveWeapons() do
        gameobj = activeWeapon.Weapon
        local name = gameobj:GetName()
        if name == name_Blades and not firstBladeHit then
          firstBladeHit = true
        elseif name == targetWeapon then
          break
        end
      end
    end
    if gameobj ~= nil then
      animationUtil.SpawnFX(particleName, gameobj, jointName)
    end
    local goEquipFlourishJoint = util.GetUiObjByName("EquipFlourishGameObject")
    animationUtil.SpawnFX("equip_pickupslot_flourish", goEquipFlourishJoint, "")
    local weaponUpgradeSlotsList = currState.menu:GetList("MainList")
    local selectedButton = weaponUpgradeSlotsList:GetSelectedButton()
    selectedButton:Update()
    selectedButton:UpdateNewIcon(nil)
    local useOnGainFocus = true
    local selectEquippedItem = true
    local isOnEquip = true
    self:UpdatePickupListBySlot(currState, currPickupSlotName, useOnGainFocus, selectEquippedItem, isOnEquip)
  end
end
function WeaponOn:SubList_BackReleaseHandler(currState, weaponUpgradeSlotName)
  self:SendEventUI("EVT_WeaponMenu_Submenu_OnBackRelease", weaponUpgradeSlotName)
  local weaponUpgradeList = currState.menu:GetList(weaponUpgradeSlotName)
  local hideList = true
  local clearButtons = false
  weaponUpgradeList:Deactivate(hideList, clearButtons)
  local sortListName = self.menu:get_sortListName()
  local sortList = self.menu:GetList(sortListName)
  local sortList_hideList = true
  local sortList_clearButtons = false
  sortList:Deactivate(sortList_hideList, sortList_clearButtons)
  local weaponUpgradeSlotsList = currState.menu:GetList("MainList")
  local showList = true
  local useOnGainFocus = true
  weaponUpgradeSlotsList:Activate(showList, useOnGainFocus)
  weaponUpgradeSlotsList:ShowHeader()
  local submenuList = self.menu:GetList(consts.inworldMenu_SubmenuList)
  local submenuList_showList = true
  local submenuList_useOnGainFocus = true
  submenuList:Activate(submenuList_showList, submenuList_useOnGainFocus)
  local currWeapon = submenuList:GetSelectedItem()
  local currWeaponSlot = pickupUtil.GetSlotName(currWeapon)
  animationUtil.ResetPickupSlotPositions(weaponUpgradeSlotsList)
  self:UpdateCamera(currState.substateCharacter, currWeaponSlot)
  self:SendEventToUIFsm("inWorldMenu", "EVT_CLEAR_ATTRIBUTES_PREVIEW", currState.substateCharacter)
  local lookingAtSon = currState.substateCharacter == pickupConsts.TAG_PICKUP_SON
  local isWeaponSpecialSlot = pickupUtil.IsWeaponSpecialSlot(weaponUpgradeSlotName)
  if not lookingAtSon and isWeaponSpecialSlot then
    self:SendEventToUIFsm("inWorldMenu", "EVT_SHOW_ATTRIBUTE_CARD")
  end
  self._gemInfoCard:HideCard()
  Audio.PlaySound("SND_UX_Menu_Camera_Pan_Medium")
  Audio.PlaySound("SND_UX_Menu_Tick_Back")
end
function WeaponOn:Menu_Back_ReleaseHandler()
  self:SendEventToUIFsm("globalMenu", "EVT_TURN_OFF_GLOBAL_MENU")
end
function WeaponOn:Menu_Options_ReleaseHandler()
  self:Menu_Back_ReleaseHandler()
end
function WeaponOn:Menu_Triangle_ReleaseHandler()
  self:SendEventToUIFsm("globalMenu", "EVT_OPEN_SETTINGS_MENU")
end
function WeaponOn:EVT_TURN_OFF_WEAPON_MENU()
  self:Goto("WeaponOff")
end
function Axe:Setup()
  self.weaponOn = self:GetState("WeaponOn")
  self.substateCharacter = pickupConsts.TAG_PICKUP_KRATOS
  self.substatePickup = pickupConsts.Axe
  self.substatePickupSlot = pickupConsts.Slot_PrimaryWeapon
  self.weaponOn:SubmenuSetup(self)
end
function Axe:Enter()
  self.weaponOn:SubmenuEnter(self)
end
function Axe:Exit()
  self.weaponOn:SubmenuExit(self)
end
function Axe:MainList_AdvanceReleaseHandler()
  self.weaponOn:MainList_AdvanceReleaseHandler(self)
end
function Axe:MainList_BackReleaseHandler()
  self.weaponOn:MainList_BackReleaseHandler()
end
function Axe:SubList_Button_Update(button)
  self.weaponOn:SubList_Button_Update(self, button)
end
function Axe:SubList_Button_OnGainFocus(button)
  self.weaponOn:SubList_Button_OnGainFocus(self, button)
end
function Axe:SubList_Button_OnLoseFocus(button)
  self.weaponOn:SubList_Button_OnLoseFocus(self, button)
end
function Axe:SubList_AdvanceReleaseHandler(pickupSlotName)
  self.weaponOn:SubList_AdvanceReleaseHandler(self, pickupSlotName)
end
function Axe:SubList_BackReleaseHandler(weaponUpgradeSlotName)
  self.weaponOn:SubList_BackReleaseHandler(self, weaponUpgradeSlotName)
end
function Axe:SubList_R3ReleaseHandler(pickupSlotName)
  self.weaponOn:ToggleItemComparison(self)
end
function Axe:EVT_Square_Press()
  self.weaponOn:Submenu_SquarePressHandler(self)
end
function Axe:EVT_EQUIPPED_ATTRIBUTES_UPDATED()
  self.weaponOn:Submenu_EquippedAttributesUpdatedHandler(self)
end
function Shield:Setup()
  self.weaponOn = self:GetState("WeaponOn")
  self.substateCharacter = pickupConsts.TAG_PICKUP_KRATOS
  self.substatePickup = pickupConsts.Shield
  self.substatePickupSlot = pickupConsts.Slot_ArmorShield
  self.weaponOn:SubmenuSetup(self)
end
function Shield:Enter()
  self.weaponOn:SubmenuEnter(self)
end
function Shield:Exit()
  self.weaponOn:SubmenuExit(self)
end
function Shield:MainList_AdvanceReleaseHandler()
  self.weaponOn:MainList_AdvanceReleaseHandler(self)
end
function Shield:MainList_BackReleaseHandler()
  self.weaponOn:MainList_BackReleaseHandler()
end
function Shield:SubList_Button_Update(button)
  self.weaponOn:SubList_Button_Update(self, button)
end
function Shield:SubList_Button_OnGainFocus(button)
  self.weaponOn:SubList_Button_OnGainFocus(self, button)
end
function Shield:SubList_Button_OnLoseFocus(button)
  self.weaponOn:SubList_Button_OnLoseFocus(self, button)
end
function Shield:SubList_AdvanceReleaseHandler(pickupSlotName)
  self.weaponOn:SubList_AdvanceReleaseHandler(self, pickupSlotName)
end
function Shield:SubList_BackReleaseHandler(weaponUpgradeSlotName)
  self.weaponOn:SubList_BackReleaseHandler(self, weaponUpgradeSlotName)
end
function Shield:SubList_R3ReleaseHandler(pickupSlotName)
  self.weaponOn:ToggleItemComparison(self)
end
function Shield:EVT_Square_Press()
  self.weaponOn:Submenu_SquarePressHandler(self)
end
function Blades:Setup()
  self.weaponOn = self:GetState("WeaponOn")
  self.substateCharacter = pickupConsts.TAG_PICKUP_KRATOS
  self.substatePickup = pickupConsts.Blades
  self.substatePickupSlot = pickupConsts.Slot_Blades
  self.weaponOn:SubmenuSetup(self)
end
function Blades:Enter()
  self.weaponOn:SubmenuEnter(self)
end
function Blades:Exit()
  self.weaponOn:SubmenuExit(self)
end
function Blades:MainList_AdvanceReleaseHandler()
  self.weaponOn:MainList_AdvanceReleaseHandler(self)
end
function Blades:MainList_BackReleaseHandler()
  self.weaponOn:MainList_BackReleaseHandler()
end
function Blades:SubList_Button_Update(button)
  self.weaponOn:SubList_Button_Update(self, button)
end
function Blades:SubList_Button_OnGainFocus(button)
  self.weaponOn:SubList_Button_OnGainFocus(self, button)
end
function Blades:SubList_Button_OnLoseFocus(button)
  self.weaponOn:SubList_Button_OnLoseFocus(self, button)
end
function Blades:SubList_AdvanceReleaseHandler(pickupSlotName)
  self.weaponOn:SubList_AdvanceReleaseHandler(self, pickupSlotName)
end
function Blades:SubList_BackReleaseHandler(weaponUpgradeSlotName)
  self.weaponOn:SubList_BackReleaseHandler(self, weaponUpgradeSlotName)
end
function Blades:SubList_R3ReleaseHandler(pickupSlotName)
  self.weaponOn:ToggleItemComparison(self)
end
function Blades:EVT_Square_Press()
  self.weaponOn:Submenu_SquarePressHandler(self)
end
function Blades:EVT_EQUIPPED_ATTRIBUTES_UPDATED()
  self.weaponOn:Submenu_EquippedAttributesUpdatedHandler(self)
end
function SonBow:Setup()
  self.weaponOn = self:GetState("WeaponOn")
  self.substateCharacter = pickupConsts.TAG_PICKUP_SON
  self.substatePickup = pickupConsts.SonBow
  self.substatePickupSlot = pickupConsts.Slot_PrimaryWeapon
  self.weaponOn:SubmenuSetup(self)
end
function SonBow:Enter()
  self.weaponOn:SubmenuEnter(self)
  local character = pickupConsts.TAG_PICKUP_SON
  local currentSlotName = pickupConsts.Slot_PrimaryWeapon
  self.weaponOn:UpdateCamera(character, currentSlotName)
  Audio.PlaySound("SND_UX_Weapon_Menu_Bow_Enter")
end
function SonBow:Exit()
  self.weaponOn:SubmenuExit(self)
  local character = pickupConsts.TAG_PICKUP_SON
  local currentSlotName = "SonBow_Exit"
  self.weaponOn:UpdateCamera(character, currentSlotName)
  Audio.PlaySound("SND_UX_Weapon_Menu_Bow_Exit")
end
function SonBow:MainList_AdvanceReleaseHandler()
  self.weaponOn:MainList_AdvanceReleaseHandler(self)
end
function SonBow:MainList_BackReleaseHandler()
  self.weaponOn:MainList_BackReleaseHandler()
end
function SonBow:SubList_Button_Update(button)
  self.weaponOn:SubList_Button_Update(self, button)
end
function SonBow:SubList_Button_OnGainFocus(button)
  self.weaponOn:SubList_Button_OnGainFocus(self, button)
end
function SonBow:SubList_Button_OnLoseFocus(button)
  self.weaponOn:SubList_Button_OnLoseFocus(self, button)
end
function SonBow:SubList_AdvanceReleaseHandler(pickupSlotName)
  self.weaponOn:SubList_AdvanceReleaseHandler(self, pickupSlotName)
end
function SonBow:SubList_BackReleaseHandler(weaponUpgradeSlotName)
  self.weaponOn:SubList_BackReleaseHandler(self, weaponUpgradeSlotName)
end
function SonBow:EVT_Square_Press()
  self.weaponOn:Submenu_SquarePressHandler(self)
end
function WeaponMenu:OnSaveCheckpoint(tab)
end
function WeaponMenu:OnRestoreCheckpoint(tab)
end
