local tablex = require("core.tablex")
local animationUtil = require("ui.animationUtil")
local attributeUtil = require("ui.attributeUtil")
local colors = require("ui.colors")
local colorUtil = require("ui.colorUtil")
local consts = require("ui.consts")
local frame = require("ui.frame")
local frameGroup = require("ui.frameGroup")
local lamsConsts = require("ui.lamsConsts")
local pickupUtil = require("ui.pickupUtil")
local pickupConsts = require("ui.pickupConsts")
local resourceConsts = require("ui.resourceConsts")
local resourceUtil = require("ui.resourceUtil")
local recipeUtil = require("ui.recipeUtil")
local runeUtil = require("ui.runeUtil")
local util = require("ui.util")
local UI = game.UI
local ExchangeCard = {}
local MAX_RECIPE_OUTPUTS = 3
function ExchangeCard.New(state, gameObjectReference)
  assert(gameObjectReference ~= nil, "GameObject reference entered into StatsCard constructor was nil.")
  assert(state ~= nil, "FSM State entered into StatsCard constructor was nil.")
  local newTable = {
    _gameObject = gameObjectReference,
    _state = state,
    _currentItemName = nil,
    _itemType = nil,
    _position = nil,
    _frameGroupCore = nil,
    children = {goBackframe = nil}
  }
  local mt = {__index = ExchangeCard}
  setmetatable(newTable, mt)
  return newTable
end
function ExchangeCard:Init()
  self.children.goRoot = self._gameObject:FindSingleGOByName("Root")
  self.children.goBackframe = self._gameObject:FindSingleGOByName("backFrameGroup")
  self.children.goBodyBackingColor = self.children.goRoot:FindSingleGOByName("body_backingColor")
  self.thItemTitle = util.GetTextHandle(self.children.goRoot, "ItemTitle")
  self.thItemIcon = util.GetTextHandle(self.children.goRoot, "ItemIcon")
  self.thItemDescription = util.GetTextHandle(self.children.goRoot, "ItemDescription")
  self.thItemAmount = util.GetTextHandle(self.children.goRoot, "ItemAmount")
  self.thItemRarity = util.GetTextHandle(self.children.goRoot, "Text_Rarity")
  for _, gameObj in pairs(self.children) do
    assert(gameObj ~= nil, "Game object in exchange card was not found.")
  end
  local showAll = true
  self:ShowCard(showAll)
  self:HideCard()
end
function ExchangeCard:UpdateAccessibilityScaling()
  if UI.GetAccessibilityScaling == nil then
    return
  end
  self.children.goRoot:SetScale(UI.GetAccessibilityScaling(1, 1.15))
end
function ExchangeCard:SetItem(itemName, itemType)
  self:ClearCard()
  if itemName == nil then
    return
  end
  if self._previousItem ~= nil and itemName ~= self._previousItem then
    animationUtil.DoDefaultTransitionAnim(self.children.goRoot)
  end
  self._previousItem = itemName
  self._itemType = itemType
  self:SetTitle(itemName)
  self:SetIcon(itemName)
  self:SetMainDescription(itemName)
  self:SetMainOutput(itemName)
  local rarity = self:GetRarity(itemName, itemType)
  self:SetRarity(rarity)
end
function ExchangeCard:SetTitle(itemName)
  local titleText = ""
  if self._itemType == resourceConsts.RESOURCE_FLAG_RESOURCE then
    titleText = resourceUtil.GetDisplayName(itemName)
  else
    titleText = recipeUtil.RecipeListItem_GetDisplayName(itemName)
  end
  UI.SetText(self.thItemTitle, titleText)
end
function ExchangeCard:SetIcon(itemName)
  local iconText = ""
  if self._itemType == resourceConsts.RESOURCE_FLAG_RESOURCE then
    iconText = resourceUtil.GetIconText(itemName)
  else
    local primaryRecipeItem = recipeUtil.RecipeListItem_GetPrimaryItem(itemName)
    local primaryItemName = recipeUtil.GetRecipeItemName(primaryRecipeItem)
    iconText = resourceUtil.GetIconText(primaryItemName)
  end
  UI.SetText(self.thItemIcon, iconText)
end
function ExchangeCard:SetMainDescription(itemName)
  local descText = ""
  if self._itemType == resourceConsts.RESOURCE_FLAG_RESOURCE then
    descText = resourceUtil.GetDescription(itemName)
  else
    local recipeName = recipeUtil.RecipeListItem_GetRecipeName(itemName)
    local primaryRecipeItem = recipeUtil.RecipeListItem_GetPrimaryItem(itemName)
    if recipeUtil.HasFlag(recipeName, resourceConsts.RESOURCE_FLAG_BUY) then
      local descLamsID = recipeUtil.GetRecipeItemLamsDescriptionId(primaryRecipeItem)
      local primaryItemName = recipeUtil.GetRecipeItemName(primaryRecipeItem)
      descText = util.GetLAMSMsg(descLamsID, primaryItemName)
    elseif recipeUtil.HasFlag(recipeName, resourceConsts.RESOURCE_FLAG_SELL) and not recipeUtil.HasFlag(recipeName, resourceConsts.RESOURCE_FLAG_ARTIFACT) then
      local descLamsID = recipeUtil.GetRecipeItemLamsDescriptionId(primaryRecipeItem)
      local primaryItemName = recipeUtil.GetRecipeItemName(primaryRecipeItem)
      descText = util.GetLAMSMsg(descLamsID, primaryItemName)
    else
      descText = recipeUtil.RecipeListItem_GetDescription(itemName)
    end
  end
  UI.SetText(self.thItemDescription, descText)
end
function ExchangeCard:SetMainOutput(itemName)
  local resourceName = itemName
  local currAmount = ""
  if self._itemType == resourceConsts.RESOURCE_FLAG_RESOURCE then
    currAmount = resourceUtil.GetAmount(resourceName)
  else
    local recipeName = recipeUtil.RecipeListItem_GetRecipeName(itemName)
    local primaryRecipeItem = recipeUtil.RecipeListItem_GetPrimaryItem(itemName)
    if recipeUtil.HasFlag(recipeName, resourceConsts.RESOURCE_FLAG_BUY) then
      currAmount = recipeUtil.GetRecipeItemAmount(primaryRecipeItem)
    elseif recipeUtil.HasFlag(recipeName, resourceConsts.RESOURCE_FLAG_SELL) and not recipeUtil.HasFlag(recipeName, resourceConsts.RESOURCE_FLAG_ARTIFACT) then
      currAmount = recipeUtil.GetRecipeItemAmount(primaryRecipeItem)
    else
      local primaryItemName = recipeUtil.GetRecipeItemName(primaryRecipeItem)
      currAmount = resourceUtil.GetAmount(primaryItemName)
    end
  end
  UI.SetText(self.thItemAmount, tostring(currAmount))
end
function ExchangeCard:SetOutputs(itemName)
  local outputsTable = recipeUtil.GetOutputRecipeItems(itemName)
  for index, outputInfo in ipairs(outputsTable) do
    self.children.goOutputGrps[index]:Show()
    local name = ""
    local amount = ""
    if outputInfo.LamsNameId ~= 0 then
      name = "[BulletPoint]" .. util.GetLAMSMsg(outputInfo.LamsNameId)
    end
    local thOutput = util.GetTextHandle(self.children.goOutputGrps[index], "Output")
    UI.SetText(thOutput, name)
    if outputInfo.Amount ~= 1 then
      local outputString = outputInfo.Amount
      if outputString ~= nil then
        amount = outputString
      end
    end
    local thAmount = util.GetTextHandle(self.children.goOutputGrps[index], "Amount")
    UI.SetText(thAmount, amount)
  end
end
function ExchangeCard:GetRarity(itemName, itemType)
  local rarity
  if itemType == resourceConsts.RESOURCE_FLAG_RESOURCE then
    rarity = resourceUtil.GetRarity(itemName)
  else
    rarity = recipeUtil.RecipeListItem_GetRecipeRarity(itemName)
  end
  return rarity
end
function ExchangeCard:SetRarity(rarity)
  local lamsID = lamsConsts[rarity]
  if lamsID == nil then
    UI.SetText(self.thItemRarity, "")
  else
    UI.SetText(self.thItemRarity, util.GetLAMSMsg(lamsID))
  end
  local color, brightColor, darkColor
  if rarity ~= nil then
    color = colors.RARITY[rarity]
    darkColor = colors.RARITY_DARK[rarity]
    brightColor = colors.RARITY_BRIGHT[rarity]
  else
    color = colors.RARITY[resourceConsts.RESOURCE_FLAG_ECONOMY]
    darkColor = colors.RARITY_DARK[resourceConsts.RESOURCE_FLAG_ECONOMY]
    brightColor = colors.RARITY_BRIGHT[resourceConsts.RESOURCE_FLAG_ECONOMY]
  end
  self:SetCardColor(color, darkColor, brightColor)
  local targetTimelinePos = 0
  local animRate = 0
  if rarity == resourceConsts.RESOURCE_FLAG_PERFECT then
    targetTimelinePos = 1
  end
  UI.Anim(self.children.goRoot, consts.AS_Forward, "", animRate, targetTimelinePos)
  UI.Anim(self.children.goBodyBackingColor, consts.AS_Forward, "", animRate, targetTimelinePos)
end
function ExchangeCard:SetCardColor(color, color_dark, color_bright)
  if color_dark == nil then
    color_dark = color
  end
  if color_bright == nil then
    color_bright = color
  end
  local childObjTable = self.children
  local attribute = "cst_EmissiveTint"
  util.SetGameObjectColor(childObjTable.goBodyBackingColor, color_dark, "itemCard_HeaderBacking", "Layer0", attribute)
  util.SetGameObjectColor(childObjTable.goBodyBackingColor, color_bright, "itemCard_HeaderBacking", "Layer1", attribute)
end
function ExchangeCard:ClearCard()
end
function ExchangeCard:ShowCard(showAll)
  self._gameObject:Show()
  self._isShowing = true
  if showAll then
    for _, obj in pairs(self.children) do
      if type(obj) == "table" then
        self:ShowCard_Recursive(obj)
      else
        obj:Show()
      end
    end
  end
end
function ExchangeCard:ShowCard_Recursive(obj)
  if type(obj) == "table" then
    for _, childObj in pairs(obj) do
      self:ShowCard_Recursive(childObj)
    end
  else
    obj:Show()
  end
end
function ExchangeCard:HideCard()
  self._isShowing = false
  self._gameObject:Hide()
end
function ExchangeCard:GetCardHeight()
  return 17.55
end
function ExchangeCard:IsShowing()
  return self._isShowing
end
function ExchangeCard:SetAmount(amount)
  UI.SetText(self.thItemAmount, tostring(amount))
end
return {ExchangeCard = ExchangeCard}
