local classlib = require("core.class")
local tablex = require("core.tablex")
local colors = require("ui.colors")
local consts = require("ui.consts")
local controlsUtil = require("ui.controlsUtil")
local fsm = require("ui.fsm")
local lamsIDs = require("ui.lamsConsts")
local list = require("ui.list")
local menu = require("ui.menu")
local resourceUtil = require("ui.resourceUtil")
local saveUtil = require("ui.saveUtil")
local slider = require("ui.slider")
local sliderHoldFill = require("ui.sliderHoldFill")
local sliderTapFill = require("ui.sliderTapFill")
local toggle = require("ui.toggle")
local control = require("ui.control")
local util = require("ui.util")
local hudConsts = require("ui.hudConsts")
local pickupConsts = require("ui.pickupConsts")
local Audio = game.Audio
local Camera = game.Camera
local Combat = game.Combat
local PhotoMode = game.PhotoMode
local UI = game.UI
local SettingsMenu = classlib.Class("SettingsMenu", fsm.UIState)
local SettingsOff = SettingsMenu:StateClass("SettingsOff", fsm.UIState)
local ShellMenuDelay = SettingsMenu:StateClass("ShellMenuDelay", fsm.UIState)
local SettingsOn = SettingsMenu:StateClass("SettingsOn", fsm.UIState)
local Options = SettingsMenu:StateClass("Options", fsm.UIState)
local MainSettings = SettingsMenu:StateClass("MainSettings", fsm.UIState)
local SaveLoad = SettingsMenu:StateClass("SaveLoad", fsm.UIState)
local ControllerSettings = SettingsMenu:StateClass("ControllerSettings", fsm.UIState)
local HUDSettings = SettingsMenu:StateClass("HUDSettings", fsm.UIState)
local PhotoModeSettings = SettingsMenu:StateClass("PhotoModeSettings", fsm.UIState)
local AccessibilitySettings = SettingsMenu:StateClass("AccessibilitySettings", fsm.UIState)
local ViewControls = SettingsMenu:StateClass("ViewControls", fsm.UIState)
local ChangeControls = SettingsMenu:StateClass("ChangeControls", fsm.UIState)
local CameraSettings = SettingsMenu:StateClass("CameraSettings", fsm.UIState)
local AudioSettings = SettingsMenu:StateClass("AudioSettings", fsm.UIState)
local VideoSettings = SettingsMenu:StateClass("VideoSettings", fsm.UIState)
local GraphicsSettings = SettingsMenu:StateClass("GraphicsSettings", fsm.UIState)
local LanguageSettings = SettingsMenu:StateClass("LanguageSettings", fsm.UIState)
local Credits = SettingsMenu:StateClass("Credits", fsm.UIState)
local Bookmarks = SettingsMenu:StateClass("Bookmarks", fsm.UIState)
local WadSelect = SettingsMenu:StateClass("WadSelect", fsm.UIState)
local LevelSelect = SettingsMenu:StateClass("Select WAD", fsm.UIState)
local BookmarkSelect = SettingsMenu:StateClass("BookmarkSelect", fsm.UIState)
local Calibration = SettingsMenu:StateClass("Calibration", fsm.UIState)
local AdvancedSettings = SettingsMenu:StateClass("AdvancedSettings", fsm.UIState)
local ScreenScale = SettingsMenu:StateClass("ScreenScale", fsm.UIState)
local settingsMenu = SettingsMenu.New("settingsMenu", {
  SettingsOff,
  ShellMenuDelay,
  SettingsOn,
  {
    Options,
    MainSettings,
    SaveLoad,
    ControllerSettings,
    HUDSettings,
    PhotoModeSettings,
    AccessibilitySettings,
    ViewControls,
    ChangeControls,
    CameraSettings,
    AudioSettings,
    VideoSettings,
    GraphicsSettings,
    LanguageSettings,
    Credits,
    Bookmarks,
    WadSelect,
    LevelSelect,
    BookmarkSelect,
    Calibration,
    AdvancedSettings,
    ScreenScale
  }
})
function SettingsMenu:Setup()
end
function SettingsMenu:Enter()
  self.from_shell = false
  self.loadScreen = false
  self:WantPadEvents(true)
  self:turnoff()
end
function SettingsMenu:turnoff()
  self:Goto("SettingsOff")
end
function SettingsMenu:SaveSettings()
  UI.SavePrefs()
  UI.SoftSavePlayerState()
end
SettingsMenu.EVT_GAME_OVER = SettingsMenu.turnoff
SettingsMenu.EVT_Restart = SettingsMenu.turnoff
function SettingsOff:Setup()
  self.settingsMenu = self:GetState("settingsMenu")
  self.settingsOn = self:GetState("SettingsOn")
  self.controllerSettings = self:GetState("ControllerSettings")
  self.optionsMenu = self:GetState("Options")
end
function SettingsOff:Enter()
  util.Hide("settingsMenu")
  self.controllerSettings:HideAllTheThings()
  self.optionsMenu:ClearButtonText()
  self.settingsMenu.from_shell = false
  self.settingsMenu.loadScreen = false
  self.settingsMenu.ignore_button_click = false
end
function SettingsOff:Exit()
end
function SettingsOff:EVT_TURN_ON_SETTINGS_MENU(instructionEntries, instructionArgs)
  self.settingsOn.menu:set_instructionEntries(instructionEntries)
  self.settingsOn.menu:set_instructionArgs(instructionArgs)
  self.settingsMenu.from_shell = instructionArgs.from_shell
  self.settingsMenu.loadScreen = instructionArgs.loadScreen
  self.settingsMenu.overrideReturnToGame = instructionArgs.overrideReturnToGame
  self.settingsMenu.shellNewGamePlus = instructionArgs.shellNewGamePlus
  if self.settingsMenu.from_shell == true then
    self.settingsMenu.ignore_button_click = true
    self:Goto("ShellMenuDelay")
  else
    self.settingsMenu.ignore_button_click = false
    local overrideGoto = instructionArgs.overrideSubmenu
    if self.settingsMenu:GetState(overrideGoto) ~= nil then
      self.settingsMenu.overrideGoto = overrideGoto
    else
      self.settingsMenu.optionsOverrideSelect = overrideGoto
    end
    self:Goto("SettingsOn")
  end
end
function ShellMenuDelay:Enter()
  UI.PadClear()
  self.settingsMenu = self:GetState("settingsMenu")
end
function ShellMenuDelay:Update()
  if not self.settingsMenu.ignore_button_click then
    self:Goto("SettingsOn")
  else
    self.settingsMenu.ignore_button_click = false
  end
end
function SettingsOn:Setup()
  self.settingsMenu = self:GetState("settingsMenu")
  self.menu = menu.Menu.New(self, {})
end
function SettingsOn:Enter()
  util.Show("settingsMenu", "GlobalHeaderBacking")
  if self.settingsMenu.overrideReturnToGame then
    self:SendEventToUIFsm("globalMenu", "EVT_SET_FOOTER_TOOL_TIP", util.GetLAMSMsg(lamsIDs.CineInProgress))
  end
  if not self.menu:HasInstructionEntryForMenuState() then
    if self.settingsMenu.from_shell then
      if self.settingsMenu.loadScreen then
        self:Goto("SaveLoad")
      else
        UI.WorldUIRender("SettingsMenuCamera")
        self:Goto("MainSettings")
      end
    else
      UI.WorldUIRender("SettingsMenuCamera")
      self:Goto(self.settingsMenu.overrideGoto or "Options")
      self.settingsMenu.overrideGoto = nil
    end
  else
    UI.WorldUIRender("SettingsMenuCamera")
    self.menu:ExecuteInstructions()
  end
end
function SettingsOn:Exit()
  self.settingsMenu:SaveSettings()
  UI.WorldUIRender("")
  util.Hide("settingsMenu", "GlobalHeaderBacking")
  if self.settingsMenu.from_shell == true then
    UI.SendEvent("*", consts.EVT_SHELL_SETTINGS_MENU_CLOSED)
  end
end
function SettingsOn:RefreshSettingsList(settingsList, settingsInfoTable, header_text)
  if settingsList ~= nil then
    local newItemArray = settingsInfoTable
    local showList = false
    local useOnGainFocus = false
    local itemDetermineFocusabilityFunc = function(settingsInfo)
      if settingsInfo.Disabled ~= nil and settingsInfo.Disabled then
        return false
      end
      return settingsInfo.ButtonType ~= "category"
    end
    local itemGetTextFunc = function(settingsInfo)
      return settingsInfo.Text
    end
    settingsList:Refresh(newItemArray, showList, useOnGainFocus, itemDetermineFocusabilityFunc, itemGetTextFunc)
    local activate_showList = true
    local activate_useOnGainFocus = true
    settingsList:Activate(activate_showList, activate_useOnGainFocus)
    settingsList:SetHeaderText(header_text)
  end
end
function SettingsOn:UpdateWadList(list, wads)
  local newItemArray = wads
  local showList = true
  local useOnGainFocus = true
  local itemDetermineFocusabilityFunc, itemGetTextFunc
  list:Refresh(newItemArray, showList, useOnGainFocus, itemDetermineFocusabilityFunc, itemGetTextFunc)
end
function SettingsOn:BackReleaseHandler()
  if self.settingsMenu.from_shell == true or self.settingsMenu.overrideReturnToGame == true then
    self:SendEventToUIFsm("globalMenu", "EVT_TURN_OFF_GLOBAL_MENU")
  else
    self:SendEventToUIFsm("globalMenu", "EVT_OPEN_IN_WORLD_MENU")
    Audio.PlaySound("SND_UX_Pause_Menu_Settings_Exit")
  end
end
function SettingsOn:OptionsReleaseHandler()
  if self.from_shell ~= true then
    self:SendEventToUIFsm("globalMenu", "EVT_TURN_OFF_GLOBAL_MENU")
  end
end
function SettingsOn:SetDefaultValues(settingsList, menu, endFunction)
  if settingsList ~= nil then
    self:SendEventUI("EVT_SHOW_SETTINGS_DIALOG_MESSAGE", {
      Header = util.GetLAMSMsg(lamsIDs.ResetToDefault),
      Body = util.GetLAMSMsg(lamsIDs.AreYouSure_SettingsReset),
      ActiveMenus = {menu},
      ActiveLists = {settingsList},
      Events = {
        Advance = {
          Text = util.GetLAMSMsg(lamsIDs.Confirm),
          Handler = function()
            endFunction()
            settingsList:SetDefaultButtonValues()
          end
        },
        Back = {
          Text = util.GetLAMSMsg(lamsIDs.Back),
          Handler = function()
            endFunction()
          end
        }
      }
    })
  end
end
function SettingsOn:CheckDisabledButton(setting_list, index, disabled)
  if setting_list == nil then
    return
  end
  local setting_button = setting_list:GetButton(index)
  if setting_button then
    if disabled then
      setting_button:SetTextColor(colors.Text_Settings_Disabled, "medium_text_settings1")
    else
      setting_button:SetTextColor(colors.Text_Settings1, "medium_text_settings1")
    end
  end
end
function SettingsOn:ClearDisableButton(setting_list, index)
  if setting_list == nil then
    return
  end
  local setting_button = setting_list:GetButton(index)
  if setting_button then
    setting_button:SetTextColor(colors.Text_Settings1, "medium_text_settings1")
  end
end
function SettingsOn:EVT_TURN_OFF_SETTINGS_MENU()
  self:Goto("SettingsOff")
end
function Options:Setup()
  self:SetAllHandlersFalse()
  self.handling_event = false
  self.anyUsedSaveSlots = false
  self.settingsMenu = self:GetState("settingsMenu")
  self.settingsOn = self:GetState("SettingsOn")
  self.menu = menu.Menu.New(self, {
    FooterButtonInfo = {
      {
        Item = "Select",
        Text = "[AdvanceButton] " .. util.GetLAMSMsg(lamsIDs.Select)
      },
      {
        Item = "Back",
        Text = "[BackButton] " .. util.GetLAMSMsg(lamsIDs.Back)
      }
    }
  })
  local optionsList = list.List.New(self, {
    MaxFocusableObjectCount = 12,
    ListObjectName = "SettingsList",
    EmptyTextLamsID = lamsIDs.NoSettingsAvailable,
    BackingAnimFrameCount = 60,
    HoverToSelect = true,
    Button_OnGainFocus = function(button)
      local currItem = button:get_item()
      self:SelectMainSetting(currItem)
    end
  })
  self.menu:SetList("optionsList", optionsList)
  self.loadButtonIndex = 3
  self.saveButtonIndex = 2
end
function Options:GetSettingsList()
  local saveTip
  if saveUtil.GetPlayerStateSoftSaveDisabledInNifleheim() then
    saveTip = util.GetLAMSMsg(lamsIDs.NiflheimDisabledTip)
  end
  local allSettingsListItems = {
    {
      Item = "Bookmarks",
      Text = "Speclizer's Menu v1.00",
      Advance = function()
        self:Goto("Bookmarks")
      end,
      Platinum = true
    },
    {
      Item = "Restart",
      Text = util.GetLAMSMsg(lamsIDs.RestartFromCheckpoint),
      Advance = function()
        self:ConfirmRestart()
      end,
      Platinum = true
    },
    {
      Item = "Save",
      Text = util.GetLAMSMsg(lamsIDs.Save),
      Advance = function()
        if saveUtil.GetPlayerStateSoftSaveDisabledInNifleheim() then
          self.handling_event = false
        else
          self:EnterSaveScreen()
        end
      end,
      Platinum = true,
      ToolTip = saveTip
    },
    {
      Item = "Load",
      Text = util.GetLAMSMsg(lamsIDs.Load),
      Advance = function()
        if self.anyUsedSaveSlots then
          self:EnterLoadScreen()
        else
          self.handling_event = false
        end
      end,
      Platinum = true
    },
    {
      Item = "Settings",
      Text = util.GetLAMSMsg(lamsIDs.Settings),
      Advance = function()
        self:Goto("MainSettings")
        Audio.PlaySound("SND_UX_Menu_Tick_Confirm")
      end,
      Platinum = true
    },
    {
      Item = "PhotoMode",
      Text = util.GetLAMSMsg(lamsIDs.EnterPhotoMode),
      Advance = function()
        self:EnterPhotoMode()
      end,
      Platinum = true
    },
    {
      Item = "NG+",
      Text = util.GetLAMSMsg(lamsIDs["NewGame+"]),
      Advance = function()
        self:ConfirmNGPlus()
      end,
      NewGamePlus = true,
      Platinum = true
    },
    {
      Item = "Quit",
      Text = util.GetLAMSMsg(lamsIDs.QuitGame),
      Advance = function()
        self:ConfirmQuit()
      end,
      Platinum = true
    }
  }
  local availableSettingsListItems = {}
  local platinum = UI.IsPlatinum ~= nil and UI.IsPlatinum()
  local newGamePlus = UI.GetNewGamePlusAvailable ~= nil and UI.GetNewGamePlusAvailable()
  for _, item in ipairs(allSettingsListItems) do
    local platCheck = not platinum or platinum and item.Platinum
    local ngPlusCheck = not item.NewGamePlus or newGamePlus and item.NewGamePlus
    if platCheck and ngPlusCheck then
      tablex.FastInsert(availableSettingsListItems, item, #availableSettingsListItems + 1)
    end
  end
  return availableSettingsListItems
end
function Options:Enter()
  self:SetAllHandlersFalse()
  self.handling_event = false
  self.anyUsedSaveSlots = saveUtil.AreThereAnyUsedSaveSlots()
  self.menu:Activate()
  local settingsListItems = self:GetSettingsList()
  local optionsList = self.menu:GetList("optionsList")
  self.settingsOn:RefreshSettingsList(optionsList, settingsListItems, util.GetLAMSMsg(lamsIDs.Options))
  self.settingsOn:CheckDisabledButton(optionsList, self.loadButtonIndex, not self.anyUsedSaveSlots)
  self.settingsOn:CheckDisabledButton(optionsList, self.saveButtonIndex, saveUtil.GetPlayerStateSoftSaveDisabledInNifleheim())
  if self.settingsMenu.optionsOverrideSelect then
    local itemID = 1
    for i = 1, #settingsListItems do
      if settingsListItems[i].Item == self.settingsMenu.optionsOverrideSelect then
        itemID = i
        break
      end
    end
    self.settingsMenu.optionsOverrideSelect = nil
    optionsList:SetSelectedButton(itemID, true)
  end
  self.menu:UpdateFooterButton("Select", false)
  self.menu:UpdateFooterButton("Back", false)
  self.menu:UpdateFooterButtonText()
  self.menu:UpdateFooterButton("Select", true)
  self.menu:UpdateFooterButton("Back", true)
  self.menu:UpdateFooterButtonText()
end
function Options:Exit()
  self:ClearButtonText()
  self.menu:Deactivate(true)
end
function Options:ClearButtonText()
  local optionsList = self.menu:GetList("optionsList")
  self.settingsOn:ClearDisableButton(optionsList, self.loadButtonIndex)
  self.settingsOn:ClearDisableButton(optionsList, self.saveButtonIndex)
end
function Options:SelectMainSetting(currItem)
  self:UpdateFooterButtonPrompt()
end
function Options:ConfirmRestart()
  if UI.IsSlotValid(-1) then
    self:SendEventUI("EVT_SHOW_SETTINGS_DIALOG_MESSAGE", {
      Header = util.GetLAMSMsg(lamsIDs.RestartFromCheckpoint),
      Body = util.GetLAMSMsg(lamsIDs.LostData),
      ActiveMenus = {
        self.menu
      },
      ActiveLists = {
        self.menu:GetList("optionsList")
      },
      Events = {
        Advance = {
          Text = util.GetLAMSMsg(lamsIDs.Confirm),
          Handler = function()
            self:Restart()
          end
        },
        Back = {
          Text = util.GetLAMSMsg(lamsIDs.Back),
          Handler = function()
            self:Enter()
          end
        }
      }
    })
  else
    self:ConfirmQuit()
  end
end
function Options:Restart()
  self.settingsMenu:SaveSettings()
  UI.SendEvent("*", consts.EVT_RESTART_GAME_SETTINGS)
end
function Options:ConfirmQuit()
  self:SendEventUI("EVT_SHOW_SETTINGS_DIALOG_MESSAGE", {
    Header = util.GetLAMSMsg(lamsIDs.QuitGame),
    Body = util.GetLAMSMsg(lamsIDs.LostData),
    ActiveMenus = {
      self.menu
    },
    ActiveLists = {
      self.menu:GetList("optionsList")
    },
    Events = {
      Advance = {
        Text = util.GetLAMSMsg(lamsIDs.Confirm),
        Handler = function()
          self:Quit()
        end
      },
      Back = {
        Text = util.GetLAMSMsg(lamsIDs.Back),
        Handler = function()
          self:Enter()
        end
      }
    }
  })
end
function Options:ConfirmNGPlus()
  self:WantPadEvents(false)
  self:SendEventUI("EVT_SHOW_SETTINGS_DIALOG_MESSAGE", {
    Header = util.GetLAMSMsg(lamsIDs["NewGame+"]),
    Body = util.GetLAMSMsg(lamsIDs["ConfirmNG+"]),
    ActiveMenus = {
      self.menu
    },
    ActiveLists = {
      self.menu:GetList("optionsList")
    },
    Events = {
      Advance = {
        Text = util.GetLAMSMsg(lamsIDs.Confirm),
        Handler = function()
          self:WantPadEvents(true)
          self.menu:GetList("optionsList"):Hide()
          self:EnterSaveScreen(true)
        end
      },
      Back = {
        Text = util.GetLAMSMsg(lamsIDs.Back),
        Handler = function()
          self:WantPadEvents(true)
          self:Enter()
        end
      }
    }
  })
end
function Options:Quit()
  self.settingsMenu:SaveSettings()
  UI.SendEvent("*", consts.EVT_QUIT_GAME_SETTINGS)
end
function Options:EnterSaveScreen(fromNGPlus)
  self.settingsMenu.loadScreen = false
  if fromNGPlus then
    self.settingsMenu.startingNGPlus = true
  end
  self:Goto("SaveLoad")
  Audio.PlaySound("SND_UX_Menu_Tick_Confirm")
end
function Options:EnterLoadScreen()
  if self.anyUsedSaveSlots then
    self.settingsMenu.loadScreen = true
    self:Goto("SaveLoad")
    Audio.PlaySound("SND_UX_Menu_Tick_Confirm")
  end
end
function Options:EnterPhotoMode()
  if game.PhotoMode ~= nil then
    self:SendEventToUIFsm("photoModeHUD", "EVT_TURN_ON_PHOTO_MODE")
    self.settingsOn:OptionsReleaseHandler()
  end
end
function Options:UpdateFooterButtonPrompt()
  self.menu:UpdateFooterButton("Select", true)
  self.menu:UpdateFooterButton("Back", true)
  self.menu:UpdateFooterButtonText()
end
function Options:EVT_MOUSE_CLICKED()
  if not self.handling_event then
    self.advance = true
  end
end
function Options:EVT_MOUSE_RIGHT_CLICKED()
  if not self.handling_event then
    self.back = true
  end
end
function Options:EVT_Back_Release()
  if not self.handling_event then
    self.back = true
  end
end
function Options:EVT_Options_Release()
  if not self.handling_event then
    self.back_options = true
  end
end
function Options:EVT_Advance_Release()
  if not self.handling_event then
    self.advance = true
  end
end
function Options:HandleAdvanceOnList()
  local optionsList = self.menu:GetList("optionsList")
  local button = optionsList:GetSelectedButton()
  local advance_event = button:get_advance()
  if advance_event ~= nil then
    advance_event()
  else
    self.handling_event = false
  end
end
function Options:SetAllHandlersFalse()
  self.back = false
  self.back_options = false
  self.advance = false
end
function Options:Update()
  if not self.handling_event then
    if self.back then
      self.handling_event = true
      self:SetAllHandlersFalse()
      self.settingsOn:BackReleaseHandler()
    elseif self.back_options then
      self.handling_event = true
      self:SetAllHandlersFalse()
      self.settingsOn:OptionsReleaseHandler()
    elseif self.advance then
      self.handling_event = true
      self:SetAllHandlersFalse()
      self:HandleAdvanceOnList()
    end
  end
end
function MainSettings:Setup()
  self.player = nil
  self.IsKeyboardAndMouse = true
  self.backPressed = false
  self.settingsMenu = self:GetState("settingsMenu")
  self.settingsOn = self:GetState("SettingsOn")
  self.optionsMenu = self:GetState("Options")
  self.menu = menu.Menu.New(self, {
    FooterButtonInfo = {
      {
        Item = "Select",
        Text = "[AdvanceButton] " .. util.GetLAMSMsg(lamsIDs.Select)
      },
      {
        Item = "Back",
        Text = "[BackButton] " .. util.GetLAMSMsg(lamsIDs.Back)
      }
    }
  })
  local mainSettingsList = list.List.New(self, {
    MaxFocusableObjectCount = 12,
    ListObjectName = "SettingsList",
    EmptyTextLamsID = lamsIDs.NoSettingsAvailable,
    BackingAnimFrameCount = 60,
    HoverToSelect = true,
    Button_OnGainFocus = function(button)
      local currItem = button:get_item()
      self:SelectMainSetting(currItem)
    end
  })
  self.menu:SetList("mainSettings", mainSettingsList)
end
function MainSettings:GetSettingsList()
  local allSettingsListItems = {
    {
      Item = "GameplaySettings",
      Text = util.GetLAMSMsg(lamsIDs.GameplaySettings),
      Advance = function()
        self:Goto("ControllerSettings")
      end,
      ShowInGame = true
    },
    {
      Item = "HUDSettings",
      Text = util.GetLAMSMsg(lamsIDs.HUD),
      Advance = function()
        self:Goto("HUDSettings")
      end,
      ShowInGame = true
    },
    {
      Item = "CameraSettings",
      Text = util.GetLAMSMsg(lamsIDs.CameraSettings),
      Advance = function()
        self:Goto("CameraSettings")
      end,
      ShowInGame = true
    },
    {
      Item = "AudioSettings",
      Text = util.GetLAMSMsg(lamsIDs.AudioSettings),
      Advance = function()
        self:Goto("AudioSettings")
      end,
      ShowInGame = true
    },
    {
      Item = "VideoSettings",
      Text = util.GetLAMSMsg(lamsIDs.Display),
      Advance = function()
        self:Goto("VideoSettings")
      end,
      ShowInGame = true
    },
    {
      Item = "GraphicsSettings",
      Text = util.GetLAMSMsg(lamsIDs.Graphics),
      Advance = function()
        self:Goto("GraphicsSettings")
      end,
      ShowInGame = true
    },
    {
      Item = "PhotoModeSettings",
      Text = util.GetLAMSMsg(lamsIDs.PhotoMode),
      Advance = function()
        self:Goto("PhotoModeSettings")
      end,
      ShowInGame = true
    },
    {
      Item = "AccessibilitySettings",
      Text = util.GetLAMSMsg(lamsIDs.Accessibility),
      Advance = function()
        self:Goto("AccessibilitySettings")
      end,
      ShowInGame = true
    },
    {
      Item = "Language",
      Text = util.GetLAMSMsg(lamsIDs.Language),
      Advance = function()
        self:Goto("LanguageSettings")
      end,
      ShowInGame = false
    },
    {
      Item = "Credits",
      Text = util.GetLAMSMsg(lamsIDs.Credits),
      Advance = function()
        self:Goto("Credits")
      end,
      ShowInGame = true
    },
    {
      Item = "ViewControls",
      Text = util.GetLAMSMsg(lamsIDs.ViewControls),
      Advance = function()
        self:Goto("ViewControls")
      end,
      ShowInGame = true
    },
    {
      Item = "ChangeControls",
      Text = util.GetLAMSMsg(lamsIDs.ChangeControls),
      Advance = function()
        if self.IsKeyboardAndMouse then
          self:GetState("ChangeControls").returnToMain = true
          self:Goto("ChangeControls")
        else
          self.handling_event = false
        end
      end,
      ShowInGame = true
    }
  }
  local availableSettingsListItems = {}
  for _, item in ipairs(allSettingsListItems) do
    if self.settingsMenu.from_shell == true or self.settingsMenu.from_shell ~= true and item.ShowInGame == true then
      tablex.FastInsert(availableSettingsListItems, item, #availableSettingsListItems + 1)
    end
  end
  self.change_controls_button_index = #availableSettingsListItems
  return availableSettingsListItems
end
function MainSettings:Enter()
  self.player = game.Player.FindPlayer()
  self.IsKeyboardAndMouse = true
  self.backPressed = false
  self:SetAllHandlersFalse()
  self.handling_event = false
  self.menu:Activate()
  local settingsListItems = self:GetSettingsList()
  local mainSettingsList = self.menu:GetList("mainSettings")
  self.settingsOn:RefreshSettingsList(mainSettingsList, settingsListItems, util.GetLAMSMsg(lamsIDs.Settings))
  self.settingsOn:ClearDisableButton(mainSettingsList, self.optionsMenu.loadButtonIndex)
  self.settingsOn:ClearDisableButton(mainSettingsList, self.optionsMenu.saveButtonIndex)
end
function MainSettings:Exit()
  self.settingsOn:ClearDisableButton(self.menu:GetList("mainSettings"), self.change_controls_button_index)
  self.menu:Deactivate(true)
end
function MainSettings:SelectMainSetting(currItem)
  self:UpdateFooterButtonPrompt()
end
function MainSettings:UpdateFooterButtonPrompt()
  self.menu:UpdateFooterButton("Select", true)
  self.menu:UpdateFooterButton("Back", true)
  self.menu:UpdateFooterButton("Close", true)
  self.menu:UpdateFooterButtonText()
end
function MainSettings:BackReleaseHandler()
  if self.settingsMenu.from_shell == true then
    self:SendEventToUIFsm("globalMenu", "EVT_TURN_OFF_GLOBAL_MENU")
  else
    self:Goto("Options")
  end
  Audio.PlaySound("SND_UX_Menu_Tick_Back")
end
function MainSettings:EVT_MOUSE_CLICKED()
  if not self.handling_event then
    self.advance = true
  end
end
function MainSettings:EVT_MOUSE_RIGHT_CLICKED()
  if not self.handling_event then
    self.back = true
  end
end
function MainSettings:EVT_Back_Press()
  self.backPressed = true
end
function MainSettings:EVT_Back_Release()
  if not self.handling_event and self.backPressed then
    self.back = true
  end
  self.backPressed = false
end
function MainSettings:EVT_Options_Release()
  if not self.handling_event then
    self.back_options = true
  end
end
function MainSettings:EVT_Advance_Release()
  if not self.handling_event then
    self.advance = true
  end
end
function MainSettings:HandleAdvanceOnList()
  local mainList = self.menu:GetList("mainSettings")
  local button = mainList:GetSelectedButton()
  local advance_event = button:get_advance()
  if advance_event ~= nil then
    advance_event()
  else
    self.handling_event = false
  end
end
function MainSettings:SetAllHandlersFalse()
  self.back = false
  self.back_options = false
  self.advance = false
end
function MainSettings:Update()
  local IsKeyboardAndMouse = self.player.Pad.IsKeyboardAndMouse
  if self.IsKeyboardAndMouse ~= IsKeyboardAndMouse then
    self.settingsOn:CheckDisabledButton(self.menu:GetList("mainSettings"), self.change_controls_button_index, not IsKeyboardAndMouse)
    self.IsKeyboardAndMouse = IsKeyboardAndMouse
  end
  if not self.handling_event then
    if self.back then
      self.handling_event = true
      self:SetAllHandlersFalse()
      self:BackReleaseHandler()
    elseif self.back_options then
      self.handling_event = true
      self:SetAllHandlersFalse()
      self.settingsOn:OptionsReleaseHandler()
    elseif self.advance then
      self.handling_event = true
      self:SetAllHandlersFalse()
      self:HandleAdvanceOnList()
    end
  end
end
function SaveLoad:Setup()
  self.settingsMenu = self:GetState("settingsMenu")
  self.settingsOn = self:GetState("SettingsOn")
  self.advance = false
  self.back = false
  self.back_options = false
  self.handling_event = false
  self.advanceSave = false
  self.advanceOverwrite = false
  self.advanceLoad = false
  self.loadNGP = false
  self.saving = false
  self.min_save_time = false
  self.finished_save = false
  self.min_time_wait = 2
  self.max_time_wait = 3
  self.showedNGPlusPrompt = false
  self.loadScreen = false
  self.menu = menu.Menu.New(self, {
    FooterButtonInfo = {
      {
        Item = "AdvanceNGP",
        Text = "[AdvanceButton] " .. util.GetLAMSMsg(lamsIDs["NewGame+"])
      },
      {
        Item = "StartNGP",
        Text = "[SquareButton] " .. util.GetLAMSMsg(lamsIDs["NewGame+"])
      },
      {
        Item = "Save",
        Text = "[AdvanceButton] " .. util.GetLAMSMsg(lamsIDs.SaveLastCheckpoint)
      },
      {
        Item = "Overwrite",
        Text = "[AdvanceButton] " .. util.GetLAMSMsg(lamsIDs.OverwriteSave)
      },
      {
        Item = "Load",
        Text = "[AdvanceButton] " .. util.GetLAMSMsg(lamsIDs.Load)
      },
      {
        Item = "Back",
        Text = "[BackButton] " .. util.GetLAMSMsg(lamsIDs.Back)
      }
    }
  })
  local saveList = "saveSlotList"
  local saveSlotList = list.List.New(self, {
    MaxFocusableObjectCount = 5,
    ListObjectName = "SaveSlotList",
    EmptyTextLamsID = lamsIDs.NoSaveSlotsAvailable,
    BackingAnimFrameCount = 60,
    HoverToSelect = true,
    Button_OnGainFocus = function(button)
      local currItem = button:get_item()
      self:SelectSaveSlot(currItem)
      self:UpdateTextValuesForSlots()
    end
  })
  self.menu:SetList(saveList, saveSlotList)
end
function SaveLoad:Enter()
  UI.SetInMenu(true)
  self.shellNewGamePlus = self.settingsMenu.shellNewGamePlus
  self.loadScreen = self.settingsMenu.loadScreen
  self.menu:Activate()
  self.advanceSave = false
  self.advanceOverwrite = false
  self.advanceLoad = false
  self.handling_event = false
  self:SetAllHandlersFalse()
  self.saving = false
  self.min_save_time = false
  self.finished_save = false
  self.settingsOn = self:GetState("SettingsOn")
  local saveSlotList = self.menu:GetList("saveSlotList")
  local subheader = util.GetLAMSMsg(lamsIDs.Save)
  if self.loadScreen then
    subheader = util.GetLAMSMsg(lamsIDs.Load)
  end
  saveSlotList:SetHeaderText(subheader)
  self:UpdateSaveSlotLists(saveSlotList)
  local currSaveData = saveSlotList:GetSelectedItem()
  if currSaveData ~= nil then
    self:SelectSaveSlot(currSaveData)
  else
    self:UpdateFooterButtonPrompt(false, false, false)
  end
  if self.settingsMenu.startingNGPlus and not self.showedNGPlusPrompt then
    self.showedNGPlusPrompt = true
    self:WantPadEvents(false)
    self.handling_event = true
    self:SendEventUI("EVT_SHOW_SETTINGS_DIALOG_MESSAGE", {
      Header = util.GetLAMSMsg(lamsIDs["NewGame+"]),
      Body = util.GetLAMSMsg(lamsIDs["SaveBeforeNG+"]),
      ActiveMenus = {
        self.menu
      },
      ActiveLists = {
        self:GetSaveSlotList()
      },
      Events = {
        Advance = {
          Text = util.GetLAMSMsg(lamsIDs.Confirm),
          Handler = function()
            self:WantPadEvents(true)
            self.handling_event = false
            self:Enter()
          end
        }
      }
    })
  end
end
function SaveLoad:ConfirmNGPlus()
  self:WantPadEvents(false)
  self:SendEventUI("EVT_SHOW_SETTINGS_DIALOG_MESSAGE", {
    Header = util.GetLAMSMsg(lamsIDs["NewGame+"]),
    Body = util.GetLAMSMsg(lamsIDs["LoadConfirmNG+"]),
    ActiveMenus = {
      self.menu
    },
    ActiveLists = {
      self:GetSaveSlotList()
    },
    Events = {
      Advance = {
        Text = util.GetLAMSMsg(lamsIDs.Confirm),
        Handler = function()
          self:WantPadEvents(true)
          self.handling_event = false
          self.settingsMenu.loadScreen = false
          self.settingsMenu.startingNGPlus = true
          self.loadNGPlusSaveSlot = self.currSaveSlotIndex
          self:Enter()
        end
      },
      Back = {
        Text = util.GetLAMSMsg(lamsIDs.Back),
        Handler = function()
          self:WantPadEvents(true)
          self:Enter()
        end
      }
    }
  })
end
function SaveLoad:Exit()
  UI.SetInMenu(false)
  local showSave, showLoad, autoSave = true, false, false
  self:UpdateFooterButtonPrompt(showSave, showLoad, autoSave)
  self.menu:Deactivate(true)
  local saveSlotList = self.menu:GetList("saveSlotList")
  saveSlotList:SetHeaderText("")
  self.settingsMenu.startingNGPlus = false
  self.showedNGPlusPrompt = false
  self.loadNGPlusSaveSlot = nil
  self.saving = false
  Audio.PlaySound("SND_UX_Menu_Tick_Back")
end
function SaveLoad:GetSaveSlotList()
  return self.menu:GetList("saveSlotList")
end
function SaveLoad:UpdateSaveSlotLists(saveSlotList)
  local manualSlots = saveUtil.GetSaveSlots(not self.loadScreen)
  local showList = true
  local useOnGainFocus = false
  local itemDetermineFocusabilityFunc, itemGetTextFunc
  saveSlotList:Refresh(manualSlots, showList, useOnGainFocus, itemDetermineFocusabilityFunc, itemGetTextFunc)
  saveSlotList:Activate(showList, useOnGainFocus)
  self:UpdateTextValuesForSlots()
end
function SaveLoad:UpdateTextValuesForButton(button)
  local goButton = button:GetGameObject()
  local goButtonChild = goButton:FindSingleGOByName("button")
  local goNoSaveData = goButtonChild:FindSingleGOByName("NoSaveData")
  local goSaveData = goButtonChild:FindSingleGOByName("SaveData")
  local goSaveImage = goButtonChild:FindSingleGOByName("SaveImage")
  local goAutoSave = goSaveData:FindSingleGOByName("AutoSave")
  local goTextObjects = goSaveData:FindSingleGOByName("TextObjects")
  local goImpossible = goSaveData:FindSingleGOByName("Impossible")
  local saveData = button:get_item()
  goButton:Show()
  goButtonChild:Show()
  if not saveUtil.IsSaveDataEmpty(saveData) then
    goNoSaveData:Hide()
    goSaveData:Show()
    goTextObjects:Show()
    local th_realmName = util.GetTextHandle(goTextObjects, "RealmName")
    local th_regionName = util.GetTextHandle(goTextObjects, "RegionName")
    local th_questName = util.GetTextHandle(goTextObjects, "QuestName")
    local th_timeStamp = util.GetTextHandle(goTextObjects, "TimeStamp")
    UI.SetText(th_realmName, saveUtil.GetSaveSlotRealm(saveData))
    UI.SetText(th_regionName, saveUtil.GetSaveSlotRegion(saveData))
    UI.SetText(th_questName, saveUtil.GetSaveSlotQuest(saveData))
    UI.SetText(th_timeStamp, saveUtil.GetSaveSlotSaveTimeString(saveData))
    goSaveImage:SetMaterialSwap(saveUtil.GetSaveSlotImageMaterial(saveData))
    goSaveImage:Show()
    if saveUtil.IsAutoSaveSlot(saveData) then
      goAutoSave:Show()
    else
      goAutoSave:Hide()
    end
    if saveUtil.IsImpossibleSaveSlot(saveData) then
      goImpossible:Show()
    else
      goImpossible:Hide()
    end
    local newGamePlus = saveUtil.GetSaveSlotNewGamePlus(saveData) == 1
    if newGamePlus then
      UI.Anim(goButtonChild, consts.AS_Forward, "button_SaveLoad_NewGamePlusColor", 0, 1)
      UI.Anim(goTextObjects, consts.AS_Forward, "", 0, 1)
      UI.AnimNoShow(goImpossible, consts.AS_Forward, "", 0, 1)
    else
      UI.Anim(goButtonChild, consts.AS_Forward, "button_SaveLoad_NewGamePlusColor", 0, 0)
      UI.Anim(goTextObjects, consts.AS_Forward, "", 0, 0)
      UI.AnimNoShow(goImpossible, consts.AS_Forward, "", 0, 0)
    end
  else
    goSaveData:Hide()
    goNoSaveData:Show()
    goSaveImage:SetMaterialSwap(saveUtil.GetSaveSlotImageMaterial(saveData))
    goSaveImage:Show()
    UI.Anim(goButtonChild, consts.AS_Forward, "button_SaveLoad_NewGamePlusColor", 0, 0)
    UI.Anim(goTextObjects, consts.AS_Forward, "", 0, 0)
    UI.AnimNoShow(goImpossible, consts.AS_Forward, "", 0, 0)
  end
end
function SaveLoad:UpdateTextValuesForSlots()
  local saveSlotList = self:GetSaveSlotList()
  for i = 1, saveSlotList._maxFocusableObjectCount do
    local button
    local buttonIndex = saveSlotList._firstFocusableButtonIndex + i - 1
    button = saveSlotList:GetButton(buttonIndex)
    if button ~= nil then
      self:UpdateTextValuesForButton(button)
    end
  end
end
function SaveLoad:SelectSaveSlot(currSaveData)
  if currSaveData == nil then
    self:UpdateFooterButtonPrompt(false, false, false)
    return
  end
  local isEmpty = saveUtil.IsSaveDataEmpty(currSaveData)
  local currSaveSlotIndex = saveUtil.GetSaveSlotIndex(currSaveData)
  local isAuto = UI.IsAutoSaveSlot(currSaveSlotIndex)
  self:UpdateFooterButtonPrompt(isEmpty, not isEmpty, isAuto)
  self.currSaveSlotIndex = currSaveSlotIndex
end
function SaveLoad:SaveGame(saveData)
  self.finished_save = false
  self.min_save_time = false
  local currSaveData = saveData
  local saveSlotList = self:GetSaveSlotList()
  if currSaveData == nil then
    currSaveData = saveSlotList:GetSelectedItem()
  end
  if currSaveData == nil then
    self:Enter()
    return
  end
  if self.loadNGPlusSaveSlot then
    local currSaveSlot = saveUtil.GetSaveSlotIndex(currSaveData)
    game.UI.CopySaveGameToManualSave(currSaveSlot, self.loadNGPlusSaveSlot)
  else
    saveUtil.SaveGame(currSaveData)
  end
  if saveData == nil then
    self:UpdateSaveSlotLists(saveSlotList)
    self:SelectSaveSlot(currSaveData)
  end
  self.currSaveSlotIndex = saveUtil.GetSaveSlotIndex(currSaveData)
  self.saving = true
  if game.build.PLATFORM_PC ~= 1 then
    self:SendEventUI("EVT_SHOW_SETTINGS_DIALOG_MESSAGE", {
      Header = util.GetLAMSMsg(lamsIDs.SavingGame),
      Body = util.GetLAMSMsg(lamsIDs.SavingDontTurnOff),
      ActiveMenus = {
        self.menu
      },
      ActiveLists = {
        self:GetSaveSlotList()
      },
      OnDialogExit = function()
        self:Enter()
      end
    })
    self:StartTimer("CheckCloseSaveMenu", self.min_time_wait, function()
      self:AfterMinimumSaveTime()
    end)
  else
    self.min_save_time = true
    self:AfterMaximumSaveTime()
  end
end
function SaveLoad:AfterMinimumSaveTime()
  self.min_save_time = true
  self:DebugPrint("After Min Save Time")
  self:StartTimer("AfterMaxSaveMenu", self.max_time_wait, function()
    self:AfterMaximumSaveTime()
  end)
end
function SaveLoad:AfterMaximumSaveTime()
  self:DebugPrint("After MAX Save Time")
  if game.build.PLATFORM_PC ~= 1 then
    self:SendEventUI("EVT_FORCE_EXIT_DIALOG")
  end
  if self.settingsMenu.startingNGPlus then
    self:TriggerNGPlus()
  end
  if game.build.PLATFORM_PC == 1 then
    self:Enter()
  end
end
function SaveLoad:EVT_ManualSaveComplete()
  self:DebugPrint("Manual Save is Completed")
  self.finished_save = true
end
function SaveLoad:TriggerNGPlus()
  self:WantPadEvents(false)
  self.settingsMenu:SaveSettings()
  if UI.ResetIntoNewGamePlus ~= nil then
    UI.ResetIntoNewGamePlus(self.currSaveSlotIndex)
  end
end
function SaveLoad:ConfirmOverwriteSaveGame()
  local saveSlotList = self:GetSaveSlotList()
  local currSaveData = saveSlotList:GetSelectedItem()
  self:SendEventUI("EVT_SHOW_SETTINGS_DIALOG_MESSAGE", {
    Header = util.GetLAMSMsg(lamsIDs.OverwriteSave),
    Body = util.GetLAMSMsg(lamsIDs.ComfirmOverwriteSave),
    ActiveMenus = {
      self.menu
    },
    ActiveLists = {
      self:GetSaveSlotList()
    },
    Events = {
      Advance = {
        Text = util.GetLAMSMsg(lamsIDs.Confirm),
        Handler = function()
          self:SaveGame(currSaveData)
        end
      },
      Back = {
        Text = util.GetLAMSMsg(lamsIDs.Back),
        Handler = function()
          self:Enter()
        end
      }
    }
  })
end
function SaveLoad:LoadGame()
  local currSaveData = self:GetSaveSlotList():GetSelectedItem()
  local saveSlot = saveUtil.GetSaveSlotIndex(currSaveData)
  if UI.IsSlotPlayGoDownloaded(saveSlot) then
    if self.settingsMenu.from_shell then
      self:DoLoadSave(saveSlot)
    else
      self:SendEventUI("EVT_SHOW_SETTINGS_DIALOG_MESSAGE", {
        Header = util.GetLAMSMsg(lamsIDs.LoadGame),
        Body = util.GetLAMSMsg(lamsIDs.LostData),
        ActiveMenus = {
          self.menu
        },
        ActiveLists = {
          self:GetSaveSlotList()
        },
        Events = {
          Advance = {
            Text = util.GetLAMSMsg(lamsIDs.Confirm),
            Handler = function()
              self:DoLoadSave(saveSlot)
            end
          },
          Back = {
            Text = util.GetLAMSMsg(lamsIDs.Back),
            Handler = function()
              self:Enter()
            end
          }
        }
      })
    end
  else
    self:SendEventUI("EVT_SHOW_SETTINGS_DIALOG_MESSAGE", {
      Header = util.GetLAMSMsg(lamsIDs.DownloadInProgress),
      ActiveMenus = {
        self.menu
      },
      ActiveLists = {
        self:GetSaveSlotList()
      },
      Events = {
        Back = {
          Text = util.GetLAMSMsg(lamsIDs.Back)
        }
      },
      Progress = {
        Get = function()
          return UI.GetSlotPlayGoDownloadProgress(saveSlot)
        end,
        Complete = function()
          self:CompletedDownload(saveSlot)
        end,
        IsComplete = function()
          return UI.IsSlotPlayGoDownloaded(saveSlot)
        end
      },
      OnDialogExit = function()
        self:Enter()
      end
    })
  end
end
function SaveLoad:DoLoadSave(saveSlot)
  UI.SetInMenu(false)
  self.settingsMenu:SaveSettings()
  UI.SendEvent("*", consts.EVT_LOAD_SAVE_SLOT_SETTINGS, saveSlot)
end
function SaveLoad:CompletedDownload(saveSlot)
  UI.SendEvent("*", consts.EVT_LOAD_SAVE_SLOT_SETTINGS, saveSlot)
end
function SaveLoad:UpdateFooterButtonPrompt(showSave, showLoad, autoSave)
  self.advanceSave = not autoSave and showSave and not self.loadScreen
  self.advanceOverwrite = not autoSave and not showSave and not self.loadScreen
  self.advanceLoad = showLoad and self.loadScreen and not self.shellNewGamePlus
  self.loadNGP = game.UI.GetNewGamePlusAvailable() and not self.shellNewGamePlus and self.loadScreen and showLoad
  self.menu:UpdateFooterButton("StartNGP", self.loadNGP)
  self.menu:UpdateFooterButton("AdvanceNGP", self.shellNewGamePlus and self.loadScreen)
  self.menu:UpdateFooterButton("Save", self.advanceSave)
  self.menu:UpdateFooterButton("Overwrite", self.advanceOverwrite)
  self.menu:UpdateFooterButton("Load", self.advanceLoad)
  self.menu:UpdateFooterButton("Back", true)
  self.menu:UpdateFooterButtonText()
end
function SaveLoad:BackReleaseHandler()
  if self.settingsMenu.from_shell then
    self:SendEventToUIFsm("globalMenu", "EVT_TURN_OFF_GLOBAL_MENU")
  else
    self:Goto("Options")
  end
end
function SaveLoad:EVT_MOUSE_CLICKED()
  local selected = UI.GetEventSenderGameObject()
  local list = self.menu:GetList("saveSlotList")
  if selected == list._scrollBarMeter:get_gameObject() then
    return
  end
  if not self.handling_event and (self.advanceSave or self.advanceOverwrite or self.advanceLoad or self.shellNewGamePlus and self.loadScreen) then
    self.advance = true
  end
end
function SaveLoad:EVT_MOUSE_RIGHT_CLICKED()
  if not self.handling_event then
    self.back = true
  end
end
function SaveLoad:EVT_Back_Release()
  if not self.handling_event then
    self.back = true
  end
end
function SaveLoad:EVT_Options_Release()
  if not self.handling_event then
    self.back_options = true
  end
end
function SaveLoad:EVT_Advance_Release()
  if not self.handling_event and (self.advanceSave or self.advanceOverwrite or self.advanceLoad or self.shellNewGamePlus and self.loadScreen) then
    self.advance = true
  end
end
function SaveLoad:EVT_Square_Release()
  if not self.handling_event and self.loadNGP then
    self.square = true
  end
end
function SaveLoad:HandleAdvance()
  if self.advanceSave then
    Audio.PlaySound("SND_UX_Warning_Prompt_Exit")
    self:SaveGame()
  elseif self.advanceOverwrite then
    self:ConfirmOverwriteSaveGame()
  elseif self.advanceLoad then
    self:LoadGame()
  elseif self.shellNewGamePlus and self.loadScreen then
    self:ConfirmNGPlus()
  end
end
function SaveLoad:SetAllHandlersFalse()
  self.back = false
  self.back_options = false
  self.advance = false
  self.square = false
end
function SaveLoad:Update()
  if not self.handling_event and not self.saving then
    if self.back then
      self.handling_event = true
      self:SetAllHandlersFalse()
      self:BackReleaseHandler()
    elseif self.back_options then
      self.handling_event = true
      self:SetAllHandlersFalse()
      self.settingsOn:OptionsReleaseHandler()
    elseif self.advance then
      self.handling_event = true
      self:SetAllHandlersFalse()
      self:HandleAdvance()
    elseif self.square then
      self.handling_event = true
      self:SetAllHandlersFalse()
      self:ConfirmNGPlus()
    end
  elseif self.min_save_time and self.finished_save then
    if self:HaveTimer("AfterMaxSaveMenu") then
      self:StopTimer("AfterMaxSaveMenu")
    end
    if game.build.PLATFORM_PC ~= 1 then
      self:SendEventUI("EVT_FORCE_EXIT_DIALOG")
    end
    if self.settingsMenu.startingNGPlus then
      self:TriggerNGPlus()
    end
    if game.build.PLATFORM_PC == 1 then
      self:Enter()
    end
  end
end
function ControllerSettings:GetControllerGOs()
  self.goController = self.goSettingsController:FindSingleGOByName("Controller")
  self.goGenericButtons = self.goSettingsController:FindSingleGOByName("GenericButtons")
  self.goTriggerButtons = self.goSettingsController:FindSingleGOByName("TriggerButtons")
  self.goFaceButtons = self.goSettingsController:FindSingleGOByName("FaceButtons")
  self.goDpadButtons = self.goSettingsController:FindSingleGOByName("DpadButtons")
  self.goJoyLButton = self.goGenericButtons:FindSingleGOByName("JoyLButton")
  self.goQuickTurnTriggerButton = self.goSettingsController:FindSingleGOByName("QuickTurnTriggerButton")
  self.th_l1Button = util.GetTextHandle(self.goTriggerButtons, "L1")
  self.th_l2Button = util.GetTextHandle(self.goTriggerButtons, "L2")
  self.th_r1Button = util.GetTextHandle(self.goTriggerButtons, "R1")
  self.th_r2Button = util.GetTextHandle(self.goTriggerButtons, "R2")
  self.th_r2Button = util.GetTextHandle(self.goTriggerButtons, "R2")
  self.th_quickTurnTrigger = util.GetTextHandle(self.goQuickTurnTriggerButton)
  self.th_faceButtons = util.GetTextHandle(self.goFaceButtons)
  self.th_dpadButtons = util.GetTextHandle(self.goDpadButtons)
  self.goFaceButtonLines = self.goSettingsController:FindSingleGOByName("FaceButtonLines")
  self.goTriggerLines = self.goSettingsController:FindSingleGOByName("TriggerLines")
  self.goDpadButtonLines = self.goSettingsController:FindSingleGOByName("DpadButtonLines")
  self.goQuickTurnLLines = self.goSettingsController:FindSingleGOByName("QuickTurnL")
  self.goQuickTurnTriggerLines = self.goSettingsController:FindSingleGOByName("QuickTurnTrigger")
  self.goFooter = self.goSettingsController:FindSingleGOByName("Footer")
  self.th_Footer = util.GetTextHandle(self.goFooter)
end
function ControllerSettings:Setup()
  self.settingsOn = self:GetState("SettingsOn")
  self.settingsMenu = self:GetState("settingsMenu")
  self.handling_event = false
  self:SetAllHandlersFalse()
  self.player = nil
  self.IsKeyboardAndMouse = false
  self.controller_highmips_request = false
  self.controllerStreamedIn = false
  self.controllerType = ""
  self.togglesetDifficulty = 0
  self.entering = false
  self.difficulty_button_index = 1
  self.goSettingsController = util.GetUiObjByName("SettingsController")
  self:GetControllerGOs()
  self.goMenuFooterToolTip = util.GetUiObjByName("MenuFooterToolTip")
  self.thTooltip = util.GetTextHandle(self.goMenuFooterToolTip)
  self.menu = menu.Menu.New(self, {
    FooterButtonInfo = {
      {
        Item = "Default",
        Text = "[TriangleButton] " .. util.GetLAMSMsg(lamsIDs.Default)
      },
      {
        Item = "Back",
        Text = "[BackButton] " .. util.GetLAMSMsg(lamsIDs.Back)
      }
    }
  })
  local controllerSettingsList = list.List.New(self, {
    MaxFocusableObjectCount = 20,
    Name = "controllerSettingsList",
    ListObjectName = "MiddleSettingsList",
    EmptyTextLamsID = lamsIDs.NoSettingsAvailable,
    BackingAnimFrameCount = 60,
    HoverToSelect = true,
    Button_OnGainFocus = function(button)
      self:ControllerSettings_Button_OnGainFocus(button)
    end
  })
  self.menu:SetList("controllerSettings", controllerSettingsList)
  self.rumbleSlider = slider.Slider.New(self, {
    Name = "RumbleSlider",
    TextFormat = "%d",
    Get = UI.GetUserRumbleStrength,
    Set = function(num)
      UI.SetUserRumbleStrength(num)
    end,
    MinValue = UI.GetUserRumbleStrengthMin(),
    MaxValue = UI.GetUserRumbleStrengthMax(),
    Delta = UI.GetUserRumbleStrengthDelta,
    Default = function()
      return 10
    end
  })
  local InteractEvadeList = {
    {
      Value = 0,
      Text = util.GetLAMSMsg(lamsIDs.Default)
    },
    {
      Value = 1,
      Text = util.GetLAMSMsg(lamsIDs.Flipped)
    }
  }
  self.interactEvadeControllerConfigToggle = toggle.Toggle.New(self, {
    Name = "InteractEvadeControllerConfig",
    Get = function()
      return self:GetInteractEvadeControllerConfig()
    end,
    Set = function(settingValue)
      self:SetInteractEvadeControllerConfig(settingValue)
    end,
    Settings = InteractEvadeList,
    OnToggleFunc = function()
      self:UpdateTheInteractConfigButtons()
    end,
    Default = function()
      return 0
    end
  })
  self.interactEvadeControllerConfigToggle.originalList = InteractEvadeList
  local CombatList = {
    {
      Value = 0,
      Text = util.GetLAMSMsg(lamsIDs.Default)
    },
    {
      Value = 1,
      Text = util.GetLAMSMsg(lamsIDs.Flipped)
    },
    {
      Value = 2,
      Text = util.GetLAMSMsg(lamsIDs.Classic)
    }
  }
  self.combatControllerConfigToggle = toggle.Toggle.New(self, {
    Name = "CombatConfig",
    Get = function()
      return self:GetCombatControllerConfig()
    end,
    Set = function(settingValue)
      self:SetCombatControllerConfig(settingValue)
    end,
    Settings = CombatList,
    OnToggleFunc = function()
      self:UpdateTheControllerConfigButtons()
    end,
    Default = function()
      return 0
    end
  })
  self.combatControllerConfigToggle.originalList = CombatList
  self.quickTurnControllerConfigToggle = toggle.Toggle.New(self, {
    Name = "QuickTurnConfig",
    Get = function()
      return self:GetQuickTurnControllerConfig()
    end,
    Set = function(settingValue)
      self:SetQuickTurnControllerConfig(settingValue)
    end,
    Settings = {
      {
        Value = 0,
        Text = util.GetLAMSMsg(lamsIDs.Default)
      },
      {
        Value = 1,
        Text = util.GetLAMSMsg(lamsIDs.Option) .. " 1"
      },
      {
        Value = 2,
        Text = util.GetLAMSMsg(lamsIDs.Option) .. " 2"
      }
    },
    OnToggleFunc = function(value)
      if self.IsKeyboardAndMouse then
        local quickTurn = self:GetQuickTurnControllerConfig()
        if quickTurn == 0 then
          UI.SetText(self.th_Footer, util.GetLAMSMsg(lamsIDs.DefaultQuickTurn))
        elseif quickTurn == 1 then
          UI.SetText(self.th_Footer, util.GetLAMSMsg(lamsIDs.QuickTurnOption1))
        elseif quickTurn == 2 then
          UI.SetText(self.th_Footer, util.GetLAMSMsg(lamsIDs.QuickTurnOption2))
        end
      else
        self:UpdateTheQuickTurnConfigButtons(value)
      end
    end,
    Default = function()
      return 0
    end
  })
  self.lockOnEnabled = toggle.Toggle.New(self, {
    Name = "LockOnEnabled",
    Get = UI.GetUserLockOn,
    Set = function(num)
      UI.SetUserLockOn(num)
    end,
    Default = function()
      return true
    end
  })
  self.pauseEnabled = toggle.Toggle.New(self, {
    Name = "PauseEnabled",
    Get = UI.GetUserPauseOnFocusLoss,
    Set = function(num)
      UI.SetUserPauseOnFocusLoss(num)
    end,
    Default = function()
      return true
    end
  })
  self.tutorialToggle = toggle.Toggle.New(self, {
    Name = "TutorialToggle",
    Get = function()
      return UI.GetUserSuppressDesignerMessages()
    end,
    Set = function(settingValue)
      UI.SetUserSuppressDesignerMessages(settingValue)
      self:SendEventUI("EVT_REFRESH_HUD")
    end,
    Settings = {
      {
        Value = false,
        Text = util.GetLAMSMsg(lamsIDs.AllTutorials)
      },
      {
        Value = true,
        Text = util.GetLAMSMsg(lamsIDs.MinimalTutorial)
      }
    },
    Default = function()
      return false
    end
  })
  self.enableCineSkipToggle = toggle.Toggle.New(self, {
    Name = "EnableCineSkipToggle",
    Get = function()
      return UI.GetCineSkipEnabled and UI.GetCineSkipEnabled()
    end,
    Set = function(settingValue)
      if UI.SetCineSkipEnabled then
        UI.SetCineSkipEnabled(settingValue)
      end
    end,
    Default = function()
      return false
    end
  })
  self.dataCollectionToggle = toggle.Toggle.New(self, {
    Name = "DataCollectionToggle",
    Get = function()
      return UI.GetAllowDataCollection()
    end,
    Set = function(settingValue)
      UI.SetAllowDataCollection(settingValue)
    end,
    Default = function()
      return false
    end
  })
end
function ControllerSettings:GetGameplayList()
  self.difficultyToggle = toggle.Toggle.New(self, {
    Name = "DifficultyToggle",
    Settings = self:GetDifficultyToggleItems(),
    Get = Combat.GetDifficulty,
    Set = function(num)
      self.togglesetDifficulty = num
    end,
    OnToggleFunc = function(num)
      self:ShouldDisableDifficultyToggle(num)
    end
  })
  local controllerSettingsInfo = {
    {
      Item = "gameplaySettingsList",
      Text = util.GetLAMSMsg(lamsIDs.Difficulty),
      ActivationList = {
        {
          Item = self.difficultyToggle
        }
      },
      Controller = false,
      ShowInShell = false,
      ToggleTip = true
    },
    {
      Item = "RumbleSlider",
      Text = util.GetLAMSMsg(lamsIDs.Rumble),
      ActivationList = {
        {
          Item = self.rumbleSlider
        }
      },
      Controller = false,
      ShowInShell = true,
      ToolTip = util.GetLAMSMsg(lamsIDs.RumbleDesc)
    },
    {
      Item = "interactEvadeControllerConfigToggle",
      Text = util.GetLAMSMsg(lamsIDs.InteractEvadeConfig),
      ActivationList = {
        {
          Item = self.interactEvadeControllerConfigToggle
        }
      },
      Controller = true,
      ShowInShell = true,
      ToolTip = util.GetLAMSMsg(lamsIDs.InteractEvadeConfigTip)
    },
    {
      Item = "combatControllerConfigToggle",
      Text = util.GetLAMSMsg(lamsIDs.CombatConfig),
      ActivationList = {
        {
          Item = self.combatControllerConfigToggle
        }
      },
      Controller = true,
      ShowInShell = true,
      ToolTip = util.GetLAMSMsg(lamsIDs.CombatConfigTip)
    },
    {
      Item = "quickTurnControllerConfigToggle",
      Text = util.GetLAMSMsg(lamsIDs.QuickTurnConfig),
      ActivationList = {
        {
          Item = self.quickTurnControllerConfigToggle
        }
      },
      Controller = true,
      ShowInShell = true,
      ToolTip = util.GetLAMSMsg(lamsIDs.QuickTurnConfigTip)
    },
    {
      Item = "LockOnEnabled",
      Text = util.GetLAMSMsg(lamsIDs.LockOn),
      ActivationList = {
        {
          Item = self.lockOnEnabled
        }
      },
      ToolTip = util.GetLAMSMsg(lamsIDs.LockOnDesc),
      Controller = false,
      ShowInShell = true
    },
    {
      Item = "TutorialBool",
      Text = util.GetLAMSMsg(lamsIDs.TutorialMode),
      ActivationList = {
        {
          Item = self.tutorialToggle
        }
      },
      Controller = false,
      ShowInShell = true,
      ToolTip = util.GetLAMSMsg(lamsIDs.TutorialModeTip)
    },
    {
      Item = "PauseEnabled",
      Text = util.GetLAMSMsg(lamsIDs.PauseOnFocusLoss),
      ActivationList = {
        {
          Item = self.pauseEnabled
        }
      },
      ToolTip = util.GetLAMSMsg(lamsIDs.PauseOnFocusLossDesc),
      Controller = false,
      ShowInShell = true
    },
    {
      Item = "EnableCineSkip",
      Text = util.GetLAMSMsg(lamsIDs.EnableCineSkip),
      ActivationList = {
        {
          Item = self.enableCineSkipToggle
        }
      },
      Controller = false,
      ShowInShell = true,
      NewGamePlus = true,
      ToolTip = util.GetLAMSMsg(lamsIDs.EnableCineSkipDesc)
    },
    {
      Item = "DataCollection",
      Text = util.GetLAMSMsg(45153),
      ActivationList = {
        {
          Item = self.dataCollectionToggle
        }
      },
      Controller = false,
      ShowInShell = true,
      ToolTip = util.GetLAMSMsg(45154)
    }
  }
  local availableSettingsListItems = {}
  for _, item in ipairs(controllerSettingsInfo) do
    if (self.settingsMenu.from_shell ~= true or self.settingsMenu.from_shell == true and item.ShowInShell == true) and (UI.GetNewGamePlusAvailable() and item.NewGamePlus or not item.NewGamePlus) then
      tablex.FastInsert(availableSettingsListItems, item, #availableSettingsListItems + 1)
    end
  end
  return availableSettingsListItems
end
function ControllerSettings:Enter()
  self.player = game.Player.FindPlayer()
  self.IsKeyboardAndMouse = false
  self.goController:SetMaterialSwap("")
  self.controller_highmips_request = false
  self.controllerStreamedIn = false
  self.controllerType = ""
  self.handling_event = false
  self:SetAllHandlersFalse()
  self.entering = true
end
function ControllerSettings:ReEnterAfterDefault()
  self.handling_event = false
  self:SetAllHandlersFalse()
  if self.controllerStreamedIn then
    self:AfterStreamedIn()
  end
end
function ControllerSettings:AfterStreamedIn()
  self.menu:Activate()
  local controllerSettingsList = self.menu:GetList("controllerSettings")
  local controllerSettingsInfo = self:GetGameplayList()
  controllerSettingsList:Deactivate()
  self.settingsOn:RefreshSettingsList(controllerSettingsList, controllerSettingsInfo, util.GetLAMSMsg(lamsIDs.GameplaySettings))
  if self.entering then
    self.togglesetDifficulty = Combat.GetDifficulty()
  else
    self.difficultyToggle:ForceSetValue(self.togglesetDifficulty)
  end
  if not self.settingsMenu.from_shell and saveUtil.GetPlayerStateSoftSaveDisabledInNifleheim() then
    self.settingsOn:CheckDisabledButton(controllerSettingsList, self.difficulty_button_index, true)
    self.settingsOn:CheckDisabledButton(self.difficultyToggle, self.difficulty_button_index, true)
  end
  Audio.PlaySound("SND_UX_Menu_Tick_Confirm")
end
function ControllerSettings:Exit()
  if not self.settingsMenu.from_shell then
    local controllerSettingsList = self.menu:GetList("controllerSettings")
    self.settingsOn:ClearDisableButton(controllerSettingsList, self.difficulty_button_index)
    local diff_list = self:GetDifficultyToggleItems()
    for i, _ in ipairs(diff_list) do
      self.settingsOn:ClearDisableButton(self.difficultyToggle, i)
    end
  end
  local controllerSettingsInfo = self:GetGameplayList()
  local controllerSettingsList = self.menu:GetList("controllerSettings")
  controllerSettingsList:Deactivate()
  self.settingsOn:RefreshSettingsList(controllerSettingsList, controllerSettingsInfo, util.GetLAMSMsg(lamsIDs.GameplaySettings))
  for i = 1, #controllerSettingsInfo do
    if controllerSettingsInfo[i].ActivationList ~= nil then
      local toggle = controllerSettingsInfo[i].ActivationList[1].Item
      local setting_button = controllerSettingsList:GetButton(i)
      if toggle == self.interactEvadeControllerConfigToggle or toggle == self.combatControllerConfigToggle then
        local newList = toggle.originalList
        setting_button:SetTextColor(colors.Text_Settings1, "medium_text_settings1")
        toggle:EnableText()
        toggle:OverrideSettings(newList)
      end
    end
  end
  self.entering = false
  self.controllerStreamedIn = true
  self.menu:Deactivate(true)
  self:HideAllTheThings()
  Audio.PlaySound("SND_UX_Menu_Tick_Back")
end
function ControllerSettings:Update()
  local IsKeyboardAndMouse = self.player.Pad.IsKeyboardAndMouse
  if IsKeyboardAndMouse ~= self.IsKeyboardAndMouse then
    local controllerSettingsInfo = self:GetGameplayList()
    local controllerSettingsList = self.menu:GetList("controllerSettings")
    controllerSettingsList:Deactivate()
    self.settingsOn:RefreshSettingsList(controllerSettingsList, controllerSettingsInfo, util.GetLAMSMsg(lamsIDs.GameplaySettings))
    for i = 1, #controllerSettingsInfo do
      if controllerSettingsInfo[i].ActivationList ~= nil then
        local toggle = controllerSettingsInfo[i].ActivationList[1].Item
        local setting_button = controllerSettingsList:GetButton(i)
        if toggle == self.interactEvadeControllerConfigToggle or toggle == self.combatControllerConfigToggle then
          local newList = toggle.originalList
          if IsKeyboardAndMouse then
            setting_button:SetTextColor(colors.Text_Settings_Disabled, "medium_text_settings1")
            newList = {
              newList[1]
            }
            toggle:DisableText()
          else
            setting_button:SetTextColor(colors.Text_Settings1, "medium_text_settings1")
            toggle:EnableText()
          end
          toggle:OverrideSettings(newList)
        end
      end
    end
    self:HideAllTheThings()
    self.IsKeyboardAndMouse = IsKeyboardAndMouse
    if IsKeyboardAndMouse then
      local button = self.menu:GetList("controllerSettings"):GetSelectedButton()
      self:ControllerSettings_Button_OnGainFocus(button)
      self.controllerStreamedIn = true
      if self.entering then
        self:AfterStreamedIn()
        self.entering = false
      end
    else
      if self.controllerType == "xbox" then
        self.goSettingsController = util.GetUiObjByName("SettingsControllerXBox")
      else
        self.goSettingsController = util.GetUiObjByName("SettingsController")
      end
      self:GetControllerGOs()
      self.controllerStreamedIn = false
    end
  end
  local controllerType = game.GetControllerType()
  if self.controllerType ~= controllerType then
    self:HideAllTheThings()
    if controllerType == "xbox" then
      self.goSettingsController = util.GetUiObjByName("SettingsControllerXBox")
      self:GetControllerGOs()
    else
      if self.controllerType == "xbox" then
        self.goSettingsController = util.GetUiObjByName("SettingsController")
        self:GetControllerGOs()
      end
      self.goController:SetMaterialSwap(controllerType)
    end
    self.controllerType = controllerType
    self.controllerStreamedIn = false
  end
  if not self.controllerStreamedIn then
    self.goController:RequestLoadHighMips(true)
    if not self.controller_highmips_request then
      self.controller_highmips_request = true
    elseif self.goController:CheckMaterialFullyStreamed() then
      self.controllerStreamedIn = true
      if self.entering then
        self:AfterStreamedIn()
        self.entering = false
      end
      local button = self.menu:GetList("controllerSettings"):GetSelectedButton()
      self:ControllerSettings_Button_OnGainFocus(button)
    end
  end
  if not self.handling_event then
    if self.back then
      self.handling_event = true
      self:SetAllHandlersFalse()
      self:HandleBackButton(true)
    elseif self.back_options then
      self.handling_event = true
      self:SetAllHandlersFalse()
      self:HandleBackButton(false)
    elseif self.triangle then
      self.handling_event = true
      self:SetAllHandlersFalse()
      self:SetDefaultValues()
    end
  end
end
function ControllerSettings:HideAllTheThings()
  self.goSettingsController:Hide()
  self.goController:Hide()
  self.goFaceButtons:Hide()
  self.goFaceButtonLines:Hide()
  self.goTriggerButtons:Hide()
  self.goTriggerLines:Hide()
  self.goFooter:Hide()
  self:HideAllQuickTurnGOs()
end
function ControllerSettings:HideAllQuickTurnGOs()
  self.goDpadButtonLines:Hide()
  self.goDpadButtons:Hide()
  self.goQuickTurnLLines:Hide()
  self.goQuickTurnTriggerLines:Hide()
  self.goQuickTurnTriggerButton:Hide()
  self.goFaceButtons:Hide()
  self.goFaceButtonLines:Hide()
end
function ControllerSettings:GetDifficultyToggleItems()
  local impossible_description = util.GetLAMSMsg(lamsIDs.ImpossibleDesc) .. "\n" .. util.GetLAMSMsg(lamsIDs.ImpossibleDescNote)
  local t = {
    {
      Value = 0,
      Text = util.GetLAMSMsg(lamsIDs.EasyFlavor),
      ToolTip = util.GetLAMSMsg(lamsIDs.EasyDesc)
    },
    {
      Value = 1,
      Text = util.GetLAMSMsg(lamsIDs.NormalFlavor),
      ToolTip = util.GetLAMSMsg(lamsIDs.NormalDesc)
    },
    {
      Value = 2,
      Text = util.GetLAMSMsg(lamsIDs.HardFlavor),
      ToolTip = util.GetLAMSMsg(lamsIDs.HardDesc)
    },
    {
      Value = 3,
      Text = util.GetLAMSMsg(lamsIDs.ImpossibleFlavor),
      ToolTip = impossible_description
    }
  }
  if saveUtil.GetPlayerStateSoftSaveDisabledInNifleheim() then
    local diff = Combat.GetDifficulty()
    local temp_data = t[diff + 1]
    temp_data.ToolTip = util.GetLAMSMsg(lamsIDs.NiflheimDisabledTip)
    return {temp_data}
  else
    return t
  end
end
function ControllerSettings:ChangeDifficultyConfirmation(num)
  self:SendEventUI("EVT_SHOW_SETTINGS_DIALOG_MESSAGE", {
    Header = util.GetLAMSMsg(lamsIDs.DifficultyChanged),
    Body = util.GetLAMSMsg(lamsIDs.DifficultyChangeBody),
    ActiveMenus = {
      self.menu
    },
    ActiveLists = {
      self.menu:GetList("controllerSettingsList")
    },
    Events = {
      Advance = {
        Text = util.GetLAMSMsg(lamsIDs.Confirm),
        Handler = function()
          self:ChangeDifficulty(num)
        end
      },
      Back = {
        Text = util.GetLAMSMsg(lamsIDs.Back),
        Handler = function()
          self:Enter()
        end
      }
    }
  })
end
function ControllerSettings:CanChooseDifficulty(num)
  local currentDifficulty = Combat.GetDifficulty()
  if currentDifficulty == 3 and num ~= 3 then
    return false
  elseif currentDifficulty ~= 3 and num == 3 then
    return false
  end
  return true
end
function ControllerSettings:ShouldDisableDifficultyToggle(num)
  local currentDifficulty = Combat.GetDifficulty()
  if self:CanChooseDifficulty(num) and not saveUtil.GetPlayerStateSoftSaveDisabledInNifleheim() then
    self.difficultyToggle:EnableText()
  else
    self.difficultyToggle:DisableText()
  end
end
function ControllerSettings:ChangeDifficulty(num)
  Combat.SetDifficulty(num)
  self.settingsMenu:SaveSettings()
  UI.SendEvent("*", consts.EVT_RESTART_GAME_SETTINGS)
end
function ControllerSettings:UpdateTheInteractConfigButtons()
  self.goFaceButtons:Show()
  self.goFaceButtonLines:Show()
  controlsUtil.UpdateFaceButtonText(self.th_faceButtons, {
    "CircleButton",
    "XButton"
  })
end
function ControllerSettings:UpdateTheControllerConfigButtons()
  self.goFaceButtons:Show()
  self.goFaceButtonLines:Show()
  self.goTriggerButtons:Show()
  self.goTriggerLines:Show()
  controlsUtil.UpdateFaceButtonText(self.th_faceButtons, {
    "SquareButton",
    "TriangleButton"
  })
  controlsUtil.UpdateButton({
    button = "L1Button",
    th = self.th_l1Button,
    left_align = false
  })
  controlsUtil.UpdateButton({
    button = "L2Button",
    th = self.th_l2Button,
    left_align = false
  })
  controlsUtil.UpdateButton({
    button = "R1Button",
    th = self.th_r1Button,
    left_align = true
  })
  controlsUtil.UpdateButton({
    button = "R2Button",
    th = self.th_r2Button,
    left_align = true
  })
end
function ControllerSettings:UpdateTheQuickTurnConfigButtons(value)
  if value == nil then
    value = self:GetQuickTurnControllerConfig()
  end
  self:HideAllQuickTurnGOs()
  if value == 0 then
    UI.SetText(self.th_dpadButtons, util.GetLAMSMsg(lamsIDs.DefaultQuickTurn))
    self.goDpadButtonLines:Show()
    self.goDpadButtons:Show()
  elseif value == 1 then
    if self:GetCombatControllerConfig() ~= 2 then
      self.goQuickTurnTriggerButton:Show()
      UI.SetText(self.th_quickTurnTrigger, util.GetLAMSMsg(lamsIDs.QuickTurnOption1))
      self.goQuickTurnTriggerLines:Show()
    else
      self.goFaceButtons:Show()
      self.goFaceButtonLines:Show()
      UI.SetText(self.th_faceButtons, util.GetLAMSMsg(lamsIDs.QuickTurnOption1))
    end
  elseif value == 2 then
    self.goQuickTurnLLines:Show()
  end
end
function ControllerSettings:ControllerSettings_Button_OnGainFocus(button)
  local currControllerSettingsInfo = button:get_item()
  self:HideAllTheThings()
  if currControllerSettingsInfo.Controller then
    if self.IsKeyboardAndMouse then
      if currControllerSettingsInfo.Item == "quickTurnControllerConfigToggle" then
        self.goSettingsController:Show()
        self.goFooter:Show()
        local quickTurn = self:GetQuickTurnControllerConfig()
        if quickTurn == 0 then
          UI.SetText(self.th_Footer, util.GetLAMSMsg(lamsIDs.DefaultQuickTurn))
        elseif quickTurn == 1 then
          UI.SetText(self.th_Footer, util.GetLAMSMsg(lamsIDs.QuickTurnOption1))
        elseif quickTurn == 2 then
          UI.SetText(self.th_Footer, util.GetLAMSMsg(lamsIDs.QuickTurnOption2))
        end
      end
    else
      UI.AnimNoShow(self.goController, consts.AS_Forward, "", 0, 0)
      UI.AnimNoShow(self.goFaceButtons, consts.AS_Forward, "", 0, 0)
      UI.AnimNoShow(self.goTriggerButtons, consts.AS_Forward, "", 0, 0)
      UI.AnimNoShow(self.goDpadButtons, consts.AS_Forward, "", 0, 0)
      UI.AnimNoShow(self.goJoyLButton, consts.AS_Forward, "", 0, 0)
      UI.AnimNoShow(self.goQuickTurnTriggerButton, consts.AS_Forward, "", 0, 0)
      self.goSettingsController:Show()
      self.goController:Show()
      local item = currControllerSettingsInfo.Item
      if item == "interactEvadeControllerConfigToggle" then
        self:UpdateTheInteractConfigButtons()
      elseif item == "combatControllerConfigToggle" then
        self:UpdateTheControllerConfigButtons()
      elseif item == "quickTurnControllerConfigToggle" then
        self:UpdateTheQuickTurnConfigButtons()
      end
    end
  end
  self:UpdateFooterButtonPrompt()
end
function ControllerSettings:SetDefaultValues()
  local controllerSettingsList = self.menu:GetList("controllerSettings")
  self.settingsOn:SetDefaultValues(controllerSettingsList, self.menu, function()
    self:ReEnterAfterDefault()
  end)
end
function ControllerSettings:UpdateFooterButtonPrompt()
  self.menu:UpdateFooterButton("Default", true)
  self.menu:UpdateFooterButton("Back", true)
  self.menu:UpdateFooterButtonText()
end
function ControllerSettings:GetInteractEvadeControllerConfig()
  return game.GetCurrentControllerLayoutForConfig(tweaks.eControllerConfigTypes.kCC_InteractEvade)
end
function ControllerSettings:SetInteractEvadeControllerConfig(id)
  game.SetControllerLayoutForConfig(tweaks.eControllerConfigTypes.kCC_InteractEvade, id)
end
function ControllerSettings:GetCombatControllerConfig()
  return game.GetCurrentControllerLayoutForConfig(tweaks.eControllerConfigTypes.kCC_Combat)
end
function ControllerSettings:SetCombatControllerConfig(id)
  game.SetControllerLayoutForConfig(tweaks.eControllerConfigTypes.kCC_Combat, id)
end
function ControllerSettings:GetQuickTurnControllerConfig()
  if game.GetQuickTurnOptionNumber then
    return game.GetQuickTurnOptionNumber()
  end
  return 0
end
function ControllerSettings:SetQuickTurnControllerConfig(id)
  if game.SetQuickTurnOptionNumber then
    game.SetQuickTurnOptionNumber(id)
  end
end
function ControllerSettings:BackOut(back_button)
  if back_button then
    self:Goto("MainSettings")
  else
    self.settingsOn:OptionsReleaseHandler()
  end
end
function ControllerSettings:HandleBackButton(back_button)
  if self.togglesetDifficulty ~= Combat.GetDifficulty() then
    if Combat.GetDifficulty() == 3 or self.togglesetDifficulty == 3 then
      self:BackOut(back_button)
    else
      self:ChangeDifficultyConfirmation(self.togglesetDifficulty)
    end
  else
    self:BackOut(back_button)
  end
end
function ControllerSettings:EVT_MOUSE_CLICKED()
  if not self.handling_event then
    self.advance = true
  end
end
function ControllerSettings:EVT_MOUSE_RIGHT_CLICKED()
  if not self.handling_event then
    self.back = true
  end
end
function ControllerSettings:EVT_Back_Release()
  if not self.handling_event then
    self.back = true
  end
end
function ControllerSettings:EVT_Options_Release()
  if not self.handling_event then
    self.back_options = true
  end
end
function ControllerSettings:EVT_Triangle_Release()
  if not self.handling_event and not self.entering then
    self.triangle = true
  end
end
function ControllerSettings:SetAllHandlersFalse()
  self.back = false
  self.back_options = false
  self.triangle = false
end
function HUDSettings:Setup()
  self.handling_event = false
  self:SetAllHandlersFalse()
  self.settingsOn = self:GetState("SettingsOn")
  self.menu = menu.Menu.New(self, {
    FooterButtonInfo = {
      {
        Item = "Default",
        Text = "[TriangleButton] " .. util.GetLAMSMsg(lamsIDs.Default)
      },
      {
        Item = "Back",
        Text = "[BackButton] " .. util.GetLAMSMsg(lamsIDs.Back)
      }
    }
  })
  local HUDSettingsList = list.List.New(self, {
    MaxFocusableObjectCount = 20,
    ListObjectName = "MiddleSettingsList",
    EmptyTextLamsID = lamsIDs.NoSettingsAvailable,
    BackingAnimFrameCount = 60,
    HoverToSelect = true,
    Button_OnGainFocus = function(button)
      local currItem = button:get_item()
      self:SelectHUDSettings(currItem)
    end
  })
  self.menu:SetList("HUDSettings", HUDSettingsList)
  HUDSettingsList:SetSelectedButton(2, false)
  local HUDSettingsGenericToggleList = {
    {
      Value = 1,
      Text = util.GetLAMSMsg(lamsIDs.On)
    },
    {
      Value = 2,
      Text = util.GetLAMSMsg(lamsIDs.SoftTouch)
    },
    {
      Value = 0,
      Text = util.GetLAMSMsg(lamsIDs.Off)
    }
  }
  self.HUDtoggle = toggle.Toggle.New(self, {
    Name = "HUDtoggle",
    Get = function()
      return UI.GetUserHUDDisplaySetting()
    end,
    Set = function(v)
      UI.SetUserHUDDisplaySetting(v)
      local subValue = 1
      if v == hudConsts.DisplayConsts.Immersive then
        subValue = 0
      end
      self:RefreshHUDSettingsList()
      self.enemyHealthToggle:ForceSetValue(subValue)
      self.compassToggle:ForceSetValue(subValue)
      self.offScreenIndicatorToggle:ForceSetValue(subValue)
      self.combatHUDToggle:ForceSetValue(subValue)
      self.bossHealthToggle:ForceSetValue(subValue)
      self.gameNotificationsToggle:ForceSetValue(subValue == 1)
      self:SendEventUI("EVT_REFRESH_HUD")
    end,
    Settings = {
      {
        Value = hudConsts.DisplayConsts.Normal,
        Text = util.GetLAMSMsg(lamsIDs.NormalHUD),
        ToolTip = util.GetLAMSMsg(lamsIDs.NormalHUDDesc)
      },
      {
        Value = hudConsts.DisplayConsts.Immersive,
        Text = util.GetLAMSMsg(lamsIDs.ImmersiveHUD),
        ToolTip = util.GetLAMSMsg(lamsIDs.ImmersiveHUDDesc)
      },
      {
        Value = hudConsts.DisplayConsts.Custom,
        Text = util.GetLAMSMsg(lamsIDs.Custom),
        ToolTip = util.GetLAMSMsg(lamsIDs.CustomHUDDesc)
      }
    },
    Default = function()
      return 0
    end
  })
  self.enemyHealthToggle = toggle.Toggle.New(self, {
    Name = "EnemyHealthToggle",
    Get = function()
      if UI.GetUserEnemyHealthBarSetting ~= nil then
        return UI.GetUserEnemyHealthBarSetting()
      end
      return 1
    end,
    Set = function(value)
      if UI.SetUserEnemyHealthBarSetting ~= nil then
        UI.SetUserEnemyHealthBarSetting(value)
      end
      self:SendEventUI("EVT_REFRESH_HUD")
    end,
    Default = function()
      return 1
    end,
    Settings = HUDSettingsGenericToggleList
  })
  self.enemyHealthToggle.originalList = HUDSettingsGenericToggleList
  self.compassToggle = toggle.Toggle.New(self, {
    Name = "CompassToggle",
    Get = function()
      if UI.GetUserCompassSetting ~= nil then
        return UI.GetUserCompassSetting()
      end
      return 1
    end,
    Set = function(value)
      if UI.SetUserCompassSetting ~= nil then
        UI.SetUserCompassSetting(value)
      end
      self:SendEventUI("EVT_REFRESH_HUD")
    end,
    Default = function()
      return 1
    end,
    Settings = HUDSettingsGenericToggleList
  })
  self.compassToggle.originalList = HUDSettingsGenericToggleList
  self.offScreenIndicatorToggle = toggle.Toggle.New(self, {
    Name = "OffScreenIndicatorToggle",
    Get = function()
      if UI.GetUserOffscreenIndicatorSetting ~= nil then
        return UI.GetUserOffscreenIndicatorSetting()
      end
      return 1
    end,
    Set = function(value)
      if UI.SetUserOffscreenIndicatorSetting ~= nil then
        UI.SetUserOffscreenIndicatorSetting(value)
      end
      self:SendEventUI("EVT_REFRESH_HUD")
    end,
    Default = function()
      return 1
    end,
    Settings = HUDSettingsGenericToggleList
  })
  self.offScreenIndicatorToggle.originalList = HUDSettingsGenericToggleList
  self.combatHUDToggle = toggle.Toggle.New(self, {
    Name = "CombatHUDToggle",
    Get = function()
      if UI.GetUserCombatHUDSetting ~= nil then
        return UI.GetUserCombatHUDSetting()
      end
      return 1
    end,
    Set = function(value)
      if UI.SetUserCombatHUDSetting ~= nil then
        UI.SetUserCombatHUDSetting(value)
      end
      self:SendEventUI("EVT_REFRESH_HUD")
    end,
    Default = function()
      return 1
    end,
    Settings = HUDSettingsGenericToggleList
  })
  self.combatHUDToggle.originalList = HUDSettingsGenericToggleList
  self.bossHealthToggle = toggle.Toggle.New(self, {
    Name = "BossHealthToggle",
    Get = function()
      if UI.GetUserBossHealthBarSetting ~= nil then
        return UI.GetUserBossHealthBarSetting()
      end
      return 1
    end,
    Set = function(value)
      if UI.SetUserBossHealthBarSetting ~= nil then
        UI.SetUserBossHealthBarSetting(value)
      end
      self:SendEventUI("EVT_REFRESH_HUD")
    end,
    Default = function()
      return 1
    end,
    Settings = HUDSettingsGenericToggleList
  })
  self.bossHealthToggle.originalList = HUDSettingsGenericToggleList
  self.gameNotificationsToggleList = {
    {
      Value = true,
      Text = util.GetLAMSMsg(lamsIDs.On)
    },
    {
      Value = false,
      Text = util.GetLAMSMsg(lamsIDs.Off)
    }
  }
  self.gameNotificationsToggle = toggle.Toggle.New(self, {
    Name = "GameNotificationsToggle",
    Get = function()
      if UI.GetUserGameplayNotificationsSetting ~= nil then
        return UI.GetUserGameplayNotificationsSetting()
      end
      return true
    end,
    Set = function(value)
      if UI.SetUserGameplayNotificationsSetting ~= nil then
        UI.SetUserGameplayNotificationsSetting(value)
      end
      self:SendEventUI("EVT_REFRESH_HUD")
    end,
    Default = function()
      return true
    end,
    Settings = self.gameNotificationsToggleList
  })
  self.gameNotificationsToggle.originalList = self.gameNotificationsToggleList
  self.wideHUDToggle = toggle.Toggle.New(self, {
    Name = "WideHUDToggle",
    Get = UI.GetWideHUD,
    Set = UI.SetWideHUD,
    Default = function()
      return true
    end
  })
  self.hudSettingsInfo = {
    {
      Item = "WideHud",
      Text = util.GetLAMSMsg(lamsIDs.WideHUD),
      ActivationList = {
        {
          Item = self.wideHUDToggle
        }
      },
      ToolTip = util.GetLAMSMsg(lamsIDs.WideHUDTip)
    },
    {
      Item = "HUDList",
      Text = util.GetLAMSMsg(lamsIDs.HUDMode),
      ActivationList = {
        {
          Item = self.HUDtoggle
        }
      },
      ToggleTip = true
    },
    {
      Item = "CompassToggle",
      Text = util.GetLAMSMsg(lamsIDs.CompassSetting),
      ActivationList = {
        {
          Item = self.compassToggle
        }
      },
      ToolTip = util.GetLAMSMsg(lamsIDs.CompassSettingTip)
    },
    {
      Item = "EnemyHealthToggle",
      Text = util.GetLAMSMsg(lamsIDs.EnemyHealthBars),
      ActivationList = {
        {
          Item = self.enemyHealthToggle
        }
      },
      ToolTip = util.GetLAMSMsg(lamsIDs.EnemyHealthBarsTip)
    },
    {
      Item = "BossHealthToggle",
      Text = util.GetLAMSMsg(lamsIDs.BossHealthBars),
      ActivationList = {
        {
          Item = self.bossHealthToggle
        }
      },
      ToolTip = util.GetLAMSMsg(lamsIDs.BossHealthBarsTip)
    },
    {
      Item = "OffScreenIndicatorToggle",
      Text = util.GetLAMSMsg(lamsIDs.OffScreenSetting),
      ActivationList = {
        {
          Item = self.offScreenIndicatorToggle
        }
      },
      ToolTip = util.GetLAMSMsg(lamsIDs.OffScreenSettingTip)
    },
    {
      Item = "CombatHUDToggle",
      Text = util.GetLAMSMsg(lamsIDs.CombatHUDSetting),
      ActivationList = {
        {
          Item = self.combatHUDToggle
        }
      },
      ToolTip = util.GetLAMSMsg(lamsIDs.CombatHUDSettingTip)
    },
    {
      Item = "GameNotificationsToggle",
      Text = util.GetLAMSMsg(lamsIDs.GameplayNotificationsSetting),
      ActivationList = {
        {
          Item = self.gameNotificationsToggle
        }
      },
      ToolTip = util.GetLAMSMsg(lamsIDs.GameplayNotificationsSettingTip)
    }
  }
end
function HUDSettings:SetupToggleSettings(toggle)
  local currentIndex = UI.GetUserHUDDisplaySetting()
  local newList = toggle.originalList
  if currentIndex == hudConsts.DisplayConsts.Normal then
    newList = {
      newList[1]
    }
  elseif currentIndex == hudConsts.DisplayConsts.Immersive then
    local listItemIdx = toggle == self.gameNotificationsToggle and 2 or 3
    newList = {
      newList[listItemIdx]
    }
  end
  toggle:OverrideSettings(newList)
end
function HUDSettings:RefreshHUDSettingsList(disabled)
  if disabled == nil then
    disabled = UI.GetUserHUDDisplaySetting() ~= hudConsts.DisplayConsts.Custom
  end
  local HUDSettingsList = self.menu:GetList("HUDSettings")
  HUDSettingsList:Deactivate()
  self.settingsOn:RefreshSettingsList(HUDSettingsList, self.hudSettingsInfo, util.GetLAMSMsg(lamsIDs.HUD))
  for i = 3, #self.hudSettingsInfo do
    local setting_button = HUDSettingsList:GetButton(i)
    local toggle = self.hudSettingsInfo[i].ActivationList[1].Item
    if disabled then
      setting_button:SetTextColor(colors.Text_Settings_Disabled, "medium_text_settings1")
      toggle:DisableText()
    else
      setting_button:SetTextColor(colors.Text_Settings1, "medium_text_settings1")
      toggle:EnableText()
    end
    self:SetupToggleSettings(toggle)
  end
end
function HUDSettings:Enter()
  self.handling_event = false
  self:SetAllHandlersFalse()
  self.menu:Activate()
  self:RefreshHUDSettingsList()
  self:UpdateFooterButtonPrompt()
  Audio.PlaySound("SND_UX_Menu_Tick_Confirm")
end
function HUDSettings:Exit()
  self:RefreshHUDSettingsList(false)
  self.menu:Deactivate(true)
  Audio.PlaySound("SND_UX_Menu_Tick_Back")
end
function HUDSettings:SelectHUDSettings(currItem)
  self:UpdateFooterButtonPrompt()
end
function HUDSettings:SetDefaultValues()
  local HUDSettingsList = self.menu:GetList("HUDSettings")
  self.settingsOn:SetDefaultValues(HUDSettingsList, self.menu, function()
    self:Enter()
  end)
end
function HUDSettings:UpdateFooterButtonPrompt()
  self.menu:UpdateFooterButton("Default", true)
  self.menu:UpdateFooterButton("Back", true)
  self.menu:UpdateFooterButton("Close", true)
  self.menu:UpdateFooterButtonText()
end
function HUDSettings:EVT_MOUSE_CLICKED()
  if not self.handling_event then
    self.advance = true
  end
end
function HUDSettings:EVT_MOUSE_RIGHT_CLICKED()
  if not self.handling_event then
    self.back = true
  end
end
function HUDSettings:EVT_Back_Release()
  if not self.handling_event then
    self.back = true
  end
end
function HUDSettings:EVT_Options_Release()
  if not self.handling_event then
    self.back_options = true
  end
end
function HUDSettings:EVT_Triangle_Release()
  if not self.handling_event then
    self.triangle = true
  end
end
function HUDSettings:SetAllHandlersFalse()
  self.back = false
  self.back_options = false
  self.triangle = false
end
function HUDSettings:Update()
  if not self.handling_event then
    if self.back then
      self.handling_event = true
      self:SetAllHandlersFalse()
      self:Goto("MainSettings")
    elseif self.back_options then
      self.handling_event = true
      self:SetAllHandlersFalse()
      self.settingsOn:OptionsReleaseHandler()
    elseif self.triangle then
      self.handling_event = true
      self:SetAllHandlersFalse()
      self:SetDefaultValues()
    end
  end
end
function PhotoModeSettings:Setup()
  self.player = nil
  self.HasNoTouchpad = false
  self.handling_event = false
  self:SetAllHandlersFalse()
  self.settingsOn = self:GetState("SettingsOn")
  self.menu = menu.Menu.New(self, {
    FooterButtonInfo = {
      {
        Item = "Default",
        Text = "[TriangleButton] " .. util.GetLAMSMsg(lamsIDs.Default)
      },
      {
        Item = "Back",
        Text = "[BackButton] " .. util.GetLAMSMsg(lamsIDs.Back)
      }
    }
  })
  local PhotoModeSettingsList = list.List.New(self, {
    MaxFocusableObjectCount = 20,
    ListObjectName = "MiddleSettingsList",
    EmptyTextLamsID = lamsIDs.NoSettingsAvailable,
    BackingAnimFrameCount = 60,
    HoverToSelect = true,
    Button_OnGainFocus = function(button)
      local currItem = button:get_item()
      self:SelectPhotoModeSettings(currItem)
    end
  })
  self.menu:SetList("PhotoModeSettings", PhotoModeSettingsList)
  PhotoModeSettingsList:SetSelectedButton(2, false)
  self.photoModeInvertXAxisToggle = toggle.Toggle.New(self, {
    Name = "PhotoModeInvertX",
    Get = PhotoMode.GetInvertX,
    Set = function(num)
      PhotoMode.SetInvertX(num)
    end,
    Default = function()
      return false
    end
  })
  self.photoModeInvertYAxisToggle = toggle.Toggle.New(self, {
    Name = "PhotModeInvertY",
    Get = PhotoMode.GetInvertY,
    Set = function(num)
      PhotoMode.SetInvertY(num)
    end,
    Default = function()
      return false
    end
  })
  self.photoModeFromTouchpadToggle = toggle.Toggle.New(self, {
    Name = "photoModeFromTouchpadToggle",
    Get = function()
      if UI.GetPhotoModeFromTouchpad ~= nil then
        return UI.GetPhotoModeFromTouchpad()
      end
      return false
    end,
    Set = function(value)
      if UI.SetPhotoModeFromTouchpad ~= nil then
        UI.SetPhotoModeFromTouchpad(value)
      end
    end,
    Default = function()
      return false
    end
  })
  self.pmSettingsInfo = {
    {
      Item = "PhotoModeFromTouchpad",
      Text = util.GetLAMSMsg(lamsIDs.PhotoModeFromTouchpad),
      ActivationList = {
        {
          Item = self.photoModeFromTouchpadToggle
        }
      },
      ToolTip = util.GetLAMSMsg(lamsIDs.PhotoModeFromTouchpadTip)
    },
    {
      Item = "photoModeCameraHeader",
      Text = util.GetLAMSMsg(lamsIDs.CameraSettings),
      ButtonType = "category"
    },
    {
      Item = "photoModeInvertXAxisToggle",
      Text = util.GetLAMSMsg(lamsIDs.InvertXAxis),
      ActivationList = {
        {
          Item = self.photoModeInvertXAxisToggle
        }
      },
      ToolTip = util.GetLAMSMsg(lamsIDs.InvertXAxisToggleDesc)
    },
    {
      Item = "photoModeInvertYAxisToggle",
      Text = util.GetLAMSMsg(lamsIDs.InvertYAxis),
      ActivationList = {
        {
          Item = self.photoModeInvertYAxisToggle
        }
      },
      ToolTip = util.GetLAMSMsg(lamsIDs.InvertYAxisToggleDesc)
    }
  }
end
function PhotoModeSettings:Enter()
  self.player = game.Player.FindPlayer()
  self.HasNoTouchpad = false
  self.handling_event = false
  self:SetAllHandlersFalse()
  self.menu:Activate()
  local PhotoModeSettingsList = self.menu:GetList("PhotoModeSettings")
  self.settingsOn:RefreshSettingsList(PhotoModeSettingsList, self.pmSettingsInfo, util.GetLAMSMsg(lamsIDs.PhotoMode))
  self:UpdateFooterButtonPrompt()
  Audio.PlaySound("SND_UX_Menu_Tick_Confirm")
end
function PhotoModeSettings:Exit()
  local photoModeSettingsList = self.menu:GetList("PhotoModeSettings")
  photoModeSettingsList:Deactivate()
  self.settingsOn:RefreshSettingsList(photoModeSettingsList, self.pmSettingsInfo, util.GetLAMSMsg(lamsIDs.PhotoMode))
  self.photoModeFromTouchpadToggle:OverrideSettings(nil)
  self.menu:GetList("PhotoModeSettings"):GetButton(1):SetTextColor(colors.Text_Settings1, "medium_text_settings1")
  self.photoModeFromTouchpadToggle:GetButton(1):SetTextColor(colors.Text_Settings1, "medium_text_settings1")
  self.photoModeFromTouchpadToggle:GetButton(2):SetTextColor(colors.Text_Settings1, "medium_text_settings1")
  self.menu:Deactivate(true)
  Audio.PlaySound("SND_UX_Menu_Tick_Back")
end
function PhotoModeSettings:SelectPhotoModeSettings(currItem)
  self:UpdateFooterButtonPrompt()
end
function PhotoModeSettings:SetDefaultValues()
  local PhotoModeSettingsList = self.menu:GetList("PhotoModeSettings")
  self.settingsOn:SetDefaultValues(PhotoModeSettingsList, self.menu, function()
    self:Enter()
  end)
end
function PhotoModeSettings:UpdateFooterButtonPrompt()
  self.menu:UpdateFooterButton("Default", true)
  self.menu:UpdateFooterButton("Back", true)
  self.menu:UpdateFooterButton("Close", true)
  self.menu:UpdateFooterButtonText()
end
function PhotoModeSettings:EVT_MOUSE_CLICKED()
  if not self.handling_event then
    self.advance = true
  end
end
function PhotoModeSettings:EVT_MOUSE_RIGHT_CLICKED()
  if not self.handling_event then
    self.back = true
  end
end
function PhotoModeSettings:EVT_Back_Release()
  if not self.handling_event then
    self.back = true
  end
end
function PhotoModeSettings:EVT_Options_Release()
  if not self.handling_event then
    self.back_options = true
  end
end
function PhotoModeSettings:EVT_Triangle_Release()
  if not self.handling_event then
    self.triangle = true
  end
end
function PhotoModeSettings:SetAllHandlersFalse()
  self.back = false
  self.back_options = false
  self.triangle = false
end
function PhotoModeSettings:Update()
  local pad = self.player.Pad
  local HasNoTouchpad = pad.IsKeyboardAndMouse or pad.IsGeneric
  if self.HasNoTouchpad ~= HasNoTouchpad then
    local photoModeSettingsList = self.menu:GetList("PhotoModeSettings")
    photoModeSettingsList:Deactivate()
    if HasNoTouchpad then
      self.pmSettingsInfo[1].ToolTip = ""
    else
      self.pmSettingsInfo[1].ToolTip = util.GetLAMSMsg(lamsIDs.PhotoModeFromTouchpadTip)
    end
    self.settingsOn:RefreshSettingsList(photoModeSettingsList, self.pmSettingsInfo, util.GetLAMSMsg(lamsIDs.PhotoMode))
    if HasNoTouchpad then
      self.photoModeFromTouchpadToggle:OverrideSettings({
        {
          Text = util.GetLAMSMsg(lamsIDs.Off),
          Value = false
        }
      })
      self.menu:GetList("PhotoModeSettings"):GetButton(1):SetTextColor(colors.Text_Settings_Disabled, "medium_text_settings1")
      self.photoModeFromTouchpadToggle:GetButton(1):SetTextColor(colors.Text_Settings_Disabled, "medium_text_settings1")
    else
      self.photoModeFromTouchpadToggle:OverrideSettings(nil)
      self.menu:GetList("PhotoModeSettings"):GetButton(1):SetTextColor(colors.Text_Settings1, "medium_text_settings1")
      self.photoModeFromTouchpadToggle:GetButton(1):SetTextColor(colors.Text_Settings1, "medium_text_settings1")
      self.photoModeFromTouchpadToggle:GetButton(2):SetTextColor(colors.Text_Settings1, "medium_text_settings1")
    end
    self.HasNoTouchpad = HasNoTouchpad
  end
  if not self.handling_event then
    if self.back then
      self.handling_event = true
      self:SetAllHandlersFalse()
      self.back = false
      self:Goto("MainSettings")
    elseif self.back_options then
      self.handling_event = true
      self:SetAllHandlersFalse()
      self.settingsOn:OptionsReleaseHandler()
    elseif self.triangle then
      self.handling_event = true
      self:SetAllHandlersFalse()
      self:SetDefaultValues()
    end
  end
end
function AccessibilitySettings:Setup()
  self.handling_event = false
  self:SetAllHandlersFalse()
  if UI.SetMaxSubtitleScale ~= nil then
    UI.SetMaxSubtitleScale(1.3)
  end
  self.settingsOn = self:GetState("SettingsOn")
  self.menu = menu.Menu.New(self, {
    FooterButtonInfo = {
      {
        Item = "Default",
        Text = "[TriangleButton] " .. util.GetLAMSMsg(lamsIDs.Default)
      },
      {
        Item = "Back",
        Text = "[BackButton] " .. util.GetLAMSMsg(lamsIDs.Back)
      }
    }
  })
  local AccessibilitySettingsList = list.List.New(self, {
    MaxFocusableObjectCount = 20,
    ListObjectName = "MiddleSettingsList",
    EmptyTextLamsID = lamsIDs.NoSettingsAvailable,
    BackingAnimFrameCount = 60,
    HoverToSelect = true,
    Button_OnGainFocus = function(button)
      local currItem = button:get_item()
      self:SelectAccessibilitySettings(currItem)
    end
  })
  self.menu:SetList("AccessibilitySettings", AccessibilitySettingsList)
  AccessibilitySettingsList:SetSelectedButton(2, false)
  self.repeatedButtonToggle = toggle.Toggle.New(self, {
    Name = "repeatedButtonToggle",
    Get = function()
      return game.GetRepeatedButtonPressChoice()
    end,
    Set = function(val)
      game.SetRepeatedButtonPressChoice(val)
    end,
    Settings = {
      {
        Value = 0,
        Text = util.GetLAMSMsg(lamsIDs.Tap)
      },
      {
        Value = 1,
        Text = util.GetLAMSMsg(lamsIDs.Hold)
      }
    },
    Default = function()
      return 0
    end
  })
  self.ChiselDoorSkipToggle = toggle.Toggle.New(self, {
    Name = "ChiselDoorSkipToggle",
    Get = function()
      return game.GetMiniGameplaySkipped()
    end,
    Set = function(val)
      game.SetMiniGameplaySkipped(val)
    end,
    Settings = {
      {
        Value = false,
        Text = util.GetLAMSMsg(lamsIDs.Precision)
      },
      {
        Value = true,
        Text = util.GetLAMSMsg(lamsIDs.SingleButton)
      }
    },
    Default = function()
      return false
    end
  })
  self.ReticleAlwaysOnToggle = toggle.Toggle.New(self, {
    Name = "ReticleAlwaysOnToggle",
    Get = function()
      return game.GetReticleAlwaysOn()
    end,
    Set = function(val)
      game.SetReticleAlwaysOn(val)
    end,
    Settings = {
      {
        Value = false,
        Text = util.GetLAMSMsg(lamsIDs.WhenAiming)
      },
      {
        Value = true,
        Text = util.GetLAMSMsg(lamsIDs.AlwayOn)
      }
    },
    Default = function()
      return false
    end
  })
  self.sprintHoldToggle = toggle.Toggle.New(self, {
    Name = "SprintToggle",
    Get = function()
      return game.GetSprintOptionNumber and game.GetSprintOptionNumber() or 0
    end,
    Set = function(val)
      if game.SetSprintOptionNumber then
        game.SetSprintOptionNumber(val)
      end
    end,
    Settings = {
      {
        Value = 0,
        Text = util.GetLAMSMsg(lamsIDs.ClickL3)
      },
      {
        Value = 1,
        Text = util.GetLAMSMsg(lamsIDs.HoldEvade)
      }
    },
    Default = function()
      return 0
    end
  })
  self.autoSprintToggle = toggle.Toggle.New(self, {
    Name = "AutoSprint",
    Get = function()
      return game.GetAutoSprint()
    end,
    Set = function(num)
      game.SetAutoSprint(num)
    end,
    Default = function()
      return false
    end
  })
  self.autoSprintDelaySlider = slider.Slider.New(self, {
    Name = "AutoSprintDelay",
    TextFormat = "%0.2fs",
    Get = function()
      return game.GetAutoSprintDelay()
    end,
    Set = function(num)
      game.SetAutoSprintDelay(num)
    end,
    MinValue = 0,
    MaxValue = 2,
    Delta = 0.25
  })
  self.rageGrabHoldToggle = toggle.Toggle.New(self, {
    Name = "RageGrabToggle",
    Get = function()
      return game.GetRageGrabOptionNumber and game.GetRageGrabOptionNumber() or 0
    end,
    Set = function(val)
      if game.SetRageGrabOptionNumber then
        game.SetRageGrabOptionNumber(val)
      end
    end,
    Settings = {
      {
        Value = 0,
        Text = util.GetLAMSMsg(lamsIDs.ClickR3)
      },
      {
        Value = 1,
        Text = util.GetLAMSMsg(lamsIDs.HoldInteract)
      }
    },
    Default = function()
      return 0
    end
  })
  self.rageEnterToggle = toggle.Toggle.New(self, {
    Name = "rageEnterToggle",
    Get = function()
      return game.GetEnterRageModeOptionNumber and game.GetEnterRageModeOptionNumber() or 0
    end,
    Set = function(val)
      if game.SetEnterRageModeOptionNumber then
        game.SetEnterRageModeOptionNumber(val)
      end
      self:SendEventUI("EVT_RAGE_ACCESS_OPTIONS_CHANGED")
    end,
    Settings = {
      {
        Value = 0,
        Text = util.GetLAMSMsg(lamsIDs.PressL3R3)
      },
      {
        Value = 1,
        Text = util.GetLAMSMsg(lamsIDs.PressXCircle)
      }
    },
    Default = function()
      return 0
    end
  })
  self.subtitleBGAlphaToggle = toggle.Toggle.New(self, {
    Name = "SubtitleBGAlpha",
    Get = function()
      return game.UI.GetSubtitleBGAlpha and game.UI.GetSubtitleBGAlpha() or 0
    end,
    Set = function(val)
      game.UI.SetSubtitleBGAlpha(val)
    end,
    Settings = {
      {
        Value = 0,
        Text = util.GetLAMSMsg(lamsIDs.None)
      },
      {
        Value = 0.3,
        Text = util.GetLAMSMsg(lamsIDs.Light)
      },
      {
        Value = 0.6,
        Text = util.GetLAMSMsg(lamsIDs.Dark)
      }
    },
    Default = function()
      return 0
    end
  })
  self.subtitleSpeakerToggle = toggle.Toggle.New(self, {
    Name = "SubtitleSpeaker",
    Get = function()
      return game.UI.GetSubtitleSpeakerPolicy and game.UI.GetSubtitleSpeakerPolicy() or 0
    end,
    Set = function(val)
      game.UI.SetSubtitleSpeakerPolicy(val)
    end,
    Settings = {
      {
        Value = 0,
        Text = util.GetLAMSMsg(lamsIDs.Off)
      },
      {
        Value = 1,
        Text = util.GetLAMSMsg(lamsIDs.On)
      }
    },
    Default = function()
      return 1
    end
  })
  self.swayStrengthSlider = slider.Slider.New(self, {
    Name = "SwayStrength",
    TextFormat = "%d",
    Get = UI.GetUserSwayStrength,
    Set = function(num)
      UI.SetUserSwayStrength(num)
    end,
    MinValue = UI.GetUserSwayStrengthMin(),
    MaxValue = UI.GetUserSwayStrengthMax(),
    Delta = UI.GetUserSwayStrengthDelta,
    Default = function()
      return 10
    end
  })
  self.shakeStrengthSlider = slider.Slider.New(self, {
    Name = "ShakeStrength",
    TextFormat = "%d",
    Get = UI.GetUserShakeStrength,
    Set = function(num)
      UI.SetUserShakeStrength(num)
    end,
    MinValue = UI.GetUserShakeStrengthMin(),
    MaxValue = UI.GetUserShakeStrengthMax(),
    Delta = UI.GetUserShakeStrengthDelta,
    Default = function()
      return 10
    end
  })
  self.uiAccessibilityScalingSlider = slider.Slider.New(self, {
    Name = "UIAccessibilityScaling",
    TextFormat = "%0.1f",
    Get = function()
      return 10 * (UI.GetAccessibilityScaling and UI.GetAccessibilityScaling() or 0)
    end,
    Set = function(num)
      if UI.SetAccessibilityScaling ~= nil then
        local scale = num / 10
        UI.SetAccessibilityScaling(scale)
        if UI.SetSubtitleScale ~= nil then
          UI.SetSubtitleScale(scale)
        end
      end
    end,
    MinValue = 0,
    MaxValue = 10,
    Delta = 0.1,
    Default = function()
      return 0
    end
  })
  self.accessibilitySettingsInfo = {
    {
      Item = "RepeatedButtonToggle",
      Text = util.GetLAMSMsg(lamsIDs.RepeatButton),
      ActivationList = {
        {
          Item = self.repeatedButtonToggle
        }
      },
      ToolTip = util.GetLAMSMsg(lamsIDs.RepeatButtonTip)
    },
    {
      Item = "ChiselDoorSkipToggle",
      Text = util.GetLAMSMsg(lamsIDs.ChiselDoorSkip),
      ActivationList = {
        {
          Item = self.ChiselDoorSkipToggle
        }
      },
      ToolTip = util.GetLAMSMsg(lamsIDs.ChiselDoorSkipTip)
    },
    {
      Item = "ReticleAlwaysOnToggle",
      Text = util.GetLAMSMsg(lamsIDs.Reticle),
      ActivationList = {
        {
          Item = self.ReticleAlwaysOnToggle
        }
      },
      ToolTip = util.GetLAMSMsg(lamsIDs.ReticleTip)
    },
    {
      Item = "sprintHoldToggle",
      Text = util.GetLAMSMsg(lamsIDs.Sprint),
      ActivationList = {
        {
          Item = self.sprintHoldToggle
        }
      },
      ToolTip = util.GetLAMSMsg(lamsIDs.SprintAccessibilityTip)
    },
    {
      Item = "autoSprintToggle",
      Text = util.GetLAMSMsg(lamsIDs.AutoSprint),
      ActivationList = {
        {
          Item = self.autoSprintToggle
        }
      },
      ToolTip = util.GetLAMSMsg(lamsIDs.AutoSprintTip)
    },
    {
      Item = "autoSprintDelaySlider",
      Text = util.GetLAMSMsg(lamsIDs.AutoSprintDelay),
      ActivationList = {
        {
          Item = self.autoSprintDelaySlider
        }
      },
      ToolTip = util.GetLAMSMsg(lamsIDs.AutoSprintDelayTip)
    },
    {
      Item = "rageGrabHoldToggle",
      Text = util.GetLAMSMsg(lamsIDs.RageGrabAccessibility),
      ActivationList = {
        {
          Item = self.rageGrabHoldToggle
        }
      },
      ToolTip = util.GetLAMSMsg(lamsIDs.RageGrabAccessibilityTip)
    },
    {
      Item = "rageEnterToggle",
      Text = util.GetLAMSMsg(lamsIDs.RageModeAccess),
      ActivationList = {
        {
          Item = self.rageEnterToggle
        }
      },
      ToolTip = util.GetLAMSMsg(lamsIDs.RageModeAccessTip)
    },
    {
      Item = "SubtitleBGAlpha",
      Text = util.GetLAMSMsg(lamsIDs.SubtitleBGAlpha),
      ActivationList = {
        {
          Item = self.subtitleBGAlphaToggle
        }
      },
      ToolTip = util.GetLAMSMsg(lamsIDs.SubtitleBGAlphaTip)
    },
    {
      Item = "SubtitleSpeaker",
      Text = util.GetLAMSMsg(lamsIDs.SubtitleSpeaker),
      ActivationList = {
        {
          Item = self.subtitleSpeakerToggle
        }
      },
      ToolTip = util.GetLAMSMsg(lamsIDs.SubtitleSpeakerTip)
    },
    {
      Item = "swayStrengthSlider",
      Text = util.GetLAMSMsg(lamsIDs.AmbientCameraSway),
      ActivationList = {
        {
          Item = self.swayStrengthSlider
        }
      },
      ToolTip = util.GetLAMSMsg(lamsIDs.AmbientCameraSwayDesc)
    },
    {
      Item = "shakeStrengthSlider",
      Text = util.GetLAMSMsg(lamsIDs.ShakeCameraStrength),
      ActivationList = {
        {
          Item = self.shakeStrengthSlider
        }
      },
      ToolTip = util.GetLAMSMsg(lamsIDs.ShakeCameraStrengthTip)
    },
    {
      Item = "uiAccessibilityScalingSlider",
      Text = util.GetLAMSMsg(lamsIDs.UIAccessibilityScaling),
      ActivationList = {
        {
          Item = self.uiAccessibilityScalingSlider
        }
      },
      ToolTip = util.GetLAMSMsg(lamsIDs.UIAccessibilityScalingDesc)
    }
  }
end
function AccessibilitySettings:Enter()
  self.handling_event = false
  self:SetAllHandlersFalse()
  self.menu:Activate()
  local AccessibilitySettingsList = self.menu:GetList("AccessibilitySettings")
  self.settingsOn:RefreshSettingsList(AccessibilitySettingsList, self.accessibilitySettingsInfo, util.GetLAMSMsg(lamsIDs.Accessibility))
  self:UpdateFooterButtonPrompt()
  Audio.PlaySound("SND_UX_Menu_Tick_Confirm")
end
function AccessibilitySettings:Exit()
  self.menu:Deactivate(true)
  Audio.PlaySound("SND_UX_Menu_Tick_Back")
end
function AccessibilitySettings:SelectAccessibilitySettings(currItem)
  self:UpdateFooterButtonPrompt()
end
function AccessibilitySettings:SetDefaultValues()
  local AccessibilitySettingsList = self.menu:GetList("AccessibilitySettings")
  self.settingsOn:SetDefaultValues(AccessibilitySettingsList, self.menu, function()
    self:Enter()
  end)
end
function AccessibilitySettings:UpdateFooterButtonPrompt()
  self.menu:UpdateFooterButton("Default", true)
  self.menu:UpdateFooterButton("Back", true)
  self.menu:UpdateFooterButton("Close", true)
  self.menu:UpdateFooterButtonText()
end
function AccessibilitySettings:EVT_MOUSE_CLICKED()
  if not self.handling_event then
    self.advance = true
  end
end
function AccessibilitySettings:EVT_MOUSE_RIGHT_CLICKED()
  if not self.handling_event then
    self.back = true
  end
end
function AccessibilitySettings:EVT_Back_Release()
  if not self.handling_event then
    self.back = true
  end
end
function AccessibilitySettings:EVT_Options_Release()
  if not self.handling_event then
    self.back_options = true
  end
end
function AccessibilitySettings:EVT_Triangle_Release()
  if not self.handling_event then
    self.triangle = true
  end
end
function AccessibilitySettings:SetAllHandlersFalse()
  self.back = false
  self.back_options = false
  self.triangle = false
end
function AccessibilitySettings:Update()
  if not self.handling_event then
    if self.back then
      self.handling_event = true
      self:SetAllHandlersFalse()
      self.back = false
      self:Goto("MainSettings")
    elseif self.back_options then
      self.handling_event = true
      self:SetAllHandlersFalse()
      self.settingsOn:OptionsReleaseHandler()
    elseif self.triangle then
      self.handling_event = true
      self:SetAllHandlersFalse()
      self:SetDefaultValues()
    end
  end
end
function ViewControls:GetControllerGOs()
  self.goController = self.goSettingsController:FindSingleGOByName("Controller")
  if self.IsProsperoPad then
    self.goGenericButtons = self.goSettingsController:FindSingleGOByName("GenericButtonsDS5")
    self.goGenericLines = self.goSettingsController:FindSingleGOByName("GenericLinesDS5")
  else
    self.goGenericButtons = self.goSettingsController:FindSingleGOByName("GenericButtons")
    self.goGenericLines = self.goSettingsController:FindSingleGOByName("GenericLines")
  end
  self.goDpadButtons = self.goSettingsController:FindSingleGOByName("DpadButtons")
  self.goTriggerButtons = self.goSettingsController:FindSingleGOByName("TriggerButtons")
  self.goFaceButtons = self.goSettingsController:FindSingleGOByName("FaceButtons")
  self.goJoyLButton = self.goGenericButtons:FindSingleGOByName("JoyLButton")
  self.goJoyRButton = self.goGenericButtons:FindSingleGOByName("JoyRButton")
  self.goQuickTurnTriggerButton = self.goSettingsController:FindSingleGOByName("QuickTurnTriggerButton")
  self.goQuickTurnTrigger = self.goSettingsController:FindSingleGOByName("QuickTurnTrigger")
  self.th_l1Button = util.GetTextHandle(self.goTriggerButtons, "L1")
  self.th_l2Button = util.GetTextHandle(self.goTriggerButtons, "L2")
  self.th_r1Button = util.GetTextHandle(self.goTriggerButtons, "R1")
  self.th_r2Button = util.GetTextHandle(self.goTriggerButtons, "R2")
  self.th_quickTurnTrigger = util.GetTextHandle(self.goQuickTurnTriggerButton)
  self.th_DPadButtons = util.GetTextHandle(self.goDpadButtons)
  self.th_faceButtons = util.GetTextHandle(self.goFaceButtons)
  self.th_joyLButtons = util.GetTextHandle(self.goJoyLButton)
  self.th_joyRButtons = util.GetTextHandle(self.goJoyRButton)
  self.goFaceButtonLines = self.goSettingsController:FindSingleGOByName("FaceButtonLines")
  self.goDpadButtonLines = self.goSettingsController:FindSingleGOByName("DpadButtonLines")
  self.goTriggerLines = self.goSettingsController:FindSingleGOByName("TriggerLines")
  self.goMapOnlyLines = self.goSettingsController:FindSingleGOByName("MapLine")
  self.goMapAndPhotoLines = self.goSettingsController:FindSingleGOByName("PhotoModeLine")
  self.goMapOnlyText = self.goGenericButtons:FindSingleGOByName("MapText")
  self.goMapAndPhotoText = self.goGenericButtons:FindSingleGOByName("PhotoModeText")
  self.th_MapAndPhoto = util.GetTextHandle(self.goMapAndPhotoText)
end
function ViewControls:Setup()
  self.player = nil
  self.IsKeyboardAndMouse = false
  self.IsProsperoPad = false
  self.backPressed = false
  self.handling_event = false
  self:SetAllHandlersFalse()
  self.controller_highmips_request = false
  self.goSettingsController = util.GetUiObjByName("SettingsController")
  self:GetControllerGOs()
  self.controllerStreamedIn = false
  self.controllerType = ""
  self.settingsOn = self:GetState("SettingsOn")
  self.goMenuFooterToolTip = util.GetUiObjByName("MenuFooterToolTip")
  self.thTooltip = util.GetTextHandle(self.goMenuFooterToolTip)
  self.menu = menu.Menu.New(self, {
    FooterButtonInfo = {
      {
        Item = "Change",
        Text = "[AdvanceButton] " .. util.GetLAMSMsg(lamsIDs.ChangeControls)
      },
      {
        Item = "Back",
        Text = "[BackButton] " .. util.GetLAMSMsg(lamsIDs.Back)
      },
      {Item = "Close", Text = ""}
    }
  })
end
function ViewControls:Enter()
  UI.SetText(self.thTooltip, "")
  self.player = game.Player.FindPlayer()
  self.IsKeyboardAndMouse = false
  self.IsProsperoPad = false
  self.backPressed = false
  self.goSettingsController = util.GetUiObjByName("SettingsController")
  self:GetControllerGOs()
  self.handling_event = false
  self:SetAllHandlersFalse()
  self.goController:SetMaterialSwap("")
  self.controllerType = ""
  self.controller_highmips_request = false
  self.controllerStreamedIn = false
  self.menu:Activate()
  Audio.PlaySound("SND_UX_Menu_Tick_Confirm")
end
function ViewControls:Exit()
  self.controllerStreamedIn = true
  self.menu:Deactivate(true)
  self:HideGOs()
  Audio.PlaySound("SND_UX_Menu_Tick_Back")
end
function ViewControls:EVT_MOUSE_CLICKED()
  if not self.handling_event and not self.changing then
    self.advance = true
  end
end
function ViewControls:EVT_MOUSE_RIGHT_CLICKED()
  if not self.handling_event and not self.changing then
    self.back = true
  end
end
function ViewControls:EVT_Back_Press()
  self.backPressed = true
end
function ViewControls:EVT_Back_Release()
  if not self.handling_event and not self.changing and self.backPressed then
    self.back = true
  end
  self.backPressed = false
end
function ViewControls:EVT_Options_Release()
  if not self.handling_event and not self.changing then
    self.back_options = true
  end
end
function ViewControls:EVT_Triangle_Release()
  if not self.handling_event and not self.changing then
    self.triangle = true
  end
end
function ViewControls:EVT_Advance_Release()
  if not self.handling_event then
    self.advance = true
  end
end
function ViewControls:SetAllHandlersFalse()
  self.back = false
  self.back_options = false
  self.advance = false
  self.triangle = false
end
function ViewControls:Update()
  local controllerType = game.GetControllerType()
  local IsProsperoPad = self.player.Pad.IsProsperoPad
  local IsKeyboardAndMouse = self.player.Pad.IsKeyboardAndMouse
  if self.IsKeyboardAndMouse ~= IsKeyboardAndMouse then
    self:HideGOs()
    self.IsProsperoPad = IsProsperoPad
    if IsKeyboardAndMouse then
      self:ShowKeyboard()
      self.controllerStreamedIn = true
    else
      if controllerType == "xbox" then
        self.goSettingsController = util.GetUiObjByName("SettingsControllerXBox")
      else
        self.goSettingsController = util.GetUiObjByName("SettingsController")
      end
      self:GetControllerGOs()
      self.controllerStreamedIn = false
    end
    self.IsKeyboardAndMouse = IsKeyboardAndMouse
  end
  if not IsKeyboardAndMouse and self.controllerType ~= controllerType then
    self:HideGOs()
    if controllerType == "xbox" then
      self.goSettingsController = util.GetUiObjByName("SettingsControllerXBox")
    else
      self.goSettingsController = util.GetUiObjByName("SettingsController")
    end
    self.IsProsperoPad = IsProsperoPad
    self:GetControllerGOs()
    self.goController:SetMaterialSwap(controllerType)
    self.controllerType = controllerType
    self.controllerStreamedIn = false
  end
  if not self.controllerStreamedIn then
    self.goController:RequestLoadHighMips(true)
    if not self.controller_highmips_request then
      self.controller_highmips_request = true
    elseif self.goController:CheckMaterialFullyStreamed() then
      self.controllerStreamedIn = true
      self:GetControllerGOs()
      self:ShowControllers()
      self.IsProsperoPad = IsProsperoPad
    end
  elseif self.IsProsperoPad ~= IsProsperoPad then
    self:HideGOs()
    self.IsProsperoPad = IsProsperoPad
    self:GetControllerGOs()
    self:ShowControllers()
  end
  if not self.handling_event then
    if self.back then
      self.handling_event = true
      self:SetAllHandlersFalse()
      self.back = false
      self:Goto("MainSettings")
    elseif self.back_options then
      self.handling_event = true
      self:SetAllHandlersFalse()
      self.settingsOn:OptionsReleaseHandler()
    elseif self.advance then
      self:SetAllHandlersFalse()
      if IsKeyboardAndMouse then
        self.handling_event = true
        self:GetState("ChangeControls").returnToMain = false
        self:Goto("ChangeControls")
      end
    end
  end
end
function ViewControls:GetCombatControllerConfig()
  return game.GetCurrentControllerLayoutForConfig(tweaks.eControllerConfigTypes.kCC_Combat)
end
function ViewControls:ShowKeys(category, info)
  local goIcon = self.goSettingsController:FindSingleGOByName(category .. "IconText")
  goIcon:Show()
  local goAction = self.goSettingsController:FindSingleGOByName(category .. "ActionText")
  goAction:Show()
  local iconsText = ""
  local actionsText = ""
  for i = 1, #info do
    if info[i] ~= nil then
      iconsText = iconsText .. info[i].icon .. "\n"
      actionsText = actionsText .. info[i].text .. "\n"
    end
  end
  UI.SetText(util.GetTextHandle(goIcon), iconsText)
  UI.SetText(util.GetTextHandle(goAction), actionsText)
end
function ViewControls:GetSprintKey()
  if game.GetSprintOptionNumber() == 0 then
    return {
      icon = "[L3]",
      text = util.GetLAMSMsg(lamsIDs.Sprint)
    }
  else
    return {
      icon = "[EvadeButton]",
      text = util.GetLAMSMsg(lamsIDs.HoldToSprint)
    }
  end
end
function ViewControls:GetQuickTurnKey()
  local quickTurn = self:GetQuickTurnControllerConfig()
  if quickTurn == 2 then
    return "[PrimaryMoveBackwardKey]+[L3]"
  elseif quickTurn == 1 then
    return "[R1Button]+[R2Button]"
  else
    return "[DownButton]"
  end
end
function ViewControls:GetWeaponKey1()
  if resourceUtil.HasResource(pickupConsts.Blades) then
    return {
      icon = "[RightButton]",
      text = util.GetLAMSMsg(lamsIDs.RecipesAxe)
    }
  else
    return {
      icon = "[RightButton]",
      text = util.GetLAMSMsg(lamsIDs.SheatheUnsheathe)
    }
  end
end
function ViewControls:GetWeaponKey2()
  if resourceUtil.HasResource(pickupConsts.Blades) then
    return {
      icon = "[LeftButton]",
      text = util.GetLAMSMsg(lamsIDs.RecipesBlades)
    }
  else
    return nil
  end
end
function ViewControls:GetArrowTypeKey()
  if resourceUtil.HasResource("ArrowUnlock_Shock") then
    return {
      icon = "[UpButton]",
      text = util.GetLAMSMsg(lamsIDs.ArrowType)
    }
  else
    return nil
  end
end
function ViewControls:GetHudKey()
  if UI.GetUserHUDDisplaySetting() == hudConsts.DisplayConsts.Custom then
    return {
      icon = "[PrimaryHudKey]",
      text = util.GetLAMSMsg(lamsIDs.HUD)
    }
  else
    return nil
  end
end
function ViewControls:GetStunGrabKey()
  if game.GetRageGrabOptionNumber() == 0 then
    return util.GetLAMSMsg(lamsIDs.ClickR3)
  else
    return util.GetLAMSMsg(lamsIDs.HoldInteract)
  end
end
function ViewControls:GetRageKey()
  if game.GetEnterRageModeOptionNumber() == 0 then
    return "[L3] + [R3]"
  else
    return "[XButton] + [CircleButton]"
  end
end
function ViewControls:ShowKeyboard()
  self.menu:UpdateFooterButton("Change", true)
  self.menu:UpdateFooterButton("Back", true)
  self.menu:UpdateFooterButton("Close", true)
  self.menu:UpdateFooterButtonText()
  self.goSettingsController = util.GetUiObjByName("SettingsKeyboard")
  self.goSettingsController:Show()
  self:ShowKeys("Game", {
    {
      icon = "[JoystickR]",
      text = util.GetLAMSMsg(lamsIDs.Camera)
    },
    {
      icon = "[JoystickL]",
      text = util.GetLAMSMsg(lamsIDs.Move)
    },
    self:GetSprintKey(),
    {
      icon = self:GetQuickTurnKey(),
      text = util.GetLAMSMsg(lamsIDs.QuickTurnConfig)
    },
    {
      icon = "[InteractButton]",
      text = util.GetLAMSMsg(lamsIDs.Interact)
    }
  })
  self:ShowKeys("Combat", {
    self:GetWeaponKey1(),
    self:GetWeaponKey2(),
    {
      icon = "[LightAttackButton]",
      text = util.GetLAMSMsg(lamsIDs.LightAttack)
    },
    {
      icon = "[HeavyAttackButton]",
      text = util.GetLAMSMsg(lamsIDs.HeavyAttack)
    },
    {
      icon = "[AimButton]",
      text = util.GetLAMSMsg(lamsIDs.Aim)
    },
    {
      icon = "[LockOnButton]",
      text = util.GetLAMSMsg(lamsIDs.LockOnController)
    },
    {
      icon = "[AxeRecallButton]",
      text = util.GetLAMSMsg(lamsIDs.AxeRecall)
    },
    {
      icon = "[SonActionButton]",
      text = util.GetLAMSMsg(lamsIDs.SonAction)
    },
    self:GetArrowTypeKey(),
    {
      icon = "[BlockButton]",
      text = util.GetLAMSMsg(lamsIDs.BlockParry)
    },
    {
      icon = "[EvadeButton]",
      text = util.GetLAMSMsg(lamsIDs.Evade)
    },
    self:GetHudKey(),
    {
      icon = self:GetStunGrabKey(),
      text = util.GetLAMSMsg(lamsIDs.RageGrabAccessibility)
    },
    {
      icon = self:GetRageKey(),
      text = util.GetLAMSMsg(lamsIDs.Slot_SpartanRage)
    }
  })
  self:ShowKeys("Menu", {
    {
      icon = "[ESCkey]",
      text = util.GetLAMSMsg(lamsIDs.Options)
    },
    {
      icon = "[PrimaryPauseKey]",
      text = util.GetLAMSMsg(lamsIDs.Weapon)
    },
    {
      icon = "[PrimaryMapKey]",
      text = util.GetLAMSMsg(lamsIDs.Map)
    },
    {
      icon = "[PrimarySkillsKey]",
      text = util.GetLAMSMsg(lamsIDs.SkillTree)
    },
    {
      icon = "[PrimaryQuestsKey]",
      text = util.GetLAMSMsg(lamsIDs.Quests)
    },
    {
      icon = "[PrimaryPhotomodeKey]",
      text = util.GetLAMSMsg(lamsIDs.PhotoMode)
    }
  })
end
function ViewControls:ShowControllers()
  self:UpdateFooterButtonPrompt()
  self:ShowAndPlaceGO()
  self:UpdateAllDynamicValues()
end
function ViewControls:UpdateFooterButtonPrompt()
  self.menu:UpdateFooterButton("Change", false)
  self.menu:UpdateFooterButton("Back", true)
  self.menu:UpdateFooterButton("Close", true)
  self.menu:UpdateFooterButtonText()
end
function ViewControls:HideGOs()
  self.goSettingsController:Hide()
  self.goController:Hide()
  self.goGenericButtons:Hide()
  self.goDpadButtons:Hide()
  self.goJoyLButton:Hide()
  self.goJoyRButton:Hide()
  self.goTriggerButtons:Hide()
  self.goFaceButtons:Hide()
  self.goGenericLines:Hide()
  self.goFaceButtonLines:Hide()
  self.goDpadButtonLines:Hide()
  self.goTriggerLines:Hide()
  self.goQuickTurnTriggerButton:Hide()
  self.goQuickTurnTrigger:Hide()
  self.goMapOnlyText:Hide()
  self.goMapOnlyLines:Hide()
  self.goMapAndPhotoText:Hide()
  self.goMapAndPhotoLines:Hide()
end
function ViewControls:ShowAndPlaceGO()
  self.goSettingsController:Show()
  self.goController:Show()
  self.goGenericButtons:Show()
  self.goDpadButtons:Show()
  self.goJoyLButton:Show()
  self.goJoyRButton:Show()
  self.goTriggerButtons:Show()
  self.goFaceButtons:Show()
  UI.Anim(self.goController, consts.AS_Forward, "", 0, 1)
  UI.Anim(self.goTriggerButtons, consts.AS_Forward, "", 0, 1)
  UI.Anim(self.goFaceButtons, consts.AS_Forward, "", 0, 1)
  UI.Anim(self.goDpadButtons, consts.AS_Forward, "", 0, 1)
  UI.Anim(self.goJoyLButton, consts.AS_Forward, "", 0, 1)
  UI.AnimNoShow(self.goQuickTurnTriggerButton, consts.AS_Forward, "", 0, 1)
  self.goGenericLines:Show()
  self.goFaceButtonLines:Show()
  self.goTriggerLines:Show()
  self.goDpadButtonLines:Show()
end
function ViewControls:UpdateAllDynamicValues()
  local quickturn = self:GetQuickTurnControllerConfig()
  local sprint = game.GetSprintOptionNumber()
  local lockOn = UI.GetUserLockOn()
  controlsUtil.UpdateButton({
    button = "L1Button",
    th = self.th_l1Button,
    left_align = false
  })
  controlsUtil.UpdateButton({
    button = "L2Button",
    th = self.th_l2Button,
    left_align = false
  })
  controlsUtil.UpdateButton({
    button = "R1Button",
    th = self.th_r1Button,
    left_align = true
  })
  controlsUtil.UpdateButton({
    button = "R2Button",
    th = self.th_r2Button,
    left_align = true
  })
  local combatConfig = self:GetCombatControllerConfig()
  local trigger = ""
  if quickturn == 1 and combatConfig ~= 2 then
    trigger = "[R1Button]+[R2Button] " .. util.GetLAMSMsg(lamsIDs.QuickTurnConfig)
    self.goQuickTurnTriggerButton:Show()
    self.goQuickTurnTrigger:Show()
  end
  UI.SetText(self.th_quickTurnTrigger, trigger)
  controlsUtil.UpdateAllFaceButtons(self.th_faceButtons, combatConfig == 2 and quickturn == 1, sprint == 1)
  controlsUtil.UpdateJoyLButtons(self.th_joyLButtons, quickturn == 2, sprint == 0)
  controlsUtil.UpdateJoyRButtons(self.th_joyRButtons, lockOn)
  controlsUtil.UpdateDpadButtons(self.th_DPadButtons, quickturn == 0)
  if self.player.Pad.IsGeneric then
    self.goMapOnlyText:Hide()
    self.goMapOnlyLines:Hide()
    self.goMapAndPhotoText:Hide()
    self.goMapAndPhotoLines:Hide()
  elseif self.player.Pad.IsKeyboardAndMouse then
    UI.SetText(self.th_MapAndPhoto, "[PrimaryPhotomodeKey] " .. util.GetLAMSMsg(lamsIDs.PhotoMode))
    self.goMapOnlyText:Hide()
    self.goMapOnlyLines:Hide()
    self.goMapAndPhotoText:Show()
    self.goMapAndPhotoLines:Show()
  elseif game.UI.GetPhotoModeFromTouchpad and game.UI.GetPhotoModeFromTouchpad() then
    UI.SetText(self.th_MapAndPhoto, util.GetLAMSMsg(lamsIDs.PhotoMode) .. " [TouchPad]")
    self.goMapOnlyText:Hide()
    self.goMapOnlyLines:Hide()
    self.goMapAndPhotoText:Show()
    self.goMapAndPhotoLines:Show()
  else
    self.goMapOnlyText:Show()
    self.goMapOnlyLines:Show()
    self.goMapAndPhotoText:Hide()
    self.goMapAndPhotoLines:Hide()
  end
end
function ViewControls:GetQuickTurnControllerConfig()
  if game.GetQuickTurnOptionNumber then
    return game.GetQuickTurnOptionNumber()
  end
  return 0
end
function ChangeControls:Setup()
  self.player = nil
  self.handling_event = false
  self:SetAllHandlersFalse()
  self.changing = false
  self.inDialog = false
  self.previousControl = nil
  self.settingsOn = self:GetState("SettingsOn")
  self.menu = menu.Menu.New(self, {
    FooterButtonInfo = {
      {
        Item = "Default1",
        Text = "[TriangleButton] " .. util.GetLAMSMsg(lamsIDs.Default)
      },
      {
        Item = "Select",
        Text = "[AdvanceButton] " .. util.GetLAMSMsg(lamsIDs.Select)
      },
      {
        Item = "Back",
        Text = "[BackButton] " .. util.GetLAMSMsg(lamsIDs.Back)
      }
    }
  })
  local ChangeControlsList = list.List.New(self, {
    MaxFocusableObjectCount = 20,
    ListObjectName = "MiddleSettingsList",
    EmptyTextLamsID = lamsIDs.NoSettingsAvailable,
    BackingAnimFrameCount = 60,
    HoverToSelect = true,
    Button_OnGainFocus = function(button)
      local control = button._activationList[1].Item
      if self.previousControl ~= nil then
        control:SetSelectedButton(self.previousControl._focusedObjectIndex, false)
      end
      self.previousControl = control
      local currItem = button:get_item()
      self:SelectChangeControls(currItem)
    end
  })
  self.menu:SetList("ChangeControls", ChangeControlsList)
  ChangeControlsList:SetSelectedButton(2, false)
  self.lookUpControl = control.Control.New(self, {
    Name = "LookUp",
    Get = function()
      return 0
    end,
    Set = function(settingValue)
    end,
    Settings = {
      {
        Value = 0,
        Text = "[PrimaryLookUpKey]"
      },
      {
        Value = 1,
        Text = "[SecondaryLookUpKey]"
      }
    },
    Default = function()
      return 0
    end
  })
  self.lookDownControl = control.Control.New(self, {
    Name = "LookDown",
    Get = function()
      return 0
    end,
    Set = function(settingValue)
    end,
    Settings = {
      {
        Value = 0,
        Text = "[PrimaryLookDownKey]"
      },
      {
        Value = 1,
        Text = "[SecondaryLookDownKey]"
      }
    },
    Default = function()
      return 0
    end
  })
  self.lookLeftControl = control.Control.New(self, {
    Name = "LookLeft",
    Get = function()
      return 0
    end,
    Set = function(settingValue)
    end,
    Settings = {
      {
        Value = 0,
        Text = "[PrimaryLookLeftKey]"
      },
      {
        Value = 1,
        Text = "[SecondaryLookLeftKey]"
      }
    },
    Default = function()
      return 0
    end
  })
  self.lookRightControl = control.Control.New(self, {
    Name = "LookRight",
    Get = function()
      return 0
    end,
    Set = function(settingValue)
    end,
    Settings = {
      {
        Value = 0,
        Text = "[PrimaryLookRightKey]"
      },
      {
        Value = 1,
        Text = "[SecondaryLookRightKey]"
      }
    },
    Default = function()
      return 0
    end
  })
  self.moveForwardControl = control.Control.New(self, {
    Name = "MoveForward",
    Get = function()
      return 0
    end,
    Set = function(settingValue)
    end,
    Settings = {
      {
        Value = 0,
        Text = "[PrimaryMoveForwardKey]"
      },
      {
        Value = 1,
        Text = "[SecondaryMoveForwardKey]"
      }
    },
    Default = function()
      return 0
    end
  })
  self.moveBackwardControl = control.Control.New(self, {
    Name = "MoveBackward",
    Get = function()
      return 0
    end,
    Set = function(settingValue)
    end,
    Settings = {
      {
        Value = 0,
        Text = "[PrimaryMoveBackwardKey]"
      },
      {
        Value = 1,
        Text = "[SecondaryMoveBackwardKey]"
      }
    },
    Default = function()
      return 0
    end
  })
  self.moveLeftControl = control.Control.New(self, {
    Name = "MoveLeft",
    Get = function()
      return 0
    end,
    Set = function(settingValue)
    end,
    Settings = {
      {
        Value = 0,
        Text = "[PrimaryMoveLeftKey]"
      },
      {
        Value = 1,
        Text = "[SecondaryMoveLeftKey]"
      }
    },
    Default = function()
      return 0
    end
  })
  self.moveRightControl = control.Control.New(self, {
    Name = "MoveRight",
    Get = function()
      return 0
    end,
    Set = function(settingValue)
    end,
    Settings = {
      {
        Value = 0,
        Text = "[PrimaryMoveRightKey]"
      },
      {
        Value = 1,
        Text = "[SecondaryMoveRightKey]"
      }
    },
    Default = function()
      return 0
    end
  })
  self.quickTurnControl = control.Control.New(self, {
    Name = "QuickTurn",
    Get = function()
      return 0
    end,
    Set = function(settingValue)
    end,
    Settings = {
      {
        Value = 0,
        Text = "[PrimaryQuickTurnKey]"
      },
      {
        Value = 1,
        Text = "[SecondaryQuickTurnKey]"
      }
    },
    Default = function()
      return 0
    end
  })
  self.sprintControl = control.Control.New(self, {
    Name = "Sprint",
    Get = function()
      return 0
    end,
    Set = function(settingValue)
    end,
    Settings = {
      {
        Value = 0,
        Text = "[PrimarySprintKey]"
      },
      {
        Value = 1,
        Text = "[SecondarySprintKey]"
      }
    },
    Default = function()
      return 0
    end
  })
  self.aimControl = control.Control.New(self, {
    Name = "Aim",
    Get = function()
      return 0
    end,
    Set = function(settingValue)
    end,
    Settings = {
      {
        Value = 0,
        Text = "[PrimaryAimKey]"
      },
      {
        Value = 1,
        Text = "[SecondaryAimKey]"
      }
    },
    Default = function()
      return 0
    end
  })
  self.lockOnControl = control.Control.New(self, {
    Name = "LockOn",
    Get = function()
      return 0
    end,
    Set = function(settingValue)
    end,
    Settings = {
      {
        Value = 0,
        Text = "[PrimaryLockOnKey]"
      },
      {
        Value = 1,
        Text = "[SecondaryLockOnKey]"
      }
    },
    Default = function()
      return 0
    end
  })
  self.heavyAttackControl = control.Control.New(self, {
    Name = "HeavyAttack",
    Get = function()
      return 0
    end,
    Set = function(settingValue)
    end,
    Settings = {
      {
        Value = 0,
        Text = "[PrimaryHeavyAttackKey]"
      },
      {
        Value = 1,
        Text = "[SecondaryHeavyAttackKey]"
      }
    },
    Default = function()
      return 0
    end
  })
  self.lightAttackControl = control.Control.New(self, {
    Name = "LightAttack",
    Get = function()
      return 0
    end,
    Set = function(settingValue)
    end,
    Settings = {
      {
        Value = 0,
        Text = "[PrimaryLightAttackKey]"
      },
      {
        Value = 1,
        Text = "[SecondaryLightAttackKey]"
      }
    },
    Default = function()
      return 0
    end
  })
  self.blockParryControl = control.Control.New(self, {
    Name = "BlockParry",
    Get = function()
      return 0
    end,
    Set = function(settingValue)
    end,
    Settings = {
      {
        Value = 0,
        Text = "[PrimaryBlockKey]"
      },
      {
        Value = 1,
        Text = "[SecondaryBlockKey]"
      }
    },
    Default = function()
      return 0
    end
  })
  self.sheatheControl = control.Control.New(self, {
    Name = "Sheathe",
    Get = function()
      return 0
    end,
    Set = function(settingValue)
    end,
    Settings = {
      {
        Value = 0,
        Text = "[PrimarySheatheKey]"
      },
      {
        Value = 1,
        Text = "[SecondarySheatheKey]"
      }
    },
    Default = function()
      return 0
    end
  })
  self.bladesControl = control.Control.New(self, {
    Name = "Blades",
    Get = function()
      return 0
    end,
    Set = function(settingValue)
    end,
    Settings = {
      {
        Value = 0,
        Text = "[PrimaryBladesKey]"
      },
      {
        Value = 1,
        Text = "[SecondaryBladesKey]"
      }
    },
    Default = function()
      return 0
    end
  })
  self.axeRecallControl = control.Control.New(self, {
    Name = "AxeRecall",
    Get = function()
      return 0
    end,
    Set = function(settingValue)
    end,
    Settings = {
      {
        Value = 0,
        Text = "[PrimaryAxeRecallKey]"
      },
      {
        Value = 1,
        Text = "[SecondaryAxeRecallKey]"
      }
    },
    Default = function()
      return 0
    end
  })
  self.sonActionControl = control.Control.New(self, {
    Name = "SonAction",
    Get = function()
      return 0
    end,
    Set = function(settingValue)
    end,
    Settings = {
      {
        Value = 0,
        Text = "[PrimarySonActionKey]"
      },
      {
        Value = 1,
        Text = "[SecondarySonActionKey]"
      }
    },
    Default = function()
      return 0
    end
  })
  self.arrowTypeControl = control.Control.New(self, {
    Name = "ArrowType",
    Get = function()
      return 0
    end,
    Set = function(settingValue)
    end,
    Settings = {
      {
        Value = 0,
        Text = "[PrimaryArrowTypeKey]"
      },
      {
        Value = 1,
        Text = "[SecondaryArrowTypeKey]"
      }
    },
    Default = function()
      return 0
    end
  })
  self.lightRunicControl = control.Control.New(self, {
    Name = "LightRunic",
    Get = function()
      return 0
    end,
    Set = function(settingValue)
    end,
    Settings = {
      {
        Value = 0,
        Text = "[PrimaryLightRunicKey]"
      },
      {
        Value = 1,
        Text = "[SecondaryLightRunicKey]"
      }
    },
    Default = function()
      return 0
    end
  })
  self.heavyRunicControl = control.Control.New(self, {
    Name = "HeavyRunic",
    Get = function()
      return 0
    end,
    Set = function(settingValue)
    end,
    Settings = {
      {
        Value = 0,
        Text = "[PrimaryHeavyRunicKey]"
      },
      {
        Value = 1,
        Text = "[SecondaryHeavyRunicKey]"
      }
    },
    Default = function()
      return 0
    end
  })
  self.rageControl = control.Control.New(self, {
    Name = "Rage",
    Get = function()
      return 0
    end,
    Set = function(settingValue)
    end,
    Settings = {
      {
        Value = 0,
        Text = "[PrimaryRageKey]"
      },
      {
        Value = 1,
        Text = "[SecondaryRageKey]"
      }
    },
    Default = function()
      return 0
    end
  })
  self.interactControl = control.Control.New(self, {
    Name = "Interact",
    Get = function()
      return 0
    end,
    Set = function(settingValue)
    end,
    Settings = {
      {
        Value = 0,
        Text = "[PrimaryInteractKey]"
      },
      {
        Value = 1,
        Text = "[SecondaryInteractKey]"
      }
    },
    Default = function()
      return 0
    end
  })
  self.evadeControl = control.Control.New(self, {
    Name = "Evade",
    Get = function()
      return 0
    end,
    Set = function(settingValue)
    end,
    Settings = {
      {
        Value = 0,
        Text = "[PrimaryEvadeKey]"
      },
      {
        Value = 1,
        Text = "[SecondaryEvadeKey]"
      }
    },
    Default = function()
      return 0
    end
  })
  self.weaponsControl = control.Control.New(self, {
    Name = "Pause",
    Get = function()
      return 0
    end,
    Set = function(settingValue)
    end,
    Settings = {
      {
        Value = 0,
        Text = "[PrimaryPauseKey]"
      },
      {
        Value = 1,
        Text = "[SecondaryPauseKey]"
      }
    },
    Default = function()
      return 0
    end
  })
  self.mapControl = control.Control.New(self, {
    Name = "Map",
    Get = function()
      return 0
    end,
    Set = function(settingValue)
    end,
    Settings = {
      {
        Value = 0,
        Text = "[PrimaryMapKey]"
      },
      {
        Value = 1,
        Text = "[SecondaryMapKey]"
      }
    },
    Default = function()
      return 0
    end
  })
  self.skillsControl = control.Control.New(self, {
    Name = "Skills",
    Get = function()
      return 0
    end,
    Set = function(settingValue)
    end,
    Settings = {
      {
        Value = 0,
        Text = "[PrimarySkillsKey]"
      },
      {
        Value = 1,
        Text = "[SecondarySkillsKey]"
      }
    },
    Default = function()
      return 0
    end
  })
  self.questsControl = control.Control.New(self, {
    Name = "Quests",
    Get = function()
      return 0
    end,
    Set = function(settingValue)
    end,
    Settings = {
      {
        Value = 0,
        Text = "[PrimaryQuestsKey]"
      },
      {
        Value = 1,
        Text = "[SecondaryQuestsKey]"
      }
    },
    Default = function()
      return 0
    end
  })
  self.photomodeControl = control.Control.New(self, {
    Name = "Photomode",
    Get = function()
      return 0
    end,
    Set = function(settingValue)
    end,
    Settings = {
      {
        Value = 0,
        Text = "[PrimaryPhotomodeKey]"
      },
      {
        Value = 1,
        Text = "[SecondaryPhotomodeKey]"
      }
    },
    Default = function()
      return 0
    end
  })
  self.HudControl = control.Control.New(self, {
    Name = "Hud",
    Get = function()
      return 0
    end,
    Set = function(settingValue)
    end,
    Settings = {
      {
        Value = 0,
        Text = "[PrimaryHudKey]"
      },
      {
        Value = 1,
        Text = "[SecondaryHudKey]"
      }
    },
    Default = function()
      return 0
    end
  })
end
function ChangeControls:GetControlsItems()
  local allControlsItems = {
    {
      Item = "CombatHeader",
      Text = util.GetLAMSMsg(lamsIDs.CombatConfig),
      ButtonType = "category"
    },
    {
      Item = "Aim",
      Text = util.GetLAMSMsg(lamsIDs.Aim),
      ActivationList = {
        {
          Item = self.aimControl
        }
      },
      ToolTip = util.GetLAMSMsg(lamsIDs.CombatConfigTip)
    },
    {
      Item = "LockOn",
      Text = util.GetLAMSMsg(lamsIDs.LockOnController),
      ActivationList = {
        {
          Item = self.lockOnControl
        }
      },
      ToolTip = util.GetLAMSMsg(lamsIDs.CombatConfigTip)
    },
    {
      Item = "HeavyAttack",
      Text = util.GetLAMSMsg(lamsIDs.HeavyAttack),
      ActivationList = {
        {
          Item = self.heavyAttackControl
        }
      },
      ToolTip = util.GetLAMSMsg(lamsIDs.CombatConfigTip)
    },
    {
      Item = "LightAttack",
      Text = util.GetLAMSMsg(lamsIDs.LightAttack),
      ActivationList = {
        {
          Item = self.lightAttackControl
        }
      },
      ToolTip = util.GetLAMSMsg(lamsIDs.CombatConfigTip)
    },
    {
      Item = "Block",
      Text = util.GetLAMSMsg(lamsIDs.BlockParry),
      ActivationList = {
        {
          Item = self.blockParryControl
        }
      },
      ToolTip = util.GetLAMSMsg(lamsIDs.CombatConfigTip)
    },
    {
      Item = "Sheathe",
      Text = util.GetLAMSMsg(lamsIDs.SheatheUnsheathe),
      ActivationList = {
        {
          Item = self.sheatheControl
        }
      },
      ToolTip = util.GetLAMSMsg(lamsIDs.CombatConfigTip)
    },
    {
      Item = "Axe",
      Text = util.GetLAMSMsg(lamsIDs.RecipesAxe),
      ActivationList = {
        {
          Item = self.sheatheControl
        }
      },
      ToolTip = util.GetLAMSMsg(lamsIDs.CombatConfigTip)
    },
    {
      Item = "Blades",
      Text = util.GetLAMSMsg(lamsIDs.RecipesBlades),
      ActivationList = {
        {
          Item = self.bladesControl
        }
      },
      ToolTip = util.GetLAMSMsg(lamsIDs.CombatConfigTip)
    },
    {
      Item = "AxeRecall",
      Text = util.GetLAMSMsg(lamsIDs.AxeRecall),
      ActivationList = {
        {
          Item = self.axeRecallControl
        }
      },
      ToolTip = util.GetLAMSMsg(lamsIDs.CombatConfigTip)
    },
    {
      Item = "SonAction",
      Text = util.GetLAMSMsg(lamsIDs.SonAction),
      ActivationList = {
        {
          Item = self.sonActionControl
        }
      },
      ToolTip = util.GetLAMSMsg(lamsIDs.CombatConfigTip)
    },
    {
      Item = "ArrowType",
      Text = util.GetLAMSMsg(lamsIDs.ArrowType),
      ActivationList = {
        {
          Item = self.arrowTypeControl
        }
      },
      ToolTip = util.GetLAMSMsg(lamsIDs.CombatConfigTip)
    },
    {
      Item = "Hud",
      Text = util.GetLAMSMsg(lamsIDs.HUD),
      ActivationList = {
        {
          Item = self.HudControl
        }
      },
      ToolTip = util.GetLAMSMsg(lamsIDs.CombatConfigTip)
    },
    {
      Item = "LightRunic",
      Text = util.GetLAMSMsg(lamsIDs.Slot_WeaponSpecial_Light),
      ActivationList = {
        {
          Item = self.lightRunicControl
        }
      },
      ToolTip = util.GetLAMSMsg(lamsIDs.CombatConfigTip)
    },
    {
      Item = "HeavyRunic",
      Text = util.GetLAMSMsg(lamsIDs.Slot_WeaponSpecial_Heavy),
      ActivationList = {
        {
          Item = self.heavyRunicControl
        }
      },
      ToolTip = util.GetLAMSMsg(lamsIDs.CombatConfigTip)
    },
    {
      Item = "Rage",
      Text = util.GetLAMSMsg(lamsIDs.Slot_SpartanRage),
      ActivationList = {
        {
          Item = self.rageControl
        }
      },
      ToolTip = util.GetLAMSMsg(lamsIDs.CombatConfigTip)
    },
    {
      Item = "InteractEvadeHeader",
      Text = util.GetLAMSMsg(lamsIDs.InteractEvadeConfig),
      ButtonType = "category"
    },
    {
      Item = "Interact",
      Text = util.GetLAMSMsg(lamsIDs.Interact),
      ActivationList = {
        {
          Item = self.interactControl
        }
      },
      ToolTip = util.GetLAMSMsg(lamsIDs.InteractEvadeConfigTip)
    },
    {
      Item = "Evade",
      Text = util.GetLAMSMsg(lamsIDs.Evade),
      ActivationList = {
        {
          Item = self.evadeControl
        }
      },
      ToolTip = util.GetLAMSMsg(lamsIDs.InteractEvadeConfigTip)
    },
    {
      Item = "MoveHeader",
      Text = util.GetLAMSMsg(lamsIDs.Move),
      ButtonType = "category"
    },
    {
      Item = "MoveForward",
      Text = util.GetLAMSMsg(lamsIDs.MoveForward),
      ActivationList = {
        {
          Item = self.moveForwardControl
        }
      },
      ToolTip = util.GetLAMSMsg(lamsIDs.MoveConfigTip)
    },
    {
      Item = "MoveBackward",
      Text = util.GetLAMSMsg(lamsIDs.MoveBackward),
      ActivationList = {
        {
          Item = self.moveBackwardControl
        }
      },
      ToolTip = util.GetLAMSMsg(lamsIDs.MoveConfigTip)
    },
    {
      Item = "MoveLeft",
      Text = util.GetLAMSMsg(lamsIDs.MoveLeft),
      ActivationList = {
        {
          Item = self.moveLeftControl
        }
      },
      ToolTip = util.GetLAMSMsg(lamsIDs.MoveConfigTip)
    },
    {
      Item = "MoveRight",
      Text = util.GetLAMSMsg(lamsIDs.MoveRight),
      ActivationList = {
        {
          Item = self.moveRightControl
        }
      },
      ToolTip = util.GetLAMSMsg(lamsIDs.MoveConfigTip)
    },
    {
      Item = "LookUp",
      Text = util.GetLAMSMsg(lamsIDs.LookUp),
      ActivationList = {
        {
          Item = self.lookUpControl
        }
      },
      ToolTip = util.GetLAMSMsg(lamsIDs.CameraConfigTip)
    },
    {
      Item = "LookDown",
      Text = util.GetLAMSMsg(lamsIDs.LookDown),
      ActivationList = {
        {
          Item = self.lookDownControl
        }
      },
      ToolTip = util.GetLAMSMsg(lamsIDs.CameraConfigTip)
    },
    {
      Item = "LookLeft",
      Text = util.GetLAMSMsg(lamsIDs.LookLeft),
      ActivationList = {
        {
          Item = self.lookLeftControl
        }
      },
      ToolTip = util.GetLAMSMsg(lamsIDs.CameraConfigTip)
    },
    {
      Item = "LookRight",
      Text = util.GetLAMSMsg(lamsIDs.LookRight),
      ActivationList = {
        {
          Item = self.lookRightControl
        }
      },
      ToolTip = util.GetLAMSMsg(lamsIDs.CameraConfigTip)
    },
    {
      Item = "Sprint",
      Text = util.GetLAMSMsg(lamsIDs.Sprint),
      ActivationList = {
        {
          Item = self.sprintControl
        }
      },
      ToolTip = util.GetLAMSMsg(lamsIDs.SprintAccessibilityTip)
    },
    {
      Item = "QuickTurn",
      Text = util.GetLAMSMsg(lamsIDs.QuickTurnConfig),
      ActivationList = {
        {
          Item = self.quickTurnControl
        }
      },
      ToolTip = util.GetLAMSMsg(lamsIDs.QuickTurnConfigTip)
    },
    {
      Item = "MenuHeader",
      Text = util.GetLAMSMsg(lamsIDs.PauseMenu),
      ButtonType = "category"
    },
    {
      Item = "Pause",
      Text = util.GetLAMSMsg(lamsIDs.Weapon),
      ActivationList = {
        {
          Item = self.weaponsControl
        }
      },
      ToolTip = util.GetLAMSMsg(lamsIDs.MenuConfigTip)
    },
    {
      Item = "Map",
      Text = util.GetLAMSMsg(lamsIDs.Map),
      ActivationList = {
        {
          Item = self.mapControl
        }
      },
      ToolTip = util.GetLAMSMsg(lamsIDs.MenuConfigTip)
    },
    {
      Item = "Skills",
      Text = util.GetLAMSMsg(lamsIDs.SkillTree),
      ActivationList = {
        {
          Item = self.skillsControl
        }
      },
      ToolTip = util.GetLAMSMsg(lamsIDs.MenuConfigTip)
    },
    {
      Item = "Quests",
      Text = util.GetLAMSMsg(lamsIDs.Quests),
      ActivationList = {
        {
          Item = self.questsControl
        }
      },
      ToolTip = util.GetLAMSMsg(lamsIDs.MenuConfigTip)
    },
    {
      Item = "Photomode",
      Text = util.GetLAMSMsg(lamsIDs.PhotoMode),
      ActivationList = {
        {
          Item = self.photomodeControl
        }
      },
      ToolTip = util.GetLAMSMsg(lamsIDs.MenuConfigTip)
    }
  }
  local availableControlsListItems = {}
  for _, item in ipairs(allControlsItems) do
    if (item.Item ~= "Sheathe" or not resourceUtil.HasResource(pickupConsts.Blades)) and (item.Item ~= "Axe" or resourceUtil.HasResource(pickupConsts.Blades)) and (item.Item ~= "Blades" or resourceUtil.HasResource(pickupConsts.Blades)) and (item.Item ~= "ArrowType" or resourceUtil.HasResource("ArrowUnlock_Shock")) then
      tablex.FastInsert(availableControlsListItems, item, #availableControlsListItems + 1)
    end
  end
  return availableControlsListItems
end
function ChangeControls:Enter()
  game.SetOnlyShowPrimaryKey(true)
  self.player = game.Player.FindPlayer()
  self.handling_event = false
  self.inDialog = false
  self:SetAllHandlersFalse()
  self.menu:Activate()
  local ChangeControlsList = self.menu:GetList("ChangeControls")
  self.settingsOn:RefreshSettingsList(ChangeControlsList, self:GetControlsItems(), util.GetLAMSMsg(lamsIDs.ChangeControls))
  self:UpdateFooterButtonPrompt()
  Audio.PlaySound("SND_UX_Menu_Tick_Confirm")
end
function ChangeControls:Exit()
  self.menu:Deactivate(true)
  game.SetOnlyShowPrimaryKey(false)
  Audio.PlaySound("SND_UX_Menu_Tick_Back")
  self:SendEventUI("EVT_RAGE_ACCESS_OPTIONS_CHANGED")
end
function ChangeControls:SelectChangeControls(currItem)
  self:UpdateFooterButtonPrompt()
end
function ChangeControls:SetDefaultValues(mode)
  local ChangeControlsList = self.menu:GetList("ChangeControls")
  self:SendEventUI("EVT_SHOW_SETTINGS_DIALOG_MESSAGE", {
    Header = util.GetLAMSMsg(lamsIDs.ResetToDefault),
    Body = util.GetLAMSMsg(lamsIDs.AreYouSure_SettingsReset),
    ActiveMenus = {
      self.menu
    },
    ActiveLists = {ChangeControlsList},
    Events = {
      Advance = {
        Text = util.GetLAMSMsg(lamsIDs.Confirm),
        Handler = function()
          game.SetDefaultKeyboardMapping(mode)
          self:Enter()
        end
      },
      Back = {
        Text = util.GetLAMSMsg(lamsIDs.Back),
        Handler = function()
          self:Enter()
        end
      }
    }
  })
end
function ChangeControls:UpdateFooterButtonPrompt()
  self.menu:UpdateFooterButton("Default1", true)
  self.menu:UpdateFooterButton("Default2", true)
  self.menu:UpdateFooterButton("Select", true)
  self.menu:UpdateFooterButton("Back", true)
  self.menu:UpdateFooterButton("Close", true)
  self.menu:UpdateFooterButtonText()
end
function ChangeControls:EVT_MOUSE_CLICKED()
  local selected = UI.GetEventSenderGameObject()
  local list = self.menu:GetList("ChangeControls")
  if selected == list._scrollBarMeter:get_gameObject() then
    return
  end
  if not self.handling_event and not self.changing then
    self.pressed = true
  end
end
function ChangeControls:EVT_MOUSE_RELEASED()
  if not self.handling_event and not self.changing and self.pressed then
    self.advance = true
    self.pressed = false
  end
end
function ChangeControls:EVT_MOUSE_RIGHT_CLICKED()
  if not self.handling_event and not self.changing then
    self.back = true
  end
end
function ChangeControls:EVT_Back_Release()
  if not self.handling_event then
    self.back = true
  end
end
function ChangeControls:EVT_Options_Release()
  if not self.handling_event and not self.changing then
    self.back_options = true
  end
end
function ChangeControls:EVT_Triangle_Release()
  if not self.handling_event then
    self.triangle = true
  end
end
function ChangeControls:EVT_Advance_Press()
  if not self.handling_event and not self.changing then
    self.pressed = true
  end
end
function ChangeControls:EVT_Advance_Release()
  if not self.handling_event and not self.changing and self.pressed then
    self.advance = true
    self.pressed = false
  end
end
function ChangeControls:HandleAdvanceOnList()
  local ChangeControlsList = self.menu:GetList("ChangeControls")
  if not self.changing and not self.inDialog then
    local button = ChangeControlsList:GetSelectedButton()
    local toggle = button._activationList[1].Item
    local selected = toggle:GetSelectedButton()
    if selected._item.Text == "[PrimaryLightRunicKey]" then
      self.handling_event = false
      return
    end
    if selected._item.Text == "[PrimaryHeavyRunicKey]" then
      self.handling_event = false
      return
    end
    if selected._item.Text == "[PrimaryRageKey]" then
      self.handling_event = false
      return
    end
    ChangeControlsList:Deactivate()
    self.settingsOn:RefreshSettingsList(ChangeControlsList, self:GetControlsItems(), util.GetLAMSMsg(lamsIDs.ChangeControls))
    local newList
    if selected._item.Value == 0 then
      newList = {
        {
          Value = 0,
          Text = util.GetLAMSMsg(45155)
        },
        {
          Value = 1,
          Text = "[Secondary" .. button._item.Item .. "Key]"
        }
      }
      toggle:OverrideSettings(newList)
      game.AcquireNewControl("Primary" .. button._item.Item .. "Key")
    else
      newList = {
        {
          Value = 0,
          Text = "[Primary" .. button._item.Item .. "Key]"
        },
        {
          Value = 1,
          Text = util.GetLAMSMsg(45155)
        }
      }
      toggle:OverrideSettings(newList)
      game.AcquireNewControl("Secondary" .. button._item.Item .. "Key")
    end
    self.menu:Deactivate(true, false)
    self.changing = true
    self.handling_event = false
  end
end
function ChangeControls:ConfirmControls()
  local ChangeControlsList = self.menu:GetList("ChangeControls")
  self:SendEventUI("EVT_SHOW_SETTINGS_DIALOG_MESSAGE", {
    Header = util.GetLAMSMsg(lamsIDs.ChangeControls),
    Body = util.GetLAMSMsg(lamsIDs.KeysNotMapped),
    ActiveMenus = {
      self.menu
    },
    ActiveLists = {ChangeControlsList},
    Events = {
      Advance = {
        Text = util.GetLAMSMsg(lamsIDs.Confirm),
        Handler = function()
          self:Enter()
        end
      },
      Back = {
        Text = util.GetLAMSMsg(lamsIDs.Revert),
        Handler = function()
          game.RevertKeyboardMapping()
          self:Enter()
          self.back = true
        end
      }
    }
  })
end
function ChangeControls:CheckFixedMenuControls()
  local ChangeControlsList = self.menu:GetList("ChangeControls")
  self:SendEventUI("EVT_SHOW_SETTINGS_DIALOG_MESSAGE", {
    Header = util.GetLAMSMsg(lamsIDs.ChangeControls),
    Body = util.GetLAMSMsg(lamsIDs.KeyFixedMenu),
    ActiveMenus = {
      self.menu
    },
    ActiveLists = {ChangeControlsList},
    Events = {
      Advance = {
        Text = util.GetLAMSMsg(lamsIDs.Confirm),
        Handler = function()
          self:Enter()
        end
      }
    }
  })
end
function ChangeControls:HandleBackOnList()
  local ChangeControlsList = self.menu:GetList("ChangeControls")
  if self.changing then
    self.menu:Activate()
    self.settingsOn:RefreshSettingsList(ChangeControlsList, self:GetControlsItems(), util.GetLAMSMsg(lamsIDs.ChangeControls))
    local button = ChangeControlsList:GetSelectedButton()
    local toggle = button._activationList[1].Item
    local newList = {
      {
        Value = 0,
        Text = "[Primary" .. button._item.Item .. "Key]"
      },
      {
        Value = 1,
        Text = "[Secondary" .. button._item.Item .. "Key]"
      }
    }
    toggle:OverrideSettings(newList)
    self.changing = false
    self.handling_event = false
  elseif game.AreSomeFunctionsUnmapped() then
    self:ConfirmControls()
  elseif self.returnToMain then
    if not self.inDialog then
      self:Goto("MainSettings")
    end
  elseif not self.inDialog then
    self:Goto("ViewControls")
  end
end
function ChangeControls:SetAllHandlersFalse()
  self.back = false
  self.back_options = false
  self.advance = false
  self.triangle = false
  self.square = false
  self.pressed = false
end
function ChangeControls:Update()
  if not self.player.Pad.IsKeyboardAndMouse then
    game.RevertKeyboardMapping()
    self.back = true
  end
  if not self.handling_event then
    if self.back then
      self.handling_event = true
      self:SetAllHandlersFalse()
      self:HandleBackOnList()
    elseif self.back_options and not self.inDialog then
      self.handling_event = true
      self:SetAllHandlersFalse()
      self.settingsOn:OptionsReleaseHandler()
    elseif self.triangle then
      self.handling_event = true
      self:SetAllHandlersFalse()
      if self.changing then
        self.inDialog = true
        self:CheckFixedMenuControls()
        self:HandleBackOnList()
      else
        self:SetDefaultValues(1)
      end
    elseif self.square then
      self.handling_event = true
      self:SetAllHandlersFalse()
      self:SetDefaultValues(2)
    elseif self.advance then
      self.handling_event = true
      self:SetAllHandlersFalse()
      self:HandleAdvanceOnList()
    end
  end
end
function CameraSettings:Setup()
  self.handling_event = false
  self:SetAllHandlersFalse()
  self.settingsOn = self:GetState("SettingsOn")
  self.menu = menu.Menu.New(self, {
    FooterButtonInfo = {
      {
        Item = "Default",
        Text = "[TriangleButton] " .. util.GetLAMSMsg(lamsIDs.Default)
      },
      {
        Item = "Back",
        Text = "[BackButton] " .. util.GetLAMSMsg(lamsIDs.Back)
      }
    }
  })
  local CameraSettingsList = list.List.New(self, {
    MaxFocusableObjectCount = 20,
    ListObjectName = "MiddleSettingsList",
    EmptyTextLamsID = lamsIDs.NoSettingsAvailable,
    BackingAnimFrameCount = 60,
    HoverToSelect = true,
    Button_OnGainFocus = function(button)
      local currItem = button:get_item()
      self:SelectCameraSetting(currItem)
    end
  })
  self.menu:SetList("CameraSettings", CameraSettingsList)
  CameraSettingsList:SetSelectedButton(2, false)
  self.horizontalSensitivitySlider = slider.Slider.New(self, {
    Name = "HorizontalSensitivity",
    TextFormat = "%d",
    Get = UI.GetUserCameraSpeedHorizontalCamera,
    Set = function(num)
      UI.SetUserCameraSpeedHorizontalCamera(num)
    end,
    MinValue = UI.GetUserCameraSpeedMin(),
    MaxValue = UI.GetUserCameraSpeedMax(),
    Delta = UI.GetUserCameraSpeedDelta,
    Default = function()
      return 5
    end
  })
  self.verticalSensitivitySlider = slider.Slider.New(self, {
    Name = "VerticalSensitivity",
    TextFormat = "%d",
    Get = UI.GetUserCameraSpeedVerticalCamera,
    Set = function(num)
      UI.SetUserCameraSpeedVerticalCamera(num)
    end,
    MinValue = UI.GetUserCameraSpeedMin(),
    MaxValue = UI.GetUserCameraSpeedMax(),
    Delta = UI.GetUserCameraSpeedDelta,
    Default = function()
      return 5
    end
  })
  self.horizontalMouseSensitivitySlider = slider.Slider.New(self, {
    Name = "HorizontalMouseSensitivity",
    TextFormat = "%d",
    Get = UI.GetUserCameraMouseSpeedHorizontalCamera,
    Set = function(num)
      UI.SetUserCameraMouseSpeedHorizontalCamera(num)
    end,
    MinValue = UI.GetUserCameraSpeedMin(),
    MaxValue = UI.GetUserCameraSpeedMax(),
    Delta = UI.GetUserCameraSpeedDelta,
    Default = function()
      return 5
    end
  })
  self.verticalMouseSensitivitySlider = slider.Slider.New(self, {
    Name = "VerticalMouseSensitivity",
    TextFormat = "%d",
    Get = UI.GetUserCameraMouseSpeedVerticalCamera,
    Set = function(num)
      UI.SetUserCameraMouseSpeedVerticalCamera(num)
    end,
    MinValue = UI.GetUserCameraSpeedMin(),
    MaxValue = UI.GetUserCameraSpeedMax(),
    Delta = UI.GetUserCameraSpeedDelta,
    Default = function()
      return 5
    end
  })
  self.InvertXAxisToggle = toggle.Toggle.New(self, {
    Name = "InvertX",
    Get = UI.GetUserCameraRotationInvert,
    Set = function(num)
      UI.SetUserCameraRotationInvert(num)
    end,
    Default = function()
      return false
    end
  })
  self.invertYAxisToggle = toggle.Toggle.New(self, {
    Name = "InvertY",
    Get = UI.GetUserCameraElevationInvert,
    Set = function(num)
      UI.SetUserCameraElevationInvert(num)
    end,
    Default = function()
      return false
    end
  })
  self.swayStrengthSlider = slider.Slider.New(self, {
    Name = "SwayStrength",
    TextFormat = "%d",
    Get = UI.GetUserSwayStrength,
    Set = function(num)
      UI.SetUserSwayStrength(num)
    end,
    MinValue = UI.GetUserSwayStrengthMin(),
    MaxValue = UI.GetUserSwayStrengthMax(),
    Delta = UI.GetUserSwayStrengthDelta,
    Default = function()
      return 10
    end
  })
  self.shakeStrengthSlider = slider.Slider.New(self, {
    Name = "ShakeStrength",
    TextFormat = "%d",
    Get = UI.GetUserShakeStrength,
    Set = function(num)
      UI.SetUserShakeStrength(num)
    end,
    MinValue = UI.GetUserShakeStrengthMin(),
    MaxValue = UI.GetUserShakeStrengthMax(),
    Delta = UI.GetUserShakeStrengthDelta,
    Default = function()
      return 10
    end
  })
  self.aimHorizontalSensitivitySlider = slider.Slider.New(self, {
    Name = "AimHorizontalSensitivity",
    TextFormat = "%d",
    Get = UI.GetUserCameraSpeedHorizontalAiming,
    Set = function(num)
      UI.SetUserCameraSpeedHorizontalAiming(num)
    end,
    MinValue = UI.GetUserCameraSpeedMin(),
    MaxValue = UI.GetUserCameraSpeedMax(),
    Delta = UI.GetUserCameraSpeedDelta,
    Default = function()
      return 5
    end
  })
  self.aimVerticalSensitivitySlider = slider.Slider.New(self, {
    Name = "AimVerticalSensitivity",
    TextFormat = "%d",
    Get = UI.GetUserCameraSpeedVerticalAiming,
    Set = function(num)
      UI.SetUserCameraSpeedVerticalAiming(num)
    end,
    MinValue = UI.GetUserCameraSpeedMin(),
    MaxValue = UI.GetUserCameraSpeedMax(),
    Delta = UI.GetUserCameraSpeedDelta,
    Default = function()
      return 5
    end
  })
  self.aimMouseHorizontalSensitivitySlider = slider.Slider.New(self, {
    Name = "AimHorizontalSensitivity",
    TextFormat = "%d",
    Get = UI.GetUserCameraSpeedHorizontalMouseAiming,
    Set = function(num)
      UI.SetUserCameraSpeedHorizontalMouseAiming(num)
    end,
    MinValue = UI.GetUserCameraSpeedMin(),
    MaxValue = UI.GetUserCameraSpeedMax(),
    Delta = UI.GetUserCameraSpeedDelta,
    Default = function()
      return 5
    end
  })
  self.aimMouseVerticalSensitivitySlider = slider.Slider.New(self, {
    Name = "AimVerticalSensitivity",
    TextFormat = "%d",
    Get = UI.GetUserCameraSpeedVerticalMouseAiming,
    Set = function(num)
      UI.SetUserCameraSpeedVerticalMouseAiming(num)
    end,
    MinValue = UI.GetUserCameraSpeedMin(),
    MaxValue = UI.GetUserCameraSpeedMax(),
    Delta = UI.GetUserCameraSpeedDelta,
    Default = function()
      return 5
    end
  })
  self.aimAssistToggle = toggle.Toggle.New(self, {
    Name = "AimAssistToggle",
    Get = UI.GetUserAimAssist,
    Set = function(num)
      UI.SetUserAimAssist(num)
    end,
    Default = function()
      return true
    end
  })
  self.attackYawToggle = toggle.Toggle.New(self, {
    Name = "AttackYawRecenter",
    Get = UI.GetUserAttackRecenter,
    Set = function(num)
      UI.SetUserAttackRecenter(num)
    end,
    Default = function()
      return true
    end
  })
  self.combatStrafeAssistToggle = toggle.Toggle.New(self, {
    Name = "AttackYawRecenter",
    Get = UI.GetUserStrafeAssist,
    Set = function(num)
      UI.SetUserStrafeAssist(num)
    end,
    Default = function()
      return true
    end
  })
  self.cameraSettingsInfo = {
    {
      Item = "CameraSettings",
      Text = util.GetLAMSMsg(lamsIDs.CameraSettings),
      ButtonType = "category",
      ButtonType = "category"
    },
    {
      Item = "horizontalSensitivitySlider",
      Text = util.GetLAMSMsg(lamsIDs.HorizontalSensitivity),
      ActivationList = {
        {
          Item = self.horizontalSensitivitySlider
        }
      },
      ToolTip = util.GetLAMSMsg(lamsIDs.HorizontalSensitivityDesc)
    },
    {
      Item = "verticalSensitivitySlider",
      Text = util.GetLAMSMsg(lamsIDs.VerticalSensitivity),
      ActivationList = {
        {
          Item = self.verticalSensitivitySlider
        }
      },
      ToolTip = util.GetLAMSMsg(lamsIDs.VerticalSensitivityDesc)
    },
    {
      Item = "horizontalMouseSensitivitySlider",
      Text = util.GetLAMSMsg(lamsIDs.MouseHorizontal),
      ActivationList = {
        {
          Item = self.horizontalMouseSensitivitySlider
        }
      },
      ToolTip = util.GetLAMSMsg(lamsIDs.HorizontalSensitivityDesc)
    },
    {
      Item = "verticalMouseSensitivitySlider",
      Text = util.GetLAMSMsg(lamsIDs.MouseVertical),
      ActivationList = {
        {
          Item = self.verticalMouseSensitivitySlider
        }
      },
      ToolTip = util.GetLAMSMsg(lamsIDs.VerticalSensitivityDesc)
    },
    {
      Item = "InvertXAxisToggle",
      Text = util.GetLAMSMsg(lamsIDs.InvertXAxis),
      ActivationList = {
        {
          Item = self.InvertXAxisToggle
        }
      },
      ToolTip = util.GetLAMSMsg(lamsIDs.InvertXAxisToggleDesc)
    },
    {
      Item = "InvertYAxisToggle",
      Text = util.GetLAMSMsg(lamsIDs.InvertYAxis),
      ActivationList = {
        {
          Item = self.invertYAxisToggle
        }
      },
      ToolTip = util.GetLAMSMsg(lamsIDs.InvertYAxisToggleDesc)
    },
    {
      Item = "swayStrengthSlider",
      Text = util.GetLAMSMsg(lamsIDs.AmbientCameraSway),
      ActivationList = {
        {
          Item = self.swayStrengthSlider
        }
      },
      ToolTip = util.GetLAMSMsg(lamsIDs.AmbientCameraSwayDesc)
    },
    {
      Item = "shakeStrengthSlider",
      Text = util.GetLAMSMsg(lamsIDs.ShakeCameraStrength),
      ActivationList = {
        {
          Item = self.shakeStrengthSlider
        }
      },
      ToolTip = util.GetLAMSMsg(lamsIDs.ShakeCameraStrengthTip)
    },
    {
      Item = "aimCameraHeader",
      Text = util.GetLAMSMsg(lamsIDs.AimCameraHeader),
      ButtonType = "category"
    },
    {
      Item = "aimHorizontalSensitivitySlider",
      Text = util.GetLAMSMsg(lamsIDs.HorizontalSensitivity),
      ActivationList = {
        {
          Item = self.aimHorizontalSensitivitySlider
        }
      },
      ToolTip = util.GetLAMSMsg(lamsIDs.HorizontalSensitivityAimDesc)
    },
    {
      Item = "aimVerticalSensitivitySlider",
      Text = util.GetLAMSMsg(lamsIDs.VerticalSensitivity),
      ActivationList = {
        {
          Item = self.aimVerticalSensitivitySlider
        }
      },
      ToolTip = util.GetLAMSMsg(lamsIDs.VerticalSensitivityAimDesc)
    },
    {
      Item = "aimHorizontalSensitivitySlider",
      Text = util.GetLAMSMsg(lamsIDs.MouseAimHorizontal),
      ActivationList = {
        {
          Item = self.aimMouseHorizontalSensitivitySlider
        }
      },
      ToolTip = util.GetLAMSMsg(lamsIDs.HorizontalSensitivityAimDesc)
    },
    {
      Item = "aimVerticalSensitivitySlider",
      Text = util.GetLAMSMsg(lamsIDs.MouseAimVertical),
      ActivationList = {
        {
          Item = self.aimMouseVerticalSensitivitySlider
        }
      },
      ToolTip = util.GetLAMSMsg(lamsIDs.VerticalSensitivityAimDesc)
    },
    {
      Item = "AimAssist",
      Text = util.GetLAMSMsg(lamsIDs.AimAssist),
      ActivationList = {
        {
          Item = self.aimAssistToggle
        }
      },
      ToolTip = util.GetLAMSMsg(lamsIDs.AimAssistDesc)
    },
    {
      Item = "combatCameraHeader",
      Text = util.GetLAMSMsg(lamsIDs.CombatCameraHeader),
      ButtonType = "category"
    },
    {
      Item = "attackYawToggle",
      Text = util.GetLAMSMsg(lamsIDs.AttackYawToggle),
      ActivationList = {
        {
          Item = self.attackYawToggle
        }
      },
      ToolTip = util.GetLAMSMsg(lamsIDs.AttackYawToggleDesc)
    },
    {
      Item = "combatStrafeAssistToggle",
      Text = util.GetLAMSMsg(lamsIDs.StrafeAssistStrength),
      ActivationList = {
        {
          Item = self.combatStrafeAssistToggle
        }
      },
      ToolTip = util.GetLAMSMsg(lamsIDs.StrafeAssistStrengthDesc)
    }
  }
end
function CameraSettings:Enter()
  self.handling_event = false
  self:SetAllHandlersFalse()
  self.menu:Activate()
  local cameraSettingsList = self.menu:GetList("CameraSettings")
  self.settingsOn:RefreshSettingsList(cameraSettingsList, self.cameraSettingsInfo, util.GetLAMSMsg(lamsIDs.CameraSettings))
  self:UpdateFooterButtonPrompt()
  Audio.PlaySound("SND_UX_Menu_Tick_Confirm")
end
function CameraSettings:Exit()
  self.menu:Deactivate(true)
  Audio.PlaySound("SND_UX_Menu_Tick_Back")
end
function CameraSettings:SelectCameraSetting(currItem)
  self:UpdateFooterButtonPrompt()
end
function CameraSettings:SetDefaultValues()
  local CameraSettingsList = self.menu:GetList("CameraSettings")
  self.settingsOn:SetDefaultValues(CameraSettingsList, self.menu, function()
    self:Enter()
  end)
end
function CameraSettings:UpdateFooterButtonPrompt()
  self.menu:UpdateFooterButton("Default", true)
  self.menu:UpdateFooterButton("Back", true)
  self.menu:UpdateFooterButton("Close", true)
  self.menu:UpdateFooterButtonText()
end
function CameraSettings:EVT_MOUSE_CLICKED()
  if not self.handling_event then
    self.advance = true
  end
end
function CameraSettings:EVT_MOUSE_RIGHT_CLICKED()
  if not self.handling_event then
    self.back = true
  end
end
function CameraSettings:EVT_Back_Release()
  if not self.handling_event then
    self.back = true
  end
end
function CameraSettings:EVT_Options_Release()
  if not self.handling_event then
    self.back_options = true
  end
end
function CameraSettings:EVT_Triangle_Release()
  if not self.handling_event then
    self.triangle = true
  end
end
function CameraSettings:SetAllHandlersFalse()
  self.back = false
  self.back_options = false
  self.triangle = false
end
function CameraSettings:Update()
  if not self.handling_event then
    if self.back then
      self.handling_event = true
      self:SetAllHandlersFalse()
      self:Goto("MainSettings")
    elseif self.back_options then
      self.handling_event = true
      self:SetAllHandlersFalse()
      self.settingsOn:OptionsReleaseHandler()
    elseif self.triangle then
      self.handling_event = true
      self:SetAllHandlersFalse()
      self:SetDefaultValues()
    end
  end
end
function AudioSettings:Setup()
  self.settingsOn = self:GetState("SettingsOn")
  self.handling_event = false
  self:SetAllHandlersFalse()
  self.menu = menu.Menu.New(self, {
    FooterButtonInfo = {
      {
        Item = "Default",
        Text = "[TriangleButton] " .. util.GetLAMSMsg(lamsIDs.Default)
      },
      {
        Item = "Back",
        Text = "[BackButton] " .. util.GetLAMSMsg(lamsIDs.Back)
      }
    }
  })
  local audioSettingsList = list.List.New(self, {
    MaxFocusableObjectCount = 20,
    ListObjectName = "MiddleSettingsList",
    EmptyTextLamsID = lamsIDs.NoSettingsAvailable,
    BackingAnimFrameCount = 60,
    HoverToSelect = true,
    Button_OnGainFocus = function(button)
      local currItem = button:get_item()
      self:SelectAudioSetting(currItem)
    end
  })
  self.menu:SetList("AudioSettings", audioSettingsList)
  self.settingsOn = self:GetState("SettingsOn")
  self.musicVolumeSlider = slider.Slider.New(self, {
    Name = "MusicVolume",
    TextFormat = "%d",
    Sound = "SND_UX_Menu_Volume_Slider_Tick_Music",
    Get = function()
      return Audio.GetBusVolume(consts.kMusic)
    end,
    Set = function(value)
      Audio.SetBusVolume(consts.kMusic, value)
    end,
    MinValue = 0,
    MaxValue = 100,
    Delta = 1,
    Default = function()
      return 100
    end
  })
  self.dialogVolumeSlider = slider.Slider.New(self, {
    Name = "DialogVolume",
    TextFormat = "%d",
    Sound = "SND_UX_Menu_Volume_Slider_Tick_Dialog",
    Get = function()
      return Audio.GetBusVolume(consts.kDialog)
    end,
    Set = function(value)
      Audio.SetBusVolume(consts.kDialog, value)
    end,
    MinValue = 0,
    MaxValue = 100,
    Delta = 1,
    Default = function()
      return 100
    end
  })
  self.sfxVolumeSlider = slider.Slider.New(self, {
    Name = "SFXVolume",
    TextFormat = "%d",
    Sound = "SND_UX_Menu_Volume_Slider_Tick_SFX",
    Get = function()
      return Audio.GetBusVolume(consts.kSFX)
    end,
    Set = function(value)
      Audio.SetBusVolume(consts.kSFX, value)
    end,
    MinValue = 0,
    MaxValue = 100,
    Delta = 1,
    Default = function()
      return 100
    end
  })
  self.soundInBackgroundToggle = toggle.Toggle.New(self, {
    Name = "SoundInBackground",
    Get = UI.GetSoundInBackground,
    Set = function(num)
      UI.SetSoundInBackground(num)
    end,
    Default = function()
      return true
    end
  })
  self.subtitlesToggle = toggle.Toggle.New(self, {
    Name = "Subtitles",
    Get = UI.GetSubtitles,
    Set = function(num)
      UI.SetSubtitles(num)
    end,
    Default = function()
      return not self:AreLanguagesTheSame()
    end
  })
  self.panningModeToggle = toggle.Toggle.New(self, {
    Name = "PanningModeToggle",
    Get = function()
      return Audio.GetPanningMode()
    end,
    Set = function(num)
      if num == 1 then
        Audio.SetPanningModeHeadphone()
      else
        Audio.SetPanningModeSpeakers()
      end
    end,
    Settings = self:DeterminePanningModeSettings(),
    Default = function()
      return 0
    end
  })
  local panningModeTip
  if self:StereoSetupDetected() then
    panningModeTip = util.GetLAMSMsg(lamsIDs.StereoDetectedTip)
  else
    panningModeTip = util.GetLAMSMsg(lamsIDs.SurroundSoundTip)
  end
  self.audioSettingsInfo = {
    {
      Item = "musicVolumeSlider",
      Text = util.GetLAMSMsg(lamsIDs.MusicVolume),
      ActivationList = {
        {
          Item = self.musicVolumeSlider
        }
      },
      ToolTip = util.GetLAMSMsg(lamsIDs.MusicVolumeTip)
    },
    {
      Item = "dialogVolumeSlider",
      Text = util.GetLAMSMsg(lamsIDs.DialogVolume),
      ActivationList = {
        {
          Item = self.dialogVolumeSlider
        }
      },
      ToolTip = util.GetLAMSMsg(lamsIDs.DialogVolumeTip)
    },
    {
      Item = "sfxVolumeSlider",
      Text = util.GetLAMSMsg(lamsIDs.SFXVolume),
      ActivationList = {
        {
          Item = self.sfxVolumeSlider
        }
      },
      ToolTip = util.GetLAMSMsg(lamsIDs.SFXVolumeTip)
    },
    {
      Item = "soundInBackgroundToggle",
      Text = util.GetLAMSMsg(lamsIDs.SoundInBackground),
      ActivationList = {
        {
          Item = self.soundInBackgroundToggle
        }
      },
      ToolTip = util.GetLAMSMsg(lamsIDs.SoundInBackgroundTip)
    },
    {
      Item = "subtitlesToggle",
      Text = util.GetLAMSMsg(lamsIDs.Subtitles),
      ActivationList = {
        {
          Item = self.subtitlesToggle
        }
      },
      ToolTip = util.GetLAMSMsg(lamsIDs.SubtitlesTip)
    },
    {
      Item = "panningModeToggle",
      Text = util.GetLAMSMsg(lamsIDs.PanningMode),
      ActivationList = {
        {
          Item = self.panningModeToggle
        }
      },
      ToolTip = panningModeTip
    }
  }
end
function AudioSettings:Enter()
  self.handling_event = false
  self:SetAllHandlersFalse()
  self.menu:Activate()
  local audioSettingsList = self.menu:GetList("AudioSettings")
  self.settingsOn:RefreshSettingsList(audioSettingsList, self.audioSettingsInfo, util.GetLAMSMsg(lamsIDs.AudioSettings))
  self:UpdateFooterButtonPrompt()
  Audio.PlaySound("SND_UX_Menu_Tick_Confirm")
end
function AudioSettings:AreLanguagesTheSame()
  local text_code = self:GetCodeForLanguage(UI.AvailableTextLanguages, UI.GetTextLanguage)
  local speech_code = self:GetCodeForLanguage(UI.AvailableSpeechLanguages, UI.GetSpeechLanguage)
  if text_code == speech_code then
    return true
  end
  return false
end
function AudioSettings:GetCodeForLanguage(getAllFunction, getCurrentFunction)
  local current_idx = getCurrentFunction()
  local get_all_idx = getAllFunction()
  return get_all_idx[current_idx]
end
function AudioSettings:Exit()
  self.menu:Deactivate(true)
  Audio.PlaySound("SND_UX_Menu_Tick_Back")
end
function AudioSettings:StereoSetupDetected()
  if Audio.GetPlaybackSpeakerCount ~= nil and Audio.GetPlaybackSpeakerCount() <= 2 then
    return true
  end
  return false
end
function AudioSettings:DeterminePanningModeSettings()
  local t = {
    {
      Value = 0,
      Text = util.GetLAMSMsg(lamsIDs.Off)
    }
  }
  if self:StereoSetupDetected() then
    local headphones_on = {
      Value = 1,
      Text = util.GetLAMSMsg(lamsIDs.On)
    }
    tablex.FastInsert(t, headphones_on, 2)
  end
  return t
end
function AudioSettings:SelectAudioSetting(currItem)
  self:UpdateFooterButtonPrompt()
end
function AudioSettings:SetDefaultValues()
  local AudioSettingsList = self.menu:GetList("AudioSettings")
  self.settingsOn:SetDefaultValues(AudioSettingsList, self.menu, function()
    self:Enter()
  end)
end
function AudioSettings:UpdateFooterButtonPrompt()
  self.menu:UpdateFooterButton("Default", true)
  self.menu:UpdateFooterButton("Back", true)
  self.menu:UpdateFooterButtonText()
end
function AudioSettings:EVT_MOUSE_CLICKED()
  if not self.handling_event then
    self.advance = true
  end
end
function AudioSettings:EVT_MOUSE_RIGHT_CLICKED()
  if not self.handling_event then
    self.back = true
  end
end
function AudioSettings:EVT_Back_Release()
  if not self.handling_event then
    self.back = true
  end
end
function AudioSettings:EVT_Options_Release()
  if not self.handling_event then
    self.back_options = true
  end
end
function AudioSettings:EVT_Triangle_Release()
  if not self.handling_event then
    self.triangle = true
  end
end
function AudioSettings:SetAllHandlersFalse()
  self.back = false
  self.back_options = false
  self.triangle = false
end
function AudioSettings:Update()
  if not self.handling_event then
    if self.back then
      self.handling_event = true
      self:SetAllHandlersFalse()
      self:Goto("MainSettings")
    elseif self.back_options then
      self.handling_event = true
      self:SetAllHandlersFalse()
      self.settingsOn:OptionsReleaseHandler()
    elseif self.triangle then
      self.handling_event = true
      self:SetAllHandlersFalse()
      self:SetDefaultValues()
    end
  end
end
function VideoSettings:Setup()
  self.handling_event = false
  self:SetAllHandlersFalse()
  self.graphics_mode_index = 1
  self.settingsOn = self:GetState("SettingsOn")
  self.menu = menu.Menu.New(self, {
    FooterButtonInfo = {
      {
        Item = "Select",
        Text = "[AdvanceButton] " .. util.GetLAMSMsg(lamsIDs.Select)
      },
      {
        Item = "Default",
        Text = "[TriangleButton] " .. util.GetLAMSMsg(lamsIDs.Default)
      },
      {
        Item = "Back",
        Text = "[BackButton] " .. util.GetLAMSMsg(lamsIDs.Back)
      }
    }
  })
  local VideoSettingsList = list.List.New(self, {
    MaxFocusableObjectCount = 20,
    ListObjectName = "MiddleSettingsList",
    EmptyTextLamsID = lamsIDs.NoSettingsAvailable,
    BackingAnimFrameCount = 60,
    HoverToSelect = true,
    Button_OnGainFocus = function(button)
      local currItem = button:get_item()
      self:SelectVideoSetting(currItem)
    end
  })
  self.menu:SetList("VideoSettings", VideoSettingsList)
  VideoSettingsList:SetSelectedButton(2, false)
  self.motionBlurSlider = slider.Slider.New(self, {
    Name = "MotionBlur",
    TextFormat = "%d",
    Get = UI.GetUserMotionBlur,
    Set = function(num)
      UI.SetUserMotionBlur(num)
    end,
    MinValue = UI.GetUserMotionBlurMin(),
    MaxValue = UI.GetUserMotionBlurMax(),
    Delta = UI.GetUserMotionBlurDelta,
    Default = function()
      return 10
    end
  })
  self.filmGrainSlider = slider.Slider.New(self, {
    Name = "FilmGrain",
    TextFormat = "%d",
    Get = function()
      return game.GetFilmGrainScale() * 10 / 100
    end,
    Set = function(num)
      local r = num * 100 / 10
      game.SetFilmGrainScale(r)
    end,
    MinValue = 0,
    MaxValue = 10,
    Delta = 1,
    Default = function()
      return 3
    end
  })
  self.graphicsModeToggle = toggle.Toggle.New(self, {
    Name = "GraphicsModeToggle",
    Settings = self:GetGraphicsModeList(),
    Get = function()
      if game.GetEnable4K ~= nil then
        return game.GetEnable4K()
      end
      return false
    end,
    Set = function(val)
      if game.SetEnable4K ~= nil then
        game.SetEnable4K(val)
      end
    end,
    Default = function()
      return true
    end
  })
  self.resolutionsToggle = toggle.Toggle.New(self, {
    Name = "ResolutionsToggle",
    Get = function()
      if game.GetDisplayMode() == 0 then
        return game.GetResolutionIndex()
      else
        return 1
      end
    end,
    Set = function(settingValue)
      if game.GetDisplayMode() == 0 then
        self.confirmed = game.SetResolutionIndex(settingValue)
        self.updateVRAM = 2
      end
    end,
    Default = function()
      return 0
    end
  })
  self.renderToggle = toggle.Toggle.New(self, {
    Name = "RenderToggle",
    Get = function()
      return 0
    end,
    Set = function(settingValue)
    end,
    Default = function()
      return 0
    end
  })
  local aspectRatios = game.GetAspectRatios()
  local aspectRatioSettings = {}
  aspectRatioSettings[1] = {}
  aspectRatioSettings[1].Value = 0
  aspectRatioSettings[1].Text = util.GetLAMSMsg(lamsIDs.Auto)
  for i = 2, #aspectRatios do
    aspectRatioSettings[i] = {}
    aspectRatioSettings[i].Value = i - 1
    aspectRatioSettings[i].Text = aspectRatios[i]
  end
  self.aspectRatiosToggle = toggle.Toggle.New(self, {
    Name = "AspectRatiosToggle",
    Settings = aspectRatioSettings,
    Get = function()
      return game.GetAspectRatioIndex()
    end,
    Set = function(settingValue)
      self.confirmed = game.SetAspectRatioIndex(settingValue)
      self.updateVRAM = 2
    end,
    Default = function()
      return 0
    end
  })
  self.orginalDevice = game.GetDevice()
  local devices = game.GetVideoDevices()
  self.deviceSettings = {}
  for i = 1, #devices do
    self.deviceSettings[i] = {}
    self.deviceSettings[i].Value = i - 1
    self.deviceSettings[i].Text = devices[i]
  end
  self.deviceToggle = toggle.Toggle.New(self, {
    Name = "DeviceToggle",
    Get = function()
      return game.GetDevice()
    end,
    Set = function(settingValue)
      game.SetDevice(settingValue)
    end,
    Settings = self.deviceSettings,
    Default = function()
      return 0
    end
  })
  local numMonitors = game.GetNumMonitors()
  self.monitorSettings = {}
  for i = 1, numMonitors do
    self.monitorSettings[i] = {}
    self.monitorSettings[i].Value = i - 1
    self.monitorSettings[i].Text = tostring(i)
  end
  self.monitorToggle = toggle.Toggle.New(self, {
    Name = "MonitorToggle",
    Get = function()
      return game.GetMonitor()
    end,
    Set = function(settingValue)
      self.confirmed = game.SetMonitor(settingValue)
      self.updateVRAM = 2
    end,
    Settings = self.monitorSettings,
    Default = function()
      return 0
    end
  })
  self.displayModeToggle = toggle.Toggle.New(self, {
    Name = "DisplayModeToggle",
    Get = function()
      return game.GetDisplayMode()
    end,
    Set = function(settingValue)
      self.confirmed = game.SetDisplayMode(settingValue)
      self.updateVRAM = 2
    end,
    Settings = {
      {
        Value = 0,
        Text = util.GetLAMSMsg(lamsIDs.Windowed)
      },
      {
        Value = 1,
        Text = util.GetLAMSMsg(lamsIDs.Borderless)
      }
    },
    Default = function()
      return 0
    end
  })
  self.vSyncToggle = toggle.Toggle.New(self, {
    Name = "VSyncToggle",
    Get = function()
      return game.GetVSync()
    end,
    Set = function(settingValue)
      self.confirmed = game.SetVSync(settingValue)
      self:UpdateFooterButtonPrompt(false, self.confirmed)
    end,
    Settings = {
      {
        Value = false,
        Text = util.GetLAMSMsg(lamsIDs.Off)
      },
      {
        Value = true,
        Text = util.GetLAMSMsg(lamsIDs.On)
      }
    },
    Default = function()
      return false
    end
  })
  self.fpsLimitSlider = slider.Slider.New(self, {
    Name = "FpsLimit",
    TextFormat = "%s" .. util.GetLAMSMsg(lamsIDs.fps),
    Get = function()
      return game.GetFpsLimit()
    end,
    Set = function(num)
      game.SetFpsLimit(num)
    end,
    MinValue = 20,
    MaxValue = 120,
    Delta = 10,
    Default = function()
      return self.fpsLimitSlider:get_minValue()
    end
  })
  self.dlssToggle = toggle.Toggle.New(self, {
    Name = "DlssToggle",
    Get = function()
      return game.GetDLSSMode()
    end,
    Set = function(settingValue)
      game.SetDLSSMode(settingValue)
      if 0 < settingValue then
        self.fsrToggle:ForceSetValue(0)
        self.fcasSlider:SetValue(self.fcasSlider:get_minValue())
        self.scaleSlider:SetValue(self.scaleSlider:get_minValue())
        game.SetScale(self.scaleSlider:get_minValue())
      elseif game.GetFSRMode() == 0 and self.fcasSlider:get_value() <= self.fcasSlider:get_minValue() and self.scaleSlider:get_value() <= self.scaleSlider:get_minValue() then
        self.scaleSlider:SetValue(self.scaleSlider:get_maxValue())
      end
      self.updateVRAM = 2
    end,
    Default = function()
      return 0
    end
  })
  self.sharpnessSlider = slider.Slider.New(self, {
    Name = "SharpnessSlider",
    TextFormat = "%d",
    Get = function()
      return game.GetDLSSSharpness()
    end,
    Set = function(num)
      game.SetDLSSSharpness(num)
    end,
    MinValue = 0,
    MaxValue = 100,
    Default = function()
      return 35
    end
  })
  self.fsrToggle = toggle.Toggle.New(self, {
    Name = "FsrToggle",
    Get = function()
      return game.GetFSRMode()
    end,
    Set = function(settingValue)
      game.SetFSRMode(settingValue)
      if 0 < settingValue then
        self.dlssToggle:ForceSetValue(0)
        self.fcasSlider:SetValue(self.fcasSlider:get_minValue())
        self.scaleSlider:SetValue(self.scaleSlider:get_minValue())
        game.SetScale(self.scaleSlider:get_minValue())
      elseif game.GetDLSSMode() == 0 and self.fcasSlider:get_value() <= self.fcasSlider:get_minValue() and self.scaleSlider:get_value() <= self.scaleSlider:get_minValue() then
        self.scaleSlider:SetValue(self.scaleSlider:get_maxValue())
      end
      self.updateVRAM = 2
    end,
    Settings = {
      {
        Value = 0,
        Text = util.GetLAMSMsg(lamsIDs.Off)
      },
      {
        Value = 1,
        Text = util.GetLAMSMsg(lamsIDs.Performance)
      },
      {
        Value = 2,
        Text = util.GetLAMSMsg(lamsIDs.Balanced)
      },
      {
        Value = 3,
        Text = util.GetLAMSMsg(lamsIDs.Quality)
      },
      {
        Value = 4,
        Text = util.GetLAMSMsg(lamsIDs.UltraQuality)
      }
    },
    Default = function()
      return 0
    end
  })
  self.fcasSlider = slider.Slider.New(self, {
    Name = "FCAS",
    TextFormat = "%s%%",
    Get = function()
      return game.GetFCAS()
    end,
    Set = function(num)
      game.SetFCAS(num)
      if num > self.fcasSlider:get_minValue() then
        self.dlssToggle:ForceSetValue(0)
        self.fsrToggle:ForceSetValue(0)
        self.scaleSlider:SetValue(self.scaleSlider:get_minValue())
        game.SetScale(self.scaleSlider:get_minValue())
      elseif game.GetDLSSMode() == 0 and game.GetFSRMode() == 0 and self.scaleSlider:get_value() <= self.scaleSlider:get_minValue() then
        self.scaleSlider:SetValue(self.scaleSlider:get_maxValue())
      end
      self.updateVRAM = 2
    end,
    MinValue = 45,
    MaxValue = 100,
    Delta = 5,
    Default = function()
      return self.fcasSlider:get_minValue()
    end
  })
  self.scaleSlider = slider.Slider.New(self, {
    Name = "Scale",
    TextFormat = "%s%%",
    Get = function()
      return game.GetScale()
    end,
    Set = function(num)
      if num > self.scaleSlider:get_minValue() then
        game.SetScale(num)
        self.dlssToggle:ForceSetValue(0)
        self.fsrToggle:ForceSetValue(0)
        self.fcasSlider:SetValue(self.fcasSlider:get_minValue())
        self.updateVRAM = 2
      elseif game.GetDLSSMode() == 0 and game.GetFSRMode() == 0 and self.fcasSlider:get_value() <= self.fcasSlider:get_minValue() then
        self.scaleSlider:SetValue(self.scaleSlider:get_minValue() + self.scaleSlider._delta)
      end
    end,
    MinValue = 45,
    MaxValue = 100,
    Delta = 5,
    Default = function()
      return self.scaleSlider:get_maxValue()
    end
  })
  self.videoSettingsInfo = {
    {
      Item = "GraphicsModeToggle",
      Text = util.GetLAMSMsg(lamsIDs.GraphicsMode),
      ActivationList = {
        {
          Item = self.graphicsModeToggle
        }
      },
      ToggleTip = true
    },
    {
      Item = "DeviceToggle",
      Text = util.GetLAMSMsg(lamsIDs.VideoDevice),
      ActivationList = {
        {
          Item = self.deviceToggle
        }
      },
      ToolTip = util.GetLAMSMsg(lamsIDs.VideoDeviceDesc)
    },
    {
      Item = "MonitorToggle",
      Text = util.GetLAMSMsg(lamsIDs.Monitor),
      ActivationList = {
        {
          Item = self.monitorToggle
        }
      },
      ToolTip = util.GetLAMSMsg(lamsIDs.MonitorDesc)
    },
    {
      Item = "DisplayModeToggle",
      Text = util.GetLAMSMsg(lamsIDs.DisplayMode),
      ActivationList = {
        {
          Item = self.displayModeToggle
        }
      },
      ToolTip = util.GetLAMSMsg(lamsIDs.DisplayModeDesc)
    },
    {
      Item = "ResolutionsToggle",
      Text = util.GetLAMSMsg(lamsIDs.OutputResolution),
      ActivationList = {
        {
          Item = self.resolutionsToggle
        }
      },
      ToolTip = util.GetLAMSMsg(lamsIDs.OutputResolutionDesc)
    },
    {
      Item = "RenderToggle",
      Text = util.GetLAMSMsg(lamsIDs.RenderResolution),
      ActivationList = {
        {
          Item = self.renderToggle
        }
      },
      ToolTip = util.GetLAMSMsg(lamsIDs.RenderResolutionDesc)
    },
    {
      Item = "AspectRatiosToggle",
      Text = util.GetLAMSMsg(lamsIDs.AspectRatio),
      ActivationList = {
        {
          Item = self.aspectRatiosToggle
        }
      },
      ToolTip = util.GetLAMSMsg(lamsIDs.AspectRatioDesc)
    },
    {
      Item = "VSyncToggle",
      Text = util.GetLAMSMsg(lamsIDs.Vsync),
      ActivationList = {
        {
          Item = self.vSyncToggle
        }
      },
      ToolTip = util.GetLAMSMsg(lamsIDs.VsyncDesc)
    },
    {
      Item = "FpsLimitSlider",
      Text = util.GetLAMSMsg(lamsIDs.FpsLimit),
      ActivationList = {
        {
          Item = self.fpsLimitSlider
        }
      },
      ToolTip = util.GetLAMSMsg(lamsIDs.FpsLimitDesc)
    },
    {
      Item = "motionBlurSlider",
      Text = util.GetLAMSMsg(lamsIDs.MotionBlur),
      ActivationList = {
        {
          Item = self.motionBlurSlider
        }
      },
      ToolTip = util.GetLAMSMsg(lamsIDs.MotionBlurDesc)
    },
    {
      Item = "FilmGrainSlider",
      Text = util.GetLAMSMsg(lamsIDs.FilmGrain),
      ActivationList = {
        {
          Item = self.filmGrainSlider
        }
      },
      ToolTip = util.GetLAMSMsg(lamsIDs.FilmGrainDesc)
    },
    {
      Item = "Calibration",
      Text = util.GetLAMSMsg(lamsIDs.CalibrationSettings),
      Advance = function()
        self:Goto("Calibration")
      end,
      ToolTip = util.GetLAMSMsg(lamsIDs.CalibrationSettingsDesc)
    },
    {
      Item = "Advanced",
      Text = util.GetLAMSMsg(lamsIDs.AdvancedSettings),
      Advance = function()
        self:Goto("AdvancedSettings")
      end,
      ToolTip = util.GetLAMSMsg(lamsIDs.AdvancedSettingsDesc)
    },
    {
      Item = "ScalingHeader",
      Text = util.GetLAMSMsg(lamsIDs.scaling),
      ButtonType = "category"
    },
    {
      Item = "DlssToggle",
      Text = util.GetLAMSMsg(lamsIDs.DLSS),
      ActivationList = {
        {
          Item = self.dlssToggle
        }
      },
      ToolTip = util.GetLAMSMsg(lamsIDs.DLSSDesc)
    },
    {
      Item = "SharpnessSlider",
      Text = util.GetLAMSMsg(lamsIDs.Sharpness),
      ActivationList = {
        {
          Item = self.sharpnessSlider
        }
      },
      ToolTip = util.GetLAMSMsg(lamsIDs.SharpnessDesc)
    },
    {
      Item = "FSRToggle",
      Text = util.GetLAMSMsg(lamsIDs.FSR),
      ActivationList = {
        {
          Item = self.fsrToggle
        }
      },
      ToolTip = util.GetLAMSMsg(lamsIDs.FSRDesc)
    },
    {
      Item = "ScaleSlider",
      Text = util.GetLAMSMsg(lamsIDs.Scale),
      ActivationList = {
        {
          Item = self.scaleSlider
        }
      },
      ToolTip = util.GetLAMSMsg(lamsIDs.ScaleDesc)
    }
  }
  self.ResolutionsToggleIndex = 5
  self.RenderToggleIndex = 6
  self.DLSSToggleIndex = 14
  self.SharpnessSliderIndex = 15
  self.FSRToggleIndex = 16
  if game.CanRender4K == nil then
    table.remove(self.videoSettingsInfo, self.graphics_mode_index)
    self.ResolutionsToggleIndex = 4
    self.RenderToggleIndex = 5
    self.DLSSToggleIndex = 13
    self.SharpnessSliderIndex = 14
    self.FSRToggleIndex = 15
  end
  if not game.HasFSR() then
    table.remove(self.videoSettingsInfo, self.FSRToggleIndex)
  end
end
function VideoSettings:RefreshResolutionSettingsList()
  local resolutions = game.GetResolutions()
  local videoSettingsList = self.menu:GetList("VideoSettings")
  videoSettingsList:Deactivate()
  self.settingsOn:RefreshSettingsList(videoSettingsList, self.videoSettingsInfo, util.GetLAMSMsg(lamsIDs.Display))
  local numMonitors = game.GetNumMonitors()
  self.monitorSettings = {}
  for i = 1, numMonitors do
    self.monitorSettings[i] = {}
    self.monitorSettings[i].Value = i - 1
    self.monitorSettings[i].Text = tostring(i)
  end
  self.monitorToggle:OverrideSettings(self.monitorSettings)
  self.resolutionSettings = {}
  if game.GetDisplayMode() == 0 then
    for i = 1, #resolutions do
      self.resolutionSettings[i] = {}
      self.resolutionSettings[i].Value = i - 1
      self.resolutionSettings[i].Text = resolutions[i]
    end
    if game.GetResolutionIndex() < #resolutions then
      self.resolutionSettings[game.GetResolutionIndex() + 1].Text = game.GetOutputWidth() .. " x " .. game.GetOutputHeight()
    end
    self.settingsOn:ClearDisableButton(videoSettingsList, self.ResolutionsToggleIndex)
    self.settingsOn:ClearDisableButton(self.resolutionsToggle, 1)
    self.resolutionsToggle:OverrideSettings(self.resolutionSettings)
  else
    self.resolutionSettings[1] = {}
    self.resolutionSettings[1].Value = 0
    self.resolutionSettings[1].Text = game.GetOutputWidth() .. " x " .. game.GetOutputHeight()
    self.resolutionsToggle:OverrideSettings(self.resolutionSettings)
    self.settingsOn:CheckDisabledButton(videoSettingsList, self.ResolutionsToggleIndex, true)
    self.settingsOn:CheckDisabledButton(self.resolutionsToggle, 1, true)
  end
  if game.HasDLSS() then
    if game.HasUltraDLSS() then
      self.dlssSettings = {
        {
          Value = 0,
          Text = util.GetLAMSMsg(lamsIDs.Off)
        },
        {
          Value = 1,
          Text = util.GetLAMSMsg(lamsIDs.UltraPerformance)
        },
        {
          Value = 2,
          Text = util.GetLAMSMsg(lamsIDs.Performance)
        },
        {
          Value = 3,
          Text = util.GetLAMSMsg(lamsIDs.Balanced)
        },
        {
          Value = 4,
          Text = util.GetLAMSMsg(lamsIDs.Quality)
        },
        {
          Value = 5,
          Text = util.GetLAMSMsg(lamsIDs.UltraQuality)
        }
      }
    else
      self.dlssSettings = {
        {
          Value = 0,
          Text = util.GetLAMSMsg(lamsIDs.Off)
        },
        {
          Value = 1,
          Text = util.GetLAMSMsg(lamsIDs.UltraPerformance)
        },
        {
          Value = 2,
          Text = util.GetLAMSMsg(lamsIDs.Performance)
        },
        {
          Value = 3,
          Text = util.GetLAMSMsg(lamsIDs.Balanced)
        },
        {
          Value = 4,
          Text = util.GetLAMSMsg(lamsIDs.Quality)
        }
      }
    end
    self.settingsOn:ClearDisableButton(videoSettingsList, self.DLSSToggleIndex)
    self.settingsOn:ClearDisableButton(self.dlssToggle, 1)
  else
    if game.DLSSNeedsUpdate() then
      self.dlssSettings = {
        {
          Value = 0,
          Text = util.GetLAMSMsg(lamsIDs.UpdateRequired)
        }
      }
    else
      self.dlssSettings = {
        {
          Value = 0,
          Text = util.GetLAMSMsg(lamsIDs.NotSupported)
        }
      }
    end
    self.settingsOn:CheckDisabledButton(videoSettingsList, self.DLSSToggleIndex, true)
    self.settingsOn:CheckDisabledButton(self.dlssToggle, 1, true)
  end
  self.dlssToggle:OverrideSettings(self.dlssSettings)
  self.renderSettings = {}
  self.renderSettings[1] = {}
  self.renderSettings[1].Value = 0
  self.renderSettings[1].Text = game.GetRenderWidth() .. " x " .. game.GetRenderHeight()
  self.renderToggle:OverrideSettings(self.renderSettings)
  self.displayMode = 0
end
function VideoSettings:Enter()
  self.orginalDevice = game.GetDevice()
  self.handling_event = false
  self.confirmed = true
  self.updateVRAM = 0
  self.displayMode = game.GetDisplayMode()
  self:SetAllHandlersFalse()
  self.menu:Activate()
  local videoSettingsList = self.menu:GetList("VideoSettings")
  self.settingsOn:RefreshSettingsList(videoSettingsList, self.videoSettingsInfo, util.GetLAMSMsg(lamsIDs.Display))
  self:UpdateFooterButtonPrompt(true, self.confirmed)
  Audio.PlaySound("SND_UX_Menu_Tick_Confirm")
  self.availableVRAM = game.GetAvailableVRAM()
  self.usedVRAM = game.GetUsedVRAM()
  self:CheckGraphicsMode(videoSettingsList)
  self.settingsOn:CheckDisabledButton(videoSettingsList, self.RenderToggleIndex, true)
  self.settingsOn:CheckDisabledButton(self.renderToggle, 1, true)
  self:RefreshResolutionSettingsList()
end
function VideoSettings:UpdateSetting(value, toggle)
  local currValueIndex = 1
  for buttonIndex, button in ipairs(toggle._buttonArray) do
    local setting = button:get_item()
    local settingValue = setting.Value
    if settingValue == value then
      currValueIndex = buttonIndex
      break
    end
  end
  toggle:SetSelectedButton(currValueIndex, false)
end
function VideoSettings:EVT_UPDATE_WINDOW_SIZE()
  self.updateVRAM = 1
  self:UpdateSetting(game.GetResolutionIndex(), self.resolutionsToggle)
end
function VideoSettings:EVT_UPDATE_WINDOW_POSITION()
  self.updateVRAM = 1
  self:UpdateSetting(game.GetResolutionIndex(), self.resolutionsToggle)
  self:UpdateSetting(game.GetMonitor(), self.monitorToggle)
  self:UpdateSetting(game.GetDisplayMode(), self.displayModeToggle)
end
function VideoSettings:Exit()
  self:SendEventToUIFsm("globalMenu", "EVT_SET_FOOTER_INFO_TEXT", "")
  if game.CanRender4K ~= nil then
    local videoSettingsList = self.menu:GetList("VideoSettings")
    self.settingsOn:ClearDisableButton(videoSettingsList, self.graphics_mode_index)
    self.settingsOn:ClearDisableButton(self.graphicsModeToggle, 1)
  end
  local videoSettingsList = self.menu:GetList("VideoSettings")
  self.settingsOn:ClearDisableButton(videoSettingsList, self.ResolutionsToggleIndex)
  self.settingsOn:ClearDisableButton(self.resolutionsToggle, 1)
  self.settingsOn:ClearDisableButton(videoSettingsList, self.RenderToggleIndex)
  self.settingsOn:ClearDisableButton(self.renderToggle, 1)
  self.settingsOn:ClearDisableButton(videoSettingsList, self.DLSSToggleIndex)
  self.settingsOn:ClearDisableButton(self.dlssToggle, 1)
  self.menu:Deactivate(true)
  Audio.PlaySound("SND_UX_Menu_Tick_Back")
end
function VideoSettings:CheckGraphicsMode(videoSettingsList)
  local canRender = false
  if game.CanRender4K ~= nil then
    self.settingsOn:CheckDisabledButton(videoSettingsList, self.graphics_mode_index, not canRender)
    self.settingsOn:CheckDisabledButton(self.graphicsModeToggle, 1, not canRender)
  end
end
function VideoSettings:GetGraphicsModeList()
  local t = {
    {
      Value = false,
      Text = util.GetLAMSMsg(lamsIDs.FavorPerformance),
      ToolTip = util.GetLAMSMsg(lamsIDs.FavorPerformanceDesc)
    },
    {
      Value = true,
      Text = util.GetLAMSMsg(lamsIDs.FavorResolution),
      ToolTip = util.GetLAMSMsg(lamsIDs.FavorResolutionDesc)
    }
  }
  if game.CanRender4K == nil or not game.CanRender4K() then
    local disabled = t[2]
    disabled.ToolTip = util.GetLAMSMsg(lamsIDs.OptionAvailablePS4Pro)
    return {disabled}
  end
  return t
end
function VideoSettings:SelectVideoSetting(currItem)
  if currItem ~= nil and (currItem.Item == "Calibration" or currItem.Item == "ScreenScale") then
    self:UpdateFooterButtonPrompt(true, self.confirmed)
  else
    self:UpdateFooterButtonPrompt(false, self.confirmed)
  end
end
function VideoSettings:SetDefaultValues()
  local VideoSettingsList = self.menu:GetList("VideoSettings")
  self.settingsOn:SetDefaultValues(VideoSettingsList, self.menu, function()
    self:Enter()
  end)
end
function VideoSettings:ConfirmDeviceChange()
  self:SendEventUI("EVT_SHOW_SETTINGS_DIALOG_MESSAGE", {
    Header = util.GetLAMSMsg(lamsIDs.VideoDevice),
    Body = util.GetLAMSMsg(lamsIDs.VideoDeviceChange),
    ActiveMenus = {
      self.menu
    },
    ActiveLists = {
      self.menu:GetList("VideoSettings")
    },
    Events = {
      Advance = {
        Text = util.GetLAMSMsg(lamsIDs.Confirm),
        Handler = function()
          self:Enter()
          self:SetAllHandlersFalse()
          self:Goto("MainSettings")
        end
      }
    }
  })
end
function VideoSettings:HandleAdvanceOnList()
  local optionsList = self.menu:GetList("VideoSettings")
  local button = optionsList:GetSelectedButton()
  if button ~= nil then
    local advance_event = button:get_advance()
    if advance_event ~= nil then
      advance_event()
    else
      self.handling_event = false
    end
  end
end
function VideoSettings:UpdateFooterButtonPrompt(advance, confirmed)
  self.menu:UpdateFooterButton("Select", advance)
  self.menu:UpdateFooterButton("Confirm", not confirmed)
  self.menu:UpdateFooterButton("Default", true)
  self.menu:UpdateFooterButton("Back", true)
  self.menu:UpdateFooterButton("Close", true)
  self.menu:UpdateFooterButtonText()
end
function VideoSettings:EVT_Advance_Release()
  if not self.handling_event then
    self.advance = true
  end
end
function VideoSettings:EVT_MOUSE_CLICKED()
  if not self.handling_event then
    self.advance = true
  end
end
function VideoSettings:EVT_MOUSE_RIGHT_CLICKED()
  if not self.handling_event then
    self.back = true
  end
end
function VideoSettings:EVT_Back_Release()
  if not self.handling_event then
    self.back = true
  end
end
function VideoSettings:EVT_Options_Release()
  if not self.handling_event then
    self.back_options = true
  end
end
function VideoSettings:EVT_Triangle_Release()
  if not self.handling_event then
    self.triangle = true
  end
end
function VideoSettings:SetAllHandlersFalse()
  self.back = false
  self.back_options = false
  self.triangle = false
  self.advance = false
end
function VideoSettings:Update()
  if self.updateVRAM > 0 then
    self.updateVRAM = self.updateVRAM - 1
    if self.updateVRAM == 0 then
      self.usedVRAM = game.GetUsedVRAM()
      self:UpdateFooterButtonPrompt(false, self.confirmed)
      self:RefreshResolutionSettingsList()
    end
  elseif game.GetDLSSMode() == 0 and game.GetFSRMode() == 0 and self.fcasSlider:get_value() <= self.fcasSlider:get_minValue() and self.scaleSlider:get_value() <= self.scaleSlider:get_minValue() then
    self.scaleSlider:SetValue(self.scaleSlider:get_maxValue())
    self.updateVRAM = 2
  else
    local displayMode = game.GetDisplayMode()
    if self.displayMode ~= displayMode then
      self.displayMode = displayMode
      self:UpdateSetting(displayMode, self.displayModeToggle)
    end
  end
  local text = util.GetLAMSMsg(lamsIDs.VramUsed) .. " " .. self.usedVRAM .. " / " .. self.availableVRAM .. "MB"
  self:SendEventToUIFsm("globalMenu", "EVT_SET_FOOTER_INFO_TEXT", text)
  if not self.handling_event then
    if self.back then
      self.handling_event = true
      if game.GetDevice() == self.orginalDevice then
        self:SetAllHandlersFalse()
        self:Goto("MainSettings")
      else
        self:ConfirmDeviceChange()
      end
    elseif self.back_options then
      self.handling_event = true
      self:SetAllHandlersFalse()
      self.settingsOn:OptionsReleaseHandler()
    elseif self.triangle then
      self.handling_event = true
      self:SetAllHandlersFalse()
      self:SetDefaultValues()
    elseif self.advance then
      self.handling_event = true
      self:SetAllHandlersFalse()
      self:HandleAdvanceOnList()
    end
  end
end
function GraphicsSettings:Setup()
  self.handling_event = false
  self:SetAllHandlersFalse()
  self.desiredTexture = game.GetTextureQuality()
  self.currentTexture = self.desiredTexture
  self.desiredSetting = nil
  self.currentSetting = nil
  self.settingsOn = self:GetState("SettingsOn")
  self.menu = menu.Menu.New(self, {
    FooterButtonInfo = {
      {
        Item = "Select",
        Text = "[AdvanceButton] " .. util.GetLAMSMsg(lamsIDs.Select)
      },
      {
        Item = "Default",
        Text = "[TriangleButton] " .. util.GetLAMSMsg(lamsIDs.Default)
      },
      {
        Item = "Back",
        Text = "[BackButton] " .. util.GetLAMSMsg(lamsIDs.Back)
      }
    }
  })
  local GraphicsSettingsList = list.List.New(self, {
    MaxFocusableObjectCount = 20,
    ListObjectName = "MiddleSettingsList",
    EmptyTextLamsID = lamsIDs.NoSettingsAvailable,
    BackingAnimFrameCount = 60,
    HoverToSelect = true,
    Button_OnGainFocus = function(button)
      local currItem = button:get_item()
      self:SelectGraphicsSetting(currItem)
    end
  })
  self.menu:SetList("GraphicsSettings", GraphicsSettingsList)
  GraphicsSettingsList:SetSelectedButton(2, false)
  local disableText = util.GetLAMSMsg(lamsIDs.Disable)
  local minText = util.GetLAMSMsg(lamsIDs.Min)
  local lowText = util.GetLAMSMsg(lamsIDs.Low)
  local medText = util.GetLAMSMsg(lamsIDs.Original)
  local highText = util.GetLAMSMsg(lamsIDs.High)
  local ultraText = util.GetLAMSMsg(lamsIDs.Ultra)
  local ultraPlusText = util.GetLAMSMsg(lamsIDs.UltraPlus)
  self.listDisable = {
    {
      Value = 0,
      Text = disableText,
      Name = "Disable"
    },
    {
      Value = 1,
      Text = medText,
      Name = "Original"
    },
    {
      Value = 2,
      Text = highText,
      Name = "High"
    }
  }
  self.listMin = {
    {
      Value = 0,
      Text = minText,
      Name = "Min"
    },
    {
      Value = 1,
      Text = lowText,
      Name = "Low"
    },
    {
      Value = 2,
      Text = medText,
      Name = "Original"
    },
    {
      Value = 3,
      Text = highText,
      Name = "High"
    },
    {
      Value = 4,
      Text = ultraText,
      Name = "Ultra"
    }
  }
  self.listMed = {
    {
      Value = 0,
      Text = medText,
      Name = "Original"
    },
    {
      Value = 1,
      Text = highText,
      Name = "High"
    },
    {
      Value = 2,
      Text = ultraText,
      Name = "Ultra"
    }
  }
  self.listHigh = {
    {
      Value = 0,
      Text = lowText,
      Name = "Low"
    },
    {
      Value = 1,
      Text = medText,
      Name = "Original"
    },
    {
      Value = 2,
      Text = highText,
      Name = "High"
    }
  }
  self.listUltra = {
    {
      Value = 0,
      Text = lowText,
      Name = "Low"
    },
    {
      Value = 1,
      Text = medText,
      Name = "Original"
    },
    {
      Value = 2,
      Text = highText,
      Name = "High"
    },
    {
      Value = 3,
      Text = ultraText,
      Name = "Ultra"
    }
  }
  self.listUltraPlus = {
    {
      Value = 0,
      Text = disableText,
      Name = "Disable"
    },
    {
      Value = 1,
      Text = medText,
      Name = "Original"
    },
    {
      Value = 2,
      Text = highText,
      Name = "High"
    },
    {
      Value = 3,
      Text = ultraText,
      Name = "Ultra"
    },
    {
      Value = 4,
      Text = ultraPlusText,
      Name = "Plus"
    }
  }
  local customText = util.GetLAMSMsg(lamsIDs.Custom)
  self.presetToggle = toggle.Toggle.New(self, {
    Name = "Preset",
    Get = function()
      return game.GetVideoPreset()
    end,
    Set = function(settingValue)
      self.confirmed = game.SetVideoPreset(settingValue)
      self:RefreshGraphicsSettingsList()
      self.updateVRAM = 2
    end,
    Settings = {
      {
        Value = 0,
        Text = lowText,
        Name = "Low"
      },
      {
        Value = 1,
        Text = medText,
        Name = "Orginal"
      },
      {
        Value = 2,
        Text = highText,
        Name = "High"
      },
      {
        Value = 3,
        Text = ultraText,
        Name = "Ultra"
      },
      {Value = 4, Text = customText}
    },
    Default = function()
      return 1
    end
  })
  self.textureToggle = toggle.Toggle.New(self, {
    Name = "Texture",
    Get = function()
      return self.desiredTexture
    end,
    Set = function(settingValue)
      self.desiredTexture = settingValue
    end,
    Settings = self.listMin,
    Default = function()
      return 2
    end
  })
  self.textureToggle.hasPreset = true
  self.textureToggle.hasUltra = true
  self.modelToggle = toggle.Toggle.New(self, {
    Name = "Model",
    Get = function()
      return game.GetModelQuality()
    end,
    Set = function(settingValue)
      self.confirmed = game.SetModelQuality(settingValue)
      self:SelectGraphicsSetting(GraphicsSettingsList:GetSelectedItem())
      self.updateVRAM = 2
    end,
    Settings = self.listMed,
    Default = function()
      return 0
    end
  })
  self.modelToggle.hasPreset = true
  self.modelToggle.hasUltra = true
  self.filterToggle = toggle.Toggle.New(self, {
    Name = "Filter",
    Get = function()
      return game.GetFilterQuality()
    end,
    Set = function(settingValue)
      self.confirmed = game.SetFilterQuality(settingValue)
      self:SelectGraphicsSetting(GraphicsSettingsList:GetSelectedItem())
      self.updateVRAM = 2
    end,
    Settings = self.listMin,
    Default = function()
      return 2
    end
  })
  self.filterToggle.hasPreset = true
  self.filterToggle.hasUltra = true
  self.shadowToggle = toggle.Toggle.New(self, {
    Name = "Shadows",
    Get = function()
      return game.GetShadowQuality()
    end,
    Set = function(settingValue)
      self.confirmed = game.SetShadowQuality(settingValue)
      self:SelectGraphicsSetting(GraphicsSettingsList:GetSelectedItem())
      self.updateVRAM = 2
    end,
    Settings = self.listUltra,
    Default = function()
      return 1
    end
  })
  self.shadowToggle.hasPreset = true
  self.shadowToggle.hasUltra = true
  self.reflectionToggle = toggle.Toggle.New(self, {
    Name = "Reflections",
    Get = function()
      return game.GetReflectionQuality()
    end,
    Set = function(settingValue)
      self.confirmed = game.SetReflectionQuality(settingValue)
      self:SelectGraphicsSetting(GraphicsSettingsList:GetSelectedItem())
      self.updateVRAM = 2
    end,
    Settings = self.listUltraPlus,
    Default = function()
      return 1
    end
  })
  self.reflectionToggle.hasPreset = true
  self.reflectionToggle.hasUltra = true
  self.atmosphereToggle = toggle.Toggle.New(self, {
    Name = "Atmosphere",
    Get = function()
      return game.GetAtmosphereQuality()
    end,
    Set = function(settingValue)
      self.confirmed = game.SetAtmosphereQuality(settingValue)
      self:SelectGraphicsSetting(GraphicsSettingsList:GetSelectedItem())
      self.updateVRAM = 2
    end,
    Settings = self.listUltra,
    Default = function()
      return 1
    end
  })
  self.atmosphereToggle.hasPreset = true
  self.atmosphereToggle.hasUltra = true
  self.occlusionToggle = toggle.Toggle.New(self, {
    Name = "Occlusion",
    Get = function()
      return game.GetOcclusionQuality()
    end,
    Set = function(settingValue)
      self.confirmed = game.SetOcclusionQuality(settingValue)
      self:SelectGraphicsSetting(GraphicsSettingsList:GetSelectedItem())
      self.updateVRAM = 2
    end,
    Settings = self.listDisable,
    Default = function()
      return 1
    end
  })
  self.occlusionToggle.hasPreset = true
  self.occlusionToggle.hasUltra = false
  self.GraphicsSettingsInfo = {
    {
      Item = "Preset",
      Text = util.GetLAMSMsg(lamsIDs.Preset),
      ActivationList = {
        {
          Item = self.presetToggle
        }
      },
      ToolTip = util.GetLAMSMsg(lamsIDs.PresetDesc)
    },
    {
      Item = "Texture",
      Text = util.GetLAMSMsg(lamsIDs.TextureQuality),
      ActivationList = {
        {
          Item = self.textureToggle
        }
      },
      ToolTip = util.GetLAMSMsg(lamsIDs.TextureQualityDesc)
    },
    {
      Item = "Model",
      Text = util.GetLAMSMsg(lamsIDs.ModelQuality),
      ActivationList = {
        {
          Item = self.modelToggle
        }
      },
      ToolTip = util.GetLAMSMsg(lamsIDs.ModelQualityDesc)
    },
    {
      Item = "Filter",
      Text = util.GetLAMSMsg(lamsIDs.AnisotropicFilter),
      ActivationList = {
        {
          Item = self.filterToggle
        }
      },
      ToolTip = util.GetLAMSMsg(lamsIDs.AnisotropicFilterDesc)
    },
    {
      Item = "Shadows",
      Text = util.GetLAMSMsg(lamsIDs.Shadows),
      ActivationList = {
        {
          Item = self.shadowToggle
        }
      },
      ToolTip = util.GetLAMSMsg(lamsIDs.ShadowsDesc)
    },
    {
      Item = "Reflections",
      Text = util.GetLAMSMsg(lamsIDs.Reflections),
      ActivationList = {
        {
          Item = self.reflectionToggle
        }
      },
      ToolTip = util.GetLAMSMsg(lamsIDs.ReflectionsDesc)
    },
    {
      Item = "Atmosphere",
      Text = util.GetLAMSMsg(lamsIDs.Atmospherics),
      ActivationList = {
        {
          Item = self.atmosphereToggle
        }
      },
      ToolTip = util.GetLAMSMsg(lamsIDs.AtmosphericsDesc)
    },
    {
      Item = "Occlusion",
      Text = util.GetLAMSMsg(lamsIDs.AmbientOcclusion),
      ActivationList = {
        {
          Item = self.occlusionToggle
        }
      },
      ToolTip = util.GetLAMSMsg(lamsIDs.AmbientOcclusionDesc)
    }
  }
  self.goSettingsController = util.GetUiObjByName("SettingsController")
  self.goController = self.goSettingsController:FindSingleGOByName("Controller")
  self.controllerStreamedIn = true
  self.waitingForSwap = false
  self.waitingForStream = false
end
function GraphicsSettings:RefreshGraphicsSettingsList(disabled)
  if disabled == nil then
    disabled = game.GetVideoPreset() ~= 4
  end
  local settingsList = self.menu:GetList("GraphicsSettings")
  settingsList:Deactivate()
  self.settingsOn:RefreshSettingsList(settingsList, self.GraphicsSettingsInfo, util.GetLAMSMsg(lamsIDs.Graphics))
  for i = 1, #self.GraphicsSettingsInfo do
    if self.GraphicsSettingsInfo[i].ActivationList ~= nil then
      local toggle = self.GraphicsSettingsInfo[i].ActivationList[1].Item
      if toggle.hasPreset ~= nil and toggle.hasPreset then
        local setting_button = settingsList:GetButton(i)
        local newList
        if toggle == self.modelToggle then
          newList = self.listMed
        elseif toggle == self.textureToggle or toggle == self.filterToggle then
          newList = self.listMin
        elseif toggle == self.occlusionToggle then
          newList = self.listDisable
        elseif toggle == self.reflectionToggle then
          newList = self.listUltraPlus
        elseif toggle.hasUltra ~= nil and toggle.hasUltra then
          newList = self.listUltra
        else
          newList = self.listHigh
        end
        if disabled then
          local preset = game.GetVideoPreset()
          setting_button:SetTextColor(colors.Text_Settings_Disabled, "medium_text_settings1")
          if toggle == self.modelToggle then
            newList = {
              newList[math.max(preset, 1)]
            }
          elseif toggle == self.textureToggle or toggle == self.filterToggle then
            newList = {
              newList[preset + 2]
            }
          elseif toggle.hasUltra ~= nil and toggle.hasUltra then
            newList = {
              newList[preset + 1]
            }
          else
            newList = {
              newList[math.min(preset + 1, 3)]
            }
          end
          toggle:DisableText()
        else
          setting_button:SetTextColor(colors.Text_Settings1, "medium_text_settings1")
          toggle:EnableText()
        end
        toggle:OverrideSettings(newList)
      end
    end
  end
end
function GraphicsSettings:Enter()
  self.handling_event = false
  self.confirmed = true
  self.updateVRAM = 0
  self.desiredTexture = game.GetTextureQuality()
  self.currentTexture = self.desiredTexture
  self.desiredSetting = nil
  self.currentSetting = nil
  self:SetAllHandlersFalse()
  self.menu:Activate()
  self:RefreshGraphicsSettingsList()
  self:UpdateFooterButtonPrompt(true, self.confirmed)
  Audio.PlaySound("SND_UX_Menu_Tick_Confirm")
  self.availableVRAM = game.GetAvailableVRAM()
  self.usedVRAM = game.GetUsedVRAM()
  self.goSettingsController:Show()
  UI.AnimNoShow(self.goController, consts.AS_Forward, "", 0, 0)
  self.controllerStreamedIn = true
  self.waitingForSwap = false
  self.waitingForStream = false
  UI.AlphaFade(self.goController, 0.001, 0)
  self.goController:Show()
  local GraphicsSettingsList = self.menu:GetList("GraphicsSettings")
  self:SelectGraphicsSetting(GraphicsSettingsList:GetSelectedItem())
end
function GraphicsSettings:UpdateSetting(value, toggle)
  local currValueIndex = 1
  for buttonIndex, button in ipairs(toggle._buttonArray) do
    local setting = button:get_item()
    local settingValue = setting.Value
    if settingValue == value then
      currValueIndex = buttonIndex
      break
    end
  end
  toggle:SetSelectedButton(currValueIndex, false)
end
function GraphicsSettings:EVT_UPDATE_WINDOW_SIZE()
  self.updateVRAM = 1
end
function GraphicsSettings:Exit()
  self:SendEventToUIFsm("globalMenu", "EVT_SET_FOOTER_INFO_TEXT", "")
  self.goSettingsController:Hide()
  self.goController:Hide()
  UI.AlphaFade(self.goController, 1, 0)
  self:RefreshGraphicsSettingsList(false)
  self.menu:Deactivate(true)
  Audio.PlaySound("SND_UX_Menu_Tick_Back")
end
function GraphicsSettings:SetDefaultValues()
  local GraphicsSettingsList = self.menu:GetList("GraphicsSettings")
  self.settingsOn:SetDefaultValues(GraphicsSettingsList, self.menu, function()
    self:Enter()
  end)
end
function GraphicsSettings:HandleAdvanceOnList()
  local optionsList = self.menu:GetList("GraphicsSettings")
  local button = optionsList:GetSelectedButton()
  local advance_event = button:get_advance()
  if advance_event ~= nil then
    advance_event()
  else
    self.handling_event = false
  end
end
function GraphicsSettings:UpdateFooterButtonPrompt(advance, confirmed)
  self.menu:UpdateFooterButton("Select", advance)
  self.menu:UpdateFooterButton("Confirm", not confirmed)
  self.menu:UpdateFooterButton("Default", true)
  self.menu:UpdateFooterButton("Back", true)
  self.menu:UpdateFooterButton("Close", true)
  self.menu:UpdateFooterButtonText()
end
function GraphicsSettings:EVT_Advance_Release()
  if not self.handling_event then
    self.advance = true
  end
end
function GraphicsSettings:EVT_MOUSE_CLICKED()
  if not self.handling_event then
    self.advance = true
  end
end
function GraphicsSettings:Back()
  game.RevertGraphicsSettings()
  self.presetToggle:ForceSetValue(game.GetVideoPreset())
  self.resolutionsToggle:ForceSetValue(game.GetResolutionIndex())
  self.aspectRatiosToggle:ForceSetValue(game.GetAspectRatioIndex())
  self.textureToggle:ForceSetValue(game.GetTextureQuality())
  self.modelToggle:ForceSetValue(game.GetModelQuality())
  self.filterToggle:ForceSetValue(game.GetFilterQuality())
  self.shadowToggle:ForceSetValue(game.GetShadowQuality())
  self.reflectionToggle:ForceSetValue(game.GetReflectionQuality())
  self.atmosphereToggle:ForceSetValue(game.GetAtmosphereQuality())
  self.occlusionToggle:ForceSetValue(game.GetOcclusionQuality())
  self.monitorToggle:ForceSetValue(game.GetMonitor())
  self.displayModeToggle:ForceSetValue(game.GetDisplayMode())
  self.vSyncToggle:ForceSetValue(game.GetVSync())
  local settingsList = self.menu:GetList("GraphicsSettings")
  settingsList:Deactivate()
  self.settingsOn:RefreshSettingsList(settingsList, self.GraphicsSettingsInfo, util.GetLAMSMsg(lamsIDs.Video))
end
function GraphicsSettings:EVT_MOUSE_RIGHT_CLICKED()
  if not self.handling_event then
    self.back = true
  end
end
function GraphicsSettings:EVT_Back_Release()
  if not self.handling_event then
    self.back = true
  end
end
function GraphicsSettings:EVT_Options_Release()
  if not self.handling_event then
    self.back_options = true
  end
end
function GraphicsSettings:EVT_Triangle_Release()
  if not self.handling_event then
    self.triangle = true
  end
end
function GraphicsSettings:SetAllHandlersFalse()
  self.back = false
  self.back_options = false
  self.triangle = false
  self.advance = false
end
function GraphicsSettings:SelectGraphicsSetting(currItem)
  if currItem then
    local toggle = currItem.ActivationList[1].Item
    if toggle == self.presetToggle then
      self.desiredSetting = nil
    else
      local settings = toggle:get_settings()
      local value
      if toggle.hasPreset and game.GetVideoPreset() ~= 4 then
        value = 0
      else
        value = toggle._Get()
      end
      local name = settings[value + 1].Name
      if name then
        self.desiredSetting = currItem.Item .. name
      else
        self.desiredSetting = nil
      end
    end
  end
  self:UpdateFooterButtonPrompt()
end
function GraphicsSettings:StartMaterialSwap()
  self.goController:SetMaterialSwap(self.currentSetting)
  self.goController:RequestLoadHighMips(true)
  self.waitingForStream = true
  self:StartTimer("Stream", 0.03, function()
    self.waitingForStream = false
  end)
  self.controllerStreamedIn = false
end
function GraphicsSettings:Update()
  if self.currentTexture ~= self.desiredTexture then
    self.confirmed = game.SetTextureQuality(self.desiredTexture)
    local GraphicsSettingsList = self.menu:GetList("GraphicsSettings")
    self:SelectGraphicsSetting(GraphicsSettingsList:GetSelectedItem())
    self.updateVRAM = 2
    self.currentTexture = self.desiredTexture
  end
  if self.currentSetting ~= self.desiredSetting and not self.waitingForSwap then
    self.currentSetting = self.desiredSetting
    if self.currentSetting then
      self.waitingForSwap = true
      self:StartTimer("Swap", 0.03, function()
        self:StartMaterialSwap()
      end)
    end
    UI.AlphaFade(self.goController, 0.001, 0)
  end
  if not self.controllerStreamedIn and not self.waitingForStream and self.goController:CheckMaterialFullyStreamed() then
    if self.currentSetting == self.desiredSetting then
      UI.AlphaFade(self.goController, 1, 0)
    end
    self.waitingForSwap = false
    self.controllerStreamedIn = true
  end
  if self.updateVRAM > 0 then
    self.updateVRAM = self.updateVRAM - 1
    if self.updateVRAM == 0 then
      self.usedVRAM = game.GetUsedVRAM()
      self:UpdateFooterButtonPrompt(false, self.confirmed)
    end
  end
  local text = util.GetLAMSMsg(lamsIDs.VramUsed) .. " " .. self.usedVRAM .. " / " .. self.availableVRAM .. "MB"
  self:SendEventToUIFsm("globalMenu", "EVT_SET_FOOTER_INFO_TEXT", text)
  if not self.handling_event then
    if self.back then
      self.handling_event = true
      self:SetAllHandlersFalse()
      self:Goto("MainSettings")
    elseif self.back_options then
      self.handling_event = true
      self:SetAllHandlersFalse()
      self.settingsOn:OptionsReleaseHandler()
    elseif self.triangle then
      self.handling_event = true
      self:SetAllHandlersFalse()
      self:SetDefaultValues()
    elseif self.advance then
      self.handling_event = true
      self:SetAllHandlersFalse()
      self:HandleAdvanceOnList()
    end
  end
end
function LanguageSettings:Setup()
  if game.CHECK_FEATURE("GoldPrint") then
    engine.GoldPrint("INT_85097: called LanguageSettings:Setup()\n" .. debug.traceback())
  end
  self.leaving = false
  self.currentLanguageSlot = -1
  self.originalLanguageSlot = -1
  self.settingsOn = self:GetState("SettingsOn")
  self.menu = menu.Menu.New(self, {
    FooterButtonInfo = {
      {
        Item = "Download",
        Text = "[AdvanceButton] " .. util.GetLAMSMsg(lamsIDs.Download),
        EventHandlers = {
          {
            Events = {
              "EVT_Advance_Release"
            },
            Handler = function()
              self:DownloadLanguage()
            end
          }
        }
      },
      {
        Item = "StopDownload",
        Text = "[AdvanceButton] " .. util.GetLAMSMsg(lamsIDs.StopDownload),
        EventHandlers = {
          {
            Events = {
              "EVT_Advance_Release"
            },
            Handler = function()
              self:StopDownloadLanguage()
            end
          }
        }
      },
      {
        Item = "Back",
        Text = "[BackButton] " .. util.GetLAMSMsg(lamsIDs.Back),
        EventHandlers = {
          {
            Events = {
              "EVT_Back_Release",
              "EVT_MOUSE_RIGHT_CLICKED"
            },
            Handler = function()
              self:HandleBackButton()
            end
          }
        }
      },
      {
        Item = "Close",
        Text = "",
        EventHandlers = {
          {
            Events = {
              "EVT_Options_Release"
            },
            Handler = function()
              self:HandleOptionsButton()
            end
          }
        }
      }
    }
  })
  local languageSettingsList = list.List.New(self, {
    MaxFocusableObjectCount = 20,
    ListObjectName = "MiddleSettingsList",
    EmptyTextLamsID = lamsIDs.NoSettingsAvailable,
    BackingAnimFrameCount = 60,
    HoverToSelect = true,
    Button_OnGainFocus = function(button)
      local currItem = button:get_item()
      self:SelectLanguageSetting(currItem)
    end
  })
  self.menu:SetList("languageSettings", languageSettingsList)
  self.settingsOn = self:GetState("SettingsOn")
  self.TextLanguageToggle = toggle.Toggle.New(self, {
    Name = "TextLanguageToggle",
    Get = function()
      return UI.GetTextLanguage()
    end,
    Set = function(value)
      UI.SetTextLanguage(value)
    end,
    Settings = self:GetAvailableLanguageOptions(UI.AvailableTextLanguages)
  })
  self.SpeechLanguageToggle = toggle.Toggle.New(self, {
    Name = "SpeechLanguageToggle",
    Get = function()
      return UI.GetSpeechLanguage()
    end,
    Set = function(value)
      self:TrySetSpeechLanguage(value)
    end,
    Settings = self:GetAvailableLanguageOptions(UI.AvailableSpeechLanguages),
    OnToggleFunc = function(num)
      self:OnSpeechLanguageToggle(num)
    end
  })
  self.subtitlesToggle = toggle.Toggle.New(self, {
    Name = "Subtitles",
    Get = UI.GetSubtitles,
    Set = function(num)
      UI.SetSubtitles(num)
    end
  })
  self.languageSettingsInfo = {
    {
      Item = "LanguageSetting",
      Text = util.GetLAMSMsg(lamsIDs.TextLanguage),
      ActivationList = {
        {
          Item = self.TextLanguageToggle
        }
      }
    },
    {
      Item = "SpeechLanguage",
      Text = util.GetLAMSMsg(lamsIDs.SpeechLanguage),
      ActivationList = {
        {
          Item = self.SpeechLanguageToggle
        }
      }
    },
    {
      Item = "Subtitles",
      Text = util.GetLAMSMsg(lamsIDs.Subtitles),
      ActivationList = {
        {
          Item = self.subtitlesToggle
        }
      }
    }
  }
end
function LanguageSettings:Enter()
  if game.CHECK_FEATURE("GoldPrint") then
    engine.GoldPrint("INT_85097: called LanguageSettings:Enter()\n" .. debug.traceback())
  end
  self.currentLanguageSlot = UI.GetSpeechLanguage()
  self.originalLanguageSlot = self.currentLanguageSlot
  self.leaving = false
  self.language_button = false
  self.curr_item = nil
  self.menu:Activate()
  local languageList = self.menu:GetList("languageSettings")
  self.settingsOn:RefreshSettingsList(languageList, self.languageSettingsInfo, util.GetLAMSMsg(lamsIDs.Language))
  Audio.PlaySound("SND_UX_Menu_Tick_Confirm")
end
function LanguageSettings:Exit()
  if game.CHECK_FEATURE("GoldPrint") then
    engine.GoldPrint("INT_85097: called LanguageSettings:Exit()\n" .. debug.traceback())
  end
  self.leaving = false
  self.menu:ForceDeactivate()
  self:TurnOffProgressDialog()
  Audio.PlaySound("SND_UX_Menu_Tick_Back")
end
function LanguageSettings:Update()
  if not self.leaving then
    if not self.language_button then
      self:OnSpeechLanguageToggle(self.currentLanguageSlot)
    else
      self:SelectLanguageSetting(self.curr_item)
    end
  end
end
function LanguageSettings:SetupDownloadProgressDialog(languageIndex, status)
  self:SendEventUI("EVT_SHOW_SETTINGS_DIALOG_MESSAGE", {
    Status = util.GetLAMSMsg(lamsIDs[status]),
    Housing = false,
    Progress = {
      Get = function()
        return UI.GetSpeechLanguagePlayGoDownloadProgress(languageIndex)
      end,
      IsComplete = function()
        return UI.IsSpeechLanguagePlayGoDownloaded(languageIndex)
      end
    },
    Language = true
  })
end
function LanguageSettings:TurnOffProgressDialog()
  self:SendEventUI("EVT_IN_GAME_TURNOFF_PLAYGO_PROGRESS")
end
function LanguageSettings:OnSpeechLanguageToggle(languageIndex)
  local download = false
  local downloading = false
  if UI.IsSpeechLanguagePlayGoDownloaded(languageIndex) then
    if languageIndex ~= UI.GetSpeechLanguage() then
      self:TrySetSpeechLanguage(languageIndex)
    end
    self:TurnOffProgressDialog()
    self.SpeechLanguageToggle:EnableText()
  else
    local downloadStatus
    if UI.IsSpeechLanguagePlayGoDownloading(languageIndex) then
      downloading = true
      downloadStatus = "Downloading"
    elseif UI.IsSpeechLanguagePlayGoRequired(languageIndex) then
      downloading = true
      downloadStatus = "Queued"
    else
      download = true
      downloadStatus = "NotDownloading"
    end
    self.SpeechLanguageToggle:DisableText()
    self:SetupDownloadProgressDialog(languageIndex, downloadStatus)
  end
  self:UpdateFooterButtonPrompt(download, downloading)
end
function LanguageSettings:SelectLanguageSetting(currItem)
  self.curr_item = currItem
  local download = false
  local stopDownload = false
  if currItem ~= nil and currItem.Item == "SpeechLanguage" then
    self.language_button = true
    self:OnSpeechLanguageToggle(self.currentLanguageSlot)
  else
    self:TurnOffProgressDialog()
    self.language_button = false
    self:UpdateFooterButtonPrompt(download, stopDownload)
  end
end
function LanguageSettings:UpdateFooterButtonPrompt(download, stopDownload)
  self.menu:UpdateFooterButton("Download", download)
  self.menu:UpdateFooterButton("StopDownload", stopDownload)
  self.menu:UpdateFooterButton("Back", true)
  self.menu:UpdateFooterButton("Close", true)
  self.menu:UpdateFooterButtonText()
end
function LanguageSettings:TrySetSpeechLanguage(languageIndex)
  if UI.IsSpeechLanguagePlayGoDownloaded(languageIndex) then
    UI.SetSpeechLanguage(languageIndex)
  end
  self.currentLanguageSlot = languageIndex
end
function LanguageSettings:DownloadLanguage()
  if not self.leaving then
    UI.SetSpeechLanguagePlayGoRequired(self.currentLanguageSlot, true)
    self:OnSpeechLanguageToggle(self.currentLanguageSlot)
  end
end
function LanguageSettings:StopDownloadLanguage()
  if not self.leaving then
    UI.SetSpeechLanguagePlayGoRequired(self.currentLanguageSlot, false)
    self:OnSpeechLanguageToggle(self.currentLanguageSlot)
  end
end
function LanguageSettings:HandleOptionsButton()
  self:HandleLeaving(false)
end
function LanguageSettings:HandleBackButton()
  self:HandleLeaving(true)
end
function LanguageSettings:DialogLeavingPrompt(backButton)
  self.leaving = true
  local language = self:GetLanguageFromAvailableSpeechLanguages(self.originalLanguageSlot)
  local languageList = self.menu:GetList("languageSettings")
  local currItem = languageList:GetSelectedItem()
  self.curr_item = currItem
  self:SendEventUI("EVT_SHOW_SETTINGS_DIALOG_MESSAGE", {
    Header = util.GetLAMSMsg(lamsIDs.LanguageDownloadInProgress),
    Body = util.GetLAMSMsg(lamsIDs.LanguageCantSwitchText) .. "\n" .. language,
    ActiveMenus = {
      self.menu
    },
    ActiveLists = {
      self.menu:GetList("languageSettings")
    },
    Events = {
      Advance = {
        Text = util.GetLAMSMsg(lamsIDs.Confirm),
        Handler = function()
          self:ReactivateThenLeave(backButton)
        end
      },
      Back = {
        Text = util.GetLAMSMsg(lamsIDs.Back),
        Handler = function()
          self:ReactivateThenEnter()
        end
      }
    }
  })
end
function LanguageSettings:GetLanguageFromAvailableSpeechLanguages(slot)
  local lang_idxs = UI.AvailableSpeechLanguages()
  for i, code in ipairs(lang_idxs) do
    if i == slot then
      return util.GetLAMSMsg(lamsIDs[consts.LanguageCodes[code + 1]])
    end
  end
end
function LanguageSettings:HandleLeaving(backButton)
  self.leaving = true
  if self.currentLanguageSlot == self.originalLanguageSlot then
    self:LeaveMenu(backButton)
  elseif not UI.IsSpeechLanguagePlayGoDownloaded(self.currentLanguageSlot) then
    self:TurnOffProgressDialog()
    self:DialogLeavingPrompt(backButton)
  else
    self:LeaveMenu(backButton)
  end
end
function LanguageSettings:ReactivateThenEnter()
  if self.originalLanguageSlot ~= UI.GetSpeechLanguage() then
    UI.SetSpeechLanguage(self.originalLanguageSlot)
  end
  self:Enter()
end
function LanguageSettings:ReactivateThenLeave(backButton)
  self.menu:UpdateFooterButtons(false, false)
  if self.originalLanguageSlot ~= UI.GetSpeechLanguage() then
    UI.SetSpeechLanguage(self.originalLanguageSlot)
  end
  self:LeaveMenu(backButton)
end
function LanguageSettings:LeaveMenu(backButton)
  if backButton then
    if self:IsActive() then
      self:Goto("MainSettings")
    else
      self:Exit()
    end
  else
    self.settingsOn:OptionsReleaseHandler()
  end
end
function LanguageSettings:GetAvailableLanguageOptions(getTextFunction)
  local languages = {}
  local lang_idxs = getTextFunction()
  for i, code in ipairs(lang_idxs) do
    local temp = {
      Value = i,
      Text = util.GetLAMSMsg(lamsIDs[consts.LanguageCodes[code + 1]])
    }
    tablex.FastInsert(languages, temp, #languages + 1)
  end
  return languages
end
function Credits:Setup()
  self.settingsOn = self:GetState("SettingsOn")
  self.now_scrolling = false
  self.ScrollStartLines = 6
  self.ScrollMaxLines = 32
  self.ScrollFramesPerLine = 20
  self.credits_text = UI.GetTeamCredits()
  self.sieStart, self.sieEnd, self.num_lines = UI.FindTagsAndCountNewLines(self.credits_text, [[

<H1>Sony Interactive Entertainment</>]], [[

<H1>Santa Monica Studio</>]])
  self.sieStart = (self.sieStart - 4) * self.ScrollFramesPerLine
  self.sieEnd = (self.sieEnd - 4) * self.ScrollFramesPerLine
  self.num_lines = self.num_lines + 22
  self.system_menu = false
  self.settingsOn = self:GetState("SettingsOn")
  self.menu = menu.Menu.New(self, {
    FooterButtonInfo = {
      {
        Item = "Back",
        Text = "[BackButton] " .. util.GetLAMSMsg(lamsIDs.Back),
        EventHandlers = {
          {
            Events = {
              "EVT_Back_Release",
              "EVT_MOUSE_RIGHT_CLICKED"
            },
            Handler = function()
              self:Goto("MainSettings")
            end
          }
        }
      },
      {
        Item = "Close",
        Text = "",
        EventHandlers = {
          {
            Events = {
              "EVT_Options_Release"
            },
            Handler = function()
              self.settingsOn:OptionsReleaseHandler()
            end
          }
        }
      }
    }
  })
end
function Credits:Enter()
  self.system_menu = false
  self.menu:Activate()
  self:UpdateFooterButtonPrompt()
  self.creditScrollingObj = util.GetUiObjByName("Credit_Scrolling")
  self.th_creditScrolling = util.GetTextHandle(self.creditScrollingObj)
  UI.AlphaFade(self.creditScrollingObj, 0, 0)
  self.creditScrollingObj:Show()
  self.now_scrolling = true
  self.nScrollingFrames = 0
  self.up_fast = false
  self.down_fast = false
  UI.ScissorRect(0, 1, 0.15, 0.85)
  UI.SetTextScroll(self.th_creditScrolling, self.ScrollStartLines - self.ScrollMaxLines, self.ScrollMaxLines)
  UI.SetText(self.th_creditScrolling, self.credits_text)
  UI.AlphaFade(self.creditScrollingObj, 1, 1)
  game.Audio.PlaySound("SND_UX_Pause_Menu_Settings_Credits_Enter")
  game.Audio.StartMusic("SND_MX_credits_roll_02_in")
  game.Audio.SetWwiseState("Credits", "in")
  Audio.PlaySound("SND_UX_Menu_Tick_Confirm")
end
function Credits:Update()
  local frameInc = UI.GetFrameTime() * 30
  if self.nScrollingFrames > self.sieStart and self.nScrollingFrames < self.sieEnd then
    frameInc = frameInc * 2
  end
  if self.now_scrolling and not self.system_menu then
    if UI.IsPlatinum() then
      self.nScrollingFrames = self.nScrollingFrames + frameInc
    elseif self.up_fast then
      self.nScrollingFrames = self.nScrollingFrames + 16 * frameInc
    elseif self.down_fast then
      self.nScrollingFrames = self.nScrollingFrames - 16 * frameInc
    else
      self.nScrollingFrames = self.nScrollingFrames + frameInc
    end
    local first_line = self.nScrollingFrames / self.ScrollFramesPerLine
    local iFirst_line = math.floor(first_line)
    UI.SetTextScroll(self.th_creditScrolling, iFirst_line + self.ScrollStartLines - self.ScrollMaxLines, self.ScrollMaxLines, first_line - iFirst_line)
    if iFirst_line > self.num_lines then
      self.now_scrolling = false
      UI.AlphaFade(self.creditScrollingObj, 0, 1)
      self:Goto("MainSettings")
    end
  end
end
function Credits:EVT_SYSTEM_MENU_ENTER()
  self.system_menu = true
end
function Credits:EVT_SYSTEM_MENU_EXIT()
  self.system_menu = false
  self.up_fast = false
  self.down_fast = false
end
function Credits:EVT_Up_Press()
  self.up_fast = true
end
function Credits:EVT_Up_Release()
  self.up_fast = false
end
function Credits:EVT_Down_Press()
  self.down_fast = true
end
function Credits:EVT_Down_Release()
  self.down_fast = false
end
function Credits:UpdateFooterButtonPrompt()
  self.menu:UpdateFooterButton("Back", true)
  self.menu:UpdateFooterButton("Close", true)
  self.menu:UpdateFooterButtonText()
end
function Credits:Exit()
  UI.ScissorRect(0, 1, 0, 1)
  self.menu:Deactivate(true)
  self.creditScrollingObj:Hide()
  UI.AlphaFade(self.creditScrollingObj, 0, 1)
  game.Audio.PlaySound("SND_UX_Pause_Menu_Settings_Credits_Exit")
  game.Audio.StartMusic("SND_MX_credits_roll_02_out")
  game.Audio.SetWwiseState("Credits", "out")
  Audio.PlaySound("SND_UX_Menu_Tick_Back")
end
function Bookmarks:Setup()
  self.settingsOn = self:GetState("SettingsOn")
  self.menu = menu.Menu.New(self, {
    FooterButtonInfo = {
      {
        Item = "Select",
        Text = "[AdvanceButton] " .. util.GetLAMSMsg(lamsIDs.Select)
      },
      {
        Item = "Back",
        Text = "[BackButton] " .. util.GetLAMSMsg(lamsIDs.Back),
        EventHandlers = {
          {
            Events = {
              "EVT_Back_Release",
              "EVT_MOUSE_RIGHT_CLICKED"
            },
            Handler = function()
              self:Goto("Options")
            end
          }
        }
      },
      {
        Item = "Close",
        Text = "",
        EventHandlers = {
          {
            Events = {
              "EVT_Options_Release"
            },
            Handler = function()
              self.settingsOn:OptionsReleaseHandler()
            end
          }
        }
      }
    }
  })
  local bookmarksList = list.List.New(self, {
    MaxFocusableObjectCount = 10,
    ListObjectName = "SettingsList",
    PadRepeatRate = 8,
    EmptyTextLamsID = lamsIDs.NoBookmarksAvailable,
    BackingAnimFrameCount = 60,
    HoverToSelect = true,
    EventHandlers = {
      {
        Events = {
          "EVT_Advance_Release"
        },
        Handler = function()
          self:Bookmarks_AdvanceReleaseHandler()
        end
      }
    },
    PageUpEvents = {
      "EVT_L2_Press"
    },
    PageDownEvents = {
      "EVT_R2_Press"
    }
  })
  self.menu:SetList("bookmarks", bookmarksList)
end
function Bookmarks:Enter()
  self.menu:Activate()
  local bookmarksList = self.menu:GetList("bookmarks")
  local bookmarks = {
    "Select WAD",
	"Unlock All Trophies"
  }
  self.settingsOn:UpdateWadList(bookmarksList, bookmarks)
  self:UpdateFooterButtonPrompt()
end
function Bookmarks:Exit()
  self.menu:Deactivate(true)
end
function Bookmarks:UnlockAllTrophies()
	for i = 50,1,-1 do 
		game.UnlockTrophy(i)
	end
end
function Bookmarks:Bookmarks_AdvanceReleaseHandler()
  local bookmark = self.menu:GetList("bookmarks"):GetSelectedItem()
  if bookmark == "Unlock All Trophies" then
    local bookmarkList = self.menu:GetList("bookmarks")
	self:SendEventUI("EVT_SHOW_SETTINGS_DIALOG_MESSAGE", {
    Header = "Unlock All Trophies",
    Body = "Are you sure? This action cannot be reversed.",
    ActiveMenus = {
      self.menu
    },
    ActiveLists = {bookmarkList},
    Events = {
      Advance = {
        Text = util.GetLAMSMsg(lamsIDs.Confirm),
        Handler = function()
          self:UnlockAllTrophies()
          self:Enter()
        end
      },
      Back = {
        Text = util.GetLAMSMsg(lamsIDs.Back),
        Handler = function()
          self:Enter()
        end
      }
    }
  })
  else
	self:Goto(bookmark)
  end
end
function Bookmarks:UpdateFooterButtonPrompt()
  self.menu:UpdateFooterButton("Select", true)
  self.menu:UpdateFooterButton("Back", true)
  self.menu:UpdateFooterButton("Close", true)
  self.menu:UpdateFooterButtonText()
end
function WadSelect:Setup()
  self.settingsOn = self:GetState("SettingsOn")
  self.menu = menu.Menu.New(self, {
    FooterButtonInfo = {
      {
        Item = "Select",
        Text = "[AdvanceButton] " .. util.GetLAMSMsg(lamsIDs.Select)
      },
      {
        Item = "Back",
        Text = "[BackButton] " .. util.GetLAMSMsg(lamsIDs.Back),
        EventHandlers = {
          {
            Events = {
              "EVT_Back_Release",
              "EVT_MOUSE_RIGHT_CLICKED"
            },
            Handler = function()
              self:Goto("Bookmarks")
            end
          }
        }
      },
      {
        Item = "Close",
        Text = "",
        EventHandlers = {
          {
            Events = {
              "EVT_Options_Release"
            },
            Handler = function()
              self.settingsOn:OptionsReleaseHandler()
            end
          }
        }
      }
    }
  })
  local wadsList = list.List.New(self, {
    MaxFocusableObjectCount = 10,
    ListObjectName = "SettingsList",
    PadRepeatRate = 8,
    EmptyTextLamsID = lamsIDs.NoWadsAvailable,
    BackingAnimFrameCount = 60,
    HoverToSelect = true,
    EventHandlers = {
      {
        Events = {
          "EVT_Advance_Release"
        },
        Handler = function()
          self:Wads_AdvanceReleaseHandler()
        end
      }
    },
    PageUpEvents = {
      "EVT_L2_Press"
    },
    PageDownEvents = {
      "EVT_R2_Press"
    }
  })
  self.menu:SetList("wads", wadsList)
end
function WadSelect:Enter()
  self.menu:Activate()
  local wadsList = self.menu:GetList("wads")
  self:UpdateWadList(wadsList)
  self:UpdateFooterButtonPrompt()
end
function WadSelect:Exit()
  self.menu:Deactivate(true)
end
function WadSelect:UpdateWadList(wadsList)
  self.settingsOn:UpdateWadList(wadsList, engine.GetAvailableWads())
end
function WadSelect:Wads_AdvanceReleaseHandler()
  local wad = self.menu:GetList("wads"):GetSelectedItem()
  self.settingsOn:OptionsReleaseHandler()
  game.QuitAndJumpToWad(wad)
end
function WadSelect:UpdateFooterButtonPrompt()
  self.menu:UpdateFooterButton("Select", true)
  self.menu:UpdateFooterButton("Back", true)
  self.menu:UpdateFooterButton("Close", true)
  self.menu:UpdateFooterButtonText()
end
function LevelSelect:Setup()
  self.settingsOn = self:GetState("SettingsOn")
  self.menu = menu.Menu.New(self, {
    FooterButtonInfo = {
      {
        Item = "Select",
        Text = "[AdvanceButton] " .. util.GetLAMSMsg(lamsIDs.Select)
      },
      {
        Item = "Back",
        Text = "[BackButton] " .. util.GetLAMSMsg(lamsIDs.Back),
        EventHandlers = {
          {
            Events = {
              "EVT_Back_Release",
              "EVT_MOUSE_RIGHT_CLICKED"
            },
            Handler = function()
              self:Goto("Bookmarks")
            end
          }
        }
      },
      {
        Item = "Close",
        Text = "",
        EventHandlers = {
          {
            Events = {
              "EVT_Options_Release"
            },
            Handler = function()
              self.settingsOn:OptionsReleaseHandler()
            end
          }
        }
      }
    }
  })
  local wadsList = list.List.New(self, {
    MaxFocusableObjectCount = 10,
    ListObjectName = "SettingsList",
    PadRepeatRate = 8,
    EmptyTextLamsID = lamsIDs.NoWadsAvailable,
    BackingAnimFrameCount = 60,
    HoverToSelect = true,
    EventHandlers = {
      {
        Events = {
          "EVT_Advance_Release"
        },
        Handler = function()
          self:Wads_AdvanceReleaseHandler()
        end
      }
    },
    PageUpEvents = {
      "EVT_L2_Press"
    },
    PageDownEvents = {
      "EVT_R2_Press"
    }
  })
  self.menu:SetList("wads", wadsList)
  self.defaultLevelWads = {
    "xpl970_beachtower",
    "xpl980_beachwaterfall",
    "alf100_hengeroom",
    "alf111_mainlightspre",
    "alf112_mainlightspost",
    "alf150_bridgedark",
    "alf151_bridgelightspre",
    "alf152_bridgelightspost",
    "alf200_mazedark",
    "alf201_mazelightspre",
    "alf202_mazelightspost",
    "alf210_lakedarklh",
    "alf211_lakelhlightspre",
    "alf211_mainlightspre",
    "alf212_lakelhlightspost",
    "alf212_mainlightspost",
    "alf250_lakescripting",
    "alf300_lakedark",
    "alf301_lakelightspre",
    "alf302_lakelightspost",
    "alf320_trenchadark",
    "alf321_trenchalightspre",
    "alf322_trenchalightspost",
    "alf325_trenchadarktwr",
    "alf340_c_prisoners",
    "alf340_trenchbdark",
    "alf341_trenchopenlights",
    "alf342_trenchclosedlights",
    "alf345_trenchbdarktwr",
    "alf350_chiselsite",
    "alf355_chiseldungeon",
    "alf370_moatscripting",
    "alf371_moatlightspre",
    "alf372_moatlightspost",
    "alf390_templeextlh",
    "alf440_hiveextlh",
    "alf450_hiveext",
    "alf490_hiveintlh",
    "alf500_hiveint",
    "alf510_hiveintcine",
    "alf550_insidelight",
    "alf551_ilfirst",
    "alf552_ilsecond",
    "alf553_ilthird",
    "alf554_ilfourth",
    "alf555_ilfifth",
    "alf600_templeint",
    "alf690_lakelightlh",
    "alfdgn100_ent",
    "alfdgn110_main",
    "alfdgn200_ent",
    "alfdgn210_main",
    "cal080_global",
    "cal090_waterlevelzero",
    "cal100_hub",
    "cal110_bridgetop",
    "cal120_c_eyesontheprize",
    "cal130_templeint",
    "cal140_templeext",
    "cal145_faketempleinterior",
    "cal170_library1",
    "cal200_shore",
    "cal210_snakehead",
    "cal215_snakehorncine1",
    "cal225_riverpasslh",
    "cal240_peaksvista",
    "cal250_foothillslh",
    "cal270_stonemasonlh",
    "cal275_stonemasonlh",
    "cal300_jotunheimlh",
    "cal400_finaleworldtravel",
    "cal410_finaletowerfight",
    "cal500_runevault",
    "cal510_runevaulthalls",
    "cal512_runevaulthalls2",
    "cal520_mainroomwalls",
    "cal580_encounters",
    "cal590_c365_modiambush",
    "cal590_c440_demigodtoast",
    "cal590_runevaultelevator",
    "cal700_templeflip",
    "cal720_lowercenterroom",
    "cal740_leftwing",
    "cal750_rightwing",
    "cal780_encounters",
    "cal800_snakebelly",
    "cal820_snakebellymid",
    "foot095_frontdoor",
    "foot100_base",
    "foot110_cs185",
    "foot120_cs450",
    "foot200_mid",
    "foot240_chiselsite",
    "foot250_chiselarena",
    "foot300_hallway",
    "foot400_arena",
    "foot500_top",
    "foot510_cs190",
    "foot520_cs320",
    "for025_animres",
    "for099_rivertofreya",
    "for100_river",
    "for101_rivertohouse",
    "for200_c020_mothersknife",
    "for200_c078_ashesgathered",
    "for200_c080_baldurintro",
    "for200_c100_timetogo",
    "for200_c380_retrieveblades",
    "for200_c620_epilogue",
    "for200_house",
    "for250_chiselsite",
    "for255_chiselentrance",
    "for260_chiseldungeon",
    "for300_bossarena",
    "for310_bossassets",
    "for350_exittunnel",
    "for400_intersection",
    "for500_forestentry",
    "for550_brokenforest",
    "for550_cinebow",
    "for560_wolfpit",
    "for600_spire",
    "for601_landmark",
    "for650_bridge",
    "for660_templeinterior",
    "for666_loadingdoor",
    "for700_deerkill",
    "for800_trollarena",
    "for810_trollarenaexit",
    "for900_exitclimb",
    "for910_stavevista",
    "for950_riverpasslh",
    "freyaltw",
    "freyalvl100_global",
    "gbl000_fasttravel",
    "hel005_realmtower",
    "hel010_vista",
    "hel075_lights",
    "hel080_global",
    "hel100_calderaheldressing",
    "hel200_entrancedoor",
    "hel300_mainbridge",
    "hel310_monolith",
    "hel325_chiselsite",
    "hel350_chiselarena",
    "hel600_guardianarena",
    "helr010_eagle",
    "helr075_lights",
    "helr080_monolith",
    "helr090_flyover",
    "helr100_docks",
    "helr100_shell",
    "helr200_boatride",
    "helr300_helship",
    "helr400_calderareturn",
    "jot100_giantsfingers",
    "msp100_base",
    "msp100_gold_01",
    "msp100_gold_02",
    "msp100_gold_03",
    "msp100_gold_04",
    "msp100_gold_05",
    "msp100_gold_06",
    "msp110_arena01",
    "msp120_arena02",
    "msp130_arena03",
    "msp140_arena04",
    "msp150_arena05",
    "msp150_calderabridge",
    "msp160_arena06",
    "nid000_levelglobal",
    "nid100_entrance",
    "nid150_calderabridge",
    "nid200_swhallway",
    "nid220_nwhallway",
    "nid240_nehallway",
    "nid260_sehallway",
    "nid300_wroom",
    "nid310_nwroom",
    "nid320_nroom",
    "nid330_neroom",
    "nid340_eroom",
    "nid400_center",
    "nid500_mazeencounters",
    "peak075_lights",
    "peak080_global",
    "peak100_entrance",
    "peak120_cavescarelh",
    "peak140_caverndark",
    "peak170_cavecrawl",
    "peak180_enttochimneylh",
    "peak190_chimneyenter",
    "peak200_chimneylow",
    "peak200_chimneylowvista",
    "peak203_chiselsite",
    "peak205_chiselarena",
    "peak210_rollerroomlh",
    "peak260_chimneylowhall",
    "peak290_shafta",
    "peak300_chimneymid",
    "peak380_bigelevator",
    "peak400_shaftb",
    "peak500_chimneytop",
    "peak510_dragontunnellh",
    "peak520_postdragontunnel",
    "peak540_predragonarena",
    "peak550_dragonarena",
    "peak580_expresselevator",
    "peak590_chimneytophalllh",
    "peak600_shaftc",
    "peak700_summitascent",
    "peak720_summitascenthub",
    "peak730_summitclimb",
    "peak740_c310_mimirintro",
    "peak740_c470_baldurreturns",
    "peak740_summitpeak",
    "peak750_summitcave",
    "peak760_summitbase",
    "peak780_summitexit",
    "peak790_chimneytopreturn",
    "peak790_summitexit",
    "peak800_dragonride",
    "peak900_riverpassexit",
    "riv085_dangersentrancelh",
    "riv100_dangersentrance",
    "riv150_dangersmainlh",
    "riv200_c115_bridgecollapse",
    "riv200_dangersmain",
    "riv225_dangerscave",
    "riv250_dangersside",
    "riv260_shopintrocn",
    "riv300_calderavistalh",
    "riv325_dangersexit",
    "riv350_c135_canyonoverlook",
    "riv350_calderavista",
    "riv400_forestintro",
    "riv405_forestboarstartlh",
    "riv420_forestboarstart",
    "riv425_forestboartracklh",
    "riv430_c137_boarintro",
    "riv430_forestboartrack",
    "riv445_c140_boarshoot",
    "riv445_forestboarhit",
    "riv450_forestchase",
    "riv455_c150_witchintro",
    "riv455_forestfreyaintro",
    "riv470_freyapath",
    "riv475_freyahouseext",
    "riv480_c160_witchhousea",
    "riv480_c165_witchhouseb",
    "riv480_c330_headenchant",
    "riv480_c370_freyatendsson",
    "riv480_c390_sonhealed",
    "riv480_freyahouse",
    "riv575_freyavista",
    "riv600_freyariver",
    "riv925_freyacave",
    "riv935_c400_demigodreveal",
    "riv935_freyaelevator",
    "riv950_chiselsite",
    "riv975_chiselarena",
    "stn090_lakevista",
    "stn100_entrance",
    "stn105_chiselsite",
    "stn110_chiselarena",
    "stn120_cheapgiant",
    "stn130_righthand",
    "stn150_handunder",
    "stn200_lakeext",
    "stn210_loadingdoors",
    "stn220_lakecombat",
    "stn250_banditcamp",
    "stn290_sindrishop",
    "stn300_huldracamp",
    "stn350_relicroom",
    "stn400_hammerbase",
    "stn430_hammerclimb",
    "stn460_hammerhead",
    "stn500_impactsite",
    "stn550_chasmclimb",
    "stn600_chasmroom",
    "stn610_poiintrocombat",
    "stn650_chiselhallway",
    "stn660_poireturncombat",
    "stn700_c360_chiselsick",
    "stn700_chiselroom",
    "stn750_returnhallway",
    "stn800_shortcut",
    "stn905_chiselsite",
    "stn910_bossentrance",
    "stn910_c540_finalfightintro",
    "stn910_c550_shieldsdown",
    "stn910_c570_baldurdeath",
    "stn920_bosslake",
    "stn970_bossroom",
    "stn980_bossshoulder",
    "waterglobal",
    "xpl100_httk",
    "xpl125_httklh",
    "xpl150_httktemple",
    "xpl160_httkcavel",
    "xpl170_httkcaver",
    "xpl200_funeral",
    "xpl220_funerallh",
    "xpl225_funerallh",
    "xpl250_funeralinterior",
    "xpl300_stronghold",
    "xpl325_strongholdlh",
    "xpl350_stronghold",
    "xpl400_huldramines",
    "xpl425_huldramineslh",
    "xpl450_huldramines",
    "xpl475_huldramineslh",
    "xpl475_lights",
    "xpl600_masontrail",
    "xpl625_masontraillh",
    "xpl650_masontrailcave",
    "xpl850_dungeonforest",
    "xpl875_dungeonforestlh",
    "xpl900_islandarch",
    "xpl910_islandshipwreck",
    "xpl920_islandclimb",
    "xpl930_beachruins",
    "xpl940_beachcave",
    "xpl950_beachmaze",
    "xpl960_beachship"
  }
end
function LevelSelect:Enter()
  self.menu:Activate()
  local wadsList = self.menu:GetList("wads")
  self:UpdateLevelList(wadsList)
  self:UpdateFooterButtonPrompt()
end
function LevelSelect:Exit()
  self.menu:Deactivate(true)
end
function LevelSelect:FindLevelList(list, levels)
  local n = {}
  for _, levelname in ipairs(levels) do
    if list[string.lower(levelname)] ~= nil or list[levelname] ~= nil then
      n[#n + 1] = levelname
    end
  end
  return n
end
function LevelSelect:UpdateLevelList(wadsList)
  local levelWads = self:FindLevelList(tablex.Inverse(engine.GetAvailableWads()), self.defaultLevelWads)
  self.settingsOn:UpdateWadList(wadsList, self.defaultLevelWads)
end
function LevelSelect:Wads_AdvanceReleaseHandler()
  local wad = self.menu:GetList("wads"):GetSelectedItem()
  self.settingsOn:OptionsReleaseHandler()
  game.QuitAndJumpToWad(wad)
end
function LevelSelect:UpdateFooterButtonPrompt()
  self.menu:UpdateFooterButton("Select", true)
  self.menu:UpdateFooterButton("Back", true)
  self.menu:UpdateFooterButton("Close", true)
  self.menu:UpdateFooterButtonText()
end
function BookmarkSelect:Setup()
  self.settingsOn = self:GetState("SettingsOn")
  self.menu = menu.Menu.New(self, {
    FooterButtonInfo = {
      {
        Item = "Select",
        Text = "[AdvanceButton] " .. util.GetLAMSMsg(lamsIDs.Select)
      },
      {
        Item = "Back",
        Text = "[BackButton] " .. util.GetLAMSMsg(lamsIDs.Back),
        EventHandlers = {
          {
            Events = {
              "EVT_Back_Release",
              "EVT_MOUSE_RIGHT_CLICKED"
            },
            Handler = function()
              self:Goto("Bookmarks")
            end
          }
        }
      },
      {
        Item = "Close",
        Text = "",
        EventHandlers = {
          {
            Events = {
              "EVT_Options_Release"
            },
            Handler = function()
              self.settingsOn:OptionsReleaseHandler()
            end
          }
        }
      }
    }
  })
  local bookmarksList = list.List.New(self, {
    MaxFocusableObjectCount = 10,
    ListObjectName = "SettingsList",
    PadRepeatRate = 8,
    EmptyTextLamsID = lamsIDs.NoSettingsAvailable,
    BackingAnimFrameCount = 60,
    HoverToSelect = true,
    EventHandlers = {
      {
        Events = {
          "EVT_Advance_Release"
        },
        Handler = function()
          self:BookmarkSelect_AdvanceReleaseHandler()
        end
      }
    },
    PageUpEvents = {
      "EVT_L2_Press"
    },
    PageDownEvents = {
      "EVT_R2_Press"
    }
  })
  self.menu:SetList("bookmarks", bookmarksList)
end
function BookmarkSelect:Enter()
  self.menu:Activate()
  local bookmarksList = self.menu:GetList("bookmarks")
  local bookmarks = {}
  if engine.GetAvailableBookmarks ~= nil then
    bookmarks = engine.GetAvailableBookmarks()
  end
  self.settingsOn:UpdateWadList(bookmarksList, bookmarks)
  self:UpdateFooterButtonPrompt()
end
function BookmarkSelect:Exit()
  self.menu:Deactivate(true)
end
function BookmarkSelect:BookmarkSelect_AdvanceReleaseHandler()
  local bookmark = self.menu:GetList("bookmarks"):GetSelectedItem()
  self.settingsOn:OptionsReleaseHandler()
  game.QuitAndJumpToBookmark("Official/Progression/" .. bookmark)
end
function BookmarkSelect:UpdateFooterButtonPrompt()
  self.menu:UpdateFooterButton("Select", true)
  self.menu:UpdateFooterButton("Back", true)
  self.menu:UpdateFooterButton("Close", true)
  self.menu:UpdateFooterButtonText()
end
function Calibration:Setup()
  self.settingsMenu = self:GetState("settingsMenu")
end
function Calibration:Enter()
  self:WantPadEvents(false)
  self:SendEventToUIFsm("calibrationScreen", "EVT_OPEN_CALIBRATION_SETTINGS", self.settingsMenu.from_shell)
  Audio.PlaySound("SND_UX_Menu_Tick_Confirm")
end
function Calibration:Exit()
  self:WantPadEvents(true)
  Audio.PlaySound("SND_UX_Menu_Tick_Back")
end
function Calibration:EVT_CLOSE_CALIBRATION_SCREEN()
  self:Goto("VideoSettings")
end
function Calibration:EVT_CLOSE_CALIBRATION_AND_LEAVE_PAUSE()
  self:SendEventToUIFsm("globalMenu", "EVT_TURN_OFF_GLOBAL_MENU")
end
function Calibration:EVT_Back_Release()
end
function Calibration:EVT_Options_Release()
end
function AdvancedSettings:Setup()
  self.handling_event = false
  self:SetAllHandlersFalse()
  self.settingsOn = self:GetState("SettingsOn")
  self.menu = menu.Menu.New(self, {
    FooterButtonInfo = {
      {
        Item = "Default",
        Text = "[TriangleButton] " .. util.GetLAMSMsg(lamsIDs.Default)
      },
      {
        Item = "Back",
        Text = "[BackButton] " .. util.GetLAMSMsg(lamsIDs.Back)
      }
    }
  })
  local AdvancedSettingsList = list.List.New(self, {
    MaxFocusableObjectCount = 20,
    ListObjectName = "MiddleSettingsList",
    EmptyTextLamsID = lamsIDs.NoSettingsAvailable,
    BackingAnimFrameCount = 60,
    HoverToSelect = true,
    Button_OnGainFocus = function(button)
      local currItem = button:get_item()
      self:SelectAdvancedSettings(currItem)
    end
  })
  self.menu:SetList("AdvancedSettings", AdvancedSettingsList)
  AdvancedSettingsList:SetSelectedButton(1, false)
  self.AdvancedMarkersToggle = toggle.Toggle.New(self, {
    Name = "AdvancedMarkers",
    Get = function()
      return game.GetLatencyMarkers()
    end,
    Set = function(num)
      game.SetLatencyMarkers(num)
    end,
    Default = function()
      return false
    end
  })
  self.AdvancedModeToggle = toggle.Toggle.New(self, {
    Name = "AdvancedMode",
    Get = function()
      return game.GetLatencyMode()
    end,
    Set = function(num)
      game.SetLatencyMode(num)
    end,
    Settings = {
      {
        Value = 0,
        Text = util.GetLAMSMsg(lamsIDs.Off)
      },
      {
        Value = 1,
        Text = util.GetLAMSMsg(lamsIDs.On)
      },
      {
        Value = 2,
        Text = util.GetLAMSMsg(lamsIDs.OnPlusBoost)
      }
    },
    Default = function()
      return 0
    end
  })
  self.AdvancedFlashToggle = toggle.Toggle.New(self, {
    Name = "AdvancedFlash",
    Get = function()
      return game.GetLatencyFlash()
    end,
    Set = function(num)
      game.SetLatencyFlash(num)
    end,
    Default = function()
      return false
    end
  })
  self.AdvancedStatsToggle = toggle.Toggle.New(self, {
    Name = "AdvancedStats",
    Get = function()
      return game.GetLatencyStats()
    end,
    Set = function(num)
      game.SetLatencyStats(num)
    end,
    Default = function()
      return false
    end
  })
  self.settingsInfo = {
    {
      Item = "AdvancedMode",
      Text = util.GetLAMSMsg(lamsIDs.ReflexMode),
      ActivationList = {
        {
          Item = self.AdvancedModeToggle
        }
      },
      ToolTip = util.GetLAMSMsg(lamsIDs.ReflexModeDesc)
    },
    {
      Item = "AdvancedMarkers",
      Text = util.GetLAMSMsg(lamsIDs.LatencyMarkers),
      ActivationList = {
        {
          Item = self.AdvancedMarkersToggle
        }
      },
      ToolTip = util.GetLAMSMsg(lamsIDs.LatencyMarkersDesc)
    },
    {
      Item = "AdvancedFlash",
      Text = util.GetLAMSMsg(lamsIDs.LatencyFlash),
      ActivationList = {
        {
          Item = self.AdvancedFlashToggle
        }
      },
      ToolTip = util.GetLAMSMsg(lamsIDs.LatencyFlashDesc)
    }
  }
end
function AdvancedSettings:Enter()
  self.handling_event = false
  self:SetAllHandlersFalse()
  self.menu:Activate()
  local AdvancedSettingsList = self.menu:GetList("AdvancedSettings")
  self.settingsOn:RefreshSettingsList(AdvancedSettingsList, self.settingsInfo, util.GetLAMSMsg(lamsIDs.AdvancedSettings))
  if not game.HasNvidiaVendorExtensions() then
    for i = 1, #self.settingsInfo do
      self.settingsOn:CheckDisabledButton(AdvancedSettingsList, i, true)
      local toggle = self.settingsInfo[i].ActivationList[1].Item
      toggle:OverrideSettings({
        {
          Text = util.GetLAMSMsg(lamsIDs.Off),
          Value = false
        }
      })
      self.settingsOn:CheckDisabledButton(toggle, 1, true)
    end
  end
  self:UpdateFooterButtonPrompt()
  Audio.PlaySound("SND_UX_Menu_Tick_Confirm")
end
function AdvancedSettings:Exit()
  local AdvancedSettingsList = self.menu:GetList("AdvancedSettings")
  for i = 1, #self.settingsInfo do
    self.settingsOn:ClearDisableButton(AdvancedSettingsList, i)
    local toggle = self.settingsInfo[i].ActivationList[1].Item
    self.settingsOn:ClearDisableButton(toggle, 1)
  end
  self.menu:Deactivate(true)
  Audio.PlaySound("SND_UX_Menu_Tick_Back")
end
function AdvancedSettings:SelectAdvancedSettings(currItem)
  self:UpdateFooterButtonPrompt()
end
function AdvancedSettings:SetDefaultValues()
  local AdvancedSettingsList = self.menu:GetList("AdvancedSettings")
  self.settingsOn:SetDefaultValues(AdvancedSettingsList, self.menu, function()
    self:Enter()
  end)
end
function AdvancedSettings:UpdateFooterButtonPrompt()
  self.menu:UpdateFooterButton("Default", true)
  self.menu:UpdateFooterButton("Back", true)
  self.menu:UpdateFooterButton("Close", true)
  self.menu:UpdateFooterButtonText()
end
function AdvancedSettings:EVT_MOUSE_CLICKED()
  if not self.handling_event then
    self.advance = true
  end
end
function AdvancedSettings:EVT_MOUSE_RIGHT_CLICKED()
  if not self.handling_event then
    self.back = true
  end
end
function AdvancedSettings:EVT_Back_Release()
  if not self.handling_event then
    self.back = true
  end
end
function AdvancedSettings:EVT_Options_Release()
  if not self.handling_event then
    self.back_options = true
  end
end
function AdvancedSettings:EVT_Triangle_Release()
  if not self.handling_event then
    self.triangle = true
  end
end
function AdvancedSettings:SetAllHandlersFalse()
  self.back = false
  self.back_options = false
  self.triangle = false
end
function AdvancedSettings:Update()
  if not self.handling_event then
    if self.back then
      self.handling_event = true
      self:SetAllHandlersFalse()
      self.back = false
      self:Goto("VideoSettings")
    elseif self.back_options then
      self.handling_event = true
      self:SetAllHandlersFalse()
      self.settingsOn:OptionsReleaseHandler()
    elseif self.triangle then
      self.handling_event = true
      self:SetAllHandlersFalse()
      self:SetDefaultValues()
    end
  end
end
function ScreenScale:Setup()
  self.settingsMenu = self:GetState("settingsMenu")
end
function ScreenScale:Enter()
  self:WantPadEvents(false)
  self:SendEventToUIFsm("screenOffset", "EVT_OPEN_SCREEN_SCALE_SETTINGS", self.settingsMenu.from_shell)
  Audio.PlaySound("SND_UX_Menu_Tick_Confirm")
end
function ScreenScale:Exit()
  self:WantPadEvents(true)
  Audio.PlaySound("SND_UX_Menu_Tick_Back")
end
function ScreenScale:EVT_CLOSE_SCREEN_SCALE_SCREEN()
  self:Goto("VideoSettings")
end
function ScreenScale:EVT_CLOSE_SCREEN_SCALE_AND_LEAVE_PAUSE()
  self:SendEventToUIFsm("globalMenu", "EVT_TURN_OFF_GLOBAL_MENU")
end
function ScreenScale:EVT_Back_Release()
end
function ScreenScale:EVT_Options_Release()
end
function SettingsMenu:OnSaveCheckpoint(tab)
end
function SettingsMenu:OnRestoreCheckpoint(tab)
end
