local version = "v2.0";

local config = {
	enabled = true,

	quest_types = {
		regular = true,
		random = true,
		rampage = false,
		random_master_rank = true,
		random_anomaly = true
	}
};

local config_file_name = "Inifite Quest Search/config.json";

local session_manager = nil;

local quest_types = {
	invalid = {},
	regular = {
		quest_id = 0
	},
	random = {
		my_hunter_rank = 0
	},
	rampage = {
		difficulty = 0,
		quest_level = nil,
		target_enemy = nil
	},
	random_master_rank = {
		my_hunter_rank = 0,
		my_master_rank = 0
	},
	random_anomaly = {
		my_hunter_rank = 0,
		my_master_rank = 0
	}
};
local quest_type = quest_types.invalid;

local function table_deep_copy(original, copies)
	copies = copies or {};
	local original_type = type(original);
	local copy;
	if original_type == 'table' then
		if copies[original] then
			copy = copies[original];
		else
			copy = {};
			copies[original] = copy;
			for original_key, original_value in next, original, nil do
				copy[table_deep_copy(original_key, copies)] = table_deep_copy(original_value, copies);
			end
			setmetatable(copy, table_deep_copy(getmetatable(original), copies));
		end
	else -- number, string, boolean, etc
		copy = original;
	end
	return copy;
end

local function merge_tables(...)
	local tables_to_merge = {...};
	assert(#tables_to_merge > 1, "There should be at least two tables to merge them");

	for key, table in ipairs(tables_to_merge) do
		assert(type(table) == "table", string.format("Expected a table as function parameter %d", key));
	end

	local result = table_deep_copy(tables_to_merge[1]);

	for i = 2, #tables_to_merge do
		local from = tables_to_merge[i];
		for key, value in pairs(from) do
			if type(value) == "table" then
				result[key] = result[key] or {};
				assert(type(result[key]) == "table", string.format("Expected a table: '%s'", key));
				result[key] = merge_tables(result[key], value);
			else
				result[key] = value;
			end
		end
	end

	return result;
end

local function load_config()
    local loaded_config = json.load_file(config.config_file_name);
	if loaded_config ~= nil then
		log.info('[Forced Display Mode] config.json loaded successfully');
		config = merge_tables(config, loaded_config);
	else
		log.error('[Forced Display Mode] Failed to load config.json');
	end
end

local function save_config()
	-- save current config to disk, replacing any existing file
	local success = json.dump_file(config_file_name, config);
	if success then
		log.info('[Forced Display Mode] config.json saved successfully');
	else
		log.error('[Forced Display Mode] Failed to save config.json');
	end
end

local function on_draw_ui()
	local changed = false;
	if imgui.tree_node("Inifinite Quest Search " .. version) then
		changed, config.enabled = imgui.checkbox("Enabled", config.enabled);

		if imgui.tree_node("Quest Types") then
			changed, config.quest_types.regular = imgui.checkbox("Regular", config.quest_types.regular);
			changed, config.quest_types.rampage = imgui.checkbox("Rampage", config.quest_types.rampage);
			changed, config.quest_types.random = imgui.checkbox("Random", config.quest_types.random);
			changed, config.quest_types.random_master_rank = imgui.checkbox("Random MR", config.quest_types.random_master_rank);
			changed, config.quest_types.random_anomaly = imgui.checkbox("Random Anomaly", config.quest_types.random_anomaly);

			imgui.tree_pop();
		end

		imgui.tree_pop();
	end

	if changed then
		save_config();
	end
end

local session_manager_type_def = sdk.find_type_definition("snow.SnowSessionManager");
local on_timeout_matchmaking_method = session_manager_type_def:get_method("funcOnTimeoutMatchmaking");

local req_matchmaking_method = session_manager_type_def:get_method("reqMatchmakingAutoJoinSession");
local req_matchmaking_random_method = session_manager_type_def:get_method("reqMatchmakingAutoJoinSessionRandom");
local req_matchmaking_hyakuryu_method = session_manager_type_def:get_method("reqMatchmakingAutoJoinSessionHyakuryu");
local req_matchmaking_random_master_rank_method = session_manager_type_def:get_method("reqMatchmakingAutoJoinSessionRandomMasterRank");
local req_matchmaking_random_mystery_method = session_manager_type_def:get_method("reqMatchmakingAutoJoinSessionRandomMystery");

local function on_post_timeout_matchmaking(retval)
	if not config.enabled then
		return retval;
	end

	if session_manager == nil then
		session_manager = sdk.get_managed_singleton("snow.SnowSessionManager");
	end

	if session_manager == nil then
		log.info("[Infinite Quest Search] No session manager");
		return retval;
	end

	if quest_type == quest_types.regular then
		if config.quest_types.regular then
			req_matchmaking_method:call(session_manager, quest_type.quest_id);
		end

	elseif quest_type == quest_types.random then
		if config.quest_types.random then
			req_matchmaking_random_method:call(session_manager, quest_type.my_hunter_rank);
		end
		
	elseif quest_type == quest_types.rampage then
		if config.quest_types.rampade then
			req_matchmaking_hyakuryu_method:call(session_manager, quest_type.difficulty, quest_type.quest_level, quest_type.target_enemy);
		end

	elseif quest_type == quest_types.random_master_rank then
		if config.quest_types.random_master_rank then
			req_matchmaking_random_master_rank_method:call(session_manager, quest_type.my_hunter_rank, quest_type.my_master_rank);
		end

	elseif quest_type == quest_types.random_anomaly then
		if config.quest_types.random_anomaly then
			req_matchmaking_random_mystery_method:call(session_manager, quest_type.my_hunter_rank, quest_type.my_master_rank);
		end
	end
	
	return retval;
end

local function on_req_matchmaking(quest_id)
	quest_type = quest_types.regular;
	quest_type.quest_id = quest_id;
end

local function on_req_matchmaking_random(my_hunter_rank)
	quest_type = quest_types.random;
	quest_type.my_hunter_rank = my_hunter_rank;
end

local function on_req_matchmaking_rampage(difficulty, quest_level, target_enemy)
	quest_type = quest_types.rampage;
	quest_type.difficulty = difficulty;
	quest_type.quest_level = quest_level;
	quest_type.target_enemy = target_enemy;
end

local function on_req_matchmaking_random_master_rank(my_hunter_rank, my_master_rank)
	quest_type = quest_types.random_master_rank;
	quest_type.my_hunter_rank = my_hunter_rank;
	quest_type.my_master_rank = my_master_rank;
end

local function on_req_matchmaking_random_anomaly(my_hunter_rank, my_master_rank)
	quest_type = quest_types.random_anomaly;
	quest_type.my_hunter_rank = my_hunter_rank;
	quest_type.my_master_rank = my_master_rank;
end

load_config();
re.on_draw_ui(on_draw_ui);

sdk.hook(on_timeout_matchmaking_method, function(args) end,
function(retval)
	return on_post_timeout_matchmaking(retval);
end);

sdk.hook(req_matchmaking_method, function(args)
	on_req_matchmaking(sdk.to_int64(args[3]));
end, function(retval)
	return retval;
end);

sdk.hook(req_matchmaking_random_method, function(args)
	on_req_matchmaking_random(sdk.to_int64(args[3]));
end, function(retval)
	return retval;
end);

sdk.hook(req_matchmaking_hyakuryu_method, function(args)
	on_req_matchmaking_rampage(sdk.to_int64(args[3]), sdk.to_int64(args[4]), sdk.to_int64(args[5]));
end, function(retval)
	return retval;
end);

sdk.hook(req_matchmaking_random_master_rank_method, function(args)
	on_req_matchmaking_random_master_rank(sdk.to_int64(args[3]), sdk.to_int64(args[4]));
end, function(retval)
	return retval;
end);

sdk.hook(req_matchmaking_random_mystery_method, function(args)
	on_req_matchmaking_random_anomaly(sdk.to_int64(args[3]), sdk.to_int64(args[4]));
end, function(retval)
	return retval;
end);