﻿using System.IO;
using System.Xml;
using Microsoft.Build.Framework;
using Microsoft.Build.Utilities;
using SkiaSharp;

namespace Microsoft.Maui.Resizetizer
{
	/// <summary>
	/// Generates Resources/values/maui_colors.xml and Resources/drawable/maui_splash_image.xml for Android splash screens
	/// </summary>
	public class GenerateSplashAndroidResources : Task
	{
		[Required]
		public string ColorsFile { get; set; }

		[Required]
		public string DrawableFile { get; set; }

		[Required]
		public ITaskItem[] MauiSplashScreen { get; set; }

		public override bool Execute()
		{
			var splash = MauiSplashScreen[0];
			WriteColors(splash);
			WriteDrawable(splash);
			return !Log.HasLoggedErrors;
		}

		static readonly XmlWriterSettings Settings = new XmlWriterSettings { Indent = true };
		const string Namespace = "http://schemas.android.com/apk/res/android";
		const string Comment = "This file was auto-generated by .NET MAUI.";

		void WriteColors(ITaskItem splash)
		{
			Directory.CreateDirectory(Path.GetDirectoryName(ColorsFile));

			using var writer = XmlWriter.Create(ColorsFile, Settings);
			writer.WriteComment(Comment);
			writer.WriteStartElement("resources");

			var colorMetadata = splash.GetMetadata("Color");
			var color = Utils.ParseColorString(colorMetadata);
			if (color != null)
			{
				writer.WriteStartElement("color");
				writer.WriteAttributeString("name", "maui_splash_color");
				writer.WriteString(color.ToString());
				writer.WriteEndElement();
			}
			else if (!string.IsNullOrEmpty(colorMetadata))
			{
				Log.LogWarning($"Unable to parse color value '{colorMetadata}' for '{splash.ItemSpec}'.");
			}

			writer.WriteEndDocument();
		}

		void WriteDrawable(ITaskItem splash)
		{
			Directory.CreateDirectory(Path.GetDirectoryName(DrawableFile));

			using var writer = XmlWriter.Create(DrawableFile, Settings);
			writer.WriteComment(Comment);
			writer.WriteStartElement("layer-list");
			writer.WriteAttributeString("xmlns", "android", ns: null, value: Namespace);

			writer.WriteStartElement("item");
			writer.WriteAttributeString("android", "drawable", Namespace, "@drawable/" + Path.GetFileNameWithoutExtension(splash.ItemSpec.ToLowerInvariant()));

			writer.WriteEndDocument();
		}
	}
}
