/*
* Smart Moving Reloaded
* Copyright (C) 2018  Tommsy64
*
* Smart Moving Reloaded is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* Smart Moving Reloaded is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with Smart Moving Reloaded.  If not, see <http://www.gnu.org/licenses/>.
*/

package com.tommsy.smartmoving.client.model;

import java.util.List;
import java.util.Random;

import lombok.NonNull;
import lombok.Setter;

import org.lwjgl.opengl.GL11;

import net.minecraft.client.entity.AbstractClientPlayer;
import net.minecraft.client.model.ModelBiped;
import net.minecraft.client.model.ModelRenderer;
import net.minecraft.entity.Entity;
import net.minecraft.entity.EntityLivingBase;
import net.minecraft.util.EnumHandSide;
import net.minecraft.util.math.MathHelper;
import net.minecraft.util.math.Vec3d;

import com.tommsy.smartmoving.client.SmartMovingAbstractClientPlayer;
import com.tommsy.smartmoving.client.renderer.ModelPreviousRotationRenderer;
import com.tommsy.smartmoving.client.renderer.ModelRotationRenderer;
import com.tommsy.smartmoving.client.renderer.ModelRotationRenderer.RotationOrder;
import com.tommsy.smartmoving.client.renderer.ScaleType;
import com.tommsy.smartmoving.common.SmartMovingPlayerState;

import static com.tommsy.smartmoving.client.renderer.RenderUtils.Eighth;
import static com.tommsy.smartmoving.client.renderer.RenderUtils.Half;
import static com.tommsy.smartmoving.client.renderer.RenderUtils.Quarter;
import static com.tommsy.smartmoving.client.renderer.RenderUtils.RadianToAngle;
import static com.tommsy.smartmoving.client.renderer.RenderUtils.Sixteenth;
import static com.tommsy.smartmoving.client.renderer.RenderUtils.Sixtyfourth;
import static com.tommsy.smartmoving.client.renderer.RenderUtils.Thirtysecond;
import static com.tommsy.smartmoving.client.renderer.RenderUtils.Whole;

public class SmartMovingModelBipedHandler {

    // These are the same object, just different types
    private final SmartMovingModelBiped smModel;
    private final ModelBiped model;

    private final boolean doGLMatrixPop;

    protected ModelPreviousRotationRenderer bipedOuter;
    protected ModelRotationRenderer bipedTorso;
    protected ModelRotationRenderer bipedBody;
    protected ModelRotationRenderer bipedBreast;
    protected ModelRotationRenderer bipedNeck;
    protected ModelRotationRenderer bipedHead;
    protected ModelRotationRenderer bipedRightShoulder;
    protected ModelRotationRenderer bipedRightArm;
    protected ModelRotationRenderer bipedLeftShoulder;
    protected ModelRotationRenderer bipedLeftArm;
    protected ModelRotationRenderer bipedPelvic;
    protected ModelRotationRenderer bipedRightLeg;
    protected ModelRotationRenderer bipedLeftLeg;

    protected ModelRotationRenderer bipedHeadwear;

    // Updated from RenderPlayer
    public float rotationYaw;
    /**
     * Set to true when the model is being rendered in the inventory GUI
     */
    public boolean isBeingRenderedInInventory;

    @Setter
    @NonNull
    private ScaleType scaleArm = ScaleType.Scale, scaleLeg = ScaleType.Scale;

    public SmartMovingModelBipedHandler(SmartMovingModelBiped smModel) {
        this(smModel, true);
    }

    /**
     * Make sure to call {@link #initialize()} after construction.
     */
    protected SmartMovingModelBipedHandler(SmartMovingModelBiped smModel, final boolean doGLMatrixPop) {
        model = (this.smModel = smModel).getImplementation();
        this.doGLMatrixPop = doGLMatrixPop;

        // Clear boxList that holds default ModelRenderers
        // The custom ones are automatically added back
        model.field_78092_r.clear();
        // Can't call initialize from constructor because fields are not accessible until object done constructing
    }

    public final void initialize() {
        initializeRenderers();
        reset(); // Set default rotation points
    }

    protected void initializeRenderers() {
        bipedOuter = new ModelPreviousRotationRenderer(model, -1, -1, null);
        bipedOuter.fadeEnabled = true;

        bipedTorso = new ModelRotationRenderer(model, bipedOuter);
        bipedBody = new ModelRotationRenderer(model, bipedTorso, model.field_78115_e);
        bipedBreast = new ModelRotationRenderer(model, bipedTorso);
        bipedNeck = new ModelRotationRenderer(model, bipedBreast);
        bipedHead = new ModelRotationRenderer(model, bipedNeck, model.field_78116_c);
        bipedRightShoulder = new ModelRotationRenderer(model, bipedBreast);
        bipedRightArm = new ModelRotationRenderer(model, bipedRightShoulder, model.field_178723_h);
        bipedLeftShoulder = new ModelRotationRenderer(model, bipedBreast);
        bipedLeftShoulder.field_78809_i = true;
        bipedLeftArm = new ModelRotationRenderer(model, bipedLeftShoulder, model.field_178724_i);
        bipedPelvic = new ModelRotationRenderer(model, bipedTorso);
        bipedRightLeg = new ModelRotationRenderer(model, bipedPelvic, model.field_178721_j);
        bipedLeftLeg = new ModelRotationRenderer(model, bipedPelvic, model.field_178722_k);

        bipedHeadwear = new ModelRotationRenderer(model, bipedHead, model.field_178720_f);

        model.field_78115_e = bipedBody;
        model.field_78116_c = bipedHead;
        model.field_178720_f = bipedHeadwear;
        model.field_178723_h = bipedRightArm;
        model.field_178724_i = bipedLeftArm;
        model.field_178721_j = bipedRightLeg;
        model.field_178722_k = bipedLeftLeg;
    }

    public void preRender(Entity entity, float limbSwing, float limbSwingAmount, float ageInTicks, float netHeadYaw, float headPitch, float scale) {
        GL11.glPushMatrix();
        bipedBody.ignoreRender = bipedHead.ignoreRender = bipedRightArm.ignoreRender = bipedLeftArm.ignoreRender = bipedRightLeg.ignoreRender = bipedLeftLeg.ignoreRender = bipedHeadwear.ignoreRender = true;
    }

    public void postRender(Entity entity, float limbSwing, float limbSwingAmount, float ageInTicks, float netHeadYaw, float headPitch, float scale) {
        bipedBody.ignoreRender = bipedHead.ignoreRender = bipedRightArm.ignoreRender = bipedLeftArm.ignoreRender = bipedRightLeg.ignoreRender = bipedLeftLeg.ignoreRender = bipedHeadwear.ignoreRender = false;

        bipedOuter.func_78785_a(scale);

        bipedOuter.renderIgnoreBase(scale);
        bipedTorso.renderIgnoreBase(scale);
        bipedBody.renderIgnoreBase(scale);
        bipedBreast.renderIgnoreBase(scale);
        bipedNeck.renderIgnoreBase(scale);
        bipedHead.renderIgnoreBase(scale);
        bipedRightShoulder.renderIgnoreBase(scale);
        bipedRightArm.renderIgnoreBase(scale);
        bipedLeftShoulder.renderIgnoreBase(scale);
        bipedLeftArm.renderIgnoreBase(scale);
        bipedPelvic.renderIgnoreBase(scale);
        bipedRightLeg.renderIgnoreBase(scale);
        bipedLeftLeg.renderIgnoreBase(scale);

        bipedHeadwear.renderIgnoreBase(scale);

        popMatrix();
    }

    /**
     * @param totalHorizontalDistance The total, accumulated distance the entity has moved
     * @param currentHorizontalSpeed The entity's magnitude of horizontal velocity.
     * @return True if {@linkplain ModelBiped#setRotationAngles(float, float, float, float, float, float, Entity)} should not be called
     */
    public boolean preSetRotationAngles(float totalHorizontalDistance, float currentHorizontalSpeed, float ageInTicks, float headYawAngle, float headPitchAngle, float scaleFactor,
            Entity entity) {
        reset(); // Reset all model renderers to their default states

        if (isBeingRenderedInInventory) {
            bipedBody.ignoreBase = true;
            bipedHead.ignoreBase = true;
            bipedRightArm.ignoreBase = true;
            bipedLeftArm.ignoreBase = true;
            bipedRightLeg.ignoreBase = true;
            bipedLeftLeg.ignoreBase = true;

            bipedBody.forceRender = false;
            bipedHead.forceRender = false;
            bipedRightArm.forceRender = false;
            bipedLeftArm.forceRender = false;
            bipedRightLeg.forceRender = false;
            bipedLeftLeg.forceRender = false;

            bipedRightArm.func_78793_a(-5F, 2.0F, 0.0F);
            bipedLeftArm.func_78793_a(5F, 2.0F, 0.0F);
            bipedRightLeg.func_78793_a(-2F, 12F, 0.0F);
            bipedLeftLeg.func_78793_a(2.0F, 12F, 0.0F);

            return false;
        }

        final AbstractClientPlayer player = (AbstractClientPlayer) entity;

        bipedOuter.field_78796_g = rotationYaw / RadianToAngle;
        bipedOuter.fadeRotateAngleY = !player.func_184218_aH();

        boolean isStandardAnimation = false;

        // Handle smart moving state...

        final float partialTicks = ageInTicks - player.field_70173_aa;

        // TODO: The original smart moving takes a moving average of the horizontal speed to get a smoother value.
        // float diffX = (float) (entity.posX - entity.prevPosX), diffZ = (float) (entity.posZ - entity.prevPosZ);
        final float horizontalSpeed = currentHorizontalSpeed; // MathHelper.sqrt(diffX * diffX + diffZ * diffZ);

        SmartMovingAbstractClientPlayer smPlayer = (SmartMovingAbstractClientPlayer) player;
        SmartMovingPlayerState state = smPlayer.getState();
        if (state.isCrawling) {
            float distance = totalHorizontalDistance * 1.3F;
            float walkFactor = factor(horizontalSpeed, 0F, 0.12951545F);
            float standFactor = factor(horizontalSpeed, 0.12951545F, 0F);

            bipedTorso.rotationOrder = RotationOrder.YZX; // Change the rotation order to match the new horizontal orientation

            bipedTorso.field_82907_q = -0.935f; // Shifts entire player toward the ground

            bipedTorso.field_78795_f = Quarter - Thirtysecond; // Tilt player to be horizontal
            // Because the rotation order is changed above from XYZ to YZX, rotationPoint Y affects rotateAngle X; point Y, angle Z; point Z, angle X
            bipedTorso.field_78797_d = 3F; // Adjust the rotation point so that the body actually touches the ground
            bipedTorso.field_78808_h = MathHelper.func_76134_b(distance + Quarter) * Sixtyfourth * walkFactor; // Roll torso (and everything attached to it) back and forth a bit

            bipedBody.field_78796_g = MathHelper.func_76134_b(distance + Half) * Sixtyfourth * walkFactor; // Roll body back and forth a bit

            bipedHead.field_78808_h = -headYawAngle / RadianToAngle; // Left to right rotation of head
            bipedHead.field_78798_e = -1.5F; // Shifts the head a little bit closer to the chest/ground

            bipedHead.field_78795_f = -Eighth; // Rotates head toward the ground

            // Leg X rotation picks the leg off the ground, making it look like it is pushing/pulling it

            float legRotateXOne = (MathHelper.func_76134_b(distance - Quarter) * Sixtyfourth + Thirtysecond) * walkFactor + Thirtysecond * standFactor;
            float legRotateXTwo = (MathHelper.func_76134_b(distance - Half - Quarter) * Sixtyfourth + Thirtysecond) * walkFactor + Thirtysecond * standFactor;

            // When moving forward, legs push the ground; when moving backward, legs pull on the ground.
            if (player.field_191988_bg >= 0) {
                bipedRightLeg.field_78795_f = legRotateXOne;
                bipedLeftLeg.field_78795_f = legRotateXTwo;
            } else {
                bipedRightLeg.field_78795_f = legRotateXTwo;
                bipedLeftLeg.field_78795_f = legRotateXOne;
            }

            bipedRightLeg.field_78808_h = (MathHelper.func_76134_b(distance - Quarter) + 1F) * 0.25F * walkFactor + Thirtysecond * standFactor;
            bipedLeftLeg.field_78808_h = (MathHelper.func_76134_b(distance - Quarter) - 1F) * 0.25F * walkFactor - Thirtysecond * standFactor;

            // Makes knees look like they are bending
            if (scaleLeg != ScaleType.NoScaleStart)
                setLegScales(
                        0.95F + (MathHelper.func_76134_b(distance + Quarter - Quarter) - 1F) * 0.25F * walkFactor,
                        0.95F + (MathHelper.func_76134_b(distance - Quarter - Quarter) - 1F) * 0.25F * walkFactor);

            bipedRightArm.rotationOrder = RotationOrder.YZX;
            bipedLeftArm.rotationOrder = RotationOrder.YZX;

            bipedRightArm.field_78795_f = Half + Eighth - Thirtysecond;
            bipedLeftArm.field_78795_f = Half + Eighth - Thirtysecond;

            bipedRightArm.field_78808_h = ((MathHelper.func_76134_b(distance + Half)) * Sixtyfourth + Thirtysecond) * walkFactor + Sixteenth * standFactor;
            bipedLeftArm.field_78808_h = ((MathHelper.func_76134_b(distance + Half)) * Sixtyfourth - Thirtysecond) * walkFactor - Sixteenth * standFactor;

            bipedRightArm.field_78796_g = -Quarter;
            bipedLeftArm.field_78796_g = Quarter;

            // Makes arms look like they are being picked up and put back on the ground
            if (scaleArm != ScaleType.NoScaleStart)
                setArmScales(
                        0.95F + (MathHelper.func_76134_b(distance + Quarter) - 1F) * 0.15F * walkFactor,
                        0.95F + (MathHelper.func_76134_b(distance - Quarter) - 1F) * 0.15F * walkFactor);
        } else if (player.func_184613_cA())
            animateElytraFlying((EntityLivingBase) entity, partialTicks, ageInTicks);
        else
            isStandardAnimation = true;

        animateHeadRotation(isStandardAnimation, headYawAngle, headPitchAngle, player.func_184613_cA());

        if (player.func_70608_bn())
            animateSleeping(isStandardAnimation);

        float elytraMagnitude = player.func_184599_cB() > 4 ? getElytraMagnitude(entity) : 1;
        if (isStandardAnimation || elytraMagnitude != 1) {
            animateArmSwinging(totalHorizontalDistance, currentHorizontalSpeed, elytraMagnitude);
        }

        if (model.field_78093_q)
            animateRiding(isStandardAnimation);

        if (model.field_187075_l == ModelBiped.ArmPose.ITEM)
            this.animateLeftArmItemHolding(isStandardAnimation);

        if (model.field_187076_m == ModelBiped.ArmPose.ITEM)
            this.animateRightArmItemHolding(isStandardAnimation);

        if (model.field_78095_p > 0F)
            animateSwinging(entity);
        if (model.field_78117_n)
            animateSneaking(isStandardAnimation);

        animateArms(isStandardAnimation, ageInTicks);

        if (model.field_187076_m == ModelBiped.ArmPose.BOW_AND_ARROW)
            animateBowAimingRight(ageInTicks);
        else if (model.field_187075_l == ModelBiped.ArmPose.BOW_AND_ARROW)
            animateBowAimingLeft(ageInTicks);

        // ???
        if (!bipedOuter.fadeRotateAngleX)
            bipedOuter.previous.rotateAngleX = bipedOuter.field_78795_f;

        if (!bipedOuter.fadeRotateAngleY)
            bipedOuter.previous.rotateAngleY = bipedOuter.field_78796_g;

        bipedOuter.fadeIntermediate(ageInTicks);
        bipedOuter.fadeStore(ageInTicks);

        return true;
    }

    private static float getElytraMagnitude(Entity entity) {
        float f = (float) (entity.field_70159_w * entity.field_70159_w + entity.field_70181_x * entity.field_70181_x + entity.field_70179_y * entity.field_70179_y);
        f = f / 0.2F;
        f = f * f * f;
        if (f < 1.0F)
            return 1;
        return f;
    }

    public void postSetRotationAngles(float limbSwing, float limbSwingAmount, float ageInTicks, float netHeadYaw, float headPitch, float scaleFactor, Entity entityIn) {

    }

    public void reset() {
        bipedOuter.reset();
        bipedTorso.reset();
        bipedBody.reset();
        bipedBreast.reset();
        bipedNeck.reset();
        bipedHead.reset();
        bipedRightShoulder.reset();
        bipedRightArm.reset();
        bipedLeftShoulder.reset();
        bipedLeftArm.reset();
        bipedPelvic.reset();
        bipedRightLeg.reset();
        bipedLeftLeg.reset();

        bipedHeadwear.reset();

        bipedRightShoulder.func_78793_a(-5F, 2.0F, 0.0F);
        bipedLeftShoulder.func_78793_a(5F, 2.0F, 0.0F);
        bipedPelvic.func_78793_a(0.0F, 12.0F, 0.1F);
        bipedRightLeg.func_78793_a(-1.9F, 0.0F, 0.0F);
        bipedLeftLeg.func_78793_a(1.9F, 0.0F, 0.0F);
    }

    private void popMatrix() {
        if (doGLMatrixPop)
            GL11.glPopMatrix();
    }

    private void animateHeadRotation(boolean isStandardAnimation, float headYawAngle, float headPitchAngle, boolean elytraFlying) {
        if (!isStandardAnimation) { return; }
        bipedNeck.ignoreBase = !elytraFlying;
        bipedHead.field_78796_g = (elytraFlying ? 0 : rotationYaw + headYawAngle) / RadianToAngle;
        bipedHead.field_78795_f = elytraFlying ? -Eighth : headPitchAngle / RadianToAngle;
    }

    private void animateSleeping(boolean isStandardAnimation) {
        if (!isStandardAnimation) { return; }
        bipedNeck.ignoreBase = false;
        bipedHead.field_78796_g = 0F;
        bipedHead.field_78795_f = Eighth;
        bipedTorso.field_78798_e = -17F;
    }

    private void animateArmSwinging(float totalHorizontalDistance, float currentHorizontalSpeed, float elytraMagnitude) {
        bipedRightArm.field_78795_f = MathHelper.func_76134_b(totalHorizontalDistance * 0.6662F + Half) * 2.0F * currentHorizontalSpeed * 0.5F / elytraMagnitude;
        bipedLeftArm.field_78795_f = MathHelper.func_76134_b(totalHorizontalDistance * 0.6662F) * 2.0F * currentHorizontalSpeed * 0.5F / elytraMagnitude;

        bipedRightLeg.field_78795_f = MathHelper.func_76134_b(totalHorizontalDistance * 0.6662F) * 1.4F * currentHorizontalSpeed / elytraMagnitude;
        bipedLeftLeg.field_78795_f = MathHelper.func_76134_b(totalHorizontalDistance * 0.6662F + Half) * 1.4F * currentHorizontalSpeed / elytraMagnitude;
    }

    private void animateRiding(boolean isStandardAnimation) {
        if (!isStandardAnimation) { return; }
        bipedRightArm.field_78795_f += -0.6283185F;
        bipedLeftArm.field_78795_f += -0.6283185F;
        bipedRightLeg.field_78795_f = -1.256637F;
        bipedLeftLeg.field_78795_f = -1.256637F;
        bipedRightLeg.field_78796_g = 0.3141593F;
        bipedLeftLeg.field_78796_g = -0.3141593F;
    }

    private void animateLeftArmItemHolding(boolean isStandardAnimation) {
        if (!isStandardAnimation) { return; }
        bipedLeftArm.field_78795_f = bipedLeftArm.field_78795_f * 0.5F - 0.3141593F;
    }

    private void animateRightArmItemHolding(boolean isStandardAnimation) {
        if (!isStandardAnimation) { return; }
        bipedRightArm.field_78795_f = bipedRightArm.field_78795_f * 0.5F - 0.3141593F;
    }

    private void animateSwinging(Entity entity) {
        float angle = MathHelper.func_76126_a(MathHelper.func_76129_c(model.field_78095_p) * Whole) * 0.2F;
        bipedBreast.field_78796_g = bipedBody.field_78796_g += angle;
        bipedBreast.rotationOrder = bipedBody.rotationOrder = ModelRotationRenderer.RotationOrder.YXZ;
        bipedLeftArm.field_78795_f += angle;

        EnumHandSide enumhandside = model.func_187072_a(entity);
        if (enumhandside == EnumHandSide.LEFT)
            this.bipedBody.field_78796_g *= -1.0F;

        float f1 = 1.0F - model.field_78095_p;
        f1 = f1 * f1;
        f1 = f1 * f1;
        f1 = 1.0F - f1;
        float f2 = MathHelper.func_76126_a(f1 * Half);
        float f3 = MathHelper.func_76126_a(model.field_78095_p * Half) * -(this.bipedHead.field_78795_f - 0.7F) * 0.75F;

        ModelRenderer modelrenderer = this.getArmForSide(enumhandside);
        modelrenderer.field_78795_f = (float) ((double) modelrenderer.field_78795_f - ((double) f2 * 1.2D + (double) f3));
        modelrenderer.field_78796_g += this.bipedBody.field_78796_g * 2.0F;
        modelrenderer.field_78808_h += MathHelper.func_76126_a(model.field_78095_p * Half) * -0.4F;
    }

    private void animateSneaking(boolean isStandardAnimation) {
        if (!isStandardAnimation) { return; }
        bipedTorso.field_78795_f += 0.5F;
        bipedRightLeg.field_78795_f += -0.5F;
        bipedLeftLeg.field_78795_f += -0.5F;
        bipedRightArm.field_78795_f += -0.1F;
        bipedLeftArm.field_78795_f += -0.1F;

        bipedPelvic.field_82908_p = -0.13652F;
        bipedPelvic.field_82907_q = -0.05652F;

        bipedBreast.field_82908_p = -0.01872F;
        bipedBreast.field_82907_q = -0.07502F;

        bipedNeck.field_82908_p = 0.0621F;
    }

    private void animateArms(boolean isStandardAnimation, float totalTime) {
        if (!isStandardAnimation) { return; }
        bipedRightArm.field_78808_h += MathHelper.func_76134_b(totalTime * 0.09F) * 0.05F + 0.05F;
        bipedLeftArm.field_78808_h -= MathHelper.func_76134_b(totalTime * 0.09F) * 0.05F + 0.05F;
        bipedRightArm.field_78795_f += MathHelper.func_76126_a(totalTime * 0.067F) * 0.05F;
        bipedLeftArm.field_78795_f -= MathHelper.func_76126_a(totalTime * 0.067F) * 0.05F;
    }

    private void animateBowAimingLeft(float totalTime) {
        bipedRightArm.field_78808_h = 0.0F;
        bipedLeftArm.field_78808_h = 0.0F;
        bipedRightArm.field_78796_g = -0.1F + bipedHead.field_78796_g - bipedOuter.field_78796_g - 0.4F;
        bipedLeftArm.field_78796_g = 0.1F + bipedHead.field_78796_g - bipedOuter.field_78796_g;
        bipedRightArm.field_78795_f = -1.570796F + bipedHead.field_78795_f;
        bipedLeftArm.field_78795_f = -1.570796F + bipedHead.field_78795_f;
        bipedRightArm.field_78808_h += MathHelper.func_76134_b(totalTime * 0.09F) * 0.05F + 0.05F;
        bipedLeftArm.field_78808_h -= MathHelper.func_76134_b(totalTime * 0.09F) * 0.05F + 0.05F;
        bipedRightArm.field_78795_f += MathHelper.func_76126_a(totalTime * 0.067F) * 0.05F;
        bipedLeftArm.field_78795_f -= MathHelper.func_76126_a(totalTime * 0.067F) * 0.05F;
    }

    private void animateBowAimingRight(float totalTime) {
        bipedRightArm.field_78808_h = 0.0F;
        bipedLeftArm.field_78808_h = 0.0F;
        bipedRightArm.field_78796_g = -0.1F + bipedHead.field_78796_g - bipedOuter.field_78796_g;
        bipedLeftArm.field_78796_g = 0.1F + bipedHead.field_78796_g - bipedOuter.field_78796_g + 0.4F;
        bipedRightArm.field_78795_f = -1.570796F + bipedHead.field_78795_f;
        bipedLeftArm.field_78795_f = -1.570796F + bipedHead.field_78795_f;
        bipedRightArm.field_78808_h += MathHelper.func_76134_b(totalTime * 0.09F) * 0.05F + 0.05F;
        bipedLeftArm.field_78808_h -= MathHelper.func_76134_b(totalTime * 0.09F) * 0.05F + 0.05F;
        bipedRightArm.field_78795_f += MathHelper.func_76126_a(totalTime * 0.067F) * 0.05F;
        bipedLeftArm.field_78795_f -= MathHelper.func_76126_a(totalTime * 0.067F) * 0.05F;
    }

    private void animateElytraFlying(EntityLivingBase entity, float partialTicks, float totalTime) {
        bipedTorso.rotationOrder = RotationOrder.YZX;
        bipedTorso.field_78797_d = 23.5F;
        bipedTorso.field_82908_p = -1.15f;
        float f = entity.func_184599_cB() + partialTicks;
        float f1 = MathHelper.func_76131_a(f * f / 100.0F, 0.0F, 1.0F);
        bipedTorso.field_78795_f = f1 * (Quarter + entity.field_70125_A / RadianToAngle);
        Vec3d vec3d = entity.func_70676_i(partialTicks);
        double d0 = entity.field_70159_w * entity.field_70159_w + entity.field_70179_y * entity.field_70179_y;
        double d1 = vec3d.field_72450_a * vec3d.field_72450_a + vec3d.field_72449_c * vec3d.field_72449_c;

        if (d0 > 0.0D && d1 > 0.0D) {
            double d2 = (entity.field_70159_w * vec3d.field_72450_a + entity.field_70179_y * vec3d.field_72449_c) / (Math.sqrt(d0) * Math.sqrt(d1));
            double d3 = entity.field_70159_w * vec3d.field_72449_c - entity.field_70179_y * vec3d.field_72450_a;
            bipedTorso.field_78796_g = (float) (Math.signum(d3) * Math.acos(d2));
        }

        bipedRightArm.field_78808_h += MathHelper.func_76134_b(totalTime * 0.09F) * 0.05F + 0.05F;
        bipedLeftArm.field_78808_h -= MathHelper.func_76134_b(totalTime * 0.09F) * 0.05F + 0.05F;
        bipedRightArm.field_78795_f += MathHelper.func_76126_a(totalTime * 0.067F) * 0.05F;
        bipedLeftArm.field_78795_f -= MathHelper.func_76126_a(totalTime * 0.067F) * 0.05F;
    }

    private void setArmScales(float rightScale, float leftScale) {
        if (scaleArm == ScaleType.Scale) {
            bipedRightArm.scaleY = rightScale;
            bipedLeftArm.scaleY = leftScale;
        } else if (scaleArm == ScaleType.NoScaleEnd) {
            bipedRightArm.field_82908_p -= (1F - rightScale) * 0.5F;
            bipedLeftArm.field_82908_p -= (1F - leftScale) * 0.5F;
        }
    }

    private void setLegScales(float rightScale, float leftScale) {
        if (scaleLeg == ScaleType.Scale) {
            bipedRightLeg.scaleY = rightScale;
            bipedLeftLeg.scaleY = leftScale;
        } else if (scaleLeg == ScaleType.NoScaleEnd) {
            bipedRightLeg.field_82908_p -= (1F - rightScale) * 0.5F;
            bipedLeftLeg.field_82908_p -= (1F - leftScale) * 0.5F;
        }
    }

    private static float factor(float a, float b, float c) {
        if (b > c) {
            if (a <= c)
                return 1F;
            if (a >= b)
                return 0F;
            return (b - a) / (b - c);
        } else {
            if (a >= c)
                return 1F;
            if (a <= b)
                return 0F;
            return (a - b) / (c - b);
        }
    }

    private ModelRenderer getArmForSide(EnumHandSide side) {
        return side == EnumHandSide.LEFT ? this.bipedLeftArm : this.bipedRightArm;
    }

    public ModelRenderer getRandomModelBox(Random rand) {
        List<ModelRenderer> boxList = model.field_78092_r;
        final int size = boxList.size();
        int renderersWithBoxes = 0;
        final int[] stack = new int[size + 1];

        for (int i = 0; i < size; i++) {
            ModelRenderer renderer = boxList.get(i);
            if (canBeRandomBoxSource(renderer))
                stack[renderersWithBoxes++] = i;
        }

        if (renderersWithBoxes != 0)
            return boxList.get(stack[rand.nextInt(renderersWithBoxes)]);

        return null;
    }

    private static boolean canBeRandomBoxSource(ModelRenderer renderer) {
        return renderer.field_78804_l != null && renderer.field_78804_l.size() > 0 &&
                (!(renderer instanceof ModelRotationRenderer) || ((ModelRotationRenderer) renderer).canBeRandomBoxSource());
    }
}
