local large_monster = {};
local singletons;
local customization_menu;
local config;
local language;
local table_helpers;
local health_UI_entity;
local stamina_UI_entity;
local rage_UI_entity;
local ailment_UI_entity;
local ailment_buildup;
local screen;
local drawing;
local ailments;
local player;
local time;

local body_part;
local part_names;

large_monster.list = {};

function large_monster.new(enemy)
	local monster = {};
	monster.enemy = enemy;
	monster.is_large = true;

	monster.id = 0;
	monster.unique_id = 0;

	monster.health = 0;
	monster.max_health = 999999;
	monster.health_percentage = 0;
	monster.missing_health = 0;

	monster.capture_health = 0;
	monster.capture_percentage = 0;

	monster.dead_or_captured = false;
	monster.is_disp_icon_mini_map = true;

	monster.is_tired = false;
	monster.stamina = 0;
	monster.max_stamina = 1000;
	monster.stamina_percentage = 0;
	monster.missing_stamina = 0;

	monster.tired_timer = 0;
	monster.tired_duration = 600;

	monster.tired_total_seconds_left = 0;
	monster.tired_minutes_left = 0;
	monster.tired_seconds_left = 0;
	monster.tired_timer_percentage = 0;

	monster.is_in_rage = false;
	monster.rage_point = 0;
	monster.rage_limit = 3000;
	monster.rage_count = 0;
	monster.rage_percentage = 0;

	monster.rage_timer = 0;
	monster.rage_duration = 600;

	monster.rage_total_seconds_left = 0;
	monster.rage_minutes_left = 0;
	monster.rage_seconds_left = 0;
	monster.rage_timer_percentage = 0;

	monster.position = Vector3f.new(0, 0, 0);
	monster.distance = 0;

	monster.name = "Large Monster";
	monster.size = 1;
	monster.small_border = 0;
	monster.big_border = 5;
	monster.king_border = 10;
	monster.crown = "";

	monster.parts = {};

	monster.ailments = ailments.init_ailments();

	monster.rider_id = -1;

	large_monster.init(monster, enemy);
	large_monster.init_static_UI(monster);
	large_monster.init_dynamic_UI(monster);
	large_monster.init_highlighted_UI(monster);

	if large_monster.list[enemy] == nil then
		large_monster.list[enemy] = monster;
	end

	large_monster.update_position(enemy, monster);
	large_monster.update(enemy, monster);

	local physical_param = large_monster.update_health(enemy, monster);
	large_monster.update_parts(enemy, monster, physical_param);
	large_monster.update_stamina(enemy, monster, nil);
	large_monster.update_stamina_timer(enemy, monster, nil);
	large_monster.update_rage(enemy, monster, nil);
	large_monster.update_rage_timer(enemy, monster, nil);

	return monster;
end

function large_monster.get_monster(enemy)
	local monster = large_monster.list[enemy];
	if monster == nil then
		monster = large_monster.new(enemy);
	end
	return monster;
end

local enemy_character_base_type_def = sdk.find_type_definition("snow.enemy.EnemyCharacterBase");

local enemy_type_field = enemy_character_base_type_def:get_field("<EnemyType>k__BackingField");
local get_monster_list_register_scale_method = enemy_character_base_type_def:get_method("get_MonsterListRegisterScale");

local message_manager_type_def = sdk.find_type_definition("snow.gui.MessageManager");
local get_enemy_name_message_method = message_manager_type_def:get_method("getEnemyNameMessage");

local enemy_manager_type_def = sdk.find_type_definition("snow.enemy.EnemyManager");
local find_enemy_size_info_method = enemy_manager_type_def:get_method("findEnemySizeInfo");

local size_info_type = find_enemy_size_info_method:get_return_type();
local get_small_border_method = size_info_type:get_method("get_SmallBorder");
local get_big_border_method = size_info_type:get_method("get_BigBorder");
local get_king_border_method = size_info_type:get_method("get_KingBorder");

local get_set_info_method = enemy_character_base_type_def:get_method("get_SetInfo");

local set_info_type = get_set_info_method:get_return_type();
local get_unique_id_method = set_info_type:get_method("get_UniqueId");

function large_monster.init(monster, enemy)
	local enemy_type = enemy_type_field:get_data(enemy);
	if enemy_type == nil then
		customization_menu.status = "No enemy type";
		return;
	end

	monster.id = enemy_type;

	local enemy_name = get_enemy_name_message_method:call(singletons.message_manager, enemy_type);
	if enemy_name ~= nil then
		monster.name = enemy_name;
	end
	
	local set_info = get_set_info_method:call(enemy);
	if set_info ~= nil then
		local unique_id = get_unique_id_method:call(set_info);
		if unique_id ~= nil then
			monster.unique_id = unique_id;
		end
	end

	local size_info = find_enemy_size_info_method:call(singletons.enemy_manager, enemy_type);
	if size_info ~= nil then
		local small_border = get_small_border_method:call(size_info);
		local big_border = get_big_border_method:call(size_info);
		local king_border = get_king_border_method:call(size_info);

		local size = get_monster_list_register_scale_method:call(enemy);

		if small_border ~= nil then
			monster.small_border = small_border;
		end

		if big_border ~= nil then
			monster.big_border = big_border;
		end

		if king_border ~= nil then
			monster.king_border = king_border;
		end

		if size ~= nil then
			monster.size = size;
		end

		if monster.size <= monster.small_border then
			monster.crown = language.current_language.UI.mini;
		elseif monster.size >= monster.king_border then
			monster.crown = language.current_language.UI.gold;
		elseif monster.size >= monster.big_border then
			monster.crown = language.current_language.UI.silver;
		end
	end
end

function large_monster.init_dynamic_UI(monster)
	local cached_config = config.current_config.large_monster_UI.dynamic;
	local global_scale_modifier = config.current_config.global_settings.modifiers.global_scale_modifier;

	monster.dynamic_name_label = table_helpers.deep_copy(cached_config.monster_name_label);

	monster.health_dynamic_UI = health_UI_entity.new(
		cached_config.health.visibility,
		cached_config.health.bar,
		cached_config.health.text_label,
		cached_config.health.value_label,
		cached_config.health.percentage_label
	);

	monster.health_dynamic_UI.bar.capture_line.offset.x = monster.health_dynamic_UI.bar.capture_line.offset.x * global_scale_modifier;
	monster.health_dynamic_UI.bar.capture_line.offset.y = monster.health_dynamic_UI.bar.capture_line.offset.y * global_scale_modifier;
	monster.health_dynamic_UI.bar.capture_line.size.width = monster.health_dynamic_UI.bar.capture_line.size.width * global_scale_modifier;
	monster.health_dynamic_UI.bar.capture_line.size.height = monster.health_dynamic_UI.bar.capture_line.size.height * global_scale_modifier;

	monster.health_dynamic_UI.bar.colors = cached_config.health.bar.normal_colors;

	monster.stamina_dynamic_UI = stamina_UI_entity.new(
		cached_config.stamina.visibility,
		cached_config.stamina.bar,
		cached_config.stamina.text_label,
		cached_config.stamina.value_label,
		cached_config.stamina.percentage_label,
		cached_config.stamina.timer_label
	);

	monster.rage_dynamic_UI = rage_UI_entity.new(
		cached_config.rage.visibility,
		cached_config.rage.bar,
		cached_config.rage.text_label,
		cached_config.rage.value_label,
		cached_config.rage.percentage_label,
		cached_config.rage.timer_label
	);

	for REpart, part in pairs(monster.parts) do
		body_part.init_dynamic_UI(part);
	end

	monster.ailment_dynamic_UI = ailment_UI_entity.new(
		cached_config.ailments.visibility,
		cached_config.ailments.bar,
		cached_config.ailments.ailment_name_label,
		cached_config.ailments.text_label,
		cached_config.ailments.value_label,
		cached_config.ailments.percentage_label,
		cached_config.ailments.timer_label
	);

	ailments.init_ailment_buildup_dynamic_UI(monster.ailments);
end

function large_monster.init_static_UI(monster)
	local cached_config = config.current_config.large_monster_UI.static;
	local global_scale_modifier = config.current_config.global_settings.modifiers.global_scale_modifier;

	monster.static_name_label = table_helpers.deep_copy(cached_config.monster_name_label);
	
	monster.static_name_label.offset.x = monster.static_name_label.offset.x * global_scale_modifier;
	monster.static_name_label.offset.y = monster.static_name_label.offset.y * global_scale_modifier;

	monster.health_static_UI = health_UI_entity.new(
		cached_config.health.visibility,
		cached_config.health.bar,
		cached_config.health.text_label,
		cached_config.health.value_label,
		cached_config.health.percentage_label
	);

	monster.health_static_UI.bar.capture_line.offset.x = monster.health_static_UI.bar.capture_line.offset.x * global_scale_modifier;
	monster.health_static_UI.bar.capture_line.offset.y = monster.health_static_UI.bar.capture_line.offset.y * global_scale_modifier;
	monster.health_static_UI.bar.capture_line.size.width = monster.health_static_UI.bar.capture_line.size.width * global_scale_modifier;
	monster.health_static_UI.bar.capture_line.size.height = monster.health_static_UI.bar.capture_line.size.height * global_scale_modifier;

	monster.health_static_UI.bar.colors = cached_config.health.bar.normal_colors;

	monster.stamina_static_UI = stamina_UI_entity.new(
		cached_config.stamina.visibility,
		cached_config.stamina.bar,
		cached_config.stamina.text_label,
		cached_config.stamina.value_label,
		cached_config.stamina.percentage_label,
		cached_config.stamina.timer_label
	);
	
	monster.rage_static_UI = rage_UI_entity.new(
		cached_config.rage.visibility,
		cached_config.rage.bar,
		cached_config.rage.text_label,
		cached_config.rage.value_label,
		cached_config.rage.percentage_label,
		cached_config.rage.timer_label
	);
	
	for REpart, part in pairs(monster.parts) do
		body_part.init_static_UI(part);
	end

	monster.ailment_static_UI = ailment_UI_entity.new(
		cached_config.ailments.visibility,
		cached_config.ailments.bar,
		cached_config.ailments.ailment_name_label,
		cached_config.ailments.text_label,
		cached_config.ailments.value_label,
		cached_config.ailments.percentage_label,
		cached_config.ailments.timer_label
	);

	ailments.init_ailment_buildup_static_UI(monster.ailments);
end

function large_monster.init_highlighted_UI(monster)
	local cached_config = config.current_config.large_monster_UI.highlighted;
	local global_scale_modifier = config.current_config.global_settings.modifiers.global_scale_modifier;

	monster.highlighted_name_label = table_helpers.deep_copy(cached_config.monster_name_label);

	monster.health_highlighted_UI = health_UI_entity.new(
		cached_config.health.visibility,
		cached_config.health.bar,
		cached_config.health.text_label,
		cached_config.health.value_label,
		cached_config.health.percentage_label
	);

	monster.health_highlighted_UI.bar.capture_line.offset.x = monster.health_highlighted_UI.bar.capture_line.offset.x * global_scale_modifier;
	monster.health_highlighted_UI.bar.capture_line.offset.y = monster.health_highlighted_UI.bar.capture_line.offset.y * global_scale_modifier;
	monster.health_highlighted_UI.bar.capture_line.size.width = monster.health_highlighted_UI.bar.capture_line.size.width * global_scale_modifier;
	monster.health_highlighted_UI.bar.capture_line.size.height = monster.health_highlighted_UI.bar.capture_line.size.height * global_scale_modifier;

	monster.health_highlighted_UI.bar.colors = cached_config.health.bar.normal_colors;

	monster.stamina_highlighted_UI = stamina_UI_entity.new(
		cached_config.stamina.visibility,
		cached_config.stamina.bar,
		cached_config.stamina.text_label,
		cached_config.stamina.value_label,
		cached_config.stamina.percentage_label,
		cached_config.stamina.timer_label
	);
	
	monster.rage_highlighted_UI = rage_UI_entity.new(
		cached_config.rage.visibility,
		cached_config.rage.bar,
		cached_config.rage.text_label,
		cached_config.rage.value_label,
		cached_config.rage.percentage_label,
		cached_config.rage.timer_label
	);
	
	for REpart, part in pairs(monster.parts) do
		body_part.init_highlighted_UI(part);
	end

	monster.ailment_highlighted_UI = ailment_UI_entity.new(
		cached_config.ailments.visibility,
		cached_config.ailments.bar,
		cached_config.ailments.ailment_name_label,
		cached_config.ailments.text_label,
		cached_config.ailments.value_label,
		cached_config.ailments.percentage_label,
		cached_config.ailments.timer_label
	);

	ailments.init_ailment_buildup_highlighted_UI(monster.ailments);
end

local physical_param_field = enemy_character_base_type_def:get_field("<PhysicalParam>k__BackingField");
local stamina_param_field = enemy_character_base_type_def:get_field("<StaminaParam>k__BackingField");
local anger_param_field = enemy_character_base_type_def:get_field("<AngerParam>k__BackingField");
local damage_param_field = enemy_character_base_type_def:get_field("<DamageParam>k__BackingField");

local check_die_method = enemy_character_base_type_def:get_method("checkDie");
local is_disp_icon_mini_map_method = enemy_character_base_type_def:get_method("isDispIconMiniMap");

local physical_param_type = physical_param_field:get_type();
local get_vital_method = physical_param_type:get_method("getVital");
local get_capture_hp_vital_method = physical_param_type:get_method("get_CaptureHpVital");

local vital_param_type = get_vital_method:get_return_type();
local get_current_method = vital_param_type:get_method("get_Current");
local get_max_method = vital_param_type:get_method("get_Max");

local stamina_param_type = stamina_param_field:get_type();
local is_tired_method = stamina_param_type:get_method("isTired");
local get_stamina_method = stamina_param_type:get_method("getStamina");
local get_max_stamina_method = stamina_param_type:get_method("getMaxStamina");

local get_remaining_tired_time_method = stamina_param_type:get_method("getStaminaRemainingTime");
local get_total_tired_time_method = stamina_param_type:get_method("get_TiredSec");

local anger_param_type = anger_param_field:get_type();
local is_anger_method = anger_param_type:get_method("isAnger");
local get_anger_point_method = anger_param_type:get_method("get_AngerPoint");
local get_limit_anger_method = anger_param_type:get_method("get_LimitAnger");

local get_remaining_anger_time_method = anger_param_type:get_method("getAngerRemainingTime");
local get_total_anger_time_method = anger_param_type:get_method("get_TimerAnger");

local mario_param_field = enemy_character_base_type_def:get_field("<MarioParam>k__BackingField");

local mario_param_type = mario_param_field:get_type();
local get_is_marionette_method = mario_param_type:get_method("get_IsMarionette");
local get_mario_player_index_method = mario_param_type:get_method("get_MarioPlayerIndex");

local get_pos_field = enemy_character_base_type_def:get_method("get_Pos");

function large_monster.update_position(enemy, monster)
	if not config.current_config.large_monster_UI.dynamic.enabled and config.current_config.large_monster_UI.static.sorting.type ~= "Distance" then
		return;
	end

	local position = get_pos_field:call(enemy);
	if position ~= nil then
		monster.position = position;
	end
end

-- Code by coavins
function large_monster.update_all_riders()
	for enemy, monster in pairs(large_monster.list) do
		-- get marionette rider
		local mario_param = enemy:get_field("<MarioParam>k__BackingField");
		if mario_param ~= nil then
			local is_marionette = get_is_marionette_method:call(mario_param);
			if is_marionette then
				local player_id = get_mario_player_index_method:call(mario_param);
				if monster.rider_id ~= player_id then
					monster.rider_id = player_id;
				end
			end
		end
	end
	
end

function large_monster.update(enemy, monster)
	local cached_config = config.current_config.large_monster_UI;

	if not cached_config.dynamic.enabled
	and not cached_config.static.enabled
	and not cached_config.highlighted.enabled then
		return;
	end

	local dead_or_captured = check_die_method:call(enemy);
	monster.dead_or_captured = (dead_or_captured == nil and false) or dead_or_captured;
	local is_disp_icon_mini_map = is_disp_icon_mini_map_method:call(enemy);
	monster.is_disp_icon_mini_map = (is_disp_icon_mini_map == nil and false) or is_disp_icon_mini_map;

	ailments.update_ailments(enemy, monster);
	
end

function large_monster.update_health(enemy, monster)
	local cached_config = config.current_config.large_monster_UI;

	if not cached_config.dynamic.enabled
	and not cached_config.static.enabled
	and not cached_config.highlighted.enabled then
		return;
	end

	if not cached_config.dynamic.health.visibility
	and not cached_config.static.health.visibility
	and not cached_config.highlighted.health.visibility then
		return nil;
	end

	local physical_param = physical_param_field:get_data(enemy)
	if physical_param == nil then
		customization_menu.status = "No physical param";
		return nil;
	end

	local vital_param = get_vital_method:call(physical_param, 0, 0);
	if vital_param == nil then
		customization_menu.status = "No vital param";
		return nil;
	end

	monster.health = get_current_method:call(vital_param) or monster.health;
	monster.max_health = get_max_method:call(vital_param) or monster.max_health;
	monster.capture_health = get_capture_hp_vital_method:call(physical_param) or monster.capture_health;

	monster.missing_health = monster.max_health - monster.health;
	if monster.max_health ~= 0 then
		monster.health_percentage = monster.health / monster.max_health;
		monster.capture_percentage = monster.capture_health / monster.max_health;
	end

	return physical_param;
end

function large_monster.update_stamina(enemy, monster, stamina_param)
	local cached_config = config.current_config.large_monster_UI;

	if not cached_config.dynamic.enabled
	and not cached_config.static.enabled
	and not cached_config.highlighted.enabled then
		return;
	end

	if not cached_config.dynamic.stamina.visibility
	and not cached_config.static.stamina.visibility
	and not cached_config.highlighted.stamina.visibility then
		return;
	end

	if stamina_param == nil then
		stamina_param = stamina_param_field:get_data(enemy);
		if stamina_param == nil then
			customization_menu.status = "No stamina param";
			return;
		end
	end

	local is_tired = is_tired_method:call(stamina_param, false);
	if is_tired ~= nil then
		monster.is_tired = is_tired;
	end

	monster.stamina  = get_stamina_method:call(stamina_param) or monster.stamina;
	monster.max_stamina  = get_max_stamina_method:call(stamina_param) or monster.max_stamina;

	monster.missing_stamina = monster.max_stamina - monster.stamina;
	if monster.max_stamina  ~= 0 then
		monster.stamina_percentage = monster.stamina / monster.max_stamina;
	end
end

function large_monster.update_stamina_timer(enemy, monster, stamina_param)
	local cached_config = config.current_config.large_monster_UI;

	if not cached_config.dynamic.enabled
	and not cached_config.static.enabled
	and not cached_config.highlighted.enabled then
		return;
	end
	
	if not cached_config.dynamic.stamina.visibility
	and not cached_config.static.stamina.visibility
	and not cached_config.highlighted.stamina.visibility then
		return;
	end

	if stamina_param == nil then
		stamina_param = stamina_param_field:get_data(enemy);
		if stamina_param == nil then
			customization_menu.status = "No stamina param";
			return;
		end
	end

	local is_tired = is_tired_method:call(stamina_param, false);
	if is_tired ~= nil then
		monster.is_tired = is_tired;
	end


	monster.tired_timer = get_remaining_tired_time_method:call(stamina_param) or monster.tired_timer;
	monster.tired_duration = get_total_tired_time_method:call(stamina_param) or monster.tired_duration;

	if monster.is_tired then
		monster.tired_total_seconds_left = monster.tired_timer;
		if monster.tired_total_seconds_left < 0 then
			monster.tired_total_seconds_left = 0;
		end

		monster.tired_minutes_left = math.floor(monster.tired_total_seconds_left / 60);
		monster.tired_seconds_left = monster.tired_total_seconds_left - 60 * monster.tired_minutes_left;

		if monster.tired_duration ~= 0 then
			monster.tired_timer_percentage = monster.tired_total_seconds_left / monster.tired_duration;
		end
	end
end

function large_monster.update_rage(enemy, monster, anger_param)
	local cached_config = config.current_config.large_monster_UI;

	if not cached_config.dynamic.enabled
	and not cached_config.static.enabled
	and not cached_config.highlighted.enabled then
		return;
	end

	if not cached_config.dynamic.rage.visibility
	and not cached_config.static.rage.visibility
	and not cached_config.highlighted.rage.visibility then
		return;
	end

	if anger_param == nil then
		anger_param = anger_param_field:get_data(enemy);
		if anger_param == nil then
			customization_menu.status = "No anger param";
			return;
		end
	end


	monster.rage_point = get_anger_point_method:call(anger_param) or monster.rage_point;
	monster.rage_limit = get_limit_anger_method:call(anger_param)or monster.rage_limit;
	
	if monster.rage_limit ~= 0 then
		monster.rage_percentage = monster.rage_point / monster.rage_limit;
	end
end

function large_monster.update_rage_timer(enemy, monster, anger_param)
	local cached_config = config.current_config.large_monster_UI;

	if not cached_config.dynamic.enabled
	and not cached_config.static.enabled
	and not cached_config.highlighted.enabled then
		return;
	end

	if not cached_config.dynamic.rage.visibility
	and not cached_config.static.rage.visibility
	and not cached_config.highlighted.rage.visibility then
		return;
	end

	if anger_param == nil then
		anger_param = anger_param_field:get_data(enemy);
		if anger_param == nil then
			customization_menu.status = "No anger param";
			return;
		end
	end

	local is_in_rage = is_anger_method:call(anger_param);
	if is_in_rage ~= nil then
		monster.is_in_rage = is_in_rage;
	end
	
	monster.rage_timer = get_remaining_anger_time_method:call(anger_param) or monster.rage_timer;
	monster.rage_duration = get_total_anger_time_method:call(anger_param) or monster.rage_duration;

	if monster.is_in_rage then
		monster.rage_total_seconds_left = monster.rage_timer;
		if monster.rage_total_seconds_left < 0 then
			monster.rage_total_seconds_left = 0;
		end

		monster.rage_minutes_left = math.floor(monster.rage_total_seconds_left / 60);
		monster.rage_seconds_left = monster.rage_total_seconds_left - 60 * monster.rage_minutes_left;

		if monster.rage_duration ~= 0 then
			monster.rage_timer_percentage = monster.rage_total_seconds_left / monster.rage_duration;
		end
	end
end

function large_monster.update_parts(enemy, monster, physical_param)
	local cached_config = config.current_config.large_monster_UI;

	if not cached_config.dynamic.enabled
	and not cached_config.static.enabled
	and not cached_config.highlighted.enabled then
		return;
	end

	if not cached_config.dynamic.body_parts.visibility
	and not cached_config.static.body_parts.visibility
	and not cached_config.highlighted.body_parts.visibility then
		return;
	end

	if not cached_config.dynamic.body_parts.part_health.visibility
	and not cached_config.dynamic.body_parts.part_break.visibility
	and not cached_config.dynamic.body_parts.part_loss.visibility
	and not cached_config.static.body_parts.part_health.visibility
	and not cached_config.static.body_parts.part_break.visibility
	and not cached_config.static.body_parts.part_loss.visibility
	and not cached_config.highlighted.body_parts.part_health.visibility
	and not cached_config.highlighted.body_parts.part_break.visibility
	and not  cached_config.highlighted.body_parts.part_loss.visibility then
		return;
	end

	if physical_param == nil then
		physical_param = physical_param_field:get_data(enemy)
		if physical_param == nil then
			customization_menu.status = "No physical param";
			return nil;
		end
	end

	local damage_param = damage_param_field:get_data(enemy);
	if damage_param == nil then
		customization_menu.status = "No damage param";
		return;
	end

	local enemy_parts_damage_info = damage_param:get_field("_EnemyPartsDamageInfo");
	if enemy_parts_damage_info == nil then
		customization_menu.status = "No parts damage info";
		return;
	end

	local enemy_parts_info_array = enemy_parts_damage_info:call("get_PartsInfo");
	if enemy_parts_info_array == nil then
		customization_menu.status = "No parts damage info array";
		return;
	end

	local enemy_parts_info_array_size = enemy_parts_info_array:get_size();

	local part_id = 1;
	for i = 0, enemy_parts_info_array_size - 1 do
		local enemy_parts_info = enemy_parts_info_array[i];
		
		local part = monster.parts[part_id];
		if part == nil then
			local part_name = part_names.get_part_name(monster.id, part_id);
			if part_name == nil then
				goto continue;
			else
				part = body_part.new(part_id, part_name);
				monster.parts[part_id] = part;
			end
		end
		
		if cached_config.dynamic.body_parts.part_health.visibility
		or cached_config.static.body_parts.part_health.visibility
		or cached_config.highlighted.body_parts.part_health.visibility then
			local part_vital = physical_param:call("getVital", 1, i);
			if part_vital ~= nil then
				local part_current = part_vital:call("get_Current") or -1;
				local part_max = part_vital:call("get_Max") or -1;

				body_part.update_flinch(part, part_current, part_max);
			
			end
		end

		if cached_config.dynamic.body_parts.part_break.visibility
		or cached_config.static.body_parts.part_break.visibility
		or cached_config.highlighted.body_parts.part_break.visibility then
			local part_break_vital = physical_param:call("getVital", 2, i);
			if part_break_vital ~= nil then
				local part_break_current = part_break_vital:call("get_Current") or -1;
				local part_break_max = part_break_vital:call("get_Max") or -1;
				local part_break_count = -1;
				local part_break_max_count = -1;

				if enemy_parts_info ~= nil then
					part_break_count = enemy_parts_info:call("get_PartsBreakDamageLevel") or part_break_count;
					part_break_max_count = enemy_parts_info:call("get_PartsBreakDamageMaxLevel") or part_break_max_count;
				end
				
				body_part.update_break(part, part_break_current, part_break_max, part_break_count, part_break_max_count)
			end
		end
		
		if cached_config.dynamic.body_parts.part_loss.visibility
		or cached_config.static.body_parts.part_loss.visibility
		or cached_config.highlighted.body_parts.part_loss.visibility then
			local part_loss_vital = physical_param:call("getVital", 3, i);
			if part_loss_vital ~= nil then
				local part_loss_current = part_loss_vital:call("get_Current") or -1;
				local part_loss_max = part_loss_vital:call("get_Max") or -1;
				local is_severed = false;

				if enemy_parts_info ~= nil then
					local _is_severed = enemy_parts_info:call("get_PartsLossState");
					if _is_severed ~= nil then
						is_severed = _is_severed;
					end
				end
				
				body_part.update_loss(part, part_loss_current, part_loss_max, is_severed)
			end
		end
		
		part_id = part_id + 1;
		::continue::
	end
end

function large_monster.draw_dynamic(monster, position_on_screen, opacity_scale)
	local cached_config = config.current_config.large_monster_UI.dynamic;
	local global_scale_modifier = config.current_config.global_settings.modifiers.global_scale_modifier;

	local monster_name_text = "";
	if cached_config.monster_name_label.include.monster_name then
		monster_name_text = string.format("%s ", monster.name);
	end

	if cached_config.monster_name_label.include.monster_id then
		monster_name_text = monster_name_text .. tostring(monster.id) .. " ";
	end

	if cached_config.monster_name_label.include.crown and monster.crown ~= "" then
		monster_name_text = monster_name_text .. string.format("%s ", monster.crown);
	end
	if cached_config.monster_name_label.include.size then
		monster_name_text = monster_name_text .. string.format("#%.0f ", 100 * monster.size);
	end

	if cached_config.monster_name_label.include.scrown_thresholds then
		monster_name_text = monster_name_text .. string.format("<=%.0f >=%.0f >=%.0f", 100 * monster.small_border,
				100 * monster.big_border, 100 * monster.king_border);
	end

	if monster.health < monster.capture_health then
		monster.health_dynamic_UI.bar.colors = cached_config.health.bar.capture_colors;
	else
		monster.health_dynamic_UI.bar.colors = cached_config.health.bar.normal_colors;
	end

	drawing.draw_label(monster.dynamic_name_label, position_on_screen, opacity_scale, monster_name_text);

	local health_position_on_screen = {
		x = position_on_screen.x + cached_config.health.offset.x * global_scale_modifier,
		y = position_on_screen.y + cached_config.health.offset.y * global_scale_modifier
	};

	local stamina_position_on_screen = {
		x = position_on_screen.x + cached_config.stamina.offset.x * global_scale_modifier,
		y = position_on_screen.y + cached_config.stamina.offset.y * global_scale_modifier
	};

	local rage_position_on_screen = {
		x = position_on_screen.x + cached_config.rage.offset.x * global_scale_modifier,
		y = position_on_screen.y + cached_config.rage.offset.y * global_scale_modifier
	};

	local parts_position_on_screen = {
		x = position_on_screen.x + cached_config.body_parts.offset.x * global_scale_modifier,
		y = position_on_screen.y + cached_config.body_parts.offset.y * global_scale_modifier
	};

	local ailments_position_on_screen = {
		x = position_on_screen.x + cached_config.ailments.offset.x * global_scale_modifier,
		y = position_on_screen.y + cached_config.ailments.offset.y * global_scale_modifier
	};

	local ailment_buildups_position_on_screen = {
		x = position_on_screen.x + cached_config.ailment_buildups.offset.x * global_scale_modifier,
		y = position_on_screen.y + cached_config.ailment_buildups.offset.y * global_scale_modifier
	};
	
	health_UI_entity.draw(monster, monster.health_dynamic_UI, health_position_on_screen, opacity_scale);
	drawing.draw_capture_line(monster.health_dynamic_UI.bar, health_position_on_screen, opacity_scale, monster.capture_percentage);

	stamina_UI_entity.draw(monster, monster.stamina_dynamic_UI, stamina_position_on_screen, opacity_scale);
	rage_UI_entity.draw(monster, monster.rage_dynamic_UI, rage_position_on_screen, opacity_scale);

	local last_part_position_on_screen = body_part.draw_dynamic(monster, parts_position_on_screen, opacity_scale);

	if cached_config.ailments.settings.offset_is_relative_to_parts then
		if last_part_position_on_screen ~= nil then
			ailments_position_on_screen = {
				x = last_part_position_on_screen.x + config.current_config.large_monster_UI.highlighted.ailments.relative_offset.x * global_scale_modifier,
				y = last_part_position_on_screen.y + config.current_config.large_monster_UI.highlighted.ailments.relative_offset.y * global_scale_modifier
			};
		end
	end

	ailments.draw_dynamic(monster, ailments_position_on_screen, opacity_scale);
	ailment_buildup.draw_dynamic(monster, ailment_buildups_position_on_screen, opacity_scale);
end

function large_monster.draw_static(monster, position_on_screen, opacity_scale)
	local cached_config = config.current_config.large_monster_UI.static;
	local global_scale_modifier = config.current_config.global_settings.modifiers.global_scale_modifier;

	local monster_name_text = "";
	if cached_config.monster_name_label.include.monster_name then
		monster_name_text = string.format("%s ", monster.name);
	end

	if cached_config.monster_name_label.include.monster_id then
		monster_name_text = monster_name_text .. tostring(monster.id) .. " ";
	end

	if cached_config.monster_name_label.include.crown and monster.crown ~= "" then
		monster_name_text = monster_name_text .. string.format("%s ", monster.crown);
	end
	if cached_config.monster_name_label.include.size then
		monster_name_text = monster_name_text .. string.format("#%.0f ", 100 * monster.size);
	end

	if cached_config.monster_name_label.include.scrown_thresholds then
		monster_name_text = monster_name_text .. string.format("<=%.0f >=%.0f >=%.0f", 100 * monster.small_border,
				100 * monster.big_border, 100 * monster.king_border);
	end

	if monster.health < monster.capture_health then
		monster.health_static_UI.bar.colors = cached_config.health.bar.capture_colors;
	else
		monster.health_static_UI.bar.colors = cached_config.health.bar.normal_colors;
	end

	drawing.draw_label(monster.static_name_label, position_on_screen, opacity_scale, monster_name_text);

	local health_position_on_screen = {
		x = position_on_screen.x + cached_config.health.offset.x * global_scale_modifier,
		y = position_on_screen.y + cached_config.health.offset.y * global_scale_modifier
	};

	local stamina_position_on_screen = {
		x = position_on_screen.x + cached_config.stamina.offset.x * global_scale_modifier,
		y = position_on_screen.y + cached_config.stamina.offset.y * global_scale_modifier
	};

	local rage_position_on_screen = {
		x = position_on_screen.x + cached_config.rage.offset.x * global_scale_modifier,
		y = position_on_screen.y + cached_config.rage.offset.y * global_scale_modifier
	};

	local parts_position_on_screen = {
		x = position_on_screen.x + cached_config.body_parts.offset.x * global_scale_modifier,
		y = position_on_screen.y + cached_config.body_parts.offset.y * global_scale_modifier
	};

	local ailments_position_on_screen = {
		x = position_on_screen.x + cached_config.ailments.offset.x * global_scale_modifier,
		y = position_on_screen.y + cached_config.ailments.offset.y * global_scale_modifier
	};

	local ailment_buildups_position_on_screen = {
		x = position_on_screen.x + cached_config.ailment_buildups.offset.x * global_scale_modifier,
		y = position_on_screen.y + cached_config.ailment_buildups.offset.y * global_scale_modifier
	};

	health_UI_entity.draw(monster, monster.health_static_UI, health_position_on_screen, opacity_scale);
	drawing.draw_capture_line(monster.health_static_UI.bar, health_position_on_screen, opacity_scale, monster.capture_percentage);

	stamina_UI_entity.draw(monster, monster.stamina_static_UI, stamina_position_on_screen, opacity_scale);
	rage_UI_entity.draw(monster, monster.rage_static_UI, rage_position_on_screen, opacity_scale);

	local last_part_position_on_screen = body_part.draw_static(monster, parts_position_on_screen, opacity_scale);

	if cached_config.ailments.settings.offset_is_relative_to_parts then
		if last_part_position_on_screen ~= nil then
			ailments_position_on_screen = {
				x = last_part_position_on_screen.x + cached_config.ailments.relative_offset.x * global_scale_modifier,
				y = last_part_position_on_screen.y + cached_config.ailments.relative_offset.y * global_scale_modifier
			};
		end
	end

	ailments.draw_static(monster, ailments_position_on_screen, opacity_scale);
	ailment_buildup.draw_static(monster, ailment_buildups_position_on_screen, opacity_scale);
end

function large_monster.draw_highlighted(monster, position_on_screen, opacity_scale)
	local cached_config = config.current_config.large_monster_UI.highlighted;
	local global_scale_modifier = config.current_config.global_settings.modifiers.global_scale_modifier;

	local monster_name_text = "";
	if cached_config.monster_name_label.include.monster_name then
		monster_name_text = string.format("%s ", monster.name);
	end

	if cached_config.monster_name_label.include.monster_id then
		monster_name_text = monster_name_text .. tostring(monster.id) .. " ";
	end

	if cached_config.monster_name_label.include.crown and monster.crown ~= "" then
		monster_name_text = monster_name_text .. string.format("%s ", monster.crown);
	end
	if cached_config.monster_name_label.include.size then
		monster_name_text = monster_name_text .. string.format("#%.0f ", 100 * monster.size);
	end

	if cached_config.monster_name_label.include.scrown_thresholds then
		monster_name_text = monster_name_text .. string.format("<=%.0f >=%.0f >=%.0f", 100 * monster.small_border,
				100 * monster.big_border, 100 * monster.king_border);
	end

	if monster.health < monster.capture_health then
		monster.health_highlighted_UI.bar.colors = cached_config.health.bar.capture_colors;
	else
		monster.health_highlighted_UI.bar.colors = cached_config.health.bar.normal_colors;
	end

	drawing.draw_label(monster.highlighted_name_label, position_on_screen, opacity_scale, monster_name_text);

	local health_position_on_screen = {
		x = position_on_screen.x + cached_config.health.offset.x * global_scale_modifier,
		y = position_on_screen.y + cached_config.health.offset.y * global_scale_modifier
	};

	local stamina_position_on_screen = {
		x = position_on_screen.x + cached_config.stamina.offset.x * global_scale_modifier,
		y = position_on_screen.y + cached_config.stamina.offset.y * global_scale_modifier
	};

	local rage_position_on_screen = {
		x = position_on_screen.x + cached_config.rage.offset.x * global_scale_modifier,
		y = position_on_screen.y + cached_config.rage.offset.y * global_scale_modifier
	};

	local parts_position_on_screen = {
		x = position_on_screen.x + cached_config.body_parts.offset.x * global_scale_modifier,
		y = position_on_screen.y + cached_config.body_parts.offset.y * global_scale_modifier
	};

	local ailments_position_on_screen = {
		x = position_on_screen.x + cached_config.ailments.offset.x * global_scale_modifier,
		y = position_on_screen.y + cached_config.ailments.offset.y * global_scale_modifier
	};

	local ailment_buildups_position_on_screen = {
		x = position_on_screen.x + cached_config.ailment_buildups.offset.x * global_scale_modifier,
		y = position_on_screen.y + cached_config.ailment_buildups.offset.y * global_scale_modifier
	};

	health_UI_entity.draw(monster, monster.health_highlighted_UI, health_position_on_screen, opacity_scale);
	drawing.draw_capture_line(monster.health_highlighted_UI.bar, health_position_on_screen, opacity_scale, monster.capture_percentage);

	stamina_UI_entity.draw(monster, monster.stamina_highlighted_UI, stamina_position_on_screen, opacity_scale);
	rage_UI_entity.draw(monster, monster.rage_highlighted_UI, rage_position_on_screen, opacity_scale);

	local last_part_position_on_screen = body_part.draw_highlighted(monster, parts_position_on_screen, opacity_scale);

	if cached_config.ailments.settings.offset_is_relative_to_parts then
		if last_part_position_on_screen ~= nil then
			ailments_position_on_screen = {
				x = last_part_position_on_screen.x + cached_config.ailments.relative_offset.x * global_scale_modifier,
				y = last_part_position_on_screen.y + cached_config.ailments.relative_offset.y * global_scale_modifier
			};
		end
	end

	ailments.draw_highlighted(monster, ailments_position_on_screen, opacity_scale);
	ailment_buildup.draw_highlighted(monster, ailment_buildups_position_on_screen, opacity_scale);
end

function  large_monster.init_list()
	large_monster.list = {};
end

function large_monster.init_module()
	singletons = require("MHR_Overlay.Game_Handler.singletons");
	customization_menu = require("MHR_Overlay.UI.customization_menu");
	language = require("MHR_Overlay.Misc.language");
	config = require("MHR_Overlay.Misc.config");
	table_helpers = require("MHR_Overlay.Misc.table_helpers");
	body_part = require("MHR_Overlay.Monsters.body_part");
	health_UI_entity = require("MHR_Overlay.UI.UI_Entities.health_UI_entity");
	stamina_UI_entity = require("MHR_Overlay.UI.UI_Entities.stamina_UI_entity");
	rage_UI_entity = require("MHR_Overlay.UI.UI_Entities.rage_UI_entity");
	ailment_UI_entity = require("MHR_Overlay.UI.UI_Entities.ailment_UI_entity");

	screen = require("MHR_Overlay.Game_Handler.screen");
	drawing = require("MHR_Overlay.UI.drawing");
	part_names = require("MHR_Overlay.Misc.part_names");
	ailments = require("MHR_Overlay.Monsters.ailments");
	player = require("MHR_Overlay.Damage_Meter.player");
	time = require("MHR_Overlay.Game_Handler.time");
	ailment_buildup = require("MHR_Overlay.Monsters.ailment_buildup");
end

return large_monster;