# Copyright (C) <2021>  <Reston Stanton>

# ##### BEGIN GPL LICENSE BLOCK #####
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTIBILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
# General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program. If not, see <http://www.gnu.org/licenses/>.
#
# ##### END GPL LICENSE BLOCK #####

import bpy

from . import default
from ..Shared import ui, uiCore, modifiers, material, particles

"""
 ██████   ██████            ███                ██████████                                       ███
░░██████ ██████            ░░░                ░░███░░░░███                                     ░░░
 ░███░█████░███   ██████   ████  ████████      ░███   ░░███  ██████  █████████████    ██████   ████  ████████
 ░███░░███ ░███  ░░░░░███ ░░███ ░░███░░███     ░███    ░███ ███░░███░░███░░███░░███  ░░░░░███ ░░███ ░░███░░███
 ░███ ░░░  ░███   ███████  ░███  ░███ ░███     ░███    ░███░███ ░███ ░███ ░███ ░███   ███████  ░███  ░███ ░███
 ░███      ░███  ███░░███  ░███  ░███ ░███     ░███    ███ ░███ ░███ ░███ ░███ ░███  ███░░███  ░███  ░███ ░███
 █████     █████░░████████ █████ ████ █████    ██████████  ░░██████  █████░███ █████░░████████ █████ ████ █████
░░░░░     ░░░░░  ░░░░░░░░ ░░░░░ ░░░░ ░░░░░    ░░░░░░░░░░    ░░░░░░  ░░░░░ ░░░ ░░░░░  ░░░░░░░░ ░░░░░ ░░░░ ░░░░░
"""


def draw_domain(self: uiCore.Core, context):
    domain = context.scene.MantaPro.active_domain
    modifier_settings = modifiers.get_modifier_domain(domain=domain)

    self.set_layout("col")
    self.enabled(ui.get_baking(modifier_settings, "ANY"))

    self.prop(modifier_settings, "resolution_max", text="Resolution Divisions")
    self.prop(modifier_settings, "time_scale")
    self.prop(modifier_settings, "cfl_condition", text="CFL Number")

    self.set_layout("col", align=True, reset=True)
    self.enabled(ui.get_baking(modifier_settings, "DATA"))
    self.icon_prop(modifier_settings, "use_adaptive_timesteps")
    if modifier_settings.use_adaptive_timesteps:
        self.prop(modifier_settings, "timesteps_max", text="Timesteps Maximum")
        self.prop(modifier_settings, "timesteps_min", text="Timesteps Minimum")

    self.set_layout("col", align=True, boxed=True)
    if context.scene.use_gravity:
        self.label("Using Scene Gravity")
    else:
        self.prop(modifier_settings, "gravity")

    self.set_layout("col")
    self.icon_prop(modifier_settings, "delete_in_obstacle",
                   text="Delete in Obstacles")

    if modifier_settings.cache_type == 'MODULAR':
        self.separator(1)
        ui.bake_data(self, modifier_settings, "LIQUID")


# Blender
def draw_liquid(self: uiCore.Core, context):
    domain = context.scene.MantaPro.active_domain
    modifier_settings = modifiers.get_modifier_domain(domain=domain)
    self.set_layout("col")
    self.enabled(ui.get_baking(modifier_settings, "DATA"))
    self.label_prop(modifier_settings, "simulation_method")
    self.prop(modifier_settings, "flip_ratio")
    self.prop(modifier_settings, "sys_particle_maximum")

    self.set_layout("col", align=True)
    self.prop(modifier_settings, "particle_radius", text="Particle Radius")
    self.prop(modifier_settings, "particle_number", text="Particle Sampleing")
    self.prop(modifier_settings, "particle_randomness",
              text="Particle Randomness")

    self.set_layout("col", align=True)
    self.prop(modifier_settings, "particle_max", text="Particles Maximum")
    self.prop(modifier_settings, "particle_min", text="Particles Minimum")

    self.set_layout("col")
    self.prop(modifier_settings, "particle_band_width",
              text="Marrow Band Width")

    self.set_layout("col", align=True)
    self.icon_prop(modifier_settings, "use_fractions")
    if modifier_settings.use_fractions:
        self.prop(modifier_settings, "fractions_distance", text="Obstacle Distance")
        self.prop(modifier_settings, "fractions_threshold", text="Obstacle Threshold")


def draw_viscosity(self: uiCore.Core, context):
    domain = context.scene.MantaPro.active_domain
    modifier_settings = modifiers.get_modifier_domain(domain=domain)

    self.set_layout("col")
    self.enabled(ui.get_baking(modifier_settings, "ANY"))
    if modifier_settings.use_viscosity:
        self.prop(modifier_settings, "viscosity_value",)
    else:
        self.label(text="Viscosity is disabled")


def draw_diffusion(self: uiCore.Core, context):
    domain = context.scene.MantaPro.active_domain
    modifier_settings = modifiers.get_modifier_domain(domain=domain)

    self.set_layout("col", align=True)
    self.enabled(ui.get_baking(modifier_settings, "ANY"))
    if modifier_settings.use_diffusion:
        self.prop(modifier_settings, "viscosity_base", text="Base")
        self.prop(modifier_settings, "viscosity_exponent",
                  text="Exponent", slider=True)
        self.separator()
        self.prop(modifier_settings, "surface_tension", text="Surface Tension")
    else:
        self.label(text="Diffusion is disabled")


def draw_mesh(self: uiCore.Core, context):
    domain = context.scene.MantaPro.active_domain
    modifier_settings = modifiers.get_modifier_domain(domain=domain)

    self.set_layout("col")
    self.enabled(ui.get_baking(modifier_settings, "MESH"))
    self.prop(modifier_settings, "mesh_scale", text="Upres Factor")
    self.prop(modifier_settings, "mesh_particle_radius", text="Particle Radius")

    self.separator(0.4)
    self.icon_prop(modifier_settings, "use_speed_vectors",
                   text="Use Speed Vectors")
    self.separator()
    self.label_prop(modifier_settings, "mesh_generator", text="Mesh Generator")

    self.set_layout("col", align=True)
    self.prop(modifier_settings, "mesh_smoothen_pos", text="Smoothing Positive")
    self.prop(modifier_settings, "mesh_smoothen_neg", text="Smoothing Negative")

    self.separator()
    self.prop(modifier_settings, "mesh_concave_upper", text="Concavity Upper")
    self.prop(modifier_settings, "mesh_concave_lower", text="Concavity Lower")
    self.separator(2)

    if modifier_settings.cache_type == 'MODULAR':
        ui.bake_mesh(self, modifier_settings, "LIQUID")


def draw_cache(self: uiCore.Core, context):
    domain = context.scene.MantaPro.active_domain
    modifier_settings = modifiers.get_modifier_domain(domain=domain)
    self.set_layout("col")
    self.prop(modifier_settings, "cache_directory", text="")

    self.set_layout("col", align=True)
    self.prop(modifier_settings, "cache_frame_start", text="Frame Start")
    self.prop(modifier_settings, "cache_frame_end", text="Frame End")

    if modifier_settings.cache_type != 'REPLAY':
        self.separator(0.5)
        self.prop(modifier_settings, "cache_frame_offset")

    self.set_layout("col")
    self.label_prop(modifier_settings, "cache_type", text="Type")

    self.ui_reset()
    self.icon_prop(modifier_settings, "cache_resumable", text="Is Resumable")

    self.ui_reset()
    if modifier_settings.cache_type == 'REPLAY':
        self.operator("mantapro.replay_clear", text="Clear Cache", icon="LOOP_BACK")

    self.ui_reset()
    self.enabled(ui.get_baking(modifier_settings, "DATA"))
    self.label_prop(modifier_settings, "cache_data_format", text="Format Volumes")

    self.ui_reset()
    self.enabled(ui.get_baking(modifier_settings, "MESH"))
    self.label_prop(modifier_settings, "cache_mesh_format", text="Format Meshes")

    ui.cache_worning(self, modifier_settings)

    ui.bake_all(self, modifier_settings, "GAS")


def draw_cache_advanced(self: uiCore.Core, context):
    domain = context.scene.MantaPro.active_domain
    modifier_settings = modifiers.get_modifier_domain(domain=domain)
    self.set_layout("col", align=True)
    self.enabled(not modifier_settings.has_cache_baked_any)
    self.label_prop(modifier_settings, "openvdb_cache_compress_type", text="Compression Volumes")
    self.label_prop(modifier_settings, "openvdb_data_depth", text="Precision Volumes")
# end Blender


def draw_material(self: uiCore.Core, context):
    default.draw_material(self, context)


"""
 ██████   ██████            ███                   ██████  ████
░░██████ ██████            ░░░                   ███░░███░░███
 ░███░█████░███   ██████   ████  ████████       ░███ ░░░  ░███   ██████  █████ ███ █████
 ░███░░███ ░███  ░░░░░███ ░░███ ░░███░░███     ███████    ░███  ███░░███░░███ ░███░░███
 ░███ ░░░  ░███   ███████  ░███  ░███ ░███    ░░░███░     ░███ ░███ ░███ ░███ ░███ ░███
 ░███      ░███  ███░░███  ░███  ░███ ░███      ░███      ░███ ░███ ░███ ░░███████████
 █████     █████░░████████ █████ ████ █████     █████     █████░░██████   ░░████░████
░░░░░     ░░░░░  ░░░░░░░░ ░░░░░ ░░░░ ░░░░░     ░░░░░     ░░░░░  ░░░░░░     ░░░░ ░░░░
"""


def draw_flow(self: uiCore.Core, context):
    default.draw_flow(self, context)


"""
 ██████   ██████            ███                            ██████     ██████                     █████
░░██████ ██████            ░░░                            ███░░███   ███░░███                   ░░███
 ░███░█████░███   ██████   ████  ████████       ██████   ░███ ░░░   ░███ ░░░   ██████   ██████  ███████    ██████  ████████
 ░███░░███ ░███  ░░░░░███ ░░███ ░░███░░███     ███░░███ ███████    ███████    ███░░███ ███░░███░░░███░    ███░░███░░███░░███
 ░███ ░░░  ░███   ███████  ░███  ░███ ░███    ░███████ ░░░███░    ░░░███░    ░███████ ░███ ░░░   ░███    ░███ ░███ ░███ ░░░
 ░███      ░███  ███░░███  ░███  ░███ ░███    ░███░░░    ░███       ░███     ░███░░░  ░███  ███  ░███ ███░███ ░███ ░███
 █████     █████░░████████ █████ ████ █████   ░░██████   █████      █████    ░░██████ ░░██████   ░░█████ ░░██████  █████
░░░░░     ░░░░░  ░░░░░░░░ ░░░░░ ░░░░ ░░░░░     ░░░░░░   ░░░░░      ░░░░░      ░░░░░░   ░░░░░░     ░░░░░   ░░░░░░  ░░░░░

"""


def draw_effector(self: uiCore.Core, context):
    effector = context.scene.MantaPro.active_effector
    modifier_settings = modifiers.find_modifier(effector).effector_settings
    self.set_layout("col")
    self.label_prop(modifier_settings, "effector_type")
    self.separator(0.5)
    self.prop(modifier_settings, "subframes", text="Sampling Subframes")
    self.prop(modifier_settings, "surface_distance", text="Surface Thickness")
    self.separator(0.5)
    self.icon_prop(modifier_settings, "use_effector", text="Use Effector")
    self.icon_prop(modifier_settings, "use_plane_init")


"""
  █████████             █████        ██████████                                       ███
 ███░░░░░███           ░░███        ░░███░░░░███                                     ░░░
░███    ░░░  █████ ████ ░███████     ░███   ░░███  ██████  █████████████    ██████   ████  ████████
░░█████████ ░░███ ░███  ░███░░███    ░███    ░███ ███░░███░░███░░███░░███  ░░░░░███ ░░███ ░░███░░███
 ░░░░░░░░███ ░███ ░███  ░███ ░███    ░███    ░███░███ ░███ ░███ ░███ ░███   ███████  ░███  ░███ ░███
 ███    ░███ ░███ ░███  ░███ ░███    ░███    ███ ░███ ░███ ░███ ░███ ░███  ███░░███  ░███  ░███ ░███
░░█████████  ░░████████ ████████     ██████████  ░░██████  █████░███ █████░░████████ █████ ████ █████
 ░░░░░░░░░    ░░░░░░░░ ░░░░░░░░     ░░░░░░░░░░    ░░░░░░  ░░░░░ ░░░ ░░░░░  ░░░░░░░░ ░░░░░ ░░░░ ░░░░░
"""


def draw_particles(self: uiCore.Core, context):
    domain = context.scene.MantaPro.active_domain
    modifier_settings = modifiers.get_modifier_domain(domain=domain)

    self.set_layout("row", align=True)
    self.prop(modifier_settings, "use_spray_particles")
    self.prop(modifier_settings, "use_foam_particles")
    self.prop(modifier_settings, "use_bubble_particles")

    self.set_layout("col")
    if modifier_settings.use_spray_particles or modifier_settings.use_foam_particles or modifier_settings.use_bubble_particles:
        self.label_prop(modifier_settings,
                        "sndparticle_combined_export", text="Combined Export")

        self.ui_reset()
        self.enabled(ui.get_baking(modifier_settings, "PARTICLE"))
        self.prop(modifier_settings, "particle_scale", text="Upres Factor")

        self.set_layout("col", align=True)
        self.prop(modifier_settings, "sndparticle_potential_max_wavecrest",
                  text="Wave Crest Potential Maximum")
        self.prop(modifier_settings, "sndparticle_potential_min_wavecrest",
                  text="Wave Crest Potential Minimum")

        self.separator()
        self.prop(modifier_settings, "sndparticle_potential_max_trappedair",
                  text="Trapped Air Potential Maximum")
        self.prop(modifier_settings, "sndparticle_potential_min_trappedair",
                  text="Trapped Air Potential Minimum")

        self.separator()
        self.prop(modifier_settings, "sndparticle_potential_max_energy",
                  text="Kinetic Energy Potential Maximum")
        self.prop(modifier_settings, "sndparticle_potential_min_energy",
                  text="Kinetic Energy Potential Minimum")

        self.separator()
        self.prop(modifier_settings, "sndparticle_potential_radius")
        self.prop(modifier_settings, "sndparticle_update_radius",
                  text="Particle Update Radius")

        self.separator()
        self.prop(modifier_settings, "sndparticle_sampling_wavecrest",
                  text="Wave Crest Particle Sampling")
        self.prop(modifier_settings, "sndparticle_sampling_trappedair",
                  text="Trapped Air Particle Sampling")

        self.separator()
        self.prop(modifier_settings, "sndparticle_life_max",
                  text="Particle Life Maximum")
        self.prop(modifier_settings, "sndparticle_life_min",
                  text="Particle Life Minimum")

        self.separator()
        self.prop(modifier_settings, "sndparticle_bubble_buoyancy")
        self.prop(modifier_settings, "sndparticle_bubble_drag")

        self.separator()
        self.label_prop(modifier_settings, "sndparticle_boundary",
                        text="Particles in Boundary")

        if modifier_settings.cache_type == 'MODULAR':
            self.separator(2)
            ui.bake_particles(self, modifier_settings, "LIQUID")
    else:
        self.label(text="Particles are disabled")


def draw_guides(self: uiCore.Core, context):
    self.set_layout("col")
    modifier_settings = modifiers.get_modifier_domain(context)
    self.enabled(ui.get_baking(modifier_settings, "DATA"))

    if modifier_settings.use_guide:
        self.set_layout("col")
        self.prop(modifier_settings, "guide_alpha")
        self.prop(modifier_settings, "guide_beta")
        self.prop(modifier_settings, "guide_vel_factor")
        self.label_prop(modifier_settings, "guide_source")

        if modifier_settings.guide_source == "DOMAIN":
            self.label_prop(modifier_settings, "guide_parent")
        else:
            self.separator(1)
            ui.bake_guides(self, modifier_settings, "LIQUID")
    else:
        self.label(text="Guides are disabled")


def draw_collections(self: uiCore.Core, context):
    domain = context.scene.MantaPro.active_domain
    modifier_settings = modifiers.get_modifier_domain(domain=domain)

    self.set_layout("col")
    self.label_prop(modifier_settings, "fluid_group", text="Flow")
    self.label_prop(modifier_settings, "effector_group", text="Effector")


def draw_border(self: uiCore.Core, context):
    default.draw_border(self, context)


def draw_field(self: uiCore.Core, context):
    domain = context.scene.MantaPro.active_domain
    modifier_settings = modifiers.get_modifier_domain(domain=domain)

    self.set_layout("col")
    self.label_prop(modifier_settings.effector_weights, "collection", text="Effector Collection")
    ui.field_weights(self, modifier_settings)


def draw_view(self: uiCore.Core, context):
    domain = context.scene.MantaPro.active_domain
    modifier_settings = modifiers.get_modifier_domain(domain=domain)

    self.set_layout("col")
    self.prop(modifier_settings, "display_thickness")
    self.label_prop(modifier_settings, "display_interpolation")
    self.separator(0.25)
    if not modifier_settings.use_slice:
        self.prop(modifier_settings, "slice_per_voxel")
    else:
        self.label(text="Slice Per Voxel is disabled")


def draw_view_slice(self: uiCore.Core, context):
    domain = context.scene.MantaPro.active_domain
    modifier_settings = modifiers.get_modifier_domain(domain=domain)

    self.set_layout("col")
    if modifier_settings.use_slice:
        self.label_prop(modifier_settings, "slice_axis", text="Axis")
        self.separator(0.25)
        self.prop(modifier_settings, "slice_depth", slider=True)
        if modifier_settings.display_interpolation == 'CLOSEST' or modifier_settings.color_ramp_field == 'FLAGS':
            self.icon_prop(modifier_settings, "show_gridlines")
        else:
            self.label(text="Gridlines are disabled")

    else:
        self.label(text="Slice is disabled")


def draw_view_grid(self: uiCore.Core, context):
    domain = context.scene.MantaPro.active_domain
    modifier_settings = modifiers.get_modifier_domain(domain=domain)

    self.set_layout("col")
    if modifier_settings.use_color_ramp:
        self.label_prop(modifier_settings, "color_ramp_field", text="Field")

        if modifier_settings.color_ramp_field != 'FLAGS':
            self.prop(modifier_settings, "color_ramp_field_scale")

        color_ramp_compatible = ['FORCE_Z', 'FORCE_Y', 'FORCE_X', 'VELOCITY_Z', 'VELOCITY_Y', 'VELOCITY_X']
        if modifier_settings.color_ramp_field in color_ramp_compatible:
            self.ui_layout.template_color_ramp(modifier_settings, "color_ramp", expand=True)

    else:
        self.label(text="Grid Display is disabled")


def draw_view_vector(self: uiCore.Core, context):
    domain = context.scene.MantaPro.active_domain
    modifier_settings = modifiers.get_modifier_domain(domain=domain)

    self.set_layout("col")

    if modifier_settings.show_velocity:
        self.label_prop(modifier_settings, "vector_display_type", text="Display As")
        if modifier_settings.vector_display_type == 'MAC':
            self.set_layout("col", align=True)
            self.icon_prop(modifier_settings, "vector_show_mac_x")
            self.icon_prop(modifier_settings, "vector_show_mac_y")
            self.icon_prop(modifier_settings, "vector_show_mac_z")
        else:
            self.icon_prop(modifier_settings, "vector_scale_with_magnitude")
        self.label_prop(modifier_settings, "vector_field", text="Field")
        self.prop(modifier_settings, "vector_scale")

        if not modifier_settings.use_guide and modifier_settings.vector_field == 'GUIDE_VELOCITY':
            self.label(text="Enable Guides first! Defaulting to Fluid Velocity", icon='INFO')

    else:
        self.label(text="Vector Display is disabled")


"""
  █████████             █████           ██████  ████
 ███░░░░░███           ░░███           ███░░███░░███
░███    ░░░  █████ ████ ░███████      ░███ ░░░  ░███   ██████  █████ ███ █████
░░█████████ ░░███ ░███  ░███░░███    ███████    ░███  ███░░███░░███ ░███░░███
 ░░░░░░░░███ ░███ ░███  ░███ ░███   ░░░███░     ░███ ░███ ░███ ░███ ░███ ░███
 ███    ░███ ░███ ░███  ░███ ░███     ░███      ░███ ░███ ░███ ░░███████████
░░█████████  ░░████████ ████████      █████     █████░░██████   ░░████░████
 ░░░░░░░░░    ░░░░░░░░ ░░░░░░░░      ░░░░░     ░░░░░  ░░░░░░     ░░░░ ░░░░
"""


def draw_velocity(self: uiCore.Core, context):
    default.draw_velocity(self, context)


def draw_source(self: uiCore.Core, context):
    modifier_settings = modifiers.find_modifier(
        context.scene.MantaPro.active_flow).flow_settings
    self.label_prop(modifier_settings, "flow_source")
    ui.flow_source(self, modifier_settings)


def draw_flow_display(self: uiCore.Core, context):
    default.draw_flow_display(self, context)


"""
  █████████             █████                    ██████     ██████                     █████
 ███░░░░░███           ░░███                    ███░░███   ███░░███                   ░░███
░███    ░░░  █████ ████ ░███████      ██████   ░███ ░░░   ░███ ░░░   ██████   ██████  ███████    ██████  ████████
░░█████████ ░░███ ░███  ░███░░███    ███░░███ ███████    ███████    ███░░███ ███░░███░░░███░    ███░░███░░███░░███
 ░░░░░░░░███ ░███ ░███  ░███ ░███   ░███████ ░░░███░    ░░░███░    ░███████ ░███ ░░░   ░███    ░███ ░███ ░███ ░░░
 ███    ░███ ░███ ░███  ░███ ░███   ░███░░░    ░███       ░███     ░███░░░  ░███  ███  ░███ ███░███ ░███ ░███
░░█████████  ░░████████ ████████    ░░██████   █████      █████    ░░██████ ░░██████   ░░█████ ░░██████  █████
 ░░░░░░░░░    ░░░░░░░░ ░░░░░░░░      ░░░░░░   ░░░░░      ░░░░░      ░░░░░░   ░░░░░░     ░░░░░   ░░░░░░  ░░░░░
"""


def draw_effector_display(self: uiCore.Core, context):
    default.draw_effector_display(self, context)


"""
  █████████             █████                           █████
 ███░░░░░███           ░░███                           ░░███
░███    ░░░  █████ ████ ░███████      █████  █████ ████ ░███████
░░█████████ ░░███ ░███  ░███░░███    ███░░  ░░███ ░███  ░███░░███
 ░░░░░░░░███ ░███ ░███  ░███ ░███   ░░█████  ░███ ░███  ░███ ░███
 ███    ░███ ░███ ░███  ░███ ░███    ░░░░███ ░███ ░███  ░███ ░███
░░█████████  ░░████████ ████████     ██████  ░░████████ ████████
 ░░░░░░░░░    ░░░░░░░░ ░░░░░░░░     ░░░░░░    ░░░░░░░░ ░░░░░░░░
"""


def draw_advanced_particle(self: uiCore.Core, context):
    default.draw_advanced_particle(self, context)


def draw_custom_particle(self: uiCore.Core, context):
    default.draw_custom_particle(self, context)


def draw_advanced_diffusion(self: uiCore.Core, context):
    default.draw_advanced_diffusion(self, context)


def draw_advanced_cache(self: uiCore.Core, context):
    default.draw_advanced_cache(self, context)


def draw_advanced_gravity(self: uiCore.Core, context):
    default.draw_advanced_gravity(self, context)


"""
 ██████   ██████                       █████              ███████████
░░██████ ██████                       ░░███              ░░███░░░░░███
 ░███░█████░███   ██████   ████████   ███████    ██████   ░███    ░███ ████████   ██████
 ░███░░███ ░███  ░░░░░███ ░░███░░███ ░░░███░    ░░░░░███  ░██████████ ░░███░░███ ███░░███
 ░███ ░░░  ░███   ███████  ░███ ░███   ░███      ███████  ░███░░░░░░   ░███ ░░░ ░███ ░███
 ░███      ░███  ███░░███  ░███ ░███   ░███ ███ ███░░███  ░███         ░███     ░███ ░███
 █████     █████░░████████ ████ █████  ░░█████ ░░████████ █████        █████    ░░██████
░░░░░     ░░░░░  ░░░░░░░░ ░░░░ ░░░░░    ░░░░░   ░░░░░░░░ ░░░░░        ░░░░░      ░░░░░░



"""


def draw_tools(self: uiCore.Core, context):
    ui.MantaPro_tools(self, context)


def draw_cache_tools(self: uiCore.Core, context):
    self.operator("mantapro.move_cache", icon="FILE_TICK",
                  settings=[["remove_dir", True]])


def draw_particle_tools(self: uiCore.Core, context):
    domain = context.scene.MantaPro.active_domain
    modifier_settings = modifiers.get_modifier_domain(domain=domain)
    domain_settings = domain.MantaPro
    spray, foam, bubble = particles.find_particle(context)

    self.operator("mantapro_liquid.link_particles",
                  icon="LINKED", text="Link Default Particles")

    self.separator(2)
    self.icon_prop(domain_settings, "link_particle_size",
                   icon_on="LINKED", icon_off="UNLINKED")

    # particel scaling
    if domain_settings.link_particle_size:
        # linked scale
        self.ui_refresh()
        self.prop(domain_settings, "particle_size", slider=True)
    else:
        # unlinked scale
        self.set_layout("col", align=True)
        if modifier_settings.sndparticle_combined_export == "OFF":
            self.prop(domain_settings, "spray_size", slider=True)
            self.prop(domain_settings, "foam_size", slider=True)
            self.prop(domain_settings, "bubble_size", slider=True)
        elif modifier_settings.sndparticle_combined_export == "SPRAY_FOAM":
            self.prop(domain_settings, "bubble_size", slider=True)
            self.prop(domain_settings, "sprayfoam_size", slider=True)
        elif modifier_settings.sndparticle_combined_export == "SPRAY_BUBBLES":
            self.prop(domain_settings, "foam_size", slider=True)
            self.prop(domain_settings, "spraybubble_size", slider=True)
        elif modifier_settings.sndparticle_combined_export == "FOAM_BUBBLES":
            self.prop(domain_settings, "spray_size", slider=True)
            self.prop(domain_settings, "foambubble_size", slider=True)
        else:
            self.prop(domain_settings, "sprayfoambubble_size", slider=True)

    self.set_layout("col", align=True)

    # view amount
    systems = particles.get_sys(domain)
    for i in systems:
        self.prop(i, "display_percentage", slider=True,
                  text=particles.sys_to_name(i) + " Display Amount")

    self.separator(2)
    self.operator("mantapro_liquid.toggle_liquid_particles",
                  icon="VIS_SEL_10")

    self.set_layout("col", align=True)
    self.separator()

    if modifier_settings.use_spray_particles:
        if "Spray Particle System" in domain.modifiers:
            modifer = context.scene.MantaPro.active_domain.modifiers[
                "Spray Particle System"].particle_system.settings
            self.label_prop(modifer, "display_method", text="Spray Display")

    if modifier_settings.use_foam_particles:
        if "Foam Particle System" in domain.modifiers:
            modifer = context.scene.MantaPro.active_domain.modifiers[
                "Foam Particle System"].particle_system.settings
            self.label_prop(modifer, "display_method", text="Foam Display")

    if modifier_settings.use_bubble_particles:
        if "Bubble Particle System" in domain.modifiers:
            modifer = context.scene.MantaPro.active_domain.modifiers[
                "Bubble Particle System"].particle_system.settings
            self.label_prop(modifer, "display_method", text="Bubble Display")

    self.separator()
    self.operator("mantapro_liquid.quick_toggle_particles",
                  icon="PIVOT_ACTIVE")
