# Copyright (C) <2021>  <Reston Stanton>

# ##### BEGIN GPL LICENSE BLOCK #####
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTIBILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
# General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program. If not, see <http://www.gnu.org/licenses/>.
#
# ##### END GPL LICENSE BLOCK #####


import bpy
from .. import Shared
from ..Shared import uiCore, ui_textures
from ..Shared import polls as P

from .. import preferences, Icons, QuickSetup

from ..Liquid import ui as ui_liquid
from .. Gas import ui as ui_gas

import os


"""
 █████  █████ █████       ████   ███           █████
░░███  ░░███ ░░███       ░░███  ░░░           ░░███
 ░███   ░███  ░███        ░███  ████   █████  ███████    █████
 ░███   ░███  ░███        ░███ ░░███  ███░░  ░░░███░    ███░░
 ░███   ░███  ░███        ░███  ░███ ░░█████   ░███    ░░█████
 ░███   ░███  ░███        ░███  ░███  ░░░░███  ░███ ███ ░░░░███
 ░░████████   █████       █████ █████ ██████   ░░█████  ██████
  ░░░░░░░░   ░░░░░       ░░░░░ ░░░░░ ░░░░░░     ░░░░░  ░░░░░░
"""


class OBJECT_UL_flows(bpy.types.UIList):
    MP_order_id = 1

    def draw_item(self, context, layout, data, item, icon, active_data, active_propname):
        if not item.obj:
            return
        ob = data

        domain = Shared.modifiers.get_modifier_domain(context)
        if not item.obj.users_scene:
            layout.alert = True
        if self.layout_type in {'DEFAULT'}:
            if Shared.modifiers.check_type_gas(item.obj, "FLOW"):
                modifier = Shared.modifiers.find_modifier(
                    item.obj).flow_settings
                if modifier.flow_type == "SMOKE":
                    Icon = Icons.get_icon_id("flow_gas_smoke")
                elif modifier.flow_type == "BOTH":
                    Icon = Icons.get_icon_id("flow_gas_both")
                else:
                    Icon = Icons.get_icon_id("flow_gas_fire")
            else:
                Icon = Icons.get_icon_id("flow_liquid")

            if domain:
                if domain.fluid_group:
                    if domain.fluid_group in item.obj.users_collection:
                        layout.enabled = True
                    else:
                        layout.enabled = False
            layout.scale_x = 1
            layout.prop(item.obj, "name", text="",
                        emboss=False, icon_value=Icon)

            if not layout.enabled:
                layout.label(text="Not in " + domain.fluid_group.name)


"""
  █████████             █████        █████               ████
 ███░░░░░███           ░░███        ░░███               ░░███
░███    ░░░  █████ ████ ░███████     ░███████    ██████  ░███  ████████   ██████  ████████   █████
░░█████████ ░░███ ░███  ░███░░███    ░███░░███  ███░░███ ░███ ░░███░░███ ███░░███░░███░░███ ███░░
 ░░░░░░░░███ ░███ ░███  ░███ ░███    ░███ ░███ ░███████  ░███  ░███ ░███░███████  ░███ ░░░ ░░█████
 ███    ░███ ░███ ░███  ░███ ░███    ░███ ░███ ░███░░░   ░███  ░███ ░███░███░░░   ░███      ░░░░███
░░█████████  ░░████████ ████████     ████ █████░░██████  █████ ░███████ ░░██████  █████     ██████
 ░░░░░░░░░    ░░░░░░░░ ░░░░░░░░     ░░░░ ░░░░░  ░░░░░░  ░░░░░  ░███░░░   ░░░░░░  ░░░░░     ░░░░░░
                                                               ░███
                                                               █████
                                                              ░░░░░
"""


class FlowSub:
    bl_parent_id = "VIEW3D_PT_MantaProFlow"
    bl_space_type = 'VIEW_3D'
    bl_region_type = 'UI'
    bl_category = "MantaPro"
    bl_context = "objectmode"


"""
 ███████████ ████                                                                      ████
░░███░░░░░░█░░███                                                                     ░░███
 ░███   █ ░  ░███   ██████  █████ ███ █████    ████████   ██████   ████████    ██████  ░███
 ░███████    ░███  ███░░███░░███ ░███░░███    ░░███░░███ ░░░░░███ ░░███░░███  ███░░███ ░███
 ░███░░░█    ░███ ░███ ░███ ░███ ░███ ░███     ░███ ░███  ███████  ░███ ░███ ░███████  ░███
 ░███  ░     ░███ ░███ ░███ ░░███████████      ░███ ░███ ███░░███  ░███ ░███ ░███░░░   ░███
 █████       █████░░██████   ░░████░████       ░███████ ░░████████ ████ █████░░██████  █████
░░░░░       ░░░░░  ░░░░░░     ░░░░ ░░░░        ░███░░░   ░░░░░░░░ ░░░░ ░░░░░  ░░░░░░  ░░░░░
                                               ░███
                                               █████
                                              ░░░░░
"""


class VIEW3D_PT_MantaProFlowMain(uiCore.Core, bpy.types.Panel):
    bl_label = ""  # MantaPro Flow
    bl_space_type = 'VIEW_3D'
    bl_region_type = 'UI'
    bl_category = "MantaPro"
    bl_context = "objectmode"
    MP_order_id = 7

    def draw_header(self, context):
        row = self.layout.row(align=True)
        row.label(text="MantaPro Flow",
                  icon_value=Icons.get_icon_id("flow_mixed"))

    def draw(self, context):
        settings = context.scene.MantaPro
        layout = self.layout

        self.set_layout("row", align=True)

        col = layout.column()
        col.template_list("OBJECT_UL_flows", "", settings,
                          "flows", settings, "flows_index")

        self.set_layout("row", align=True)

        icon = "BLENDER"

        if settings.active_flow:
            modifier = Shared.modifiers.find_modifier(settings.active_flow)
            if modifier:
                if modifier.flow_settings.flow_type == "LIQUID":
                    icon = "MATFLUID"
                else:
                    icon = "MOD_FLUID"
        try:
            self.label(
                text=settings.flows[settings.flows_index].obj.name, icon=icon)
        except:
            self.label(text="")

        self.operator("mantapro_gas.make_paint", icon="BRUSH_DATA", text="")
        self.operator("mantapro.make_flow", icon="ADD", text="")
        self.operator("mantapro.remove", text="", icon="REMOVE", settings=[
            ["kind", "FLOW"]])
        self.operator("mantapro.find_flow", icon="ZOOM_ALL", text="",)


class VIEW3D_PT_MantaProFlow(uiCore.Core, bpy.types.Panel):
    bl_parent_id = "VIEW3D_PT_MantaProFlowMain"
    bl_space_type = 'VIEW_3D'
    bl_region_type = 'UI'
    bl_category = "MantaPro"
    bl_context = "objectmode"
    bl_label = "settings"
    MP_order_id = 9

    @ classmethod
    def poll(cls, context):
        return P.FLOW(context)

    def draw(self, context):
        flow = context.scene.MantaPro.active_flow
        if Shared.modifiers.check_type_gas(flow, "FLOW"):
            ui_gas.draw(self, context, "flow")
        else:
            ui_liquid.draw(self, context, "flow")
        self.set_layout("col")


"""
 ███████████ █████                                                      █████
░░███░░░░░░█░░███                                                      ░░███
 ░███   █ ░  ░███         ██████  █████ ███ █████     █████  █████ ████ ░███████   █████
 ░███████    ░███        ███░░███░░███ ░███░░███     ███░░  ░░███ ░███  ░███░░███ ███░░
 ░███░░░█    ░███       ░███ ░███ ░███ ░███ ░███    ░░█████  ░███ ░███  ░███ ░███░░█████
 ░███  ░     ░███      █░███ ░███ ░░███████████      ░░░░███ ░███ ░███  ░███ ░███ ░░░░███
 █████       ███████████░░██████   ░░████░████       ██████  ░░████████ ████████  ██████
░░░░░       ░░░░░░░░░░░  ░░░░░░     ░░░░ ░░░░       ░░░░░░    ░░░░░░░░ ░░░░░░░░  ░░░░░░
"""


class VIEW3D_PT_FlowSourceSub(uiCore.Core, FlowSub, bpy.types.Panel):
    bl_label = "Source"
    MP_order_id = 11
    bl_options = {"DEFAULT_CLOSED"}

    @ classmethod
    def poll(cls, context):
        return P.FLOW(context) and not P.UI_BLENDER()

    def draw_header(self, context):
        self.set_layout("row")
        self.label(icon="CON_OBJECTSOLVER")

    def draw(self, context):
        domain = context.scene.MantaPro.active_domain
        if Shared.modifiers.check_category(domain) == "GAS":
            ui_gas.draw(self, context, "source")
        else:
            ui_liquid.draw(self, context, "source")


class VIEW3D_PT_FlowVelocitySub(uiCore.Core, FlowSub, bpy.types.Panel):
    bl_label = "Initial Velocity"
    MP_order_id = 42

    bl_options = {"DEFAULT_CLOSED"}

    @ classmethod
    def poll(cls, context):
        return P.FLOW(context) and not P.FLOW_OUTFLOW(context) and not P.UI_BLENDER()

    def draw_header(self, context):
        self.set_layout("row")
        modifier_settings = Shared.modifiers.find_modifier(
            context.scene.MantaPro.active_flow).flow_settings
        self.label(icon="ARROW_LEFTRIGHT")
        self.prop(modifier_settings, "use_initial_velocity", text="")

    def draw(self, context):
        domain = context.scene.MantaPro.active_domain
        if Shared.modifiers.check_category(domain) == "GAS":
            ui_gas.draw(self, context, "velocity")
        else:
            ui_liquid.draw(self, context, "velocity")


class VIEW3D_PT_FlowDisplaySubGas(uiCore.Core, FlowSub, bpy.types.Panel):
    bl_label = "Display"
    MP_order_id = 79

    bl_options = {"DEFAULT_CLOSED"}

    @ classmethod
    def poll(cls, context):
        return P.FLOW(context)

    def draw_header(self, context):
        self.set_layout("row")
        self.label(icon="RESTRICT_VIEW_OFF")

    def draw(self, context):
        domain = context.scene.MantaPro.active_domain
        if Shared.modifiers.check_category(domain) == "GAS":
            ui_gas.draw(self, context, "flow_display")
        else:
            ui_liquid.draw(self, context, "flow_display")


class VIEW3D_PT_FlowEmberSub(uiCore.Core, FlowSub, bpy.types.Panel):
    bl_label = "Ember"
    MP_order_id = 78

    bl_options = {"DEFAULT_CLOSED"}

    @ classmethod
    def poll(cls, context):
        return P.FLOW(context)

    def draw_header(self, context):
        self.set_layout("row")
        self.label(icon=Icons.get_icon_id("ember"))

    def draw(self, context):
        flow = context.scene.MantaPro.active_flow
        self.set_layout("col", align=True, boxed=True)
        self.ui_layout.scale_y = 0.8
        system = Shared.particles.find_ember_system(flow)

        self.label(text="It is Recommended to Bake the Fluid", icon="INFO")
        self.label(text="Simulation Before Setting up an Ember System")

        self.set_layout("col")
        if not system:
            self.operator("mantapro.setup_ember", icon="ADD")
        else:
            self.operator("mantapro.remove_ember", icon="REMOVE")


"""
 ███████████  ████                          █████
░░███░░░░░███░░███                         ░░███
 ░███    ░███ ░███   ██████  ████████    ███████   ██████  ████████
 ░██████████  ░███  ███░░███░░███░░███  ███░░███  ███░░███░░███░░███
 ░███░░░░░███ ░███ ░███████  ░███ ░███ ░███ ░███ ░███████  ░███ ░░░
 ░███    ░███ ░███ ░███░░░   ░███ ░███ ░███ ░███ ░███░░░   ░███
 ███████████  █████░░██████  ████ █████░░████████░░██████  █████
░░░░░░░░░░░  ░░░░░  ░░░░░░  ░░░░ ░░░░░  ░░░░░░░░  ░░░░░░  ░░░░░
"""


class VIEW3D_PT_FlowSourceBlender(uiCore.Core, FlowSub, bpy.types.Panel):
    bl_label = "Flow Source"
    MP_order_id = 66
    bl_options = {"DEFAULT_CLOSED"}

    @ classmethod
    def poll(cls, context):
        return P.FLOW(context) and P.UI_BLENDER()

    def draw(self, context):
        domain = context.scene.MantaPro.active_domain
        if Shared.modifiers.check_category(domain) == "GAS":
            ui_gas.draw(self, context, "source")
        else:
            ui_liquid.draw(self, context, "source")


class VIEW3D_PT_FlowVelocityBlender(uiCore.Core, FlowSub, bpy.types.Panel):
    bl_label = "Initial Velocity"
    MP_order_id = 67

    @ classmethod
    def poll(cls, context):
        return P.FLOW(context) and not P.FLOW_OUTFLOW(context) and P.UI_BLENDER()

    def draw_header(self, context):
        self.set_layout("row")
        modifier_settings = Shared.modifiers.find_modifier(
            context.scene.MantaPro.active_flow).flow_settings
        self.prop(modifier_settings, "use_initial_velocity", text="")

    def draw(self, context):
        domain = context.scene.MantaPro.active_domain
        if Shared.modifiers.check_category(domain) == "GAS":
            ui_gas.draw(self, context, "velocity")
        else:
            ui_liquid.draw(self, context, "velocity")
