# Copyright (C) <2021>  <Reston Stanton>

# ##### BEGIN GPL LICENSE BLOCK #####
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTIBILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
# General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program. If not, see <http://www.gnu.org/licenses/>.
#
# ##### END GPL LICENSE BLOCK #####


import bpy
from .Shared import modifiers, particles


"""
 █████  █████               █████            █████                █████               ████
░░███  ░░███               ░░███            ░░███                ░░███               ░░███
 ░███   ░███  ████████   ███████   ██████   ███████    ██████     ░███████    ██████  ░███  ████████   ██████  ████████   █████
 ░███   ░███ ░░███░░███ ███░░███  ░░░░░███ ░░░███░    ███░░███    ░███░░███  ███░░███ ░███ ░░███░░███ ███░░███░░███░░███ ███░░
 ░███   ░███  ░███ ░███░███ ░███   ███████   ░███    ░███████     ░███ ░███ ░███████  ░███  ░███ ░███░███████  ░███ ░░░ ░░█████
 ░███   ░███  ░███ ░███░███ ░███  ███░░███   ░███ ███░███░░░      ░███ ░███ ░███░░░   ░███  ░███ ░███░███░░░   ░███      ░░░░███
 ░░████████   ░███████ ░░████████░░████████  ░░█████ ░░██████     ████ █████░░██████  █████ ░███████ ░░██████  █████     ██████
  ░░░░░░░░    ░███░░░   ░░░░░░░░  ░░░░░░░░    ░░░░░   ░░░░░░     ░░░░ ░░░░░  ░░░░░░  ░░░░░  ░███░░░   ░░░░░░  ░░░░░     ░░░░░░
              ░███                                                                          ░███
              █████                                                                         █████
             ░░░░░                                                                         ░░░░░
"""


def change_particle_size(context, domain, kind, size):
    sys = particles.get_sys(domain)
    sim = None
    if kind in particles.sys_name_list:
        for i in sys:
            if kind in i.name:
                sim = i
                break
    else:
        return

    if sim is None:
        return

    if sim.render_type == "OBJECT":
        object = sim.instance_object
        if not object.MantaPro.original_scale:
            object.MantaPro.original_scale = object.scale
            object.MantaPro.new_scale = object.scale

        if (object.scale[0] != object.MantaPro.new_scale[0]) or (object.scale[1] != object.MantaPro.new_scale[1]) or (object.scale[2] != object.MantaPro.new_scale[2]):
            object.MantaPro.original_scale = object.scale

        original_scale = object.MantaPro.original_scale

        object.scale = [original_scale[0] * size,
                        original_scale[1] * size, original_scale[2] * size]
        object.MantaPro.new_scale = object.scale

    if sim.render_type == "COLLECTION":
        coll = sim.instance_collection
        for i in coll.objects:
            object = i

            if not object.MantaPro.original_scale:
                object.MantaPro.original_scale = object.scale
                object.MantaPro.new_scale = object.scale

            if (object.scale[0] != object.MantaPro.new_scale[0]) or (object.scale[1] != object.MantaPro.new_scale[1]) or (object.scale[2] != object.MantaPro.new_scale[2]):
                object.MantaPro.original_scale = object.scale

            original_scale = object.MantaPro.original_scale

            object.scale = [original_scale[0] * size,
                            original_scale[1] * size, original_scale[2] * size]
            object.MantaPro.new_scale = object.scale


"""
 █████  █████               █████            █████
░░███  ░░███               ░░███            ░░███
 ░███   ░███  ████████   ███████   ██████   ███████    ██████
 ░███   ░███ ░░███░░███ ███░░███  ░░░░░███ ░░░███░    ███░░███
 ░███   ░███  ░███ ░███░███ ░███   ███████   ░███    ░███████
 ░███   ░███  ░███ ░███░███ ░███  ███░░███   ░███ ███░███░░░
 ░░████████   ░███████ ░░████████░░████████  ░░█████ ░░██████
  ░░░░░░░░    ░███░░░   ░░░░░░░░  ░░░░░░░░    ░░░░░   ░░░░░░
              ░███
              █████
             ░░░░░
"""

updating = False


def update_flow_type(self, context):

    modifier = modifiers.find_modifier(context.scene.MantaPro.active_flow)
    if modifier:
        modifier_settings = modifier.flow_settings
        modifier_settings.flow_type = self.flow_type


def update_particle_size(self, context):
    domain = context.scene.MantaPro.active_domain
    settings = domain.MantaPro

    size = settings.particle_size
    link = settings.link_particle_size

    spray_size = settings.spray_size
    foam_size = settings.foam_size
    bubble_size = settings.bubble_size
    sprayfoam_size = settings.sprayfoam_size
    spraybubble_size = settings.spraybubble_size
    foambubble_size = settings.foambubble_size
    sprayfoambubble_size = settings.sprayfoambubble_size

    # "SprayFoam", "SprayBubble", "FoamBubble", "SprayFoamBubble"

    if link:
        change_particle_size(context, domain, "Spray", size)
        change_particle_size(context, domain, "Foam", size)
        change_particle_size(context, domain, "Bubble", size)
        change_particle_size(context, domain, "SprayFoam", size)
        change_particle_size(context, domain, "SprayBubble", size)
        change_particle_size(context, domain, "FoamBubble", size)
        change_particle_size(context, domain, "SprayFoamBubble", size)

    else:
        change_particle_size(context, domain, "Spray", spray_size)
        change_particle_size(context, domain, "Foam", foam_size)
        change_particle_size(context, domain, "Bubble", bubble_size)
        change_particle_size(context, domain, "SprayFoam", sprayfoam_size)
        change_particle_size(context, domain, "SprayBubble", spraybubble_size)
        change_particle_size(context, domain, "FoamBubble", foambubble_size)
        change_particle_size(
            context, domain, "SprayFoamBubble", sprayfoambubble_size)


def update_animate_texture(self, context):
    flow = context.scene.MantaPro.active_flow
    speed = self.animate_speed
    frame_start = context.scene.frame_start
    frame_end = context.scene.frame_end

    modifier = modifiers.find_modifier(flow)
    if modifier:
        modifier_settings = modifier.flow_settings

        if self.animate_texture:
            modifier_settings.texture_offset = 0
            modifier_settings.keyframe_insert(
                "texture_offset", frame=frame_start)

            modifier_settings.texture_offset = frame_end * (speed / 100)
            modifier_settings.keyframe_insert(
                "texture_offset", frame=frame_end)

            f_curve = flow.animation_data.action.fcurves.find(
                modifier_settings.path_from_id("texture_offset"))

            for i in f_curve.keyframe_points:
                i.interpolation = "LINEAR"

        else:
            f_curve = flow.animation_data.action.fcurves.find(
                modifier_settings.path_from_id("texture_offset"))
            flow.animation_data.action.fcurves.remove(f_curve)


def update_domains_index(self, context):
    # handle no objects
    if len(self.domains.keys()) < 1:
        return

    # make sure index is not to large
    index = self.domains_index
    while len(self.domains) <= index:
        if index == -1:
            break
        index -= 1

    # set active domain
    item = self.domains[index]
    self.active_domain = item.obj


def update_flows_index(self, context):
    # handle no objects
    if len(self.flows.keys()) < 1:
        return

    # make sure index is not to large
    index = self.flows_index
    while len(self.flows) <= index:
        if index == -1:
            break
        index -= 1

    # set active flow
    item = self.flows[index]
    self.active_flow = item.obj


def update_effectors_index(self, context):
    # handle no objects
    if len(self.effectors.keys()) < 1:
        return

    # make sure index is not to large
    index = self.effectors_index
    while len(self.effectors) <= index:
        if index == -1:
            break
        index -= 1

    # set active effector
    item = self.effectors[index]
    self.active_effector = item.obj


def update_forces_index(self, context):
    # handle no objects
    if len(self.forces.keys()) < 1:
        return

    # make sure index is not to large
    index = self.forces_index
    while len(self.forces) <= index:
        if index == -1:
            break
        index -= 1

    # set active force
    item = self.forces[index]
    self.active_force = item.obj


def update_auto_ops(self, context):
    if self.auto_ops:
        bpy.ops.mantapro.auto_ops("INVOKE_DEFAULT")


"""
 █████        ███           █████                ████
░░███        ░░░           ░░███                ░░███
 ░███        ████   █████  ███████       ██████  ░███   ██████    █████   █████   ██████   █████
 ░███       ░░███  ███░░  ░░░███░       ███░░███ ░███  ░░░░░███  ███░░   ███░░   ███░░███ ███░░
 ░███        ░███ ░░█████   ░███       ░███ ░░░  ░███   ███████ ░░█████ ░░█████ ░███████ ░░█████
 ░███      █ ░███  ░░░░███  ░███ ███   ░███  ███ ░███  ███░░███  ░░░░███ ░░░░███░███░░░   ░░░░███
 ███████████ █████ ██████   ░░█████    ░░██████  █████░░████████ ██████  ██████ ░░██████  ██████
░░░░░░░░░░░ ░░░░░ ░░░░░░     ░░░░░      ░░░░░░  ░░░░░  ░░░░░░░░ ░░░░░░  ░░░░░░   ░░░░░░  ░░░░░░
"""


class MantaProDomains(bpy.types.PropertyGroup):
    obj: bpy.props.PointerProperty(
        type=bpy.types.Object,)
    name: bpy.props.StringProperty(
        name="Name", subtype='NONE', default="default name")
    index: bpy.props.IntProperty(
        name="Index", subtype='NONE', default=0, min=0)


class MantaProFlows(bpy.types.PropertyGroup):
    obj: bpy.props.PointerProperty(
        type=bpy.types.Object,)
    name: bpy.props.StringProperty(
        name="Name", subtype='NONE', default="default name")
    index: bpy.props.IntProperty(
        name="Index", subtype='NONE', default=0, min=0)


class MantaProEffectors(bpy.types.PropertyGroup):
    obj: bpy.props.PointerProperty(
        type=bpy.types.Object,)
    name: bpy.props.StringProperty(
        name="Name", subtype='NONE', default="default name")
    index: bpy.props.IntProperty(
        name="Index", subtype='NONE', default=0, min=0)


class MantaProForces(bpy.types.PropertyGroup):
    obj: bpy.props.PointerProperty(
        type=bpy.types.Object,)
    name: bpy.props.StringProperty(
        name="Name", subtype='NONE', default="default name")
    index: bpy.props.IntProperty(
        name="Index", subtype='NONE', default=0, min=0)


"""
  █████████
 ███░░░░░███
░███    ░░░   ██████   ██████  ████████    ██████     ████████  ████████   ██████  ████████   █████
░░█████████  ███░░███ ███░░███░░███░░███  ███░░███   ░░███░░███░░███░░███ ███░░███░░███░░███ ███░░
 ░░░░░░░░███░███ ░░░ ░███████  ░███ ░███ ░███████     ░███ ░███ ░███ ░░░ ░███ ░███ ░███ ░███░░█████
 ███    ░███░███  ███░███░░░   ░███ ░███ ░███░░░      ░███ ░███ ░███     ░███ ░███ ░███ ░███ ░░░░███
░░█████████ ░░██████ ░░██████  ████ █████░░██████     ░███████  █████    ░░██████  ░███████  ██████
 ░░░░░░░░░   ░░░░░░   ░░░░░░  ░░░░ ░░░░░  ░░░░░░      ░███░░░  ░░░░░      ░░░░░░   ░███░░░  ░░░░░░
                                                      ░███                         ░███
                                                      █████                        █████
                                                     ░░░░░                        ░░░░░
"""


class SelectionProps(bpy.types.PropertyGroup):
    v_domain: bpy.props.PointerProperty(
        name="v_domain", type=bpy.types.Object)
    v_flow: bpy.props.PointerProperty(
        name="v_flow", type=bpy.types.Object)
    v_effector: bpy.props.PointerProperty(
        name="v_effector", type=bpy.types.Object)
    v_force: bpy.props.PointerProperty(
        name="v_force", type=bpy.types.Object)


class AutoOps(bpy.types.PropertyGroup):
    auto_find: bpy.props.BoolProperty(
        name="Use Auto Find", description="automatically find all fluid sim objects in the scene", default=True)
    auto_remove: bpy.props.BoolProperty(
        name="Use Auto Remove", description="automatically remove any object form the lists if it is not in the scene", default=True)
    auto_v_selection: bpy.props.BoolProperty(name="Sync Viewport Selection to Lists",
                                             description="selected objects from the viewport get selected in the lists ", default=False)

    updated_preferences: bpy.props.BoolProperty(default=False)


class SceneProps(bpy.types.PropertyGroup):
    active_domain: bpy.props.PointerProperty(
        name="domain", type=bpy.types.Object)
    active_flow: bpy.props.PointerProperty(
        name="flow", type=bpy.types.Object)
    active_effector: bpy.props.PointerProperty(
        name="effector", type=bpy.types.Object)
    active_force: bpy.props.PointerProperty(
        name="force", type=bpy.types.Object)

    domains: bpy.props.CollectionProperty(type=MantaProDomains)
    flows: bpy.props.CollectionProperty(type=MantaProFlows)
    effectors: bpy.props.CollectionProperty(type=MantaProEffectors)
    forces: bpy.props.CollectionProperty(type=MantaProForces)

    selection: bpy.props.PointerProperty(type=SelectionProps)

    domains_index: bpy.props.IntProperty(
        name="Index", subtype='NONE', default=0, min=0, update=update_domains_index)
    flows_index: bpy.props.IntProperty(
        name="Index", subtype='NONE', default=0, min=0, update=update_flows_index)
    effectors_index: bpy.props.IntProperty(
        name="Index", subtype='NONE', default=0, min=0, update=update_effectors_index)
    forces_index: bpy.props.IntProperty(
        name="Index", subtype='NONE', default=0, min=0, update=update_forces_index)

    auto_ops: bpy.props.BoolProperty(
        name="Use Auto Operations", description="use auto operations that auto find new objects and remove deleted objects", default=False, update=update_auto_ops)

    auto_ops_settings: bpy.props.PointerProperty(type=AutoOps)

    # quick setup save
    quick_setup_save_collection: bpy.props.PointerProperty(
        name="Collection", type=bpy.types.Collection,
        description="The collection that contains all the objects and collections to be saved as a preset")

    quick_setup_save_category: bpy.props.EnumProperty(
        items=[("LIQUID", "Liquid", "Liquid Category", 0),
               ("GAS", "Gas", "Gas Category", 1),
               ("OTHER", "Other", "Other Category", 2)],
        default="OTHER",
        name="Category",
        description="The category that the preset will be displayed under \n (if not OTHER it adds an '_CATEGORY' to the end of the collection name)")

    quick_setup_save_index: bpy.props.IntProperty(
        name="Index", default=4, min=0, max=999,
        description="Index for the showing the preset in lists (higher numbers are placed lower) \n (if two presets have the same index, they are sorted alphabetically)")

    quick_setup_save_name: bpy.props.StringProperty(
        name="Name", default="Preset", description="The name for the preset")


"""
    ███████    █████          ███                     █████
  ███░░░░░███ ░░███          ░░░                     ░░███
 ███     ░░███ ░███████      █████  ██████   ██████  ███████      ████████  ████████   ██████  ████████   █████
░███      ░███ ░███░░███    ░░███  ███░░███ ███░░███░░░███░      ░░███░░███░░███░░███ ███░░███░░███░░███ ███░░
░███      ░███ ░███ ░███     ░███ ░███████ ░███ ░░░   ░███        ░███ ░███ ░███ ░░░ ░███ ░███ ░███ ░███░░█████
░░███     ███  ░███ ░███     ░███ ░███░░░  ░███  ███  ░███ ███    ░███ ░███ ░███     ░███ ░███ ░███ ░███ ░░░░███
 ░░░███████░   ████████      ░███ ░░██████ ░░██████   ░░█████     ░███████  █████    ░░██████  ░███████  ██████
   ░░░░░░░    ░░░░░░░░       ░███  ░░░░░░   ░░░░░░     ░░░░░      ░███░░░  ░░░░░      ░░░░░░   ░███░░░  ░░░░░░
                         ███ ░███                                 ░███                         ░███
                        ░░██████                                  █████                        █████
                         ░░░░░░                                  ░░░░░                        ░░░░░
"""


def update_ember_domain(self, context):
    obj = self.id_data
    force = obj.MantaPro.ember_force_object
    if force:
        force.field.source_object = self.ember_domain


def poll_ember_domain(self, object):
    modifier = modifiers.find_modifier(object)
    if modifier:
        if modifier.fluid_type == "DOMAIN":
            return True
    return False


class ObjectProps(bpy.types.PropertyGroup):

    particle_collection: bpy.props.PointerProperty(type=bpy.types.Collection)

    # particle size values
    size_max = 100
    size_soft_max = 5
    size_default = 1

    # particle size
    link_particle_size: bpy.props.BoolProperty(
        name="Link particle size", default=True, description="link particle size", update=update_particle_size)

    particle_size: bpy.props.FloatProperty(
        name="Particel size", default=size_default, min=0, max=size_max, soft_max=size_soft_max, update=update_particle_size)

    spray_size: bpy.props.FloatProperty(
        name="spray size", default=size_default, min=0, max=size_max, soft_max=size_soft_max, update=update_particle_size)

    foam_size: bpy.props.FloatProperty(
        name="foam size", default=size_default, min=0, max=size_max, soft_max=size_soft_max, update=update_particle_size)

    bubble_size: bpy.props.FloatProperty(
        name="bubble size", default=size_default, min=0, max=size_max, soft_max=size_soft_max, update=update_particle_size)

    sprayfoam_size: bpy.props.FloatProperty(
        name="spray+foam size", default=size_default, min=0, max=size_max, soft_max=size_soft_max, update=update_particle_size)

    spraybubble_size: bpy.props.FloatProperty(
        name="spray+bubble size", default=size_default, min=0, max=size_max, soft_max=size_soft_max, update=update_particle_size)

    foambubble_size: bpy.props.FloatProperty(
        name="foam+bubble size", default=size_default, min=0, max=size_max, soft_max=size_soft_max, update=update_particle_size)

    sprayfoambubble_size: bpy.props.FloatProperty(
        name="spray+foam+bubble size", default=size_default, min=0, max=size_max, soft_max=size_soft_max, update=update_particle_size)

    # Other

    moved_cache: bpy.props.BoolProperty(name="moved cache", default=False)

    locked_cache: bpy.props.BoolProperty(
        name="Cache Lock", description="lock the cache to prevent accidental deletion", default=False)

    original_scale: bpy.props.FloatVectorProperty(precision=4)
    new_scale: bpy.props.FloatVectorProperty(precision=4)

    # flow settings
    flow_type: bpy.props.EnumProperty(
        items=[("FIRE", "Fire", "Add fire", 0),
               ("BOTH", "Fire + Smoke", "Add fire and smoke", 1),
               ("SMOKE", "Smoke", "Add smoke", 2)],

        name="Flow Type",
        update=update_flow_type,)

    animate_texture: bpy.props.BoolProperty(
        name="Animate Texture", default=False, description="animate texture using drivers", update=update_animate_texture)

    animate_speed: bpy.props.FloatProperty(
        name="speed", default=0.5, soft_max=1, soft_min=0, precision=5, update=update_animate_texture)

    ember_particle_object: bpy.props.PointerProperty(type=bpy.types.Object)
    ember_force_object: bpy.props.PointerProperty(type=bpy.types.Object)

    ember_domain: bpy.props.PointerProperty(
        type=bpy.types.Object, update=update_ember_domain, poll=poll_ember_domain)

    # force settings

    force_ember: bpy.props.BoolProperty(default=False)


"""
   █████████           ████  ████                     █████     ███
  ███░░░░░███         ░░███ ░░███                    ░░███     ░░░
 ███     ░░░   ██████  ░███  ░███   ██████   ██████  ███████   ████   ██████  ████████      ████████  ████████   ██████  ████████   █████
░███          ███░░███ ░███  ░███  ███░░███ ███░░███░░░███░   ░░███  ███░░███░░███░░███    ░░███░░███░░███░░███ ███░░███░░███░░███ ███░░
░███         ░███ ░███ ░███  ░███ ░███████ ░███ ░░░   ░███     ░███ ░███ ░███ ░███ ░███     ░███ ░███ ░███ ░░░ ░███ ░███ ░███ ░███░░█████
░░███     ███░███ ░███ ░███  ░███ ░███░░░  ░███  ███  ░███ ███ ░███ ░███ ░███ ░███ ░███     ░███ ░███ ░███     ░███ ░███ ░███ ░███ ░░░░███
 ░░█████████ ░░██████  █████ █████░░██████ ░░██████   ░░█████  █████░░██████  ████ █████    ░███████  █████    ░░██████  ░███████  ██████
  ░░░░░░░░░   ░░░░░░  ░░░░░ ░░░░░  ░░░░░░   ░░░░░░     ░░░░░  ░░░░░  ░░░░░░  ░░░░ ░░░░░     ░███░░░  ░░░░░      ░░░░░░   ░███░░░  ░░░░░░
                                                                                            ░███                         ░███
                                                                                            █████                        █████
                                                                                           ░░░░░                        ░░░░░
"""


def update_use_defaults(self, context):
    global updating
    if not updating:
        updating = True
        if self.use_defaults:
            self.liquid_resolution = True
            self.gas_resolution = True
            self.liquid_sim_speed = True
            self.gas_sim_speed = True
            self.gas_flow_type = True
            self.liquid_set_collection = True
            self.gas_set_collection = True
        else:
            self.liquid_resolution = False
            self.gas_resolution = False
            self.liquid_sim_speed = False
            self.gas_sim_speed = False
            self.gas_flow_type = False
            self.liquid_set_collection = False
            self.gas_set_collection = False
        updating = False


def update_collection_other(self, context):
    global updating
    updating = True
    if not (self.liquid_resolution and self.gas_resolution and self.liquid_sim_speed and self.gas_sim_speed and self.gas_flow_type and self.liquid_set_collection and self.gas_set_collection):
        self.use_defaults = False
    else:
        self.use_defaults = True
    updating = False


class CollectionProps(bpy.types.PropertyGroup):
    use_defaults: bpy.props.BoolProperty(
        name="use all defaults", default=False, update=update_use_defaults)

    use_materials: bpy.props.BoolProperty(name="use materials", default=True,
                                          description="adds the default material to domian objects on import")
    add_particles: bpy.props.BoolProperty(name="add particles", default=True,
                                          description="adds the default particle objects on import for liquid domains")

    liquid_resolution: bpy.props.BoolProperty(
        name="liquid resolution", default=True, update=update_collection_other)
    gas_resolution: bpy.props.BoolProperty(
        name="gas resolution", default=True, update=update_collection_other)
    liquid_sim_speed: bpy.props.BoolProperty(
        name="liquid sim speed", default=True, update=update_collection_other)
    gas_sim_speed: bpy.props.BoolProperty(
        name="gas sim speed", default=True, update=update_collection_other)
    gas_flow_type: bpy.props.BoolProperty(
        name="gas flow type", default=False, update=update_collection_other)
    liquid_set_collection: bpy.props.BoolProperty(
        name="liquid set collection", default=True, update=update_collection_other)
    gas_set_collection: bpy.props.BoolProperty(
        name="gas set collection", default=True, update=update_collection_other)


"""
 ██████   ██████            █████                        ███            ████
░░██████ ██████            ░░███                        ░░░            ░░███
 ░███░█████░███   ██████   ███████    ██████  ████████  ████   ██████   ░███     ████████  ████████   ██████  ████████   █████
 ░███░░███ ░███  ░░░░░███ ░░░███░    ███░░███░░███░░███░░███  ░░░░░███  ░███    ░░███░░███░░███░░███ ███░░███░░███░░███ ███░░
 ░███ ░░░  ░███   ███████   ░███    ░███████  ░███ ░░░  ░███   ███████  ░███     ░███ ░███ ░███ ░░░ ░███ ░███ ░███ ░███░░█████
 ░███      ░███  ███░░███   ░███ ███░███░░░   ░███      ░███  ███░░███  ░███     ░███ ░███ ░███     ░███ ░███ ░███ ░███ ░░░░███
 █████     █████░░████████  ░░█████ ░░██████  █████     █████░░████████ █████    ░███████  █████    ░░██████  ░███████  ██████
░░░░░     ░░░░░  ░░░░░░░░    ░░░░░   ░░░░░░  ░░░░░     ░░░░░  ░░░░░░░░ ░░░░░     ░███░░░  ░░░░░      ░░░░░░   ░███░░░  ░░░░░░
                                                                                 ░███                         ░███
                                                                                 █████                        █████
                                                                                ░░░░░                        ░░░░░
"""


class MaterialProps(bpy.types.PropertyGroup):
    version: bpy.props.FloatProperty(
        name="Material version", default=1, min=1, precision=1)


"""
 ███████████                       █████     ███           ████
░░███░░░░░███                     ░░███     ░░░           ░░███
 ░███    ░███  ██████   ████████  ███████   ████   ██████  ░███   ██████     ████████  ████████   ██████  ████████   █████
 ░██████████  ░░░░░███ ░░███░░███░░░███░   ░░███  ███░░███ ░███  ███░░███   ░░███░░███░░███░░███ ███░░███░░███░░███ ███░░
 ░███░░░░░░    ███████  ░███ ░░░   ░███     ░███ ░███ ░░░  ░███ ░███████     ░███ ░███ ░███ ░░░ ░███ ░███ ░███ ░███░░█████
 ░███         ███░░███  ░███       ░███ ███ ░███ ░███  ███ ░███ ░███░░░      ░███ ░███ ░███     ░███ ░███ ░███ ░███ ░░░░███
 █████       ░░████████ █████      ░░█████  █████░░██████  █████░░██████     ░███████  █████    ░░██████  ░███████  ██████
░░░░░         ░░░░░░░░ ░░░░░        ░░░░░  ░░░░░  ░░░░░░  ░░░░░  ░░░░░░      ░███░░░  ░░░░░      ░░░░░░   ░███░░░  ░░░░░░
                                                                             ░███                         ░███
                                                                             █████                        █████
                                                                            ░░░░░                        ░░░░░
"""


class ParticleProps(bpy.types.PropertyGroup):
    is_ember: bpy.props.BoolProperty(default=False)


"""
                              ███           █████
                             ░░░           ░░███
 ████████   ██████   ███████ ████   █████  ███████    ██████  ████████
░░███░░███ ███░░███ ███░░███░░███  ███░░  ░░░███░    ███░░███░░███░░███
 ░███ ░░░ ░███████ ░███ ░███ ░███ ░░█████   ░███    ░███████  ░███ ░░░
 ░███     ░███░░░  ░███ ░███ ░███  ░░░░███  ░███ ███░███░░░   ░███
 █████    ░░██████ ░░███████ █████ ██████   ░░█████ ░░██████  █████
░░░░░      ░░░░░░   ░░░░░███░░░░░ ░░░░░░     ░░░░░   ░░░░░░  ░░░░░
                    ███ ░███
                   ░░██████
                    ░░░░░░
"""

classes = (
    MantaProDomains,
    MantaProFlows,
    MantaProEffectors,
    MantaProForces,

    SelectionProps,
    AutoOps,

    SceneProps,
    ObjectProps,
    CollectionProps,
    MaterialProps,
    ParticleProps,

)


def register():
    for i in classes:
        bpy.utils.register_class(i)
    bpy.types.Scene.MantaPro = bpy.props.PointerProperty(type=SceneProps)
    bpy.types.Object.MantaPro = bpy.props.PointerProperty(type=ObjectProps)
    bpy.types.Collection.MantaPro = bpy.props.PointerProperty(
        type=CollectionProps)
    bpy.types.Material.MantaPro = bpy.props.PointerProperty(type=MaterialProps)
    bpy.types.ParticleSettings.MantaPro = bpy.props.PointerProperty(
        type=ParticleProps)


def unregister():
    for i in classes:
        bpy.utils.unregister_class(i)
    del bpy.types.Scene.MantaPro
    del bpy.types.Object.MantaPro
    del bpy.types.Collection.MantaPro
    del bpy.types.Material.MantaPro
    del bpy.types.ParticleSettings.MantaPro
