# Copyright (C) <2021>  <Reston Stanton>

# ##### BEGIN GPL LICENSE BLOCK #####
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTIBILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
# General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program. If not, see <http://www.gnu.org/licenses/>.
#
# ##### END GPL LICENSE BLOCK #####

import bpy
import bpy_extras
import os
import zipfile

from ..Shared import modifiers, material, particles, ops
from .. import preferences, Icons

"""
   █████████            █████          ██████                                  █████     ███
  ███░░░░░███          ░░███          ███░░███                                ░░███     ░░░
 ███     ░░░   ██████  ███████       ░███ ░░░  █████ ████ ████████    ██████  ███████   ████   ██████  ████████    █████
░███          ███░░███░░░███░       ███████   ░░███ ░███ ░░███░░███  ███░░███░░░███░   ░░███  ███░░███░░███░░███  ███░░
░███    █████░███████   ░███       ░░░███░     ░███ ░███  ░███ ░███ ░███ ░░░   ░███     ░███ ░███ ░███ ░███ ░███ ░░█████
░░███  ░░███ ░███░░░    ░███ ███     ░███      ░███ ░███  ░███ ░███ ░███  ███  ░███ ███ ░███ ░███ ░███ ░███ ░███  ░░░░███
 ░░█████████ ░░██████   ░░█████      █████     ░░████████ ████ █████░░██████   ░░█████  █████░░██████  ████ █████ ██████
  ░░░░░░░░░   ░░░░░░     ░░░░░      ░░░░░       ░░░░░░░░ ░░░░ ░░░░░  ░░░░░░     ░░░░░  ░░░░░  ░░░░░░  ░░░░ ░░░░░ ░░░░░░
"""

built_in = ["000Default_Gas", "000Default_Liquid", "001Fire_Canvas_Gas", "003Explosion_Gas",
            "001Water_Fall_Liquid", "002Cloud_Gas", "002Splash_Liquid", "003Slime_Liquid"]


def get_all():
    liquid = []
    gas = []
    other = []
    base = os.path.abspath(os.path.dirname(__file__))
    all = os.listdir(base)
    for i in all:
        if i.endswith("Liquid.blend"):
            liquid.append(i)
        elif i.endswith("Gas.blend"):
            gas.append(i)
        elif i.endswith(".blend"):
            other.append(i)

    return convert_all(liquid, gas, other)


def get_custom():
    skip = built_in
    liquid, gas, other = get_all()
    c_liquid = []
    c_gas = []
    c_other = []

    for i in liquid:
        if i in skip:
            continue
        c_liquid.append(i)

    for i in gas:
        if i in skip:
            continue
        c_gas.append(i)

    for i in other:
        if i in skip:
            continue
        c_other.append(i)
    return c_liquid, c_gas, c_other


def convert_all(liquid: list, gas: list, other: list):
    new_liquid = []
    new_gas = []
    new_other = []

    for i in liquid:
        new_liquid.append(i[:-6])

    for i in gas:
        new_gas.append(i[:-6])

    for i in other:
        new_other.append(i[:-6])

    return new_liquid, new_gas, new_other


"""
    ███████                                            █████
  ███░░░░░███                                         ░░███
 ███     ░░███ ████████   ██████  ████████   ██████   ███████    ██████  ████████   █████
░███      ░███░░███░░███ ███░░███░░███░░███ ░░░░░███ ░░░███░    ███░░███░░███░░███ ███░░
░███      ░███ ░███ ░███░███████  ░███ ░░░   ███████   ░███    ░███ ░███ ░███ ░░░ ░░█████
░░███     ███  ░███ ░███░███░░░   ░███      ███░░███   ░███ ███░███ ░███ ░███      ░░░░███
 ░░░███████░   ░███████ ░░██████  █████    ░░████████  ░░█████ ░░██████  █████     ██████
   ░░░░░░░     ░███░░░   ░░░░░░  ░░░░░      ░░░░░░░░    ░░░░░   ░░░░░░  ░░░░░     ░░░░░░
               ░███
               █████
              ░░░░░
"""


class MantaProSetupMenu(bpy.types.Operator):
    bl_idname = "mantapro.quick_setup_menu"
    bl_label = "MantaPro Quick Setup"
    bl_description = "quick setup for simulations"

    bl_options = {'REGISTER', 'UNDO'}

    def invoke(self, context, event):
        wm = context.window_manager
        return wm.invoke_popup(self, width=150)
        return {'FINISHED'}

    def execute(self, context):
        return {'FINISHED'}

    def draw(self, context):
        liquid, gas, other = get_all()
        layout = self.layout

        col = layout.box().column(align=True)
        for i in liquid:
            settings = col.operator(
                "mantapro.quick_setup_op", icon_value=Icons.get_icon_id("domain_liquid"), text=i[3:-6].replace("_", " "))
            settings.preset_name = i

        col = layout.box().column(align=True)
        for i in gas:
            settings = col.operator(
                "mantapro.quick_setup_op", icon_value=Icons.get_icon_id("domain_gas"), text=i[3:-3].replace("_", " "))
            settings.preset_name = i

        if other:
            col = layout.box().column(align=True)
            for i in other:
                settings = col.operator(
                    "mantapro.quick_setup_op", icon_value=Icons.get_icon_id("domain_mixed"), text=i[3:].replace("_", " "))
                settings.preset_name = i


class MantaProQuickSetupOp(bpy.types.Operator):
    bl_idname = "mantapro.quick_setup_op"
    bl_label = "MantaPro Preset Setup"
    bl_description = "quick preset setup for simulations"

    bl_options = {'REGISTER', 'UNDO'}

    preset_name:  bpy.props.StringProperty(
        name="preset_name", default="Default_Liquid")

    def execute(self, context):
        load(context, self.preset_name, self)

        if preferences.get_preferences().auto_ops:
            context.scene.MantaPro.auto_ops = True
        return {'FINISHED'}


class MantaProQuickSetupSaveOp(bpy.types.Operator):
    bl_idname = "mantapro.quick_setup_save_op"
    bl_label = "MantaPro Preset Save"
    bl_description = "save quick preset setup for simulations"

    bl_options = {'REGISTER', 'UNDO'}

    @classmethod
    def poll(self, context):
        return context.scene.MantaPro.quick_setup_save_collection

    def execute(self, context):
        save(self, context)
        return {'FINISHED'}


class MantaProQuickSetupRemoveMenu(bpy.types.Operator):
    bl_idname = "mantapro.quick_setup_remove_menu"
    bl_label = "Remove Preset"
    bl_description = "remove preset"

    bl_options = {'REGISTER', 'UNDO'}

    preset: bpy.props.StringProperty(name="preset")

    def invoke(self, context, event):
        wm = context.window_manager
        return wm.invoke_props_dialog(self, width=300)

    def execute(self, context):
        bpy.ops.mantapro.quick_setup_remove_op(preset=self.preset)
        return {'FINISHED'}

    def draw(self, context):
        layout = self.layout
        layout.alert = True
        layout.label(text="Are you sure you want to remove")
        layout.alert = False
        layout.label(text=self.preset[3:].replace("_", " "))
        layout.alert = True
        layout.label(text="This action is PERMANENT and CANNOT be UNDONE")
        layout.alert = False

        row = layout.row()


class MantaProQuickSetupRemoveOp(bpy.types.Operator):
    bl_idname = "mantapro.quick_setup_remove_op"
    bl_label = "MantaPro Preset Remove"
    bl_description = "remove preset"

    bl_options = {'REGISTER'}

    preset: bpy.props.StringProperty(name="preset")

    def execute(self, context):
        if self.preset:
            remove(self, context, self.preset)
            context.area.tag_redraw()
        return {'FINISHED'}


class MantaProQuickSetupExportOp(bpy.types.Operator, bpy_extras.io_utils.ExportHelper):
    bl_idname = "mantapro.quick_setup_export_op"
    bl_label = "Export"
    bl_description = "export preset setup for simulations"

    bl_options = {'REGISTER'}

    filename_ext = ".zip"
    filepath: bpy.props.StringProperty(default='MantaPro_Quick_Setup')

    compression: bpy.props.EnumProperty(name="Compression", description="compression used for zip file",
                                        items=[("zipfile.ZIP_STORED", "STORED", "No compression", 0),
                                               ("zipfile.ZIP_DEFLATED", "DEFLATED",
                                                "Low compression", 1),
                                               ("zipfile.ZIP_BZIP2", "BZIP2",
                                                "Medium compression", 2),
                                               ("zipfile.ZIP_BZIP2", "LZMA", "High compression", 3)])

    def execute(self, context):
        Export(self, context)
        return {'FINISHED'}


class MantaProQuickSetupImportOp(bpy.types.Operator, bpy_extras.io_utils.ImportHelper):
    bl_idname = "mantapro.quick_setup_import_op"
    bl_label = "Import"
    bl_description = "import preset setup for simulations"

    filter_glob: bpy.props.StringProperty(
        default='*.zip', options={'HIDDEN'})

    filename: bpy.props.StringProperty(default='')

    bl_options = {'REGISTER'}

    def execute(self, context):
        Import(self, context)
        return {'FINISHED'}


"""
 █████                                               █████                ███ ██████████                                           █████          ██████                                  █████     ███
░░███                                               ░░███               ███░ ░░███░░░░░█                                          ░░███          ███░░███                                ░░███     ░░░
 ░███  █████████████   ████████   ██████  ████████  ███████           ███░    ░███  █ ░  █████ █████ ████████  ████████   ██████  ███████       ░███ ░░░  █████ ████ ████████    ██████  ███████   ████   ██████  ████████    █████
 ░███ ░░███░░███░░███ ░░███░░███ ███░░███░░███░░███░░░███░          ███░      ░██████   ░░███ ░░███ ░░███░░███░░███░░███ ███░░███░░░███░       ███████   ░░███ ░███ ░░███░░███  ███░░███░░░███░   ░░███  ███░░███░░███░░███  ███░░
 ░███  ░███ ░███ ░███  ░███ ░███░███ ░███ ░███ ░░░   ░███         ███░        ░███░░█    ░░░█████░   ░███ ░███ ░███ ░░░ ░███ ░███  ░███       ░░░███░     ░███ ░███  ░███ ░███ ░███ ░░░   ░███     ░███ ░███ ░███ ░███ ░███ ░░█████
 ░███  ░███ ░███ ░███  ░███ ░███░███ ░███ ░███       ░███ ███   ███░          ░███ ░   █  ███░░░███  ░███ ░███ ░███     ░███ ░███  ░███ ███     ░███      ░███ ░███  ░███ ░███ ░███  ███  ░███ ███ ░███ ░███ ░███ ░███ ░███  ░░░░███
 █████ █████░███ █████ ░███████ ░░██████  █████      ░░█████  ███░            ██████████ █████ █████ ░███████  █████    ░░██████   ░░█████      █████     ░░████████ ████ █████░░██████   ░░█████  █████░░██████  ████ █████ ██████
░░░░░ ░░░░░ ░░░ ░░░░░  ░███░░░   ░░░░░░  ░░░░░        ░░░░░  ░░░             ░░░░░░░░░░ ░░░░░ ░░░░░  ░███░░░  ░░░░░      ░░░░░░     ░░░░░      ░░░░░       ░░░░░░░░ ░░░░ ░░░░░  ░░░░░░     ░░░░░  ░░░░░  ░░░░░░  ░░░░ ░░░░░ ░░░░░░
                       ░███                                                                          ░███
                       █████                                                                         █████
                      ░░░░░                                                                         ░░░░░
"""


def Import(self, context):
    base = os.path.abspath(os.path.dirname(__file__))
    path = self.filepath

    zip = zipfile.ZipFile(path, "r")
    try:
        for i in zip.namelist():
            if os.path.exists(os.path.join(base, i)):
                self.report({"WARNING"}, i +
                            " already exists and had been skiped")
                continue
            zip.extract(i, base)
    except:
        zip.close()
    zip.close()
    self.report({"INFO"}, "Successfully imported " + bpy.path.basename(path))


def Export(self, context):
    liquid, gas, other = get_custom()
    base = os.path.abspath(os.path.dirname(__file__))
    zip_file = zipfile.ZipFile(
        self.filepath, 'w', compression=eval(self.compression))

    try:
        for i in liquid:
            file_name = i + ".blend"
            file = os.path.join(base, file_name)
            zip_file.write(file, file_name)

        for i in gas:
            file_name = i + ".blend"
            file = os.path.join(base, file_name)
            zip_file.write(file, file_name)

        for i in other:
            file_name = i + ".blend"
            file = os.path.join(base, file_name)
            zip_file.write(file, file_name)
    except:
        zip_file.close()

    zip_file.close()
    self.report({"INFO"}, "Successfully exported presets")


"""
 ███████████                                                                ██████                                  █████     ███
░░███░░░░░███                                                              ███░░███                                ░░███     ░░░
 ░███    ░███   ██████  █████████████    ██████  █████ █████  ██████      ░███ ░░░  █████ ████ ████████    ██████  ███████   ████   ██████  ████████    █████
 ░██████████   ███░░███░░███░░███░░███  ███░░███░░███ ░░███  ███░░███    ███████   ░░███ ░███ ░░███░░███  ███░░███░░░███░   ░░███  ███░░███░░███░░███  ███░░
 ░███░░░░░███ ░███████  ░███ ░███ ░███ ░███ ░███ ░███  ░███ ░███████    ░░░███░     ░███ ░███  ░███ ░███ ░███ ░░░   ░███     ░███ ░███ ░███ ░███ ░███ ░░█████
 ░███    ░███ ░███░░░   ░███ ░███ ░███ ░███ ░███ ░░███ ███  ░███░░░       ░███      ░███ ░███  ░███ ░███ ░███  ███  ░███ ███ ░███ ░███ ░███ ░███ ░███  ░░░░███
 █████   █████░░██████  █████░███ █████░░██████   ░░█████   ░░██████      █████     ░░████████ ████ █████░░██████   ░░█████  █████░░██████  ████ █████ ██████
░░░░░   ░░░░░  ░░░░░░  ░░░░░ ░░░ ░░░░░  ░░░░░░     ░░░░░     ░░░░░░      ░░░░░       ░░░░░░░░ ░░░░ ░░░░░  ░░░░░░     ░░░░░  ░░░░░  ░░░░░░  ░░░░ ░░░░░ ░░░░░░
"""


def remove(self, context, preset):
    base = os.path.abspath(os.path.dirname(__file__))
    file = os.path.join(base, preset + ".blend")
    if not os.path.exists(file):
        self.report({"ERROR"}, "Preset file not found")
        return
    os.remove(file)
    self.report({"WARNING"}, "Successfully removed " +
                preset[3:].replace("_", " "))


"""
  █████████                                       ██████                                  █████     ███
 ███░░░░░███                                     ███░░███                                ░░███     ░░░
░███    ░░░   ██████   █████ █████  ██████      ░███ ░░░  █████ ████ ████████    ██████  ███████   ████   ██████  ████████    █████
░░█████████  ░░░░░███ ░░███ ░░███  ███░░███    ███████   ░░███ ░███ ░░███░░███  ███░░███░░░███░   ░░███  ███░░███░░███░░███  ███░░
 ░░░░░░░░███  ███████  ░███  ░███ ░███████    ░░░███░     ░███ ░███  ░███ ░███ ░███ ░░░   ░███     ░███ ░███ ░███ ░███ ░███ ░░█████
 ███    ░███ ███░░███  ░░███ ███  ░███░░░       ░███      ░███ ░███  ░███ ░███ ░███  ███  ░███ ███ ░███ ░███ ░███ ░███ ░███  ░░░░███
░░█████████ ░░████████  ░░█████   ░░██████      █████     ░░████████ ████ █████░░██████   ░░█████  █████░░██████  ████ █████ ██████
 ░░░░░░░░░   ░░░░░░░░    ░░░░░     ░░░░░░      ░░░░░       ░░░░░░░░ ░░░░ ░░░░░  ░░░░░░     ░░░░░  ░░░░░  ░░░░░░  ░░░░ ░░░░░ ░░░░░░
"""


def save(self, context):
    settings = context.scene.MantaPro
    collection = settings.quick_setup_save_collection
    category = settings.quick_setup_save_category
    index = settings.quick_setup_save_index
    q_name = bpy.path.clean_name(settings.quick_setup_save_name)

    original_name = collection.name
    collection.name = q_name

    index_str = str(index)

    for i in range(3-len(index_str)):
        index_str = "0" + index_str

    name = index_str + collection.name
    if category != "OTHER":
        name += "_" + category.capitalize()
        collection.name += "_" + category.capitalize()

    base = os.path.abspath(os.path.dirname(__file__))
    blendfile = os.path.join(base, name + '.blend')

    # check to see if preset already exists
    if os.path.exists(blendfile):
        self.report({"ERROR"}, "A preset with that name already exists")
        return

    # save collection
    bpy.data.libraries.write(blendfile, {collection}, fake_user=True)

    collection.name = original_name

    self.report({"INFO"}, "Successfully exported preset")


"""
 █████                              █████       ██████                                  █████     ███
░░███                              ░░███       ███░░███                                ░░███     ░░░
 ░███         ██████   ██████    ███████      ░███ ░░░  █████ ████ ████████    ██████  ███████   ████   ██████  ████████    █████
 ░███        ███░░███ ░░░░░███  ███░░███     ███████   ░░███ ░███ ░░███░░███  ███░░███░░░███░   ░░███  ███░░███░░███░░███  ███░░
 ░███       ░███ ░███  ███████ ░███ ░███    ░░░███░     ░███ ░███  ░███ ░███ ░███ ░░░   ░███     ░███ ░███ ░███ ░███ ░███ ░░█████
 ░███      █░███ ░███ ███░░███ ░███ ░███      ░███      ░███ ░███  ░███ ░███ ░███  ███  ░███ ███ ░███ ░███ ░███ ░███ ░███  ░░░░███
 ███████████░░██████ ░░████████░░████████     █████     ░░████████ ████ █████░░██████   ░░█████  █████░░██████  ████ █████ ██████
░░░░░░░░░░░  ░░░░░░   ░░░░░░░░  ░░░░░░░░     ░░░░░       ░░░░░░░░ ░░░░ ░░░░░  ░░░░░░     ░░░░░  ░░░░░  ░░░░░░  ░░░░ ░░░░░ ░░░░░░
"""


def load(context, name: str, self):
    base = os.path.abspath(os.path.dirname(__file__))
    blendfile = os.path.join(base, name + '.blend')

    if not os.path.exists(blendfile):
        self.report(
            {"ERROR"}, "Unable to find preset '" + str(blendfile) + "'")
        return

    # load collection
    with bpy.data.libraries.load(blendfile) as (data_from, data_to):
        data_to.collections = [
            coll for coll in data_from.collections if coll == name[3:]]

    if not data_to.collections:
        self.report(
            {"ERROR"}, "Blendfile name does NOT mach any collections in the file")
        return
    collection = data_to.collections[0]
    context.scene.collection.children.link(collection)
    version = bpy.data.version
    for library in bpy.data.libraries:
        if library.filepath == blendfile:
            version = library.version
    add(context, collection, version)


def add(context, collection, version):
    domains = []
    flows = []
    effectors = []
    forces = []
    other = []

    # loop though main collection and categorize objects
    for obj in collection.all_objects:
        modifier = modifiers.find_modifier(obj)
        if modifier:
            # domain
            if modifier.fluid_type == "DOMAIN":
                ops.set_domain_index(context, obj)
                domains.append(obj)
                continue

            # flow
            elif modifier.fluid_type == "FLOW":
                ops.set_flow_index(context, obj)
                flows.append(obj)
                continue

            # effector
            elif modifier.fluid_type == "EFFECTOR":
                ops.set_effector_index(context, obj)
                effectors.append(obj)
                continue

        if obj.field:
            if obj.field.type != "NONE":
                ops.set_flow_index(context, obj)
                forces.append(obj)
                continue

        other.append(obj)

    # set active collection
    layer_collection = context.view_layer.layer_collection.children[collection.name]
    context.view_layer.active_layer_collection = layer_collection

    # set default
    set_defaults(context, collection, domains,
                 flows, effectors, forces, other, version)


def set_cache():
    name = "GET_CACHE"

    mesh = bpy.data.meshes.new(name)
    object = bpy.data.objects.new(name, mesh)

    modifier = object.modifiers.new(name, "FLUID")
    modifier.fluid_type = "DOMAIN"

    cache = modifier.domain_settings.cache_directory

    bpy.data.objects.remove(object)
    bpy.data.meshes.remove(mesh)

    return cache


def set_defaults(context, collection, domains: list, flows: list, effectors: list, forces: list, other: list, version):
    prefs = preferences.get_preferences()
    settings = collection.MantaPro

    # domains
    for obj in domains:
        modifier_settings = modifiers.get_modifier_domain(domain=obj)
        if modifiers.check_type_gas(obj, "DOMAIN"):
            set_domain_gas(context, modifier_settings, prefs, settings)
        elif modifiers.check_type_liquid(obj, "DOMAIN"):
            set_domain_liquid(context, modifier_settings,
                              prefs, settings)

        # clear cache directory
        modifier_settings.cache_directory = set_cache()

    # flows
    for obj in flows:
        modifier_settings = modifiers.find_modifier(obj).flow_settings
        if modifiers.check_type_gas(obj, "FLOW"):
            set_flow_gas(context, modifier_settings,
                         obj, prefs, settings, version)
        elif modifiers.check_type_liquid(obj, "FLOW"):
            set_flow_liquid(context, modifier_settings,
                            obj, prefs, settings, version)

    # effectors
        # they have no defaults yet

    # forces flow
        # they have no defaults yet

    # move to 3D cursor
    move(context, domains, flows, effectors, forces, other)


def move(context, domains: list, flows: list, effectors: list, forces: list, other: list):
    cursor_location = context.scene.cursor.location

    # save selection
    original_selection = context.selected_objects

    # deselect original
    bpy.ops.object.select_all(action='DESELECT')

    # select domains
    for obj in domains:
        obj.select_set(True)

    # select flows
    for obj in flows:
        obj.select_set(True)

    # select effectors
    for obj in effectors:
        obj.select_set(True)

    # set forces
    for obj in forces:
        obj.select_set(True)

    for obj in other:
        obj.select_set(True)

    # move
    bpy.ops.transform.translate(value=cursor_location, orient_type='GLOBAL',
                                orient_matrix_type='GLOBAL', use_proportional_edit=False)

    # restore original
    bpy.ops.object.select_all(action='DESELECT')

    for obj in original_selection:
        obj.select_set(True)


"""
     █████             ██████                       ████   █████       █████               ████
    ░░███             ███░░███                     ░░███  ░░███       ░░███               ░░███
  ███████   ██████   ░███ ░░░   ██████   █████ ████ ░███  ███████      ░███████    ██████  ░███  ████████   ██████  ████████   █████
 ███░░███  ███░░███ ███████    ░░░░░███ ░░███ ░███  ░███ ░░░███░       ░███░░███  ███░░███ ░███ ░░███░░███ ███░░███░░███░░███ ███░░
░███ ░███ ░███████ ░░░███░      ███████  ░███ ░███  ░███   ░███        ░███ ░███ ░███████  ░███  ░███ ░███░███████  ░███ ░░░ ░░█████
░███ ░███ ░███░░░    ░███      ███░░███  ░███ ░███  ░███   ░███ ███    ░███ ░███ ░███░░░   ░███  ░███ ░███░███░░░   ░███      ░░░░███
░░████████░░██████   █████    ░░████████ ░░████████ █████  ░░█████     ████ █████░░██████  █████ ░███████ ░░██████  █████     ██████
 ░░░░░░░░  ░░░░░░   ░░░░░      ░░░░░░░░   ░░░░░░░░ ░░░░░    ░░░░░     ░░░░ ░░░░░  ░░░░░░  ░░░░░  ░███░░░   ░░░░░░  ░░░░░     ░░░░░░
                                                                                                 ░███
                                                                                                 █████
                                                                                                ░░░░░
"""


def update_velocity(version, velocity_coord):
    if version[0] <= 2:
        if version[1] < 90:
            for i in range(len(velocity_coord)):
                if velocity_coord[i]:
                    velocity_coord[i] = velocity_coord[i] * 12


def set_domain_gas(context, modifier_settings, preferences, settings):
    if settings.gas_resolution:
        modifier_settings.resolution_max = preferences.gas_resolution
    if settings.gas_sim_speed:
        modifier_settings.time_scale = preferences.gas_sim_speed

    if not settings.gas_set_collection:
        modifier_settings.fluid_group = None
        modifier_settings.effector_group = None
        modifier_settings.effector_weights.collection = None

    if settings.use_materials:
        material.add_material(modifier_settings.id_data, "MantaPro Gas")


def set_domain_liquid(context, modifier_settings, preferences, settings):
    if settings.liquid_resolution:
        modifier_settings.resolution_max = preferences.liquid_resolution
    if settings.liquid_sim_speed:
        modifier_settings.time_scale = preferences.liquid_sim_speed

    if not settings.liquid_set_collection:
        if not preferences.liquid_set_collection:
            modifier_settings.fluid_group = None
            modifier_settings.effector_group = None
            modifier_settings.effector_weights.collection = None

    if settings.use_materials:
        material.add_material(modifier_settings.id_data, "MantaPro Liquid")

    if settings.add_particles:
        collection = particles.add_particles(context)[3]
        modifier_settings.id_data.MantaPro.particle_collection = collection


def set_flow_gas(context, modifier_settings, obj, preferences, settings, version):
    if settings.gas_flow_type:
        modifier_settings.flow_type = preferences.gas_flow_type
        obj.MantaPro.flow_type = preferences.gas_flow_type
    update_velocity(version, modifier_settings.velocity_coord)


def set_flow_liquid(context, modifier_settings, obj, preferences, settings, version):
    update_velocity(version, modifier_settings.velocity_coord)


classes = [MantaProQuickSetupOp, MantaProSetupMenu,
           MantaProQuickSetupSaveOp, MantaProQuickSetupRemoveMenu,
           MantaProQuickSetupRemoveOp, MantaProQuickSetupExportOp,
           MantaProQuickSetupImportOp]


def register():
    for i in classes:
        bpy.utils.register_class(i)


def unregister():
    for i in classes:
        bpy.utils.unregister_class(i)
