# Copyright (C) <2021>  <Reston Stanton>

# ##### BEGIN GPL LICENSE BLOCK #####
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTIBILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
# General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program. If not, see <http://www.gnu.org/licenses/>.
#
# ##### END GPL LICENSE BLOCK #####


import bpy
import os


def need_update(domain, version: float):
    if is_material(domain):
        if round(domain.material_slots[0].material.MantaPro.version, 3) == version:
            return False
        else:
            return True


def is_eevee_setup(context):
    if context.engine == "BLENDER_EEVEE":
        if context.scene.eevee.use_ssr:
            if context.scene.eevee.use_ssr_refraction:
                return True
    return False


def add_material(obj, name):
    base = os.path.abspath(os.path.dirname(__file__))
    base = os.path.dirname(base)

    blendfile = os.path.join(base, *('Assets', 'materials.blend'))

    with bpy.data.libraries.load(blendfile) as (data_from, data_to):
        data_to.materials = [m for m in data_from.materials if m == name]

    data_to.materials[0].name = name
    obj.active_material = data_to.materials[0]


def is_material(obj):
    return len(obj.material_slots) >= 1


def get_gas(domain):
    node = None
    if is_material(domain):
        if "MantaPro Gas" in domain.material_slots[0].name:
            if "Principled Volume" in domain.material_slots[0].material.node_tree.nodes:
                node = domain.material_slots[0].material.node_tree.nodes["Principled Volume"]
    return node


def get_liquid_nodes(domain):
    volume = None
    glass = None
    if is_material(domain):
        if "MantaPro Liquid" in domain.material_slots[0].name:
            if "Volume Absorption" in domain.material_slots[0].material.node_tree.nodes:
                volume = domain.material_slots[0].material.node_tree.nodes["Volume Absorption"]
            if "Glass BSDF" in domain.material_slots[0].material.node_tree.nodes:
                glass = domain.material_slots[0].material.node_tree.nodes["Glass BSDF"]
    return volume, glass


def get_name(obj):
    if is_material(obj):
        if "MantaPro Gas" in obj.material_slots[0].name:
            return obj.material_slots[0].name
        elif "MantaPro Liquid" in obj.material_slots[0].name:
            return obj.material_slots[0].name
        elif "MantaPro Ember" in obj.material_slots[0].name:
            return obj.material_slots[0].name


def get_ember_nodes(ember):
    emission = None
    color_ramp = None
    math = None
    eevee_emission = None
    eevee_mix = None
    if is_material(ember):
        if "MantaPro Ember" in ember.material_slots[0].name:
            if "Emission" in ember.material_slots[0].material.node_tree.nodes:
                emission = ember.material_slots[0].material.node_tree.nodes["Emission"]
            if "ColorRamp" in ember.material_slots[0].material.node_tree.nodes:
                color_ramp = ember.material_slots[0].material.node_tree.nodes["ColorRamp"]
            if "Math" in ember.material_slots[0].material.node_tree.nodes:
                math = ember.material_slots[0].material.node_tree.nodes["Math"]
            if "Emission Eevee" in ember.material_slots[0].material.node_tree.nodes:
                eevee_emission = ember.material_slots[0].material.node_tree.nodes["Emission Eevee"]
            if "Mix Shader Eevee" in ember.material_slots[0].material.node_tree.nodes:
                eevee_mix = ember.material_slots[0].material.node_tree.nodes["Mix Shader Eevee"]

    return emission, color_ramp, math, eevee_emission, eevee_mix


"""
 ██████   ██████            ████   █████     ███                    ████
░░██████ ██████            ░░███  ░░███     ░░░                    ░░███
 ░███░█████░███  █████ ████ ░███  ███████   ████   ██████   ██████  ░███   ██████  ████████
 ░███░░███ ░███ ░░███ ░███  ░███ ░░░███░   ░░███  ███░░███ ███░░███ ░███  ███░░███░░███░░███
 ░███ ░░░  ░███  ░███ ░███  ░███   ░███     ░███ ░███ ░░░ ░███ ░███ ░███ ░███ ░███ ░███ ░░░
 ░███      ░███  ░███ ░███  ░███   ░███ ███ ░███ ░███  ███░███ ░███ ░███ ░███ ░███ ░███
 █████     █████ ░░████████ █████  ░░█████  █████░░██████ ░░██████  █████░░██████  █████
░░░░░     ░░░░░   ░░░░░░░░ ░░░░░    ░░░░░  ░░░░░  ░░░░░░   ░░░░░░  ░░░░░  ░░░░░░  ░░░░░
"""


def has_multicolor(domain):
    return "Multicolor" in domain.material_slots[0].material.node_tree.nodes


def is_multicolor(domain, principled_volume):
    node = False
    if has_multicolor(domain):
        if len(principled_volume.inputs['Color'].links) > 0:
            node = principled_volume.inputs['Color'].links[0].from_node.name == "Multicolor"
    return node


def get_multicolor(domain, principled_volume):
    if is_multicolor(domain, principled_volume):
        return domain.material_slots[0].material.node_tree.nodes["Multicolor"]


"""
 ███████████ ████
░░███░░░░░░█░░███                                         ███
 ░███   █ ░  ░███   ██████   █████████████    ██████     ░███
 ░███████    ░███  ░░░░░███ ░░███░░███░░███  ███░░███ ███████████
 ░███░░░█    ░███   ███████  ░███ ░███ ░███ ░███████ ░░░░░███░░░
 ░███  ░     ░███  ███░░███  ░███ ░███ ░███ ░███░░░      ░███
 █████       █████░░████████ █████░███ █████░░██████     ░░░
░░░░░       ░░░░░  ░░░░░░░░ ░░░░░ ░░░ ░░░░░  ░░░░░░
"""


def has_flame(domain):
    return "Flame+" in domain.material_slots[0].material.node_tree.nodes


def is_flame(domain, principled_volume):
    node = False
    if has_flame(domain):
        if len(principled_volume.inputs['Density'].links) > 0 and len(principled_volume.inputs['Temperature'].links) > 0:
            node = (principled_volume.inputs['Temperature'].links[0].from_node.name == "Flame+" and
                    principled_volume.inputs['Density'].links[0].from_node.name == "Flame+")
    return node


def get_flame(domain, principled_volume):
    if is_flame(domain, principled_volume):
        return domain.material_slots[0].material.node_tree.nodes["Flame+"]
