# Copyright (C) <2021>  <Reston Stanton>

# ##### BEGIN GPL LICENSE BLOCK #####
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTIBILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
# General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program. If not, see <http://www.gnu.org/licenses/>.
#
# ##### END GPL LICENSE BLOCK #####


import bpy
from . import material, modifiers

"""
  █████████
 ███░░░░░███
░███    ░░░  █████ ████  █████     ████████    ██████   █████████████    ██████   █████
░░█████████ ░░███ ░███  ███░░     ░░███░░███  ░░░░░███ ░░███░░███░░███  ███░░███ ███░░
 ░░░░░░░░███ ░███ ░███ ░░█████     ░███ ░███   ███████  ░███ ░███ ░███ ░███████ ░░█████
 ███    ░███ ░███ ░███  ░░░░███    ░███ ░███  ███░░███  ░███ ░███ ░███ ░███░░░   ░░░░███
░░█████████  ░░███████  ██████     ████ █████░░████████ █████░███ █████░░██████  ██████
 ░░░░░░░░░    ░░░░░███ ░░░░░░     ░░░░ ░░░░░  ░░░░░░░░ ░░░░░ ░░░ ░░░░░  ░░░░░░  ░░░░░░
              ███ ░███
             ░░██████
              ░░░░░░
"""

sys_name_list = ["Spray", "Foam", "Bubble", "SprayFoam",
                 "SprayBubble", "FoamBubble", "SprayFoamBubble"]
sys_ui_name_list = ["Spray", "Foam", "Bubble", "Spray+Foam",
                    "Spray+Bubble", "Foam+Bubble", "Spray+Foam+Bubble"]


"""
    ██████                                  █████     ███
   ███░░███                                ░░███     ░░░
  ░███ ░░░  █████ ████ ████████    ██████  ███████   ████   ██████  ████████    █████
 ███████   ░░███ ░███ ░░███░░███  ███░░███░░░███░   ░░███  ███░░███░░███░░███  ███░░
░░░███░     ░███ ░███  ░███ ░███ ░███ ░░░   ░███     ░███ ░███ ░███ ░███ ░███ ░░█████
  ░███      ░███ ░███  ░███ ░███ ░███  ███  ░███ ███ ░███ ░███ ░███ ░███ ░███  ░░░░███
  █████     ░░████████ ████ █████░░██████   ░░█████  █████░░██████  ████ █████ ██████
 ░░░░░       ░░░░░░░░ ░░░░ ░░░░░  ░░░░░░     ░░░░░  ░░░░░  ░░░░░░  ░░░░ ░░░░░ ░░░░░░
"""


def get_sys(domain):
    export = modifiers.get_modifier_domain(
        domain=domain).sndparticle_combined_export
    sys_list = []

    if export == "OFF":
        if "Spray Particle System" in domain.modifiers:
            sys_list.append(
                domain.modifiers["Spray Particle System"].particle_system.settings)
        if "Foam Particle System" in domain.modifiers:
            sys_list.append(
                domain.modifiers["Foam Particle System"].particle_system.settings)
        if "Bubble Particle System" in domain.modifiers:
            sys_list.append(
                domain.modifiers["Bubble Particle System"].particle_system.settings)

    elif export == "SPRAY_FOAM":
        if "Bubble Particle System" in domain.modifiers:
            sys_list.append(
                domain.modifiers["Bubble Particle System"].particle_system.settings)
        if "Spray + Foam Particle System" in domain.modifiers:
            sys_list.append(
                domain.modifiers["Spray + Foam Particle System"].particle_system.settings)

    elif export == "SPRAY_BUBBLES":
        if "Foam Particle System" in domain.modifiers:
            sys_list.append(
                domain.modifiers["Foam Particle System"].particle_system.settings)
        if "Spray + Bubble Particle System" in domain.modifiers:
            sys_list.append(
                domain.modifiers["Spray + Bubble Particle System"].particle_system.settings)

    elif export == "FOAM_BUBBLES":
        if "Spray Particle System" in domain.modifiers:
            sys_list.append(
                domain.modifiers["Spray Particle System"].particle_system.settings)
        if "Foam + Bubble Particle System" in domain.modifiers:
            sys_list.append(
                domain.modifiers["Foam + Bubble Particle System"].particle_system.settings)

    elif export == "SPRAY_FOAM_BUBBLES":
        if "Spray + Foam + Bubble Particle System" in domain.modifiers:
            sys_list.append(
                domain.modifiers["Spray + Foam + Bubble Particle System"].particle_system.settings)

    return sys_list


def find_particle(context):
    spray = False
    foam = False
    bubble = False
    domain = context.scene.MantaPro.active_domain

    if domain.MantaPro.particle_collection:
        for obj in domain.MantaPro.particle_collection.objects:
            if "SPRAY_PARTICLE" in obj.name:
                spray = obj
            if "FOAM_PARTICLE" in obj.name:
                foam = obj
            if "BUBBLE_PARTICLE" in obj.name:
                bubble = obj
    else:
        return spray, foam, bubble

    return spray, foam, bubble


def add_particles(context):

    # add spray

    bpy.ops.mesh.primitive_ico_sphere_add(subdivisions=1, radius=0.5,
                                          enter_editmode=False, align='WORLD', location=(-2, 0, -5))
    spray = context.active_object
    spray.name = "SPRAY_PARTICLE"
    spray.hide_render = True
    spray.scale = [0.5, 0.5, 0.5]
    material.add_material(spray, "MantaPro Liquid Spray")

    bpy.ops.mesh.primitive_ico_sphere_add(subdivisions=1, radius=0.5,
                                          enter_editmode=False, align='WORLD', location=(0, 0, -5))
    foam = context.active_object
    foam.name = "FOAM_PARTICLE"
    foam.hide_render = True
    foam.scale = [0.5, 0.5, 0.5]
    material.add_material(foam, "MantaPro Liquid Foam")

    bpy.ops.mesh.primitive_ico_sphere_add(subdivisions=1, radius=0.5,
                                          enter_editmode=False, align='WORLD', location=(2, 0, -5))
    bubble = context.active_object
    bubble.name = "BUBBLE_PARTICLE"
    bubble.hide_render = True
    bubble.scale = [0.5, 0.5, 0.5]
    material.add_material(bubble, "MantaPro Liquid Bubble")

    # add to collection
    collection = bpy.data.collections.new(name="MANTAPRO_PARTICLES")
    bpy.context.scene.collection.children.link(
        collection)

    spray.users_collection[0].objects.unlink(spray)
    foam.users_collection[0].objects.unlink(foam)
    bubble.users_collection[0].objects.unlink(bubble)

    collection.objects.link(spray)
    collection.objects.link(foam)
    collection.objects.link(bubble)

    collection.hide_viewport = True

    context.scene.MantaPro.active_domain.MantaPro.particle_collection = collection

    return spray, foam, bubble, collection


def sync_particles(context, domain):
    modifier_settings = modifiers.get_modifier_domain(
        domain=domain)
    spray, foam, bubble = find_particle(context)
    if not domain.MantaPro.particle_collection:
        spray, foam, bubble, collection = add_particles(context)

    if modifier_settings.use_spray_particles:
        modifer_spray = domain.modifiers[
            "Spray Particle System"].particle_system.settings
        modifer_spray.render_type = "OBJECT"
        modifer_spray.display_method = "DOT"
        modifer_spray.instance_object = spray

    if modifier_settings.use_foam_particles:
        modifer_foam = domain.modifiers[
            "Foam Particle System"].particle_system.settings
        modifer_foam.render_type = "OBJECT"
        modifer_foam.instance_object = foam
        modifer_foam.display_method = "DOT"

    if modifier_settings.use_bubble_particles:
        modifer_bubble = domain.modifiers[
            "Bubble Particle System"].particle_system.settings
        modifer_bubble.render_type = "OBJECT"
        modifer_bubble.instance_object = bubble
        modifer_bubble.display_method = "DOT"

    if not spray and not foam and not bubble:
        add_particles(context)


def is_linked(sys):
    if sys.render_type == "OBJECT":
        if sys.instance_object:
            return True
        else:
            return False

    elif sys.render_type == "COLLECTION":
        if sys.instance_collection:
            return True
        else:
            return False
    else:
        return False


def sys_to_name(sys):
    name = "ERROR"
    for j in range(len(sys_name_list)):
        if sys_name_list[j] in sys.name:
            name = sys_ui_name_list[j]
    return name


"""
                          █████                              ██████                                  █████     ███
                         ░░███                              ███░░███                                ░░███     ░░░
  ██████  █████████████   ░███████   ██████  ████████      ░███ ░░░  █████ ████ ████████    ██████  ███████   ████   ██████  ████████    █████
 ███░░███░░███░░███░░███  ░███░░███ ███░░███░░███░░███    ███████   ░░███ ░███ ░░███░░███  ███░░███░░░███░   ░░███  ███░░███░░███░░███  ███░░
░███████  ░███ ░███ ░███  ░███ ░███░███████  ░███ ░░░    ░░░███░     ░███ ░███  ░███ ░███ ░███ ░░░   ░███     ░███ ░███ ░███ ░███ ░███ ░░█████
░███░░░   ░███ ░███ ░███  ░███ ░███░███░░░   ░███          ░███      ░███ ░███  ░███ ░███ ░███  ███  ░███ ███ ░███ ░███ ░███ ░███ ░███  ░░░░███
░░██████  █████░███ █████ ████████ ░░██████  █████         █████     ░░████████ ████ █████░░██████   ░░█████  █████░░██████  ████ █████ ██████
 ░░░░░░  ░░░░░ ░░░ ░░░░░ ░░░░░░░░   ░░░░░░  ░░░░░         ░░░░░       ░░░░░░░░ ░░░░ ░░░░░  ░░░░░░     ░░░░░  ░░░░░  ░░░░░░  ░░░░ ░░░░░ ░░░░░░



"""


def add_ember_particles(context):
    # add particle object
    bpy.ops.mesh.primitive_ico_sphere_add(subdivisions=1, radius=0.5,
                                          enter_editmode=False, align='WORLD', location=(-2, 0, -6))
    ember = context.active_object
    ember.name = "MantaPro_Ember"
    ember.hide_render = True
    ember.scale = [0.1, 0.1, 0.1]
    material.add_material(ember, "MantaPro Ember")

    # add to collection
    collection = bpy.data.collections.new(name="MantaPro_Ember")
    bpy.context.scene.collection.children.link(collection)

    ember.users_collection[0].objects.unlink(ember)

    collection.objects.link(ember)

    collection.hide_viewport = True

    return ember, collection


def find_ember_system(flow):
    for i in flow.modifiers:
        if i.type == "PARTICLE_SYSTEM":
            if i.particle_system.settings.MantaPro.is_ember:
                return i.particle_system
    return None
