/*
 * Decompiled with CFR 0.152.
 */
package mobac.mapsources.loader;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.HttpURLConnection;
import java.net.URL;
import java.security.CodeSigner;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.cert.Certificate;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Enumeration;
import java.util.Iterator;
import java.util.List;
import java.util.ServiceLoader;
import java.util.jar.Attributes;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;
import java.util.jar.Manifest;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.zip.ZipEntry;
import java.util.zip.ZipException;
import java.util.zip.ZipFile;
import javax.swing.JOptionPane;
import mobac.exceptions.MapSourceCreateException;
import mobac.exceptions.UnrecoverableDownloadException;
import mobac.exceptions.UpdateFailedException;
import mobac.mapsources.MapSourcesManager;
import mobac.mapsources.loader.MapPackClassLoader;
import mobac.program.Logging;
import mobac.program.ProgramInfo;
import mobac.program.interfaces.MapSource;
import mobac.program.model.MapSourceLoaderInfo;
import mobac.program.model.Settings;
import mobac.utilities.GUIExceptionHandler;
import mobac.utilities.I18nUtils;
import mobac.utilities.Utilities;
import mobac.utilities.file.DirOrFileExtFilter;
import mobac.utilities.file.FileExtFilter;
import org.apache.commons.codec.binary.Hex;
import org.apache.log4j.Level;
import org.apache.log4j.Logger;

public class MapPackManager {
    private final Logger log = Logger.getLogger(MapPackManager.class);
    private final int requiredMapPackVersion;
    private final File mapPackDir;
    private final X509Certificate mapPackCert;

    public MapPackManager(File mapPackDir) throws CertificateException, IOException {
        this.mapPackDir = mapPackDir;
        this.requiredMapPackVersion = Integer.parseInt(System.getProperty("mobac.mappackversion"));
        CertificateFactory cf = CertificateFactory.getInstance("X.509");
        Collection<? extends Certificate> certs = cf.generateCertificates(Utilities.loadResourceAsStream("cert/MapPack.cer"));
        this.mapPackCert = (X509Certificate)certs.iterator().next();
    }

    public void installUpdates() throws IOException {
        File[] newMapPacks = this.mapPackDir.listFiles(new FileExtFilter(".jar.new"));
        if (newMapPacks == null) {
            throw new IOException("Failed to enumerate installable mappacks");
        }
        for (File newMapPack : newMapPacks) {
            try {
                this.testMapPack(newMapPack);
                String name = newMapPack.getName();
                name = name.substring(0, name.length() - 4);
                File oldMapPack = new File(this.mapPackDir, name);
                if (oldMapPack.isFile()) {
                    File oldMapPack2 = new File(this.mapPackDir, name + ".old");
                    Utilities.renameFile(oldMapPack, oldMapPack2);
                }
                if (newMapPack.renameTo(oldMapPack)) continue;
                throw new IOException("Failed to rename file: " + newMapPack);
            }
            catch (CertificateException e) {
                Utilities.deleteFile(newMapPack);
                this.log.error("Map pack certificate cerificateion failed (" + newMapPack.getName() + ") installation aborted and file was deleted");
            }
        }
    }

    public List<File> getAllMapPackFiles() {
        return Utilities.traverseFolder(this.mapPackDir, new DirOrFileExtFilter(".jar"));
    }

    public void loadMapPacks(MapSourcesManager mapSourcesManager) throws IOException, CertificateException {
        List<File> mapPacks = this.getAllMapPackFiles();
        for (File mapPackFile : mapPacks) {
            File oldMapPackFile = new File(mapPackFile.getAbsolutePath() + ".old");
            try {
                this.loadMapPack(mapPackFile, mapSourcesManager);
                if (!oldMapPackFile.isFile()) continue;
                Utilities.deleteFile(oldMapPackFile);
            }
            catch (MapSourceCreateException e) {
                if (oldMapPackFile.isFile()) {
                    mapPackFile.deleteOnExit();
                    File newMapPackFile = new File(mapPackFile.getAbsolutePath() + ".new");
                    Utilities.renameFile(oldMapPackFile, newMapPackFile);
                    try {
                        JOptionPane.showMessageDialog(null, I18nUtils.localizedStringForKey("msg_update_map_pack_error", new Object[0]), I18nUtils.localizedStringForKey("msg_update_map_pack_error_title", new Object[0]), 1);
                        System.exit(1);
                    }
                    catch (Exception e1) {
                        this.log.error(e1.getMessage(), e1);
                    }
                }
                GUIExceptionHandler.processException(e);
            }
            catch (CertificateException e) {
                throw e;
            }
            catch (Exception e) {
                throw new IOException("Failed to load map pack: " + mapPackFile, e);
            }
        }
    }

    public void loadMapPack(File mapPackFile, MapSourcesManager mapSourcesManager) throws CertificateException, IOException, MapSourceCreateException {
        URL url = mapPackFile.toURI().toURL();
        MapPackClassLoader urlCl = new MapPackClassLoader(url, ClassLoader.getSystemClassLoader());
        InputStream manifestIn = urlCl.getResourceAsStream("META-INF/MANIFEST.MF");
        String rev = null;
        if (manifestIn != null) {
            Manifest mf = new Manifest(manifestIn);
            rev = mf.getMainAttributes().getValue("MapPackRevision");
            manifestIn.close();
            if (rev != null) {
                rev = "exported".equals(rev) ? ProgramInfo.getRevisionStr() : Integer.toString(Utilities.parseSVNRevision(rev));
            }
            mf = null;
        }
        MapSourceLoaderInfo loaderInfo = new MapSourceLoaderInfo(MapSourceLoaderInfo.LoaderType.MAPPACK, mapPackFile, rev);
        Iterator<MapSource> iterator = ServiceLoader.load(MapSource.class, urlCl).iterator();
        while (iterator.hasNext()) {
            try {
                MapSource ms = iterator.next();
                ms.setLoaderInfo(loaderInfo);
                mapSourcesManager.addMapSource(ms);
                this.log.trace("Loaded map source: " + ms.toString() + " (name: " + ms.getName() + ")");
            }
            catch (Error e) {
                urlCl = null;
                throw new MapSourceCreateException("Failed to load a map sources from map pack: " + mapPackFile.getName() + " " + e.getMessage(), e);
            }
        }
    }

    public String downloadMD5SumList() throws IOException, UpdateFailedException {
        int responseCode;
        String md5eTag = Settings.getInstance().mapSourcesUpdate.etag;
        this.log.debug("Last md5 eTag: " + md5eTag);
        String updateUrl = System.getProperty("mobac.updateurl");
        if (updateUrl == null) {
            throw new RuntimeException("Update url not present");
        }
        byte[] data = null;
        HttpURLConnection conn = (HttpURLConnection)new URL(updateUrl).openConnection();
        conn.setInstanceFollowRedirects(false);
        if (md5eTag != null) {
            conn.addRequestProperty("If-None-Match", md5eTag);
        }
        if ((responseCode = conn.getResponseCode()) == 304) {
            this.log.debug("No newer md5 file available");
            return null;
        }
        if (responseCode != 200) {
            throw new UpdateFailedException("Invalid HTTP response: " + responseCode + " for update url " + conn.getURL());
        }
        InputStream in = conn.getInputStream();
        data = Utilities.getInputBytes(in);
        in.close();
        Settings.getInstance().mapSourcesUpdate.etag = conn.getHeaderField("ETag");
        this.log.debug("New md5 file retrieved");
        String md5sumList = new String(data);
        return md5sumList;
    }

    public void cleanMapPackDir() throws IOException {
        File[] unverifiedMapPacks;
        File[] newMapPacks;
        for (File newMapPack : newMapPacks = this.mapPackDir.listFiles(new FileExtFilter(".jar.new"))) {
            Utilities.deleteFile(newMapPack);
        }
        for (File unverifiedMapPack : unverifiedMapPacks = this.mapPackDir.listFiles(new FileExtFilter(".jar.unverified"))) {
            Utilities.deleteFile(unverifiedMapPack);
        }
    }

    public int updateMapPacks() throws UpdateFailedException, UnrecoverableDownloadException, IOException {
        String[] outdatedMapPacks;
        String updateBaseUrl = System.getProperty("mobac.updatebaseurl");
        if (updateBaseUrl == null) {
            throw new RuntimeException("Update base url not present");
        }
        this.cleanMapPackDir();
        String md5sumList = this.downloadMD5SumList();
        if (md5sumList == null) {
            return 0;
        }
        if (md5sumList.length() == 0) {
            return -1;
        }
        int updateCount = 0;
        for (String mapPack : outdatedMapPacks = this.searchForOutdatedMapPacks(md5sumList)) {
            this.log.debug("Updaing map pack: " + mapPack);
            try {
                File newMapPackFile = this.downloadMapPack(updateBaseUrl, mapPack);
                try {
                    this.testMapPack(newMapPackFile);
                }
                catch (CertificateException e) {
                    this.log.error(e.getMessage(), e);
                    Utilities.deleteFile(newMapPackFile);
                    continue;
                }
                this.log.debug("Verification of map pack \"" + mapPack + "\" passed successfully");
                int newRev = this.getMapPackRevision(newMapPackFile);
                File oldMapPack = new File(this.mapPackDir, mapPack);
                int oldRev = -1;
                if (oldMapPack.isFile()) {
                    oldRev = this.getMapPackRevision(oldMapPack);
                }
                if (newRev < oldRev) {
                    this.log.warn("Downloaded map pack was older than existing map pack - ignoring update");
                    Utilities.deleteFile(newMapPackFile);
                    continue;
                }
                String name = newMapPackFile.getName();
                name = name.replace(".unverified", ".new");
                File f = new File(newMapPackFile.getParentFile(), name);
                Utilities.renameFile(newMapPackFile, f);
                ++updateCount;
            }
            catch (IOException e) {
                this.log.error(e.getMessage(), e);
            }
        }
        return updateCount;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public int getMapPackRevision(File mapPackFile) throws ZipException, IOException {
        try (ZipFile zip = new ZipFile(mapPackFile);){
            ZipEntry entry = zip.getEntry("META-INF/MANIFEST.MF");
            if (entry == null) {
                throw new ZipException("Unable to find MANIFEST.MF");
            }
            Manifest mf = new Manifest(zip.getInputStream(entry));
            Attributes a = mf.getMainAttributes();
            String mpv = a.getValue("MapPackRevision").trim();
            int n = Utilities.parseSVNRevision(mpv);
            return n;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public File downloadMapPack(String baseURL, String mapPackFilename) throws IOException {
        if (!mapPackFilename.endsWith(".jar")) {
            throw new IOException("Invalid map pack filename");
        }
        byte[] mapPackData = Utilities.downloadHttpFile(baseURL + mapPackFilename);
        File newMapPackFile = new File(this.mapPackDir, mapPackFilename + ".unverified");
        FileOutputStream out = new FileOutputStream(newMapPackFile);
        try {
            out.write(mapPackData);
            out.flush();
        }
        finally {
            Utilities.closeStream(out);
        }
        this.log.debug("New map pack \"" + mapPackFilename + "\" successfully downloaded");
        return newMapPackFile;
    }

    public String[] searchForOutdatedMapPacks(String md5sumList) throws UpdateFailedException {
        ArrayList<String> outdatedMappacks = new ArrayList<String>();
        String[] md5s = md5sumList.split("[\\n\\r]+");
        Pattern linePattern = Pattern.compile("([0-9a-f]{32}) (mp-[\\w]+\\.jar)");
        for (String line : md5s) {
            if ((line = line.trim()).length() == 0) continue;
            Matcher m = linePattern.matcher(line);
            if (!m.matches()) {
                throw new UpdateFailedException("Invalid content found in md5 list: \"" + line + "\"");
            }
            String md5 = m.group(1);
            String filename = m.group(2);
            File mapPackFile = new File(this.mapPackDir, filename + ".new");
            if (!mapPackFile.isFile()) {
                mapPackFile = new File(this.mapPackDir, filename);
            }
            if (!mapPackFile.isFile()) {
                outdatedMappacks.add(filename);
                this.log.debug("local map pack file missing: " + filename);
                continue;
            }
            try {
                String localmd5 = this.generateMappackMD5(mapPackFile);
                if (localmd5.equals(md5)) continue;
                this.log.debug("Found outdated map pack: \"" + filename + "\" local md5: " + localmd5 + " remote md5: " + md5);
                outdatedMappacks.add(filename);
            }
            catch (Exception e) {
                this.log.error("Failed to generate md5sum of " + mapPackFile, e);
            }
        }
        String[] result = new String[outdatedMappacks.size()];
        outdatedMappacks.toArray(result);
        return result;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public String generateMappackMD5(File mapPackFile) throws IOException, NoSuchAlgorithmException {
        try (ZipFile zip = new ZipFile(mapPackFile);){
            Enumeration<? extends ZipEntry> entries = zip.entries();
            MessageDigest md5Total = MessageDigest.getInstance("MD5");
            MessageDigest md5 = MessageDigest.getInstance("MD5");
            while (entries.hasMoreElements()) {
                String name;
                ZipEntry entry = entries.nextElement();
                if (entry.isDirectory() || (name = entry.getName()).toUpperCase().startsWith("META-INF")) continue;
                md5.reset();
                InputStream in = zip.getInputStream(entry);
                byte[] data = Utilities.getInputBytes(in);
                in.close();
                byte[] digest = md5.digest(data);
                this.log.trace("Hashsum " + Hex.encodeHexString(digest) + " includes \"" + name + "\"");
                md5Total.update(digest);
                md5Total.update(name.getBytes());
            }
            String md5sum = Hex.encodeHexString(md5Total.digest());
            this.log.trace("md5sum of " + mapPackFile.getName() + ": " + md5sum);
            String string = md5sum;
            return string;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void testMapPack(File mapPackFile) throws IOException, CertificateException {
        String fileName = mapPackFile.getName();
        try (JarFile jf = new JarFile(mapPackFile, true);){
            Enumeration<JarEntry> it = jf.entries();
            while (it.hasMoreElements()) {
                JarEntry entry = it.nextElement();
                if (!entry.getName().endsWith(".class")) continue;
                Utilities.readFully(jf.getInputStream(entry));
                if (entry.getCodeSigners() == null) {
                    throw new CertificateException("Unsigned class file found: " + entry.getName());
                }
                CodeSigner signer = entry.getCodeSigners()[0];
                List<? extends Certificate> cp = signer.getSignerCertPath().getCertificates();
                if (cp.size() > 1) {
                    throw new CertificateException("Signature certificate not accepted: certificate path contains more than one certificate");
                }
                if (this.mapPackCert.equals(cp.get(0))) continue;
                throw new CertificateException("Signature certificate not accepted: not the MapPack signer certificate");
            }
            Manifest mf = jf.getManifest();
            Attributes a = mf.getMainAttributes();
            String mpv = a.getValue("MapPackVersion");
            if (mpv == null) {
                throw new IOException("MapPackVersion info missing!");
            }
            int mapPackVersion = Integer.parseInt(mpv);
            if (this.requiredMapPackVersion != mapPackVersion) {
                throw new IOException("This pack \"" + fileName + "\" is not compatible with this MOBAC version.");
            }
            ZipEntry entry = jf.getEntry("META-INF/services/mobac.program.interfaces.MapSource");
            if (entry == null) {
                throw new IOException("MapSources services list is missing in file " + fileName);
            }
        }
    }

    public static void main(String[] args) {
        try {
            Logging.configureConsoleLogging(Level.DEBUG);
            ProgramInfo.initialize();
            MapPackManager mpm = new MapPackManager(new File("mapsources"));
            mpm.updateMapPacks();
        }
        catch (Exception e) {
            e.printStackTrace();
        }
    }
}

