/*
 * Decompiled with CFR 0.152.
 */
package org.mapsforge.map.reader;

import java.io.File;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.util.ArrayList;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.mapsforge.core.model.BoundingBox;
import org.mapsforge.core.model.LatLong;
import org.mapsforge.core.model.Tag;
import org.mapsforge.core.model.Tile;
import org.mapsforge.core.util.LatLongUtils;
import org.mapsforge.core.util.MercatorProjection;
import org.mapsforge.map.datastore.MapDataStore;
import org.mapsforge.map.datastore.MapReadResult;
import org.mapsforge.map.datastore.PoiWayBundle;
import org.mapsforge.map.datastore.PointOfInterest;
import org.mapsforge.map.datastore.Way;
import org.mapsforge.map.reader.IndexCache;
import org.mapsforge.map.reader.QueryParameters;
import org.mapsforge.map.reader.ReadBuffer;
import org.mapsforge.map.reader.header.MapFileException;
import org.mapsforge.map.reader.header.MapFileHeader;
import org.mapsforge.map.reader.header.MapFileInfo;
import org.mapsforge.map.reader.header.SubFileParameter;

public class MapFile
extends MapDataStore {
    private static final long BITMASK_INDEX_OFFSET = 0x7FFFFFFFFFL;
    private static final long BITMASK_INDEX_WATER = 0x8000000000L;
    private static final Byte DEFAULT_START_ZOOM_LEVEL = 12;
    private static final int INDEX_CACHE_SIZE = 64;
    private static final String INVALID_FIRST_WAY_OFFSET = "invalid first way offset: ";
    private static final Logger LOGGER = Logger.getLogger(MapFile.class.getName());
    private static final int POI_FEATURE_ELEVATION = 32;
    private static final int POI_FEATURE_HOUSE_NUMBER = 64;
    private static final int POI_FEATURE_NAME = 128;
    private static final int POI_LAYER_BITMASK = 240;
    private static final int POI_LAYER_SHIFT = 4;
    private static final int POI_NUMBER_OF_TAGS_BITMASK = 15;
    private static final String READ_ONLY_MODE = "r";
    private static final byte SIGNATURE_LENGTH_BLOCK = 32;
    private static final byte SIGNATURE_LENGTH_POI = 32;
    private static final byte SIGNATURE_LENGTH_WAY = 32;
    private static final String TAG_KEY_ELE = "ele";
    private static final String TAG_KEY_HOUSE_NUMBER = "addr:housenumber";
    private static final String TAG_KEY_NAME = "name";
    private static final String TAG_KEY_REF = "ref";
    private static final int WAY_FEATURE_DATA_BLOCKS_BYTE = 8;
    private static final int WAY_FEATURE_DOUBLE_DELTA_ENCODING = 4;
    private static final int WAY_FEATURE_HOUSE_NUMBER = 64;
    private static final int WAY_FEATURE_LABEL_POSITION = 16;
    private static final int WAY_FEATURE_NAME = 128;
    private static final int WAY_FEATURE_REF = 32;
    private static final int WAY_LAYER_BITMASK = 240;
    private static final int WAY_LAYER_SHIFT = 4;
    private static final int WAY_NUMBER_OF_TAGS_BITMASK = 15;
    public static final MapFile TEST_MAP_FILE = new MapFile();
    public static boolean wayFilterEnabled = true;
    public static int wayFilterDistance = 20;
    private final IndexCache databaseIndexCache;
    private final long fileSize;
    private final RandomAccessFile inputFile;
    private final MapFileHeader mapFileHeader;
    private final ReadBuffer readBuffer;
    private final long timestamp;

    private MapFile() {
        this.databaseIndexCache = null;
        this.fileSize = 0L;
        this.inputFile = null;
        this.mapFileHeader = null;
        this.readBuffer = null;
        this.timestamp = System.currentTimeMillis();
    }

    public MapFile(File mapFile) {
        this(mapFile, null);
    }

    public MapFile(File mapFile, String language) {
        super(language);
        if (mapFile == null) {
            throw new MapFileException("mapFile must not be null");
        }
        try {
            if (!mapFile.exists()) {
                throw new MapFileException("file does not exist: " + mapFile);
            }
            if (!mapFile.isFile()) {
                throw new MapFileException("not a file: " + mapFile);
            }
            if (!mapFile.canRead()) {
                throw new MapFileException("cannot read file: " + mapFile);
            }
            this.inputFile = new RandomAccessFile(mapFile, READ_ONLY_MODE);
            this.fileSize = this.inputFile.length();
            this.readBuffer = new ReadBuffer(this.inputFile);
            this.mapFileHeader = new MapFileHeader();
            this.mapFileHeader.readHeader(this.readBuffer, this.fileSize);
            this.databaseIndexCache = new IndexCache(this.inputFile, 64);
            this.timestamp = mapFile.lastModified();
        }
        catch (Exception e) {
            LOGGER.log(Level.SEVERE, e.getMessage(), e);
            this.closeFile();
            throw new MapFileException(e.getMessage());
        }
    }

    public MapFile(String mapFileName, String language) {
        this(new File(mapFileName), language);
    }

    @Override
    public BoundingBox boundingBox() {
        return this.getMapFileInfo().boundingBox;
    }

    @Override
    public void close() {
        this.closeFile();
    }

    private void closeFile() {
        try {
            this.databaseIndexCache.destroy();
            this.inputFile.close();
        }
        catch (Exception e) {
            LOGGER.log(Level.SEVERE, e.getMessage(), e);
        }
    }

    private void decodeWayNodesDoubleDelta(LatLong[] waySegment, double tileLatitude, double tileLongitude) {
        double wayNodeLatitude = tileLatitude + LatLongUtils.microdegreesToDegrees(this.readBuffer.readSignedInt());
        double wayNodeLongitude = tileLongitude + LatLongUtils.microdegreesToDegrees(this.readBuffer.readSignedInt());
        waySegment[0] = new LatLong(wayNodeLatitude, wayNodeLongitude);
        double previousSingleDeltaLatitude = 0.0;
        double previousSingleDeltaLongitude = 0.0;
        for (int wayNodesIndex = 1; wayNodesIndex < waySegment.length; ++wayNodesIndex) {
            double doubleDeltaLatitude = LatLongUtils.microdegreesToDegrees(this.readBuffer.readSignedInt());
            double doubleDeltaLongitude = LatLongUtils.microdegreesToDegrees(this.readBuffer.readSignedInt());
            double singleDeltaLatitude = doubleDeltaLatitude + previousSingleDeltaLatitude;
            double singleDeltaLongitude = doubleDeltaLongitude + previousSingleDeltaLongitude;
            waySegment[wayNodesIndex] = new LatLong(wayNodeLatitude += singleDeltaLatitude, wayNodeLongitude += singleDeltaLongitude);
            previousSingleDeltaLatitude = singleDeltaLatitude;
            previousSingleDeltaLongitude = singleDeltaLongitude;
        }
    }

    private void decodeWayNodesSingleDelta(LatLong[] waySegment, double tileLatitude, double tileLongitude) {
        double wayNodeLatitude = tileLatitude + LatLongUtils.microdegreesToDegrees(this.readBuffer.readSignedInt());
        double wayNodeLongitude = tileLongitude + LatLongUtils.microdegreesToDegrees(this.readBuffer.readSignedInt());
        waySegment[0] = new LatLong(wayNodeLatitude, wayNodeLongitude);
        for (int wayNodesIndex = 1; wayNodesIndex < waySegment.length; ++wayNodesIndex) {
            waySegment[wayNodesIndex] = new LatLong(wayNodeLatitude += LatLongUtils.microdegreesToDegrees(this.readBuffer.readSignedInt()), wayNodeLongitude += LatLongUtils.microdegreesToDegrees(this.readBuffer.readSignedInt()));
        }
    }

    @Override
    public long getDataTimestamp(Tile tile) {
        return this.timestamp;
    }

    public MapFileInfo getMapFileInfo() {
        return this.mapFileHeader.getMapFileInfo();
    }

    public String[] getMapLanguages() {
        String languagesPreference = this.getMapFileInfo().languagesPreference;
        if (languagesPreference != null && !languagesPreference.trim().isEmpty()) {
            return languagesPreference.split(",");
        }
        return null;
    }

    private PoiWayBundle processBlock(QueryParameters queryParameters, SubFileParameter subFileParameter, BoundingBox boundingBox, double tileLatitude, double tileLongitude) {
        if (!this.processBlockSignature()) {
            return null;
        }
        int[][] zoomTable = this.readZoomTable(subFileParameter);
        int zoomTableRow = queryParameters.queryZoomLevel - subFileParameter.zoomLevelMin;
        int poisOnQueryZoomLevel = zoomTable[zoomTableRow][0];
        int waysOnQueryZoomLevel = zoomTable[zoomTableRow][1];
        int firstWayOffset = this.readBuffer.readUnsignedInt();
        if (firstWayOffset < 0) {
            LOGGER.warning(INVALID_FIRST_WAY_OFFSET + firstWayOffset);
            return null;
        }
        if ((firstWayOffset += this.readBuffer.getBufferPosition()) > this.readBuffer.getBufferSize()) {
            LOGGER.warning(INVALID_FIRST_WAY_OFFSET + firstWayOffset);
            return null;
        }
        boolean filterRequired = queryParameters.queryZoomLevel > subFileParameter.baseZoomLevel;
        List<PointOfInterest> pois = this.processPOIs(tileLatitude, tileLongitude, poisOnQueryZoomLevel, boundingBox, filterRequired);
        if (pois == null) {
            return null;
        }
        if (this.readBuffer.getBufferPosition() > firstWayOffset) {
            LOGGER.warning("invalid buffer position: " + this.readBuffer.getBufferPosition());
            return null;
        }
        this.readBuffer.setBufferPosition(firstWayOffset);
        List<Way> ways = this.processWays(queryParameters, waysOnQueryZoomLevel, boundingBox, filterRequired, tileLatitude, tileLongitude);
        if (ways == null) {
            return null;
        }
        return new PoiWayBundle(pois, ways);
    }

    private MapReadResult processBlocks(QueryParameters queryParameters, SubFileParameter subFileParameter, BoundingBox boundingBox) throws IOException {
        boolean queryIsWater = true;
        boolean queryReadWaterInfo = false;
        MapReadResult mapFileReadResult = new MapReadResult();
        for (long row = queryParameters.fromBlockY; row <= queryParameters.toBlockY; ++row) {
            for (long column = queryParameters.fromBlockX; column <= queryParameters.toBlockX; ++column) {
                long nextBlockPointer;
                long currentBlockPointer;
                long blockNumber = row * subFileParameter.blocksWidth + column;
                long currentBlockIndexEntry = this.databaseIndexCache.getIndexEntry(subFileParameter, blockNumber);
                if (queryIsWater) {
                    queryIsWater &= (currentBlockIndexEntry & 0x8000000000L) != 0L;
                    queryReadWaterInfo = true;
                }
                if ((currentBlockPointer = currentBlockIndexEntry & 0x7FFFFFFFFFL) < 1L || currentBlockPointer > subFileParameter.subFileSize) {
                    LOGGER.warning("invalid current block pointer: " + currentBlockPointer);
                    LOGGER.warning("subFileSize: " + subFileParameter.subFileSize);
                    return null;
                }
                if (blockNumber + 1L == subFileParameter.numberOfBlocks) {
                    nextBlockPointer = subFileParameter.subFileSize;
                } else {
                    nextBlockPointer = this.databaseIndexCache.getIndexEntry(subFileParameter, blockNumber + 1L) & 0x7FFFFFFFFFL;
                    if (nextBlockPointer > subFileParameter.subFileSize) {
                        LOGGER.warning("invalid next block pointer: " + nextBlockPointer);
                        LOGGER.warning("sub-file size: " + subFileParameter.subFileSize);
                        return null;
                    }
                }
                int currentBlockSize = (int)(nextBlockPointer - currentBlockPointer);
                if (currentBlockSize < 0) {
                    LOGGER.warning("current block size must not be negative: " + currentBlockSize);
                    return null;
                }
                if (currentBlockSize == 0) continue;
                if (currentBlockSize > ReadBuffer.getMaximumBufferSize()) {
                    LOGGER.warning("current block size too large: " + currentBlockSize);
                    continue;
                }
                if (currentBlockPointer + (long)currentBlockSize > this.fileSize) {
                    LOGGER.warning("current block largher than file size: " + currentBlockSize);
                    return null;
                }
                this.inputFile.seek(subFileParameter.startAddress + currentBlockPointer);
                if (!this.readBuffer.readFromFile(currentBlockSize)) {
                    LOGGER.warning("reading current block has failed: " + currentBlockSize);
                    return null;
                }
                double tileLatitude = MercatorProjection.tileYToLatitude(subFileParameter.boundaryTileTop + row, subFileParameter.baseZoomLevel);
                double tileLongitude = MercatorProjection.tileXToLongitude(subFileParameter.boundaryTileLeft + column, subFileParameter.baseZoomLevel);
                try {
                    PoiWayBundle poiWayBundle = this.processBlock(queryParameters, subFileParameter, boundingBox, tileLatitude, tileLongitude);
                    if (poiWayBundle == null) continue;
                    mapFileReadResult.add(poiWayBundle);
                    continue;
                }
                catch (ArrayIndexOutOfBoundsException e) {
                    LOGGER.log(Level.SEVERE, e.getMessage(), e);
                }
            }
        }
        if (queryIsWater && queryReadWaterInfo) {
            mapFileReadResult.isWater = true;
        }
        return mapFileReadResult;
    }

    private boolean processBlockSignature() {
        String signatureBlock;
        if (this.mapFileHeader.getMapFileInfo().debugFile && !(signatureBlock = this.readBuffer.readUTF8EncodedString(32)).startsWith("###TileStart")) {
            LOGGER.warning("invalid block signature: " + signatureBlock);
            return false;
        }
        return true;
    }

    private List<PointOfInterest> processPOIs(double tileLatitude, double tileLongitude, int numberOfPois, BoundingBox boundingBox, boolean filterRequired) {
        ArrayList<PointOfInterest> pois = new ArrayList<PointOfInterest>();
        Tag[] poiTags = this.mapFileHeader.getMapFileInfo().poiTags;
        for (int elementCounter = numberOfPois; elementCounter != 0; --elementCounter) {
            boolean featureElevation;
            String signaturePoi;
            if (this.mapFileHeader.getMapFileInfo().debugFile && !(signaturePoi = this.readBuffer.readUTF8EncodedString(32)).startsWith("***POIStart")) {
                LOGGER.warning("invalid POI signature: " + signaturePoi);
                return null;
            }
            double latitude = tileLatitude + LatLongUtils.microdegreesToDegrees(this.readBuffer.readSignedInt());
            double longitude = tileLongitude + LatLongUtils.microdegreesToDegrees(this.readBuffer.readSignedInt());
            byte specialByte = this.readBuffer.readByte();
            byte layer = (byte)((specialByte & 0xF0) >>> 4);
            byte numberOfTags = (byte)(specialByte & 0xF);
            ArrayList<Tag> tags = new ArrayList<Tag>();
            for (byte tagIndex = numberOfTags; tagIndex != 0; tagIndex = (byte)(tagIndex - 1)) {
                int tagId = this.readBuffer.readUnsignedInt();
                if (tagId < 0 || tagId >= poiTags.length) {
                    LOGGER.warning("invalid POI tag ID: " + tagId);
                    return null;
                }
                tags.add(poiTags[tagId]);
            }
            byte featureByte = this.readBuffer.readByte();
            boolean featureName = (featureByte & 0x80) != 0;
            boolean featureHouseNumber = (featureByte & 0x40) != 0;
            boolean bl = featureElevation = (featureByte & 0x20) != 0;
            if (featureName) {
                tags.add(new Tag(TAG_KEY_NAME, this.extractLocalized(this.readBuffer.readUTF8EncodedString())));
            }
            if (featureHouseNumber) {
                tags.add(new Tag(TAG_KEY_HOUSE_NUMBER, this.readBuffer.readUTF8EncodedString()));
            }
            if (featureElevation) {
                tags.add(new Tag(TAG_KEY_ELE, Integer.toString(this.readBuffer.readSignedInt())));
            }
            LatLong position = new LatLong(latitude, longitude);
            if (filterRequired && !boundingBox.contains(position)) continue;
            pois.add(new PointOfInterest(layer, tags, position));
        }
        return pois;
    }

    private LatLong[][] processWayDataBlock(double tileLatitude, double tileLongitude, boolean doubleDeltaEncoding) {
        int numberOfWayCoordinateBlocks = this.readBuffer.readUnsignedInt();
        if (numberOfWayCoordinateBlocks < 1 || numberOfWayCoordinateBlocks > Short.MAX_VALUE) {
            LOGGER.warning("invalid number of way coordinate blocks: " + numberOfWayCoordinateBlocks);
            return null;
        }
        LatLong[][] wayCoordinates = new LatLong[numberOfWayCoordinateBlocks][];
        for (int coordinateBlock = 0; coordinateBlock < numberOfWayCoordinateBlocks; ++coordinateBlock) {
            int numberOfWayNodes = this.readBuffer.readUnsignedInt();
            if (numberOfWayNodes < 2 || numberOfWayNodes > Short.MAX_VALUE) {
                LOGGER.warning("invalid number of way nodes: " + numberOfWayNodes);
                return null;
            }
            LatLong[] waySegment = new LatLong[numberOfWayNodes];
            if (doubleDeltaEncoding) {
                this.decodeWayNodesDoubleDelta(waySegment, tileLatitude, tileLongitude);
            } else {
                this.decodeWayNodesSingleDelta(waySegment, tileLatitude, tileLongitude);
            }
            wayCoordinates[coordinateBlock] = waySegment;
        }
        return wayCoordinates;
    }

    private List<Way> processWays(QueryParameters queryParameters, int numberOfWays, BoundingBox boundingBox, boolean filterRequired, double tileLatitude, double tileLongitude) {
        ArrayList<Way> ways = new ArrayList<Way>();
        Tag[] wayTags = this.mapFileHeader.getMapFileInfo().wayTags;
        BoundingBox wayFilterBbox = boundingBox.extendMeters(wayFilterDistance);
        for (int elementCounter = numberOfWays; elementCounter != 0; --elementCounter) {
            boolean featureWayDoubleDeltaEncoding;
            String signatureWay;
            if (this.mapFileHeader.getMapFileInfo().debugFile && !(signatureWay = this.readBuffer.readUTF8EncodedString(32)).startsWith("---WayStart")) {
                LOGGER.warning("invalid way signature: " + signatureWay);
                return null;
            }
            int wayDataSize = this.readBuffer.readUnsignedInt();
            if (wayDataSize < 0) {
                LOGGER.warning("invalid way data size: " + wayDataSize);
                return null;
            }
            if (queryParameters.useTileBitmask) {
                int tileBitmask = this.readBuffer.readShort();
                if ((queryParameters.queryTileBitmask & tileBitmask) == 0) {
                    this.readBuffer.skipBytes(wayDataSize - 2);
                    continue;
                }
            } else {
                this.readBuffer.skipBytes(2);
            }
            byte specialByte = this.readBuffer.readByte();
            byte layer = (byte)((specialByte & 0xF0) >>> 4);
            byte numberOfTags = (byte)(specialByte & 0xF);
            ArrayList<Tag> tags = new ArrayList<Tag>();
            for (byte tagIndex = numberOfTags; tagIndex != 0; tagIndex = (byte)(tagIndex - 1)) {
                int tagId = this.readBuffer.readUnsignedInt();
                if (tagId < 0 || tagId >= wayTags.length) {
                    LOGGER.warning("invalid way tag ID: " + tagId);
                    return null;
                }
                tags.add(wayTags[tagId]);
            }
            byte featureByte = this.readBuffer.readByte();
            boolean featureName = (featureByte & 0x80) != 0;
            boolean featureHouseNumber = (featureByte & 0x40) != 0;
            boolean featureRef = (featureByte & 0x20) != 0;
            boolean featureLabelPosition = (featureByte & 0x10) != 0;
            boolean featureWayDataBlocksByte = (featureByte & 8) != 0;
            boolean bl = featureWayDoubleDeltaEncoding = (featureByte & 4) != 0;
            if (featureName) {
                tags.add(new Tag(TAG_KEY_NAME, this.extractLocalized(this.readBuffer.readUTF8EncodedString())));
            }
            if (featureHouseNumber) {
                tags.add(new Tag(TAG_KEY_HOUSE_NUMBER, this.readBuffer.readUTF8EncodedString()));
            }
            if (featureRef) {
                tags.add(new Tag(TAG_KEY_REF, this.readBuffer.readUTF8EncodedString()));
            }
            LatLong labelPosition = this.readOptionalLabelPosition(tileLatitude, tileLongitude, featureLabelPosition);
            int wayDataBlocks = this.readOptionalWayDataBlocksByte(featureWayDataBlocksByte);
            if (wayDataBlocks < 1) {
                LOGGER.warning("invalid number of way data blocks: " + wayDataBlocks);
                return null;
            }
            for (int wayDataBlock = 0; wayDataBlock < wayDataBlocks; ++wayDataBlock) {
                LatLong[][] wayNodes = this.processWayDataBlock(tileLatitude, tileLongitude, featureWayDoubleDeltaEncoding);
                if (wayNodes == null || filterRequired && wayFilterEnabled && !wayFilterBbox.intersectsArea(wayNodes)) continue;
                ways.add(new Way(layer, tags, wayNodes, labelPosition));
            }
        }
        return ways;
    }

    @Override
    public synchronized MapReadResult readMapData(Tile tile) {
        try {
            QueryParameters queryParameters = new QueryParameters();
            queryParameters.queryZoomLevel = this.mapFileHeader.getQueryZoomLevel(tile.zoomLevel);
            SubFileParameter subFileParameter = this.mapFileHeader.getSubFileParameter(queryParameters.queryZoomLevel);
            if (subFileParameter == null) {
                LOGGER.warning("no sub-file for zoom level: " + queryParameters.queryZoomLevel);
                return null;
            }
            queryParameters.calculateBaseTiles(tile, subFileParameter);
            queryParameters.calculateBlocks(subFileParameter);
            return this.processBlocks(queryParameters, subFileParameter, tile.getBoundingBox());
        }
        catch (IOException e) {
            LOGGER.log(Level.SEVERE, e.getMessage(), e);
            return null;
        }
    }

    private LatLong readOptionalLabelPosition(double tileLatitude, double tileLongitude, boolean featureLabelPosition) {
        if (featureLabelPosition) {
            double latitude = tileLatitude + LatLongUtils.microdegreesToDegrees(this.readBuffer.readSignedInt());
            double longitude = tileLongitude + LatLongUtils.microdegreesToDegrees(this.readBuffer.readSignedInt());
            return new LatLong(latitude, longitude);
        }
        return null;
    }

    private int readOptionalWayDataBlocksByte(boolean featureWayDataBlocksByte) {
        if (featureWayDataBlocksByte) {
            return this.readBuffer.readUnsignedInt();
        }
        return 1;
    }

    private int[][] readZoomTable(SubFileParameter subFileParameter) {
        int rows = subFileParameter.zoomLevelMax - subFileParameter.zoomLevelMin + 1;
        int[][] zoomTable = new int[rows][2];
        int cumulatedNumberOfPois = 0;
        int cumulatedNumberOfWays = 0;
        for (int row = 0; row < rows; ++row) {
            zoomTable[row][0] = cumulatedNumberOfPois += this.readBuffer.readUnsignedInt();
            zoomTable[row][1] = cumulatedNumberOfWays += this.readBuffer.readUnsignedInt();
        }
        return zoomTable;
    }

    public void restrictToZoomRange(byte minZoom, byte maxZoom) {
        this.getMapFileInfo().zoomLevelMax = maxZoom;
        this.getMapFileInfo().zoomLevelMin = minZoom;
    }

    @Override
    public LatLong startPosition() {
        if (null != this.getMapFileInfo().startPosition) {
            return this.getMapFileInfo().startPosition;
        }
        return this.getMapFileInfo().boundingBox.getCenterPoint();
    }

    @Override
    public Byte startZoomLevel() {
        if (null != this.getMapFileInfo().startZoomLevel) {
            return this.getMapFileInfo().startZoomLevel;
        }
        return DEFAULT_START_ZOOM_LEVEL;
    }

    @Override
    public boolean supportsTile(Tile tile) {
        return tile.getBoundingBox().intersects(this.getMapFileInfo().boundingBox);
    }
}

